/*=============================================================================
*
*   WCSLIB - an implementation of the FITS WCS proposal.
*   Copyright (C) 1995, Mark Calabretta
*
*   This library is free software; you can redistribute it and/or modify it
*   under the terms of the GNU Library General Public License as published
*   by the Free Software Foundation; either version 2 of the License, or (at
*   your option) any later version.
*
*   This library is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library
*   General Public License for more details.
*
*   You should have received a copy of the GNU Library General Public License
*   along with this library; if not, write to the Free Software Foundation,
*   Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*
*   Correspondence concerning WCSLIB may be directed to:
*      Internet email: mcalabre@atnf.csiro.au
*      Postal address: Dr. Mark Calabretta,
*                      Australia Telescope National Facility,
*                      P.O. Box 76,
*                      Epping, NSW, 2121,
*                      AUSTRALIA
*
*=============================================================================
*
*   C routines which implement the FITS World Coordinate System (WCS)
*   convention.
*
*   Summary of routines
*   -------------------
*   wcsfwd() and wcsrev() are high level driver routine for the WCS linear
*   transformation, spherical coordinate transformation, and spherical
*   projection routines.
*
*   Given an element of (lng,lat) and of (x,y) a hybrid routine, wcsmix(),
*   iteratively solves for the two unknown elements.
*
*
*   Forward transformation; wcsfwd()
*   --------------------------------
*   Compute pixel coordinate for given celestial coordinates (lng,lat).
*
*   Given:
*      pcode[4] char     WCS projection code (see below).
*      lng,lat  double   Celestial longitude and latitude, in degrees.
*
*   Given and returned:
*      cel      celprm*  Spherical coordinate transformation parameters
*                        (usage is described in the prologue to "cel.c").
*
*   Returned:
*      phi,     double*  Longitude and latitude in the native coordinate
*      theta             system of the projection, in degrees.
*
*   Given and returned:
*      prj      prjprm*  Projection parameters (usage is described in the
*                        prologue to "proj.c").
*
*   Returned:
*      imgcrd   double[] Image coordinate.  The first two elements of this
*                        vector are assumed to be the projected x-, and
*                        y-coordinates, in "degrees".
*
*   Given and returned:
*      lin      linprm*  Linear transformation parameters (usage is described
*                        in the prologue to "lin.c").
*
*   Returned:
*      pixcrd   double[] Pixel coordinate.
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Invalid coordinate transformation parameters.
*                           2: Invalid projection parameters.
*                           3: Invalid value of (lng,lat).
*                           4: Invalid linear transformation parameters.
*
*   Reverse transformation; wcsrev()
*   --------------------------------
*   Compute the celestial coordinates (lng,lat) for a given pixel coordinate.
*
*   Given:
*      pcode[4] char     WCS projection code (see below).
*      pixcrd   double[] Pixel coordinate.
*
*   Given and returned:
*      lin      linprm*  Linear transformation parameters (usage is described
*                        in the prologue to "lin.c").
*
*   Returned:
*      imgcrd   double[] Image coordinate.  The first two elements of this
*                        vector are assumed to be the projected x-, and
*                        y-coordinates, in "degrees".
*
*   Given and returned:
*      prj      prjprm*  Projection parameters (usage is described in the
*                        prologue to "proj.c").
*
*   Returned:
*      phi,     double*  Longitude and latitude in the native coordinate
*      theta             system of the projection, in degrees.
*
*   Given and returned:
*      cel      celprm*  Spherical coordinate transformation parameters
*                        (usage is described in the prologue to "cel.c").
*
*   Returned:
*      lng,lat  double*  Celestial longitude and latitude, in degrees.
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Invalid coordinate transformation parameters.
*                           2: Invalid projection parameters.
*                           3: Invalid value of (lng,lat).
*                           4: Invalid linear transformation parameters.
*
*   Hybrid transformation; wcsmix()
*   -------------------------------
*   Given an element of (lng,lat) and of a pixel coordinate solve for the
*   unknowns by iterating on the unknown celestial coordinate element.
*
*   Given:
*      pcode[4] char     WCS projection code (see below).
*      mix      int      The absolute value indicates which element of the
*                        pixel coordinate (pixcrd) is given (counting from 1),
*                        and the sign indicates which element of the celestial
*                        coordinate is given:
*                           mix < 0: celestial longitude is given,
*                                    latitude returned.
*                           mix > 0: celestial latitude is given,
*                                    longitude returned.
*      vspan[2] double   Solution interval for the celestial coordinate, in
*                        degrees.
*      vstep    double   Step size for solution search, in degrees.  If zero,
*                        a sensible, although perhaps non-optimal default will
*                        be used.
*      viter    int      If a solution is not found then the step size will be
*                        halved and the search recommenced.  viter controls
*                        how many times the step size is halved.  The allowed
*                        range is 0 - 5.
*
*   Given or returned:
*      lng,lat  double*  Celestial longitude and latitude, in degrees.  Which
*                        is given and which returned depends on the value of
*                        'mix'.
*
*   Given and returned:
*      cel      celprm*  Spherical coordinate transformation parameters
*                        (usage is described in the prologue to "cel.c").
*
*   Returned:
*      phi,     double*  Longitude and latitude in the native coordinate
*      theta             system of the projection, in degrees.
*
*   Given and returned:
*      prj      prjprm*  Projection parameters (usage is described in the
*                        prologue to "proj.c").
*
*   Returned:
*      imgcrd   double[] Image coordinate.  It is assumed that the projected
*                        x-, and y-coordinates are the first two elements of
*                        this vector.
*
*   Given and returned:
*      lin      linprm*  Linear transformation parameters (usage is described
*                        in the prologue to "lin.c").
*
*   Given and returned:
*      pixcrd   double[] Pixel coordinate.  The element indicated by mix is
*                        given and the remaining elements are returned.
*
*   Function return value:
*               int      Error status
*                           0: Success.
*                           1: Invalid coordinate transformation parameters.
*                           2: Invalid projection parameters.
*                           3: Coordinate transformation error.
*                           4: Invalid linear transformation parameters.
*                           5: No solution found in the specified interval.
*
*   Algorithm:
*      Initially the specified solution interval is checked to see if it's a
*      "crossing" interval.  If it isn't, a search is made for a crossing
*      solution by iterating on the unknown celestial coordinate starting at
*      the upper limit of the solution interval and decrementing by the
*      specified step size.  A crossing is indicated if the trial value of the
*      pixel coordinate steps through the value specified.  If a crossing
*      interval is found then the solution is determined by a modified form of
*      "regula falsi" division of the crossing interval.  If no crossing
*      interval was found within the specified solution interval then a search
*      is made for a "non-crossing" solution as may arise from a point of
*      tangency.  The process is complicated by having to make allowance for
*      the discontinuities that occur in all map projections.
*
*      Once one solution has been determined others may be found by subsequent
*      invokations of wcsmix() with suitably restricted solution intervals.
*
*      Note that there is a circumstance where the problem posed to wcsmix()
*      is ill-conditioned and it may fail to find a valid solution where one
*      does exist.  This may arise when the solution point lies at a native
*      pole of a projection in which the pole is represented as a finite
*      interval.
*
*      Because of its generality wcsmix() is very compute-intensive.  For
*      compute-limited applications more efficient special-case solvers could
*      be written for simple projections, for example non-oblique cylindrical
*      projections.
*
*   WCS projection codes
*   --------------------
*   Zenithals/azimuthals:
*      AZP: zenithal/azimuthal perspective
*      TAN: gnomonic
*      SIN: synthesis (generalized orthographic)
*      STG: stereographic
*      ARC: zenithal/azimuthal equidistant
*      ZPN: zenithal/azimuthal polynomial
*      ZEA: zenithal/azimuthal equal area
*      AIR: Airy
*
*   Cylindricals:
*      CYP: cylindrical perspective
*      CAR: Cartesian
*      MER: Mercator
*      CEA: cylindrical equal area
*
*   Conics:
*      COP: conic perspective
*      COD: conic equidistant
*      COE: conic equal area
*      COO: conic orthomorphic
*
*   Polyconics:
*      BON: Bonne
*      PCO: polyconic
*
*   Pseudo-cylindricals:
*      GLS: Sanson-Flamsteed (global sinusoidal)
*      PAR: parabolic
*      MOL: Mollweide
*
*   Conventional:
*      AIT: Hammer-Aitoff
*
*   Quad-cubes:
*      CSC: COBE quadrilateralized spherical cube
*      QSC: quadrilateralized spherical cube
*      TSC: tangential spherical cube
*
*   Author: Mark Calabretta, Australia Telescope National Facility
*   $Id: wcs.c,v 2.0 1995/09/11 08:44:44 mcalabre Exp $
*===========================================================================*/

#include "wcs.h"
#include <stdlib.h>

#ifndef __STDC__
#ifndef const
#define const
#endif
#endif

#ifdef SIGNBIT
#define signbit(X) ((X) < 0.0 ? 0 : 1)
#endif


int wcsfwd(pcode, lng, lat, cel, phi, theta, prj, imgcrd, lin, pixcrd)

char   pcode[4];
double lng, lat;
struct celprm *cel;
double *phi, *theta;
struct prjprm *prj;
double imgcrd[];
struct linprm *lin;
double pixcrd[];

{
   int    err, j;

   /* Compute projected coordinates. */
   if (err = celfwd(pcode, lng, lat, cel, theta, phi, prj,
                    &imgcrd[0], &imgcrd[1])) {
      return err;
   }

   for (j = 2; j < lin->naxis; imgcrd[j++] = 0.0);

   /* Apply forward linear transformation. */
   if (linfwd(imgcrd, lin, pixcrd)) {
      return 4;
   }

   return 0;
}

/*--------------------------------------------------------------------------*/

int wcsrev(pcode, pixcrd, lin, imgcrd, prj, phi, theta, cel, lng, lat)

char   pcode[4];
double pixcrd[];
struct linprm *lin;
double imgcrd[];
struct prjprm *prj;
double *phi, *theta;
struct celprm *cel;
double *lng, *lat;

{
   int    err;

   /* Apply reverse linear transformation. */
   if (linrev(pixcrd, lin, imgcrd)) {
      return 4;
   }

   /* Compute projected coordinates. */
   if (err = celrev(pcode, imgcrd[0], imgcrd[1], prj, phi, theta, cel,
                    lng, lat)) {
      return err;
   }

   return 0;
}

/*--------------------------------------------------------------------------*/

int wcsmix(pcode, mix, vspan, vstep, viter, lng, lat, cel, phi, theta, prj, 
           imgcrd, lin, pixcrd)

char   pcode[4];
int    mix;
double vspan[2], vstep;
int    viter;
double *lng, *lat;
struct celprm *cel;
double *phi, *theta;
struct prjprm *prj;
double imgcrd[];
struct linprm *lin;
double pixcrd[];

{
   const int niter = 60;
   int    crossed, err, istep, iter, nix, retry;
   const double tol = 1.0e-10;
   double lambda, span[2], step;
   double pixnix;
   double lng0, lng0m, lng1, lng1m;
   double lat0, lat0m, lat1, lat1m;
   double d, d0, d0m, d1, d1m, dx;
   double dabs, dmin, lmin;

   /* Check vspan. */
   if (vspan[0] <= vspan[1]) {
      span[0] = vspan[0];
      span[1] = vspan[1];
   } else {
      /* Swap them. */
      span[0] = vspan[1];
      span[1] = vspan[0];
   }

   /* Check vstep. */
   step = fabs(vstep);
   if (step == 0.0) {
      step = (span[1] - span[0])/10.0;
      if (step > 1.0 || step == 0.0) step = 1.0;
   }

   /* Check viter. */
   if (viter < 0) {
      viter = 0;
   } else if (viter > 5) {
      viter = 5;
   }

   /* Given pixel element. */
   nix = abs(mix) - 1;
   pixnix = pixcrd[nix];

   /* Iterate on the step size. */
   for (istep = 0; istep <= viter; istep++) {
      if (istep) step /= 2.0;

      /* Iterate on the sky coordinate between the specified range. */
      if (mix < 0) {
         /* Celestial longitude is given. */

         /* Check whether the solution interval is a crossing interval. */
         lat0 = span[0];
         if (err = wcsfwd(pcode, *lng, lat0, cel, phi, theta, prj, imgcrd,
                          lin, pixcrd)) {
            return err;
         }
         d0 = pixcrd[nix] - pixnix;

         dabs = fabs(d0);
         if (dabs < tol) {
            *lat = lat0;
            return 0;
         }

         lat1 = span[1];
         if (err = wcsfwd(pcode, *lng, lat1, cel, phi, theta, prj, imgcrd,
                          lin, pixcrd)) {
            return err;
         }
         d1 = pixcrd[nix] - pixnix;

         dabs = fabs(d1);
         if (dabs < tol) {
            *lat = lat1;
            return 0;
         }
         lmin = lat1;
         dmin = dabs;

         /* Check for a crossing point. */
         if (signbit(d0) != signbit(d1)) {
            crossed = 1;
            dx = d1;
         } else {
            crossed = 0;
            lat0 = span[1];
         }

         for (retry = 0; retry < 4; retry++) {
            /* Refine the solution interval. */
            while (lat0 > span[0]) {
               lat0 -= step;
               if (lat0 < span[0]) lat0 = span[0];
               if (err = wcsfwd(pcode, *lng, lat0, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                  return err;
               }
               d0 = pixcrd[nix] - pixnix;

               /* Check for a solution. */
               dabs = fabs(d0);
               if (dabs < tol) {
                  *lat = lat0;
                  return 0;
               }

               /* Record the point of closest approach. */
               if (dabs < dmin) {
                  lmin = lat0;
                  dmin = dabs;
               }

               /* Check for a crossing point. */
               if (signbit(d0) != signbit(d1)) {
                  crossed = 2;
                  dx = d0;
                  break;
               }

               /* Advance to the next subinterval. */
               lat1 = lat0;
               d1 = d0;
            }

            if (crossed) {
               /* A crossing point was found. */
               for (iter = 0; iter < niter; iter++) {
                  /* Use regula falsi division of the interval. */
                  lambda = d0/(d0-d1);
                  if (lambda < 0.1) {
                     lambda = 0.1;
                  } else if (lambda > 0.9) {
                     lambda = 0.9;
                  }

                  *lat = lat0 + lambda*(lat1 - lat0);
                  if (err = wcsfwd(pcode, *lng, *lat, cel, phi, theta, prj,
                                   imgcrd, lin, pixcrd)) {
                     return err;
                  }
                  d = pixcrd[nix] - pixnix;

                  /* Check for a solution. */
                  dabs = fabs(d);
                  if (dabs < tol) return 0;

                  /* Record the point of closest approach. */
                  if (dabs < dmin) {
                     lmin = *lat;
                     dmin = dabs;
                  }

                  if (signbit(d0) == signbit(d)) {
                     lat0 = *lat;
                     d0 = d;
                  } else {
                     lat1 = *lat;
                     d1 = d;
                  }
               }

               /* No convergence, must have been a discontinuity. */
               if (crossed == 1) lat0 = span[1];
               lat1 = lat0;
               d1 = dx;
               crossed = 0;

            } else {
               /* No crossing point; look for a tangent point. */
               if (lmin == span[0]) break;
               if (lmin == span[1]) break;

               *lat = lmin;
               lat0 = *lat - step;
               if (lat0 < span[0]) lat0 = span[0];
               lat1 = *lat + step;
               if (lat1 > span[1]) lat1 = span[1];

               if (err = wcsfwd(pcode, *lng, lat0, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                  return err;
               }
               d0 = fabs(pixcrd[nix] - pixnix);

               d  = dmin;
               if (err = wcsfwd(pcode, *lng, lat1, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                  return err;
               }
               d1 = fabs(pixcrd[nix] - pixnix);

               for (iter = 0; iter < niter; iter++) {
                  lat0m = (lat0 + *lat)/2.0;
                  if (err = wcsfwd(pcode, *lng, lat0m, cel, phi, theta, prj,
                                   imgcrd, lin, pixcrd)) {
                     return err;
                  }
                  d0m = fabs(pixcrd[nix] - pixnix);
                  if (d0m < tol) {
                     *lat = lat0m;
                     return 0;
                  }

                  lat1m = (lat1 + *lat)/2.0;
                  if (err = wcsfwd(pcode, *lng, lat1m, cel, phi, theta, prj,
                                   imgcrd, lin, pixcrd)) {
                     return err;
                  }
                  d1m = fabs(pixcrd[nix] - pixnix);
                  if (d1m < tol) {
                     *lat = lat1m;
                     return 0;
                  }

                  if (d0m < d && d0m <= d1m) {
                     lat1 = *lat;
                     d1   = d;
                     *lat = lat0m;
                     d    = d0m;
                  } else if (d1m < d) {
                     lat0 = *lat;
                     d0   = d;
                     *lat = lat1m;
                     d    = d1m;
                  } else {
                     lat0 = lat0m;
                     d0   = d0m;
                     lat1 = lat1m;
                     d1   = d1m;
                  }
               }
            }
         }

      } else {
         /* Celestial latitude is given. */

         /* Check whether the solution interval is a crossing interval. */
         lng0 = span[0];
         if (err = wcsfwd(pcode, lng0, *lat, cel, phi, theta, prj, imgcrd,
                          lin, pixcrd)) {
            return err;
         }
         d0 = pixcrd[nix] - pixnix;

         dabs = fabs(d0);
         if (dabs < tol) {
            *lng = lng0;
            return 0;
         }

         lng1 = span[1];
         if (err = wcsfwd(pcode, lng1, *lat, cel, phi, theta, prj, imgcrd,
                          lin, pixcrd)) {
            return err;
         }
         d1 = pixcrd[nix] - pixnix;

         dabs = fabs(d1);
         if (dabs < tol) {
            *lng = lng1;
            return 0;
         }
         lmin = lng1;
         dmin = dabs;

         /* Check for a crossing point. */
         if (signbit(d0) != signbit(d1)) {
            crossed = 1;
            dx = d1;
         } else {
            crossed = 0;
            lng0 = span[1];
         }

         for (retry = 0; retry < 4; retry++) {
            /* Refine the solution interval. */
            while (lng0 > span[0]) {
               lng0 -= step;
               if (lng0 < span[0]) lng0 = span[0];
               if (err = wcsfwd(pcode, lng0, *lat, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                  return err;
               }
               d0 = pixcrd[nix] - pixnix;

               /* Check for a solution. */
               dabs = fabs(d0);
               if (dabs < tol) {
                  *lng = lng0;
                  return 0;
               }

               /* Record the point of closest approach. */
               if (dabs < dmin) {
                  lmin = lng0;
                  dmin = dabs;
               }

               /* Check for a crossing point. */
               if (signbit(d0) != signbit(d1)) {
                  crossed = 2;
                  dx = d0;
                  break;
               }

               /* Advance to the next subinterval. */
               lng1 = lng0;
               d1 = d0;
            }

            if (crossed) {
               /* A crossing point was found. */
               for (iter = 0; iter < niter; iter++) {
                  /* Use regula falsi division of the interval. */
                  lambda = d0/(d0-d1);
                  if (lambda < 0.1) {
                     lambda = 0.1;
                  } else if (lambda > 0.9) {
                     lambda = 0.9;
                  }

                  *lng = lng0 + lambda*(lng1 - lng0);
                  if (err = wcsfwd(pcode, *lng, *lat, cel, phi, theta, prj,
                                   imgcrd, lin, pixcrd)) {
                     return err;
                  }
                  d = pixcrd[nix] - pixnix;

                  /* Check for a solution. */
                  dabs = fabs(d);
                  if (dabs < tol) return 0;

                  /* Record the point of closest approach. */
                  if (dabs < dmin) {
                     lmin = *lng;
                     dmin = dabs;
                  }

                  if (signbit(d0) == signbit(d)) {
                     lng0 = *lng;
                     d0 = d;
                  } else {
                     lng1 = *lng;
                     d1 = d;
                  }
               }

               /* No convergence, must have been a discontinuity. */
               if (crossed == 1) lng0 = span[1];
               lng1 = lng0;
               d1 = dx;
               crossed = 0;

            } else {
               /* No crossing point; look for a tangent point. */
               if (lmin == span[0]) break;
               if (lmin == span[1]) break;

               *lng = lmin;
               lng0 = *lng - step;
               if (lng0 < span[0]) lng0 = span[0];
               lng1 = *lng + step;
               if (lng1 > span[1]) lng1 = span[1];

               if (err = wcsfwd(pcode, lng0, *lat, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                  return err;
               }
               d0 = fabs(pixcrd[nix] - pixnix);
               d  = dmin;
               if (err = wcsfwd(pcode, lng1, *lat, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                  return err;
               }
               d1 = fabs(pixcrd[nix] - pixnix);

               for (iter = 0; iter < niter; iter++) {
                  lng0m = (lng0 + *lng)/2.0;
                  if (err = wcsfwd(pcode, lng0m, *lat, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                     return err;
                  }
                  d0m = fabs(pixcrd[nix] - pixnix);
                  if (d0m < tol) {
                     *lng = lng0m;
                     return 0;
                  }

                  lng1m = (lng1 + *lng)/2.0;
                  if (err = wcsfwd(pcode, lng1m, *lat, cel, phi, theta, prj,
                                imgcrd, lin, pixcrd)) {
                     return err;
                  }
                  d1m = fabs(pixcrd[nix] - pixnix);
                  if (d1m < tol) {
                     *lng = lng1m;
                     return 0;
                  }

                  if (d0m < d && d0m <= d1m) {
                     lng1 = *lng;
                     d1   = d;
                     *lng = lng0m;
                     d    = d0m;
                  } else if (d1m < d) {
                     lng0 = *lng;
                     d0   = d;
                     *lng = lng1m;
                     d    = d1m;
                  } else {
                     lng0 = lng0m;
                     d0   = d0m;
                     lng1 = lng1m;
                     d1   = d1m;
                  }
               }
            }
         }
      }
   }

   /* No solution. */
   return 4;

}
