/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* machine dependent code for SUN SPARCstation running X windows */ 

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>     /* includes strings.h, sys/time.h, and sys/file.h */

#include <stdio.h>
#include <stdlib.h>
#include "skyview.h"
#include "img.h"
#include "img_x.h"
#include "parse.h"
#include "data_types.h"

typedef unsigned long Pixel;
extern unsigned long black_pixval, white_pixval;
extern Atom wm_delete_window;
extern Visual *visual;
extern int debug, server_mode, s0_server_mode, graphics;
extern char server_str[];
extern int miny, maxy;
extern int lines, pixels;
extern struct img *curr_img;
extern FILE *session, *debug_file;
extern int llow_save, hhigh_save;

static unsigned int mag_height, mag_width;
extern Window mag_win, wedge_win;
extern unsigned char *image_org;
extern unsigned int width_org, height_org;
extern unsigned border_width;   /* border four pixels wide */
extern unsigned long pixval[];
extern int ads_server_mode;

extern Display *display;
extern int screen;
extern unsigned int depth, bytes_per_pixel;

extern GC image_gc;
extern GC gcxor;
extern GC wedge_gc;
extern Pixmap icon_pixmap;

static XImage *magImage;
static int mag_on;
static unsigned char *mag_image_data;
static int mag_zoom_factor = 8;


/*  -- everything below here appears both globally and in image structure -- */
extern XImage *theImage;
extern Window win;
extern unsigned int win_width, win_height;
extern unsigned char *image_zoomed;
extern unsigned int width_zoomed, height_zoomed;
extern double win_zoom;
extern int win_x_offset, win_y_offset;


/* function prototypes */
#ifdef _NO_PROTO

static void create_mag_window();
static void zoom_smaller();
static void zoom_smaller16();
static void zoom_smaller32();
static void zoom_bigger();
static void zoom_bigger16();
static void zoom_bigger32();

#else

static void create_mag_window(void);
static void zoom_smaller(unsigned char *inptr, unsigned char *outptr, 
    double zoom,
    unsigned int in_width, unsigned int in_height, 
    unsigned int out_width, unsigned int out_height, 
    int ixoffset, int iyoffset);
static void zoom_smaller16(INT16 *inptr, INT16 *outptr, double zoom,
    unsigned int in_width, unsigned int in_height, 
    unsigned int out_width, unsigned int out_height, 
    int ixoffset, int iyoffset);
static void zoom_smaller32(int *inptr, int *outptr, double zoom,
    unsigned int in_width, unsigned int in_height, 
    unsigned int out_width, unsigned int out_height, 
    int ixoffset, int iyoffset);
static void zoom_bigger(unsigned char *inptr, unsigned char *outptr, int zoom,
    int in_width, int in_height, 
    int out_width, int out_height, 
    int ixoffset, int iyoffset);
static void zoom_bigger16(INT16 *inptr, INT16 *outptr, int zoom,
    int in_width, int in_height, 
    int out_width, int out_height, 
    int ixoffset, int iyoffset);
static void zoom_bigger32(int *inptr, int *outptr, int zoom,
    int in_width, int in_height, 
    int out_width, int out_height, 
    int ixoffset, int iyoffset);

#endif /* _NO_PROTO */

void
mag_cmd(argc, argv)
int argc;
char *argv[];
{
    if (!graphics)
	return;

    if (argc == 1)
    {
	if(server_mode == FALSE)
	    printf("magnifier %s;  zoom factor = %d\n", 
		mag_on?"on":"off", mag_zoom_factor);
    }
    else
    {
	if (strncmp(argv[1], "off", 2) == 0)
	{
	    mag_on = FALSE;
	    mag_destroy();
	}
	else if (strcmp(argv[1], "on") == 0)
	{
	    mag_on = TRUE;
	    create_mag_window();
	}
	else
	{
	    getint(argv[1], &mag_zoom_factor);
	}
	if (argc == 3)
	    getint(argv[2], &mag_zoom_factor);
    }

    if(server_mode == TRUE)
    {
	sprintf(server_str, "%s", mag_on?"on":"off");
        srv_string("mag_status", server_str);
        sprintf(server_str, "%d", mag_zoom_factor);
        srv_real("mag_zoom_factor", server_str);
    }
}


static void
create_mag_window()
{
    XSizeHints size_hints;
    int x, y;            /* window position */
    char *window_name;
    char *icon_name;
    int cmdc;
    char **cmdv;

    if (mag_win != 0)
    {
	mag_expose();
	return;
    }
    window_name = "magnifier";
    icon_name = window_name;
    mag_width = 150;
    mag_height = 150;
    x = DisplayWidth(display, screen) - mag_width - 2 * border_width;
    y = 0;


    mag_win = new_window(x, y, mag_width, mag_height);

    /* initialize size hint property for window manager */
    size_hints.flags = USPosition | USSize | PMinSize;
    size_hints.x = x;
    size_hints.y = y;
    size_hints.width = mag_width;
    size_hints.height = mag_height;
    size_hints.min_width = 50;
    size_hints.min_height = 50;
    cmdc = 0;
    cmdv = NULL;

    /* set Properties for window manager (always before mapping) */
    XSetStandardProperties(display, mag_win, window_name, icon_name,
	icon_pixmap, cmdv, cmdc, &size_hints);

#ifdef WithdrawnState
    (void) XSetWMProtocols (display, mag_win, &wm_delete_window, 1);
#endif /* WithdrawnState */

    /* Select event types wanted */
    XSelectInput(display, mag_win, KeyPressMask | ExposureMask |
	ButtonPressMask | StructureNotifyMask);

    /* get buffer for pixels */
    mag_image_data = (unsigned char *) calloc(mag_width* mag_height, 
	bytes_per_pixel);

    if (!mag_image_data)
    {
	error1("magnifier error: unable to allocate space for pixels");
	return;
    }
    magImage = XCreateImage(display, DefaultVisual(display, screen), 
	depth, ZPixmap, 0, (char *) mag_image_data,
	mag_width, mag_height, 8, 0);

    if (!magImage)
    {
	error1("magnifier error: unable to create Ximage");
	free(mag_image_data);
	return;
    }

    /* Display window */
    XMapWindow(display, mag_win);
}


void
mag_expose()
{
}

void
mag_motion(wind, x, y)
Window wind;
int x, y;
{
    int xc, yc;
    struct img *tmpimg;
    int i_coord[2];
    double mag_zoom;
    int mag_x_offset, mag_y_offset;

    if (debug)
	fprintf(debug_file, "mag_motion:  wind = %ld  x = %d  y = %d\n", wind, x, y);

    if (wind != win)
    {
	tmpimg = find_WIN(wind);
	if (tmpimg == NULL)
	    return;
    }

    mag_zoom = mag_zoom_factor * win_zoom;

    /* convert X window x,y to our SC coordinates */
    i_coord[0] = x;
    i_coord[1] = y;
    WIN_to_SC(i_coord);
    xc = i_coord[0];
    yc = i_coord[1];

    /* find new screen origin using new zoom factor */
    mag_x_offset = xc - (int) ((mag_width / 2) / mag_zoom);
    mag_y_offset = maxy - miny - yc - (int) ((mag_height / 2) / mag_zoom);

    if (debug)
    {
	fprintf(debug_file, 
	    "mag_motion:  window id %lx   zoom factor %g\n", win, mag_zoom);
	fprintf(debug_file, 
	    "mag_motion:  mag_x_offset = %d  mag_y_offset = %d\n",
	    mag_x_offset, mag_y_offset);
    }

    if (mag_zoom >= 1.0)
    {
    if (depth > 16)
    zoom_bigger32( (int *) image_org, (int *) mag_image_data, (int) mag_zoom, 
	(int) width_org, (int) height_org, (int) mag_width, (int) mag_height, 
	mag_x_offset, mag_y_offset);
    else if (depth > 8)
    zoom_bigger16( (INT16 *) image_org, (INT16 *) mag_image_data, (int) mag_zoom, 
	(int) width_org, (int) height_org, (int) mag_width, (int) mag_height, 
	mag_x_offset, mag_y_offset);
    else
    zoom_bigger(image_org, mag_image_data, (int) mag_zoom, (int) width_org, 
	(int) height_org, (int) mag_width, (int) mag_height, mag_x_offset, mag_y_offset);
    }
    else
    {
    if (depth > 16)
    zoom_smaller32( (int *)image_org, (int *) mag_image_data,  mag_zoom, 
	width_org, height_org, mag_width, mag_height, 
	mag_x_offset, mag_y_offset);
    else if (depth > 8)
    zoom_smaller16( (INT16 *)image_org, (INT16 *) mag_image_data,  mag_zoom, 
	width_org, height_org, mag_width, mag_height, 
	mag_x_offset, mag_y_offset);
    else
    zoom_smaller(image_org, mag_image_data,  mag_zoom, width_org, 
	height_org, mag_width, mag_height, mag_x_offset, mag_y_offset);
    }
    XPutImage(display, mag_win, image_gc, magImage, 0,0,0,0,
	magImage->width, magImage->height);

#ifdef UNDEF
#define CROSSLEN 5
    cross_x = (int)((mag_width / 2) / mag_zoom) * mag_zoom + mag_zoom / 2;
    cross_y = (int)((mag_height / 2) / mag_zoom) * mag_zoom + mag_zoom / 2;
    XDrawLine(display, mag_win, gcxor, cross_x, cross_y-CROSSLEN, 
	cross_x, cross_y+CROSSLEN);
    XDrawLine(display, mag_win, gcxor, cross_x-CROSSLEN, cross_y, 
	cross_x+CROSSLEN, cross_y);
#endif /* UNDEF */
}

void
mag_destroy()
{
    if (mag_win != 0)
    {
	XDestroyWindow(display, mag_win);
	mag_win = 0;
    }
}

void
mag_configure(width, height)
unsigned int width, height;
{
    if ((mag_width != width) || (mag_height != height))
    {
	mag_width = width;
	mag_height = height;

	if (mag_image_data != NULL)
	    free(mag_image_data);

	/* get buffer for pixels */
	mag_image_data = (unsigned char *) calloc(mag_width* mag_height, 
	    bytes_per_pixel);
	if (!mag_image_data)
	{
	    error1("magnifier error: unable to allocate space for pixels");
	    return;
	}
	magImage->data = (char *) mag_image_data;
	magImage->width = mag_width;
	magImage->height = mag_height;
	magImage->bytes_per_line = mag_width * bytes_per_pixel;
    }
}

void
initial_zoom()
{
}

void
trackball_zoom()
{
    unsigned char *pan_image_org;
    XImage *panImage;
    Pixel back, border;
    int x, y, box_height, box_width;
    int done, xc, yc;
    double tmpzoom, pan_zoom;
    int pan_x = 0, pan_y = 0;
    char anin[MAXLINE];
    Window pan_win;
    unsigned int pan_width, pan_height, pan_zoom_inverse;
    int dummy_dx, dummy_dy;
    int what;
    Window root, child;
    int root_x, root_y;
    unsigned int keys_buttons;

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    pan_zoom_inverse = height_org / 150 + 1;
    if (pan_zoom_inverse > width_org / 150 + 1)
	pan_zoom_inverse = width_org / 150 + 1;
    pan_zoom = 1 / (double) pan_zoom_inverse;
    pan_width = width_org / pan_zoom_inverse;
    pan_height = height_org / pan_zoom_inverse;

    back = white_pixval;
    border = black_pixval;

    pan_win = XCreateSimpleWindow(display, win,
	pan_x, pan_y, pan_width, pan_height, border_width,
	border, back);

    
    /* Select event types wanted */
    XSelectInput(display, pan_win, KeyPressMask | ExposureMask |
	ButtonPressMask | StructureNotifyMask);

    /* Display window */
    XMapWindow(display, pan_win);


    /* get space for pan image */
    pan_image_org = (unsigned char *) calloc(pan_width * pan_height, 
	bytes_per_pixel);
    if (!pan_image_org)
    {
	error1("error: unable to allocate space for zoom pixels");
	return;
    }
    panImage = XCreateImage(display, visual, depth, ZPixmap, 0, 
	(char *) pan_image_org, pan_width, pan_height, 8, 0);
    if (!panImage)
    {
	error1("error: unable to create zoom Ximage");
	free(pan_image_org);
	return;
    }

    /* subsample the image */
    if (depth > 16)
    zoom_smaller32( (int *) image_org, (int *) pan_image_org, pan_zoom, 
	width_org, height_org, pan_width, pan_height, 0, 0);
    else if (depth > 8)
    zoom_smaller16( (INT16 *) image_org, (INT16 *) pan_image_org, pan_zoom, 
	width_org, height_org, pan_width, pan_height, 0, 0);
    else
    zoom_smaller(image_org, pan_image_org, pan_zoom, 
	width_org, height_org, pan_width, pan_height, 0, 0);

    tmpzoom = win_zoom;
    box_width = win_width / (pan_zoom_inverse * tmpzoom);
    box_height = win_height / (pan_zoom_inverse *tmpzoom);
    /* get current location of pointer */
    XQueryPointer(display, pan_win,
	&root, &child, &root_x, &root_y, 
	&xc, &yc, &keys_buttons);
    XPutImage(display, pan_win, image_gc, panImage, 0,0,0,0,
	panImage->width, panImage->height);

    x = xc;   /* keep purify happy */
    y = yc;   /* keep purify happy */

    set_motion_notify(pan_win);
    done = 0;
    while(!done)
    {
	draw_box(pan_win, gcxor, xc, yc, box_width, box_height);

	wait_for_event(anin, &what, &x, &y, &dummy_dx, &dummy_dy, 0);

	/* undraw old line */
	draw_box(pan_win, gcxor, xc, yc, box_width, box_height);

	switch(what)
	{
	case Button1:
	    if (tmpzoom >= 1.0)
		tmpzoom = tmpzoom + 1.0;
	    else
		tmpzoom = tmpzoom * 2;
	    if(server_mode == FALSE)
		printf("Zoom factor %f\n", tmpzoom);
	    box_width = win_width / (pan_zoom_inverse * tmpzoom);
	    box_height = win_height / (pan_zoom_inverse *tmpzoom);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    if (tmpzoom > 1.0)
		tmpzoom = tmpzoom - 1.0;
	    else
		tmpzoom = tmpzoom / 2;
	    if(server_mode == FALSE)
		printf("Zoom factor %f\n", tmpzoom);
	    box_width = win_width / (pan_zoom_inverse * tmpzoom);
	    box_height = win_height / (pan_zoom_inverse *tmpzoom);
	    break;
	case POINTERMOTION:
	    if (debug)
		fprintf(debug_file, "zoom_pan:  x = %d  y = %d\n", x, y);
	    xc = x;
	    yc = y;
	    break;
	}
    }
    clear_motion_notify();
    XDestroyWindow(display, pan_win);
    xc = (xc * (int)width_org) / (int) pan_width;
    yc = (((int)pan_height - yc) * (int)height_org) / (int) pan_height;
    if (debug)
	fprintf(debug_file, "zoom_pan:  xc = %d  yc = %d\n", xc, yc);

    do_zoom_pan(4, &tmpzoom, &xc, &yc, 1);

}

void
do_zoom_pan(argc, tmpzoom, xc, yc, do_expose)
int argc, *xc, *yc;
int do_expose;   /* 1: queue up an expose event,    0: don't    */
double *tmpzoom;
{
    int i_coord[2];
    XEvent report;

	/* argc == 1 is a special case - do not change win_zoom */
	/*   or win_x_offset or win_y_offset.                   */
    if (argc != 1)
    {
	if (argc == 2)    /* if he specified only the zoom factor */
	{
	    /* get old center x,y  using old zoom factor */
	    i_coord[0] = win_width / 2;
	    i_coord[1] = win_height / 2;
	    WIN_to_SC(i_coord);
	    *xc = i_coord[0];
	    *yc = i_coord[1];
	}
	if (argc != 3)    /* if he specified a new zoom */
	{
	    win_zoom = *tmpzoom;
	}
	else
	{
	    *tmpzoom = win_zoom;
	}

	/* find new screen origin using new zoom factor */
	win_x_offset = Round(*xc - (win_width / 2 + 0.5) / win_zoom + 0.5);
	win_y_offset = Round(height_org - 1.0 - *yc + 0.5 
	    - (win_height / 2 + 0.5) / win_zoom);
    }

    if (debug)
    {
	fprintf(debug_file, 
	    "do_zoom_pan:  window id 0x%lx   zoom factor %f\n", win, win_zoom);
	fprintf(debug_file, 
	    "do_zoom_pan:  win_x_offset = %d  win_y_offset = %d\n",
	    win_x_offset, win_y_offset);
    }
    width_zoomed = win_width;
    height_zoomed = win_height;
    if (image_zoomed != NULL)
	free(image_zoomed);
    image_zoomed = (unsigned char *) calloc(width_zoomed * height_zoomed, 
	bytes_per_pixel);
    if (image_zoomed == NULL)
    {
	/* we're in trouble - fall back to unzoomed image */
	win_zoom = 1.0;
	win_x_offset = 0;
	win_y_offset = 0;
	theImage->data = (char *) image_org;
	theImage->width = width_org;
	theImage->height = height_org;
	theImage->bytes_per_line = width_org * bytes_per_pixel;

	/* fix return values */
	*tmpzoom = 1.0;
	get_center_xy(i_coord);
	*xc = i_coord[0];
	*yc = i_coord[1];

	error1("zoom_pan error:  unable to allocate memory for zoomed image");
    }
    else
    {
    fill_img(curr_img); /* force stuff into img struct */
    if (win_zoom >= 1.0)
    {
    if (depth > 16)
    zoom_bigger32( (int *)image_org, (int *) image_zoomed, (int) win_zoom, 
	(int) width_org, (int) height_org, (int) width_zoomed, (int) height_zoomed, 
	win_x_offset, win_y_offset);
    else if (depth > 8)
    zoom_bigger16( (INT16 *)image_org, (INT16 *) image_zoomed, (int) win_zoom, 
	(int) width_org, (int) height_org, (int) width_zoomed, (int) height_zoomed, 
	win_x_offset, win_y_offset);
    else
    zoom_bigger(image_org, image_zoomed, (int) win_zoom, (int) width_org, 
	(int) height_org, (int) width_zoomed, (int) height_zoomed, win_x_offset, win_y_offset);
    }
    else
    {
    if (depth > 16)
    zoom_smaller32( (int *) image_org, (int *) image_zoomed,  win_zoom, 
	width_org, height_org, width_zoomed, height_zoomed, 
	win_x_offset, win_y_offset);
    else if (depth > 8)
    zoom_smaller16( (INT16 *) image_org, (INT16 *) image_zoomed,  win_zoom, 
	width_org, height_org, width_zoomed, height_zoomed, 
	win_x_offset, win_y_offset);
    else
    zoom_smaller(image_org, image_zoomed,  win_zoom, width_org, 
	height_org, width_zoomed, height_zoomed, win_x_offset, win_y_offset);
    }
    theImage->data = (char *) image_zoomed;
    theImage->width = width_zoomed;
    theImage->height = height_zoomed;
    theImage->bytes_per_line = width_zoomed * bytes_per_pixel;
    }

    if (do_expose && graphics)
    {
	/* now queue up a fake expose event (instead of doing XPutImage */
	report.type = Expose;
	report.xexpose.window = win;
	report.xexpose.x = 0;
	report.xexpose.y = 0;
	report.xexpose.width = width_zoomed;
	report.xexpose.height = height_zoomed;
	report.xexpose.count = 0;
	XSendEvent(display, win, False, ExposureMask, &report);
    }
}

static
void zoom_bigger(inptr, outptr, zoom, in_width, in_height, 
    out_width, out_height, ixoffset, iyoffset)
unsigned char *inptr;	     /* source of pixels */
register unsigned char *outptr;  /* destination */
int zoom;		/* zoom factor */
int in_height;	/* input height */
int in_width;		/* input width */
int out_width, out_height;	/* output width and height */
int ixoffset, iyoffset; /* position on input image corresponding to 0,0 */
			/* on output image */
{
#define DEBUG_ZOOM 1
    register unsigned char *p;
    register int count;
    register unsigned char *lineend; /* pointer to end of a line */
    unsigned char *inputend;	/* pointer to end of input buffer */
    unsigned char *outputend;	/* pointer to end of output buffer */
    unsigned char *thisline;	/* pointer to line to replicate */
    unsigned char *nextline;	/* pointer to end of replicate area */
    int i_pixels;  	        /* pixels to use from input line to avoid */
				/* overflow of output line */
    int o_zeros;  	     /* number of leading zeros on each output line */
    int clip_right_flag;	/* TRUE if pixel replication must stop */
    				/* before input line is used up */

    inputend = inptr + in_width * in_height;
    outputend = outptr + out_width * out_height;

    /* handle y offset first */
    if (iyoffset < 0)
    {
	/* fill first blank lines with zeros */
	lineend = outptr - (out_width * iyoffset * zoom);
	while ((outptr < lineend) && (outptr < outputend))
	    *outptr++ = pixval[0];
    }
    else
    {
	/* start partway into input image */
	inptr += in_width * iyoffset;
    }

    /* now handle x offset */
    clip_right_flag = FALSE;
    if (ixoffset < 0)
    {
	/* each line needs to have leading zeros */
	o_zeros = -(zoom * ixoffset);   /* number of leading zeros */
	if (o_zeros + in_width * zoom > out_width)  /* if overflows output line */
	{
	    i_pixels = (out_width - o_zeros) / zoom;
	    clip_right_flag = TRUE;
	    if (o_zeros > out_width)
		inptr = inputend;   /* flag that we are completely off screen */
	}
	else
	    i_pixels = in_width;
    }
    else
    {
	/* need to start partway into each input line */
	o_zeros = 0;
	i_pixels = in_width - ixoffset;
	if (i_pixels <= 0)
	    inptr = inputend;   /* flag that we are completely off screen */
	else
	{
	    inptr += ixoffset;
	    if (i_pixels * zoom > out_width)
	    {
		i_pixels = out_width / zoom;
		clip_right_flag = TRUE;
	    }
	}
    }


    /* outer loop */
    while ((outptr < outputend ) && (inptr < inputend))
    {
	/* remember start of this line */
	thisline = outptr;

	/* replicate the first line */

	/* fill in leading zeros, if any */
	lineend = thisline + o_zeros;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

	/* do the pixel replication */
	p = inptr;
	lineend = inptr + i_pixels;
	inptr += in_width;    /* ready for the next loop iteration */
	while(p < lineend )
	{
	    count = 0;
	    while( count++ < zoom )
		*outptr++ = *p;
	    p++;
	};

	/* finish the line */
	lineend = thisline + out_width;
	if (clip_right_flag)
	{
	    /* replicate last pixel (less than zoom times) */
	    while (outptr < lineend)
		*outptr++ = *p;
	}
	else
	{
	    /* fill in trailing zeros */
	    while (outptr < lineend)
		*outptr++ = pixval[0];    /* pixel value for color zero */
	}

#ifdef DEBUG_ZOOM
	if (p > inputend)    
	{
	    printf("zoom_bigger error: p too big\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
	if (outptr > outputend)    
	{
	    printf("zoom_bigger error: outptr too big (case 1)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

	/* now rubber stamp the line for the replication in y */
	p = thisline;				/* source */
	outptr = thisline + out_width;		/* destination */
	nextline = thisline + zoom * out_width;	/* limit */
	if (nextline > outputend)
	    nextline = outputend;

	while( outptr < nextline ) 
	    *outptr++ = *p++;

#ifdef DEBUG_ZOOM
	if (outptr > outputend)    
	{
	    printf("zoom_bigger error: outptr too big (case 2)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

    }
    /* fill rest of buffer (if any) with zeros */
    while (outptr < outputend)
	*outptr++ = pixval[0];    /* pixel value for color zero */
}

/* zoom_bigger16 is same as zoom_bigger, but does 16bit pixels */

static
void zoom_bigger16(inptr, outptr, zoom, in_width, in_height, 
    out_width, out_height, ixoffset, iyoffset)
INT16 *inptr;	     /* source of pixels */
INT16 *outptr;  /* destination */
int zoom;		/* zoom factor */
int in_height;	/* input height */
int in_width;		/* input width */
int out_width, out_height;	/* output width and height */
int ixoffset, iyoffset; /* position on input image corresponding to 0,0 */
			/* on output image */
{
#define DEBUG_ZOOM 1
    register INT16 *p;
    register int count;
    register INT16 *lineend; /* pointer to end of a line */
    INT16 *inputend;	/* pointer to end of input buffer */
    INT16 *outputend;	/* pointer to end of output buffer */
    INT16 *thisline;	/* pointer to line to replicate */
    INT16 *nextline;	/* pointer to end of replicate area */
    int i_pixels;  	        /* pixels to use from input line to avoid */
				/* overflow of output line */
    int o_zeros;  	     /* number of leading zeros on each output line */
    int clip_right_flag;	/* TRUE if pixel replication must stop */
    				/* before input line is used up */

    inputend = inptr + in_width * in_height;
    outputend = outptr + out_width * out_height;

    /* handle y offset first */
    if (iyoffset < 0)
    {
	/* fill first blank lines with zeros */
	lineend = outptr - (out_width * iyoffset * zoom);
	while ((outptr < lineend) && (outptr < outputend))
	    *outptr++ = pixval[0];
    }
    else
    {
	/* start partway into input image */
	inptr += in_width * iyoffset;
    }

    /* now handle x offset */
    clip_right_flag = FALSE;
    if (ixoffset < 0)
    {
	/* each line needs to have leading zeros */
	o_zeros = -(zoom * ixoffset);   /* number of leading zeros */
	if (o_zeros + in_width * zoom > out_width)  /* if overflows output line */
	{
	    i_pixels = (out_width - o_zeros) / zoom;
	    clip_right_flag = TRUE;
	    if (o_zeros > out_width)
		inptr = inputend;   /* flag that we are completely off screen */
	}
	else
	    i_pixels = in_width;
    }
    else
    {
	/* need to start partway into each input line */
	o_zeros = 0;
	i_pixels = in_width - ixoffset;
	if (i_pixels <= 0)
	    inptr = inputend;   /* flag that we are completely off screen */
	else
	{
	    inptr += ixoffset;
	    if (i_pixels * zoom > out_width)
	    {
		i_pixels = out_width / zoom;
		clip_right_flag = TRUE;
	    }
	}
    }


    /* outer loop */
    while ((outptr < outputend ) && (inptr < inputend))
    {
	/* remember start of this line */
	thisline = outptr;

	/* replicate the first line */

	/* fill in leading zeros, if any */
	lineend = thisline + o_zeros;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

	/* do the pixel replication */
	p = inptr;
	lineend = inptr + i_pixels;
	inptr += in_width;    /* ready for the next loop iteration */
	while(p < lineend )
	{
	    count = 0;
	    while( count++ < zoom )
		*outptr++ = *p;
	    p++;
	};

	/* finish the line */
	lineend = thisline + out_width;
	if (clip_right_flag)
	{
	    /* replicate last pixel (less than zoom times) */
	    while (outptr < lineend)
		*outptr++ = *p;
	}
	else
	{
	    /* fill in trailing zeros */
	    while (outptr < lineend)
		*outptr++ = pixval[0];    /* pixel value for color zero */
	}

#ifdef DEBUG_ZOOM
	if (p > inputend)    
	{
	    printf("zoom_bigger error: p too big\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
	if (outptr > outputend)    
	{
	    printf("zoom_bigger error: outptr too big (case 1)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

	/* now rubber stamp the line for the replication in y */
	p = thisline;				/* source */
	outptr = thisline + out_width;		/* destination */
	nextline = thisline + zoom * out_width;	/* limit */
	if (nextline > outputend)
	    nextline = outputend;

	while( outptr < nextline ) 
	    *outptr++ = *p++;

#ifdef DEBUG_ZOOM
	if (outptr > outputend)    
	{
	    printf("zoom_bigger error: outptr too big (case 2)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

    }
    /* fill rest of buffer (if any) with zeros */
    while (outptr < outputend)
	*outptr++ = pixval[0];    /* pixel value for color zero */
}

/* zoom_bigger32 is same as zoom_bigger, but does 32bit pixels */

static
void zoom_bigger32(inptr, outptr, zoom, in_width, in_height, 
    out_width, out_height, ixoffset, iyoffset)
int *inptr;	     /* source of pixels */
register int *outptr;  /* destination */
int zoom;		/* zoom factor */
int in_height;	/* input height */
int in_width;		/* input width */
int out_width, out_height;	/* output width and height */
int ixoffset, iyoffset; /* position on input image corresponding to 0,0 */
			/* on output image */
{
#define DEBUG_ZOOM 1
    register int *p;
    register int count;
    register int *lineend; /* pointer to end of a line */
    int *inputend;	/* pointer to end of input buffer */
    int *outputend;	/* pointer to end of output buffer */
    int *thisline;	/* pointer to line to replicate */
    int *nextline;	/* pointer to end of replicate area */
    int i_pixels;  	        /* pixels to use from input line to avoid */
				/* overflow of output line */
    int o_zeros;  	     /* number of leading zeros on each output line */
    int clip_right_flag;	/* TRUE if pixel replication must stop */
    				/* before input line is used up */

    inputend = inptr + in_width * in_height;
    outputend = outptr + out_width * out_height;

    /* handle y offset first */
    if (iyoffset < 0)
    {
	/* fill first blank lines with zeros */
	lineend = outptr - (out_width * iyoffset * zoom);
	while ((outptr < lineend) && (outptr < outputend))
	    *outptr++ = pixval[0];
    }
    else
    {
	/* start partway into input image */
	inptr += in_width * iyoffset;
    }

    /* now handle x offset */
    clip_right_flag = FALSE;
    if (ixoffset < 0)
    {
	/* each line needs to have leading zeros */
	o_zeros = -(zoom * ixoffset);   /* number of leading zeros */
	if (o_zeros + in_width * zoom > out_width)  /* if overflows output line */
	{
	    i_pixels = (out_width - o_zeros) / zoom;
	    clip_right_flag = TRUE;
	    if (o_zeros > out_width)
		inptr = inputend;   /* flag that we are completely off screen */
	}
	else
	    i_pixels = in_width;
    }
    else
    {
	/* need to start partway into each input line */
	o_zeros = 0;
	i_pixels = in_width - ixoffset;
	if (i_pixels <= 0)
	    inptr = inputend;   /* flag that we are completely off screen */
	else
	{
	    inptr += ixoffset;
	    if (i_pixels * zoom > out_width)
	    {
		i_pixels = out_width / zoom;
		clip_right_flag = TRUE;
	    }
	}
    }


    /* outer loop */
    while ((outptr < outputend ) && (inptr < inputend))
    {
	/* remember start of this line */
	thisline = outptr;

	/* replicate the first line */

	/* fill in leading zeros, if any */
	lineend = thisline + o_zeros;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

	/* do the pixel replication */
	p = inptr;
	lineend = inptr + i_pixels;
	inptr += in_width;    /* ready for the next loop iteration */
	while(p < lineend )
	{
	    count = 0;
	    while( count++ < zoom )
		*outptr++ = *p;
	    p++;
	};

	/* finish the line */
	lineend = thisline + out_width;
	if (clip_right_flag)
	{
	    /* replicate last pixel (less than zoom times) */
	    while (outptr < lineend)
		*outptr++ = *p;
	}
	else
	{
	    /* fill in trailing zeros */
	    while (outptr < lineend)
		*outptr++ = pixval[0];    /* pixel value for color zero */
	}

#ifdef DEBUG_ZOOM
	if (p > inputend)    
	{
	    printf("zoom_bigger error: p too big\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
	if (outptr > outputend)    
	{
	    printf("zoom_bigger error: outptr too big (case 1)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

	/* now rubber stamp the line for the replication in y */
	p = thisline;				/* source */
	outptr = thisline + out_width;		/* destination */
	nextline = thisline + zoom * out_width;	/* limit */
	if (nextline > outputend)
	    nextline = outputend;

	while( outptr < nextline ) 
	    *outptr++ = *p++;

#ifdef DEBUG_ZOOM
	if (outptr > outputend)    
	{
	    printf("zoom_bigger error: outptr too big (case 2)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

    }
    /* fill rest of buffer (if any) with zeros */
    while (outptr < outputend)
	*outptr++ = pixval[0];    /* pixel value for color zero */
}


static
void zoom_smaller(inptr, outptr, zoom, in_width, in_height, 
    out_width, out_height, ixoffset, iyoffset)
unsigned char *inptr;		/* source of pixels */
register unsigned char *outptr;	/* destination */
double zoom;		       /* zoom factor */
unsigned int in_height;		/* input height */
unsigned int in_width;		/* input width */
unsigned int out_width, out_height;	/* output width and height */
int ixoffset, iyoffset; /* position on input image corresponding to 0,0 */
			/* on output image */
{
#define DEBUG_ZOOM 1
    register unsigned char *p;
    register int count;
    register unsigned char *lineend;	/* pointer to end of a line */
    unsigned char *inputend;		/* pointer to end of input buffer */
    unsigned char *outputend;		/* pointer to end of output buffer */
    unsigned char *thisline;	/* pointer to line to replicate */
    int i_pixels;  	        /* pixels to use from input line to avoid */
				/* overflow of output line */
    int o_zeros;  	     /* number of leading zeros on each output line */

    inputend = inptr + in_width * in_height;
    outputend = outptr + out_width * out_height;

    /* handle y offset first */
    if (iyoffset < 0)
    {
	/* fill first rows with zeros */
	lineend = outptr - out_width * (int) (iyoffset * zoom);
	while ((outptr < lineend) && (outptr < outputend))
	    *outptr++ = pixval[0];
    }
    else
    {
	/* start partway into input image */
	inptr += in_width * iyoffset;
    }

    /* now handle x offset */
    if (ixoffset < 0)
    {
	/* each line needs to have leading zeros */
	o_zeros = -(zoom * ixoffset);   /* number of leading zeros */
	if (o_zeros + (int) (in_width * zoom) > out_width)  /* if overflows output line */
	{
	    i_pixels = (out_width - o_zeros) / zoom;
	    if (o_zeros > out_width)
		inptr = inputend;   /* flag that we are completely off screen */
	}
	else
	    i_pixels = in_width;
    }
    else
    {
	/* need to start partway into each input line */
	o_zeros = 0;
	i_pixels = in_width - ixoffset;
	if (i_pixels <= 0)
	    inptr = inputend;   /* flag that we are completely off screen */
	else
	{
	    inptr += ixoffset;
	    if (i_pixels * zoom > out_width)
		i_pixels = out_width / zoom;
	}
    }


    /* quit when output buffer is full */
    while ((outptr < outputend ) && (inptr < inputend))
    {
	/* remember start of this line */
	thisline = outptr;

	/* make one line with decimation */

	/* fill in leading zeros, if any */
	lineend = thisline + o_zeros;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

	/* do the pixel decimation */
	p = inptr;
	lineend = inptr + i_pixels;
	count = Round(1 / zoom);
	inptr += in_width * count;    /* ready for the next loop iteration */
	while(p < lineend )
	{
	    *outptr++ = *p;
	    p += count;
	};

	/* fill in trailing zeros, if any */
	lineend = thisline + out_width;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

#ifdef DEBUG_ZOOM
	if (outptr > outputend)    
	{
	    printf("zoom_smaller error: outptr too big (case 1)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

    }
    /* fill rest of buffer (if any) with zeros */
    while (outptr < outputend)
	*outptr++ = pixval[0];    /* pixel value for color zero */
}

/* zoom_smaller16 is same as zoom_smaller, but does 16bit pixels */

static
void zoom_smaller16(inptr, outptr, zoom, in_width, in_height, 
    out_width, out_height, ixoffset, iyoffset)
INT16 *inptr;		/* source of pixels */
register INT16 *outptr;	/* destination */
double zoom;		       /* zoom factor */
unsigned int in_height;		/* input height */
unsigned int in_width;		/* input width */
unsigned int out_width, out_height;	/* output width and height */
int ixoffset, iyoffset; /* position on input image corresponding to 0,0 */
			/* on output image */
{
#define DEBUG_ZOOM 1
    register INT16 *p;
    register int count;
    register INT16 *lineend;	/* pointer to end of a line */
    INT16 *inputend;		/* pointer to end of input buffer */
    INT16 *outputend;		/* pointer to end of output buffer */
    INT16 *thisline;	/* pointer to line to replicate */
    int i_pixels;  	        /* pixels to use from input line to avoid */
				/* overflow of output line */
    int o_zeros;  	     /* number of leading zeros on each output line */

    inputend = inptr + in_width * in_height;
    outputend = outptr + out_width * out_height;

    /* handle y offset first */
    if (iyoffset < 0)
    {
	/* fill first rows with zeros */
	lineend = outptr - out_width * (int) (iyoffset * zoom);
	while ((outptr < lineend) && (outptr < outputend))
	    *outptr++ = pixval[0];
    }
    else
    {
	/* start partway into input image */
	inptr += in_width * iyoffset;
    }

    /* now handle x offset */
    if (ixoffset < 0)
    {
	/* each line needs to have leading zeros */
	o_zeros = -(zoom * ixoffset);   /* number of leading zeros */
	if (o_zeros + (int) (in_width * zoom) > out_width)  /* if overflows output line */
	{
	    i_pixels = (out_width - o_zeros) / zoom;
	    if (o_zeros > out_width)
		inptr = inputend;   /* flag that we are completely off screen */
	}
	else
	    i_pixels = in_width;
    }
    else
    {
	/* need to start partway into each input line */
	o_zeros = 0;
	i_pixels = in_width - ixoffset;
	if (i_pixels <= 0)
	    inptr = inputend;   /* flag that we are completely off screen */
	else
	{
	    inptr += ixoffset;
	    if (i_pixels * zoom > out_width)
		i_pixels = out_width / zoom;
	}
    }


    /* quit when output buffer is full */
    while ((outptr < outputend ) && (inptr < inputend))
    {
	/* remember start of this line */
	thisline = outptr;

	/* make one line with decimation */

	/* fill in leading zeros, if any */
	lineend = thisline + o_zeros;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

	/* do the pixel decimation */
	p = inptr;
	lineend = inptr + i_pixels;
	count = Round(1 / zoom);
	inptr += in_width * count;    /* ready for the next loop iteration */
	while(p < lineend )
	{
	    *outptr++ = *p;
	    p += count;
	};

	/* fill in trailing zeros, if any */
	lineend = thisline + out_width;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

#ifdef DEBUG_ZOOM
	if (outptr > outputend)    
	{
	    printf("zoom_smaller error: outptr too big (case 1)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

    }
    /* fill rest of buffer (if any) with zeros */
    while (outptr < outputend)
	*outptr++ = pixval[0];    /* pixel value for color zero */
}

/* zoom_smaller32 is same as zoom_smaller, but does 32bit pixels */

static
void zoom_smaller32(inptr, outptr, zoom, in_width, in_height, 
    out_width, out_height, ixoffset, iyoffset)
int *inptr;		/* source of pixels */
register int *outptr;	/* destination */
double zoom;		       /* zoom factor */
unsigned int in_height;		/* input height */
unsigned int in_width;		/* input width */
unsigned int out_width, out_height;	/* output width and height */
int ixoffset, iyoffset; /* position on input image corresponding to 0,0 */
			/* on output image */
{
#define DEBUG_ZOOM 1
    register int *p;
    register int count;
    register int *lineend;	/* pointer to end of a line */
    int *inputend;		/* pointer to end of input buffer */
    int *outputend;		/* pointer to end of output buffer */
    int *thisline;	/* pointer to line to replicate */
    int i_pixels;  	        /* pixels to use from input line to avoid */
				/* overflow of output line */
    int o_zeros;  	     /* number of leading zeros on each output line */

    inputend = inptr + in_width * in_height;
    outputend = outptr + out_width * out_height;

    /* handle y offset first */
    if (iyoffset < 0)
    {
	/* fill first rows with zeros */
	lineend = outptr - out_width * (int) (iyoffset * zoom);
	while ((outptr < lineend) && (outptr < outputend))
	    *outptr++ = pixval[0];
    }
    else
    {
	/* start partway into input image */
	inptr += in_width * iyoffset;
    }

    /* now handle x offset */
    if (ixoffset < 0)
    {
	/* each line needs to have leading zeros */
	o_zeros = -(zoom * ixoffset);   /* number of leading zeros */
	if (o_zeros + (int) (in_width * zoom) > out_width)  /* if overflows output line */
	{
	    i_pixels = (out_width - o_zeros) / zoom;
	    if (o_zeros > out_width)
		inptr = inputend;   /* flag that we are completely off screen */
	}
	else
	    i_pixels = in_width;
    }
    else
    {
	/* need to start partway into each input line */
	o_zeros = 0;
	i_pixels = in_width - ixoffset;
	if (i_pixels <= 0)
	    inptr = inputend;   /* flag that we are completely off screen */
	else
	{
	    inptr += ixoffset;
	    if (i_pixels * zoom > out_width)
		i_pixels = out_width / zoom;
	}
    }


    /* quit when output buffer is full */
    while ((outptr < outputend ) && (inptr < inputend))
    {
	/* remember start of this line */
	thisline = outptr;

	/* make one line with decimation */

	/* fill in leading zeros, if any */
	lineend = thisline + o_zeros;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

	/* do the pixel decimation */
	p = inptr;
	lineend = inptr + i_pixels;
	count = Round(1 / zoom);
	inptr += in_width * count;    /* ready for the next loop iteration */
	while(p < lineend )
	{
	    *outptr++ = *p;
	    p += count;
	};

	/* fill in trailing zeros, if any */
	lineend = thisline + out_width;
	while (outptr < lineend)
	    *outptr++ = pixval[0];    /* pixel value for color zero */

#ifdef DEBUG_ZOOM
	if (outptr > outputend)    
	{
	    printf("zoom_smaller error: outptr too big (case 1)\n");
	    printf("out_width = %d  out_height = %d\n", out_width, out_height);
	    exit(1);
	}
#endif /* DEBUG_ZOOM */

    }
    /* fill rest of buffer (if any) with zeros */
    while (outptr < outputend)
	*outptr++ = pixval[0];    /* pixel value for color zero */
}



/*******************************************************************/
/*                                                                 */
/* ST_BALL uses the pointer to stretch the color table           */
/*                                                                 */
/*******************************************************************/

void
st_ball()
{
    int low, high, temp;
    char comm[MAXLINE];
    int done, what;
    int x, y;
    int dx, dy, total_dx = 0, dx_div, total_dy = 0, dy_div;
    Window st_win;
    Window root, child;
    int root_x, root_y;
    unsigned int keys_buttons;

    if (!graphics)
	return;

    if (set_current_window())
	st_win = win;
    else
    {
	if (wedge_win == 0)
	{
	    error1("no images currently being displayed");
	    return;
	}
	else
	{
	    st_win = wedge_win;
	}
    }

    XQueryPointer(display, st_win,
	&root, &child, &root_x, &root_y, 
	&x, &y, &keys_buttons);

    low = llow_save;
    high = hhigh_save;

    if(server_mode == FALSE)
	printf("set stretch using mouse, then hit CR (here)\n");

    set_motion_notify(st_win);
    done = 0;
    while(!done)
    {
	wait_for_event(comm, &what, &x, &y, &dx, &dy, 0);

	switch(what)
	{
	case Button1:
	    temp = low;
	    low = high;
	    high = temp;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    low = 0;
	    high = 255;
	    break;
	case POINTERMOTION:
	    /* reduce sensitivity  by factor of 2 */
	    total_dx  += dx;
	    dx_div = total_dx / 2;
	    total_dx = total_dx % 2;
	    total_dy  -= dy;
	    dy_div = total_dy / 2;
	    total_dy = total_dy % 2;
	    low  -= dy_div;
	    high += dy_div;
	    low  += dx_div;
	    high += dx_div;
	    break;
	} 

	if ((!s0_server_mode) && (server_mode == FALSE))
	{
	    printf("stretch: %d to %d   \r", low, high);
	    if (debug)
		fprintf(debug_file, "\n");
	    fflush(stdout);
	}
	stretch(low,high);
    } 
    clear_motion_notify();

    if(server_mode == FALSE)
	printf("\nfinal stretch is %d to %d\n",low,high);
    else
    {
	sprintf(server_str, "%d", low);
	srv_string("stretch_min", server_str);
	sprintf(server_str, "%d", high);
	srv_string("stretch_max", server_str);
    }

    fprintf(session, "final stretch is %d to %d\n",low,high);
}
