/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* machine dependent code for SUN SPARCstation running X windows */ 

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>     /* includes strings.h, sys/time.h, and sys/file.h */
#include <X11/keysym.h>

typedef unsigned long Pixel;

#include "skyview.h"
#include "parse.h"
#include "img_x.h"
#include "img.h"
#include "data_types.h"
#include "job.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/time.h>
#include <string.h>

#define DEF 1

#include "icon_bitmap"
#include "old_bitmap"
#define BITMAPDEPTH 1
#define SMALL 1
#define OK 0

extern int fixed_ct;
extern int graphics, ttyfd, debug, server_mode, s0_server_mode, s1_server_mode;
extern int cmdline_mode, accepting_stdin;
extern FILE *debug_file;
extern char server_str[];
extern char plane;
extern int vax_bytes;
extern int frame, frame_advance, imio_wants_char, imio_ok, be_quiet;
extern int minx_next, miny_next;
extern int frame_graphics;
extern char current_clt[], fi_name[], filnam[];
extern char save_line[];
extern char ov_color[];
extern int mapped;
extern int pixels, lines;  /* pixels per line, lines in image */
extern int graphics_window;

extern struct img *curr_img;


static char Id[] = "@(#)xwindows.c   RBH        04/09/2003>";


static int dispcells;
static Window dummy_window;
static int want_motion = 0;
Cursor cursor;
static int xserver_fd;
static int sync_mode = FALSE;
static int do_zoom_flag = 0;
static int button_down = FALSE;
static Window grabbed_window = 0;
static int use_default_color = TRUE;
static int save_u_d_c;  /* save copy of use_default_color */
static int hold_exposes = 0;
static int swap_pixels;

static int ov_colors_n = 0;   /* ov_colors in use */
static int ov_colors_save = 0;   /* ov_colors not to be released  */
static struct {
    XColor color_spec;
    GC gc;
    unsigned long pixel;
    char name[25];
    } ov_colors[256];


int JUPhardware = FALSE;
int force_colormap = FALSE;   /* flag to install X windows colormap */
int desired_colors;
char visualName[80];
int eight_bit = 0;
char default_font_name[MAXPATHLEN];
static unsigned long this_pixval, blank_pixval, default_blank_pixval;
unsigned long black_pixval, white_pixval;

#include "event_types.h"

Display *display;
int screen;
Visual *visual;
unsigned int depth, bytes_per_pixel;

GC image_gc;
GC wedge_gc, hist_gc;
GC gcxor, gc_overlay, gc_red, gc_green, gc_blue, gc_white, gc_black;
static GC gc_user, save_gc_user, gc_transparent, sicon_gc;
XFontStruct *default_font;
Font default_font_id;
Font label_font_id;
extern Font agra_font_id;
Colormap theCmap;
unsigned int ncolors;    /* number of colors in our private color map */
#define MAX_COLORS 256
static unsigned long colors[MAX_COLORS];
unsigned long pixval[MAX_COLORS];
XColor exact_defs[MAX_COLORS];
static Pixel red_max, green_max, blue_max;
static Pixel red_mult, green_mult, blue_mult;
Pixel red_mask, green_mask, blue_mask;
Pixmap icon_pixmap;

unsigned border_width = 4;   /* border four pixels wide */

Window sc_win = 0;
Window hist_win = 0;
Window mag_win = 0;
Window wedge_win = 0;
Window cur_window = 0; /* window last receiving tty or button input */
Atom wm_delete_window;

static unsigned int sicon_width, sicon_height;
static Pixmap sicon_immap;
static Window sicon_win = 0;


/*  -- everything below here appears both globally and in image structure -- */
extern int minx, miny, maxx, maxy;
extern int image_frame;
Window win;
unsigned int win_width, win_height;
XImage *theImage;
unsigned char *image_org;
unsigned int width_org, height_org;
unsigned char *image_zoomed;
unsigned width_zoomed, height_zoomed;
double win_zoom = 1.0;
int win_x_offset, win_y_offset;
struct framestruct *frameptr;


/* function prototypes */
#ifdef _NO_PROTO

static void server_icon();
static void sicon_expose();
static void sicon_configure();
static void sicon_destroy();
static void new_image();
static void fill_black();
static void get_visual();
static void swap_Pixel();
static void swap_short();
static void get_colormap();
static void do_sleep();
static int is_viewable();
static char *visual_name();
static int visual_class();

#else

static void server_icon(void);
static void sicon_expose(void);
static void sicon_configure(unsigned int width, unsigned int height);
static void sicon_destroy(void);
static void new_image(void);
static void fill_black(void);
static void get_visual(void);
static void swap_Pixel(Pixel *pixel);
static void swap_short(Pixel *pixel);
static void get_colormap(void);
static void do_sleep(long sec, long usec);
static int is_viewable(void);
static char *visual_name(int class);
static int visual_class(char *vis_name);

#endif /* _NO_PROTO */

int
init_graphics()
{
    unsigned long valuemask;
    XGCValues values;
    int status, i, j;
    unsigned long plane_masks[1];  /* unused */


    if (!graphics)
    {
#ifdef NOTDEF
	depth = 8;    /* just to make other routines work right */
#endif /* NOTDEF */

    if (depth > 8)
    {
	bytes_per_pixel = 4;
	red_mask = 0xff;
	green_mask = 0xff00;
	blue_mask = 0xff0000;
    }
    else
	bytes_per_pixel = 1;

	gc_transparent = (GC) 0;

	gc_red   = get_rgb("red");
	gc_green = get_rgb("green");
	gc_blue  = get_rgb("blue");
	gc_white = get_rgb("white");
	gc_black = get_rgb("black");

	for (i = 0; i < 255; i++)
	{
	    pixval[i] = (int) ((double) i * 240. / 255.);
	    if (depth > 8)
		pixval[i] = pixval[i] + (pixval[i] << 8) + (pixval[i] << 16);
	    colors[i] = i;
	    default_blank_pixval = 0;
	    blank_pixval = default_blank_pixval;
	}

	ncolors = 240;

	label_font_id = agra_font_id;

	return(FALSE);
    }

    /* get screen size from display structure macro */
    screen = DefaultScreen(display);

    dispcells = DisplayCells(display, screen);

    get_visual();

    wm_delete_window = XInternAtom (display, "WM_DELETE_WINDOW", False);



    /* get the x window server fd */
    xserver_fd = ConnectionNumber(display);


    /* NOW DO COLORS */

    if ((visual->class == PseudoColor) || (visual->class == DirectColor))
    {
    /* allocate as many color cells as we can */
    if (desired_colors == 0)
	ncolors = dispcells;  /* try for all of them */
    else
	ncolors = desired_colors;
    if (debug)
	fprintf(debug_file, "trying for %d colors\n", ncolors);
    for (;;)
    {
	if (XAllocColorCells (display, theCmap, False, plane_masks, 0,
	colors, ncolors))
	    break;
	ncolors--;
	if ((desired_colors > 0) && (theCmap == DefaultColormap(display, screen)))
	{
	    if(server_mode == FALSE)
		printf( 
    "Couldnt get %d colors in default colormap. Creating private colormap\n",
		ncolors+1);
	    get_colormap();
	    ncolors = dispcells;  /* start over */
	}
	if (ncolors <= 20)
	{
	    error1("error: couldn't allocate 20 read/write colors");
	    exit(1);
	}
	if ((ncolors <= 64) && (theCmap == DefaultColormap(display, screen)))
	{
	    if(server_mode == FALSE)
	    {
		printf(
"Fewer than 64 colors available in default colormap.  Creating private colormap\n");
	    }
	    get_colormap();
	    ncolors = dispcells - 6;
	}
    }

    if (debug)
	fprintf(debug_file,
	    "there are %d read/write color cells available\n", ncolors);
    XFreeColors(display, theCmap, colors, ncolors, 0L);
    if (ncolors > 20)
	ncolors -= 20;  /* allocate all but 20 of them */
    if (ncolors < 10)
	ncolors = 4;  /* get at least 4, even if he forces less */
    if (!XAllocColorCells (display, theCmap, False, plane_masks, 0,
	colors, ncolors))
    {
	error1("bizarre error allocating colors");
	exit(1);
    }
    if (debug)
	fprintf(debug_file, "allocated %d read/write color cells\n", ncolors);

    for (i = 0, j = 0; i < ncolors; i++)
    {
	/* set pixel value in struct to the allocated one */
	if ((colors[i] < 254) || (visual->class != PseudoColor))
	/* save 254 and 255 for OpenWindows black and white */
	{
	    exact_defs[j].pixel = colors[i];
	    j++;
	}
    }
    ncolors = j;

    for (i = 0; i < 255; i++)                    /* set 0 - 254 */
	pixval[i] = colors[(ncolors * i) / 255];
    /* Thus, pixval[254] = colors[ncolors - 1]  */
    /*   ( pixval[255] is set to white above )  */

    if ((visual->class == DirectColor) && (swap_pixels))
    {
	for (i = 0; i < 255; i++)
	{
	    if (depth > 16)
		swap_Pixel(&pixval[i]);
	    else if (depth > 8)
	    {
		swap_short(&pixval[i]);
	    }
	}
    }

    default_blank_pixval = colors[0];
    blank_pixval = default_blank_pixval;
    }
    else   /* read-only colormap */
    {
	if (visual->class == TrueColor)
	{
	    for (i = 0; i < 255; i++)
	    {
		pixval[i] = ((i * (red_max+1))   / 256) * red_mult   + 
			    ((i * (green_max+1)) / 256) * green_mult + 
			    ((i * (blue_max+1))  / 256) * blue_mult;
		if (swap_pixels)
		{
		    if (depth > 16)
			swap_Pixel(&pixval[i]);
		    else if (depth > 8)
		    {
			swap_short(&pixval[i]);
		    }
		}
	    }
	}
	else
	{
	    for (i = 0; i < 255; i++)                    /* set 0 - 254 */
		pixval[i] = i;
	}
    }

    /* get existing rgb's in case he does a wedge now */
    XQueryColors(display, theCmap, exact_defs, ncolors);
    /* this also sets exact_defs[i].flags = DoRed | DoGreen | DoBlue  */

    load_font(&default_font, default_font_name);
    default_font_id = default_font->fid;
    label_font_id = default_font->fid;

    /* create GC's */
    /* first must create a drawable - cannot use root window because */
    /* it may not be same depth as our selected visual */
    dummy_window = new_window(0,0,1,1);
    /* note that we never map this dummy window */


    /* create GC's for overlays */
    status = set_overlay_color("red");
    gc_red = gc_user;

    status = set_overlay_color("green");
    gc_green = gc_user;

    status = set_overlay_color("blue");
    gc_blue = gc_user;

    status = set_overlay_color("white");
    white_pixval = this_pixval;
    gc_white = gc_user;

    status = set_overlay_color("black");
    black_pixval = this_pixval;
    gc_black = gc_user;

    ov_colors_save = ov_colors_n;
    status = set_overlay_color(ov_color);

    gc_overlay = gc_white;   /* make sure that gc_overlay is valid */


    values.font = default_font_id;
    valuemask = GCForeground | GCFont;

    /* create GC for image */
    values.foreground = black_pixval;
    image_gc = XCreateGC(display, dummy_window,
	valuemask, &values);

    /* create GC for wedge */
    wedge_gc = XCreateGC(display, dummy_window, 
	valuemask, &values);

    /* create GC for server icon */
    sicon_gc = XCreateGC(display, dummy_window, 
	valuemask, &values);

    /* create GC for drawing histogram (white foreground) */
    values.foreground = white_pixval;
    hist_gc = XCreateGC(display, dummy_window,
	valuemask, &values);

    /* create GC for drawing with XOR */
    values.foreground = 0x80;
    values.function = GXxor;
    valuemask |= GCFunction;
    gcxor = XCreateGC(display, dummy_window, 
	valuemask, &values);

    /* create GC for transparent (no draw at all */
    values.foreground = 0x80;
    values.function = GXnoop;
    valuemask |= GCFunction;
    gc_transparent = XCreateGC(display, dummy_window, 
	valuemask, &values);

    /* Create pixmap of depth 1 (bitmap) for icon */
    icon_pixmap = XCreateBitmapFromData(display, RootWindow(display, screen),
	(char *) icon_bitmap_bits, icon_bitmap_width, icon_bitmap_height);

    if (s0_server_mode)
	server_icon();

    /* Create crosshairs cursor (now done in resources_x.c) */
    /*
    cursor = XCreateFontCursor(display, XC_crosshair);
    */

    return(TRUE);
}

static void
server_icon()
{
    XSizeHints size_hints;
    int x = 800, y = 0;            /* window position */
    char *window_name;
    char *sicon_name;
    int cmdc;
    char **cmdv;
    XSetWindowAttributes attributes;
    unsigned long valuemask;
    XWMHints wmhints;

    /* Create pixmap of depth "depth" for displaying icon in a mapped window */
    sicon_immap = XCreatePixmapFromBitmapData(display, 
	RootWindow(display, screen),
	(char *) old_bitmap_bits, old_bitmap_width, old_bitmap_height,
	WhitePixel(display, screen), BlackPixel(display, screen), depth);

    window_name = "skyview";
    sicon_name = " ";
    sicon_width = old_bitmap_width;
    sicon_height = old_bitmap_height;

    attributes.background_pixel = BlackPixel(display, screen);
    attributes.border_pixel = BlackPixel(display, screen);
    attributes.colormap = theCmap;
    attributes.backing_store = WhenMapped;
    attributes.bit_gravity= StaticGravity;
    valuemask = CWBackPixel | CWBorderPixel | CWColormap | 
	CWBackingStore | CWBitGravity;
    sicon_win = XCreateWindow(display, RootWindow(display, screen),
	x, y, sicon_width, sicon_height, border_width,
	depth, InputOutput, visual, valuemask, &attributes);

    /* initialize size hint property for window manager */
    size_hints.flags = USSize | USPosition;
    size_hints.x = x;
    size_hints.y = y;
    size_hints.width = sicon_width;
    size_hints.height = sicon_height;
    cmdc = 0;
    cmdv = NULL;

    /* set Properties for window manager (always before mapping) */
    XSetStandardProperties(display, sicon_win, window_name, sicon_name,
	icon_pixmap, cmdv, cmdc, &size_hints);

    wmhints.initial_state = IconicState;
    wmhints.flags = StateHint | IconPixmapHint;
    wmhints.icon_pixmap = icon_pixmap;
    XSetWMHints(display, sicon_win, &wmhints);

#ifdef WithdrawnState
    (void) XSetWMProtocols (display, sicon_win, &wm_delete_window, 1);
#endif /* WithdrawnState */

    /* Select event types wanted */
    XSelectInput(display, sicon_win,  ExposureMask |  StructureNotifyMask);

    /* Display window */
    XMapWindow(display, sicon_win);

}

static void
sicon_expose()
{
    if (sicon_win == 0)
	return;
    XCopyArea(display, sicon_immap, sicon_win, sicon_gc, 0, 0, 
	sicon_width, sicon_height, 0, 0);
}

static void
sicon_configure(width, height)
unsigned width, height;
{
    sicon_width = width;
    sicon_height = height;
}

static void
sicon_destroy()
{
    if (sicon_win != 0)
    {
	XDestroyWindow(display, sicon_win);
	sicon_win = 0;
    }
}


int
show_frame(geom, zp, labelptr)
char *geom, *labelptr;
int zp;
{
    XClassHint class_hints;
    XSizeHints size_hints;
    char window_name[MAXPATHLEN];
    char resource_name[100];
    char icon_name[20];
    int cmdc;
    char **cmdv;
    struct img *tmpimg;
    XEvent report;
    int parse_result;
    int nbr, y, delta_y;

    image_frame = frame;

    if ((depth < 8) && graphics)
    {
	error1("Cannot paint images on a monochrome display");
	return(FALSE);
    }
    width_org = maxx - minx + 1;
    height_org = maxy - miny + 1;
    image_zoomed = NULL;

    /* if the frame exists, use the same window */
    tmpimg = peek_FRAME(frame);
    if ((tmpimg != NULL) && (geom == NULL))
    {
	/* repaint into existing window */
	frameptr = tmpimg->frameptr;
	if (zp)
	{
	    if ((win_zoom > 1.0001) || (win_zoom < .99999) ||
		(win_x_offset != 0) || (win_y_offset != 0))
		    if (graphics)
			XClearWindow(display, tmpimg->frameptr->win);
	    win_zoom = 1.0;
	    win_x_offset = 0;
	    win_y_offset = 0;
	}
	else
	{
	    win_zoom = tmpimg->frameptr->win_zoom;
	    win_x_offset = tmpimg->frameptr->win_x_offset ;
	    win_y_offset = tmpimg->frameptr->win_y_offset ;
	}
	win = tmpimg->frameptr->win;
	win_width = tmpimg->frameptr->win_width;
	win_height = tmpimg->frameptr->win_height;

	/* make sure not pointing to image_zoomed */
	tmpimg->frameptr->theImage->data = (char *) tmpimg->frameptr->image_org;
	tmpimg->frameptr->theImage->width = tmpimg->frameptr->width_org;
	tmpimg->frameptr->theImage->height = tmpimg->frameptr->height_org;
	tmpimg->frameptr->theImage->bytes_per_line = tmpimg->frameptr->width_org * bytes_per_pixel;

	if (tmpimg->frameptr->image_zoomed != NULL)
	{
	    free(tmpimg->frameptr->image_zoomed);
	    tmpimg->frameptr->image_zoomed = NULL;
	}

	if ((width_org <= tmpimg->frameptr->width_org) && 
	    (height_org <= tmpimg->frameptr->height_org))
	{
	    /* new image is same size or smaller */
	    width_org = tmpimg->frameptr->width_org;
	    height_org = tmpimg->frameptr->height_org;
	    theImage = tmpimg->frameptr->theImage;
	    image_org = tmpimg->frameptr->image_org;
	}
	else
	{
	    /* image is larger */
	    if (width_org < tmpimg->frameptr->width_org)
		width_org = tmpimg->frameptr->width_org;
	    if (height_org < tmpimg->frameptr->height_org)
		height_org = tmpimg->frameptr->height_org;
	    
	    new_image();    /* get image_org, theImage */
	    if (theImage == NULL)
		return(FALSE);
	    delta_y = height_org - tmpimg->frameptr->height_org;
	    for (y = 0; y < tmpimg->frameptr->height_org; y++)
	    {
		nbr = tmpimg->frameptr->width_org;
		if (depth > 16)
		{
		    INT32 *inptr32, *outptr32;;

		    inptr32 = (INT32 *) (tmpimg->frameptr->image_org + 
			y * tmpimg->frameptr->width_org * 4);
		    outptr32 = (INT32 *) (image_org +
			(y + delta_y) * width_org * 4);
		    while (nbr--)
			*outptr32++ = *inptr32++;
		}
		else if (depth > 8)
		{
		    INT16 *inptr16, *outptr16;;

		    inptr16 = (INT16 *) (tmpimg->frameptr->image_org + 
			y * tmpimg->frameptr->width_org * 2);
		    outptr16 = (INT16 *) (image_org +
			(y + delta_y) * width_org * 2);
		    while (nbr--)
			*outptr16++ = *inptr16++;
		}
		else   /* depth == 8 */
		{
		    INT8 *inptr8, *outptr8;

		    inptr8 = tmpimg->frameptr->image_org + 
			y * tmpimg->frameptr->width_org;
		    outptr8 = image_org + (y + delta_y) * width_org;
		    while (nbr--)
			*outptr8++ = *inptr8++;
		}
	    }

	    if (graphics)
		XDestroyImage(tmpimg->frameptr->theImage);
	    tmpimg->frameptr->image_org = NULL;
	    tmpimg->frameptr->theImage = NULL;
	}

	if (labelptr != NULL)
	    sprintf(window_name, "%d %s", frame, labelptr);
	else
	    sprintf(window_name,"%s  (frame %d)", strrchr(fi_name, '/') + 1, frame);
	change_win_name(window_name);

	/* now queue up a fake expose event */
	if (graphics)
	{
	    report.type = Expose;
	    report.xexpose.window = win;
	    report.xexpose.x = 0;
	    report.xexpose.y = 0;
	    report.xexpose.width = win_width;
	    report.xexpose.height = win_height;
	    report.xexpose.count = 0;
	    if ((win_zoom > 1.0001) || (win_zoom < .99999) ||
		(win_x_offset != 0) || (win_y_offset != 0))
	    {

		do_zoom_flag = win;   /* flag for later call to do_zoom_pan */
		/* (cannot zp until pixels are here) (write_AOI) */
	    }
	    XSendEvent(display, win, False, ExposureMask, &report);
	}

	cur_window = win;   /* make it the current window */
	return(TRUE);
    }

    frameptr = (struct framestruct *) malloc(sizeof(struct framestruct));
    win_zoom = 1.0;
    win_x_offset = 0;
    win_y_offset = 0;
    win_width = width_org;
    win_height = height_org;

    new_image();    /* get image_org, theImage */
    if (theImage == NULL)
	return(FALSE);

    if (!graphics)
    {
	win = frame;   /* dummy non-zero window id */
	cur_window = win;   /* make it the current window */
	return(TRUE);
    }
    size_hints.flags = PSize;
    size_hints.x = 0;
    size_hints.y = 0;
    if (geom != NULL)
    {
	next_frame();
	image_frame = frame;
	parse_result = XParseGeometry(geom, &size_hints.x, &size_hints.y,
	    &win_width, &win_height);
	if (parse_result & (WidthValue | HeightValue))
	    size_hints.flags |= USSize;

	if (parse_result & XValue)
	{
	    if (parse_result & XNegative)
	    {
		size_hints.x = DisplayWidth(display, screen) + size_hints.x
		    - win_width;
	    }
	    size_hints.flags |= USPosition;
	}
	if (parse_result & YValue)
	{
	    if (parse_result & YNegative)
	    {
		size_hints.y = DisplayHeight(display, screen) + size_hints.y
		    - win_height;
	    }
	    size_hints.flags |= USPosition;
	}

    }

    sprintf(icon_name,"frame %d", frame);
    if (labelptr != NULL)
	sprintf(window_name, "%d %s", frame, labelptr);
    else
	sprintf(window_name,"%s  (frame %d)", strrchr(fi_name, '/') + 1, frame);

    win = new_window(size_hints.x, size_hints.y, win_width, win_height);


    size_hints.width = win_width;
    size_hints.height = win_height;
    cmdc = 0;
    cmdv = NULL;

    /* DEBUG FOLLOWS */
    sprintf(resource_name, "frame%d", frame);
    class_hints.res_name = resource_name;
    class_hints.res_class = "Skyview";
    XSetClassHint(display, win, &class_hints);
    /* END DEBUG */

    /* set Properties for window manager */
    XSetStandardProperties(display, win, window_name, icon_name,
	icon_pixmap, cmdv, cmdc, &size_hints);

    change_win_name(window_name);

#ifdef WithdrawnState
    /* XSetWMProtocol not available in X11R3 */
    /*    (WithdrawnState is defined in Xutil.h for X11R4, but not X11R3) */
    /*   Too bad that XlibSpecificationRelease wasn't defined until X11R5 */
    (void) XSetWMProtocols (display, win, &wm_delete_window, 1);
#endif /* WithdrawnState */

    /* set crosshair cursor */
    XDefineCursor(display, win, cursor);
    
    /* Select event types wanted */
    XSelectInput(display, win, KeyPressMask | ExposureMask |
	ButtonPressMask | StructureNotifyMask | PointerMotionMask |
	PointerMotionHintMask);

    /* Display window */
    XMapWindow(display, win);
    XFlush(display);  /* let server start chewing on this */
    cur_window = win;   /* make it the current window */
    return(TRUE);
}

void
wait_for_windowmgr(force)
int force;
{
    if (!graphics)
	return;

    if ((sync_mode) || (force))
    {
	/* wait for the MapNotify - let window manager do its thing */
	XFlush(display);
	do_sleep((long) 1, (long) 999999);  /* let window manager execute */
	/* specifying 999999 usec swaps us out only until the next event, */
	/* (the results of the window manager Reparenting or Mapping) */
	/* with a max of 1 sec if no event happens */
	/* Without this kludge, set_current_window() finds  */
	/* that the window is not Viewable by virtue of the */
	/* window manager not having been able to map the window */
    }
}

Window new_window(x, y, width, height)
int x, y;
unsigned int width, height;
{
    XSetWindowAttributes attributes;
    unsigned long valuemask;

    attributes.background_pixel = black_pixval;
    attributes.border_pixel = black_pixval;
    attributes.colormap = theCmap;
    attributes.backing_store = WhenMapped;
    attributes.bit_gravity= StaticGravity;
    valuemask = CWBackPixel | CWBorderPixel | CWColormap | 
	CWBackingStore | CWBitGravity;
    return(XCreateWindow(display, RootWindow(display, screen),
	x, y, width, height, border_width,
	depth, InputOutput, visual, valuemask, &attributes));
}

void
change_win_name(window_name)
char *window_name;
{
    char tmp_name[300];

    if (!graphics)
	return;
    sprintf(tmp_name, "IPAC skyview: %s", window_name);
    XStoreName(display, win, tmp_name);
}

/* new_image creates theImage */
/* uses width_org, height_org, and bytes_per_pixel */
/* sets theImage, image_org */
static void
new_image()
{
    static XImage dummyImage;

    /* get buffer for pixels */
    image_org = (unsigned char *) calloc(width_org * height_org , 
	bytes_per_pixel);
    if (!image_org)
    {
	error1("error: unable to allocate space for pixels");
	theImage = NULL;
	return;
    }

    fill_black();     /* fill with current black pixval */
    /* (in case we write into one plane only or we are enlarging the buffer) */

    if (!graphics)
    {
	theImage = &dummyImage;
	frame_graphics = 0;
	return;
    }

    theImage = XCreateImage(display, visual, depth, ZPixmap, 0, 
	(char *) image_org, width_org, height_org, 8, 0);
    if (!theImage)
    {
	error1("error: unable to create Ximage");
	free(image_org);
	image_org = NULL;
    }
    frame_graphics = 1;
}

/* fill_black fills image_org with black pixels */
static void
fill_black()
{
    int nbr;

    nbr = width_org * height_org;
    if (depth > 16)
    {
	INT32 zeroval, *ptr32;
	ptr32 = (INT32 *) image_org;
	zeroval = pixval[0];
	while (nbr--)
	    *ptr32++ = zeroval;
    }
    else if (depth > 8)
    {
	INT16 zeroval, *ptr16;
	ptr16 = (INT16 *) image_org;
	zeroval = pixval[0];
	while (nbr--)
	    *ptr16++ = zeroval;
    }
    else
    {
	/* depth == 8 */
	INT8 zeroval, *ptr8;

	ptr8 = image_org;
	zeroval = pixval[0];
	while (nbr--)
	    *ptr8++ = zeroval;
    }
}

/* ************************************************************ */
/*  get_visual searches the list of visuals for the best visual */
/*     Sets global variables visual, theCmap, visual->class     */
/* ************************************************************ */

static void
get_visual()
{
    XVisualInfo vTemplate;    /* template of the visual we want */
    XVisualInfo *visualList;  /* list of XVisualInfo structs that match */
    int visualsMatched;       /* number of visuals that match */
    int visual_index;
    int requested_index, default_index;
    int i, goodness8[6], goodness32[6];
    int this_goodness, best_goodness;
    int shift;
    int desired_visual_class;

    desired_visual_class = visual_class(visualName);

    switch(desired_visual_class)
    {
    case NOT_SPECIFIED:
	if (DisplayPlanes(display, screen) == 16)
	{
	    desired_visual_class = DEFAULT;
	    if (debug)
		printf(
		    "favoring default visual, since this is a 16-bit screen\n");
	}
	break;
    case DEFAULT:
	if (debug)
	    printf("user specified default visual\n");
	break;
    default:
	if (debug)
	    printf("user specified visual [%s]  class = %d\n", visualName, 
		visual_class(visualName));
	break;
    }

    /* look for the best visual */
    /* try the default visual */
    visual = DefaultVisual(display, screen);
    theCmap = DefaultColormap(display, screen);
    depth = DisplayPlanes(display, screen);
    if (depth > 16)
	bytes_per_pixel = 4;
    else if (depth > 8)
	bytes_per_pixel = 2;
    else
	bytes_per_pixel = 1;


	
    /* rank the goodness of visuals for 8 and 32 bit depth displays */
    goodness8[PseudoColor] = 5;
    goodness8[DirectColor] = 4;
    goodness8[TrueColor]   = 3;
    goodness8[StaticColor] = 2;
    goodness8[GrayScale]   = 1;
    goodness8[StaticGray]  = 0;

    goodness32[DirectColor] = 15;
    goodness32[TrueColor]   = 14;
    goodness32[PseudoColor] = 13;
    goodness32[StaticColor] = 12;
    goodness32[GrayScale]   = 11;
    goodness32[StaticGray]  = 10;

    if (eight_bit)
    {
	/* he wants to favor 8 bit stuff */
	if(server_mode == FALSE)
	    fprintf(debug_file, "favoring 8 bit visuals per user request\n");
	goodness8[PseudoColor] = 25;
	goodness8[DirectColor] = 24;
	goodness8[TrueColor]   = 23;
	goodness8[StaticColor] = 22;
	goodness8[GrayScale]   = 21;
	goodness8[StaticGray]  = 20;
    }

    /*
     * Set up the XVisualInfo template so that it returns
     * a list of all the visuals defined on the
     * current screen by the X server
     */
    vTemplate.screen = screen;

    visualList = XGetVisualInfo (display, (long) VisualScreenMask,
	&vTemplate, &visualsMatched);

    if ( visualList == NULL )
    {
	error1("skyview:  cannot find any visuals");
	/* just use the default visual */
	return;
    }

    /* find the visual with the best goodness */

    best_goodness = -1;
    visual_index = -1;
    default_index = -1;
    requested_index = -1;

    for (i = 0; i < visualsMatched; i++)
    {
	if (visualList[i].visual == DefaultVisual(display, screen))
	    default_index = i;   /* remember which one is the default */

	if (visualList[i].class == desired_visual_class)
	    requested_index = i;  /* this one matches users request */

	if (visualList[i].depth > 8)
	    this_goodness = goodness32[visualList[i].class];
	else if (visualList[i].depth == 8)
	    this_goodness = goodness8[visualList[i].class];
	else
	    continue;

	if ((this_goodness > best_goodness) && 
	    (visualList[i].colormap_size > 20))
	{
	    /* this visual is better */
	    visual_index = i;
	    best_goodness = this_goodness;
	}

    }

    if (best_goodness == -1)
    {
	error1("skyview:  cannot find any visual with depth 8 or more");
	/* just use the default visual */
	return;
    }

    if (desired_visual_class == DEFAULT)
    {
	if (default_index >= 0)
	    visual_index = default_index;
	else
	{
	    sprintf(server_str, 
		"FATAL ERROR:  cannot find default visual");
	    error1(server_str);
	    exit(1);
	}
    }
    else if (desired_visual_class != NOT_SPECIFIED)
    {
	if (requested_index >= 0)
	{
	    visual_index = requested_index;
	}
	else
	{
	    sprintf(server_str, 
		"FATAL ERROR:  cannot find a visual of type %s", visualName);
	    error1(server_str);
	    exit(1);
	}
    }

    depth = visualList[visual_index].depth;
    if (depth > 16)
	bytes_per_pixel = 4;
    else if (depth > 8)
	bytes_per_pixel = 2;
    else
	bytes_per_pixel = 1;
    visual = visualList[visual_index].visual;

    if (debug)
	fprintf(debug_file,
	    "selected visual of class %d (%s)  depth %d\n", visual->class, 
		visual_name(visual->class), depth);

    /* if it's not the default, get a colormap */
    if (visual != DefaultVisual(display, screen))
    {
	get_colormap();
    }
    if ((visual->class == TrueColor) || (visual->class == DirectColor))
    {
	/* need to find red, green, blue field placement and width */
	red_mask = visualList[visual_index].red_mask;
	green_mask = visualList[visual_index].green_mask;
	blue_mask = visualList[visual_index].blue_mask;

	red_max = red_mask;
	green_max = green_mask;
	blue_max = blue_mask;

	shift = 0;
	while ((red_max & 0x01) == 0)
	{
	    shift++;
	    red_max >>= 1;
	}
	red_mult = 1 << shift;

	shift = 0;
	while ((green_max & 0x01) == 0)
	{
	    shift++;
	    green_max >>= 1;
	}
	green_mult = 1 << shift;

	shift = 0;
	while ((blue_max & 0x01) == 0)
	{
	    shift++;
	    blue_max >>= 1;
	}
	blue_mult = 1 << shift;

	if ((depth > 8) &&
	    (((vax_bytes == 0) && (ImageByteOrder(display) == LSBFirst)) ||
	    ((vax_bytes == 1) && (ImageByteOrder(display) == MSBFirst))))
	{
	    /* client (xlib) and server have differing byte-sex */
	    /* swap bytes in masks */
	    swap_pixels = 1;
	    if (depth > 16)
	    {
		swap_Pixel(&red_mask);
		swap_Pixel(&green_mask);
		swap_Pixel(&blue_mask);
	    }
	    else if (depth > 8)
	    {
		swap_short(&red_mask);
		swap_short(&green_mask);
		swap_short(&blue_mask);
	    }
	}
	else
	    swap_pixels = 0;

    }

    /* All done with visual information.  Free it.  */

    XFree(visualList);
}

static char *visual_name(class)
int class;
{
    if (class == DirectColor)
	return("DirectColor");
    if (class == TrueColor)
	return("TrueColor");
    if (class == PseudoColor)
	return("PseudoColor");
    if (class == StaticColor)
	return("StaticColor");
    if (class == GrayScale)
	return("GrayScale");
    if (class == StaticGray)
	return("StaticGray");
    return(NULL);
}

static int visual_class(vis_name)
char *vis_name;
{
    if (strcasecmp(vis_name, "DirectColor") == 0)
	return(DirectColor);
    if (strcasecmp(vis_name, "TrueColor") == 0)
	return(TrueColor);
    if (strcasecmp(vis_name, "PseudoColor") == 0)
	return(PseudoColor);
    if (strcasecmp(vis_name, "StaticColor") == 0)
	return(StaticColor);
    if (strcasecmp(vis_name, "GrayScale") == 0)
	return(GrayScale);
    if (strcasecmp(vis_name, "StaticGray") == 0)
	return(StaticGray);
    if (strcasecmp(vis_name, "default") == 0)
	return(DEFAULT);
    if (strcasecmp(vis_name, "not_specified") == 0)
	return(NOT_SPECIFIED);
    
    sprintf(server_str,
	"FATAL ERROR: unrecognized visual: %s", vis_name);
    error1(server_str);
    exit(1);
}

static void
swap_Pixel(pixel)
Pixel *pixel;   /* assumes that Pixel (an unsigned long) is 4 bytes */
{
    unsigned char *p, temp;

    p = (unsigned char *) pixel;
    temp = p[0];
    p[0] = p[3];
    p[3] = temp;
    temp = p[1];
    p[1] = p[2];
    p[2] = temp;
}

static void
swap_short(pixel)
Pixel *pixel;   /* assumes that short is 2 bytes */
{
    unsigned char *p, temp;

    p = (unsigned char *) pixel;
    temp = p[2];
    p[2] = p[3];
    p[3] = temp;
}

static void
get_colormap()
{
#define RETAINCOLORS 16         /* number of colors to copy from default clt */
    Pixel black_pixel, white_pixel;
    int retain1, retain2, nextfree;
    XColor colorcell_def[256];
    unsigned long colors[256];
    unsigned long plane_masks[1];  /* unused */
    int i;

    theCmap = XCreateColormap(display, RootWindow(display, screen),
	visual, AllocNone);

    if ((visual->class == PseudoColor) || (visual->class == DirectColor))
    {
	if (visual->map_entries > RETAINCOLORS)
	{
	/* Copy some colors from the default color table      */
	/* to minimize flashing and preserve black and white. */
	/* We wil copy the first RETAINCOLORS colors plus     */
	/* BlackPixel and WhitePixel                          */

	if (DefaultDepth(display, screen) == 24)
	{
	    for (i = 0; i < 256; i++)
		colorcell_def[i].pixel = i * 0x10000 + i * 0x100 + i;
	}
	else
	{
	    for (i = 0; i < 256; i++)
		colorcell_def[i].pixel = i;
	}
	/* get all the default colors */
	XQueryColors(display, DefaultColormap(display, screen),
	    colorcell_def, 256);

	/* allocate all the cells of our new colormap */
	if (!XAllocColorCells (display, theCmap, False, plane_masks, 0,
	    colors, 256))
	{
	    error1(
	    "X windows error:  cannot allocate colors in private color map");
	}

	for (i = 0; i < 256; i++)
	{
	    colorcell_def[i].pixel = colors[i];
	}


	/* move all the colors into our color table */
	XStoreColors (display, theCmap, colorcell_def, 256);

	/* free all colors except 0..(RETAINCOLORS-1) and B and W */

	/* find black and white pixels in colors array */
	black_pixel = BlackPixel(display, screen);
	for (i = 0; i < 256; i++)
	{
	    if (black_pixel == colors[i])
		break;
	}
	retain1 = i;

	white_pixel = WhitePixel(display, screen);
	for (i = 0; i < 256; i++)
	{
	    if (white_pixel == colors[i])
		break;
	}
	if (retain1 < i)
	{
	    retain2 = i;
	}
	else
	{
	    retain2 = retain1;
	    retain1 = i;
	}


	if ((desired_colors > 0) && (dispcells - desired_colors < RETAINCOLORS))
	{
	    /* he wants so many colors that we can't save RETAINCOLORS */
	    nextfree = dispcells - desired_colors;
	}
	else
	    nextfree = RETAINCOLORS;
	if (retain1 > nextfree)
	{
	    XFreeColors(display, theCmap, &colors[nextfree],
		(int) (retain1 - nextfree), 0L);
	    nextfree = retain1 + 1;
	} else if (retain1 == nextfree)
	{
	    nextfree++;
	}

	if (retain2 > nextfree)
	{
	    XFreeColors(display, theCmap, &colors[nextfree],
		(int) (retain2 - nextfree), 0L);
	    nextfree = retain2 + 1;
	} else if (retain2 == nextfree)
	{
	    nextfree++;
	}

	if (256 > nextfree)
	{
	    XFreeColors(display, theCmap, &colors[nextfree],
		(int) (256 - nextfree), 0L);
	}

	}
    }

    /* now we want to insure that colormap will be installed */
    /* The window manager is supposed to do it, but Nelson's does not */

    if (force_colormap)
	XInstallColormap(display, theCmap);
}

void
load_font(font_info1, fontname)
XFontStruct **font_info1;
char *fontname;
{
    /* Access font */
    if ((*font_info1 = XLoadQueryFont(display,fontname)) != NULL)
    {
	return;
    }
    else
    {
	if(server_mode == FALSE)
	{
	    printf("cannot open font %s\n", fontname);
	}
    }
    /* try for a usually available  font */
    if ((*font_info1 = XLoadQueryFont(display,
	"-*-*-*-R-*-*-*-120-*-*-*-*-ISO8859-1")) != NULL)
    {
	return;
    }
    else
    {
	if(server_mode == FALSE)
	{
	    printf("cannot open font %s\n", fontname);
	    exit(1);
	}
    }
}

void
close_graphics()
{
    XFreePixmap(display, icon_pixmap);
    XFreeFont(display, default_font);
    XFreeGC(display, image_gc);
    XCloseDisplay(display);
}

void
erase(argc, argv)
int argc;
char **argv;
{
    struct img *save_imgp, *tmpimg;
    int bundle_nbr, status, frame_nbr, dummy, dummyx, dummyy;
    int nbr;
    XEvent report;

    if (argc == 1)
    {
	if (cur_window == 0)
	{
	    /* erase all windows */
	    tmpimg = find_last_img(); 
	    while (tmpimg != NULL)
	    {
		delete_WIN();
		tmpimg = find_last_img();
	    }
	    clear_img();  /* clear entire screen mgmt */
	    delete_all_RGB();
	}
	else
	{
	    /* erase just this window */
	    if (cur_window == wedge_win)
	    {
		wedge_destroy();
	    }
	    else if (cur_window == hist_win)
	    {
		hist_destroy();
	    }
	    else if (cur_window == sc_win)
	    {
		sc_destroy();
	    }
	    else if (cur_window == mag_win)
	    {
		mag_destroy();
	    }
	    else
	    {
		tmpimg = find_WIN(cur_window);		/* get globals */
		if (tmpimg != NULL)
		{
		    delete_WIN();
		}
	    }
	    cur_window = 0;
	}
    }
    else    /* (argc > 1) */
    {
	if (isint(argv[1]))
	{
	    /*  erase frame  */
	    dummy = getint(argv[1], &frame_nbr);
	    tmpimg = peek_FRAME(frame_nbr);
	    if (tmpimg != NULL)
	    {
		fill_glob(tmpimg);
		delete_WIN();
	    }
	}
	else if ((argv[1][1] == 'o') && (argv[1][2] == '\0'))
	{
	    /*  erase overlay */
	    erase_graphics_color(argv[1]);
	    expose_all();
	}
	else if (strcmp(argv[1], "bundle") == 0)
	{
	    if (argc >= 3)
	    {
		status = getint(argv[2], &bundle_nbr);
		if (status)
		{
		    erase_graphics_bundle(bundle_nbr);
		    expose_all();
		}
	    }
	    else
	    {
		if (graphics)
		    erase_graphics_trackball();
	    }
	}
	else
	{
	    if (argv[1][1] == '\0')  /* if it's one character only */
	    {
	    /* erase image plane (a, r, g, or b) */
	    if (argv[1][0] == 'a')
	    {
		fill_black();   /* fill pixel buffer (image_org) with black */
		save_imgp = curr_img;
		for (;;)
		{
		    tmpimg = find_WIN_again(save_imgp);
		    if (tmpimg == NULL)
			break;
		    delete_img(tmpimg);        /* delete from screen mgmt */
		};
		fill_glob(save_imgp);           /* refill globals */
	    }
	    else
	    {
		if (depth > 16)
		{
		    INT32 zeroval, *ptr32, set_mask, clear_mask;
		    switch(argv[1][0])
		    {
			case 'r':
			    set_mask = red_mask;
			    break;
			case 'g':
			    set_mask = green_mask;
			    break;
			case 'b':
			    set_mask = blue_mask;
			    break;
			default:
			    sprintf(server_str,
				"illegal plane in erase command: %c", argv[1][0]);
			    error1(server_str);
			    return;

		    }
		    clear_mask = ~set_mask;
		    ptr32 = (INT32 *) image_org;
		    zeroval = pixval[0];
		    nbr = width_org * height_org;
		    while (nbr--)
		    {
			*ptr32 = (set_mask & zeroval) | (clear_mask & *ptr32);
			ptr32++;
		    }
		}
		else if (depth > 8)
		{
		    INT16 zeroval, *ptr16, set_mask, clear_mask;
		    switch(argv[1][0])
		    {
			case 'r':
			    set_mask = red_mask;
			    break;
			case 'g':
			    set_mask = green_mask;
			    break;
			case 'b':
			    set_mask = blue_mask;
			    break;
			default:
			    sprintf(server_str,
				"illegal plane in erase command: %c", argv[1][0]);
			    error1(server_str);
			    return;

		    }
		    clear_mask = ~set_mask;
		    ptr16 = (INT16 *) image_org;
		    zeroval = pixval[0];
		    nbr = width_org * height_org;
		    while (nbr--)
		    {
			*ptr16 = (set_mask & zeroval) | (clear_mask & *ptr16);
			ptr16++;
		    }
		}
		else
		{
		    error1("cannot erase image planes on 8 plane graphics systems");
		    return;
		}
	    }
	    if (image_zoomed != NULL)
	    {
		if (graphics)
		    do_zoom_pan(1, &win_zoom, &dummyx, &dummyy, 1);
	    }
	    else
	    {
		/* queue up a fake expose event */
		report.type = Expose;
		report.xexpose.window = win;
		report.xexpose.x = 0;
		report.xexpose.y = 0;
		report.xexpose.width = win_width;
		report.xexpose.height = win_height;
		report.xexpose.count = 0;
		if (graphics)
		    XSendEvent(display, win, False, ExposureMask, &report);
	    }
	    }
	    else
	    {
		/* erase arbitrary color */
		erase_graphics_color(argv[1]);
		expose_all();
	    }
	}
    }
}

void
expose_all()
{
    struct img *savimg, *tmpimg;

    if (!graphics)
	return;

    savimg = curr_img;   /* save current image */

    tmpimg = find_last_img(); 
    while (tmpimg != NULL)
    {
	if (theImage != NULL)
	{
	    XPutImage(display, win, image_gc, theImage, 0,0,0,0,
		theImage->width, theImage->height);
	    repaint_graphics(win);
	}
	tmpimg = find_previous_img(tmpimg);
    }
    fill_glob(savimg);   /* restore current image */
}


/************************************************************/
/*                                                          */
/* WAIT_FOR_EVENT waits for the next user action.           */
/*                                                          */
/* Arguments:                                               */
/*   pptr:  returns pointer to TTYINPUT string              */
/*   what:  returns indication of what user action occurred */
/*      Button1                                             */
/*      Button2                                             */
/*      Button3                                             */
/*      TTYINPUT (from keyboard)                            */
/*      POINTERMOTION                                       */
/*    x, y:  final position of pointer in WIN coords.  Will */
/*           be constrained to visible screen.              */
/*           If the pointer device returns relative values  */
/*           (dx and dy), then x and y must have            */
/*           legitimate values on input.                    */
/*    dx, dy:  pointer device motion.  dy is positive in    */
/*             up direction.                                */
/*    nowait: true specifies to do any pending X events,    */
/*            but to not hang on user input                 */
/*                                                          */
/************************************************************/

void
wait_for_event(pptr, what, x, y, dx, dy, nowait)
char *pptr;     /* place to store keyboard input line */
int *what;      /* what ended event wait:        */
int *x, *y;
int *dx, *dy;
int nowait;
{
    struct img *tmpimg, *savimg;
    int count, length;
    int bufsize = 20;
    char *cmdv[10], buffer[20], *pptr_tmp, *first_newline;
    int n_read, bytes_wanted;
    int done;
    XEvent report;
    KeySym keysym;
    XComposeStatus compose;
    Window root, child;
    int root_x, root_y;
    unsigned int keys_buttons;
    int last_x, last_y;
    int dummyx, dummyy;
    fd_set fdset;
    static int hist_is_exposed = 0;
    static int wedge_is_exposed = 0;

    /* get events */
    *pptr = '\0';
    *what = 0;
    *dx = 0;
    *dy = 0;

    done = 0;
    while (!done)
    {
	if ((graphics) && (XPending(display)))
	{
	    XNextEvent(display, &report);
	}
	else
	{
	    if (nowait)
		return;

	    /* wait for either stdin or X server event */
	    FD_ZERO(&fdset);
	    if (accepting_stdin)
		FD_SET(ttyfd, &fdset);
	    if (graphics)
		FD_SET(xserver_fd, &fdset);
	    add_callbacks(&fdset);

	    if (select(FD_SETSIZE, &fdset,
		(fd_set *) 0,(fd_set *) 0,(struct timeval *) 0)  < 0)
	    {
		if (errno != EINTR)
		{
		    sperror("select error", errno);
		}
		continue;
	    }
	    if ((graphics) && (FD_ISSET(xserver_fd, &fdset)))
	    {
		XNextEvent(display, &report);
	    }
	    else if (FD_ISSET(ttyfd, &fdset))
	    {
		if (save_line[0] != '\0')    /* if some of last read is left */
		{
		    strcpy(pptr, save_line);
		    bytes_wanted = MAXLINE -1 - strlen(pptr);
		    pptr_tmp = pptr + strlen(pptr);
		}
		else
		{
		    bytes_wanted = MAXLINE-1;
		    pptr_tmp = pptr;
		}

		n_read = read(ttyfd, pptr_tmp, bytes_wanted);
		if (n_read == 0)
		{
		    /* EOF on stdin means we're all done */
		    strcpy(pptr, "unix (simulated)");
		}
		else 
		{
		    *(pptr_tmp + n_read) = '\0'; /* terminate the string */
		    first_newline = strchr(pptr, '\n');
		    if (first_newline == NULL)
		    {
			/* no newline - save all of it  */
			strcpy(save_line, pptr); 
			*pptr = '\0';  /* make like there's nothing here */
		    }
		    else
		    {
			strcpy(save_line, first_newline+1);  /* save the tail */
			*first_newline = '\0';  /* change newline to null */
		    }
		}
		report.type = TTYINPUT;
	    } 
	    else
	    {
		/* some forked job now is sending back output */
		    do_callback(&fdset);
		    continue;
	    }
	}

	if (debug)
	{
	    if (report.type == TTYINPUT)
	    {
		fprintf(debug_file, "skyview:  TTYINPUT event received\n");
		fprintf(debug_file, "  RBH the TTYINPUT is %s\n", pptr);
	    }
	    else
		fprintf(debug_file, 
		    "skyview:  %s event received   win = 0x%lx   %s\n",
		    event_names[report.type], report.xany.window, 
		    report.xany.send_event?"SEND_EVENT":"");
	}

	switch (report.type)
	{
	    case Expose:
		if (debug)
		{
		    fprintf(debug_file,
	    "   expose:  win = 0x%lx x = %d y = %d width = %d height = %d\n", 
		    report.xexpose.window,
		    report.xexpose.x, report.xexpose.y, 
		    report.xexpose.width, report.xexpose.height);
		}
#ifdef NOTDEF
		/* get rid of all other Expose events for this window */
		while (XCheckTypedWindowEvent(display, report.xexpose.window,
		    Expose, &report))
		{
		    if (debug)
		    {
		    printf("  another expose event via XCheckTypedWEvent\n");
		    printf("   expose:  x = %d ", report.xexpose.x);
		    printf(" y = %d ", report.xexpose.y);
		    printf(" width = %d ", report.xexpose.width);
		    printf(" height = %d\n", report.xexpose.height);
		    }
		}
#endif /* NOTDEF */
		    if (report.xexpose.window == wedge_win)
		    {
			wedge_expose(0);
			wedge_is_exposed = 1;
		    }
		    else if (report.xexpose.window == hist_win)
		    {
			hist_expose();
			hist_is_exposed = 1;
		    }
		    else if (report.xexpose.window == sc_win)
		    {
			sc_expose();
			done = 1;
		    }
		    else if (report.xexpose.window == mag_win)
		    {
			mag_expose();
		    }
		    else if (report.xexpose.window == sicon_win)
		    {
			sicon_expose();
		    }
		    else
		    {
			savimg = curr_img;   /* save current image */
			tmpimg = find_WIN(report.xexpose.window);
			if (tmpimg != NULL)
			{
			    /* check if this window needs zooming */
			    if (do_zoom_flag == win)
			    {
				do_zoom_pan(1, &win_zoom, &dummyx, &dummyy,0);
				XPutImage(display, win, image_gc, theImage, 
				    0, 0, 0, 0, width_zoomed, height_zoomed);
				do_zoom_flag = 0;
			    }
			    else
			    {
				if (hold_exposes == win) /* if we're waiting */
							/* for a send_event */
				{
				    if (report.xexpose.send_event == 1) 
				    {
					/* this is what we're waiting for */
					hold_exposes = 0;
				    }
				    else
				    {
					/* discard this partial expose */
					if (debug)
					    fprintf(debug_file, 
					    "expose: discarding this expose\n");
					if (savimg != NULL)
					{
					    if (debug)
						fprintf(debug_file, 
						"expose: restoring frame\n");
					    fill_glob(savimg);
					}
					break;
				    }
				}
				XPutImage(display, win, image_gc, theImage, 
				    report.xexpose.x,
				    report.xexpose.y,
				    report.xexpose.x,
				    report.xexpose.y,
				    report.xexpose.width,
				    report.xexpose.height);
			    }
			    if (hist_win)
				hist_expose();

			    /* if no more exposes coming */
			    if (report.xexpose.count == 0)
				repaint_graphics(win);

			    if (savimg != NULL)
			    {
				if (debug)
				    fprintf(debug_file, 
				    "expose: restoring frame.\n");
				fill_glob(savimg);   /* restore current image */
			    }
			}
		    }
		break;
	    case ConfigureNotify:
		/* window has been resized, change width and height */
		if (debug)
		{
		    fprintf(debug_file,
	"   configure:  win = 0x%lx  x = %d y = %d width = %d height = %d\n", 
			report.xconfigure.window,
			report.xconfigure.x, report.xconfigure.y,
			report.xconfigure.width, report.xconfigure.height);
		}

		if (report.xconfigure.window == wedge_win)
		{
		    wedge_configure((unsigned) report.xconfigure.width,
				    (unsigned) report.xconfigure.height);

		    /* now if this is a reduction in the size of an existing  */
		    /* window, we will get no expose and must repaint it here */
		    if (wedge_is_exposed)
		    {
			/* see if an expose is coming */
			if (XCheckTypedWindowEvent(display, wedge_win, 
			    Expose, &report))
			{
			    /* handle the expose normally */
			    XPutBackEvent(display, &report);
			}
			else
			{
			    /* no expose is coming - repaint it here */
			    wedge_expose(0);
			}
		    }
		}
		else if (report.xconfigure.window == hist_win)
		{
		    hist_configure((unsigned) report.xconfigure.width,
				   (unsigned) report.xconfigure.height);

		    /* now if this is a reduction in the size of an existing  */
		    /* window, we will get no expose and must repaint it here */
		    if (hist_is_exposed)
		    {
			/* see if an expose is coming */
			if (XCheckTypedWindowEvent(display, hist_win, 
			    Expose, &report))
			{
			    /* handle the expose normally */
			    XPutBackEvent(display, &report);
			}
			else
			{
			    /* no expose is coming - repaint it here */
			    hist_expose();
			}
		    }
		}
		else if (report.xconfigure.window == sc_win)
		{
		    sc_configure((unsigned) report.xconfigure.width,
				 (unsigned) report.xconfigure.height);
		}
		else if (report.xconfigure.window == mag_win)
		{
		    mag_configure((unsigned) report.xconfigure.width,
				  (unsigned) report.xconfigure.height);
		}
		else if (report.xconfigure.window == sicon_win)
		{
		    sicon_configure((unsigned) report.xconfigure.width,
				    (unsigned) report.xconfigure.height);
		}
		else
		{
		    savimg = curr_img;   /* save current image */
		    tmpimg = find_WIN(report.xconfigure.window);
		    if (tmpimg != NULL)
		    {
			if ((win_zoom > 1.0001) || (win_zoom < .99999) ||
			    (win_x_offset != 0) || (win_y_offset != 0))
			{
			    /* if larger, redo the zoomed image */
			    if ((report.xconfigure.width > win_width) ||
			    (report.xconfigure.height > win_height))
			    {
				win_width = report.xconfigure.width;
				win_height = report.xconfigure.height;
				do_zoom_pan(1, &win_zoom, &dummyx, &dummyy, 0);
			    }
			}
			win_width = report.xconfigure.width;
			win_height = report.xconfigure.height;
#ifdef NOTDEF
			if ((graphics_window) && (filnam[0] == '\0'))
			{
			    pixels = win_width;
			    lines = win_height;
			    maxx = pixels -1;
			    maxy = lines - 1;
			    height_org = win_height;
			}
#endif /* NOTDEF */
			fill_img(tmpimg); /* force stuff into img struct */

			if (savimg != NULL)
			{
			    if (debug)
				fprintf(debug_file, 
				"configure: restoring frame\n");
			    fill_glob(savimg);   /* restore current image */
			}
		    }
		}
		break;
	    case MapNotify:
		savimg = curr_img;   /* save current image */
		tmpimg = find_WIN(report.xmap.window);
		if (tmpimg != NULL)
		{
		    mapped = 1;
		    /* force into img structure */
		    if (tmpimg->frameptr != 0)
			tmpimg->frameptr->mapped = 1;

		    if (savimg != NULL)
		    {
			if (debug)
			    fprintf(debug_file, 
			    "mapnotify: restoring frame\n");
			fill_glob(savimg);   /* restore current image */
		    }
		}
		break;
	    case UnmapNotify:
		if (report.xunmap.window == hist_win)
		{
		    hist_is_exposed = 0;
		}
		else if (report.xunmap.window == wedge_win)
		{
		    wedge_is_exposed = 0;
		}
		else
		{
		    savimg = curr_img;   /* save current image */
		    tmpimg = find_WIN(report.xunmap.window);
		    if (tmpimg != NULL)
		    {
			mapped = 0;
			/* force into img structure */
			if (tmpimg->frameptr != 0)
			    tmpimg->frameptr->mapped = 0;

			if (savimg != NULL)
			{
			    if (savimg == tmpimg)
			    {
				/* (don't restore this unmapped image) */
				curr_img = NULL;
			    }
			    else
			    {
				if (debug)
				    fprintf(debug_file, 
				    "unmapnotify: restoring frame\n");
				fill_glob(savimg);   /* restore current image */
			    }
			}
		    }
		}
		break;
	    case ButtonPress:
		cur_window = report.xbutton.window;

		*x = report.xbutton.x;
		*y = report.xbutton.y;

		*what = report.xbutton.button;
		button_down = TRUE;

#ifdef ALADIN
/*$$$$$$$$$$$$$$$ Houri Ziaeepour 26 Aug. 1995 $$$$$$$$$$$$$$$$$*/

                XSendEvent (display, cur_window, FALSE, ButtonReleaseMask,
                   &report);

/*$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$*/
#endif /* ALADIN */

		done = 1;
		break;
	    case ButtonRelease:
		button_down = FALSE;
		break;
	    case KeyPress:

		count = XLookupString(&report.xkey, buffer, bufsize, &keysym,
				      &compose);

		buffer[count] = '\0';

		if (cmdline_mode)
		{
		    if ((keysym == XK_q) || (keysym == XK_Q))
			exit(0);
		}

		if ((s0_server_mode) || (s1_server_mode))
		    break;   /* ignor input typed into windows */
		cur_window = report.xkey.window;
		if (win != report.xkey.window)
		    tmpimg = find_WIN(report.xkey.window);

		/* print char and store it   done if CR */


		if (imio_ok)
		{
		    /* this is an IRAF window */
		    /*   either send it to IRAF or throw it away */
		    if ((imio_wants_char) && (buffer[0] != '\0'))
		    {
			/* send it to IRAF */
			cursor_callback(report.xkey.x, report.xkey.y, buffer[0]);
		    }
		    break;
		}

		if ((keysym == XK_Return) || (keysym == XK_KP_Enter) ||
		    (keysym == XK_Linefeed))
		{
		    putchar('\n');
		    *what = TTYINPUT;
		    done = 1;
		    break;
		}
		else if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
		    || ((keysym >= XK_space) && (keysym <= XK_asciitilde))
		    || ((keysym >= XK_F1) && (keysym <= XK_F35)))
		{
		    if (strlen(pptr) + strlen(buffer) >= (size_t) MAXLINE)
			XBell(display, 100);
		    else
		    {
			strcat(pptr, buffer);
			fputs(buffer, stdout);
			fflush(stdout);
		    }
		}
		else if ((keysym == XK_BackSpace) || (keysym == XK_Delete))
		{
		    if ((length = strlen(pptr)) > 0)
		    {
			pptr[length - 1] = '\0';
			putchar('\b');
			fflush(stdout);
		    }
		    else
			XBell(display, 100);
		}
		break;
	    case MotionNotify:
		if ((mag_win) || (want_motion))
		{
		    while (XCheckMaskEvent(display, PointerMotionMask, 
			&report));
		    if ((report.xmotion.window != win) &&
			(report.xmotion.window != grabbed_window))
		    {
			tmpimg = find_WIN(report.xmotion.window);
			if (tmpimg == NULL)
			    break;  /* the window is gone - do nothing */
		    }
		    last_x = *x;
		    last_y = *y;
		    if (!XQueryPointer(display, report.xmotion.window,
			&root, &child, &root_x, &root_y, 
			x, y, &keys_buttons))
		    {
			if (debug)
			    fprintf(debug_file, 
				"MotionNotify:  pointer is in another screen\n");
		    }
		    else
		    {
			if ((mag_win) && ( !(keys_buttons & ShiftMask)))
			    mag_motion(report.xmotion.window, *x, *y);
			if (want_motion)
			{
			    *dx = *x - last_x;
			    *dy = last_y - *y;  /* positive in up direction */
			    *what = POINTERMOTION;
			    done = 1;
			}
		    }
		}
		break;
	    case DestroyNotify:
		break;
	    case ClientMessage:
		if ( report.xclient.data.l[0] == wm_delete_window) 
		{
		    if (debug)
			fprintf(debug_file, "got the delete message\n");
		    if (report.xclient.window == wedge_win)
		    {
			wedge_destroy();
			wedge_is_exposed = 0;
		    }
		    else if (report.xclient.window == hist_win)
		    {
			hist_destroy();
			hist_is_exposed = 0;
		    }
		    else if (report.xclient.window == sc_win)
		    {
			sc_destroy();
		    }
		    else if (report.xclient.window == mag_win)
		    {
			mag_destroy();
		    }
		    else if (report.xclient.window == sicon_win)
		    {
			sicon_destroy();
		    }
		    else
		    {
			cur_window = report.xclient.window;
			cmdv[0] = "erase";
			erase(1, cmdv);
		    }
		}
		break;
	    case TTYINPUT:
		cur_window = 0;
		*what = TTYINPUT;
		done = 1;
		break;
	    default:
		if (debug)
		{
		    fprintf(debug_file, "  Unhandled event: %s  window = 0x%lx\n",
		    event_names[report.type], report.xany.window);
		}
		break;
	}  /* end switch */
    }  /* end while */
}

void
repaint_x(min_x, max_x, min_y, max_y)
int min_x, max_x, min_y, max_y;  /* SC coordinates */
{
    int width1, height1, minx1, miny1, maxx1, maxy1;
    int i_coord[2];
    int dummyx, dummyy;
    XEvent report;

    i_coord[0] = min_x;
    i_coord[1] = min_y;
    SC_to_WIN(i_coord);
    minx1 = i_coord[0];
    miny1 = i_coord[1];

    i_coord[0] = max_x;
    i_coord[1] = max_y;
    SC_to_WIN(i_coord);
    maxx1 = i_coord[0];
    if (i_coord[1] > miny1)
    {
	maxy1 = i_coord[1];
    }
    else
    {
	maxy1 = miny1;
	miny1 = i_coord[1];
    }

    /* move from center of pixel to outer edge  */
    minx1 = minx1 - win_zoom / 2 + .5;
    miny1 = miny1 - win_zoom / 2 + .5;
    maxx1 = maxx1 + win_zoom / 2 - .5;
    maxy1 = maxy1 + win_zoom / 2 - .5;

    /* clip to window boundaries */
    if (minx1 < 0)
	minx1 = 0;
    if (miny1 < 0)
	miny1 = 0;

    width1 = maxx1 - minx1 + 1;
    height1 = maxy1 - miny1 + 1;

    /* check if this window is zoomed or panned  */
    if ((win_zoom > 1.0001) || (win_zoom < .99999) ||
	(win_x_offset != 0) || (win_y_offset != 0))
    {
	/* future:  do_zoom_pan should only have to work on affected area */
	do_zoom_pan(1, &win_zoom, &dummyx, &dummyy, 0);
	/* now queue up a fake expose event (instead of doing XPutImage) */
	report.type = Expose;
	report.xexpose.window = win;
	report.xexpose.width = width1;
	report.xexpose.height = height1;
	report.xexpose.x = minx1;
	report.xexpose.y = miny1;
	report.xexpose.count = 0;
	XSendEvent(display, win, False, ExposureMask, &report);
    }
    else
    {
	XPutImage(display, win, image_gc, theImage, 
	    minx1, miny1, minx1, miny1, width1, height1);
	repaint_graphics(win);
    }
}


void
do_sync()
{
    int dummy_x, dummy_y; 
    int what;
    char comm[20];
    int dummy_dx, dummy_dy;

    if (!graphics)
	return;

    dummy_x = 0;   /* keep purify happy */
    dummy_y = 0;   /* keep purify happy */
    if (sync_mode)
    {
	XSync(display, False);   /* make sure that all events are queued */
	wait_for_event(comm, &what, &dummy_x, &dummy_y, 
	    &dummy_dx, &dummy_dy, 1);  /* nowait */
	XSync(display, False); /* make sure that XPutImage etc is complete */
    }
    else
    {
	/* handle any X events that are already queued */
	while (XPending(display))
	    wait_for_event(comm, &what, &dummy_x, &dummy_y, 
		&dummy_dx, &dummy_dy, 1);  /* nowait */
    }
}

void
sync_command(argc, argv)
int argc;
char **argv;
{
    if (!graphics)
	return;

    if (argc == 2)
    {
	if (strncmp(argv[1], "off", 2) == 0)
	{
	    sync_mode = FALSE;
	}
	if (strcmp(argv[1], "on") == 0)
	{
	    sync_mode = TRUE;
	}
    }
    if(server_mode == FALSE)
	printf("sync is %s\n", sync_mode?"on":"off");
    else
    {
	srv_string("sync", sync_mode?"on":"off");
    }
}


void
ball_setup()
{
}

void
flush_graphics()
{
}

void
set_overlay()
{
}

void
set_clt(r, g, b)
unsigned char *r, *g, *b;
{
    int i, maxcolor, this_def;
    int startat = 0;
    int len = 256;

    if (!graphics)
    {
        if(fixed_ct == TRUE)
        {
	    for(i = 0; i < ncolors; i++)
	    {
	        exact_defs[i].red =   r[i];
	        exact_defs[i].green = g[i];
	        exact_defs[i].blue =  b[i];
	    }

	    return;
        }
        for (i = 0; i < 256; i++)
        {
	    this_def = (240 * i) / 255;
	    exact_defs[this_def].red =   r[i];
	    exact_defs[this_def].green = g[i];
	    exact_defs[this_def].blue =  b[i];
        }
	return;
    }

    if(fixed_ct == TRUE)
    {
	for(i = 0; i < ncolors; i++)
	{
	    exact_defs[i].red =   r[i] << 8;
	    exact_defs[i].green = g[i] << 8;
	    exact_defs[i].blue =  b[i] << 8;
	}
    }
    else
    {
	maxcolor = startat + len;
	for (i = startat; i < maxcolor; i++)
	{
	    this_def = (ncolors * i) / 255;
	    exact_defs[this_def].red =   r[i] << 8;
	    exact_defs[this_def].green = g[i] << 8;
	    exact_defs[this_def].blue =  b[i] << 8;
	    /* some exact_defs will be stored into more than once */
	}
	/* make sure that zero is not overwritten */
	exact_defs[0].red =   r[0] << 8;
	exact_defs[0].green = g[0] << 8;
	exact_defs[0].blue =  b[0] << 8;
    }

    if ((visual->class == PseudoColor) || (visual->class == DirectColor))
	XStoreColors (display, theCmap, exact_defs, ncolors);

    wedge_expose(1);   /* repaint the wedge if it exists */
}

/*ARGSUSED*/
void
map_for_AOI(this_plane)
int this_plane;
{
}

void
map_normal()
{
}

void
write_AOI(xmin, ymin, xmax, ymax, inptr)
int xmin,ymin,xmax,ymax;
int *inptr;
{
    int nbr;
    int i;
    unsigned int imgheight;
    int cur_line;

    cur_line = ymin;
    nbr = (xmax - xmin + 1) * (ymax - ymin + 1);

    /*
    imgheight = maxy - miny + 1;
    */
    imgheight = height_org;

    if (plane == 'a')
    {
	/* move pixels into pixel buffer */
	if (depth > 16)
	{
	    INT32 *ptr32;

	    ptr32 = (INT32 *) (image_org + 
		((imgheight - cur_line - 1) * width_org + xmin) * 4);
					 /* each pixel is 4 times as long */
	    for (i = 0; i < nbr; i++)
	    {
		if (*inptr == -1)
		{
		    *ptr32++ = blank_pixval;
		    inptr++;
		}
		else
		    *ptr32++ = pixval[*inptr++];
	    }
	}
	else if (depth > 8)
	{
	    INT16 *ptr16;

	    ptr16 = (INT16 *) (image_org + 
		((imgheight - cur_line - 1) * width_org + xmin) * 2);
					 /* each pixel is 2 times as long */
	    for (i = 0; i < nbr; i++)
	    {
		if (*inptr == -1)
		{
		    *ptr16++ = blank_pixval;
		    inptr++;
		}
		else
		    *ptr16++ = pixval[*inptr++];
	    }
	}
	else
	{
	    /* depth == 8 */
	    INT8 *ptr8;

	    ptr8 = image_org + 
		(imgheight - cur_line - 1) * width_org + xmin;
	    for (i = 0; i < nbr; i++)
	    {
		if(fixed_ct == TRUE)
		    *ptr8++ = colors[*inptr++];
		else if (*inptr == -1)
		{
		    *ptr8++ = blank_pixval;
		    inptr++;
		}
		else
		    *ptr8++ = pixval[*inptr++];
	    }
	}
    }
    else
    {

	if (depth > 16)
	{
	    INT32 *ptr32, set_mask, clear_mask;
	    switch(plane)
	    {
		case 'r':
		    set_mask = red_mask;
		    break;
		case 'g':
		    set_mask = green_mask;
		    break;
		case 'b':
		    set_mask = blue_mask;
		    break;

	    }
	    clear_mask = ~set_mask;
	    ptr32 = (INT32 *) (image_org + 
		((imgheight - cur_line - 1) * width_org + xmin) * 4);
					 /* each pixel is 4 times as long */
	    for (i = 0; i < nbr; i++)
	    {
		if (*inptr == -1)
		{
		*ptr32 = (set_mask & blank_pixval) | (clear_mask & *ptr32);
		inptr++;
		ptr32++;
		}
		else
		{
		*ptr32 = (set_mask & pixval[*inptr++]) | (clear_mask & *ptr32);
		ptr32++;
		}
	    }
	}
	else if (depth > 8)
	{
	    INT16 *ptr16, set_mask, clear_mask;
	    switch(plane)
	    {
		case 'r':
		    set_mask = red_mask;
		    break;
		case 'g':
		    set_mask = green_mask;
		    break;
		case 'b':
		    set_mask = blue_mask;
		    break;

	    }
	    clear_mask = ~set_mask;
	    ptr16 = (INT16 *) (image_org + 
		((imgheight - cur_line - 1) * width_org + xmin) * 2);
					 /* each pixel is 2 times as long */
	    for (i = 0; i < nbr; i++)
	    {
		if (*inptr == -1)
		{
		*ptr16 = (set_mask & blank_pixval) | (clear_mask & *ptr16);
		inptr++;
		ptr16++;
		}
		else
		{
		*ptr16 = (set_mask & pixval[*inptr++]) | (clear_mask & *ptr16);
		ptr16++;
		}
	    }
	}
	else
	{
	    /* depth == 8 */
	    error1("error in paint:  8-bit display cannot be mapped");
#ifdef NOTDEF
	    INT8 *ptr8, set_mask, clear_mask;
	    switch(plane)
	    {
		case 'r':
		    set_mask = red_mask;
		    break;
		case 'g':
		    set_mask = green_mask;
		    break;
		case 'b':
		    set_mask = blue_mask;
		    break;

	    }
	    clear_mask = ~set_mask;
	    ptr8 = image_org + 
		(imgheight - cur_line - 1) * width_org + xmin;
	    for (i = 0; i < nbr; i++)
	    {
		if (*inptr == -1)
		{
		*ptr8 = (set_mask & blank_pixval) | (clear_mask & *ptr8);
		inptr++;
		ptr8++;
		}
		else
		{
		*ptr8 = (set_mask & pixval[*inptr++]) | (clear_mask & *ptr8);
		ptr8++;
		}
	    }
#endif /* NOTDEF */
	}
    }
}

void
map(argc, argv)
int argc;
char **argv;
{
    if(argc == 1)
    {
	if(server_mode == FALSE)
	{
	    printf("current mapping: %c\n", plane);
	    return;
	}
    }

    if(argc == 2)
    {
	if ((depth <= 8) && (argv[1][0] != 'a'))
	{
	    error1("map function not available on 8 plane graphics systems");
	    return;
	}
	switch(argv[1][0])
	{
	case 'a':
	case 'r':
	case 'g':
	case 'b':
	    plane = argv[1][0];
	    break;
	}
    }

    if(debug)
	fprintf(debug_file, "map:  plane = %c\n", plane);

    if (server_mode == TRUE)
    {
	sprintf(server_str, "%c", plane);
	srv_string("mapped_plane", server_str);
    }

    image_setup(plane);
}

/* ************************************************************ */
/*  lookup_RGB                                                  */
/*  finds an exactly matching entry in the ov_colors database   */
/*                                                              */
/*  entry:  XColor                                              */
/*                                                              */
/*  returns the GC, or NULL if not found                        */
/* ************************************************************ */

GC lookup_RGB(user_color)
XColor *user_color;
{
    int i;

    for (i = 0; i < ov_colors_n; i++)
    {
	if ((user_color->red   == ov_colors[i].color_spec.red)   &&
	    (user_color->green == ov_colors[i].color_spec.green) &&
	    (user_color->blue  == ov_colors[i].color_spec.blue))
	{
	    return(ov_colors[i].gc);
	}
    }
    return(NULL);
}

GC delete_RGB(user_color)  /* delete one color - return the GC of that color */
XColor *user_color;
{
    int i;
    unsigned long pixel_delete[1];
    GC gc_delete;

    for (i = 0; i < ov_colors_n; i++)
    {
	if ((user_color->red   == ov_colors[i].color_spec.red)   &&
	    (user_color->green == ov_colors[i].color_spec.green) &&
	    (user_color->blue  == ov_colors[i].color_spec.blue))
	break;
    }
    if (i == ov_colors_n)
	return (NULL);    /* not found */

    if ((i < ov_colors_save) || (gc_user == ov_colors[i].gc))
	return (ov_colors[i].gc);    /* one of our permanent colors or */
				  /* the currently selected user color */

    if (debug)
	fprintf(debug_file, "delete_RGB:  deleting GC 0x%p\n", 
	    (void *) ov_colors[i].gc);
    XFreeGC(display, ov_colors[i].gc);
    /* now compress out the RGB entry */
    gc_delete = ov_colors[i].gc;
    pixel_delete[0] = ov_colors[i].pixel;
    for (; i < (ov_colors_n - 1); i++)
    {
	ov_colors[i].color_spec.red  = ov_colors[i+1].color_spec.red;
	ov_colors[i].color_spec.green = ov_colors[i+1].color_spec.green;
	ov_colors[i].color_spec.blue = ov_colors[i+1].color_spec.blue;
	ov_colors[i].gc              = ov_colors[i+1].gc;
	ov_colors[i].pixel           = ov_colors[i+1].pixel;
	strcpy(ov_colors[i].name, ov_colors[i+1].name);
    }
    ov_colors_n--;

    /* now deallocate color if not in use */
    for (i = 0; i < ov_colors_n; i++)
    {
	if (ov_colors[i].pixel == pixel_delete[0])
	    break;
    }
    if (i == ov_colors_n)
    {
	/* not found - delete it */
	if (debug)
	    fprintf(debug_file, "delete_RGB:  deallocating pixel %ld\n", 
		pixel_delete[0]);
	XFreeColors(display, theCmap, pixel_delete, 1, 0L);
    }
    return(gc_delete);
}

void
delete_all_RGB()
{
    int i, new_n;
    XColor user_color;
    GC tmp_gc;

    new_n = ov_colors_save;

    i = ov_colors_save;
    while (i < ov_colors_n)
    {
	if (gc_user == ov_colors[i].gc)
	{
	    /* this color is still in use - keep it */
	    new_n++;
	    i++;
	}
	else
	{
	    user_color.red = ov_colors[i].color_spec.red;
	    user_color.green = ov_colors[i].color_spec.green;
	    user_color.blue = ov_colors[i].color_spec.blue;
	    tmp_gc = delete_RGB(&user_color);
	}
    }
    ov_colors_n = new_n;
}

int
set_overlay_color(color)
char *color;
{
    XColor user_color;
    GC tmp_gc;
    unsigned long valuemask;
    XGCValues values;
    int i;

    if (strcmp(color, "default") == 0)
    {
	gc_user = gc_white;
	use_default_color = 1;
	return(TRUE);
    }

    if (strcmp(color, "transparent") == 0)
    {
	gc_user = gc_transparent;
	use_default_color = 0;
	return(TRUE);
    }

    if (!graphics)
    {
	gc_user = get_rgb(color);

	if(gc_user == NULL)
	    return(FALSE);
	 
	use_default_color = FALSE;
	this_pixval = dmpcolor(gc_user);
	return(TRUE);
    }

    /* look for existing name in RGB database */
    for (i = 0; i < ov_colors_n; i++)
    {
	if (strcmp(color, ov_colors[i].name) == 0)
	{
	    /* found it */
	    gc_user = ov_colors[i].gc;
	    use_default_color = FALSE;
	    this_pixval = ov_colors[i].pixel;
	    return(TRUE);
	}
    }

    if (!XParseColor(display, theCmap, color, &user_color))
    {
	sprintf(server_str, 
	    "color %s is not in X windows color list", color);
	error1(server_str);
	return(FALSE);
    }

    /* look for existing entry in RGB database */
    for (i = 0; i < ov_colors_n; i++)
    {
	if ((user_color.red   == ov_colors[i].color_spec.red)   &&
	    (user_color.green == ov_colors[i].color_spec.green) &&
	    (user_color.blue  == ov_colors[i].color_spec.blue))
	{
	    /* found it */
	    gc_user = ov_colors[i].gc;
	    use_default_color = FALSE;
	    this_pixval = ov_colors[i].pixel;
	    return(TRUE);
	}
    }

    if (!XAllocColor(display, theCmap, &user_color))
    {
	sprintf(server_str, 
	    "unable to allocate color %s  - colormap is full", color);
	error1(server_str);
	return(FALSE);
    }

    /* create GC for user specified overlay color */
    values.foreground = user_color.pixel;
    values.font = default_font_id;
    valuemask = GCForeground | GCFont;
    tmp_gc = XCreateGC(display, dummy_window, 
	valuemask, &values);
    if (tmp_gc == NULL)
    {
	sprintf(server_str, "unable to allocate GC for overlay color %s", color);
	error1(server_str);
	return(FALSE);
    }

    ov_colors[ov_colors_n].color_spec.red   = user_color.red;
    ov_colors[ov_colors_n].color_spec.green = user_color.green;
    ov_colors[ov_colors_n].color_spec.blue  = user_color.blue;
    ov_colors[ov_colors_n].pixel = user_color.pixel;
    ov_colors[ov_colors_n].gc = tmp_gc;
    this_pixval = ov_colors[ov_colors_n].pixel;
    strcpy(ov_colors[ov_colors_n].name, color);
    if (debug)
	fprintf(debug_file, 
	"set_overlay_color: adding %s to RGB database  pixel = %ld  GC = 0x%p\n", 
	color, user_color.pixel, (void *) tmp_gc);
    ov_colors_n++;

    gc_user = tmp_gc;
    use_default_color = FALSE;
    return(TRUE);
}

int
set_blank_color(color)
char *color;
{
    if (strcmp(color, "default") == 0)
    {
	blank_pixval = default_blank_pixval;
	return(TRUE);
    }
    else
    {
	if (set_overlay_color(color))
	{
	    blank_pixval = this_pixval;
	    return(TRUE);
	}
	else
	    return(FALSE);
    }
}

void
save_overlay_color()
{
    save_u_d_c = use_default_color;
    save_gc_user = gc_user;
}

void
restore_overlay_color()
{
    use_default_color = save_u_d_c;
    gc_user = save_gc_user;
}

int
good_overlay_color(color)    /* tests if specified color is a good X color */
char *color;
{
    XColor user_color;
    int i;
    GC gc;

    if(!graphics)
    {
	gc = get_rgb(color);

	if(gc == NULL)
	    return(0);
        else
	    return(1);
    }

    if (strcmp(color, "transparent") == 0)
	return(1);

    /* look for existing name in RGB database */
    for (i = 0; i < ov_colors_n; i++)
    {
	if (strcmp(color, ov_colors[i].name) == 0)
	{
	    /* found it */
	    return(1);
	}
    }

    return(XParseColor(display, theCmap, color, &user_color));
}


/*ARGSUSED*/
void
overlay_color(c)   /* set color (for overlay) */
int c;
{
}

void
overlay_setup(plane1)
int plane1;
{
    if (!use_default_color)
    {
	/* ignor color specified by calling routine */
	/* instead use the color specified by user */
	gc_overlay = gc_user;
	return;
    }

    switch(plane1)
    {
    case 'r':
	gc_overlay = gc_red;
	break;
    case 'g':
	gc_overlay = gc_green;
	break;
    case 'b':
	gc_overlay = gc_blue;
	break;
    case 'w':
	gc_overlay = gc_white;
	break;
    case 'B':
	gc_overlay = gc_black;
	break;
    }
}

/*ARGSUSED*/
void
image_setup(pl)
int pl;
{
    /* This stuff is all commented out */
    /* It didn't work on Gregorich's original 24-bit Rasterops (?) X server */
    /* It might work with newer X servers */
#ifdef UNDEF
    int i;

    /*
    if (depth > 8)
    */
    switch (pl)
    {
	case 'a':
	    XSetPlaneMask(display, image_gc, AllPlanes);
	    plane = pl;
	    break;
	case 'r':
	    XSetPlaneMask(display, image_gc, red_mask);
	    plane = pl;
	    break;
	case 'g':
	    XSetPlaneMask(display, image_gc, green_mask);
	    plane = pl;
	    break;
	case 'b':
	    XSetPlaneMask(display, image_gc, blue_mask);
	    plane = pl;
	    break;
    }
#endif /* UNDEF */
}

void
zoom_setup()
{
}

void
set_motion_notify(wind)
Window wind;
{
    grabbed_window = wind;

    XGrabPointer(display, wind, False, (unsigned) (ButtonPressMask | 
	ButtonReleaseMask | PointerMotionMask | PointerMotionHintMask), 
	GrabModeAsync, GrabModeAsync, None, None, CurrentTime);

    want_motion = 1;
}

void
clear_motion_notify()
{
     XEvent report;

    /* since a ButtonPress stops the grab, wait for the accompanying */
    /*    ButtonRelease so that it won't be delivered to another window */
    if (button_down)
	XWindowEvent(display, grabbed_window, ButtonReleaseMask, &report);
    button_down = FALSE;
    XUngrabPointer(display, CurrentTime);
    XSync(display, False);   /* RBH DEBUG */

    want_motion = 0;
    grabbed_window = 0;
}


void
not_implemented()
{
    error1("feature not implemented in X windows version");
}


int parse_size(geom, width_ptr, height_ptr)
char *geom;
int *width_ptr, *height_ptr;
{
    int parse_result, dummyx, dummyy;

    parse_result = XParseGeometry(geom, &dummyx, &dummyy,
	(unsigned int *) width_ptr, (unsigned int *) height_ptr);
    if ((parse_result & WidthValue) && (parse_result & HeightValue))
	return 1;
    else
	return 0;
}

void
get_center_xy(i_coord)
int i_coord[2];
{
    i_coord[0] = Round((double)win_width / 2);
    i_coord[1] = Round((double)win_height / 2);
    WIN_to_SC(i_coord);
}


void
d_get_center_xy(d_coord)
double d_coord[2];
{
    d_coord[0] = (double)win_width / 2;
    d_coord[1] = (double)win_height / 2;
    dWIN_to_SC(d_coord);
}

int
get_screen_height()     /* return window height in SC coordinates */
{
    return(win_height / win_zoom);
}

int
get_screen_width()     /* return window width in SC coordinates */
{
    return(win_width / win_zoom);
}

int
get_screen_bottom()     /* return bottom edge in SC coordinates */
{
    int i_coord[2];

    i_coord[0] = 0;
    i_coord[1] = win_height - 1;
    WIN_to_SC(i_coord);
    return(i_coord[1]);
}

int
get_screen_left()     /* return left edge in SC coordinates */
{
    int i_coord[2];

    i_coord[0] = 0;
    i_coord[1] = 0;
    WIN_to_SC(i_coord);
    return(i_coord[0]);
}

/***********************************************************************/
/*                                                                     */
/* SET_CURRENT_WINDOW loads global variables for the "current" window. */
/*   The "current window" must be mapped (not iconified), and          */
/*   is defined as:				                       */
/*                                                                     */
/*	1.  The window last receiving input.      		       */
/*	else (last input was to tty or wedge window) 		       */
/*	2.  The current window (global variables are not changed)      */
/*	else (win == 0, meaning that there is no current window)       */
/*	3.  The most recently painted window			       */
/*	else (there are no images on the screen)		       */
/*	4.  win = 0						       */
/*                                                                     */
/*      Returns 1 if able to find a window.                            */
/*              0 otherwise.                                           */
/*                                                                     */
/***********************************************************************/

int
set_current_window()
{
    struct img *tmpimg;

    if (cur_window != 0)
    {
	if (cur_window == win)
	{
	    if (is_viewable())
		return(1);                     /* case 1 */
	}
	tmpimg = find_WIN(cur_window);      /* sets curr_img */
	if (tmpimg != NULL)
	{
	    if (is_viewable())
		return(1);          		/* case 1 */
	}
    }

    if (win != 0)
    {
	if (is_viewable())
	    return(1);				/* case 2 */
    }

    tmpimg = find_last_img();
    if (tmpimg != NULL)
    {
	if (is_viewable())
	    return(1);
    }

    return(0);
}

static int
is_viewable()
{
    XWindowAttributes window_attributes;

    if (!graphics)
	return(1);  /* play like everything is viewable */

    if (mapped)
	return(1);

    XGetWindowAttributes(display, win, &window_attributes);
    if (window_attributes.map_state == IsViewable)
	return(1);
    wait_for_windowmgr(TRUE);  /* give it a second chance */
    XGetWindowAttributes(display, win, &window_attributes);
    if (window_attributes.map_state == IsViewable)
	return(1);
    return(0);
}


void
frame_cmd(argc, argv)
int argc;
char *argv[];
{
    struct img *imgp;
    int next_frame;

    if (argc > 1)
    {
	if (strncmp(argv[1], "ad", 2) == 0)
	{
	    frame_advance = TRUE;
	}
	else if (strncmp(argv[1], "noad", 4) == 0)
	{
	    frame_advance = FALSE;
	}
	else
	    getint(argv[1], &frame);   /* get frame number */
	    imgp = peek_FRAME(frame);
	    if (imgp != NULL)
		fill_glob(imgp);
    }
    else if(server_mode == FALSE)
    {
	printf("current frame is %d", image_frame);
	next_frame = frame;
	if (frame_advance)
	{
	    while (peek_FRAME(next_frame) != NULL)
		next_frame++;
	}
	printf("    next frame is %d      frame advance is %s\n", 
	    next_frame, frame_advance?"on":"off");
    }

    if (server_mode == TRUE)
    {
	next_frame = frame;
	if (frame_advance)
	{
	    while (peek_FRAME(next_frame) != NULL)
		next_frame++;
	}
	sprintf(server_str, "%d", image_frame);
	srv_real("current_frame", server_str);
	sprintf(server_str, "%d", next_frame);
	srv_real("next_frame", server_str);
	sprintf(server_str, "%s", frame_advance?"on":"off");
	srv_string("frame_advance", server_str);
    }
}

void
next_frame()
{
    int i;

    i = frame;
    while (peek_FRAME(i) != NULL)
	i++;

    frame = i;
}

/*ARGSUSED*/
void
origin(argc, argv)
int argc;
char *argv[];
{
    not_implemented();
#ifdef NOTDEF
    minx_next = atoi(argv[1]);
    miny_next = atoi(argv[2]);
#endif /* NOTDEF */
}

void
border(argc, argv)
int argc;
char **argv;
{
    int border_flag, status, std_border;
    int border_top, border_bottom, border_left, border_right;
    int border_rem;
    int dummyx, dummyy;

    std_border = 50;
    border_flag = FALSE;

    switch(argc)
    {
	case 2:
	    if (getint(argv[1], &std_border) == FALSE)
		break;;
	    /* if OK, fall thru */
	    /*FALLTHROUGH*/

	case 1:
	    /* set to default border */
	    border_top = std_border;
	    border_bottom = std_border;
	    border_left = std_border;
	    border_right = std_border;
	    border_flag = TRUE;
	    break;

	case 5:
	    status = TRUE;
	    status &= getint(argv[1], &border_top);
	    status &= getint(argv[2], &border_bottom);
	    status &= getint(argv[3], &border_left);
	    status &= getint(argv[4], &border_right);
	    if (status)
		border_flag = TRUE;
	    else
		error1("bad argument to border command");
	    break;
	default:
	    error1("wrong number of arguments to border command");
    }
    if (debug)
    {
	fprintf(debug_file,
	    "border_flag = %d\n    border_top = %d border_bottom = %d",
	    border_flag, border_top, border_bottom);
	fprintf(debug_file, " border_left = %d  border_right = %d\n",
	    border_left, border_right);
    }

    if (border_flag)
    {
	if (!set_current_window())
	{
	    error1("no images currently being displayed");
	    return;
	}
	if (win_zoom > 1.5)
	{
	    /* (top border must be integer number of pixels) */
	    border_rem = border_top % (int) win_zoom;
	    if (border_rem)
	    {
		/* increase to next integer pixel size */
		border_top += win_zoom - border_rem;
	    }
	    /* (left border must be integer number of pixels) */
	    border_rem = border_left % (int) win_zoom;
	    if (border_rem)
	    {
		/* increase to next integer pixel size */
		border_left += win_zoom - border_rem;
	    }
	}

	win_width = width_org * win_zoom + border_left + border_right;
	win_height = height_org * win_zoom + border_top + border_bottom;

	if(graphics)
	{
	   XResizeWindow(display, win, win_width, win_height);
	   XSync(display, False); /* make sure that the expose events arrive */
		    /* ahead of the expose event sent by do_zoom_pan() below */
		   
	}

	win_x_offset = - border_left / win_zoom;
	win_y_offset = - border_top / win_zoom;
	do_zoom_pan(1, &win_zoom, &dummyx, &dummyy, 1);

	/* flag to ignor partial exposes resulting from the XResizeWindow, */
	/* since do_zoom_pan sends an expose event for the whole new window */
	hold_exposes = win;  

	if ((server_mode == TRUE) && (!be_quiet))
	{
	    sprintf(server_str, "%d", border_top);
	    srv_real("border_top", server_str);
	    sprintf(server_str, "%d", border_bottom);
	    srv_real("border_bottom", server_str);
	    sprintf(server_str, "%d", border_left);
	    srv_real("border_left", server_str);
	    sprintf(server_str, "%d", border_right);
	    srv_real("border_right", server_str);
	}

    }
}

void
sleep_cmd(argc, argv)
int argc;
char *argv[];
{
    int n;

    if(server_mode == TRUE)
    {
       error_na("sleep command not allowed in server mode");
       return;
    }

    if (argc > 1)
    {
	if (getint(argv[1], &n))
	{
	    do_sleep((long) n, (long) 0);
	}
    }
}

static void
do_sleep(sec, usec)
long sec,usec;
{
    int dummy_x, dummy_y, dummy_dx, dummy_dy, what;
    char comm[20];
    fd_set fdset;
    long end_sec;
    struct timeval timeout, currenttime;
    int status;

    /* we want the sleep to be interruptable by X events, but */
    /* then we must restart the sleep for the remainder of    */
    /* the desired sleep duration */
    gettimeofday(&currenttime, (struct timezone *) 0);
    end_sec = currenttime.tv_sec + sec;   /* desired end time */

    for (;;)
    {
	if ((sec <= 0) && (usec == 0))
	    return;

	/* wait for either timeout or X server event */
	FD_ZERO(&fdset);
	if (graphics)
	    FD_SET(xserver_fd, &fdset);
	add_callbacks(&fdset);
	timeout.tv_sec = sec;
	timeout.tv_usec = usec;

	status = select(FD_SETSIZE, &fdset, (fd_set *) 0,(fd_set *) 0, &timeout);
	if (status < 0)
	{
	    sperror("select error", errno);
	    continue;
	}
	else if (status == 0)
	{
	    /* got the timeout - sleep is done */
	    return;
	}
	else if (FD_ISSET(xserver_fd, &fdset))
	{
	    /* process the X events */
	    dummy_x = 0;   /* keep purify happy */
	    dummy_y = 0;   /* keep purify happy */
	    while (XPending(display))
	    {
		wait_for_event(comm, &what, &dummy_x, &dummy_y,
		    &dummy_dx, &dummy_dy, 1);  /* nowait */
	    }
	}
	else
	{
	    /* some forked job now is sending back output */
	    do_callback(&fdset);
	}
	/* how much of desired sleep time remains */
	gettimeofday(&currenttime, (struct timezone *) 0);
	sec = end_sec - currenttime.tv_sec;
	usec = 0;
    }
}


/*********************************************************************/
/* 								     */
/* WIN_to_SC                                                         */
/* converts window coordinates to screen coordinates                 */
/* 								     */
/*********************************************************************/

void
WIN_to_SC(i_coord)
int i_coord[2];
{
    double d_coord[2];

    if (debug)
	fprintf(debug_file,
	    "WIN_to_SC: in x=%d, y=%d\n", i_coord[0], i_coord[1]);

    d_coord[0] = i_coord[0];
    d_coord[1] = i_coord[1];
    dWIN_to_SC(d_coord);
    i_coord[0] = Round(d_coord[0]);
    i_coord[1] = Round(d_coord[1]);

    if (debug == TRUE)
	fprintf(debug_file,
	    "WIN_to_SC: out x=%d, y=%d\n", i_coord[0], i_coord[1]);
}

/*********************************************************************/
/* 								     */
/* dWIN_to_SC                                                        */
/* converts window coordinates to screen coordinates                 */
/* (double precision)                                                */
/* 								     */
/*********************************************************************/

void
dWIN_to_SC(d_coord)
double d_coord[2];
{
    if (debug)
	fprintf(debug_file,
	    "dWIN_to_SC: in x=%g, y=%g\n", d_coord[0], d_coord[1]);

    d_coord[0] = (d_coord[0] + 0.5) / win_zoom + win_x_offset - 0.5;
    d_coord[1] = (int) height_org - 1 -  win_y_offset + 0.5 - 
	((d_coord[1] + 0.5) / win_zoom);


    if (debug)
	fprintf(debug_file,
	    "dWIN_to_SC: out x=%g, y=%g\n", d_coord[0], d_coord[1]);
}

/*********************************************************************/
/* 								     */
/* SC_to_WIN                                                         */
/* converts screen coordinates to window coordinates                 */
/* 								     */
/*********************************************************************/
void
SC_to_WIN(i_coord)
int i_coord[2];
{
    double d_coord[2];

    if (debug)
	fprintf(debug_file,
	    "SC_to_WIN: in x=%d, y=%d\n", i_coord[0], i_coord[1]);

    d_coord[0] = i_coord[0];
    d_coord[1] = i_coord[1];
    dSC_to_WIN(d_coord);
    i_coord[0] = Round(d_coord[0]);
    i_coord[1] = Round(d_coord[1]);

    if (debug == TRUE)
	fprintf(debug_file,
	    "SC_to_WIN: out x=%d, y=%d\n", i_coord[0], i_coord[1]);
}

/*********************************************************************/
/* 								     */
/* dSC_to_WIN                                                        */
/* converts screen coordinates to window coordinates                 */
/* (double precision)						     */
/* 								     */
/*********************************************************************/

void
dSC_to_WIN(d_coord)
double d_coord[2];
{
    if (debug)
	fprintf(debug_file,
	    "dSC_to_WIN: in x=%g, y=%g\n", d_coord[0], d_coord[1]);

    /* the 0.5 factors are because the center of the edge pixel is */
    /* exactly 0.5 pixels in from the edge of the window           */
    d_coord[0] = (d_coord[0] - win_x_offset + 0.5) * win_zoom - 0.5;
    d_coord[1] = (height_org - 1 - d_coord[1] - win_y_offset + 0.5) * win_zoom
	- 0.5;

    if (debug)
	fprintf(debug_file,
	    "dSC_to_WIN: out x=%g, y=%g\n", d_coord[0], d_coord[1]);
}
