/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "parse.h"
#include "fits.h"
#include "job.h"
#include "area.h"
#include "im_proto.h"
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <signal.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#define POLYLINE 0
#define FILLAREA 1

extern int debug, graphics, server_mode, ttyfd;
extern int ads_server_mode;
extern char server_str[];
extern int conserve_mem, use_projtype, mouse_is_on, default_u16, reread_on;
extern int u16, imio_on;
extern int pixels, lines, image_frame;
extern char bunit[]; 
extern char hist_filnam[], filnam[], mod_filnam[], fi_name[];
extern double blank, blank_flag, report_blank, cdelt1;
extern int blank_set;
extern double b_zero, bscale;
extern FILE *session, *debug_file;
extern FILE *cmdfile;
extern struct hdrstruct filehdr;
extern int area_number; 
extern int control_c;
extern struct area_struct area[MAX_AREAS];
extern struct img *curr_img;

char ov_color[25] = "default";
static char desired_sex_str[25] = "default";
static char blank_color[25] = "default";
double zap_value;
int zap_blank = TRUE;  /* TRUE means use BLANK value for zapping */
double default_blank;
int default_blank_set = FALSE;

static char *tek_ptr;


/* function prototypes */
#ifdef _NO_PROTO

static void on_off();
static void do_tek();
static void do_post();
static int get_tek_char();
static void flatten_callback();
int fgetc();

#else

static void on_off(int *var, char *str);
static void do_tek(FILE *fd, int getdims, int (*get_a_char)(FILE *));
static void do_post(FILE *fd, int (*get_a_char)(FILE *));
static int get_tek_char(FILE *fd);
static void flatten_callback(struct job_info *job);

#endif /* _NO_PROTO */

int      /* returns 1 if OK, 0 if error */
parse_equinox(p, outstr, equinox_val, equinox_char)
char *p;        /* input */
char *outstr;  /* output - properly formed string (e.g. J2000.0) */
double *equinox_val;  /* output - e.g. 2000.0 */
char *equinox_char;   /* output - upper case J or B */
{
    double dbl_temp;
    char *ptr, eq_letter;

    if ((*p == 'J') || (*p == 'j') ||
	(*p == 'B') || (*p == 'b'))
    {
	eq_letter = toupper(*p);
	p++;
    }
    else
	eq_letter = '\0';

    /* check validity */
    dbl_temp = strtod(p, &ptr);
    if ((ptr == p) || (*ptr != '\0'))
    {
	return(0);
    }
    else
    {
	if (eq_letter == '\0')
	{
	    if (dbl_temp >= 2000.0)
		eq_letter = 'J';
	    else
		eq_letter = 'B';
	}

	*equinox_val = dbl_temp;
	*equinox_char = eq_letter;
	outstr[0] = eq_letter;
	strcpy(&outstr[1], p);
    }
    return(1);
}

void
set(argc, argv)
int argc;
char **argv;
{
    int status;

    if (argc == 1)
    {
	if (server_mode == FALSE)
	{
	   printf("(per image):\n");
	   if (default_blank_set)
	       printf("  blank           %g", default_blank);
	   else
	       printf("  blank           undefined");
	   printf("   (used only if BLANK missing from header)\n");
	   printf("  blank_color     %s\n", blank_color);
	   printf("  u16             %s\n", default_u16?"on":"off");
	   printf("  projtype        %s\n", use_projtype?"on":"off");
	   printf("\n(global):\n");
	   printf("  mouse           %s\n", mouse_is_on?"on":"off");
	   printf("  reread          %s\n", reread_on?"on":"off");
	   printf("  conserve_mem    %s\n", conserve_mem?"on":"off");
	   printf("  color           %s\n", ov_color);
	   printf("  sexigesimal     %s\n", desired_sex_str);
	   if (zap_blank)
	       printf("  zap_value       blank\n");
	   else
	       printf("  zap_value       %g\n", zap_value);
	   printf("  imio            %s\n", imio_on?"on":"off");
	}
	else
	{
	    if (default_blank_set)
	    {
		sprintf(server_str, "%g", default_blank);
		srv_real("blank", server_str);
	    }
	    else
		srv_string("blank", "undefined");
	    srv_string("blank_color",  blank_color);
	    srv_string("u16",  default_u16?"on":"off");
	    srv_string("use_projtype",  use_projtype?"on":"off");
	    srv_string("mouse",  mouse_is_on?"on":"off");
	    srv_string("reread",  reread_on?"on":"off");
	    srv_string("conserve_mem",  conserve_mem?"on":"off");
	    srv_string("overlay_color",  ov_color);
	    srv_string("sexigesimal",  desired_sex_str);
	    if (zap_blank)
		srv_string("zap_value", "blank");
	    else
	    {
		sprintf(server_str, "%g", zap_value);
		srv_real("zap_value", server_str);
	    }
	    srv_string("imio_on",  imio_on?"on":"off");
	}
	return;
    }
    else if (argc == 3)
    {
	switch(cmd(argv[1]))
	{
	case MOUSE:
	    on_off(&mouse_is_on, argv[2]);
	    return;
	case PROJTYPE:
	    on_off(&use_projtype, argv[2]);
	    return;
	case U16:
	    on_off(&default_u16, argv[2]);
	    hist_filnam[0] = '\0';  /* invalidate saved histogram */
	                            /* in case it's 16 bit */
	    return;
	case REREAD:
	    on_off(&reread_on, argv[2]);
	    return;
	case CONSERVE_MEM:
	    on_off(&conserve_mem, argv[2]);
	    return;
	case BLANK_COLOR:
	    save_overlay_color();
	    status = set_blank_color(argv[2]);
	    if (status)
		strcpy(blank_color, argv[2]);
	    restore_overlay_color();
	    return;
	case COLOR:
	    status = set_overlay_color(argv[2]);
	    if (status)
		strcpy(ov_color, argv[2]);
	    return;
	case SEXIGESIMAL:
	    status = set_desired_sex(argv[2]);
	    if (status)
		strcpy(desired_sex_str, argv[2]);
	    return;
	case ZAP:
	    if (strcmp(argv[2], "blank") == 0)
	    {
		zap_blank = TRUE;
	    }
	    else
	    {
		if (isdbl(argv[2], &zap_value))
		{
		    zap_blank = FALSE;
		}
		else
		{
		    error1("zap value must be a number or 'blank'");
		}
	    }
	    return;
	case BLANK:
	    if (isdbl(argv[2], &default_blank))
	    {
		default_blank_set = TRUE;
	    }
	    else if (strncmp(argv[2], "un", 2) == 0)
	    {
		default_blank_set = FALSE;
	    }
	    else
	    {
		error1("blank value must be a number or 'undefined'");
	    }
	    return;
	case IMIO:
	    on_off(&imio_on, argv[2]);
	    if (imio_on)
		start_imio();  /* start IRAF pipe */
	    else
		stop_imio();
	    return;
	}
    }
    error1("illegal set command");
}

/* on_off sets variable to 0 if string is "off" */
static void
on_off(var, str)
int *var;
char *str;
{
    if (strcmp(str, "off") == 0)
	*var = 0;
    else
	*var = 1;
}


void
zap(argc, argv)
int argc;
char **argv;
{
    int xcur, ycur, dummyx, dummyy;
    int i_coord[2];
    int changes, status;
    void *filptr;  /* pointer to pixels */
    double d_coord[2];

    changes = FALSE;

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    if (filehdr.tapetype != IMFILE)
    {
	error1("cannot zap a skyview workfile");
	return;
    }

    if ((zap_blank) && (!blank_set))
    {
	error1("zap error:  blank is undefined");
	return;
    }

    if (argc < 2)
    {
	if (!graphics)
	    return;

	/* for trackball pick, start cursor in center of visible screen */
	get_center_xy(i_coord);
	xcur = i_coord[0];
	ycur = i_coord[1];

	for (;;)
	{
	    status = trackball_pick(&xcur, &ycur, &dummyx, &dummyy);
	    if (status)
	    {
		d_coord[0] = xcur;
		d_coord[1] = ycur;
		dSC_to_IRAF(d_coord);
		i_coord[0] = Round(d_coord[0]);
		i_coord[1] = Round(d_coord[1]);

		filptr = zap_flux(i_coord);
		if (filptr != NULL)   /* if the change worked */
		{
		    i_coord[0] = xcur;
		    i_coord[1] = ycur;
		    repaintm32(i_coord[0], i_coord[0], i_coord[1], i_coord[1], filptr);
		    repaint_x(i_coord[0], i_coord[0], i_coord[1], i_coord[1]);
		    changes = TRUE;
		}
		else
		    break;
	    }
	    else
		break;

	}
    }
    else if (strncmp(argv[1], "area", 4) == 0)
    {
	/* zap an area */
	int min_samp,max_samp,min_line,max_line;  
	int x, y; 
	int INTERIOR=1; 

	enable_control_c();

	if(argc > 2)
	{
	    if (!switch_area(argv[2]))
		return;
	}

	if (area[area_number].area_type == 0)
	{
	    sprintf(server_str,
		"area %d has not been defined", area_number);
	    error1(server_str);
	    return;
	}

	draw_outline('g', 2);   /* of polygon or circle */ 

	find_range( &min_samp, &max_samp, &min_line, &max_line);

	segment_setup();  /* calculate parameters defining each line */ 


	/* Loop over the possible area of the image */ 


	for(y=min_line; y <= max_line; y++) 
	{
	    x_list(y);     /* list of boundary crossings */

	    for(x=min_samp; x <= max_samp; x++) 
	    {
		if(debug == TRUE)
		    fprintf(debug_file, "zap: at position %d,%d\n", x,y);

		if(position(x)==INTERIOR && !on_horizontal(x, y)) 
		{
		    i_coord[0] = x;
		    i_coord[1] = y;
		    filptr = zap_flux(i_coord);
		    if (filptr == NULL)
			return;  /* abort out of here */
		    changes = TRUE;
		}
		if (control_c)
		{
		    if(server_mode == FALSE)
			printf("\nzap interrupted by control c\n");
		    fprintf(session, "\nzap interrupted by control c\n");
		    return;
		}
	    }
	}
	if (changes)   /* if any changes */
	{
	    d_coord[0] = min_samp;
	    d_coord[1] = min_line;
	    dIRAF_to_SC(d_coord);
	    min_samp = Round(d_coord[0]);
	    min_line = Round(d_coord[1]);

	    d_coord[0] = max_samp;
	    d_coord[1] = max_line;
	    dIRAF_to_SC(d_coord);
	    max_samp = Round(d_coord[0]);
	    max_line = Round(d_coord[1]);

	    repaintm32(min_samp, max_samp, min_line, max_line, filptr);
	    repaint_x(min_samp, max_samp, min_line, max_line);
	}
    }
    else
    {
	error1("wrong arguments to zap command");
    }
    if (changes)
	if (strcmp(hist_filnam, filnam) == 0)
	    hist_filnam[0] = '\0';       /* invalidate saved histogram */
}

void
write_image(argc, argv)
int argc;
char **argv;
{
    int fd_in[2], fd_out, totpixels, bitpix1, imstat, file_type;
    char *pixel_ptr;
    char outfile_name[MAXPATHLEN];
    char window_name[200];
    double blank_value;
    int dummy;
    unsigned char *ptr8;
    unsigned short *ptru16;
    short *ptr16;
    int i, *ptr32;

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    if (filehdr.tapetype != IMFILE)
    {
	error1("cannot write a skyview workfile");
	return;
    }

    if (argc < 2)
    {
	error1("write command requires a filename");
	return;
    }
    strcpy(outfile_name, expand_path(argv[1]));

    if (argc > 2)
    {
	if (!isdbl(argv[2], &blank_value))
	{
	    error1("blank_value not a valid number");
	    return;
	}
    }
    else
	blank_value = blank_flag;

    /* first check if file already exists */
    fd_out = im_open(outfile_name, "r");
    if (fd_out >= 0)
    {
	im_close(fd_out);
	error1("write command error:  file already exists");
	return;
    }

    fd_in[0] = im_open(filnam, "r");
    if (fd_in[0] < 0)
    {
	error1("write command error:  original file has disappeared");
	return;
    }
    im_rkey_i(fd_in[0], &totpixels, "npix");
    im_rkey_i(fd_in[0], &bitpix1, "BITPIX");
    if (mod_filnam[0] == '\0')
	pixel_ptr = (char *) imget(fd_in[0], filnam);  /* get the orig pixels */
    else
	pixel_ptr = (char *) imget(-1, mod_filnam); /* get the modified pixels */
    if (pixel_ptr == NULL)
    {
	error1("unable to read pixels from old file");
	im_close(fd_in[0]);
	return;
    }


    fd_out = im_open(outfile_name, "w");
    if (fd_out < 0)
    {
	error1("cannot open output file");
	im_close(fd_in[0]);
	return;
    }
    fd_in[1] = 0;
    imstat = im_hdr(fd_out, fd_in, 0);

    /* now fuss with blank pixels */
    if (bitpix1 > 0)
    {
	if (FINITE(blank_value))
	{
	    imstat = im_wkey_i(fd_out, (int) blank_value, "BLANK");
	    if ((blank_set) && (blank_value != blank))
	    {
		/* run through all integer pixels changing old blank to new */
		switch(bitpix1)
		{
		    case 8:
			ptr8 = (unsigned char *) pixel_ptr;
			for (i = 0; i < totpixels; i++)
			{
			    if (*ptr8 == blank)
				*ptr8 = blank_value;
			    ptr8++;
			}
			break;
		    case 16:
			if (u16)
			{
			    ptru16 = (unsigned short *) pixel_ptr;
			    for (i = 0; i < totpixels; i++)
			    {
				if (*ptru16 == blank)
				    *ptru16 = blank_value;
				ptru16++;
			    }
			}
			else
			{
			    ptr16 = (short *) pixel_ptr;
			    for (i = 0; i < totpixels; i++)
			    {
				if (*ptr16 == blank)
				    *ptr16 = blank_value;
				ptr16++;
			    }
			}
			break;
		    case 32:
			ptr32 = (int *) pixel_ptr;
			for (i = 0; i < totpixels; i++)
			{
			    if (*ptr32 == blank)
				*ptr32 = blank_value;
			    ptr32++;
			}
			break;
		}
	    }
	}
	else
	{
	    if (blank_set)
		imstat = im_wkey_i(fd_out, (int) blank, "BLANK");
	}
    }
    else
    {
	/* floating point */
	imstat = im_rkey_i(fd_out, &file_type, "filetype");
	if (file_type == 3)
	{
	    /* IRAF floating point output */
	    if (FINITE(blank_value))
	    {
		/* make sure IM converts NaN's to blank_value */
		imstat = im_wkey_i(fd_out, (int) blank_value, "BLANK");
	    }
	    else if (FINITE(report_blank))
	    {
		imstat = im_wkey_i(fd_out, (int) report_blank, "BLANK");
	    }
	    else
	    {
		error1(
		"blank_value required when writing to a floating point IRAF file");
		return;
	    }
	}
	else
	{
	    /* FITS floating point output */

	    /* delete any BLANK keyword line */
	    imstat = im_rkey_i(fd_out, &dummy, "BLANK");
	    if (imstat > 0)
		imstat = im_wkey_t(fd_out, 0, "delete");

#ifdef NOTDEF
/* blank pixels will already be NaN's */
	    if (FINITE(report_blank))
	    {
		/* make sure IM converts report_blank to NaN's */
		imstat = im_wkey_i(fd_out, (int) report_blank, "BLANK");
	    }
#endif /* NOTDEF */
	}
    }

    switch(bitpix1)
    {
	case 8:
	    imstat = im_wpix_c(fd_out, (unsigned char *) pixel_ptr, totpixels);
	    break;
	case 16:
	    imstat = im_wpix_s(fd_out, (short *)pixel_ptr, totpixels);
	    break;
	case 32:
	    imstat = im_wpix_i(fd_out, (int *) pixel_ptr, totpixels);
	    break;
	case -32:
	    imstat = im_wpix_r(fd_out, (float *) pixel_ptr, totpixels);
	    break;
	case -64:
	    imstat = im_wpix_d(fd_out, (double *) pixel_ptr, totpixels);
	    break;
    }

    /* due to the delayed open in image access, a bad status here       */
    /* may mean that the file was never opened.  In this case the fid   */
    /* inside image access is 0.  Then im_close() will close file 0     */
    /* Thus, we only close the file if the status is good.              */
    if (imstat >= 0)
    {
	im_close(fd_out);
    }
    im_close(fd_in[0]);
    if (imstat != totpixels)
    {
	error1("error writing pixels to new file");
	return;
    }

    if(server_mode == TRUE)
    {
	srv_string("filename", outfile_name);
    }
/*
From man page for image access:
IMPORTANT NOTE: the data in the array passed to im_wpix will usually be
		altered (if conversion required before writing).
This nastiness occurs specifically on a byte swapped machine from DEC.
In that case image access swaps the bytes in place, and doesn't swap
them back.  Therefore, the bytes must be discarded.
*/


    /* now make it look like a normal unmodified image */
    if (mod_filnam[0] == '\0')
    {
	force_reread(filnam);  /* junk the pixels */
    }
    else
    {
	force_reread(mod_filnam);  /* junk the pixels */
	/* change_pixel_name(mod_filnam, outfile_name, FALSE); */
	strcpy(filnam, outfile_name);
	mod_filnam[0] = '\0';
	fill_img(curr_img);      /* force into img struct for this image */
	sprintf(window_name,"%s  (frame %d)",
	    strrchr(filnam, '/') + 1, image_frame);
	change_win_name(window_name);
    }
}

/******************************************************************/
/*                                                                */
/* FLATTEN                                                        */
/* removes a plane defined by three points from an image          */
/*                                                                */
/******************************************************************/
 

void
flatten(argc, argv) 
int argc;
char **argv;
 
{ 
    FILE *flfile, *dummy_fd;
    struct job_info *dummy_job;
    char *coord_str;
    char image1[MAXPATHLEN], image2[MAXPATHLEN];
    int status;
    double z, zint, d_coord[2]; 
    double fluxes[4];
    int i, j, i_coord[2];
    int xcur, ycur, dummyx, dummyy;
    int list, cmdc;
    char s_coord[2][40], *cmdv[10];
    int coord_sys;
    int area, area2, nint;
#define NMAX 400
    int n_pts, x_fit[NMAX], y_fit[NMAX];
    struct img *imgp, *imgp_sav;
    int jsys;
    double equinox;
    int srv_elts;

    list = FALSE;

    switch(argc)
    {
    case 1:
	error1("Must give output image name.");
	return;

    case 2:
	/* check for "fl list" */
	if ((strcmp(argv[1], "li") == 0) || (strcmp(argv[1], "list") == 0))
	{
	    /* first line of list is   "[n] filename"   */
	    if (cmdfile == stdin)
		if(server_mode == FALSE)
		   printf("flatten");
	    (void) getcmd(&cmdc, cmdv);
	    if (cmdc == 1)
	    {
		area = 0;
		strcpy(image2, cmdv[0]);
	    }
	    else if (cmdc == 2)
	    {
		area = atoi(cmdv[0]);
		strcpy(image2, cmdv[1]);
	    }
	    else
	    {
		error1("first line in flatten list must have filename");
		return;
	    }
	    list = TRUE;
	}
	else
	{
	    area = 0;
	    strcpy(image2, argv[1]);
	}
	break;

    case 3:
	area = atoi(argv[1]);
	strcpy(image2, argv[2]);
	break;

    default:
	error1("wrong number of arguments to flatten");
	return;
    }

    if (ads_server_mode)
    {
	srv_start_array("point");
	srv_elts = 0;
    }


    area = area / 2;

    if(area < 0)
	area = 0;

    area2 = 2 * area + 1;

    if(area != 0) 
    {
	if(server_mode == FALSE)
	   printf("Averaging over a %dx%d box.\n", area2, area2);
	fprintf(session, "Averaging over a %dx%d box.\n", area2, area2);
    }

    strcpy(image2, expand_path(image2));

    get_center_xy(i_coord);
    xcur = i_coord[0];
    ycur = i_coord[1];

    n_pts = 0;
    for (;;)
    {
	i_coord[0] = xcur;
	i_coord[1] = ycur;
	if (list)
	{
	    if (cmdfile == stdin)
		if(server_mode == FALSE)
		   printf("flatten");
	    (void) getcmd(&cmdc, cmdv);
	    if (cmdc == 2)
	    {
		strcpy(s_coord[0], cmdv[0]);
		strcpy(s_coord[1], cmdv[1]);
		imgp = sky_to_image(s_coord, d_coord);
		if (imgp == NULL)
		{
		    error1("error in flatten list:  coordinates not on an image");
		    return;
		}
		dIM_to_SC(d_coord);
		xcur = Round(d_coord[0]);
		ycur = Round(d_coord[1]);

		status = TRUE;
	    }
	    else
	    {
		/* if he ended list with 'e' */
		if ((cmdc == 1) && (cmdv[0][0] == 'e') && (cmdv[0][1] == '\0'))
		    status = FALSE;
		else
		{
		    if(server_mode == FALSE)
		    {
		       printf("Error in flatten list:  must be exactly one x,y pair per line,\n");
		       printf("End of list must be marked by 'e'\n");
		    }
		    error1("");
		    return;
		}
	    }
	}
	else
	{
	    if (!graphics)
		return;
	    status = trackball_pick(&xcur, &ycur, &dummyx, &dummyy);
	}
	if(status)    /* Extract the flux */
	{ 
	    next_bundle();
	    mark_the_spot(xcur, ycur);

	    imgp_sav = NULL;

	    zint = 0.;
	    nint = 0;

	    for(i = -area; i <= area; ++i)
	    {
		for(j = -area; j <= area; ++j)
		{
		    d_coord[0] = xcur + i;
		    d_coord[1] = ycur + j;

		    imgp = find_SC(xcur + i, ycur + j);

		    if ((imgp != NULL) && (debug))
			fprintf(debug_file,
			    "pixel came from %s\n", imgp->filnam);

		    if (imgp_sav == NULL)
			imgp_sav = imgp;

		    if ((imgp != imgp_sav) && (imgp != NULL))
		    {
			error1("area falls on multiple images");
			return;  /* abort out of here */
		    }

		    dSC_to_IRAF(d_coord);
		    i_coord[0] = Round(d_coord[0]);
		    i_coord[1] = Round(d_coord[1]);
		    flux_IRAF(i_coord, fluxes);
		    z = fluxes[0] * bscale + b_zero;

		    if(FINITE(fluxes[0]))
		    {
			++nint;
			zint += z;
			x_fit[n_pts] = i_coord[0];
			y_fit[n_pts] = i_coord[1];
			/* x_fit and y_fit are in IRAF coordinate system */
			++n_pts;
			if (n_pts >= NMAX)
			{
			    sprintf(server_str,
			    "flatten error:  cannot use more than %d pixels", 
			    NMAX);
			    error1(server_str);
			}
		    }
		}
	    }

	    d_coord[0] = xcur;
	    d_coord[1] = ycur;
	    dSC_to_IM(d_coord);


	    if(nint == 0) 
	    {
		coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, 
		    &equinox, &jsys);
		sprintf(server_str,
		    "No flux available at %s", coord_str);
		error1(server_str);
		fprintf(session, 
		    "\nNo flux available at %s\n", coord_str);
		fflush(stdout);
	    }
	    else
	    {
		zint = zint / nint;
		coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, 
		    &equinox, &jsys);
		if(server_mode == FALSE)
		   printf("%-g %s at %s", zint, bunit, coord_str);
		else if (ads_server_mode)
		{
		    srv_coord(NULL, coord_sys, s_coord, equinox, jsys);
		    srv_elts++;
		}
		fprintf(session, "\n\n%-g %s at %s", zint, bunit, coord_str);
	     }


	     if(nint > 1)
	     {
		  if(server_mode == FALSE)
		     printf(" (using %d pixels)\n", nint);
		  fprintf(session, " (using %d pixels)\n", nint);
	    }

	}
	else
	    break;
    }

    if (ads_server_mode)
	srv_end_array(srv_elts);


    if(n_pts < 3)
    {
	error1("need at least 3 points to flatten");
	return;
    }

    if(debug)
       fprintf(debug_file, 
	   "flatten: lines, pixels = %d %d\n", lines, pixels);

    strcpy(image1, imgp->filnam);

    cmdv[0] = "flatten";
    cmdv[1] = NULL;
    if (duplex_pipe(cmdv, &dummy_fd, &flfile, &dummy_job))
    {
	dummy_job->callback = flatten_callback;
	dummy_job->valid_callback = 1;
	signal(SIGPIPE, SIG_IGN);
	fprintf(flfile, "%s\n", image1);
	fprintf(flfile, "%s\n", image2);
	if (debug)
	{
	    fprintf(debug_file, "%s\n", image1);
	    fprintf(debug_file, "%s\n", image2);
	}
	for (i = 0; i < n_pts; i++)
	{
	    fprintf(flfile, "%d %d\n", x_fit[i], y_fit[i]);
	    if (debug)
		fprintf(debug_file, "%d %d\n", x_fit[i], y_fit[i]);
	}
	fclose(flfile);
	signal(SIGPIPE, SIG_DFL);
    }
    else
	error1("Unable to start background flatten job");
}

static void
flatten_callback(job)
struct job_info *job;
{
    int dummy_char;

    while (!feof(job->fp))
	dummy_char = getc(job->fp);
    fclose(job->fp);
    del_job(job);
}


/******************************************************************/
/*                                                                */
/* REPLICATE                                                      */
/* expands an image by replicating pixels                         */
/*                                                                */
/******************************************************************/
 

void
replicate(argc, argv) 
int argc;
char **argv;
 
{ 
    char syscom[150];
    char image1[MAXPATHLEN], image2[MAXPATHLEN];
    int repl_samp, repl_line;

    if(server_mode == TRUE)
    {
       error_na("replicate command not allowed in server mode");
       return;
    }

    if (argc < 2)
    {
	error1("replicate:  need a filename");
	return;
    }

    if (fi_name[0] == '\0')
    {
	error1("replicate:  no workfile has been specified");
	return;
    }

    repl_line = 1;
    if(argc > 2)
	repl_line = atoi(argv[2]);

    repl_samp = 1;
    if(argc > 3)
	repl_samp = atoi(argv[3]);

    strcpy(image1, fi_name);
    strcpy(image2, expand_path(argv[1]));

    sprintf(syscom, "%s %s %s %d %d", 
	"replicate", image1, image2, repl_samp, repl_line);

    if(debug)
	fprintf(debug_file, "%s\n", syscom);
    
    system(syscom);
}

/*********************************************************************/ 
/*                                                                   */ 
/*  TEK_CMD  entry point for Tektronix emulator.                     */ 
/*  Only moves and draws are supported, plus graphics on/off.        */ 
/*  Specifically, alpha mode characters are not implemented.         */  
/*                                                                   */  
/*                                                                   */ 
/*********************************************************************/

#define LOX 1
#define LOY 2
#define HIX 3
#define HIY 4
#define MOVE 5
#define DRAW 6
#define ETX 0x3
#define CR 0xd
#define LF 0xa
#define CAN 0x18
#define GS 0x1d
#define US 0x1f

void
tek_cmd(argc, argv)
int argc;
char **argv;
{
    char filename[MAXPATHLEN];
    int getdims = 0;
    int i, cmdc;
    char *cmdv[2];

    filename[0] = '\0';
    for (i = 1; i < argc; i++)
    {
	if ((strcmp(argv[i], "DIMENSIONS") == 0) ||
	    (strcmp(argv[i], "dimensions") == 0))
	{
	    /* next 2 MOVE-DRAW pairs give upper right and lower left */
	    /* edges of plot in the coordinates NCAR will be sending */
	    getdims = 2;
	}
	else
	    strcpy(filename, argv[i]);
    }
    if (filename[0] != '\0')
    {
	cmdv[0] = "take";
	cmdv[1] = filename;
	cmdc = 2;
	take(cmdc, cmdv);
	if (cmdfile == stdin)
	    return;   /* if take failed */
    }
    do_tek(cmdfile, getdims, fgetc);
}

/* tek_sequence uses up the tek commands, then returns a pointer to    */
/* whatever is left                                                    */
/*    RBH - there is still a problem here and in parse.c:              */
/*          If the sequence (on or off) is split between bufferloads   */
/*          it will not be recognized                                  */

char *tek_sequence(anin)
char *anin;
{
    if (strncmp (anin, "\033[?38h", 6) == 0)
    {
	tek_ptr = anin + 6;
	do_tek(cmdfile, 0, get_tek_char);
	return(tek_ptr);
    }
    else if (strncmp (anin, "\033[?38l", 6) == 0)
    {
	return(anin + 6);   /* discard TEK_off sequence */
    }
    else
	return(anin);
}

static int
get_tek_char(fd)
FILE *fd;  /* dummy */
{
    char c;

    if (*tek_ptr)            /* if more chars left on command line */
	return(*tek_ptr++);
    if (fd == stdin)
	read(ttyfd, &c, 1);
    else
	c = fgetc(fd);
    return(c);
}

static void
do_tek(fd, getdims, get_a_char)
FILE *fd;
int getdims;
int (*get_a_char)();   /* function to call for more characters */
{
    double d_coord1[2];
    static double ncar_width = 1024;
    static double ncar_height = 780;
    static double ncar_x_offset = 0;
    static double ncar_y_offset = 0;
    int c;
    int plot_mode = FALSE;
    int lastc = LOX;
    int action;
    int tek_x, tek_y, tek_hix, tek_hiy, tek_lox, tek_loy;

    overlay_setup('w');
    next_bundle();
    for (;;)
    {
	c = get_a_char(fd);
	switch(c)
	{
	    case EOF:
	    case CAN:
	    case CR:
	    case LF:
		flush_graphics();
		return;   /* exit TEK mode */
	    case 0x1b:
		/* escape sequence */
		c = get_a_char(fd);
		if (c == ETX)
		{
		    flush_graphics();
		    return;   /* exit TEK mode */
		}
		if (c == '[')
		    if (get_a_char(fd) == '?')
			if (get_a_char(fd) == '3')
			    if (get_a_char(fd) == '8')
				if (get_a_char(fd) == 'l')
				{
				    flush_graphics();
				    return;   /* exit TEK mode */
				}
		break;
	    case GS:
		plot_mode = TRUE;
		action = MOVE;
		break;
	    case US:
		plot_mode = FALSE;
		break;
	    default:
		if (plot_mode)
		{
		    /* convert the coordinates */
		    if (c < 0x20)
			break;    /* ignor control codes */
		    if (c < 0x40)
		    {
			/* 20-3f are HIX or HIY */
			if (lastc == LOY)
			{
			    tek_hix = (c & 0x1f) << 5;
			    lastc = HIX;
			}
			else
			{
			    tek_hiy = (c & 0x1f) << 5;
			    lastc = HIY;
			}
		    }
		    else if (c < 0x60)
		    {
			/* 40-5f are LOX (causes beam movement) */
			tek_lox = c & 0x1f;
			lastc = LOX;
			tek_x = tek_hix | tek_lox;
			tek_y = tek_hiy | tek_loy;

			if (action == MOVE)
			{
			    action = DRAW;   /* make next coords draw */
			    d_coord1[0] = (tek_x - ncar_x_offset) 
				* (pixels-1) / ncar_width;
			    d_coord1[1] = (tek_y - ncar_y_offset) 
				* (lines-1) / ncar_height;
			    overlay_dmov(d_coord1[0], d_coord1[1]);
			}
			else
			{
			    if (getdims)
			    {
				/* collecting corner values */
				getdims--;
				if (getdims)
				{
				    /* first time thru - */
				    /*  we have upper right values */
				    ncar_width = tek_x;   /* save U R x */
				    ncar_height = tek_y;   /* save U R y */
				    break;
				}
				/* second time thru */
				/* - we have lower left values */
				ncar_x_offset = tek_x;
				ncar_y_offset = tek_y;
				ncar_width -= ncar_x_offset;
				ncar_height -= ncar_y_offset;
				break;
			    }
			    d_coord1[0] = (tek_x - ncar_x_offset) 
				* (pixels-1) / ncar_width;
			    d_coord1[1] = (tek_y - ncar_y_offset) 
				* (lines-1) / ncar_height;
			    overlay_ddraw(d_coord1[0], d_coord1[1]);
			}
		    }
		    else
		    {
			/* 60-7f are LOY */
			tek_loy = c & 0x1f;
			lastc = LOY;
		    }
		}
		else
		{
		    if (debug)
			fprintf(debug_file, 
			    "TEK error:  not in plot mode. char =%c\n", c);
		    else
		    {
			putchar(c);
			fflush(stdout);
		    }
		}
		break;
	}
    }
}

/*********************************************************************/ 
/*                                                                   */ 
/*  POST_CMD  entry point for Postscript emulator.                   */ 
/*  Only moves and draws are supported                               */ 
/*                                                                   */  
/*                                                                   */ 
/*********************************************************************/

void
post_cmd(argc, argv)
int argc;
char **argv;
{
    int cmdc;
    char *cmdv[2];

    if (argc > 1)
    {
	cmdv[0] = "take";
	cmdv[1] = argv[1];
	cmdc = 2;
	take(cmdc, cmdv);
	if (cmdfile == stdin)
	    return;   /* if take failed */
    }
    do_post(cmdfile, fgetc);
}


static void
do_post(fd, get_a_char)
FILE *fd;
int (*get_a_char)();   /* function to call for more characters */
{
    static int line_mode = TRUE;  /* collecting whole lines */
    int got_some_x, collecting_x, fig_type;
    char post_line[200], *line_ptr, *line_end;
    char tmp_color[200];
    char x_buf[30], y_buf[30], *x_ptr, *y_ptr;
    double d_coord1[2];
    int status;
    int c, c1;
    double x, y;
    int post_minx, post_maxx, post_miny, post_maxy;
    double post_x_scale, post_y_scale, post_x_offset, post_y_offset;

    enable_control_c();

restart_post:

    /* must start with %!  */
    c = get_a_char(fd);
    if (control_c)
	return;
    c1 = get_a_char(fd);
    if (control_c)
	return;
    if ((c != '%') || (c1 != '!'))
    {
	error1("postscript does not start with '%!'");
	return;
    }

    overlay_setup('w');
    next_bundle();

    line_ptr = post_line;
    line_end = post_line + 199;
    for (;;)
    {
	c = get_a_char(fd);
	if (control_c)
	    return;
	if (c == EOF)
	    return;
	else if (c == LF)
	{
	    *line_ptr++ = '\0';
	    if (strncmp(post_line, "%%Page:", 7) == 0)
	    {
		break;
	    }
	    else if (strncmp(post_line, "%%ImageBox:", 11) == 0)
	    {
		sscanf(&post_line[11], "%d %d %d %d", 
		    &post_minx, &post_miny, &post_maxx, &post_maxy);

		post_x_scale = ((double)pixels) / 
		   (post_maxx - post_minx);

		post_y_scale = ((double)lines) / 
		   (post_maxy - post_miny);

		post_x_offset = -.5 - post_x_scale * post_minx;
		post_y_offset = -.5 - post_y_scale * post_miny;

		strcpy(post_line, "        ");  /* clear out buffer */
	    }
	    else
	    {
		line_ptr = post_line;  /* start a new line */
	    }
	}
	else
	{
	    *line_ptr = c;
	    if (line_ptr < line_end)
		line_ptr++;
	}
    }

    line_ptr = post_line;
    save_overlay_color();
    for (;;)
    {
	if (control_c)
	    return;
	if (line_mode)
	{
	    c = get_a_char(fd);
	    if (c == EOF)
		return;
	    else if (c == LF)
	    {
		*line_ptr = '\0';
		if ((((line_ptr - post_line) >= 10) &&
		    (strncmp(post_line, "%!polyline", 10) == 0)) ||
		    (strstr(post_line, "newpath") != 0))
		{
		    line_mode = FALSE;
		    collecting_x = TRUE;
		    got_some_x = FALSE;
		    fig_type = POLYLINE;
		    x_ptr = x_buf;
		    y_ptr = y_buf;
		}
		else if (((line_ptr - post_line) >= 12) &&
		    (strncmp(post_line, "%!filledarea", 12) == 0))
		{
		    line_mode = FALSE;
		    collecting_x = TRUE;
		    got_some_x = FALSE;
		    fig_type = FILLAREA;
		    x_ptr = x_buf;
		    y_ptr = y_buf;
		}
                else if (strncmp(post_line, "%!color", 7) == 0)
                {
                    *line_ptr = '\0';
                    strcpy(tmp_color, post_line + 8);


		    if (!good_overlay_color(tmp_color))
		        break;

                    status = set_overlay_color(tmp_color);
                    if (status)
                        strcpy(ov_color, tmp_color);

		    overlay_setup('g');

                    line_ptr = post_line;
                }
		else if (strncmp(post_line, "%%EOF", 5) == 0)
		{
		    /* peek to see if we've got another postscript page */
		    c = get_a_char(fd);
		    if (c == EOF)
			return;
		    c1 = get_a_char(fd);
		    if (c1 == EOF)
		    {
			ungetc(c, fd);
			return;
		    }
		    ungetc(c1, fd);
		    ungetc(c, fd);
		    if ((c != '%') || (c1 != '!'))
		    {
			return;
		    }
		    else
			goto restart_post;
		}
		else
		{
		    line_ptr = post_line;  /* start a new line */
		    continue;
		}
	    }
	    else
	    {
		*line_ptr++ = c;
	    }
	}
	else
	{
	c = get_a_char(fd);
#ifdef NOTDEF
	putchar(c);
#endif /* NOTDEF */
	switch(c)
	{
	    case EOF:
		flush_graphics();
		return;   /* exit PS mode */
	    case 's':
		/* start looking for %!polyline again */
		line_mode = TRUE;
		line_ptr = post_line;
		break;
	    case 'f':
		/* start looking for %!filledarea again */
		overlay_darea(0., 0., 1);
		line_mode = TRUE;
		line_ptr = post_line;
		break;
	    case 'M':
	    case 'm':
		*x_ptr = '\0';
		*y_ptr = '\0';
		x = atof(x_buf);
		y = atof(y_buf);
		if(fig_type == POLYLINE)
		{
		    d_coord1[0] = x * post_x_scale + post_x_offset;
		    d_coord1[1] = y * post_y_scale + post_y_offset;
		    overlay_dmov(d_coord1[0], d_coord1[1]);
		}
		else if(fig_type == FILLAREA)
		{
		    d_coord1[0] = x * post_x_scale + post_x_offset;
		    d_coord1[1] = y * post_y_scale + post_y_offset;
		    overlay_darea(d_coord1[0], d_coord1[1], 0);
		}
		collecting_x = TRUE;
		got_some_x = FALSE;
		x_ptr = x_buf;
		y_ptr = y_buf;
		break;
	    case 'L':
	    case 'l':
		*x_ptr = '\0';
		*y_ptr = '\0';
		x = atof(x_buf);
		y = atof(y_buf);
		if(fig_type == POLYLINE)
		{
		    d_coord1[0] = x * post_x_scale + post_x_offset;
		    d_coord1[1] = y * post_y_scale + post_y_offset;
		    overlay_ddraw(d_coord1[0], d_coord1[1]);
		}
		else if(fig_type == FILLAREA)
		{
		    d_coord1[0] = x * post_x_scale + post_x_offset;
		    d_coord1[1] = y * post_y_scale + post_y_offset;
		    overlay_darea(d_coord1[0], d_coord1[1], 0);
		}
		collecting_x = TRUE;
		got_some_x = FALSE;
		x_ptr = x_buf;
		y_ptr = y_buf;
		break;
	    case ' ':
		if (got_some_x)
		    collecting_x = FALSE;
		break;
	    case CR:
	    case LF:
		break;
	    default:
		if ((isdigit(c)) || (c == '.'))
		{
		    if (collecting_x)
		    {
			*x_ptr++ = c;
			got_some_x = TRUE;
		    }
		    else
			*y_ptr++ = c;
		}
		break;
	    }
	}
    }
    restore_overlay_color();
}

void
draw_border()
{
    double d_coord1[2];

    d_coord1[0] = -0.5;
    d_coord1[1] = -0.5;
    overlay_dmov(d_coord1[0], d_coord1[1]);
    d_coord1[0] = -0.5;
    d_coord1[1] = lines - 0.5;
    overlay_ddraw(d_coord1[0], d_coord1[1]);
    d_coord1[0] = pixels - 0.5;
    d_coord1[1] = lines - 0.5;
    overlay_ddraw(d_coord1[0], d_coord1[1]);
    d_coord1[0] = pixels - 0.5;
    d_coord1[1] = -0.5;
    overlay_ddraw(d_coord1[0], d_coord1[1]);
    d_coord1[0] = -0.5;
    d_coord1[1] = -0.5;
    overlay_ddraw(d_coord1[0], d_coord1[1]);
}
