/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "parse.h"
#include "fits_table.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/param.h>


#define MAXFIELDLENGTH 60

/* begin per-file stuff - only one file for now */
static int unit = 1;
static char format_str[500];
static int table_field[14];  /* indices of fields to be used */
static int field_count;      /* count of entries in table_field */
static char *more_save;
static char more_line[MAXLINE];
/* end of per_file stuff */

struct tablename {
    struct tablename *llink, *rlink;
    char fname[1];   /* 1 for the NUL */
    };

static struct tablename head_tablename = {&head_tablename, &head_tablename};

extern char *more;
extern int debug;
extern FILE *debug_file;
extern char server_str[];
int table_file = FALSE;    /* flag that we are processing a table file */
int table_linenumber;
static char table_nam_chars[MAXPATHLEN];
char *table_nam;

/* function prototypes */
#ifdef _NO_PROTO

static void delete_table_name();
static char *new_table_name();
static char *find_table_name();

#else

static void delete_table_name(struct tablename *p);
static char *new_table_name(char *filename);
static char *find_table_name(char *filename);

#endif /* _NO_PROTO */


void
table(argc, argv)
int argc;
char **argv;
{
    int i, temp_index, nparms;

    if (argc < 3)
    {
	error1("Too few arguments to table command");
	return;
    }
    strcpy(table_nam_chars, expand_path(argv[1]));
    table_nam = new_table_name(table_nam_chars);

    topenr(unit, table_nam, &nparms);
    if (itstat())
    {

	sprintf(server_str, "unable to open table file %s: %s", 
	    table_nam, t_error());
	error1(server_str);
	return;
    }
    if (debug)
	fprintf(debug_file, 
	    "nparms = %d (number of fields in table file)\n", nparms);

    format_str[0] = '\0';
    field_count = 0;
    for (i = 2; i < argc; i++)
    {
	if (argv[i][0] == '$')
	{
	    tindp(unit, &argv[i][1], &temp_index);
	    if (temp_index == -1)
	    {
		sprintf(server_str, "no field named %s in table file %s",
		    &argv[i][1], table_nam);
		error1(server_str);
		tcls(unit);
		return;
	    }
	    else
	    {
		strcat(format_str, " %s ");
		table_field[field_count++] = temp_index;
		if (debug)
		    fprintf(debug_file, 
			"using field number %d for name =%s=\n", 
			temp_index, argv[i]);
	    }
	}
	else
	{
	    /* no dollar sign on this arg - take it literally */
	    strcat(format_str, " ");
	    strcat(format_str, argv[i]);
	}
    }
    strcat(format_str, "\n");
    more_save = more;
    if (more != NULL)
	strcpy(more_line, more);
    more = NULL;
    srvflush("table", 1, 1);
    table_file = TRUE;
    enable_control_c();
}

void table_close()
{
    tcls(unit);
    table_file = FALSE;
    table_linenumber = 0;
    more = more_save;
    if (more != NULL)
	strcpy(more, more_line);
}

void
get_table_line(anin)
char *anin;
{
    int i;
    char val[14][MAXFIELDLENGTH];

    table_linenumber = tblread(unit);
    if (itstat())
    {
	table_close();
	anin[0] = '\0';
	return;
    }
    for (i = 0; i < field_count; i++)
    {
	tgetc(unit, table_field[i], val[i]);
    }
    sprintf(anin, format_str, val[0], val[1], val[2], val[3], val[4], val[5],
    val[6], val[7], val[8], val[9], val[10], val[11], val[12], val[13]);
}





/********************************************************************/
/*                                                                  */
/*  CLEAR_TABLE_NAMES                                               */
/*  Clears entire structure remembering table filenames.            */
/*                                                                  */
/********************************************************************/

void
clear_table_names()
{
    while (head_tablename.rlink != &head_tablename)
	delete_table_name(head_tablename.rlink);
}


/********************************************************************/
/*                                                                  */
/*  NEW_TABLE_NAME                                                  */
/*  Place a new entry into the table filename structure.            */
/*  If it matches an existing name, just return a pointer to        */
/*  that name.                                                      */
/*                                                                  */
/********************************************************************/

static
char *new_table_name(filename)
char *filename;
{
    static struct tablename *tmptablename;
    struct tablename *x;
    char *tmpcharptr;
    unsigned size;

    /* see if it exists already */
    tmpcharptr = find_table_name(filename);
    if (tmpcharptr != NULL)
	return(tmpcharptr);

    size = sizeof(struct tablename) + strlen(filename);
    if (debug)
	fprintf(debug_file, "new_table_name:  allocating %d bytes\n", size);
    tmptablename = (struct tablename *) malloc(size);
    if (tmptablename == NULL)
    {
	error1("error allocating space for table filename structure\n");
	return(NULL);
    }
    /* insert to right of head_tablename */
    x = &head_tablename;
    tmptablename->llink = x;
    tmptablename->rlink = x->rlink;
    (tmptablename->rlink)->llink = tmptablename;
    x->rlink = tmptablename;

    strcpy(tmptablename->fname, filename);
    if (debug)
	fprintf(debug_file, 
	    "new_table_name: tmptablename = 0x%p\n", (void *) tmptablename);
    return(tmptablename->fname);
}


/********************************************************************/
/*                                                                  */
/*  DELETE_TABLE_NAME                                               */
/*  Deletes an entry from the tablename array.                      */
/*                                                                  */
/*  If no image exists at the address, does nothing.                */
/*                                                                  */
/********************************************************************/

static void
delete_table_name(p)
struct tablename *p;
{
    if (debug)
	fprintf(debug_file, "delete_tablename  p = 0x%p\n", (void *) p);

    (p->llink)->rlink = p->rlink;
    (p->rlink)->llink = p->llink;
    free(p);

    return;
}

/********************************************************************/
/*                                                                  */
/*  FIND_TABLE_NAME                                                 */
/*  given a filename returns a pointer to the tablename structure   */
/*  with that filename.                                             */
/*                                                                  */
/*  If tablename not found, returns NULL                            */
/*                                                                  */
/********************************************************************/

static
char *find_table_name(filename)
char *filename;
{
    struct tablename *p;

    if (debug)
	fprintf(debug_file, "find_table_name: entry  filename =%s=\n", filename);

    p = head_tablename.rlink;
    while (p != &head_tablename)
    {
	if (strcmp(filename, p->fname) == 0)
	{
	    return(p->fname);
	}
	p = p->rlink;
    }
    return(NULL);
}
