/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* skyview program */
#include "skyview.h"
#include "parse.h"
#include "im_proto.h"
#include "job.h"
#include <stdio.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <string.h>
#include <sys/file.h>
#include <pwd.h>
#include <sys/types.h>
#include <netinet/in.h>
#include "dirent.h"
#include <sys/param.h>
#include <unistd.h>
#include <stdlib.h>

char Id[] = "@(#)skyview.c   RBH        04/15/2015  (R3.8)";

extern char version[];          /* Makefile version date */

extern FILE *cmdfile;
extern int do_alias;    /* whether parser should expand aliases */
FILE *session;         /* stream pointer for session history file */
char pathnam[MAXPATHLEN];      /* argument from "cd" command */
char org_path[MAXPATHLEN];      /* path for session, alias, and color tables */
static char home_dir[MAXPATHLEN];      /* home directory */
char fi_name[MAXPATHLEN] = "";     /* filename from last "fi" command */
int mfd = -1;         /* file descripter for mag tape */
int ttyfd;           /* file descripter for users terminal */
int mouse_is_on = 1;	/* flag to allow commands from mouse buttons */
int use_projtype = 0;   /* flag to use FITS keyword PROJTYPE */
int u16;                /* flag to treat current BITPIX 16 as unsigned */
int default_u16 = 0;    /* flag to treat future BITPIX 16 as unsigned */
int reread_on = 0;      /* flag to force rereading image when repainting*/
int conserve_mem = 0;   /* flag to save memory not keeping pixels in mem */
int imio_on = 0;        /* flag to look for imtool output from iraf */
int minx_next, miny_next;  /* values from "orig" or "frame" command */
int frame = 1;
int frame_advance = FALSE;  /* automatic advance of frame with each paint */
int x_orig, y_orig;          /* values given to last j_zoom function call */
int reqband = 1;        /* argument from "plane" command */
char current_clt[MAXPATHLEN];      /* name of color table file */
int req_mode = LIN;         /* type of stretch */


int clip_top = 0, clip_bottom = 0, clip_left = 0, clip_right = 0;
int s0_server_mode = FALSE; /* flag if in -s0 server mode (mini-lib version) */
int s1_server_mode = FALSE; /* flag if in -s server mode  */
int ads_server_mode = FALSE; /* flag if in -ads server mode  */
int cmdline_mode = FALSE;   /* flag if parsing commands from command line */
int accepting_stdin = TRUE; /* flag if accepting stdin input */
int debug = FALSE;
int debug_val = 0;	/* 5:  fifo.c messages */
			/* 7:  range stretch table values */
			/* 8:  get_hist_IM */
			/* 9:  scatter_plot */
FILE *debug_file;
int server_mode = FALSE;
char server_str[4096];
int be_quiet = FALSE;  /* TRUE suppresses messages during paint, zoom, etc. */
int graphics = TRUE;   /* FALSE allows operation with no X display  */
int hist[HISTSIZ2+1];  /* histogram of image in file hist_filnam */
int screen_hist[256];  /* histogram of pixels painted in last image */
char hist_filnam[MAXPATHLEN]; /* name of file from which hist came */
int hist_band;                /* band_offset in use when hist computed */
char screen_hist_filnam[MAXPATHLEN]; /* file from which screen_hist came */
double hist_binsiz;   /* binsize of each bin in hist if file is IMFILE */
double hist_min;      /* pixval represented by first bin (if IMFILE) */
char plane = 'a';     /* image planes mapped (from "map" command) */
int control_c = FALSE;  /* flag that control c has been intercepted */
int captive = FALSE;    /* flag for captive mode */
int vax_bytes;
static char cmd_keyword[40];

/*  -- everything below here appears both globally and in image structure -- */
char filnam[MAXPATHLEN] = "";  /* current image workfile name   */
			       /* (with path already prepended) */
char mod_filnam[12];    /* scratch filename for zapped image */
struct hdrstruct filehdr;

int bitpix;         /* bits per pixel on tape and in workfile */
double blank_flag;  /* missing pixel flag - NaN in ieee machines */
int blank_set;         /* flag if blank has meaning */
double blank;          /* value of missing pixel */
double report_blank;  /* user's notion of BLANK: */
	    /* same as "blank" except for IRAF floating point images painted */
	    /* with either BLANK= keyword or "set blank" value */
int pixels, lines;  /* pixels per line, lines in image */
int naxis;
int naxis3;
double cdelt1, cdelt2, twist;  /* pixel dimensions, deepsky twist angle */
double cd1_1, cd1_2, cd2_1, cd2_2;
double dc1_1, dc1_2, dc2_1, dc2_2;
int using_cd;
double crpix1, crpix2;
double bscale, b_zero, glat, glong, crval3;
char bunit[80];   /* units of pixel flux value */
char ctype1[80], ctype2[80];  /* strings "lat", "lon", etc. */
char ctype1sav[80], ctype2sav[80];  /* strings saved for tapewrite */
char object[80];        /* name of object */
int maptype;    /* projection type */
int map_distortion;    /* flag for SIRTF distortion corrections */
int a_order, ap_order, b_order, bp_order;      /* distortion corrections */
double a[5][5], ap[5][5], b[5][5], bp[5][5];  /*  distortion corrections */
double base, noise;    /* deepsky stuff */
int dskygrid;
int x_offset;       /* relationship of screen x to image sample */
int minx = 0, maxx, miny = 0, maxy;  /* screen coords of image corners */
int sgn_y, y_offset;   /* relationship of screen y to image line  */
double iraf_max, iraf_min;  /* max, min floating pt DN values for iraf */
int band_offset;        /* reqband - 1 when image was painted */
double slow, shigh;    /* range stretch used on this image */
int mode_used;         /* log vs linear used on this image */
int samp_disp;   /* pixel displacement from left edge to 1st desired */
int line_disp;  /* line displacement from top edge to 1st desired line */

int image_frame;
int mapped;
int frame_graphics;  /* flag if this frame was painted with graphics on */
int graphics_window = 0;  /* graphics-only window */

    /* stuff below stolen from Doug Mink's saoimage */
/* saoimage/wcslib/wcs.h
   March 6, 1995
   By Doug Mink, Harvard-Smithsonian Center for Astrophysics */

    double	plate_ra;	/* Right ascension of plate center */
    double	plate_dec;	/* Declination of plate center */
    double	x_pixel_offset;	/* X pixel offset of image lower right */
    double	y_pixel_offset;	/* Y pixel offset of image lower right */
    double	x_pixel_size;	/* X pixel_size */
    double	y_pixel_size;	/* Y pixel_size */
    double	ppo_coeff[6];
    double	amd_x_coeff[20]; /* X coefficients for plate model */
    double	amd_y_coeff[20]; /* Y coefficients for plate model */
    double	plt_scale; /* plate scale arc sec per mm */


/* More stuff is found in machine dependent files (e.g xwindows.c, jup.c) */


/* function prototypes */
#ifdef _NO_PROTO

static void bang();
static void pause_cmd();
static void finish();
static void done();
static void echo();
static void do_debug();
void getctrlc();

#else

static void bang(int argc, char *argv[]);
static void pause_cmd(int argc, char *argv[]);
static void finish(void);
static void done(int argc, char *argv[]);
static void echo(int argc, char *argv[]);
static void do_debug(int argc, char *argv[]);
void getctrlc(int dummy);

#endif /* _NO_PROTO */

/*ARGSUSED*/
void getctrlc(dummy)
int dummy;
{
    control_c = TRUE;
    signal(SIGINT, SIG_DFL);    /* restore default action              */
				/* i.e. two control c's kill program   */
}

void
enable_control_c()
{
    control_c = FALSE;
    signal(SIGINT, getctrlc);
}

#ifdef NOTDEF
#include <math.h>
matherr(struct exception *x) 
{
    if ((x->type) == DOMAIN)
    {
	printf("RBH DOMAIN ERROR function = [%s]\n", x->name);
	abort();
    }
    return(0);
}
#endif /* NOTDEF */


int main(argc, argv)
int argc;
char **argv;
{
	
    char aliasname[MAXPATHLEN];     /* name of alias file */
    char *cmdv[10];
    char strng[MAXPATHLEN+2], *p, slogin[MAXPATHLEN];
    char path_tmp[MAXPATHLEN];
    int cmdc, cmdcode, redo_slogin;
    struct passwd *pwuid;
    u_long netlong;
    DIR *dirp;
    FILE *tmp_fd;


    debug_file = stdout;
    cmdfile = stdin;

    /* test for byteorder */
    netlong = 0x12345678;
    netlong = htonl(netlong);
    if (netlong == 0x12345678)
	vax_bytes = FALSE;
    else if (netlong == 0x78563412)
	vax_bytes = TRUE;
    else
    {
	printf("s error:  unrecognized byte order on this machine\n");
	exit(1);
    }

    /* get value for flagging a blank pixel */
    /* - NaN on ieee machines, */
    /* - some wierd value dreamed up by image access on non-ieee machines */
    im_blankd(&blank_flag);

    /* set master default to login directory */
    p = getenv("HOME");
    if (p == NULL)
    {
	/* no HOME environment variable - get it from /etc/passwd */
	p = getenv("USER");
	if (p == NULL)
	{
	    pwuid = getpwuid(getuid());
	}
	else
	{
	    pwuid = getpwnam(p);
	}
	strcpy(org_path, pwuid->pw_dir);
    }
    else
    {
	strcpy(org_path, p);
    }
    if (org_path[strlen(org_path)-1] != '/')
	strcat(org_path,"/");    
    strcpy(home_dir, org_path);




    /* process input arguments */

    if (argv[0][0] == '-')
	captive = TRUE;

    get_resources(&argc, argv);   /* parse command line, merge X resources */
				  /* (may set graphics = FALSE) */


    if (accepting_stdin)
	ttyfd = fileno(stdin);

    graphics = init_graphics(); 
    /* sets graphics TRUE or FALSE */

    if(server_mode == FALSE)
    {
	printf("skyview version %s\n",Id+26);
	printf(
"  Copyright (C) 1992, California Institute of Technology.\n");
	printf(
"  U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.\n\n");
    }

    if ((captive) && (server_mode == FALSE))
	printf("invoked in captive mode.\n");

    if ((graphics == FALSE) && (server_mode == FALSE))
	printf("invoked in no graphics mode\n");

    session = fopen("/dev/null", "w");
     

    if (graphics)
    {
	ball_setup();
	image_setup(plane);   /* set up board map */
    }
    color_setup();

    if (imio_on)
	start_imio();  /* start IRAF pipe */


    if ((!s0_server_mode) && (server_mode == FALSE))
	/* don't take slogin in server mode */
    {
    /* look for .slogin file */
    p = getenv("SLOGIN");
    if (p)
    {
	strcpy(slogin, p);
    }
    else
    {
	strcpy(slogin, org_path);
	strcat(slogin, ".slogin");
    }
    if (debug)
	fprintf(debug_file, "trying to open slogin file =%s=\n", slogin);
    tmp_fd = fopen(slogin, "r");
    if (tmp_fd != NULL)      /* if .slogin file exists and is readable */
    {
	fclose(tmp_fd);    /* take() will reopen the file */
	redo_slogin = TRUE;  /* assume that this will fail */
	do_alias = FALSE;  /* prevent alias expansion */
	cmdv[1] = slogin;
	take(2,cmdv);   /* take .slogin */
	if (getcmd(&cmdc, cmdv) == CD)
	{
	    if (cmdc == 2)
	    {
		if ((*cmdv[1] == '/') || (*cmdv[1] == '~'))
		{
		    /* meets all our stringent requirements */
		    strcpy(path_tmp, expand_path(cmdv[1]));
		    if (path_tmp[strlen(path_tmp)-1] != '/')
			strcat(path_tmp, "/");
		    if ((dirp = opendir(path_tmp)) != NULL) /* if new        */
							  /* directory is ok */
		    {
			redo_slogin = FALSE;
			closedir(dirp);
			strcpy(org_path, path_tmp);
		    }
		}
	    }
	}
	do_alias = TRUE;
	if (redo_slogin)
	{
	    /* the .slogin file does not contain the special cd command */
	    /* but weve already munched up the first line */
	    close_take();
	    cmdv[1] = slogin;
	    take(2,cmdv);   /* reopen .slogin for normal command loop */
	}
    }
    }
    if (debug)
	fprintf(debug_file, "final org_path =%s=\n", org_path);

    p = getcwd(strng, MAXPATHLEN+2); /* set normal default to current directory */
    if (p == NULL)
    {
	if(server_mode == FALSE)
	    printf("current directory is unreadable\n");
	pathnam[0] = '\0';;
    }
    else
    {
	if (strncmp(p, "/tmp_mnt", 8) == 0)
	    p += 8;
	strcpy(pathnam, p);
	strcat(pathnam, "/");
    }

    strcpy(aliasname, org_path);
    strcat(aliasname, "alias");
    load_aliases(aliasname); 

    helpinit();

    enable_control_c();

    for (;;)
    {

	check_forks();

	srvinit();
	cmdcode = getcmd(&cmdc, cmdv);
	if (cmdcode == NIL)
	{
	    cmd_keyword[0] = '\0';
	}
	else
	{
	    strcpy(cmd_keyword, cmdv[0]);
	}
	switch (cmdcode)
	{
	    case AL:
		alias(cmdc,cmdv);
		break;
	    case AN:
		label(cmdc, cmdv);
		break;
	    case PLANE:
		chgband(cmdc, cmdv);
		break;
	    case BANG:
		bang(cmdc, cmdv);
		break;
	    case BL:
		background(cmdc, cmdv);
		break;
	    case BORDER:
		border(cmdc, cmdv);
		break;
            case CA:
		change_area(cmdc, cmdv);
		break;
	    case CD:
		ch_dir(cmdc, cmdv);
		break;
	    case CIMG:
		sho_curr_img(cmdc, cmdv);
		break;
	    case CL:
		clip(cmdc, cmdv);
		break;
	    case CMAP:
		curr_map_parms(cmdc, cmdv);
		break;
	    case CONTOUR:
		cplot(cmdc, cmdv);
		break;
	    case CROP:
		crop(cmdc, cmdv);
		break;
	    case CS:
		set_coord_system(cmdc, cmdv);
		break;
	    case CT:
		chgtbl(cmdc, cmdv);
		break;
	    case CU:
		cutoff(cmdc, cmdv);
		break;
	    case DA:
		define_area(cmdc, cmdv);
		break;
	    case DE:
		do_debug(cmdc, cmdv);
		break;
	    case DF:
		delete_file(cmdc, cmdv);
		break;
	    case DIRR:      /* dir command */
		dir(cmdc, cmdv);
		break;
	    case DR:
		rddisk(cmdc, cmdv);
		break;
	    case ECHO1:
		echo(cmdc, cmdv);
		break;
	    case ER:
		erase(cmdc, cmdv);
		break;
	    case EX:
		examine_area(cmdc, cmdv);
		break;
	    case FI:
		chgfil(cmdc, cmdv);
		break;
	    case FL:
		flatten(cmdc, cmdv);
		break;
	    case FOV:
		fov(cmdc, cmdv);
		break;
	    case FP:
		find_plate(cmdc, cmdv);
		break;
	    case FR:
		frame_cmd(cmdc, cmdv);
		break;
	    case GEX:
		gexamine(cmdc, cmdv);
		break;
	    case GPICK:
		gpick(cmdc, cmdv);
		break;
	    case GPOINT:
		gpoint(cmdc, cmdv);
		break;
	    case GRID:
		grid_cmd(cmdc,cmdv);
		break;
	    case HE:
		header(cmdc,cmdv);
		break;
	    case HI:
		hi_cmd(cmdc, cmdv);
		break;
	    case HO:
		ho_cmd(cmdc, cmdv);
		break;
	    case MA:
		map(cmdc, cmdv);
		break;
	    case MARK:
		mark(cmdc, cmdv);
		break;
	    case MAG:
		mag_cmd(cmdc, cmdv);
		break;
	    case MAN:
		manual_page(cmdc, cmdv);
		break;
	    case MD:
		memdump(cmdc, cmdv);
		break;
	    case MO:
		mode_set(cmdc, cmdv);
		break;
	    case MOVIE:
		movie(cmdc, cmdv);
		break;
	    case MOVIE_PREP:
		movie_prep(cmdc, cmdv);
		break;
	    case ONN:
		on(cmdc, cmdv);
		break;
	    case OF:
		off(cmdc, cmdv);
		break;
	    case OR:
		origin(cmdc, cmdv);
		break;
	    case PAINT_BLANK:
		paint_blank(cmdc, cmdv);
		break;
	    case PAUSE:
		pause_cmd(cmdc, cmdv);
		break;
	    case PH:
		print_history(cmdc, cmdv);
		break;
	    case PICK:
		pick(cmdc, cmdv);
		break;
	    case PIXCOORD:
		pixcoord(cmdc, cmdv);
		break;
	    case POINT:
		point(cmdc, cmdv);
		break;
	    case POSTSCRIPT:
		post_cmd(cmdc, cmdv);
		break;
	    case PWD:
		pwd(cmdc, cmdv);
		break;
	    case RA:
		range(cmdc, cmdv);
		break;
	    case RE:
		replicate(cmdc, cmdv);
		break;
	    case REW:
		not_implemented();
		break;
	    case SC:
		scatter_plot(cmdc, cmdv);
		break;
	    case SD:
		screen_dump(cmdc, cmdv);
		break;
	    case SET:
		set(cmdc, cmdv);
		break;
	    case SH:
		set_history(cmdc, cmdv);
		break;
	    case SKY_BOX:
		sky_box(cmdc, cmdv);
		break;
	    case SL:
		slice(cmdc, cmdv);
		break;
	    case SLEEP:
		sleep_cmd(cmdc, cmdv);
		break;
	    case SMONGO:
		smongo(cmdc, cmdv);
		break;
	    case SPLIT:
		split(cmdc, cmdv);
		break;
	    case ST:
		st_cmd(cmdc, cmdv);
		break;
	    case SYNC:
		sync_command(cmdc, cmdv);
		break;
	    case TA:
		take(cmdc, cmdv);
		break;
	    case TABLE:
		table(cmdc, cmdv);
		break;
	    case TEK:
		tek_cmd(cmdc, cmdv);
		break;
	    case TEST:
		stest(cmdc, cmdv);
		break;
	    case TR:
		not_implemented();
		break;
	    case TW:
		not_implemented();
		break;
	    case UNIX:
		done(cmdc, cmdv);
		break;
	    case VE:
		vector(cmdc, cmdv);
		break;
	    case VERSION:
		if(server_mode == FALSE)
		    printf("%s\n%s\n", version, Id);
		else
		{
		    srv_string("make_date", version);
		    srv_string("version", Id+26);
		}
		break;
	    case WE:
		wedgecmd(cmdc, cmdv);
		break;
	    case WRITE:
		write_image(cmdc, cmdv);
		break;
	    case ZAP:
		zap(cmdc, cmdv);
		break;
	    case ZP:
		zoom_pan(cmdc, cmdv);
		break;
	    case QMARK:
		help(cmdc,cmdv);
		break;
	    case COMMENT_LINE:
	    case NIL:
		break;
	    default:
		if(server_mode == FALSE)
		{
		    error1("no such command;  enter ? for list of commands");
		}
		else
		{
		    error1("No such command");
		}
		break;
	}

	fflush(debug_file);
	if (cmd_keyword[0] != '\0')
	    srvflush(cmd_keyword, 1, 1);
    }
}

static void 
do_debug(argc, argv)
int argc;
char *argv[];
{
    int dummy;

    if (argc > 1)
    {
	if (getint(argv[1], &dummy))
	{
	    debug = TRUE;
	    debug_val = dummy;
	}
    }
    else
    {
	debug = !debug;
    }

    if (debug)
	fprintf(debug_file,
	    "ENTERING DEBUG MODE !!!  To get out enter 'de' again\n");
}

static void 
bang(argc, argv)
int argc;
char *argv[];
{
    int i;
    char sysstr[512];

    if (captive)
    {
	error1("bang command not allowed in demo mode");
	return;
    }

    if(server_mode == TRUE)
    {
       error_na("bang command not allowed in server mode");
       return;
    }

    sysstr[0] = '\0';
    for (i=0; i<argc; i++)
	{
	strcat(sysstr, argv[i]);
	strcat(sysstr, " ");
	}
    sysstr[0] = ' ';      /* change bang to space */
    if (debug)
    {
	fprintf(debug_file, "passing to system: '%s'\n", sysstr);
	for (i=0; i<20; i++)
	    fprintf(debug_file, "sysstr[%d] = %d\n", i, sysstr[i]);
    }
    fflush(stdout);    /* make sure that output stays in order */
    i = system(sysstr);
    if (i != 0)
    {
	sprintf(server_str, "bad return from system =%d", i);
	error1(server_str);
    }
}

static void 
echo(argc, argv)
int argc;
char *argv[];
{
    int i;
    char sysstr[80];

    sysstr[0] = '\0';
    for (i=1; i<argc; i++)
    {
	strcat(sysstr, argv[i]);
	strcat(sysstr, " ");
    }
    if(server_mode == FALSE)
	printf("%s\n", sysstr);
    else
    {
	srv_string("echo", sysstr);
    }
}

static void 
pause_cmd(argc, argv)
int argc;
char *argv[];
{
    int dummy_x, dummy_y, dummy_dx, dummy_dy, what;
    char comm[2000];

    if(server_mode == TRUE)
    {
	error_na("pause not allowed in server mode");
	return;
    }

    if (argc > 1)
	printf("%s", argv[1]);
    else
	printf("Press <<Return key>> to continue ");

    fflush(stdout);

    wait_for_event(comm, &what, &dummy_x, &dummy_y, &dummy_dx, &dummy_dy, 0);
    if (debug)
    {
	if (comm[0] != '\0')
	    fprintf(debug_file, 
		"pause terminated when user entered [%s]\n", comm);
    }
}

/*ARGSUSED*/
static void
done(argc, argv)     /* UNIX command */
int argc;
char **argv;
{
    finish();
    srvflush(argv[0], 1, 1);
    exit(0);
}


static void
finish()
{
    cmdfile = stdin;  /* turn off take file -- so that srvflush will print */
    kill_contour_job();
    if (graphics)
	close_graphics();
    if ((debug_file != stdout) && (debug_file != stderr))
	fclose(debug_file);
    fclose(session);
#ifdef NOTDEF
    if (mfd >= 0)
	rewind_tape();
#endif /* NOTDEFF */
}
