/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#define _XOPEN_SOURCE
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>     /* includes strings.h, sys/time.h, and sys/file.h */

#include "parse.h"
#include "skyview.h"
#include "img.h"
#include "img_x.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/file.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <ctype.h>
#include <sys/param.h>
#include "jpeglib.h"
#include <setjmp.h>

#define DISKFILE 999

typedef unsigned long Pixel;

int DEBUG = 0;   /* for xvgifwr */

extern FILE *session, *debug_file;
extern int graphics, control_c;
extern int mfd, debug, server_mode, vax_bytes;
extern char server_str[];
extern int filnbr, tappos;
extern char plane;
extern char *hdr_buf, *hdr_bufp_end;

extern Display *display;
extern int screen;
extern unsigned int depth;
extern Window win;
extern Cursor cursor;
extern XImage *theImage;
extern Visual *visual;
extern GC image_gc;
extern unsigned char *image_org;
extern unsigned int width_org, height_org;

static int selection;
static int color_selection;
static char fname[MAXPATHLEN];
static int do_gif;
static int do_jpeg;
static int do_ppm;
static int do_fullimage;
static int quality;

/* function prototypes */
#ifdef _NO_PROTO

static void ipl_dump();
static void blinn_dump();
static Window pick_a_window();
Window XmuClientWindow ();

#else

static void ipl_dump(void);
static void blinn_dump(void);
static Window pick_a_window(void);
Window XmuClientWindow(Display *, Window);
int WriteGIF(FILE *, unsigned char *, int, int, char *, char *, char *, int, int);

#endif /* _NO_PROTO */

struct my_error_mgr 
{
    struct jpeg_error_mgr pub;    /* "public" fields */
    jmp_buf setjmp_buffer;        /* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
*/

METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
    char jpeg_error_message[JMSG_LENGTH_MAX];

    /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
    my_error_ptr myerr = (my_error_ptr) cinfo->err;

    /* format the message. */
    (*cinfo->err->format_message) (cinfo, jpeg_error_message);
    error1(jpeg_error_message);

    /* Return control to the setjmp point */
    longjmp(myerr->setjmp_buffer, 1);
}

void
screen_dump(argc, argv)
int argc;
char *argv[];
{
    char *tapename;
    int arg_index, tmpnbr, nfilnbr;
    int sel;
#ifdef NOTDEF
    struct mtop  mstop;

    if (mfd < 0)
    {
	tappos = 1;
	filnbr = 0;   /* will be incremented by 1 (default) */
    }
#endif /* NOTDEF */

    /* set up defaults */
#ifdef NOTDEF
    nfilnbr = filnbr + 1;
#endif /* NOTDEF */
    tmpnbr = -1;
    selection = IPL;
    color_selection = WHITE;
    do_gif = 0;
    do_jpeg = 0;
    do_ppm = 0;
    do_fullimage = 0;

    for (arg_index = 1; arg_index < argc; arg_index++)
    {
	if (debug)
	    fprintf(debug_file, 
		"sd: argument %d is =%s=\n", arg_index, argv[arg_index]);
	/* see if its a file number or a workfile name */
	if (isint(argv[arg_index]))
	{
#ifdef NOTDEF
	    /* its a number */
	    if (debug)
		   fprintf(debug_file, "it is a number\n");
	    if (getint(argv[arg_index], &tmpnbr) == FALSE)
		return;
	    if ((*argv[arg_index] == '+') || (*argv[arg_index] == '-'))
	    {
		nfilnbr = filnbr + tmpnbr;
	    }
	    else
	    {
		nfilnbr = tmpnbr;
	    }
#endif /* NOTDEF */
	}
	else
	{
	    sel = cmd(argv[arg_index]);
	    if (debug)
		fprintf(debug_file, "sel = %d\n", sel);
	    switch(sel)
	    {
	    case RED:
	    case GREEN:
	    case BLUE:
		if (debug)
		    fprintf(debug_file, "it is a color selection\n");
		color_selection = sel;
		break;
	    case GIF:
		do_gif = 1;
		break;
	    case JPEG:
		do_jpeg = 1;
		break;
	    case PPM:
		    do_ppm = 1;
		break;
	    case PPMBIN:
		    do_ppm = 2;
		break;
	    case FULLIMAGE:
		do_fullimage = 1;
		break;
	    case IPL:
	    case BLINN:
	    case BLINNCOLOR:
		if (debug)
		    fprintf(debug_file, "it is a dump selection\n");
		selection = sel;
		break;
	    default:
		/* assume its a filename */
		if (debug)
		    fprintf(debug_file, "it is a filename\n");
		strcpy(fname, expand_path(argv[arg_index]));
		selection = DISKFILE;
		break;
	    }
	}
    }
    if (selection != DISKFILE)
    {
#ifdef NOTDEF
    wait((int *)0);    /* let child processes die */
    if (mfd < 0)
    {

	if ((tapename = getenv("STAPE")) == NULL)
	    tapename = "/dev/rmt12";
	mfd = open(tapename, O_RDWR, 0);
	if (mfd < 0)
	{
	    sperror(tapename, errno);
	    return;
	}
    }
    filnbr = nfilnbr;   /* didnt want to change file nbr if file open blew */

    if (debug)
	fprintf(debug_file, 
	"seeking tape filenumber %d   current tappos=%d\n", filnbr, tappos);
    if (filnbr > tappos)
    {
	mstop.mt_op = MTFSF;
	mstop.mt_count = filnbr - tappos;
	ioctl(mfd, MTIOCTOP, &mstop);
    }
    else    /* filnbr <= tappos */
    {
	if (filnbr != 1)
	{
	    mstop.mt_op = MTBSF;
	    mstop.mt_count = tappos - filnbr + 1;
	    ioctl(mfd, MTIOCTOP, &mstop);

	    mstop.mt_op = MTFSF;
	    mstop.mt_count = 1;
	    ioctl(mfd, MTIOCTOP, &mstop);
	}
    }
    tappos = filnbr;
#endif /* NOTDEF */
    }
    else
    {
	if (tmpnbr == -1)
	    quality = 75;
	else
	    quality = tmpnbr;
    }



    switch(selection)
    {
    case IPL:
    case DISKFILE:
	ipl_dump();
	break;
    case BLINN:
    case BLINNCOLOR:
	blinn_dump();
	break;
    }

    if (selection != DISKFILE)
    {
#ifdef NOTDEF
	mstop.mt_op = MTWEOF;    /* write second EOF  */
	mstop.mt_count = 1;
	ioctl(mfd, MTIOCTOP, &mstop);
	mstop.mt_op = MTBSF;   /* backspace over one of them */
	mstop.mt_count = 1;
	ioctl(mfd, MTIOCTOP, &mstop);
#endif /* NOTDEF */
    }
}

static void
blinn_dump()
{
    not_implemented();
}

static void
ipl_dump()
{
    unsigned char ch[3], *pixp, *pixp1;
    int hdr_bytes, blocks, i, y, tfd;
    char *pixbuf, tfilnam[MAXPATHLEN];
    char buf[2880], *bufp;
    register int k;
#ifdef NOTDEF
    struct mtop  mstop;
#endif /* NOTDEF */
    XColor colorcell_defs[256];
    XImage *tmpImage, *thisImage;
    Pixmap pixmap;
    Window sd_window;
    unsigned int sd_width, sd_height;
    Window root;
    int dummyi;
    unsigned int dummy;
    XWindowAttributes win_attributes;
    int sd_x, sd_y, rx, ry, xright, xbelow;
    Window dummywin;
    int byte_pad, redshift, greenshift, blueshift;
    unsigned int *pix32;
    Pixel redmask, greenmask, bluemask, tempval;
    FILE *fd;
    char rmap[256], gmap[256], bmap[256];
    int status, numcols, colorstyle;
    struct img *tmpimg;
    struct jpeg_compress_struct cinfo;
    struct my_error_mgr jerr;
    /* JSAMPROW row_pointer[1]; */      /* pointer to JSAMPLE row[s] */
    int row_stride;               /* physical row width in image buffer */
    JSAMPARRAY buffer;            /* Output row buffer */
    JSAMPLE *buffer_ptr;


    if (!graphics)
    {
	error1("sd command not allowed in no-graphics mode");
	return;
    }

    if ((depth > 8) && do_gif)
    {
	error1("no gif screendump in 24 plane mode - suggest ppm or ppmbin");
	return;
    }

    if(server_mode == FALSE)
    {
       printf("\nPlease select the window to be dumped by clicking the\n");
       printf("       mouse in that window.\n");
    }

    if(server_mode == TRUE)
        sd_window = win;
    else
        sd_window = pick_a_window();

    if (sd_window) 
    {
	if (XGetGeometry (display, sd_window, &root, &dummyi, &dummyi,
		    &sd_width, &sd_height, &dummy, &dummy) &&
	sd_window != root)
	{
	    /*
	    printf("cursor is not in root window\n");
	    */
	    sd_window = XmuClientWindow (display, sd_window);
	}
    }
    else
    {
	error1("sdump:  error finding which window contains cursor");
	return;
    }

    if (!XGetWindowAttributes(display, sd_window, &win_attributes))
    {
	error1("sdump error:  can't get window attributes");
	return;
    }

    /* do colors */
    if (depth > 8)
    {
	for (i = 0; i < 256; i++)
	    colorcell_defs[i].pixel = (i << 16) + (i << 8) + i;
    }
    else
    {
	for (i = 0; i < 256; i++)
	    colorcell_defs[i].pixel = i;
    }
    XQueryColors(display, win_attributes.colormap, colorcell_defs, 256);

    if (do_fullimage)
    {
	tmpimg = find_WIN(sd_window);
	if (tmpimg == NULL)
	{
	    error1("screen_dump error:  cannot do a fullimage dump of a non Skyview windows");
	    return;
	}
	pixmap = XCreatePixmap(display, win, width_org, height_org, 
	    theImage->depth);
	if (pixmap == 0)
	{
	    error1("screen_dump error:  cannot allocate space for pixmap");
	    return;
	}
	tmpImage = XCreateImage(display, visual, theImage->depth,
	    theImage->format, 0, (char *) image_org, 
	    width_org, height_org, 8, 0);
	sd_width = width_org;
	sd_height = height_org;
	XPutImage(display, pixmap, image_gc, tmpImage,
	    0, 0, 0, 0, width_org, height_org);
	tmpImage->data = NULL;
	XDestroyImage(tmpImage);
	repaint_graphics(pixmap);   /* repaint the graphics into this pixmap */
	thisImage = XGetImage(display, pixmap, 0, 0, width_org, height_org,
	    (unsigned long) AllPlanes, ZPixmap);
	XFreePixmap(display, pixmap);
	if (thisImage == NULL)
	{
	    error1("screen_dump:  cannot read the pixmap");
	    return;
	}
	thisImage->red_mask = theImage->red_mask;
	thisImage->green_mask = theImage->green_mask;
	thisImage->blue_mask = theImage->blue_mask;
    }
    else
    {
    XGetGeometry (display, sd_window, &root, &dummyi, &dummyi,
	&sd_width, &sd_height, &dummy, &dummy);


    /* now clip at screen boundaries to avoid BadMatch error in XGetImage */
    (void) XTranslateCoordinates(display, sd_window, win_attributes.root, 0, 0,
	&rx, &ry, &dummywin);
    xright = DisplayWidth(display, screen) - rx
	- win_attributes.border_width * 2 - win_attributes.width;
    xbelow = DisplayHeight(display, screen) - ry
	- win_attributes.border_width * 2 - win_attributes.height;

    sd_x = 0;
    if (rx < 0)
    {
	sd_x -= rx;
	sd_width += rx;
    }
    if (xright < 0)
    {
	sd_width += xright;
    }

    sd_y = 0;
    if (ry < 0)
    {
	sd_y -= ry;
	sd_height += ry;
    }
    if (xbelow < 0)
    {
	sd_height += xbelow;
    }
    
    thisImage = XGetImage(display, sd_window, sd_x, sd_y, sd_width, sd_height,
	(unsigned long) AllPlanes, ZPixmap);
    if (thisImage == NULL)
    {
	error1("sdump:  cannot access the window");
	return;
    }
    }

    byte_pad = thisImage->bitmap_pad / 8;
    if (thisImage->depth > 8)
    {
	redmask = thisImage->red_mask;
	greenmask = thisImage->green_mask;
	bluemask = thisImage->blue_mask;

	/* find out how much to shift each pixval */
	/* i.e how many shifts needed to right justify mask value */
	/* first red */
	tempval = thisImage->red_mask;
	redshift = 0;
	while ((tempval & 0x1) == 0)
	{
	    redshift++;
	    tempval >>= 1;
	}

	/* next green */
	tempval = thisImage->green_mask;
	greenshift = 0;
	while ((tempval & 0x1) == 0)
	{
	    greenshift++;
	    tempval >>= 1;
	}

	/* last blue */
	tempval = thisImage->blue_mask;
	blueshift = 0;
	while ((tempval & 0x1) == 0)
	{
	    blueshift++;
	    tempval >>= 1;
	}
    }

    if ((do_gif) || (do_ppm) || (do_jpeg))
    {
	if (thisImage->depth == 8)
	{
	    /* first get rid of the end_of_line padding */
	    pixp = (unsigned char *) thisImage->data;
	    pixp1 = pixp;
	    for (y = 0; y < sd_height; y++)
	    {
		for (k=0; k < sd_width; k++)
		{
		    /* move pixel */
		    *pixp1++ = *pixp++;
		}

		while ((k % byte_pad) != 0)
		{
		    pixp++;
		    k++;
		}
	    }
	}

	fd = fopen(fname, "wb");
	if (fd == NULL)
	{
	    sperror(fname, errno);
	    return;
	}
	if (do_gif)
	{
	    for (i = 0; i < 256; i++)
	    {
		rmap[i] = colorcell_defs[i].red >> 8;
		gmap[i] = colorcell_defs[i].green >> 8;
		bmap[i] = colorcell_defs[i].blue >> 8;
	    }
	    numcols = 256;
	    colorstyle = 0;
	    WriteGIF(fd, (unsigned char *) thisImage->data, 
		sd_width, sd_height, 
		rmap, gmap, bmap, numcols, colorstyle);
	}
	else if (do_ppm == 1)
	{
	    fprintf(fd, "P3\n");
	    fprintf(fd, "%d %d\n", sd_width, sd_height);
	    fprintf(fd, "255\n");
	    pixp = (unsigned char *) thisImage->data;
	    pix32 = (unsigned int *) thisImage->data;
	    if (thisImage->depth > 8)
	    {
		for (i = 0; i < sd_width*sd_height; i++)
		{
		    fprintf(fd, "%d %d %d ", 
			colorcell_defs[((*pix32) & redmask)
			    >> redshift].red >> 8,
			colorcell_defs[((*pix32) & greenmask)
			    >> greenshift].green >> 8,
			colorcell_defs[((*pix32) & bluemask)
			    >> blueshift].blue >> 8);
		    pix32++;
		    if ( (i % 6) == 5)
			fprintf(fd, "\n");
		}
	    }
	    else
	    {
		for (i = 0; i < sd_width*sd_height; i++)
		{
		    fprintf(fd, "%d %d %d ", 
			colorcell_defs[*pixp].red >> 8,
			colorcell_defs[*pixp].green >> 8,
			colorcell_defs[*pixp].blue >> 8);
		    pixp++;
		    if ( (i % 6) == 5)
			fprintf(fd, "\n");
		}
	    }
	}

	else if (do_ppm == 2)
	{
	    fprintf(fd, "P6\n");
	    fprintf(fd, "%d %d\n", sd_width, sd_height);
	    fprintf(fd, "255\n");
	    pixp = (unsigned char *) thisImage->data;
	    pix32 = (unsigned int *) thisImage->data;
	    if (thisImage->depth > 8)
	    {
		for (i = 0; i < sd_width*sd_height; i++)
		{
		    ch[0] = colorcell_defs[((*pix32) & redmask)
			>> redshift].red >> 8;
		    ch[1] = colorcell_defs[((*pix32) & greenmask)
			>> greenshift].green >> 8;
		    ch[2] = colorcell_defs[((*pix32) & bluemask)
			>> blueshift].blue >> 8;
		    fwrite(ch, sizeof(ch[0]), 3, fd);
		    pix32++;
		}
	    }
	    else
	    {
		for (i = 0; i < sd_width*sd_height; i++)
		{
		    ch[0] = colorcell_defs[*pixp].red >> 8,
		    ch[1] = colorcell_defs[*pixp].green >> 8,
		    ch[2] = colorcell_defs[*pixp].blue >> 8;
		    fwrite(ch, sizeof(ch[0]), 3, fd);
		    pixp++;
		}
	    }
	}
	else if (do_jpeg)
	{
	    pixp = (unsigned char *) thisImage->data;
	    pix32 = (unsigned int *) thisImage->data;

    /* We have to set up the error handler first, in case the initialization
    * step fails.  (Unlikely, but it could happen if you are out of memory.)
    * This routine fills in the contents of struct jerr, and returns jerr's
    * address which we place into the link field in cinfo.
    */
	    cinfo.err = jpeg_std_error(&jerr.pub);
	    jerr.pub.error_exit = my_error_exit;
	    /* Establish the setjmp return context for my_error_exit to use. */
	    if (setjmp(jerr.setjmp_buffer)) 
	    {
		/* If we get here, the JPEG code has signaled an error.
		* We need to clean up the JPEG object, close the input file
		*/
		jpeg_destroy_compress(&cinfo);
		fclose(fd);
		goto end_sd;
	    }
	    /* Now we can initialize the JPEG compression object. */
	    jpeg_create_compress(&cinfo);
	    jpeg_stdio_dest(&cinfo, fd);
  /* Step 3: set parameters for compression */

  /* First we supply a description of the input image.
   * Four fields of the cinfo struct must be filled in:
   */
  cinfo.image_width = sd_width; 	/* image width and height, in pixels */
  cinfo.image_height = sd_height;
  cinfo.input_components = 3;		/* # of color components per pixel */
  cinfo.in_color_space = JCS_RGB; 	/* colorspace of input image */
  /* Now use the library's routine to set default compression parameters.
   * (You must set at least cinfo.in_color_space before calling this,
   * since the defaults depend on the source color space.)
   */
  jpeg_set_defaults(&cinfo);
  /* Now you can set any non-default parameters you wish to.
   * Here we just illustrate the use of quality (quantization table) scaling:
   */
  jpeg_set_quality(&cinfo, quality, TRUE /* limit to baseline-JPEG values */);

  /* Step 4: Start compressor */

  /* TRUE ensures that we will write a complete interchange-JPEG file.
   * Pass TRUE unless you are very sure of what you're doing.
   */
  jpeg_start_compress(&cinfo, TRUE);

  /* Step 5: while (scan lines remain to be written) */
  /*           jpeg_write_scanlines(...); */

  /* Here we use the library's state variable cinfo.next_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   * To keep things simple, we pass one scanline per call; you can pass
   * more if you wish, though.
   */
  row_stride = sd_width * 3;	/* JSAMPLEs per row in image_buffer */
  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
    ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

  enable_control_c();

  while (cinfo.next_scanline < cinfo.image_height) {
    /* jpeg_write_scanlines expects an array of pointers to scanlines.
     * Here the array is only one element long, but you could pass
     * more than one scanline at a time if that's more convenient.
     */
		buffer_ptr = &buffer[0][0];
	    if (thisImage->depth > 8)
	    {
		for (i = 0; i < sd_width; i++)
		{
		    *buffer_ptr++ = colorcell_defs[((*pix32) & redmask)
			>> redshift].red >> 8;
		    *buffer_ptr++ = colorcell_defs[((*pix32) & greenmask)
			>> greenshift].green >> 8;
		    *buffer_ptr++ = colorcell_defs[((*pix32) & bluemask)
			>> blueshift].blue >> 8;
		    pix32++;
		}
	    }
	    else
	    {
		for (i = 0; i < sd_width; i++)
		{
		    *buffer_ptr++ = colorcell_defs[*pixp].red >> 8,
		    *buffer_ptr++ = colorcell_defs[*pixp].green >> 8,
		    *buffer_ptr++ = colorcell_defs[*pixp].blue >> 8;
		    pixp++;
		}
	    }
    (void) jpeg_write_scanlines(&cinfo, buffer, 1);
    if (control_c)
	break;
  }

  /* Step 6: Finish compression */

  if (!control_c)
      jpeg_finish_compress(&cinfo);
  /* After finish_compress, we can close the output file. */

  /* Step 7: release JPEG compression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_compress(&cinfo);

  /* And we're done! */
	}

	status = fclose(fd);
	if (status)
	{
	    sperror(fname, errno);
	    goto end_sd;
	}
	goto end_sd;
    }

    if (debug)
	fprintf(debug_file, "doing ipl dump\n");

    pixbuf = malloc(sd_width);

    if (selection == DISKFILE)
    {
	/* set up FITS header */
	for (i=0, bufp = buf; i<2880; i++, bufp++)
	    *bufp = ' ';
	strncpy(buf, "END", 3);
	hdr_buf = buf;
	(void) hdrinit(2880);
	hdrlogical("SIMPLE", 'T');
	hdrint("BITPIX", 8);
	hdrint("NAXIS", 3);
	hdrint("NAXIS1", (int) sd_width);
	hdrint("NAXIS2", (int) sd_height);
	hdrint("NAXIS3", 1);
	hdrint("CDELT2", -1);
	hdrlogical("BLOCKED", 'T');
	comment("SCREEN DUMP FILE FROM S");

	blocks = (hdr_bufp_end - hdr_buf) / 2880;
	if ((blocks * 2880) != (hdr_bufp_end - hdr_buf))
	    blocks++;  /* round up to next integral number of blocks */
	hdr_bytes = blocks * 2880;
    }

    /* first do red */
    if ((color_selection != BLUE) && (color_selection != GREEN))
    {
    if (selection == DISKFILE)
    {
	/* open the file */

	strcpy(tfilnam, fname);
	strcat(tfilnam, ".r");
	if(server_mode == FALSE)
	    printf("opening disk file: %s\n", tfilnam);
	tfd = open(tfilnam, O_RDWR | O_CREAT | O_TRUNC, 0644);
	if (tfd < 0)
	{
	    sperror(tfilnam, errno);
	    goto abort_sd;
	}

	write(tfd, hdr_buf, hdr_bytes);    /* write FITS header */
    }
    else
    {
#ifdef NOTDEF
	tfd = mfd;
	if(server_mode == FALSE)
	    printf("now writing red file to tape\n");
#endif /* NOTDEF */
    }

    pix32 = (unsigned int *) thisImage->data;  /* in case it's 32 bit */
    pixp = (unsigned char *) thisImage->data;  /* in case it's 8 bit */

    for (y = 0; y < sd_height; y++)
    {

	/* massage pixels here */

	if (thisImage->depth > 8)
	{
	    for (k=0; k < sd_width; k++)
	    {
		/* apply color table */
		pixbuf[k] = colorcell_defs[((*pix32++) & redmask) >> redshift].red >> 8;
	    }
	}
	else
	{
	    for (k=0; k < sd_width; k++)
	    {
		/* apply color table */
		pixbuf[k] = colorcell_defs[*pixp++].red >> 8;
	    }

	    /* now, since the Sun server gives us a multiple of 4 bytes for
	    each line - (byte_pad = 4;  this is surely machine dependent) */
	    /* I think that the 4 is found from thisImage->bitmap_pad / 8 */
	    while ((k % byte_pad) != 0)
	    {
		pixp++;
		k++;
	    }
	}

	if (!vax_bytes)
	{
	    /* IPL has vaxes - write in vax order */
	    if (selection != DISKFILE)
		swab(pixbuf, pixbuf, sd_width);
	}

	write(tfd, pixbuf, sd_width);
    }

    if (selection == DISKFILE)
    {

	status = close(tfd);
	if (status == -1)
	{
	    sperror(tfilnam, errno);
	    goto abort_sd;
	}
    }
    else
    {
#ifdef NOTDEF
	mstop.mt_op = MTWEOF;    /* write EOF on tape */
	mstop.mt_count = 1;
	ioctl(mfd, MTIOCTOP, &mstop);

	tappos++;
	filnbr = tappos - 1;
#endif /* NOTDEF */
    }
    }

    /* now do green */
    if ((color_selection != BLUE) && (color_selection != RED))
    {
    if (selection == DISKFILE)
    {
	/* open the file */

	strcpy(tfilnam, fname);
	strcat(tfilnam, ".g");
	if(server_mode == FALSE)
	    printf("opening disk file: %s\n", tfilnam);
	tfd = open(tfilnam, O_RDWR | O_CREAT | O_TRUNC, 0644);
	if (tfd < 0)
	{
	    sperror(tfilnam, errno);
	    goto abort_sd;
	}

	write(tfd, hdr_buf, hdr_bytes);    /* write FITS header */
    }
    else
    {
#ifdef NOTDEF
	tfd = mfd;
	if(server_mode == FALSE)
	    printf("now writing green file to tape\n");
#endif /* NOTDEF */
    }

    pix32 = (unsigned int *) thisImage->data;  /* in case it's 32 bit */
    pixp = (unsigned char *) thisImage->data;  /* in case it's 8 bit */

    for (y = 0; y < sd_height; y++)
    {

	/* massage pixels here */

	if (thisImage->depth > 8)
	{
	    for (k=0; k < sd_width; k++)
	    {
		/* apply color table */
		pixbuf[k] = colorcell_defs[((*pix32++) & greenmask) >> greenshift].green >> 8;
	    }
	}
	else
	{
	    for (k=0; k < sd_width; k++)
	    {
		/* apply color table */
		pixbuf[k] = colorcell_defs[*pixp++].green >> 8;
	    }

	    /* now, since the Sun server gives us a multiple of 4 bytes for
	    each line - (byte_pad = 4;  this is surely machine dependent) */
	    /* I think that the 4 is found from thisImage->bitmap_pad / 8 */
	    while ((k % byte_pad) != 0)
	    {
		pixp++;
		k++;
	    }
	}

    if (!vax_bytes)
    {
	if (selection != DISKFILE)
	    swab(pixbuf, pixbuf, sd_width);
    }

    write(tfd, pixbuf, sd_width);
    }

    if (selection == DISKFILE)
    {
	status = close(tfd);
	if (status == -1)
	{
	    sperror(tfilnam, errno);
	    goto abort_sd;
	}
    }
    else
    {
#ifdef NOTDEF
	mstop.mt_op = MTWEOF;    /* write EOF on tape */
	mstop.mt_count = 1;
	ioctl(mfd, MTIOCTOP, &mstop);

	tappos++;
	filnbr = tappos - 1;
#endif /* NOTDEF */
    }
    }

    /* now do blue */
    if ((color_selection != GREEN) && (color_selection != RED))
    {
    if (selection == DISKFILE)
    {
	/* open the file */

	strcpy(tfilnam, fname);
	strcat(tfilnam, ".b");
	if(server_mode == FALSE)
	    printf("opening disk file: %s\n", tfilnam);
	tfd = open(tfilnam, O_RDWR | O_CREAT | O_TRUNC, 0644);
	if (tfd < 0)
	{
	    sperror(tfilnam, errno);
	    goto abort_sd;
	}

	write(tfd, hdr_buf, hdr_bytes);    /* write FITS header */
    }
    else
    {
	tfd = mfd;
	if(server_mode == FALSE)
	    printf("now writing blue file to tape\n");
    }


    pix32 = (unsigned int *) thisImage->data;  /* in case it's 32 bit */
    pixp = (unsigned char *) thisImage->data;  /* in case it's 8 bit */

    for (y = 0; y < sd_height; y++)
    {

	/* massage pixels here */

	if (thisImage->depth > 8)
	{
	    for (k=0; k < sd_width; k++)
	    {
		/* apply color table */
		pixbuf[k] = colorcell_defs[((*pix32++) & bluemask) >> blueshift].blue >> 8;
	    }
	}
	else
	{
	    for (k=0; k < sd_width; k++)
	    {
		/* apply color table */
		pixbuf[k] = colorcell_defs[*pixp++].blue >> 8;
	    }

	    /* now, since the Sun server gives us a multiple of 4 bytes for
	    each line - (byte_pad = 4;  this is surely machine dependent) */
	    /* I think that the 4 is found from thisImage->bitmap_pad / 8 */
	    while ((k % byte_pad) != 0)
	    {
		pixp++;
		k++;
	    }
	}

	if (!vax_bytes)
	{
	    if (selection != DISKFILE)
		swab(pixbuf, pixbuf, sd_width);
	}

	write(tfd, pixbuf, sd_width);
    }

    if (selection == DISKFILE)
    {
	status = close(tfd);
	if (status == -1)
	{
	    sperror(tfilnam, errno);
	    goto abort_sd;
	}
    }
    else
    {
#ifdef NOTDEF
	mstop.mt_op = MTWEOF;    /* write EOF on tape */
	mstop.mt_count = 1;
	ioctl(mfd, MTIOCTOP, &mstop);

	tappos++;
	filnbr = tappos - 1;
#endif /* NOTDEF */
    }
    }

    if (selection != DISKFILE)
    {
    if(server_mode == FALSE)
	printf("Tell IPL that image is %d lines by %d samples\n", 
	    sd_height + 47, sd_width);
    fprintf(session, "Tell IPL that image is %d lines by %d samples\n", 
	sd_height + 47, sd_width);
    }

abort_sd:
    free(pixbuf);
end_sd:
    XDestroyImage(thisImage);
}

/*ARGSUSED*/
void FatalError(msg)
char *msg;
{
}




static Window pick_a_window()
{
    Window win1;
    Window root;
    XEvent report;
    int done, got_buttonpress;

    root = RootWindow(display,screen);
    if (XGrabPointer(display, root, False,
	(unsigned) (ButtonPressMask|ButtonReleaseMask), GrabModeSync,
	GrabModeAsync, root, cursor, CurrentTime) != GrabSuccess)
    {
	error1("unable to grab the pointer");
	return(None);
    }

    got_buttonpress = FALSE;
    done = FALSE;
    while (!done)
    {
	/* let server process events until we get a button event */
	XAllowEvents(display, SyncPointer, CurrentTime);

	/* wait here for the button event */
	XWindowEvent(display, root, ButtonPressMask|ButtonReleaseMask, &report);
	switch (report.type) 
	{
	    case ButtonPress:
		win1 = report.xbutton.subwindow;
		if (win1 == None)
		    win1 = root;
		got_buttonpress = TRUE;
		break;
	    case ButtonRelease:
		if (got_buttonpress)
		    done = TRUE;
		break;
	}
    } 
    XUngrabPointer(display, CurrentTime);
    return(win1);
}
