/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "parse.h"
#include "fits.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>


extern FILE *session, *debug_file;
extern int band_offset, reqband, debug, server_mode, req_mode, maptype, u16;
extern int mode_used;
extern char server_str[];
extern int x_orig, y_orig;
extern int clip_top, clip_bottom, clip_left, clip_right;
extern double bscale, b_zero;
extern char filnam[], mod_filnam[], radecsys[];
extern int bitpix, minx, maxx, miny, maxy;
extern int x_offset, sgn_y, y_offset;
extern struct hdrstruct filehdr;
extern struct framestruct *frameptr;
extern char bunit[], ctype1[], ctype2[];
extern double glong, glat, crval3, cdelt1, cdelt2, twist, file_equinox;
extern double cd1_1, cd1_2, cd2_1, cd2_2;
extern double dc1_1, dc1_2, dc2_1, dc2_2;
extern int using_cd;
extern double	plate_ra;	/* Right ascension of plate center */
extern double	plate_dec;	/* Declination of plate center */
extern double	x_pixel_offset;	/* X pixel offset of image lower right */
extern double	y_pixel_offset;	/* Y pixel offset of image lower right */
extern double	x_pixel_size;	/* X pixel_size */
extern double	y_pixel_size;	/* Y pixel_size */
extern double	ppo_coeff[6];
extern double	amd_x_coeff[20]; /* X coefficients for plate model */
extern double	amd_y_coeff[20]; /* Y coefficients for plate model */
extern double	plt_scale; /* plate scale arc sec per mm */

extern int pixels, lines;
extern double blank;
extern int blank_set;
extern double crpix1, crpix2, iraf_max, iraf_min;
extern char ctype1sav[], ctype2sav[], object[];
extern double base, noise;
extern int dskygrid, naxis, naxis3;
extern int file_jsys, file_coord_sys;
extern double slow, shigh;    /* range stretch used on this image */
extern int samp_disp;   /* pixel displacement from left edge to 1st desired */
extern int line_disp;  /* line displacement from top edge to 1st desired line */
extern double win_zoom;
extern unsigned int width_org, height_org;
extern int graphics_window;

extern int map_distortion;    /* flag for SIRTF distortion corrections */
extern int a_order, ap_order, b_order, bp_order;   /* distortion corrections */
extern double a[5][5], ap[5][5], b[5][5], bp[5][5]; /*  distortion corrections */

extern int output_coord_sys, output_jsys;
extern double output_equinox;

extern int imio_frame, imio_config, imio_ok;
extern char *imio_filptr, wcsbuf[];
extern int image_frame;

extern double report_blank;  /* user's notion of BLANK: */
	    /* same as "blank" except for IRAF floating point images painted */
	    /* with either BLANK= keyword or "set blank" value */

struct img *curr_img = NULL;  /* pointer to current image */
struct img head_img = {&head_img, &head_img};

static double   dtr = M_PI/180;


/* function prototypes */
#ifdef _NO_PROTO
static int check_COORD();
#else
static int check_COORD(double coord[], double d_coord[]);
#endif /* _NO_PROTO */

void
sho_curr_img(argc, argv)
int argc;
char *argv[];
{
    char fname[MAXPATHLEN];
    struct img *imgp;
    double d_coord[2];
    char *coord_str;
    char s_coord[2][40];
    int coord_sys;
    int jsys;
    double equinox;

    if (argc == 1)
    {
	if (curr_img == NULL)
	    error1("no current image");
	else
	{
	    if(server_mode == FALSE)
	    {
		if (mod_filnam[0] == '\0')
		    printf("current image: %s\n", filnam);
		else
		    printf("current image: %s (modified)\n", filnam);

		printf("  painted with range %g to %g\n", 
		    dn_to_jy(slow), dn_to_jy(shigh));
		printf("  projection type:  %s\n", maptype_string(maptype));
		if ((file_coord_sys == EQ) || (file_coord_sys == EC))
		{
		    printf("  coordinates:  %s  %c%6.1f\n", 
			coord_name(file_coord_sys),
			(file_jsys == 1 || file_jsys == 3)?'B':'J', 
			file_equinox);
		}
		else
		{
		    printf("  coordinates:  %s\n", 
			coord_name(file_coord_sys));
		}
		if (!blank_set)
		    printf("  blank           undefined\n");
		else if (bitpix < 0)
		    printf("  blank           %g\n", report_blank);
		else
		    printf("  blank           %g\n", blank);
		if (bitpix == 16)
		    printf("  %s 16 bit image\n", u16?"unsigned":"signed");
		printf("  mode %s\n", (mode_used == LIN)? "linear" : "logarithmic");
		printf("  frame %d\n", image_frame);
		d_get_center_xy(d_coord);
		dSC_to_IM(d_coord);
		coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, 
		    &equinox, &jsys);
		printf("Zoom factor %f\nCentered on %s\n", win_zoom, coord_str);
	    }
	    else
	    {
		sprintf(server_str, "%s%s",
		    filnam, mod_filnam[0]?" (modified)":"");
		srv_string("filename", server_str);

		sprintf(server_str, "%-g", dn_to_jy(slow));
		srv_real("range_min_val", server_str);
		sprintf(server_str, "%-g", dn_to_jy(shigh));
		srv_real("range_max_val", server_str);

		srv_string("projection_type", maptype_string(maptype));
		srv_string("file_coord_sys", coord_name(file_coord_sys));
		if ((file_coord_sys == EQ) || (file_coord_sys == EC))
		{
		    sprintf(server_str, "%c%6.1f", 
			(file_jsys == 1 || file_jsys == 3)?'B':'J', 
			file_equinox);
		    srv_string("file_equinox", server_str);
		}

		if (!blank_set)
		{
		    srv_string("blank", "undefined");
		}
		else if (bitpix < 0)
		{
		    sprintf(server_str, "%g", report_blank);
		    srv_real("blank", server_str);
		}
		else
		{
		    sprintf(server_str, "%g", blank);
		    srv_real("blank", server_str);
		}

		srv_string("mode", (mode_used == LIN)? "linear" : "logarithmic");
		sprintf(server_str, "%d", image_frame);
		srv_real("frame", server_str);
		d_get_center_xy(d_coord);
		dSC_to_IM(d_coord);
		coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, 
		    &equinox, &jsys);
		sprintf(server_str, "%f", win_zoom);
		srv_real("zoom_factor", server_str);
		srv_coord("center", coord_sys, s_coord, equinox, jsys);
	    }
	}
    }
    else
    {

	strcpy(fname, expand_path(argv[1]));
	imgp = find_FILNAM(fname);
	if (imgp == NULL)
	{
	    sprintf(server_str,"No image has been painted from %s", fname);
	    error1(server_str);
	}
	else
	{
	    update_cur_window();
	}
    }
}


void
curr_map_parms(argc, argv)
int argc;
char *argv[];
{
    double d_coord[2];
    double d_coordt[2];
    char s_coord[2][40];
    char s_coordt[2][40];
    double xsize, ysize;

    double radius;
    double dlon, dlat, lon_min, lon_max, lat_min, lat_max;
    int coord_sys, target_coord_sys;
    char *cmdv[10];
    int cmdc;
    char jchar;
    char file_eq_str[100];
    char target_eq_str[100];
    int proj_jsys, target_jsys;
    double proj_equinox, target_equinox;

    if (curr_img == NULL)
	error1("no current image");
    else
    {

	d_coord[0] = ((double)width_org / 2.) - 0.5;
	d_coord[1] = ((double)height_org / 2.) - 0.5;

	d_coordt[0] = d_coord[0];
	d_coordt[1] = d_coord[1];

	dSC_to_IM(d_coord);
	dSC_to_IM(d_coordt);

	save_coord_sys();

	/* first get map center in target coord system, (specified by argv) */
	if(argc > 1)
	   set_coord_sys3(argc, argv, 1);   /* change to target system */
	 
	(void) dimage_to_sky(d_coordt, s_coordt, &coord_sys, 
	    &target_equinox, &target_jsys);
	str_to_coord(s_coordt, d_coordt);
	target_coord_sys = output_coord_sys;
	if ((output_coord_sys == EQ) || (output_coord_sys == EC))
	{
	    jchar = 'B';
	    if ((output_jsys == 0) || (output_jsys == 13))
		jchar = 'J';
	    sprintf(target_eq_str, "%c%f", jchar, output_equinox);
	}
	else
	{
	    target_eq_str[0] = '\0';
	}


	/* now get map center in file coord system */
	cmdv[1] = coord_name(file_coord_sys);
	if ((file_coord_sys == EQ) || (file_coord_sys == EC))
	{
	    jchar = 'B';
	    if ((file_jsys == 0) || (file_jsys == 13))
		jchar = 'J';
	    sprintf(file_eq_str, "%c%f", jchar, file_equinox);
	    cmdv[2] = file_eq_str;
	    cmdv[3] = "output";
	    cmdc = 4;
	}
	else
	{
	    file_eq_str[0] = '\0';
	    cmdv[2] = "output";
	    cmdc = 3;
	}
	set_coord_sys3(cmdc, cmdv, 1);   /* change to file's coord system */

	(void) dimage_to_sky(d_coord, s_coord, &coord_sys, 
	    &proj_equinox, &proj_jsys);
	str_to_coord(s_coord, d_coord);
	restore_coord_sys();

	if (maptype == PLATE)
	{
	    xsize = fabs((double) width_org *
		plt_scale * x_pixel_size / 1000 / 3600);
	    ysize = fabs((double) height_org *
		plt_scale * y_pixel_size / 1000 / 3600);
	}
	else
	{
	    xsize = fabs((double) width_org * cdelt1);
	    ysize = fabs((double) height_org * cdelt2);
	}

	radius = sqrt(xsize*xsize + ysize*ysize)/2.;

        if(radius > 90.)
	   dlon = 360.;
        else if(d_coordt[1] < 90.)
	   dlon = asin(sin(radius * dtr)/cos(d_coordt[1] * dtr))/dtr;
        else
	   dlon = 360.;
  
        dlat = radius;
  
        lat_max = d_coordt[1] + dlat;
        if(lat_max > 90.)
	   dlon = 360;
  
        lat_min = d_coordt[1] - dlat;
        if(lat_min < -90.)
	   dlon = 360;
  
        lon_max = d_coordt[0] + dlon;
        if(lon_max > 360.)
	   lon_max = 0;

        lon_min = d_coordt[0] - dlon;
        if(lon_min < 0.)
	   lon_min = 0;

	if(server_mode == FALSE)
	{
	    printf("Projection:         %s\n", maptype_string(maptype));

	    printf("Projection coord sys: %s  %s\n", 
	        coord_name(file_coord_sys), file_eq_str);

	    printf("Projection center: (%11.6f, %11.6f)\n", glong, glat);

	    printf("Map center:        (%11.6f, %11.6f)\n", 
	        d_coord[0], d_coord[1]);

	    printf("in target system:  (%11.6f, %11.6f) %s  %s\n", 
	        d_coordt[0], d_coordt[1], coord_name(target_coord_sys),
		target_eq_str);

	    printf("Size (showing):    %-gx%-g degrees\n", xsize, ysize);

	    printf("Search box:        %-g to %-g (longitude)\n", 
	        lon_min, lon_max);

	    printf("                   %-g to %-g (latitude)\n", 
	        lat_min, lat_max);
	}
	else
	{
	    srv_string("projection_type", maptype_string(maptype));

	    sprintf(s_coord[0], "%11.6f", glong);
	    sprintf(s_coord[1], "%11.6f", glat);
	    srv_coord("proj_center", file_coord_sys, s_coord, 
		proj_equinox, proj_jsys);

	    sprintf(s_coord[0], "%11.6f", d_coord[0]);
	    sprintf(s_coord[1], "%11.6f", d_coord[1]);
	    srv_coord("map_center", file_coord_sys, s_coord, 
		proj_equinox, proj_jsys);

	    sprintf(server_str, "%-g", xsize);
	    srv_real("xsize", server_str);
	    sprintf(server_str, "%-g", ysize);
	    srv_real("ysize", server_str);

	    sprintf(s_coord[0], "%11.6f", d_coordt[0]);
	    sprintf(s_coord[1], "%11.6f", d_coordt[1]);
	    srv_coord("target_map_center", target_coord_sys, s_coord, 
		target_equinox, target_jsys);

	    sprintf(server_str, "%-g", lon_min);
	    srv_real("lon_min", server_str);
	    sprintf(server_str, "%-g", lon_max);
	    srv_real("lon_max", server_str);

	    sprintf(server_str, "%-g", lat_min);
	    srv_real("lat_min", server_str);
	    sprintf(server_str, "%-g", lat_max);
	    srv_real("lat_max", server_str);
	    sprintf(server_str, "%f", win_zoom);
	    srv_real("win_zoom", server_str);
	    sprintf(server_str, "%d", lines);
	    srv_real("lines", server_str);
	    sprintf(server_str, "%d", pixels);
	    srv_real("pixels", server_str);
	}
    }
}

void
chgband(argc, argv)
int argc;
char *argv[];
{
    if (argc == 1)
    {
	if(server_mode == FALSE)
	    printf("requested plane = %d\n", reqband); 
    }
    else if (argc == 2)
    {
	reqband = atoi(argv[1]);
    }
    else
    {
	error1("wrong number of arguments to PL command");
	return;
    }

    if(server_mode == TRUE)
    {
	sprintf(server_str, "%d", reqband);
	srv_real("plane", server_str);
    }
}

void
clip(argc, argv)
int argc;
char *argv[];
{
    int i;

    if (argc == 1)
    {
	if(server_mode == FALSE)
	    printf("clipping TOP %d  BOTTOM %d  LEFT %d  RIGHT %d\n", 
		clip_top, clip_bottom, clip_left, clip_right);
	fprintf(session, "clipping TOP %d  BOTTOM %d  LEFT %d  RIGHT %d\n", 
	    clip_top, clip_bottom, clip_left, clip_right);
    }
    else
    {
	for (i = 1; i < argc; i += 2)
	{
	    if ((i + 1) == argc)
	    {
		error1("bad pairing of arguments to CL command");
		return;
	    }
	    switch(cmd(argv[i]))
	    {
		case TOP:
		    if (!getint(argv[i + 1], &clip_top))
			return;
		    break;
		case BOTTOM:
		    if (!getint(argv[i + 1], &clip_bottom))
			return;
		    break;
		case RIGHT:
		    if (!getint(argv[i + 1], &clip_right))
			return;
		    break;
		case LEFT:
		    if (!getint(argv[i + 1], &clip_left))
			return;
		    break;
		default:
		    error1("no keyword in clip command");
		    return;
	    }
	}
    }

    if (server_mode == TRUE)
    {
	sprintf(server_str, "%d", clip_top);
	srv_real("clip_top", server_str);
	sprintf(server_str, "%d", clip_bottom);
	srv_real("clip_bottom", server_str);
	sprintf(server_str, "%d", clip_left);
	srv_real("clip_left", server_str);
	sprintf(server_str, "%d", clip_right);
	srv_real("clip_right", server_str);
    }
}

void
mode_set(argc, argv)
int argc;
char *argv[];
{
    char *mode_string;

    if (argc > 2)
    {
	error1("wrong number of arguments to mode command");
    }

    if (argc == 2)
    {
	switch (cmd(argv[1]))
	{
	    case LOG:
	    case LOGLOG:
	    case LIN:
	    case LORD:
	    case TRUELOG:
	    case EQ:
	    case HIST_SPEC:
		req_mode = cmd(argv[1]);
		break;
	    default:
		error1("unknown stretch mode");
		break;
	}
    }
    if (req_mode == LIN)
	mode_string = "linear";
    else if (req_mode == LOG)
	mode_string = "logarithmic";
    else if (req_mode == LOGLOG)
	mode_string = "loglog";
    else if (req_mode == LORD)
	mode_string = "lord";
    else if (req_mode == TRUELOG)
	mode_string = "truelog";
    else if (req_mode == EQ)
	mode_string = "equilization";
    else if (req_mode == HIST_SPEC)
	mode_string = "hist_spec (hs)";
    else
	mode_string = "undefined";

    if(server_mode == FALSE)
	printf("stretch mode is %s\n", mode_string);
    else
    {
	srv_string("mode", mode_string);
    }
    fprintf(session, "stretch mode is %s\n", mode_string);
}





/********************************************************************/
/*                                                                  */
/*  Begin Screen Management functions                               */
/*                                                                  */
/********************************************************************/

/********************************************************************/
/*                                                                  */
/*  CLEAR_IMG                                                       */
/*  Clears the entire screen management structure.                  */
/*                                                                  */
/********************************************************************/

void
clear_img()
{
    while (head_img.rlink != &head_img)
	delete_img(head_img.rlink);
    curr_img = NULL;

    clear_table_names();  /* no need to save tablefile names now */
}


/********************************************************************/
/*                                                                  */
/*  NEW_IMG                                                         */
/*  Place a new entry into the img structure.                       */
/*  Return a pointer to the new entry.                              */
/*  Copies values from global variables to the structure.           */
/*                                                                  */
/********************************************************************/

struct img *new_img()
{
    static struct img *tmpimg;
    struct img *x;

    tmpimg = (struct img *) malloc(sizeof(struct img));
    if (tmpimg == NULL)
    {
	if(server_mode == FALSE)
	    printf("skyview:  error allocating space for img structure\n");
	return(tmpimg);
    }
    /* insert to right of head_img */
    x = &head_img;
    tmpimg->llink = x;
    tmpimg->rlink = x->rlink;
    (tmpimg->rlink)->llink = tmpimg;
    x->rlink = tmpimg;

    if (debug)
	fprintf(debug_file, "new_img: tmpimg=0x%p\n", (void *) tmpimg);
    fill_img(tmpimg);
    curr_img = tmpimg;
    return(tmpimg);
}

/********************************************************************/
/*                                                                  */
/*  FIND_LAST_IMG                                                   */
/*  Finds the last entry in the img array.                          */
/*  Return a pointer to that entry.                                 */
/*  Copies values from the structure to global variables            */
/*  making this the current image.                                  */
/*                                                                  */
/*  If no image remains, returns NULL and makes no changes to       */
/*  current image or globals.                                       */
/*                                                                  */
/********************************************************************/

struct img *find_last_img()
{
    struct img *p;

    p = head_img.rlink;
    if (p == &head_img)
	return(NULL);
    if (debug)
	fprintf(debug_file, "find_last_img: p=0x%p\n", (void *) p);
    fill_glob(p);
    return(p);
}

/********************************************************************/
/*                                                                  */
/*  FIND_PREVIOUS_IMG                                               */
/*  Given a pointer to an img array entry, returns a pointer to     */
/*  the previous entry.                                             */
/*  Copies values from the structure to global variables            */
/*  making this the current image.                                  */
/*                                                                  */
/*  If no image remains,                           returns NULL and */
/*  makes no changes to current image or globals.                   */
/*                                                                  */
/********************************************************************/

struct img *find_previous_img(p)
struct img *p;
{
    p = p->rlink;
    if (p == &head_img)
	return(NULL);
    if (debug)
	fprintf(debug_file, "find_previous_img: p=0x%p\n", (void *) p);
    fill_glob(p);
    return(p);
}

/********************************************************************/
/*                                                                  */
/*  CHECK_IMG                                                       */
/*  Verifies that an image pointer points to a still valid img      */
/*                                                                  */
/*  Returns 1 if image pointer is valid.                            */
/*  Returns 0 if image pointer is bad.                              */
/*                                                                  */
/********************************************************************/

int
check_img(q)
struct img *q;
{
    struct img *p;

    p = head_img.rlink;
    while (p != &head_img)
    {
	if (p == q)
	    return(1);
	p = p->rlink;
    }
    return(0);
}

/********************************************************************/
/*                                                                  */
/*  DELETE_IMG                                                      */
/*  Deletes an entry from the img array.                            */
/*  Sets curr_img to NULL.                                          */
/*                                                                  */
/*  If no image exists at the address, does nothing.                */
/*                                                                  */
/********************************************************************/

void
delete_img(p)
struct img *p;
{
    (p->llink)->rlink = p->rlink;
    (p->rlink)->llink = p->llink;
    free(p);

    curr_img = NULL;

    if (debug)
	fprintf(debug_file, "delete_img: img = 0x%p\n", (void *) p);
    return;
}

/********************************************************************/
/*                                                                  */
/*  FIND_FILNAM                                                     */
/*  given a filename     returns a pointer to the img structure of  */
/*  the image painted from that filename.                           */
/*  Copies values from the structure to global variables            */
/*  making this the current image.                                  */
/*                                                                  */
/*  Favors current image                                            */
/*                                                                  */
/*  If no image found,    returns NULL and makes no changes to      */
/*  current image.                                                  */
/*                                                                  */
/********************************************************************/

struct img *find_FILNAM(fname)
char *fname;
{
    struct img *p;

    if (debug)
	fprintf(debug_file, "find_FILNAM: entry  fname =%s=\n", fname);

    if ((strcmp(fname, filnam) == 0) && (curr_img != NULL))
    {
	return(curr_img);
    }

    p = head_img.rlink;
    while (p != &head_img)
    {
	if (strcmp(fname, p->filnam) == 0)
	{
	    fill_glob(p);
	    return(p);
	}
	p = p->rlink;
    }
    return(NULL);
}

/********************************************************************/
/*                                                                  */
/*  PEEK_FILNAM                                                     */
/*  given a filename     returns a pointer to the img structure of  */
/*  the image painted from that filename.                           */
/*  Does NOT change the global variables                            */
/*                                                                  */
/*  Favors current image                                            */
/*                                                                  */
/*  If no image found,    returns NULL.                             */
/*                                                                  */
/********************************************************************/

struct img *peek_FILNAM(fname)
char *fname;
{
    struct img *p;

    if (debug)
	fprintf(debug_file, "peek_FILNAM: entry  fname =%s=\n", fname);

    if ((strcmp(fname, filnam) == 0) && (curr_img != NULL))
    {
	return(curr_img);
    }

    p = head_img.rlink;
    while (p != &head_img)
    {
	if (strcmp(fname, p->filnam) == 0)
	{
	    return(p);
	}
	p = p->rlink;
    }
    return(NULL);
}

/********************************************************************/
/*                                                                  */
/*  FIND_SC                                                         */
/*  given a screen (x,y) returns a pointer to the img structure of  */
/*  the image containing (x,y).                                     */
/*  Copies values from the structure to global variables            */
/*  making this the current image.                                  */
/*                                                                  */
/*  Favors current image (for X windows)                            */
/*                                                                  */
/*  If no image at (x,y), returns NULL and makes no changes to      */
/*  current image.                                                  */
/*                                                                  */
/********************************************************************/

struct img *find_SC(x,y)
int x,y;
{
    struct img *p;

    if (debug)
	fprintf(debug_file, "find_SC: entry  x = %d  y = %d\n", x, y);

    if ((minx <= x) && (maxx >= x) &&
	(miny <= y) && (maxy >= y) && (curr_img != NULL))
    {
	return(curr_img);
    }

    p = head_img.rlink;
    while (p != &head_img)
    {
	if ((p->minx <= x) && (p->maxx >= x) &&
	    (p->miny <= y) && (p->maxy >= y))
	{
	    fill_glob(p);
	    return(p);
	}
	p = p->rlink;
    }
    return(NULL);

}


/********************************************************************/
/*                                                                  */
/*  PEEK_FRAME                                                      */
/*  given a frame number returns a pointer to the img structure of  */
/*  the last image painted into that frame.                         */
/*  Does NOT copy stuff from the structure to the global variables  */
/*                                                                  */
/*  If no such image exists, returns NULL                           */
/*                                                                  */
/********************************************************************/

struct img *peek_FRAME(frame)
int frame;
{
    struct img *p;

    p = head_img.rlink;
    while (p != &head_img)
    {
	if (p->image_frame == frame)
	{
	    if (debug)
		fprintf(debug_file, 
		    "peek_FRAME: frame = %d  (found it) imgptr = 0x%p\n", 
		    frame, (void *) p);
	    return(p);
	}
	p = p->rlink;
    }
    if (debug)
	fprintf(debug_file, 
	"peek_FRAME: frame = %d  NOT FOUND\n", frame);
    return(NULL);
}

/********************************************************************/
/*                                                                  */
/*  MAX_FRAME                                                       */
/*  return the maximum frame number in use                          */
/*                                                                  */
/********************************************************************/

int max_FRAME()
{
    struct img *p;
    int max_frame;

    max_frame = 0;
    p = head_img.rlink;
    while (p != &head_img)
    {
	if (p->image_frame > max_frame)
	{
	    max_frame = p->image_frame;
	}
	p = p->rlink;
    }
    return(max_frame);
}

/********************************************************************/
/*                                                                  */
/*  FIND_IM                                                         */
/*                                                                  */
/*  Finds image with samp,line visible.                             */
/*  Favors current image.                                           */
/*                                                                  */
/*  On entry:                                                       */
/*     i_coord[0] = samp                                            */
/*     i_coord[1] = line                                            */
/*                                                                  */
/*  On exit:                                                        */
/*     Returns image pointer.                                       */
/*     Copies values from the structure to global variables         */
/*     making this the current image.                               */
/*                                                                  */
/*     If no image at (x,y), returns NULL and makes no changes to   */
/*     current image.                                               */
/*                                                                  */
/*     Leaves i_coord unchanged.                                    */
/*                                                                  */
/********************************************************************/

struct img *find_IM(i_coord)
int i_coord[2];
{
    struct img *p, *save_imgp, *imgp;
    int x, y;
    double ti_coord[2];

    save_imgp = curr_img;

    ti_coord[0] = i_coord[0];
    ti_coord[1] = i_coord[1];
    dIM_to_SC(ti_coord);
    x = Round(ti_coord[0]);
    y = Round(ti_coord[1]);
    if ((minx <= x) && (maxx >= x) &&
	(miny <= y) && (maxy >= y))
	{
	    /* its on the current image, but check if visible */
	    imgp = find_SC(x,y);
	    if (imgp == save_imgp)
		return(imgp);
	}


    p = head_img.rlink;
    while (p != &head_img)
    {

	fill_glob(p);
	ti_coord[0] = i_coord[0];
	ti_coord[1] = i_coord[1];
	dIM_to_SC(ti_coord);
	x = Round(ti_coord[0]);
	y = Round(ti_coord[1]);
	if ((minx <= x) && (maxx >= x) &&
	    (miny <= y) && (maxy >= y))
	    {
	    /* its on the current image, but check if visible */
	    imgp = find_SC(x,y);
	    if (imgp == p)
		return(p);
	    }
	p = p->rlink;
    }
    /* it is not on any image */
    if (save_imgp != NULL)
	fill_glob(save_imgp);  /* refill global with current image stuff */
    return(NULL);
}

/********************************************************************/
/*                                                                  */
/*  FIND_IRAF                                                       */
/*                                                                  */
/*  Finds image with IRAF pixel,line visible.                       */
/*  Favors current image.                                           */
/*                                                                  */
/*  On entry:                                                       */
/*     d_coord[0] = IRAF pixel (double precision)                   */
/*     d_coord[1] = IRAF line (double precision)                    */
/*                                                                  */
/*  On exit:                                                        */
/*     Returns image pointer.                                       */
/*     Copies values from the structure to global variables         */
/*     making this the current image.                               */
/*                                                                  */
/*     If no image at (x,y), returns NULL and makes no changes to   */
/*     current image.                                               */
/*                                                                  */
/*     Leaves d_coord unchanged.                                    */
/*                                                                  */
/********************************************************************/

struct img *find_IRAF(d_coord)
double d_coord[2];
{
    struct img *p, *save_imgp, *imgp;
    int x, y;
    double td_coord[2];

    save_imgp = curr_img;


    td_coord[0] = d_coord[0];
    td_coord[1] = d_coord[1];
    dIRAF_to_SC(td_coord);
    x = Round(td_coord[0]);
    y = Round(td_coord[1]);
    if ((minx <= x) && (maxx >= x) &&
	(miny <= y) && (maxy >= y))
	{
	    /* its on the current image, but check if visible */
	    imgp = find_SC(x,y);
	    if (imgp == save_imgp)
		return(imgp);
	}


    p = head_img.rlink;
    while (p != &head_img)
    {

	fill_glob(p);
	td_coord[0] = d_coord[0];
	td_coord[1] = d_coord[1];
	dIRAF_to_SC(td_coord);
	x = Round(td_coord[0]);
	y = Round(td_coord[1]);
	if ((minx <= x) && (maxx >= x) &&
	    (miny <= y) && (maxy >= y))
	    {
	    /* its on the current image, but check if visible */
	    imgp = find_SC(x,y);
	    if (imgp == p)
		return(p);
	    }
	p = p->rlink;
    }
    /* it is not on any image */
    if (save_imgp != NULL)
	fill_glob(save_imgp);  /* refill global with current image stuff */
    return(NULL);
}

/********************************************************************/
/*                                                                  */
/*  FIND_IMIO                                                       */
/*                                                                  */
/*  Finds image with IRAF IMIO config and frame.                    */
/*                                                                  */
/*  On entry:                                                       */
/*     d_coord[0] = IRAF pixel (double precision)                   */
/*     d_coord[1] = IRAF line (double precision)                    */
/*                                                                  */
/*  On exit:                                                        */
/*     Returns image pointer.                                       */
/*     Copies values from the structure to global variables         */
/*     making this the current image.                               */
/*                                                                  */
/*     If no image at (x,y), returns NULL and makes no changes to   */
/*     current image.                                               */
/*                                                                  */
/*     Leaves d_coord unchanged.                                    */
/*                                                                  */
/********************************************************************/

struct img *find_IMIO(config, desired_frame)
int config, desired_frame;
{
    struct img *p;

    if ((config == imio_config) && (desired_frame == imio_frame) &&
	(curr_img != NULL))
	return(curr_img);  /* it's on the current image */


    p = head_img.rlink;
    while (p != &head_img)
    {
	if ((config == p->imio_config) && (desired_frame == p->imio_frame))
	{
	    fill_glob(p);
	    return(p);
	}
	p = p->rlink;
    }
    /* it is not on any image */
    return(NULL);
}

/********************************************************************/
/*                                                                  */
/*  FIND_COORD                                                      */
/*                                                                  */
/*  Finds image with lat,lon visible.                               */
/*  Favors current image.                                           */
/*                                                                  */
/*  On entry:                                                       */
/*     coord[0] = lon                                               */
/*     coord[1] = lat                                               */
/*                                                                  */
/*  On exit:                                                        */
/*     Returns image pointer.                                       */
/*     d_coord[0] = samp (double precision)                         */
/*     d_coord[1] = line (double precision)                         */
/*                                                                  */
/*     Copies values from the structure to global variables         */
/*     making this the current image.                               */
/*                                                                  */
/*     If no image at (lat,lon), returns NULL and makes no          */
/*     changes to current image.  In this case, the returned        */
/*     line and samp are the result of deprojection relative to     */
/*     the current image (off of the edge of the image).            */
/*                                                                  */
/********************************************************************/

struct img *find_COORD(coord, d_coord)
double coord[2];
double d_coord[2];
{
    struct img *p, *save_imgp;

    save_imgp = curr_img;

    /* see if it's on the current image */
    if (check_COORD(coord, d_coord))
	return(curr_img);

    /* see if it's on any other image */
    p = head_img.rlink;
    while (p != &head_img)
    {
	fill_glob(p);
	if (check_COORD(coord, d_coord))
	    return(p);
	p = p->rlink;
    }
    if (save_imgp != NULL)
	fill_glob(save_imgp);  /* refill global with current image stuff */
    /* return line and samp relative to current image */
    deproject(coord, d_coord);
    return(NULL);
}

static int 
check_COORD(coord, d_coord)
double coord[2];
double d_coord[2];
{
    struct img *imgp;
    double td_coord[2];
    int x, y;

    deproject(coord, d_coord);
    td_coord[0] = d_coord[0];
    td_coord[1] = d_coord[1];
    dIM_to_SC(td_coord);
    x = Round(td_coord[0]);
    y = Round(td_coord[1]);
    if ((minx <= x) && (maxx >= x) &&
	(miny <= y) && (maxy >= y))
	{
	    /* its on this image, but check if visible */
	    imgp = find_SC(x,y);
	    if (imgp == curr_img)
		return(1);
	}
    /* try the other way around the globe */
    if (coord[0] >= 0)
    {
	td_coord[0] = coord[0] - 360.0;
	td_coord[1] = coord[1];
    }
    else
    {
	td_coord[0] = coord[0] + 360.0;
	td_coord[1] = coord[1];
    }
    deproject(td_coord, d_coord);
    td_coord[0] = d_coord[0];
    td_coord[1] = d_coord[1];
    dIM_to_SC(td_coord);
    x = Round(td_coord[0]);
    y = Round(td_coord[1]);
    if ((minx <= x) && (maxx >= x) &&
	(miny <= y) && (maxy >= y))
	{
	    /* its on this image, but check if visible */
	    imgp = find_SC(x,y);
	    if (imgp == curr_img)
		return(1);
	}

    return(0);
}



/********************************************************************/
/*                                                                  */
/*  FILL_GLOB                                                       */
/*  Fills global variables with values from an image structure.     */
/*  Sets curr_img                                                   */
/*                                                                  */
/********************************************************************/

void
fill_glob(imgp)
struct img *imgp;
{
    int i, j;

    if (debug)
	fprintf(debug_file, 
"fill_glob:  imgptr = 0x%p     prior image_frame = %d  new image_frame = %d\n",
	    (void *) imgp, image_frame, imgp?imgp->image_frame:image_frame);
    if (imgp != NULL)
    {
	strcpy(filnam, imgp->filnam);
	strcpy(mod_filnam, imgp->mod_filnam);
	bitpix = imgp->bitpix;
	u16 = imgp->u16;
	blank= imgp->blank;
	blank_set= imgp->blank_set;
	report_blank= imgp->report_blank;
	crval3 = imgp->crval3;
	pixels= imgp->pixels;
	lines= imgp->lines;
	naxis= imgp->naxis;
	naxis3= imgp->naxis3;
	cdelt1= imgp->cdelt1;
	cdelt2= imgp->cdelt2;
	using_cd = imgp->using_cd;
	cd1_1 = imgp->cd1_1;
	cd1_2 = imgp->cd1_2;
	cd2_1 = imgp->cd2_1;
	cd2_2 = imgp->cd2_2;
	dc1_1 = imgp->dc1_1;
	dc1_2 = imgp->dc1_2;
	dc2_1 = imgp->dc2_1;
	dc2_2 = imgp->dc2_2;
	twist= imgp->twist;
	crpix1= imgp->crpix1;
	crpix2= imgp->crpix2;
	bscale= imgp->bscale;
	b_zero= imgp->b_zero;
	glat= imgp->glat;
	glong= imgp->glong;
	strcpy(bunit, imgp->bunit);
	strcpy(ctype1, imgp->ctype1);
	strcpy(ctype2, imgp->ctype2);
	strcpy(ctype1sav, imgp->ctype1sav);
	strcpy(ctype2sav, imgp->ctype2sav);
	strcpy(object, imgp->object);
	maptype = imgp->maptype;
	map_distortion = imgp->map_distortion;
	a_order = imgp->a_order;
	ap_order = imgp->ap_order;
	b_order = imgp->b_order;
	bp_order = imgp->bp_order;
	for (i = 0; i < 5; i++)
	    for (j = 0; j < 5; j++)
	    {
		a[i][j] = imgp->a[i][j];
		ap[i][j] = imgp->ap[i][j];
		b[i][j] = imgp->b[i][j];
		bp[i][j] = imgp->bp[i][j];
	    }
	base= imgp->base;
	noise= imgp->noise;
	dskygrid= imgp->dskygrid;
	x_offset = imgp->x_offset;
	minx= imgp->minx;
	maxx= imgp->maxx;
	miny= imgp->miny;
	maxy= imgp->maxy;
	sgn_y = imgp->sgn_y;
	y_offset = imgp->y_offset;
	filehdr.tapetype = imgp->filehdr.tapetype;
	filehdr.hdr_start= imgp->filehdr.hdr_start;
	filehdr.img_start= imgp->filehdr.img_start;
	filehdr.hist_start = imgp->filehdr.hist_start;
	filehdr.img_max = imgp->filehdr.img_max;
	filehdr.img_min = imgp->filehdr.img_min;
	filehdr.hist_bin_size = imgp->filehdr.hist_bin_size;
	file_coord_sys = imgp->file_coord_sys;
	iraf_max = imgp->iraf_max;
	iraf_min = imgp->iraf_min;
	slow = imgp->slow;
	shigh = imgp->shigh;
	mode_used = imgp->mode_used;
	samp_disp = imgp->samp_disp;
	line_disp = imgp->line_disp;
	graphics_window = imgp->graphics_window;
	file_equinox = imgp->file_equinox;
	strcpy(radecsys, imgp->radecsys);
	imio_config = imgp->imio_config;
	imio_frame = imgp->imio_frame;
	imio_ok = imgp->imio_ok;
	imio_filptr = imgp->imio_filptr;
	band_offset = imgp->band_offset;
	image_frame = imgp->image_frame;
	frameptr = imgp->frameptr;

	plate_ra = imgp->plate_ra;
	plate_dec = imgp->plate_dec;
	x_pixel_offset = imgp->x_pixel_offset;
	y_pixel_offset = imgp->y_pixel_offset;
	x_pixel_size = imgp->x_pixel_size;
	y_pixel_size = imgp->y_pixel_size;
	for (i = 0; i < 6; i++)
	    ppo_coeff[i] = imgp->ppo_coeff[i];
	for (i = 0; i < 20; i++)
	    amd_x_coeff[i] = imgp->amd_x_coeff[i];
	for (i = 0; i < 20; i++)
	    amd_y_coeff[i] = imgp->amd_y_coeff[i];
	plt_scale = imgp->plt_scale;

	fill_glob_frame(imgp);
	strncpy(wcsbuf, imgp->wcsbuf, SZ_WCSBUF);
    }

    set_file_sys();
    curr_img = imgp;
}

/********************************************************************/
/*                                                                  */
/*  FILL_IMG                                                        */
/*  Fills image structure with values from global variables         */
/*                                                                  */
/********************************************************************/

void
fill_img(imgp)
struct img *imgp;
{
    int i, j;

    if (debug)
	fprintf(debug_file, "fill_img:  imgptr = 0x%p  image_frame = %d\n",
	    (void *) imgp, image_frame);
    if (imgp != NULL)
    {
	strcpy(imgp->filnam, filnam );
	strcpy(imgp->mod_filnam, mod_filnam );
	imgp->bitpix = bitpix ;
	imgp->u16 = u16;
	imgp->blank = blank;
	imgp->blank_set = blank_set;
	imgp->report_blank = report_blank;
	imgp->crval3 = crval3;
	imgp->pixels = pixels;
	imgp->lines = lines;
	imgp->naxis = naxis;
	imgp->naxis3 = naxis3;
	imgp->cdelt1 = cdelt1;
	imgp->cdelt2 = cdelt2;
	imgp->using_cd = using_cd;
	imgp->cd1_1 = cd1_1;
	imgp->cd1_2 = cd1_2;
	imgp->cd2_1 = cd2_1;
	imgp->cd2_2 = cd2_2;
	imgp->dc1_1 = dc1_1;
	imgp->dc1_2 = dc1_2;
	imgp->dc2_1 = dc2_1;
	imgp->dc2_2 = dc2_2;
	imgp->twist = twist;
	imgp->crpix1 = crpix1;
	imgp->crpix2 = crpix2;
	imgp->bscale = bscale;
	imgp->b_zero = b_zero;
	imgp->glat = glat;
	imgp->glong = glong;
	strcpy(imgp->bunit,bunit );
	strcpy(imgp->ctype1,ctype1);
	strcpy(imgp->ctype2, ctype2);
	strcpy(imgp->ctype1sav, ctype1sav);
	strcpy(imgp->ctype2sav, ctype2sav);
	strcpy(imgp->object, object);
	imgp->maptype = maptype;
	imgp->map_distortion = map_distortion;
	imgp->a_order = a_order;
	imgp->ap_order = ap_order;
	imgp->b_order = b_order;
	imgp->bp_order = bp_order;
	for (i = 0; i < 5; i++)
	    for (j = 0; j < 5; j++)
	    {
		imgp->a[i][j] = a[i][j];
		imgp->ap[i][j] = ap[i][j];
		imgp->b[i][j] = b[i][j];
		imgp->bp[i][j] = bp[i][j];
	    }
	imgp->base = base;
	imgp->noise = noise;
	imgp->dskygrid = dskygrid;
	imgp->x_offset = x_offset;
	imgp->minx = minx;
	imgp->maxx = maxx;
	imgp->miny = miny;
	imgp->maxy = maxy;
	imgp->sgn_y = sgn_y ;
	imgp->y_offset = y_offset ;
	imgp->filehdr.tapetype = filehdr.tapetype ;
	imgp->filehdr.hdr_start = filehdr.hdr_start ;
	imgp->filehdr.img_start = filehdr.img_start ;
	imgp->filehdr.hist_start = filehdr.hist_start ;
	imgp->filehdr.img_max = filehdr.img_max ;
	imgp->filehdr.img_min = filehdr.img_min ;
	imgp->filehdr.hist_bin_size = filehdr.hist_bin_size ;
	imgp->file_coord_sys = file_coord_sys;
	imgp->iraf_max = iraf_max;
	imgp->iraf_min = iraf_min;
	imgp->band_offset = band_offset;
	imgp->slow = slow;
	imgp->shigh = shigh;
	imgp->mode_used = mode_used;
	imgp->samp_disp = samp_disp;
	imgp->line_disp = line_disp;
	imgp->graphics_window = graphics_window;
	imgp->file_equinox = file_equinox;
	strcpy(imgp->radecsys, radecsys);
	imgp->imio_config = imio_config;
	imgp->imio_frame = imio_frame;
	imgp->imio_ok = imio_ok;
	imgp->imio_filptr = imio_filptr;
	imgp->image_frame = image_frame;
	imgp->frameptr = frameptr;

	imgp->plate_ra = plate_ra;
	imgp->plate_dec = plate_dec;
	imgp->x_pixel_offset = x_pixel_offset;
	imgp->y_pixel_offset = y_pixel_offset;
	imgp->x_pixel_size = x_pixel_size;
	imgp->y_pixel_size = y_pixel_size;
	for (i = 0; i < 6; i++)
	    imgp->ppo_coeff[i] = ppo_coeff[i];
	for (i = 0; i < 20; i++)
	    imgp->amd_x_coeff[i] = amd_x_coeff[i];
	for (i = 0; i < 20; i++)
	    imgp->amd_y_coeff[i] = amd_y_coeff[i];
	imgp->plt_scale = plt_scale;

	fill_img_frame(imgp);
	strncpy(imgp->wcsbuf, wcsbuf, SZ_WCSBUF );
    }
}
