/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* range.c */

/* syntax for range command;                              */
/*   range [arg_op_arg arg_op_arg]                        */
/*     where op is + or -                                 */
/*     and arg is min, max, lastmin, lastmax, lastdiff,   */
/*       x%, or x (an absolute value)                     */

#include "skyview.h"
#include "parse.h"
#include "img.h"
#include "range.h"
#include "im_proto.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

extern FILE *session, *debug_file;
extern struct hdrstruct filehdr;
extern double iraf_min, iraf_max;
extern int server_mode;
extern char fi_name[];
extern struct img *curr_img;
extern char server_str[];
extern int debug;
extern double slow, shigh;
extern int llow_save, hhigh_save;

struct full_ra min_ra = { RA_SIMPLE, {RA_PCT, 1} , {0,0} } ;
struct full_ra max_ra = { RA_SIMPLE, {RA_PCT, 99} , {0,0} } ;
double lastmin, lastmax;

#ifdef _NO_PROTO
static char *get_arg();
static char *get_operator();
static char *one_range_report();
static char *range_operator_report();
static char *range_token_report();
static double range_token_value();
static void range_report();
#else
/* prototypes */
static char *get_arg(char *input_string, struct simple_ra *ra_token);
static char *get_operator(char *input_string, int *operator);
static char *one_range_report( struct full_ra *ra_struct);
static char *range_operator_report( int operator);
static char *range_token_report(struct simple_ra *ra_token);
static double range_token_value(struct simple_ra *ra_token);
static void range_report(void);
#endif /* _NO_PROTO */


void
range(argc, argv)
int argc;
char *argv[];
{

    if (argc == 3)
    {
	/* set ra values */
	if (!parse_minmax(argv[1], &min_ra, PARSE_MIN))
	{
	    error1("bad first argument to RA command");
	    return;
	}

	if (!parse_minmax(argv[2], &max_ra, PARSE_MAX))
	{
	    error1("bad second argument to RA command");
	    return;
	}

	if(server_mode == TRUE)
	    range_report();
	return;
    }

    if (argc == 2)
    {
	if (strncmp(argv[1], "st", 2) == 0)
	{
	    /* "ra st" command */
	    min_ra.tag = RA_SIMPLE;
	    min_ra.arg_one.tag = RA_ABS;
	    min_ra.arg_one.value = 
		(dn_to_jy(shigh) - dn_to_jy(slow)) * llow_save / 255 + 
		dn_to_jy(slow);
	    max_ra.tag = RA_SIMPLE;
	    max_ra.arg_one.tag = RA_ABS;
	    max_ra.arg_one.value = 
		(dn_to_jy(shigh) - dn_to_jy(slow)) * hhigh_save / 255 + 
		dn_to_jy(slow);


	    if(server_mode == TRUE)
		range_report();
	    return;
	}
    }

    if (argc == 1)
    {
	/* report only */
	range_report();
	return;
    }
    error1("wrong number of arguments to RA command");
}



/* get_arg parses for one range token */
/*     returns a pointer to the character which stopped the scan */
/*     Failure is indicated by return value == input_string */
static
char *get_arg(input_string, ra_token)
char *input_string;
struct simple_ra *ra_token;
{
    char *ptr;

    if (strncmp(input_string, "min", 3) == 0)
    {
	ra_token->tag = RA_MIN;
	input_string += 3;
    }
    else if (strncmp(input_string, "max", 3) == 0)
    {
	ra_token->tag = RA_MAX;
	input_string += 3;
    }
    else if (strncmp(input_string, "lastmin", 7) == 0)
    {
	ra_token->tag = RA_LASTMIN;
	input_string += 7;
    }
    else if (strncmp(input_string, "lastmax", 7) == 0)
    {
	ra_token->tag = RA_LASTMAX;
	input_string += 7;
    }
    else if (strncmp(input_string, "lastdiff", 8) == 0)
    {
	ra_token->tag = RA_LASTDIFF;
	input_string += 8;
    }
    else
    {
	ra_token->value = strtod(input_string, &ptr);
	if (ptr != input_string)
	{
	    /* got a good number */
	    input_string = ptr;
	    if (*input_string == '%')
	    {
		ra_token->tag = RA_PCT;
		input_string++;
	    }
	    else if (*input_string == 's')
	    {
		ra_token->tag = RA_SIGMA;
		input_string++;
	    }
	    else
	    {
		ra_token->tag = RA_ABS;
	    }
	}

    }
    return(input_string);
}

static
char *get_operator(input_string, operator)
char *input_string;
int *operator;
{
    if (*input_string == '+')
    {
	*operator = RA_PLUS;
	input_string++;
    }
    else if (*input_string == '-')
    {
	*operator = RA_MINUS;
	input_string++;
    }
    return(input_string);
}

int parse_minmax(input_string, ra_struct, which)
char *input_string;
struct full_ra *ra_struct;
int which;    /* which value are we parsing, min or max */
{
    char *string_remaining, *current_string;

    /* first take care of undocumented abbreviation 'm' */
    if ((input_string[0] == 'm') && (input_string[1] == '\0'))
    {
	if (which == PARSE_MIN)
	    ra_struct->arg_one.tag = RA_MIN;
	else
	    ra_struct->arg_one.tag = RA_MAX;
	ra_struct->tag = RA_SIMPLE;
	return(1);
    }
    current_string = input_string;
    string_remaining = get_arg(current_string, &ra_struct->arg_one);
    if (string_remaining == current_string)
	return(0);
    if (*string_remaining == '\0')
    {
	/* it was simple token, not an expression */
	ra_struct->tag = RA_SIMPLE;
	return(1);
    }
    current_string = string_remaining;
    string_remaining = get_operator(current_string, &ra_struct->tag);
    if (string_remaining == current_string)
	return(0);
    current_string = string_remaining;
    string_remaining = get_arg(current_string, &ra_struct->arg_two);
    if (string_remaining == current_string)
	return(0);

    return(1);
}



static
void range_report()
{
    char *report;
    int fd;
    struct img tmpimg, *save_img;
    double s_low, s_high;

    if (!server_mode)
	printf("stretching from ");
    fprintf(session, "stretching from ");

    report = one_range_report(&min_ra);
    if (server_mode)
	srv_string("range_min", report);
    else
	printf("%s", report);
    fprintf(session, "%s", report);

    if (!server_mode)
	printf(" to ");
    fprintf(session, " to ");

    report = one_range_report(&max_ra);
    if (server_mode)
	srv_string("range_max", report);
    else
	printf("%s", report);
    fprintf(session, "%s", report);

    if (!server_mode)
	printf("\n");
    fprintf(session, "\n");

    if ((min_ra.tag == RA_SIMPLE) && (min_ra.arg_one.tag == RA_ABS) &&
	(max_ra.tag == RA_SIMPLE) && (max_ra.arg_one.tag == RA_ABS))
	return;  /* no values to fetch from file */

    if (fi_name[0] == '\0')
	return;
    fd = wf_open(fi_name, &tmpimg, 0);
    if (fd < 0)
	return;

    save_img = curr_img;
    fill_glob(&tmpimg);
    curr_img = NULL;
    set_file_sys();

    check_hist(fd);
    s_low = get_minmax(&min_ra);
    s_high = get_minmax(&max_ra);

    if(!server_mode)
       printf("    (from %e to %e)\n", dn_to_jy(s_low), dn_to_jy(s_high));
    else
    {
       sprintf(server_str, "%e", dn_to_jy(s_low));
       srv_real("range_min_val", server_str);
       sprintf(server_str, "%e", dn_to_jy(s_high));
       srv_real("range_max_val", server_str);
    }
    fprintf(session, "    (from %e to %e)\n", dn_to_jy(s_low), dn_to_jy(s_high));

    if (debug)
	fprintf(debug_file, "low = %e DN   high = %e DN\n", s_low, s_high);

    if (filehdr.tapetype == IMFILE)
	im_close(fd);
    else
	close (fd);
    fill_glob(save_img);
}

static
char *one_range_report(ra_struct)
struct full_ra *ra_struct;
{
    static char report[40];

    strcpy(report, range_token_report(&ra_struct->arg_one));
    if (ra_struct->tag == RA_SIMPLE)
	return(report);
    strcat(report, range_operator_report(ra_struct->tag));
    strcat(report, range_token_report(&ra_struct->arg_two));
    return(report);
}

static
char *range_operator_report(operator)
int operator;
{
    if (operator == RA_PLUS)
	return("+");
    else if (operator == RA_MINUS)
	return("-");
    else
	return("UNKNOWN_OP");
}

static
char *range_token_report(ra_token)
struct simple_ra *ra_token;
{
    static char report[30];
    switch(ra_token->tag)
    {
	case RA_MIN:
	    strcpy(report, "MIN");
	    break;
	case RA_MAX:
	    strcpy(report, "MAX");
	    break;
	case RA_LASTMIN:
	    strcpy(report, "LASTMIN");
	    break;
	case RA_LASTMAX:
	    strcpy(report,"LASTMAX");
	    break;
	case RA_LASTDIFF:
	    strcpy(report, "LASTDIFF");
	    break;
	case RA_ABS:
	    sprintf(report, "%e", ra_token->value);
	    break;
	case RA_PCT:
	    sprintf(report, "%4.1f%%", ra_token->value);
	    break;
	case RA_SIGMA:
	    sprintf(report, "%4.1fs", ra_token->value);
	    break;
    }
    return(report);
}

double get_minmax(ra_struct)
struct full_ra *ra_struct;
{
    double retval;

    retval = range_token_value(&ra_struct->arg_one);
    switch (ra_struct->tag)
    {
	case RA_SIMPLE:
	    break;
	case RA_PLUS:
	    retval += range_token_value(&ra_struct->arg_two);
	    break;
	case RA_MINUS:
	    retval -= range_token_value(&ra_struct->arg_two);
	    break;
    }
    return(retval);
}

static
double range_token_value(ra_token)
struct simple_ra *ra_token;
{
    double retval;
    double a, b, median, sigma;

    switch(ra_token->tag)
    {
	case RA_MIN:
	    retval = iraf_min;
	    break;
	case RA_MAX:
	    retval = iraf_max;
	    break;
	case RA_LASTMIN:
	    retval = jy_to_dn(lastmin);
	    break;
	case RA_LASTMAX:
	    retval = jy_to_dn(lastmax);
	    break;
	case RA_LASTDIFF:
	    retval = jy_to_dn(lastmax) - jy_to_dn(lastmin);
	    break;
	case RA_PCT:
	    retval = get_pct(ra_token->value);
	    break;
	case RA_SIGMA:
	    a = get_pct(16);
	    median = get_pct(50);
	    b = get_pct(84);
	    sigma = (b - a) / 2.0;
	    retval = median + (ra_token->value) * sigma;
	    break;
	case RA_ABS:
	    retval = jy_to_dn(ra_token->value);
	    break;
    }
    return(retval);
}

/* return TRUE if we need to check_hist() for this range */
int need_hist()
{
    if ((min_ra.arg_one.tag == RA_MIN) || (min_ra.arg_one.tag == RA_MAX) ||
	(min_ra.arg_one.tag == RA_PCT) || (min_ra.arg_one.tag == RA_SIGMA))
	return(1);
    if (min_ra.tag != RA_SIMPLE) 
	if ((min_ra.arg_two.tag == RA_MIN) || (min_ra.arg_two.tag == RA_MAX) ||
	    (min_ra.arg_two.tag == RA_PCT) || (min_ra.arg_one.tag == RA_SIGMA))
	    return(1);
    if ((max_ra.arg_one.tag == RA_MIN) || (max_ra.arg_one.tag == RA_MAX) ||
	(max_ra.arg_one.tag == RA_PCT) || (min_ra.arg_one.tag == RA_SIGMA))
	return(1);
    if (max_ra.tag != RA_SIMPLE) 
	if ((max_ra.arg_two.tag == RA_MIN) || (max_ra.arg_two.tag == RA_MAX) ||
	    (max_ra.arg_two.tag == RA_PCT) || (min_ra.arg_one.tag == RA_SIGMA))
	    return(1);

    return(0);
}


#ifdef TEST_DRIVER

main(argc, argv)
int argc;
char **argv;
{

    range_report();

#ifdef NOTDEF
    min_ra.tag = 1;
    min_ra.arg_one.tag = 2;
    min_ra.arg_one.value = 3.0;
    min_ra.arg_two.tag = 4;
    min_ra.arg_two.value = 5.0;
    range_report();
#endif /* NOTDEF */

    if (!parse_minmax(argv[1], &min_ra, PARSE_MIN))
    {
	printf("error in range minimum value\n");
    }
    if (!parse_minmax(argv[2], &max_ra, PARSE_MAX))
    {
	printf("error in range maximum value\n");
    }
    range_report();
}
#endif /* TEST_DRIVER */
