/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/********************************************************************/
/*                                                                  */
/*  FIND_PLATE                                                      */
/*  allows the user to give a coordinate and returns the tape and   */
/*  file number for the SKYFLUX Fits tape to be mounted.            */
/*                                                                  */
/*                                                                  */
/*  Fri Mar 29 16:40:40 PST 1991   R.Ebert                          */
/*  Modification to support SSF (Super-Skyflux plate scheme)        */
/*  Now, Two calling modes:                                         */
/*     fp  lon lat                                                  */
/*     fp  ssf lon lat [band [HCON]]                                */
/*  Mon Apr  1 14:46:35 PST 1991   R.Ebert                          */
/*     fork off external job to determine the answer                */
/*                                                                  */
/*                                                                  */
/********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "skyview.h"
#include "img.h"
#include "job.h"

#define SC 0
#define IM 1
#define EQ 2
#define EC 3
#define GA 4

extern int server_mode, debug;
extern char server_str[];
extern FILE *session, *debug_file;
extern int input_jsys;
extern double input_equinox;

static char fp_errmesg[] = "Not the right number of arguments for FP command.";

#define  SSFPGM  "issaffld"    /* program generating ssf plate info */
#define  SKYPGM  "skyplate"    /* program generating Skyflux plate info */

#define  READBUFLEN 81

void
find_plate(argc,argv)

int argc;
char **argv;

{ 
     int band = 1;   /* default ssf band */
     int hcon = 0;   /* default ssf hcon */
     int stat;
     int ssf;   /* == 1 if ssf is asked for; == 0 for skyflux */

     char s_coord[2][40]; 
     double coord[2]; 

     char **pargs;
     int pargc;


     char *cmdv[7];    /*  ssf_plate  lat lon  band hcon \0 */
     char cra[20], cdec[20], cband[2], chcon[2];

     FILE *jout, *jin;
     struct job_info *job;
     char readbuf[READBUFLEN];

     if(debug)
     {
	  fprintf(debug_file,"find_plate: Got %d args\n", argc);
	  for (stat = 0; stat<argc; stat++)
	     fprintf(debug_file,
		 "find_plate: Argv[%d] \"%s\"\n", stat, argv[stat]);
     }

    if(server_mode == TRUE)
    {
       error_na("find_plate command not allowed in server mode");
       return;
    }
	  

     if(argc < 3 || argc > 6)
     {
	  error1(fp_errmesg);
	  return;
     }

     pargs = argv+1;
     pargc = argc-1;

     ssf = 1;
     if ((strcmp(*pargs,"ssf")==0) || (strncmp(*pargs,"issa", 4)==0))
     {
	ssf = 1;
	pargs++, pargc--;
     }
     else if (strncmp(*pargs,"sky", 3)==0)
     {
	ssf = 0;
	pargs++, pargc--;
     }

       if(pargc != 2)   /*if not 2 args left */
       {
	    error1(fp_errmesg);
	    return;
       }
     
     strcpy(s_coord[0], *pargs++); pargc--;
     strcpy(s_coord[1], *pargs++); pargc--;
     if(debug)
     {
	  fprintf(debug_file,"find_plate: Got lon %s\n", s_coord[0]);
	  fprintf(debug_file,"find_plate: Got lat %s\n", s_coord[1]);
     }

     if (pargc>0)
     {
         band = atoi(*pargs++);
	 pargc--;
         if(band<1 || band > 4)
         {
	    error1("Invalid IRAS Band");
	    return;
         }

         if (pargc>0)
         {
	     hcon = atoi(*pargs++);
	     pargc--;
             if(hcon<0 || hcon > 3)
             {
		 error1("Invalid IRAS HCON");
                 return;
             }
         }
     }

     stat = str_to_coord(s_coord, coord);
     if  (stat == FALSE)
     {
	error1("Invalid Coordinates");
	return;
     }

     /* Convert from the selected coordinate system to equatorial B1950 */

     jcnvc2(input_jsys, input_equinox, coord[0], coord[1],
	1, 1950.0, &coord[0], &coord[1], 0.0);

     if (debug)
     {
	 fprintf(debug_file,"find_plate:  RA = %f, Dec = %f (B1950)\n",
				    coord[0], coord[1]);
	 fprintf(debug_file,"             on %s Plates ", ssf==1?"SSF":"SkyFlux");
	 if(ssf == 1) 
	    fprintf(debug_file,"HCON %d, Band %d\n", hcon, band);
	 else         
	    fprintf(debug_file,"\n");
     }

     if ((fabs(coord[1]) > 90) || (coord[0] > 360) || (coord[0] < 0))
     {
	error1("Invalid Coordinates");
	return;
     }


     if (ssf == 1)
     {
	 cmdv[0] = SSFPGM ;
	 cmdv[1] = "-c";
	 sprintf(cra,"%.8g", coord[0]);
	 cmdv[2] = cra;
	 sprintf(cdec,"%.8g", coord[1]);
	 cmdv[3] = cdec;
         sprintf(cband,"%1d", band);
         cmdv[4] = cband;
         sprintf(chcon,"%1d", hcon);
         cmdv[5] = chcon;
         cmdv[6] = (char *) 0;
     }
     else
     {
         cmdv[0] = SKYPGM ;
	 sprintf(cra,"%.8g", coord[0]);
	 cmdv[1] = cra;
	 sprintf(cdec,"%.8g", coord[1]);
	 cmdv[2] = cdec;
         cmdv[3] = (char *) 0;
     }

     if (duplex_pipe(cmdv, &jout, &jin, &job))
     {
         fclose(jin);
         job->fp = jout;
	 job->imgp= 0;
         job->callback =  0 ;      /* register dummy callback (not used) */
	 job->valid_callback = 0;
     }
     else
     {
	 sprintf(server_str, "Unable to start job: %s", cmdv[0]);
	 error1(server_str);
         return;
     }

     while (fgets(readbuf,READBUFLEN,jout) != NULL)
     {
        fputs(readbuf, stdout);
        fputs(readbuf, session);
     }
     fflush(stdout);  /* Keep output in sync */
     fflush(session);

     fclose(jout);    /* put away loose ends */
     del_job(job);
     return;
     /* end of find_plate */
}
