/******************************************************************************
* C Source: Astrometric Calibration                                           *
*                                                                             *
* File:                                                                       *
*    astrmcal.c                                                               *
*                                                                             *
* Description:                                                                *
*    This set of functions are used for applying the astrometric of a plate   *
*    to ultimately compute the x-y position on a plate from a given right     *
*    ascension and declination. These functions were either originally        *
*    written, or ported to C by R. White.                                     *
*                                                                             *
* Module List:                                                                *
*    amdinv                                                                   *
*       Computes x-y position on a plate using CALOBCC solution.              *
*    amdpos                                                                   *
*       Routine to convert x,y to RA,Dec using the CALOBCC solution.          *
*    pltmodel                                                                 *
*       Computes plate model and it's partial derivatives.                    *
*    ppoinv                                                                   *
*       Computes x-y position on a plate using PDS orientation solution.      *
*    xypos                                                                    *
*       Computes x-y from right ascension and declination.                    *
*                                                                             *
* History:                                                                    *
*    Created    : 13-AUG-93 by J. Doggett,                                    *
*       for the Association of Universities for Research in Astronomy, Inc.   *
*          at the Space Telescope Science Institute,                          *
*       to facilitate the accessing of data from the CD-ROM set of the        *
*          Digitized Sky Survey.                                              *
*    Delivered  : 01-MAR-94 by J. Doggett,                                    *
*       GetImage Version 1.0                                                  *
*                                                                             *
*    Updated    : 01-AUG-94 by J. Doggett,                                    *
*       Updated pltmodel function.                                            *
*    Redelivered: 30-NOV-94 by J. Doggett,                                    *
*       GetImage Version 1.1.                                                 *
*                                                                             *
* Copyright (c) 1993, 1994, Association of Universities for Research in       *
* Astronomy, Inc.  All Rights Reserved.                                       *
* Produced under National Aeronautics and Space Administration grant          *
* NAG W-2166.                                                                 *
******************************************************************************/
#include <math.h>

#include "skyview.h"
#include "img.h"

#define ARCSECONDS_PER_RADIAN 206264.8062470964

/******************************************************************************
* Module:                                                                     *
*    amdinv                                                                   *
*                                                                             *
* File:                                                                       *
*    astrmcal.c                                                               *
*                                                                             *
* Description:                                                                *
*    Computes the x,y position on the plate from the right ascension and      *
*    declination using the inverse of the CALOBCC solution. Newton's method   *
*    is used to iterate from a starting position until convergence is         *
*    reached.                                                                 *
*                                                                             *
* Return Value:                                                               *
*    none                                                                     *
*                                                                             *
* Arguments:                                                                  *
*                                                                             *
*    Input Arguments:                                                         *
*       header                                                                *
*          HEADER, Reference                                                  *
*             Structure with header information.                              *
*       ra                                                                    *
*          double, Value                                                      *
*             The right asension of the coordinates to convert to x-y.        *
*       dec                                                                   *
*          double, Value                                                      *
*             The declination of the coordinates to convert to x-y.           *
*       mag                                                                   *
*          float, Value                                                       *
*             Magnitude for the calibration solution.                         *
*       col                                                                   *
*          float, Value                                                       *
*             Color for the calibration solution.                             *
*    Input/Output Arguments:                                                  *
*       none                                                                  *
*    Output Arguments:                                                        *
*       x                                                                     *
*          float, Reference                                                   *
*             Result x position.                                              *
*       y                                                                     *
*          float, Reference                                                   *
*             Result y position.                                              *
*                                                                             *
* History:                                                                    *
*    Created    : 07-MAY-90 by R. White,                                      *
*       Converted from GASP IDL routine to Fortran.                           *
*    Delivered  : 30-JUL-91 by R. White,                                      *
*       Converted from Fortran to C.                                          *
*                                                                             *
*    Updated    : 13-AUG-93 by J. Doggett,                                    *
*       Adopted into software used to facilitate the accessing of data from   *
*          the CD-ROM set of the Digitized Sky Survey.                        *
*    Updated    : 14-DEC-93 by J. Doggett,                                    *
*       Modified to reflect change in calling sequence to traneqstd function. *
*    Redelivered: 01-MAR-94 by J. Doggett,                                    *
*       GetImage Version 1.0                                                  *
******************************************************************************/
extern void
amdinv(imgp,ra,dec,mag,col,x,y)
struct img *imgp;
double ra, dec;
double mag, col;
double *x, *y;
{
int i, max_iterations;
float tolerance;
double xi, eta, object_x, object_y, delta_x, delta_y, f, fx, fy, g, gx, gy;
    void pltmodel();
    void traneqstd();

    /*
     *  Initialize
     */
    i = 0;
    max_iterations = 50;
    tolerance = 0.0000005;
    delta_x = tolerance;
    delta_y = tolerance;
    /*
     *  Convert RA and Dec to St.coords
     */
    traneqstd(imgp->plate_ra,imgp->plate_dec,ra,dec,&xi,&eta);
    /*
     *  Set initial value for x,y
     */
    object_x = xi/imgp->plt_scale;
    object_y = eta/imgp->plt_scale;
    /*
     *  Iterate by Newtons method
     */
    for(i = 0; i < max_iterations; i++) {
        pltmodel(imgp,object_x,object_y,mag,col,&f,&fx,&fy,&g,&gx,&gy);
        f = f-xi;
        g = g-eta;
        delta_x = (-f*gy+g*fy)/(fx*gy-fy*gx);
        delta_y = (-g*fx+f*gx)/(fx*gy-fy*gx);
        object_x = object_x+delta_x;
        object_y = object_y+delta_y;
        if ((fabs(delta_x) < tolerance) && (fabs(delta_y) < tolerance)) break;
    }
    /*
     *  Convert mm from plate center to pixels
     */
    *x = (imgp->ppo_coeff[2]-object_x*1000.0)/imgp->x_pixel_size;
    *y = (imgp->ppo_coeff[5]+object_y*1000.0)/imgp->y_pixel_size;

    /* RBH added */
#ifdef NOTDEF
    *x = *x + 1 - imgp->x_pixel_offset;
    *y = *y + 1 - imgp->y_pixel_offset;
#endif /* NOTDEF */
    *x = *x;
    *y = *y;
    /* end RBH added */

} /* amdinv */
/******************************************************************************
* Module:                                                                     *
*    pltmodel                                                                 *
*                                                                             *
* File:                                                                       *
*    astrmcal.c                                                               *
*                                                                             *
* Description:                                                                *
*    Computes values of a plate model and its parital derivatives for use in  *
*    computing inverse astrometric solutions.                                 *
*                                                                             *
* Return Value:                                                               *
*    none                                                                     *
*                                                                             *
* Arguments:                                                                  *
*                                                                             *
*    Input Arguments:                                                         *
*       h                                                                     *
*          HEADER, Reference                                                  *
*             Structure with header information.                              *
*       x                                                                     *
*          double, Value                                                      *
*             X pixel position.                                               *
*       y                                                                     *
*          double, Value                                                      *
*             Y pixel position.                                               *
*       mag                                                                   *
*          double, Value                                                      *
*             Magnitude                                                       *
*       color                                                                 *
*          double, Value                                                      *
*             Color                                                           *
*    Input/Output Arguments:                                                  *
*       none                                                                  *
*    Output Arguments:                                                        *
*       f                                                                     *
*          double, Reference                                                  *
*             Xi standard coordinate.                                         *
*       fx                                                                    *
*          double, Reference                                                  *
*             Derivative of xi with respect to x.                             *
*       fy                                                                    *
*          double, Reference                                                  *
*             Derivative of xi with respect to y.                             *
*       g                                                                     *
*          double, Reference                                                  *
*             eta standard coordinate.                                        *
*       gx                                                                    *
*          double, Reference                                                  *
*             Derivative of eta with respect to x.                            *
*       gy                                                                    *
*          double, Reference                                                  *
*             Derivative of eta with respect to y.                            *
*                                                                             *
* History:                                                                    *
*    Created    : 07-MAY-90 by R. White,                                      *
*       Converted from GASP IDL routine to Fortran.                           *
*    Delivered  : 30-JUL-91 by R. White,                                      *
*       Converted from Fortran to C.                                          *
*                                                                             *
*    Updated    : 13-AUG-93 by J. Doggett,                                    *
*       Adopted into software used to facilitate the accessing of data from   *
*          the CD-ROM set of the Digitized Sky Survey.                        *
*******************************************************************************

*******************
* pltmodel page 2 *
*******************

*******************************************************************************
*    Redelivered: 01-MAR-94 by J. Doggett,                                    *
*       GetImage Version 1.0                                                  *
*                                                                             *
*    Updated    : 01-AUG-94 by J. Doggett,                                    *
*       Corrected error in term [11] of the derivative of Y model wrt y.      *
*    Redelivered: 30-NOV-94 by J. Doggett,                                    *
*       GetImage Version 1.1.                                                 *
******************************************************************************/
void
pltmodel(imgp,x,y,mag,colour,f,fx,fy,g,gx,gy)
struct img *imgp;
double x,y,mag,colour;
double *f, *fx, *fy, *g, *gx, *gy;
{
double cjunk,x4,y4;

    /*
     *  X plate model
     */
    cjunk=(x*x+y*y)*(x*x+y*y);
    x4 = (x*x)*(x*x);
    y4 = (y*y)*(y*y);
    *f=imgp->amd_x_coeff[0]*x                 + imgp->amd_x_coeff[1]*y              +
       imgp->amd_x_coeff[2]                   + imgp->amd_x_coeff[3]*x*x            +
       imgp->amd_x_coeff[4]*x*y               + imgp->amd_x_coeff[5]*y*y            +
       imgp->amd_x_coeff[6]*(x*x+y*y)         + imgp->amd_x_coeff[7]*x*x*x          +
       imgp->amd_x_coeff[8]*x*x*y             + imgp->amd_x_coeff[9]*x*y*y          +
       imgp->amd_x_coeff[10]*y*y*y            + imgp->amd_x_coeff[11]*x*(x*x+y*y)   +
       imgp->amd_x_coeff[12]*x*cjunk          + imgp->amd_x_coeff[13]*mag           +
       imgp->amd_x_coeff[14]*mag*mag          + imgp->amd_x_coeff[15]*mag*mag*mag   +
       imgp->amd_x_coeff[16]*mag*x            + imgp->amd_x_coeff[17]*mag*(x*x+y*y) +
       imgp->amd_x_coeff[18]*mag*x*(x*x+y*y)  + imgp->amd_x_coeff[19]*colour;
    /*
     *  Derivative of X model wrt x
     */
    *fx=imgp->amd_x_coeff[0]                              +
        imgp->amd_x_coeff[3]*2.0*x                        +
        imgp->amd_x_coeff[4]*y                            +
        imgp->amd_x_coeff[6]*2.0*x                        +
        imgp->amd_x_coeff[7]*3.0*x*x                      +
        imgp->amd_x_coeff[8]*2.0*x*y                      +
        imgp->amd_x_coeff[9]*y*y                          +
        imgp->amd_x_coeff[11]*(3.0*x*x+y*y)               +
        imgp->amd_x_coeff[12]*(5.0*x4  +6.0*x*x*y*y+y4  ) +
        imgp->amd_x_coeff[16]*mag                         +
        imgp->amd_x_coeff[17]*mag*2.0*x                   +
        imgp->amd_x_coeff[18]*mag*(3.0*x*x+y*y);
    /*
     *  Derivative of X model wrt y
     */
    *fy=imgp->amd_x_coeff[1]                     +
        imgp->amd_x_coeff[4]*x                   +
        imgp->amd_x_coeff[5]*2.0*y               +
        imgp->amd_x_coeff[6]*2.0*y               +
        imgp->amd_x_coeff[8]*x*x                 +
        imgp->amd_x_coeff[9]*x*2.0*y             +
        imgp->amd_x_coeff[10]*3.0*y*y            +
        imgp->amd_x_coeff[11]*2.0*x*y            +
        imgp->amd_x_coeff[12]*4.0*x*y*(x*x+y*y)  +

/******************
* pltmodel page 3 *
******************/

        imgp->amd_x_coeff[17]*mag*2.0*y          +
        imgp->amd_x_coeff[18]*mag*2.0*x*y;
    /*
     *  Y plate model
     */
    *g=imgp->amd_y_coeff[0]*y                + imgp->amd_y_coeff[1]*x              +
       imgp->amd_y_coeff[2]                  + imgp->amd_y_coeff[3]*y*y            +
       imgp->amd_y_coeff[4]*y*x              + imgp->amd_y_coeff[5]*x*x            +
       imgp->amd_y_coeff[6]*(x*x+y*y)        + imgp->amd_y_coeff[7]*y*y*y          +
       imgp->amd_y_coeff[8]*y*y*x            + imgp->amd_y_coeff[9]*y*x*x          +
       imgp->amd_y_coeff[10]*x*x*x           + imgp->amd_y_coeff[11]*y*(x*x+y*y)   +
       imgp->amd_y_coeff[12]*y*cjunk         + imgp->amd_y_coeff[13]*mag           +
       imgp->amd_y_coeff[14]*mag*mag         + imgp->amd_y_coeff[15]*mag*mag*mag   +
       imgp->amd_y_coeff[16]*mag*y           + imgp->amd_y_coeff[17]*mag*(x*x+y*y) +
       imgp->amd_y_coeff[18]*mag*y*(x*x+y*y) + imgp->amd_y_coeff[19]*colour;
    /*
     *  Derivative of Y model wrt x
     */
    *gx=imgp->amd_y_coeff[1]                    +
        imgp->amd_y_coeff[4]*y                  +
        imgp->amd_y_coeff[5]*2.0*x              +
        imgp->amd_y_coeff[6]*2.0*x              +
        imgp->amd_y_coeff[8]*y*y                +
        imgp->amd_y_coeff[9]*y*2.0*x            +
        imgp->amd_y_coeff[10]*3.0*x*x           +
        imgp->amd_y_coeff[11]*2.0*x*y           +
        imgp->amd_y_coeff[12]*4.0*x*y*(x*x+y*y) +
        imgp->amd_y_coeff[17]*mag*2.0*x         +
        imgp->amd_y_coeff[18]*mag*y*2.0*x;
    /*
     *  Derivative of Y model wrt y
     */
    *gy=imgp->amd_y_coeff[0]                              +
        imgp->amd_y_coeff[3]*2.0*y                        +
        imgp->amd_y_coeff[4]*x                            +
        imgp->amd_y_coeff[6]*2.0*y                        +
        imgp->amd_y_coeff[7]*3.0*y*y                      +
        imgp->amd_y_coeff[8]*2.0*y*x                      +
        imgp->amd_y_coeff[9]*x*x                          +
        imgp->amd_y_coeff[11]*(x*x+3.0*y*y)               +
        imgp->amd_y_coeff[12]*(5.0*y4  +6.0*x*x*y*y+x4  ) +
        imgp->amd_y_coeff[16]*mag                         +
        imgp->amd_y_coeff[17]*mag*2.0*y                   +
        imgp->amd_y_coeff[18]*mag*(x*x+3.0*y*y);

} /* pltmodel */

/******************************************************************************
* Module:                                                                     *
*    traneqstd                                                                *
*                                                                             *
* File:                                                                       *
*    astronmy.c                                                               *
*                                                                             *
* Description:                                                                *
*    Routine to convert RA and Dec in radians to standard coordinates on a    *
*    plate.                                                                   *
*                                                                             *
* Return Value:                                                               *
*    none                                                                     *
*                                                                             *
* Arguments:                                                                  *
*                                                                             *
*    Input Arguments:                                                         *
*       plt_center_ra                                                         *
*          double, Value                                                      *
*             Plate center right ascension in radians                         *
*       plt_center_dec                                                        *
*          double, Value                                                      *
*             Plate center declination in radians                             *
*       object_ra                                                             *
*          double, Value                                                      *
*             Input right ascension in radians.                               *
*       object_dec                                                            *
*          double, Value                                                      *
*             Input declination in radians.                                   *
*    Input/Output Arguments:                                                  *
*       none                                                                  *
*    Output Arguments:                                                        *
*       Object_Xi                                                             *
*          double, Reference                                                  *
*             Output xi standard coordinate in arcseconds.                    *
*       Object_Eta                                                            *
*          double, Reference                                                  *
*             Output eta standard coordinate in arcseconds.                   *
*                                                                             *
* History:                                                                    *
*    Created    : 07-MAY-90 by R. White,                                      *
*       Converted to Fortran from IDL routine.                                *
*    Delivered  : 30-JUL-91 by R. White,                                      *
*       Convert from Fortran to C.                                            *
*                                                                             *
*    Updated    : 13-AUG-93 by J. Doggett,                                    *
*       Adopted into software used to facilitate the accessing of data from   *
*          the CD-ROM set of the Digitized Sky Survey.                        *
*    Updated    : 14-DEC-93 by J. Doggett,                                    *
*       Modified to take plate center coordinates explicitly as arguments     *
*          rather than from the plate header.                                 *
*    Redelivered: 01-MAR-94 by J. Doggett,                                    *
*       GetImage Version 1.0                                                  *
******************************************************************************/
void
traneqstd
     (plt_center_ra, plt_center_dec,object_ra,object_dec,object_xi,object_eta)
double plt_center_ra, plt_center_dec;
double object_ra, object_dec;
double *object_xi, *object_eta;
{
double div;



/*******************
* traneqstd page 2 *
*******************/

    /*
     *  Find divisor
     */

    div=(sin(object_dec)*sin(plt_center_dec)+
        cos(object_dec)*cos(plt_center_dec)*
        cos(object_ra - plt_center_ra));
    /*
     *  Compute standard coords and convert to arcsec
     */
    *object_xi=cos(object_dec)*sin(object_ra - plt_center_ra)*
        ARCSECONDS_PER_RADIAN/div;

    *object_eta=(sin(object_dec)*cos(plt_center_dec)-
        cos(object_dec)*sin(plt_center_dec)*
        cos(object_ra - plt_center_ra))*
        ARCSECONDS_PER_RADIAN/div;

} /* traneqstd */
