/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "parse.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <string.h>

#ifdef NO_FLOAT_H
#include <values.h>
#define DBL_MAX MAXDOUBLE
#define FLT_MAX MAXFLOAT
#else
#include <float.h>
#endif


double          backgnd_level = 0.;
static int      backgnd_flag = FALSE;
static double   backgnd_rms = 0.0, backgnd_stdev = 0.0;
static double   backgnd_min = 0.0, backgnd_max = 0.0;
static double   dtr = M_PI/180;
static double   rtd = 180/M_PI;
static FILE *fov_fd;
int point_mode = 0;

extern int      debug, graphics, server_mode, be_quiet;
extern char     server_str[];
extern char     plane, bunit[];
extern char filnam[], mod_filnam[];
extern char last_coord[2][40];
extern int      image_frame, output_coord_sys;
extern double   cdelt1, cdelt2, b_zero, bscale;
extern int sgn_y;
extern FILE    *session, *debug_file;
extern struct img head_img, *curr_img;
extern double win_zoom;
extern int      s1_server_mode;
extern int      ads_server_mode;

extern int bitpix;

/******************************************************************/
/* 								  */
/* PICK                                                           */
/* uses the trackball to define a point for flux extraction       */
/* 								  */
/******************************************************************/


void
pick(argc, argv)
int argc;
char **argv;
{
    int             status;
    double          z, zint, zmax;
    double          d_coord[2], fluxes[4];
    char            *cmd_name, s_coord[2][40], *coord_str;
    int             i, j, i_coord[2], coord_sys;
    int             xcur, ycur, xcur_IRAF, ycur_IRAF,  win_x, win_y;
    int             area, area2, pixelcount;
    int             n_backgnd;
    double          backgnd, backgnd_sq;
    struct img     *imgp;
    char tmp_filnam[MAXPATHLEN + 12];
    double x, y;
    double coefftot, contribtot;
    double thiscoeff;
    int interpolating;
    int jsys;
    double equinox;


    cmd_name = argv[0];

    n_backgnd = 0;
    backgnd = 0.0;
    backgnd_sq = 0.0;
    backgnd_min = DBL_MAX;
    backgnd_max = -DBL_MAX;
    area = 1;
    interpolating = 0;


    if (head_img.rlink == &head_img)
    {
	sprintf(server_str, "no images currently being displayed");
	error1(server_str);
	return;
    }

    if (strncmp(argv[argc-1], "int", 3) == 0)
    {
	argc--;
	interpolating = 1;
    }

    /* process the area if given */
    if ((argc == 2) || (argc == 4))
    {
	if (getint(*(++argv), &area) == FALSE)
	    return;
	if (area < 1)
	    area = 1;
    }

    /* process the coordinates if given */
    if (argc >= 3)
    {
	strcpy(s_coord[0], *(++argv));
	strcpy(s_coord[1], *(++argv));

	imgp = sky_to_image(s_coord, d_coord);
	if (imgp == NULL)
	{
	    error1("coordinates not on any visible image");
	    return;
	}
	dIM_to_SC(d_coord);
	xcur = Round(d_coord[0]);
	ycur = Round(d_coord[1]);
	x = d_coord[0] - xcur;
	y = d_coord[1] - ycur;
    }
    else
    {
	if (!graphics)
	    return;

	/* for trackball pick, start cursor in center of visible screen */
	get_center_xy(i_coord);
	xcur = i_coord[0];
	ycur = i_coord[1];
    }

    area = area / 2;

    area2 = 2 * area + 1;

    if ((server_mode == TRUE) && (backgnd_flag == FALSE))
    {
	if ((argc < 3) && (!point_mode))
	{
	    if(s1_server_mode)
	    {
	        srvflush(cmd_name, 1, 0);
	        /* table file header */
	        printf("table\n");
	        printf(
	        "|x           |y           |coord_sys |pixel_value |units   |");
	        printf("max_pixel_value |pixel_count |frame |filename");
	        printf("%-*.*s|\n", MAXPATHLEN+12, MAXPATHLEN+12, "");
	    }
	}
    }

    if (area != 0)
    {
	if(server_mode == FALSE)
	    printf("Averaging over a %dx%d box.\n", area2, area2);
	fprintf(session, "Averaging over a %dx%d box.\n", area2, area2);
    }

    for (;;)
    {
	if (argc < 3)		/* if using trackball */
	{
	    status = trackball_pick(&xcur, &ycur, &win_x, &win_y);
	    if (status == FALSE)
		break;
	    d_coord[0] = win_x;
	    d_coord[1] = win_y;
	    dWIN_to_SC(d_coord);
	    x = d_coord[0] - xcur;
	    y = d_coord[1] - ycur;
	}

	next_bundle();
	if (interpolating)
	{
	    overlay_setup('g');
	    do_mark(d_coord[0], d_coord[1], CROSS, 7, 0, 0);
	}
	else
	    mark_the_spot(xcur, ycur);
	zint = 0.;
	pixelcount = 0;
	zmax = -DBL_MAX;
	coefftot = 0.0;
	contribtot = 0.0;

	d_coord[0] = xcur;
	d_coord[1] = ycur;
	dSC_to_IRAF(d_coord);
	xcur_IRAF = Round(d_coord[0]);
	ycur_IRAF = Round(d_coord[1]);

	for (i = -area; i <= area; ++i)
	{
	    for (j = -area; j <= area; ++j)
	    {
		i_coord[0] = xcur_IRAF + i;
		i_coord[1] = ycur_IRAF + j;

		flux_IRAF(i_coord, fluxes);

		if (FINITE(fluxes[0]))
		{
		    z = fluxes[0] * bscale + b_zero;
		    ++pixelcount;
		    zint += z;

		    ++n_backgnd;
		    backgnd += z;
		    backgnd_sq += z * z;

		    if (z > zmax)
			zmax = z;
		    if (z > backgnd_max)
			backgnd_max = z;
		    if (z < backgnd_min)
			backgnd_min = z;
		    thiscoeff = 
		    (1 - 2.0/area2 *fabs(i - x)) * (1 - 2.0/area2 *fabs(j - y));
		    coefftot += thiscoeff;
		    contribtot += thiscoeff * z;
#ifdef NOTDEF
		    printf(
		    "thiscoeff = %g  thiscontrib = %g\n"
		    , thiscoeff, thiscoeff*z);
#endif /* NOTDEF */
		}
	    }
	}
#ifdef NOTDEF
	if (pixelcount > 1)
	 {
	    printf("contribtot = %g  coefftot = %g\n", contribtot, coefftot);
	 }
#endif  /* NOTDEF */


	d_coord[0] = xcur;
	d_coord[1] = ycur;
	dSC_to_IM(d_coord);
	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);

	if (argc < 3)
	{
	    strcpy(last_coord[0], s_coord[0]);
	    strcpy(last_coord[1], s_coord[1]);
	}


	/* first take care of session history file */
	if (pixelcount == 0)
	{
	    fprintf(session, "No flux available at %s\n", coord_str);
	}
	else
	{
	    zint = zint / pixelcount;  /* zint is also used later */
	    fprintf(session, "%-g %s at %s", zint, bunit, coord_str);
	}
	if (pixelcount > 1)
	{
	    fprintf(session, " (using %d pixels)\n", pixelcount);
	    fprintf(session, " maximum %-g %s\n", zmax, bunit);
	    if (interpolating)
		fprintf(session,
		    "interpolated flux = %g\n", contribtot / coefftot);
	}


	if(server_mode == FALSE)
	{
	    if (pixelcount == 0)
	    {
		sprintf(server_str, "No flux available at %s\n", coord_str);
		error1(server_str);
	    }
	    else
	    {
		printf("%-g %s at %s", zint, bunit, coord_str);
	    }
	    if (pixelcount > 1)
	    {
		printf(" (using %d pixels)\n", pixelcount);
		printf(" maximum %-g %s\n", zmax, bunit);
		if (interpolating)
		    printf( "interpolated flux = %g\n", contribtot / coefftot);
	    }
	}
	else if(backgnd_flag == FALSE)
	{
	    if ((argc >= 3) || (point_mode))   /* if not doing a tablefile */
	    {
		if (pixelcount == 0)
		{
		    if (strncmp(coord_str, "undefined", 9) == 0)
		    {
			error1("undefined location");
		    }
		    else
		    {
			/* output NaN */
			srv_coord_val("center", coord_sys, s_coord, equinox, 
			    jsys, "NaN", bunit);
			sprintf(server_str, "%d", pixelcount);
			srv_real("pixel_count", server_str);
			sprintf(server_str, "%d", image_frame);
			srv_real("frame", server_str);
			sprintf(server_str, "%s%s", 
			    filnam, mod_filnam[0]?" (modified)":"");
			srv_string("filename", server_str);
		    }
		}
		else
		{
		    sprintf(server_str,"%-g", zint);
		    srv_coord_val("center", coord_sys, s_coord, equinox, jsys,
			server_str, bunit);
		    sprintf(server_str, "%d", pixelcount);
		    srv_real("pixel_count", server_str);
		    sprintf(server_str, "%-g", zmax);
		    srv_real("max_pixel_value", server_str);
		    sprintf(server_str, "%d", image_frame);
		    srv_real("frame", server_str);
		    sprintf(server_str, "%s%s", 
			filnam, mod_filnam[0]?" (modified)":"");
		    srv_string("filename", server_str);
		    if (interpolating)
		    {
			sprintf(server_str, "%-g", contribtot / coefftot);
			srv_real("interpolated_flux", server_str);
		    }
		}
	    } 
	    else
	    {
		if(s1_server_mode)
		{
		    /* output table file line */
		    strcpy(tmp_filnam, filnam);
		    if (mod_filnam[0] != '\0')
		       strcat(tmp_filnam, " (modified)");
		    if (pixelcount == 0)
		    {
		    printf("%-12.12s %-12.12s %-10.10s %-12.12s %-8.8s %-12.12s     %-5d        %-3d    %-*.*s\n",
			s_coord[0], s_coord[1], coord_name(coord_sys),
			"NaN", bunit, "NaN", pixelcount, image_frame, 
			MAXPATHLEN+12, MAXPATHLEN+12, tmp_filnam);
		    }
		    else
		    {
		    printf("%-12.12s %-12.12s %-10.10s %-12.5e %-8.8s %-12.5e     %-5d        %-3d    %-*.*s\n",
			s_coord[0], s_coord[1], coord_name(coord_sys),
			zint, bunit, zmax, pixelcount, image_frame, 
			MAXPATHLEN+12, MAXPATHLEN+12, tmp_filnam);
		    }
		    fflush(stdout);
		}
		else
		{
		    /* ads server mode */
		    srv_string("state", "point");
		    if (pixelcount == 0)
			strcpy(server_str, "NaN");
		    else
			sprintf(server_str,"%-g", zint);
		    srv_coord_val("center", coord_sys, s_coord, equinox, jsys,
			server_str, bunit);
		    sprintf(server_str, "%d", pixelcount);
		    srv_real("pixel_count", server_str);
		    if (pixelcount == 0)
			strcpy(server_str, "NaN");
		    else
			sprintf(server_str, "%-g", zmax);
		    srv_real("max_pixel_value", server_str);
		    sprintf(server_str, "%d", image_frame);
		    srv_real("frame", server_str);
		    sprintf(server_str, "%s%s", 
			filnam, mod_filnam[0]?" (modified)":"");
		    srv_string("filename", server_str);

		    srvflush(cmd_name, 0, 0);
		    srvinit();
		    srv_string("cmd", cmd_name);
		}
	    }
	}

	if ((argc >= 3) || (point_mode))   /* if not using trackball */
	{
	    break;
	}
    }

    if ((server_mode == TRUE) && (backgnd_flag == FALSE) &&
	(argc < 3) && (!point_mode))
    {
	if(s1_server_mode)
	    printf("table_end\n");
	else
	{
	    srvinit();
	    srv_string("state", "done");
	}
    }

    if ((backgnd_flag == TRUE) && (n_backgnd > 0))
    {
	backgnd_level = backgnd / n_backgnd;
	backgnd_stdev = sqrt(backgnd_sq / n_backgnd - 
	    backgnd_level * backgnd_level);
	backgnd_rms = sqrt(backgnd_sq / n_backgnd );
	if (debug)
	    fprintf(debug_file, "pick:  background %-g, using %d points\n",
		   backgnd_level, n_backgnd);
    }
}


/******************************************************************/
/* 								  */
/* PIXCOORD                                                       */
/* Returns the exact coordinate of a fractional pixel             */
/* 								  */
/******************************************************************/


void
pixcoord(argc, argv)
int argc;
char **argv;
{
    double          d_coord[2];
    char            *cmd_name, s_coord[2][40], *coord_str;
    int             coord_sys;
    struct img     *imgp;
    int jsys;
    double equinox;
    int local_debug = 0;


    if (strcmp(argv[argc-1], "debug") == 0)
    {
       local_debug = 1;
       argc--;
    }

    cmd_name = argv[0];

    if (head_img.rlink == &head_img)
    {
	sprintf(server_str, "no images currently being displayed");
	error1(server_str);
	return;
    }

    /* process the coordinates if given */
    if (argc >= 3)
    {
	strcpy(s_coord[0], *(++argv));
	strcpy(s_coord[1], *(++argv));

	if(local_debug)
	{
	   printf("input s_coord[]: %s %s\n", s_coord[0], s_coord[1]);
	   fflush(stdout);
	}

	imgp = sky_to_image(s_coord, d_coord);

	if (imgp == NULL)
	{
	    error1("coordinates not on any visible image");
	    return;
	}

	if(local_debug)
	{
	   printf("after sky_to_image() d_coord[]: %-g %-g\n", d_coord[0], d_coord[1]);
	   fflush(stdout);
	}

	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);

	if(server_mode == FALSE)
	{
	    printf("Location: %s", coord_str);
	}
	else
	{
	    srv_coord("location", coord_sys, s_coord, equinox, jsys);
	}
    }
}


void
point(argc, argv)
int argc;
char **argv;
{
    point_mode = 1;
    pick(argc, argv);
    point_mode = 0;
}

void
gpoint(argc, argv)
int argc;
char **argv;
{
    point_mode = 1;
    gpick(argc, argv);
    point_mode = 0;
}

/******************************************************************/
/* 								  */
/* GPICK                                                          */
/* define a point to search for a graphics mark                   */
/* 								  */
/******************************************************************/


void
gpick(argc, argv)
int argc;
char **argv;
{
    int             status, find_status;
    double          d_coord[2];
    char            s_coord[2][40];
    int             xcur, ycur, xwin, ywin;
    int             i_coord[2];
    struct img     *imgp;
    char *cmd_name;

    cmd_name = argv[0];

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    /* process the coordinates if given */
    if (argc >= 3)
    {
	strcpy(s_coord[0], *(++argv));
	strcpy(s_coord[1], *(++argv));

	imgp = sky_to_image(s_coord, d_coord);
	dIM_to_SC(d_coord);
	dSC_to_WIN(d_coord);
	xwin = Round(d_coord[0]);
	ywin = Round(d_coord[1]);

	if (imgp == NULL)
	{
	    error1("coordinates not on any visible image");
	    return;
	}
	point_mode = 1;
	find_status = find_mark( xwin, ywin);
	point_mode = 0;
	return;
    }

    if (!graphics)
	return;

    if ((server_mode == TRUE) && (!point_mode))
    {
	if (s1_server_mode)
	{
	    srvflush(argv[0], 1, 0);
	    /* table file header */
	    printf("table\n");
	    printf(
	    "|x           |y           |coord_sys |table_line |filename");
	    printf("%-*.*s|\n", MAXPATHLEN+12, MAXPATHLEN+12, "");
	}
    }

    /* for trackball pick, start cursor in center of visible screen */
    get_center_xy(i_coord);
    xcur = i_coord[0];
    ycur = i_coord[1];

    for (;;)
    {
	status = trackball_pick(&xcur, &ycur, &xwin, &ywin);
	if (status == FALSE)
	    break;

	find_status = find_mark(xwin, ywin);
	if (point_mode || !find_status) 
	    break;
	srvflush(cmd_name, 0, 0);
	srvinit();
    }
    if ((server_mode == TRUE) && (!point_mode))
    {
	if(s1_server_mode)
	    printf("table_end\n");
	else
	    srv_string("state", "done");
    }
}


/******************************************************************/
/* 								  */
/* SKY_BOX                                                        */
/* define background using multiple boxes                         */
/* 								  */
/******************************************************************/


void
sky_box(argc, argv)
int argc;
char **argv;
{
    int             status;
    double          flux_mean, clip_level, flux_squared, stdev, z, flux_sum;
    double          d_coord[2], fluxes[4];
    char            s_coord[2][40], *coord_str;
    int             i, j, i_coord[2];
    int             coord_sys, xcur, ycur, xcur_IRAF, ycur_IRAF, dummyx, dummyy;
    int             area, area2, pixelcount, clipped_pixels;
    double grand_mean, grand_sum, grand_squared, grand_stdev, grand_stdev_sum;
    double grand_min, grand_max, min, max;
    int grand_count;
    int local_debug = 0;
    int zap_it = 0;
    int changes, min_x, max_x, min_y, max_y;
    void *filptr;   /* pointer to pixels */
    int jsys;
    double equinox;

    if (!graphics)
	return;

    if (strcmp(argv[argc-1], "debug") == 0)
    {
	local_debug = 1;
	argc--;
    }

    if (strcmp(argv[argc-1], "zap") == 0)
    {
	zap_it = 1;
	argc--;
    }

    area = 1;



    /* process the area if given */
    if (argc >= 2)
    {
	if (getint(*(++argv), &area) == FALSE)
	    return;
	if (area < 1)
	    area = 1;
    }
    if (argc >= 3)
	clip_level = atof(*(++argv));
    else
	clip_level = 0;

    /* process the coordinates if given */
    /* By specification, they are not given */

    /*  start cursor in center of visible screen */
    get_center_xy(i_coord);
    xcur = i_coord[0];
    ycur = i_coord[1];

    area = area / 2;

    area2 = 2 * area + 1;

    if (area != 0)
    {
	if(server_mode == FALSE)
	    printf("Averaging over a %dx%d box.\n", area2, area2);
	fprintf(session, "Averaging over a %dx%d box.\n", area2, area2);
    }

    grand_mean = 0.;
    grand_sum = 0.;
    grand_squared = 0.;
    grand_stdev_sum = 0.;
    grand_count = 0;;
    grand_min = DBL_MAX;
    grand_max = -DBL_MAX;

    if(s1_server_mode)
    {
	srvflush(argv[0], 1, 0);
	/* table file header */
	printf("table\n");
    }

    if ((server_mode == FALSE) || (s1_server_mode))
    {
	printf(
    "|  x          |   y         | mean       |units   |stdev       |pixels|clipped | min          | max\n");
    }

    for (;;)
    {
	status = trackball_pick(&xcur, &ycur, &dummyx, &dummyy);
	if (status == FALSE)
	    break;

	overlay_setup('g');
	overlay_color(2);
	next_bundle();
	overlay_dmov((double) (xcur-area), (double) (ycur-area));
	overlay_ddraw((double) (xcur-area), (double) (ycur+area));
	overlay_ddraw((double) (xcur+area), (double) (ycur+area));
	overlay_ddraw((double) (xcur+area), (double) (ycur-area));
	overlay_ddraw((double) (xcur-area), (double) (ycur-area));
	image_setup(plane);

	flux_sum = 0.;
	flux_squared = 0.;
	pixelcount = 0;
	clipped_pixels = 0;
	min = DBL_MAX;
	max = -DBL_MAX;

	d_coord[0] = xcur;
	d_coord[1] = ycur;
	dSC_to_IRAF(d_coord);
	xcur_IRAF = Round(d_coord[0]);
	ycur_IRAF = Round(d_coord[1]);

	for (i = -area; i <= area; ++i)
	{
	    for (j = -area; j <= area; ++j)
	    {
		i_coord[0] = xcur_IRAF + i;
		i_coord[1] = ycur_IRAF + j;

		flux_IRAF(i_coord, fluxes);


		if (local_debug)
		{
		    if (!FINITE(fluxes[0]))
			printf("got blank flux\n");
		    else
			printf("got flux %g\n", fluxes[0]);
		}

		if (FINITE(fluxes[0]))
		{
		    z = fluxes[0] * bscale + b_zero;
		    ++pixelcount;
		    flux_sum += z;
		    flux_squared += z * z;
		    if (z < min)
			min = z;
		    if (z > max)
			max = z;
		}
	    }
	}

	d_coord[0] = xcur;
	d_coord[1] = ycur;
	dSC_to_IM(d_coord);
	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);

	if (pixelcount != 0)
	{
	    flux_mean = flux_sum / pixelcount;
	    stdev = sqrt(flux_squared / pixelcount - flux_mean * flux_mean);

	    /* x y mean stddev #pixels #clipped_pixels */
	    if (local_debug)
	    {
		printf("before clipping:\n");
		printf(
	       "%-12.12s  %-12.12s  %-12g %-7.7s  %-11g  %-5d  %-5d    %-12g   %-12g\n", 
		s_coord[0], s_coord[1],
		flux_mean, bunit, stdev, pixelcount, 
		clipped_pixels, min, max);
		printf("after clipping:\n");
	    }

	flux_sum = 0.;
	flux_squared = 0.;
	pixelcount = 0;
	min = DBL_MAX;
	max = -DBL_MAX;

	for (i = -area; i <= area; ++i)
	{
	    for (j = -area; j <= area; ++j)
	    {
		i_coord[0] = xcur_IRAF + i;
		i_coord[1] = ycur_IRAF + j;

		flux_IRAF(i_coord, fluxes);

		if (FINITE(fluxes[0]))
		{
		    z = fluxes[0] * bscale + b_zero;
		    if ((clip_level == 0) ||
			(fabs(z - flux_mean) < clip_level * stdev))
		    {
			++pixelcount;
			flux_sum += z;
			flux_squared += z * z;
			if (z < min)
			    min = z;
			if (z > max)
			    max = z;
		    }
		    else
		    {
			clipped_pixels++;
			if (zap_it)
			{
			    filptr = zap_flux(i_coord);
			    if (filptr == NULL)
				return;  /* abort out of here */
			    changes = TRUE;
			}
		    }
		}
	    }
	}
	    if ((zap_it) && (changes))
	    {
		min_x = xcur - area;
		min_y = ycur - area;
		max_x = xcur + area;
		max_y = ycur + area;
		repaintm32(min_x, max_x, min_y, max_y, filptr);
		repaint_x(min_x, max_x, min_y, max_y);
	    }
	    flux_mean = flux_sum / pixelcount;
	    stdev = sqrt(flux_squared / pixelcount - flux_mean * flux_mean);
	    grand_sum += flux_mean;
	    grand_stdev_sum += stdev;
	    grand_squared += flux_mean * flux_mean;
	    grand_count++;
	    if (min < grand_min)
		grand_min = min;
	    if (max > grand_max)
		grand_max = max;

	    /* x y mean stddev #pixels #clipped_pixels */

	    if ((server_mode == FALSE) || (s1_server_mode))
	    {
	    printf(
	    "%-12.12s  %-12.12s  %-12g %-7.7s  %-11g  %-5d  %-5d    %-12g   %-12g\n", 
	    s_coord[0], s_coord[1],
	    flux_mean, bunit, stdev, pixelcount, clipped_pixels,
	    min, max);
	    }
	    else
	    {
		/* ads server mode */
		    srv_string("state", "point");
		    if (pixelcount == 0)
			strcpy(server_str, "NaN");
		    else
			sprintf(server_str,"%-g", flux_mean);
		    srv_coord("center", coord_sys, s_coord, equinox, jsys);
		    sprintf(server_str, "%g", min);
		    srv_real("min", server_str);
		    sprintf(server_str, "%g", max);
		    srv_real("max", server_str);
		    sprintf(server_str, "%g", flux_mean);
		    srv_real("mean", server_str);
		    srv_string("units", bunit);
		    sprintf(server_str, "%g", stdev);
		    srv_real("stdev", server_str);
		    sprintf(server_str, "%d", pixelcount);
		    srv_real("pixel_count", server_str);
		    sprintf(server_str, "%d", clipped_pixels);
		    srv_real("clipped_pixels", server_str);

		    srvflush(argv[0], 0, 0);
		    srvinit();
	    }

	    fprintf(session, "%-g %s at %s", flux_mean, bunit, coord_str);
	}
	else
	{
	    printf("%-22.22s                                     %-5d %-5d\n", 
	    coord_str,                     pixelcount, clipped_pixels);
	    fprintf(session, "No flux available at %s\n", coord_str);
	}
    }

    if (local_debug)
	printf(
	"grand_sum = %g  grand_count = %d  grand_squared = %g  grand_stdev_sum = %g\n",
	grand_sum, grand_count, grand_squared, grand_stdev_sum);
	grand_mean = grand_sum / grand_count;
	grand_stdev = sqrt(grand_squared / grand_count - grand_mean * grand_mean);

    if(server_mode == FALSE)
    {
	printf("mean of means = %g   mean stdev = %g  stdev of means = %g\n", 
	    grand_mean, grand_stdev_sum / grand_count, grand_stdev);
	printf("  min of mins = %g   max of maxs = %g\n", grand_min, grand_max);
    }
    else
    {
	if (s1_server_mode)
	{
	    printf("table_end\n");
	}
	else 
	{
	    srvinit();
	    srv_string("state", "done");
	}
	sprintf(server_str, "%g", grand_mean);
	srv_real("mean_of_means", server_str);
	sprintf(server_str, "%g", grand_stdev_sum / grand_count);
	srv_real("mean_stdev", server_str);
	sprintf(server_str, "%g", grand_stdev);
	srv_real("stdev_of_means", server_str);
	sprintf(server_str, "%g", grand_min);
	srv_real("min_of_mins", server_str);
	sprintf(server_str, "%g", grand_max);
	srv_real("max_of_maxs", server_str);
    }

}

/******************************************************************/
/* 								  */
/* MARK                                                           */
/* places a marker at the specified point                         */
/* 								  */
/******************************************************************/


void
mark(argc, argv)
int argc;
char **argv;
{
    double d_coord[2], tail_args[4];
    double width, height, angle;
    char s_coord[2][40];
    int i, tail_count;
    int got_coords, got_color;
    struct img     *imgp, *save_imgp;
    static int      markertype = CROSS, markersize = 7, markercolor = 2;
    static char     overlay_plane = 'g';
    static char     xcolor[25] = {'g', 'r', 'e', 'e', 'n'};

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    if (argc == 1)
    {
	/* undocumented trackball mark */
	/*  - like pick, but with subpixel accuracy, and no flux report */
	if (graphics)
	    trackball_mark();
	return;
    }

    if (argc < 3)
    {
	error1("Too few arguments for mark command");
	return;
    }


    save_imgp = curr_img;    /* save current image */
    got_coords = FALSE;
    got_color = FALSE;
    tail_count = 0;

    for (i = 1; i < argc; i++)
    {
	switch(cmd(argv[i]))
	{
	    case DOT:
	    case CROSS:
	    case ASTERISK:
	    case CIRCLE:
	    case X:
	    case BOX:
	    case DIAMOND:
	    case TRIANGLE:
	    case ELLIPSE:
		markertype = cmd(argv[i]);
		break;
	    default:
		if (!got_coords)  /* coords must come first */
		{
		    /* process the coordinates */
		    strcpy(s_coord[0], argv[i]);
		    i++;
		    if (argc > i)
			strcpy(s_coord[1], argv[i]);
		    else
		    {
			/* only one coordinate value */
			error1("mark error: incomplete coordinate pair");
			return;
		    }

		    imgp = sky_to_image(s_coord, d_coord);
		    dIM_to_SC(d_coord);

		    if (imgp == NULL)
		    {
			/* its not on current image - not an error */
			/*  - just discard it */
			return;
		    }

		    got_coords = TRUE;
		}
		else if (!got_color)
		{
		    if (good_overlay_color(argv[i]))
		    {
			switch(cmd(argv[i]))
			{
			    case RED:
				overlay_plane = 'r';
				markercolor = 1;
				strcpy(xcolor, "red");
				break;
			    case GREEN:
				overlay_plane = 'g';
				markercolor = 2;
				strcpy(xcolor, "green");
				break;
			    case BLUE:
				overlay_plane = 'b';
				markercolor = 4;
				strcpy(xcolor, "blue");
				break;
			    case WHITE:
				overlay_plane = 'w';
				markercolor = 15;
				strcpy(xcolor, "white");
				break;
			    default:
				strcpy(xcolor, argv[i]);
			}
			got_color = 1;
		    }
		}
		else
		{
		    if (!isdbl(argv[i], &tail_args[tail_count++]))
		    {
			error1("unrecognized argument in mark command");
			fill_glob(save_imgp);   /* reset current image */
			return;
		    }
		}
	}
    }


    if (!got_coords)
    {
	error1("mark error:  coordinates not recognized");
	return;
    }

    save_overlay_color();
    set_overlay_color(xcolor);
    overlay_setup(overlay_plane);
    overlay_color(markercolor);
    next_bundle();
    if (markertype == ELLIPSE)
    {
	if (tail_count < 3)
	{
	    error1("mark error:  not enough arguments to describe ellipse");
	    fill_glob(save_imgp);   /* reset current image */
	    return;
	}
	width = tail_args[0];
	height = tail_args[1];
	angle = tail_args[2];
	if ((cdelt1 != 0) &&
	    (output_coord_sys != IM) &&
	    (output_coord_sys != SC) &&
	    (output_coord_sys != IRAF))
	{
	    width = .5 + (width/ (60.0 * fabs(cdelt1)));
	    height = .5 + (height/ (60.0 * fabs(cdelt1)));
	}

	do_mark(d_coord[0], d_coord[1], markertype, (int) width, 
	    (int) height, (int) angle);
    }
    else
    {
	if (tail_count >= 1)
	    markersize = tail_args[0];
	do_mark(d_coord[0], d_coord[1], markertype, markersize, 0, 0);
    }
    image_setup(plane);
    restore_overlay_color();  /* nullify effect of tmp_overlay_color() */

    if (imgp != save_imgp)
    {
	/* mark is not on current image */
	fill_glob(save_imgp);   /* reset current image */
    }
}

/******************************************************************/
/* 								  */
/* BACKGROUND                                                     */
/* uses 'pick' to define several points whose average is used as  */
/* the background level during the 'examine' command              */
/* 								  */
/******************************************************************/

void
background(argc, argv)
int argc;
char **argv;
{
    if ((argc > 1) && (argv[1][0] == '='))
    {
	if (argc >= 3)
	    backgnd_level = atof(argv[2]);
	else
	    backgnd_level = atof(&argv[1][1]);


	if(server_mode == FALSE)
	    printf("Background level is %-g\n\n", backgnd_level);
	else
	{
	    sprintf(server_str, "%-g", backgnd_level);
	    srv_real("background_level", server_str);
	}
	fprintf(session, "Background level is %-g\n\n", backgnd_level);
	return;
    }

    else
    {
	backgnd_flag = TRUE;
	pick(argc, argv);
	backgnd_flag = FALSE;

	if(server_mode == FALSE)
	{
	    printf("Background level is %-g     min = %-g  max = %-g\n", 
		backgnd_level, backgnd_min, backgnd_max);
	    printf("          std dev = %-g   rms= %-g\n\n", 
		backgnd_stdev, backgnd_rms);
	}
	else
	{
	    sprintf(server_str, "%-g", backgnd_level);
	    srv_real("background_level", server_str);
	    sprintf(server_str, "%-g", backgnd_min);
	    srv_real("background_min", server_str);
	    sprintf(server_str, "%-g", backgnd_max);
	    srv_real("background_max", server_str);
	    sprintf(server_str, "%-g", backgnd_stdev);
	    srv_real("background_stdev", server_str);
	    sprintf(server_str, "%-g", backgnd_rms);
	    srv_real("background_rms", server_str);
	}

	fprintf(session, "Background level is %-g     min = %-g  max = %-g\n", 
	    backgnd_level, backgnd_min, backgnd_max);
	fprintf(session, "          std dev = %-g   rms= %-g\n\n", 
	    backgnd_stdev, backgnd_rms);
	return;
    }
}

/***********************************************************************/
/*                                                                     */
/* ZOOM_PAN                                                            */
/* zooms in by a factor of between 1 and 16 and pans the image         */
/*                                                                     */
/***********************************************************************/


void
zoom_pan(argc, argv)
int argc;
char **argv;

{
    struct img *imgp;
    int xc = 0, yc = 0;
    double tmpzoom, d_coord[2];
    char *coord_str;
    char s_coord[2][40];
    int coord_sys;
    int jsys;
    double equinox;


    if (argc == 1)
    {
	if (graphics)
	    trackball_zoom();
    }
    else
    {
	if (!set_current_window())
	{
	    error1("no images currently being displayed");
	    return;
	}

	if (argc > 4)
	{
	    error1("wrong number of arguments to ZP command");
	    return;
	}


	/* process the zoom factor if given */
	if ((argc == 2) || (argc == 4))
	{
	    tmpzoom = atof(*(++argv));
	    if (tmpzoom == 0.0)
	    {
		error1("bad zoom value");
		return;
	    }
	    if (tmpzoom > 1.0)
	    {
		tmpzoom = Round(tmpzoom);
	    }
	    else
	    {
		tmpzoom = 1.0 / Round(1/tmpzoom);
	    }
	}

	/* process the coordinates if given */
	if (argc >= 3)
	{
	    strcpy(s_coord[0], *(++argv));
	    strcpy(s_coord[1], *(++argv));

	    imgp = sky_to_image(s_coord, d_coord);
	    dIM_to_SC(d_coord);
	    
	    if (imgp == NULL)
	    {
		error1("coordinates not on any visible image");
		return;
	    }
	    xc = Round(d_coord[0]);
	    yc = Round(d_coord[1]);
	}
	do_zoom_pan(argc, &tmpzoom, &xc, &yc, 1);
	if ((server_mode == FALSE) && (!be_quiet))
	    printf("Zoom factor %f\n", win_zoom);
    }

    d_get_center_xy(d_coord);
    dSC_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    if (!be_quiet)
    {
	if (server_mode == FALSE)
	    printf("Centered on %s\n", coord_str);
	else
	{
	    sprintf(server_str, "%f", win_zoom);
	    srv_real("zoom_factor", server_str);
	    srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	}
    }
    fprintf(session, "Zoom factor %f\nCentered on %s\n", win_zoom, coord_str);
}




/********************************************************************** 
**                                                                   ** 
**  VECTOR                                                           ** 
**  draws a vector                                                   ** 
**                                                                   ** 
**********************************************************************/ 
 
void
vector(argc, argv) 
int argc;
char **argv;
{ 
    char s_coord[2][40];
    char s_coord2[2][40];
    char *coord_str;
    double d_coord[2], d_coord2[2];
    int coord_sys;
    int jsys;
    double equinox;

    /* process the coordinates if given */

    if(argc == 5)
    {
	strcpy(s_coord[0], argv[1]);
	strcpy(s_coord[1], argv[2]);
	strcpy(s_coord2[0], argv[3]);
	strcpy(s_coord2[1], argv[4]);

	(void) sky_to_image(s_coord, d_coord);
	(void) sky_to_image(s_coord2, d_coord2);
	dIM_to_SC(d_coord);
	dIM_to_SC(d_coord2);
    }
    else
    {
	sl_vector(d_coord, d_coord2);
    } 
    overlay_setup('g');
    overlay_color(2);
    next_bundle();
    overlay_dmov(d_coord[0], d_coord[1]);          /* draw the vector   */
    overlay_ddraw(d_coord2[0], d_coord2[1]);

    image_setup(plane);

    flush_graphics();

    /* output the info */
    dSC_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    if(server_mode == FALSE)
	printf("vector \nFrom %s", coord_str);
    else
    {
	srv_coord("coord1", coord_sys, s_coord, equinox, jsys);
    }
    fprintf(session, "vector \nFrom %s", coord_str);

    dSC_to_IM(d_coord2);
    coord_str = dimage_to_sky(d_coord2, s_coord, &coord_sys, &equinox, &jsys);
    if (argc == 5)
    {
	strcpy(last_coord[0], s_coord[0]);
	strcpy(last_coord[1], s_coord[1]);
    }
    if(server_mode == FALSE)
	printf("To %s", coord_str);
    else
    {
	srv_coord("coord2", coord_sys, s_coord, equinox, jsys);
    }

    fprintf(session, "To %s", coord_str);
} 




/********************************************************************** 
**                                                                   ** 
**  FOV                                                              ** 
**  reads a description file for an instrument Field-Of-View         ** 
**  and plots it on the current image.                               ** 
**                                                                   ** 
**  syntax:  fov filename [ra dec [rotation_angle]]                  ** 
**********************************************************************/ 

void
fov(argc, argv) 
int argc;
char **argv;
 
{ 
    char s_coord[2][40];
    int coord_sys, use_rotation;
    char *coord_str, *fov_name;
    double rot_angle, pos_angle;
    double x_center, y_center;
    double d_coord[2];
    double angle_bias;
    int jsys;
    double equinox;


    if (argc < 2)
    {
	error1("fov command requires a filename");
	return;
    }

    fov_name = sdata_filename(argv[1]);
    if (fov_name == NULL)
    {
	sprintf(server_str, "file %s not found", argv[1]);
	error1(server_str);
	return;
    }
	
    fov_fd = fopen (fov_name, "r");
    if (fov_fd == NULL)
    {
	sperror(fov_name, errno);
	return;
    }

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    /* process the coordinates if given */

    if(argc >= 4)
    {
	strcpy(s_coord[0], argv[2]);
	strcpy(s_coord[1], argv[3]);
	if (argc > 4)
	    pos_angle = dtr * atof(argv[4]);
	else
	    pos_angle = 0;
	
	if ((argc > 5) && (argv[5][0] == 'r'))
	    use_rotation = 1;
	else
	    use_rotation = 0;

	(void) sky_to_image(s_coord, d_coord);
	dIM_to_SC(d_coord);

	x_center = d_coord[0];
	y_center = d_coord[1];

	/* convert position angle (ref north) to rotation angle (ref up) */
	if (use_rotation)
	    angle_bias = 0.0;   /* don't convert */
	else
	    angle_bias = position_angle_bias(d_coord);

	if (debug)
	    fprintf(debug_file, "fov: angle_bias = %g\n", angle_bias);
	rot_angle = pos_angle - dtr * angle_bias;
	if (rot_angle < 0.0)
	    rot_angle += 2 * M_PI;


	overlay_setup('g');
	overlay_color(2);
	next_bundle();

	fov_plot(x_center, y_center, rot_angle, 0);

	image_setup(plane);

	flush_graphics();
    }
    else
    {
	if (!graphics)
	    return;

	if (argc == 3)
	    rot_angle = dtr * atof(argv[2]);
	else
	    rot_angle = 0;
#ifdef NOTDEF
	/* we have a problem here - we cannot convert from position angle */
	/* to rotation angle, becaause the conversion depends upon the */
	/* x,y position and in this interactive mode he hasn't yet */
	/* specified a position */

	/* convert position angle (ref north) to rotation angle (ref up) */
	get_center_xy(i_coord);
	d_coord[0] = i_coord[0];
	d_coord[1] = i_coord[1];
	angle_bias = position_angle_bias(d_coord);
	if (debug)
	    fprintf(debug_file, "fov: angle_bias = %g\n", angle_bias);
	rot_angle -= dtr * angle_bias;
	if (rot_angle < 0.0)
	    rot_angle += 2 * pi;
#endif /* NOTDEF */


	overlay_setup('g');
	overlay_color(2);

	fov_joy(&rot_angle, d_coord);

	image_setup(plane);
	flush_graphics();

	/* compute position angle (ref north) from rotation angle (ref up) */
	angle_bias = position_angle_bias(d_coord);
	if (debug)
	    fprintf(debug_file, "fov: angle_bias = %15.10f\n", angle_bias);
	pos_angle = rot_angle + dtr * angle_bias;
	if (pos_angle >= 2 * M_PI)
	    pos_angle -= 2 * M_PI;

	dSC_to_IM(d_coord);
	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
	if (argc < 4)
	{
	    strcpy(last_coord[0], s_coord[0]);
	    strcpy(last_coord[1], s_coord[1]);
	}
	if(server_mode == FALSE)
	    printf(
"Fov centered at %s   Position angle = %g degrees  Rotation angle = %g degrees\n", 
		coord_str, pos_angle/dtr, rot_angle/dtr);
	else
	{
	    srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	    sprintf(server_str, "%g", pos_angle/dtr);
	    srv_real("position_angle", server_str);
	    sprintf(server_str, "%g", rot_angle/dtr);
	    srv_real("rotation_angle", server_str);
	}
    } 
    if (!feof(fov_fd))
    {
	error1("fov error:  probable bad format in file");
    }
    fclose(fov_fd);
} 

/*********************************************************************/ 
/*                                                                   */ 
/*  FOV_PLOT takes graphic commands from a file and draws them       */ 
/*  on the current image.  Calls functions which insure that         */ 
/*  the graphics are properly repainted and resized when image       */  
/*  is zoomed.                                                       */  
/*                                                                   */  
/*  skeleton_only = 1 to draw only the skeleton, and do no colors    */ 
/*                = 0 to draw the permanent part                     */ 
/*                                                                   */ 
/*********************************************************************/

void
fov_plot(x_center, y_center, rot_angle, skeleton_only)
double x_center, y_center, rot_angle;
int skeleton_only;
{
    double d_coord[2];
    char command[40], input_line[80];
    char set1[25], color[25];
    double new_x, dradius;
    float x_offset, y_offset, scale, rad, x, y;
    float angle1, angle_extent;
    float x_origin, y_origin;
    int in_skeleton;
    int flip_y;

    if (debug)
	fprintf(debug_file,
	    "fov_plot:  x_center = %g  y_center = %g  rot_angle = %g\n",
	    x_center, y_center, rot_angle);

    save_overlay_color();
    scale = 1;
    x_offset = 0;
    y_offset = 0;
    x_origin = 0;
    y_origin = 0;
    in_skeleton = FALSE;
    flip_y = FALSE;

    rewind(fov_fd);
    for (;;)
    {

	if (fgets(input_line, 80, fov_fd) == NULL)
	    break;
	if (sscanf(input_line, "%s", command) != 1)
	    break;
	if (debug)
	    fprintf(debug_file, "fov_plot:  command = %s\n", command);
	if (strcmp(command, "skeleton") == 0)
	{
	    in_skeleton = TRUE;
	}
	if (strcmp(command, "flip") == 0)
	{
	    flip_y = TRUE;
	}
	else if (strcmp(command, "end") == 0)
	{
	    if (skeleton_only)
		break;            /* we are done with the skeleton */
	    in_skeleton = FALSE;
	}
	else if ((in_skeleton == TRUE) && (!skeleton_only))
	{
	    /*EMPTY*/ 
	    /* do nothing */
	}
	else if (strcmp(command, "scale") == 0)
	{
	    if (sscanf(input_line, "%s%f", command, &scale) != 2)
		break;
	}
	else if (strcmp(command, "origin") == 0)
	{
	    if (sscanf(input_line, "%s%f%f", command, &x_origin, &y_origin) != 3)
		break;
	    x_origin = - x_origin;
	    y_origin = - y_origin;
	    if (flip_y)
		y_origin = -y_origin;  /* make y positive upwards */
	}
	else if (strcmp(command, "offset") == 0)
	{
	    if (sscanf(input_line, "%s%f%f", command, &x_offset, &y_offset) != 3)
		break;
	}
	else if (strcmp(command, "move") == 0)
	{
	    if (sscanf(input_line, "%s%f%f", command, &x, &y) != 3)
		break;
	    if (flip_y)
		y = -y;
	    x = (x + x_offset) * scale + x_origin;
	    y = (y + y_offset) * scale + y_origin;
	    if (rot_angle != 0)
	    {
		new_x = x * cos(rot_angle) + y * sin(rot_angle);
		y =    -x * sin(rot_angle) + y * cos(rot_angle);
		x = new_x;
	    }
	    d_coord[0] = x_center + x / cdelt1;
	    d_coord[1] = y_center + y / cdelt2 * sgn_y;
	    overlay_dmov(d_coord[0], d_coord[1]);
	}
	else if (strcmp(command, "draw") == 0)
	{
	    if (sscanf(input_line, "%s%f%f", command, &x, &y) != 3)
		break;
	    if (flip_y)
		y = -y;
	    x = (x + x_offset) * scale + x_origin;
	    y = (y + y_offset) * scale + y_origin;
	    if (rot_angle != 0)
	    {
		new_x = x * cos(rot_angle) + y * sin(rot_angle);
		y =    -x * sin(rot_angle) + y * cos(rot_angle);
		x = new_x;
	    }
	    d_coord[0] = x_center + x / cdelt1;
	    d_coord[1] = y_center + y / cdelt2 * sgn_y;
	    overlay_ddraw(d_coord[0], d_coord[1]);
	}
	else if (strcmp(command, "arc") == 0)
	{
	    if (sscanf(input_line, "%s%f%f%f%f%f", command, &rad, &x, &y,
		&angle1, &angle_extent) != 6)
		    break;
	    if (flip_y)
		y = -y;
	    x = (x + x_offset) * scale + x_origin;
	    y = (y + y_offset) * scale + y_origin;

	    new_x = x * cos(rot_angle) + y * sin(rot_angle);
	    y =    -x * sin(rot_angle) + y * cos(rot_angle);
	    x = new_x;
	    angle1 = angle1 + 180.0 + rot_angle * rtd;

	    d_coord[0] = x_center + x / cdelt1;
	    d_coord[1] = y_center + y / cdelt2 * sgn_y;
	    overlay_dmov(d_coord[0], d_coord[1]);
	    dradius = fabs(scale * rad / cdelt1);
	    overlay_darc(dradius, angle1, angle_extent);
	}
	else if (strcmp(command, "circle") == 0)
	{
	    if (sscanf(input_line, "%s%f%f%f", command, &rad, &x, &y) != 4)
		break;
	    if (flip_y)
		y = -y;
	    x = (x + x_offset) * scale + x_origin;
	    y = (y + y_offset) * scale + y_origin;
	    if (rot_angle != 0)
	    {
		new_x = x * cos(rot_angle) + y * sin(rot_angle);
		y =    -x * sin(rot_angle) + y * cos(rot_angle);
		x = new_x;
	    }
	    d_coord[0] = x_center + x / cdelt1;
	    d_coord[1] = y_center + y / cdelt2 * sgn_y;
	    overlay_dmov(d_coord[0], d_coord[1]);
	    dradius = fabs(scale * rad / cdelt1);
	    overlay_dcircle(dradius);
	}
	else if (strcmp(command, "set") == 0)
	{
	    if (sscanf(input_line, "%s%s%s", command,  set1, color) != 3)
		break;
	    if (strcmp(set1, "shape") == 0)
		continue;
	    if (strcmp(set1, "color") != 0)
		break;
	    if (!skeleton_only)
	    {
		if (!good_overlay_color(color))
		    break;
		set_overlay_color(color);
		overlay_setup('g');   /* install the new color */
				      /* green is the default */
	    }
	}
    }
    restore_overlay_color();
}
