/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "parse.h"
#include "fits.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>

FILE *cmdfile;
char *more = NULL;
int ads_signal = 0;
int do_alias = TRUE;    /* whether parser should expand aliases */
char save_line[MAXLINE];   /* saved tail of input line when buffer overflows */
extern int debug, server_mode, s1_server_mode, ads_server_mode, control_c;
extern int cmdline_mode;
extern char server_str[];
extern int table_file;
extern FILE *session;
extern FILE *debug_file;
static int takerr = TRUE;
static int *err_cmdc;
static char **err_cmdv;
static char server_string[4096];
static int starting_array;

struct take_info {
    struct take_info *last;
    char *more_save;
    FILE *cmdfile_save;
    int take_param;
    char *param_ptr[10];
    char *next_param;
    char param_buf[MAXLINE];
    char input_line[MAXLINE];
};

static struct take_info *current_take = NULL;

/* function prototypes */
#ifdef _NO_PROTO

static void take_prep();
static void srvprint();
static void srverror();
static void put_param();

#else

static void take_prep(void);
static void srvprint(char *str);
static void srverror(char *str);
static void put_param(char *str);

#endif /* _NO_PROTO */


int
parse_fits(p)
char *p;
{
    if (memcmp (p, "SIMPLE  ", 8) == 0)
	return (SIMPLE);

    if (memcmp (p, "BITPIX  ", 8) == 0)
	return (BITPIX);

    if (memcmp (p, "NAXIS   ", 8) == 0)
	return (NAXIS);

    if (memcmp (p, "NAXIS1  ", 8) == 0)
	return (NAXIS1);

    if (memcmp (p, "NAXIS2  ", 8) == 0)
	return (NAXIS2);

    if (memcmp (p, "NAXIS3  ", 8) == 0)
	return (NAXIS3);

    if (memcmp (p, "BSCALE  ", 8) == 0)
	return (BSCALE);

    if (memcmp (p, "BZERO   ", 8) == 0)
	return (BZERO);

    if (memcmp (p, "BUNIT   ", 8) == 0)
	return (BUNIT);

    if (memcmp (p, "BLANK   ", 8) == 0)
	return (BLANK);

    if (memcmp (p, "CRVAL1  ", 8) == 0)
	return (CRVAL1);

    if (memcmp (p, "CRPIX1  ", 8) == 0)
	return (CRPIX1);

    if (memcmp (p, "CTYPE1  ", 8) == 0)
	return (CTYPE1);

    if (memcmp (p, "CDELT1  ", 8) == 0)
	return (CDELT1);

    if (memcmp (p, "CRVAL2  ", 8) == 0)
	return (CRVAL2);

    if (memcmp (p, "CRPIX2  ", 8) == 0)
	return (CRPIX2);

    if (memcmp (p, "CTYPE2  ", 8) == 0)
	return (CTYPE2);

    if (memcmp (p, "CDELT2  ", 8) == 0)
	return (CDELT2);

    if (memcmp (p, "CROTA2  ", 8) == 0)
	return (CROTA2);

    if (memcmp (p, "CRVAL3  ", 8) == 0)
	return (CRVAL3);

    if (memcmp (p, "CTYPE3  ", 8) == 0)
	return (CTYPE3);

    if (memcmp (p, "DATAMAX ", 8) == 0)
	return (DATAMAX);

    if (memcmp (p, "DATAMIN ", 8) == 0)
	return (DATAMIN);

    if (memcmp (p, "TELESCOP", 8) == 0)
	return (TELESCOP);

    if (memcmp (p, "INSTRUME", 8) == 0)
	return (INSTRUME);

    if (memcmp (p, "OBJECT  ", 8) == 0)
	return (OBJECT);

    if (memcmp (p, "EPOCH   ", 8) == 0)
	return (EPOCH);

    if (memcmp (p, "EQUINOX ", 8) == 0)
	return (EQUINOX);

    if (memcmp (p, "PROJTYPE", 8) == 0)
	return (PROJTYPE);

    if (memcmp (p, "DSKYGRID", 8) == 0)
	return (DSKYGRID);

    if (memcmp (p, "OBSERVER", 8) == 0)
	return (OBSERVER);

    if (memcmp (p, "FILENAME", 8) == 0)
	return (FILENAME);

    if (memcmp (p, "COMMENT ", 8) == 0)
	return (COMMENT);

    if (memcmp (p, "PIXTYPE ", 8) == 0)
	return (PIXTYPE);

    if (memcmp (p, "IMHFILE ", 8) == 0)
	return (IMHFILE);

    if (memcmp (p, "PIXFILE ", 8) == 0)
	return (PIXFILE);

    if (memcmp (p, "END     ", 8) == 0)
	return (END);

    if (memcmp (p, "GNOMON", 6) == 0)   /* allow aberration for Do Kester */
	return (GNOMONIC);

    if (memcmp (p, "LAMBECYL", 8) == 0)
	return (LAMBECYL);

    if (memcmp (p, "AITOFF", 6) == 0)
	return (AITOFF);

    if (memcmp (p, "ORTHOGRA", 8) == 0)
	return (ORTHOGRAPHIC);

    if (memcmp (p, "NCP", 3) == 0)
	return (NCP);

    /* now look for Digital Sky Survey plate solution cofficients keywords */
    if (memcmp (p, "PLTRAH", 6) == 0)
	return (PLTRAH);

    if (memcmp (p, "PLTRAM", 6) == 0)
	return (PLTRAM);

    if (memcmp (p, "PLTRAS", 6) == 0)
	return (PLTRAS);

    if (memcmp (p, "PLTDECSN", 8) == 0)
	return (PLTDECSN);

    if (memcmp (p, "PLTDECD", 7) == 0)
	return (PLTDECD);

    if (memcmp (p, "PLTDECM", 7) == 0)
	return (PLTDECM);

    if (memcmp (p, "PLTDECS", 7) == 0)
	return (PLTDECS);

    if (memcmp (p, "CNPIX1", 6) == 0)
	return (CNPIX1);

    if (memcmp (p, "CNPIX2", 6) == 0)
	return (CNPIX2);

    if (memcmp (p, "PLTSCALE", 8) == 0)
	return (PLTSCALE);

    if (memcmp (p, "XPIXELSZ", 8) == 0)
	return (XPIXELSZ);

    if (memcmp (p, "YPIXELSZ", 8) == 0)
	return (YPIXELSZ);

    if (memcmp (p, "PPO1", 4) == 0)
	return (PPO1);

    if (memcmp (p, "PPO2", 4) == 0)
	return (PPO2);

    if (memcmp (p, "PPO3", 4) == 0)
	return (PPO3);

    if (memcmp (p, "PPO4", 4) == 0)
	return (PPO4);

    if (memcmp (p, "PPO5", 4) == 0)
	return (PPO5);

    if (memcmp (p, "PPO6", 4) == 0)
	return (PPO6);

    if (memcmp (p, "AMDX10", 6) == 0)
	return (AMDX10);

    if (memcmp (p, "AMDX11", 6) == 0)
	return (AMDX11);

    if (memcmp (p, "AMDX12", 6) == 0)
	return (AMDX12);

    if (memcmp (p, "AMDX13", 6) == 0)
	return (AMDX13);

    if (memcmp (p, "AMDX14", 6) == 0)
	return (AMDX14);

    if (memcmp (p, "AMDX15", 6) == 0)
	return (AMDX15);

    if (memcmp (p, "AMDX16", 6) == 0)
	return (AMDX16);

    if (memcmp (p, "AMDX17", 6) == 0)
	return (AMDX17);

    if (memcmp (p, "AMDX18", 6) == 0)
	return (AMDX18);

    if (memcmp (p, "AMDX19", 6) == 0)
	return (AMDX19);

    if (memcmp (p, "AMDX20", 6) == 0)
	return (AMDX20);

    if (memcmp (p, "AMDX1", 5) == 0)
	return (AMDX1);

    if (memcmp (p, "AMDX2", 5) == 0)
	return (AMDX2);

    if (memcmp (p, "AMDX3", 5) == 0)
	return (AMDX3);

    if (memcmp (p, "AMDX4", 5) == 0)
	return (AMDX4);

    if (memcmp (p, "AMDX5", 5) == 0)
	return (AMDX5);

    if (memcmp (p, "AMDX6", 5) == 0)
	return (AMDX6);

    if (memcmp (p, "AMDX7", 5) == 0)
	return (AMDX7);

    if (memcmp (p, "AMDX8", 5) == 0)
	return (AMDX8);

    if (memcmp (p, "AMDX9", 5) == 0)
	return (AMDX9);

    if (memcmp (p, "AMDY10", 6) == 0)
	return (AMDY10);

    if (memcmp (p, "AMDY11", 6) == 0)
	return (AMDY11);

    if (memcmp (p, "AMDY12", 6) == 0)
	return (AMDY12);

    if (memcmp (p, "AMDY13", 6) == 0)
	return (AMDY13);

    if (memcmp (p, "AMDY14", 6) == 0)
	return (AMDY14);

    if (memcmp (p, "AMDY15", 6) == 0)
	return (AMDY15);

    if (memcmp (p, "AMDY16", 6) == 0)
	return (AMDY16);

    if (memcmp (p, "AMDY17", 6) == 0)
	return (AMDY17);

    if (memcmp (p, "AMDY18", 6) == 0)
	return (AMDY18);

    if (memcmp (p, "AMDY19", 6) == 0)
	return (AMDY19);

    if (memcmp (p, "AMDY20", 6) == 0)
	return (AMDY20);

    if (memcmp (p, "AMDY1", 5) == 0)
	return (AMDY1);

    if (memcmp (p, "AMDY2", 5) == 0)
	return (AMDY2);

    if (memcmp (p, "AMDY3", 5) == 0)
	return (AMDY3);

    if (memcmp (p, "AMDY4", 5) == 0)
	return (AMDY4);

    if (memcmp (p, "AMDY5", 5) == 0)
	return (AMDY5);

    if (memcmp (p, "AMDY6", 5) == 0)
	return (AMDY6);

    if (memcmp (p, "AMDY7", 5) == 0)
	return (AMDY7);

    if (memcmp (p, "AMDY8", 5) == 0)
	return (AMDY8);

    if (memcmp (p, "AMDY9", 5) == 0)
	return (AMDY9);

    if ((memcmp (p, "CD1_1", 5) == 0) || (memcmp (p, "CD001001", 5) == 0))
	return (CD1_1);

    if ((memcmp (p, "CD1_2", 5) == 0) || (memcmp (p, "CD001002", 5) == 0))
	return (CD1_2);

    if ((memcmp (p, "CD2_1", 5) == 0) || (memcmp (p, "CD002001", 5) == 0))
	return (CD2_1);

    if ((memcmp (p, "CD2_2", 5) == 0) || (memcmp (p, "CD002002", 5) == 0))
	return (CD2_2);

    return (UNRECOGNIZED);
}



int
getcmd (cmdc, cmdv)

int    *cmdc;	    /* output argument -  count of command parameters      */
char   *cmdv[];	    /* output argument - array of strings (cmd parameters) */

{
    char   *p;
    char   *ptmp;
    char *first_newline;
    int     i, len;
    static char anin[MAXLINE];
    static char an[512];
    int in_quotes;


 /* Get the string to be parsed */

    if (more == NULL)
    {

	if (table_file)
	{
	    if (control_c)
	    {
		if(server_mode == FALSE)
		    printf("\ntable file interrupted by control c\n");
		fprintf(session, "\ntable file interrupted by control c\n");
		table_close();
		more = NULL;
		*cmdc = 0;
		return(NIL);
	    }
	    get_table_line(anin);
	    if (anin[0] == '\0')
	    {
		*cmdc = 0;
		return(NIL);
	    }
	}
        else if (cmdfile == stdin)
	{
	    first_newline = strchr(save_line, '\n');
	    if (first_newline == NULL)
	    {
		/* we do not have a complete command in save_line */
		if (save_line[0] == '\0')    /* if no remaining input */
		{
		    if(server_mode == FALSE)
			printf ("> ");
		}
		fflush(stdout);
		main_command(anin);
	    }
	    else
	    {
		/* we do have a complete command in save_line -- use it */
		strcpy(anin, save_line);
		first_newline = strchr(anin, '\n');
		strcpy(save_line, first_newline+1);  /* save the tail */
		*first_newline = '\0';  /* change newline to null */
	    }
	    do_sync();
	}
	else
	{
#ifdef NOTDEF
	    save_line[0] = '\0';
#endif /* NOTDEF */
	    if (control_c)
	    {
		if(server_mode == FALSE)
		    printf("\ntake file interrupted by control c\n");
		fprintf(session, "\ntake file interrupted by control c\n");
		while (cmdfile != stdin) 
		{
		    close_take();
		}
		more = NULL;
		*cmdc = 0;
		return(NIL);
	    }

	    if (fgets (anin, MAXLINE, cmdfile) == NULL)
	    {
		close_take();
		*cmdc = 0;
		return(NIL);
	    }
	    do_sync();
	}

	if (anin[0] == '#')
	{
	    fprintf (session, ">%s\n", anin);
	    *cmdc = 1;
	    cmdv[0] = "#";
	    return(COMMENT_LINE);
	}

	len = strlen (anin);	/* fgets returns newline followed by null */
	if (anin[len-1] == '\n')
	    anin[len-1] = '\0';


	/* extract Tektronix sequences */
	while (strncmp (anin, "\033[?38", 5) == 0)
	    strcpy(anin, tek_sequence(anin));

	if ((cmd(anin) == AL) || (do_alias == FALSE) || (server_mode ==TRUE))
	{
	    /* dont expand an alias command or 1st line in .slogin, */
	    /* or any line in server mode.                          */
	    strcpy(an, anin);
	    if(debug)
		fprintf(debug_file,
		    "skipping alias expansion:   input =%s=  length = %lu\n", 
		    an, strlen(an));
	}
	else
	{
	    expand_alias (anin, an);
	}

	fprintf (session, ">%s\n", an);

	p = an;
    }
    else
    {
	p = more;
	do_sync();
    }


 /* Scan the command for semicolons and newlines */

    len = strlen (p);

    in_quotes = FALSE;

    for (i = 0; i < len; i++)
    {
	if (p[i] == '"')
	    in_quotes = !in_quotes;    /* toggle quoting on/off         */

	if (in_quotes == FALSE &&      /* if this isn't a quoted string */
		p[i] == ';')           /* change semicolon to null      */
	    p[i] = '\0';

	if (p[i] == '\n')	           /* if this is a newline */
		p[i] = '\0';

	if (p[i] == '\0')	           /* if this is a null */
	    break;		           /* stop the scan     */
    }


 /* Check to see if this is the end of the input line */

    if (i == len)
	more = NULL;
    else
	more = p + i + 1;


 /* Collect pointers to each argument for this command */

    ptmp = p;
    *cmdc = 0;
    cmdv[0] = "* nil";

    while (*ptmp == ' ')	/* discard spaces */
	ptmp++;

    while (*ptmp != '\0')	/* while some of string is left */
    {

	if (*ptmp == '"')	/* If this is a quote */
	{
	    ptmp++;     /* move past quote */
	    cmdv[*cmdc] = ptmp;	/* pointer to this arg  */
	    (*cmdc)++;
	    /*  search for ending quote */
	    while (*ptmp != '"' && *ptmp != '\0')
		ptmp++;

	    if (*ptmp == '"')
		{
		*ptmp = ' ';   /* change quote to a space */
		}
	}
	else
	{
	cmdv[*cmdc] = ptmp;	/* pointer to this arg          */
	(*cmdc)++;
	}
    /* Find the next space */
	while ((*ptmp != ' ') && (*ptmp != '\0'))
	    ptmp++;

	if (*ptmp == ' ')
	    *ptmp++ = '\0';	/* change space to null and advance */

	while (*ptmp == ' ')	/* discard additional spaces */
	    ptmp++;

    }

    /* save cmd line for error() message */
    err_cmdc = cmdc;
    err_cmdv = cmdv;


    return (cmd (cmdv[0]));
}



/***********************************************************************/
/*                                                                    */
/* TAKE                                                               */
/* Opens a take file                                                  */
/*                                                                    */
/*                                                                    */
/**********************************************************************/

void
take (argc, argv)
int     argc;
char   *argv[];
{
    char *tak_nam;
    int i, nextarg;
    FILE *tmp_cmdfile;

    if (argc < 2)
    {
	error1("take error:  need a filename");
	return;
    }

    take_prep();
    if (argc > 2)
    {
	switch(cmd(argv[2]))
	{
	    case ERROR:
		takerr = TRUE;
		nextarg = 3;
		break;
	    case NO_ERROR:
		takerr = FALSE;
		nextarg = 3;
		break;
	    default:
		nextarg = 2;
	}

	/* stash parameters */
	for (i = nextarg; i < argc; i++)
	{
	    current_take->take_param++;
	    put_param(argv[i]);
	}
    }

    tak_nam = expand_path(argv[1]);

    tmp_cmdfile = fopen (tak_nam, "r");

    if (tmp_cmdfile == NULL)
    {
	sperror(tak_nam, errno);
    }
    else
    {
	cmdfile = tmp_cmdfile;
    }
}

/* take_prep sets up for a take file, but doesn't open the file */
static void
take_prep()
{
    struct take_info *new_take;


    new_take = (struct take_info *) malloc(sizeof(struct take_info));
    if (new_take == NULL)
    {
	error1("error in take:  cannot malloc");
	return;
    }
    new_take->last = current_take;
    current_take = new_take;

    current_take->more_save = more;
    if (more != NULL)
	strcpy(current_take->input_line, more);
    current_take->cmdfile_save = cmdfile;
    current_take->take_param = 0;
    current_take->next_param = current_take->param_buf;
    more = NULL;
    enable_control_c();
}

void
close_take()
{
    struct take_info *old_take;

    if (cmdfile != stdin)
    {
	fclose(cmdfile);
	cmdfile = current_take->cmdfile_save;
	more = current_take->more_save;
	if (more != NULL)
	    strcpy(more, current_take->input_line);

	old_take = current_take;
	current_take = current_take->last;
	free(old_take);
    }
}


void
get_param(com)
char *com;
{
    int param_nbr;
    char *psave, *p, com_temp[512];

    if (current_take != NULL)
    {
	strcpy(com_temp, com);
	com[0] = '\0';
	psave = com_temp;
	p = com_temp;
	while ((p = strchr(p, '%')))
	{
	    p++;
	    if (isdigit(*p))
	    {
		param_nbr = (*p) - '0';
		if ((param_nbr <= current_take->take_param) && (param_nbr > 0))
		{
		    strncat(com, psave, p - psave - 1);
		    strcat(com, current_take->param_ptr[param_nbr]);
		    p++;
		    psave = p;
		}
	    }
	}
	strcat(com, psave);
    }
}

static void
put_param(str)
char *str;
{
    strcpy(current_take->next_param, str);
    current_take->param_ptr[current_take->take_param] = 
	current_take->next_param;
    current_take->next_param = strchr(current_take->next_param, '\0');
    current_take->next_param++;
}

void
error1(str)
char *str;
{
    int i;

    if (cmdline_mode)
	exit(0);

    if (server_mode == TRUE)
	srverror(str);
    else
	printf("%s\n", str);

    if (takerr == FALSE)  /* if errors are suppressed */
	return;

    if (server_mode == FALSE)
    {
	if ((cmdfile != stdin) || (table_file))
	{
	    printf("offending command:  ");
	    for (i = 0; i < *err_cmdc; i++)
	       printf("%s ", err_cmdv[i]);
	    printf("\n");
	}
	printf ("\7");
	fflush(stdout);
    }

    while (cmdfile != stdin) 
    {
	close_take();
    }
    more = NULL;

    if (table_file)
	table_close();
}


/* print a coordinate and pixel value in server mode */
void
srv_coord_val(name, coord_sys, s_coord, equinox, jsys, value, units)
char *name;
int coord_sys;
char s_coord[2][40];
double equinox;
int jsys;
char *value;
char *units;
{
    if(strlen(server_string) > 3700)
	return;
    if (s1_server_mode)
    {
	srv_real("pixel_value", value);
	srv_string("units", units);
	srv_coord("coord", coord_sys, s_coord, equinox, jsys);
    }
    else
    {
	sprintf(server_string, "%s, %s = [struct pixel_value = %s ", 
	    server_string, name, value);
	srv_string("units", units);
	srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	strcat(server_string, "]");
    }
}

/* print a coordinate in server mode */
void
srv_coord(name, coord_sys, s_coord, equinox, jsys)
char *name;
int coord_sys;
char s_coord[2][40];
double equinox;
int jsys;
{
    char result[256];
    char equinox_str[10];

    if (s1_server_mode)
    {
	srv_string("coord_sys", coord_name(coord_sys));
	srv_real("x", s_coord[0]);
	srv_real("y", s_coord[1]);
    }
    else
    {
	if( (int) strlen(server_string) > 4000)
	    return;

	if ((coord_sys == EQ) || (coord_sys == EC))
	{
	    sprintf(equinox_str, "%c%6.1f", 
		(jsys == 1 || jsys == 3)?'B':'J',  equinox);
	    sprintf(result,
	"[struct coord_sys = \"%s\", equinox = \"%s\", x = \"%s\", y = \"%s\"]", 
		coord_name(coord_sys), equinox_str, s_coord[0], s_coord[1]);
	}
	else
	{

	    sprintf(result,
		"[struct coord_sys = \"%s\", x = \"%s\", y = \"%s\"]", 
		coord_name(coord_sys), s_coord[0], s_coord[1]);
	}
	if (name == NULL)
	{
	    sprintf(server_string, "%s%s %s", server_string,
		starting_array?"":",", result);
	    starting_array = 0;
	}
	else
	    sprintf(server_string, "%s, %s = %s", server_string, name, result);
    }
}

void srv_start_array(name)
char *name;
{
    if (!s1_server_mode)
    {
	sprintf(server_string, "%s, %s_list = [struct %s = [array ", 
	    server_string, name, name);
	starting_array = 1;
    }
}

void srv_end_array(nbr_elts)
int nbr_elts;
{
    if (!s1_server_mode)
    {
	sprintf(server_string, "%s], number = %d]", server_string, nbr_elts);
    }
}

void
srv_real(keyword, str)
char *keyword, *str;
{
    if( (int) strlen(server_string) > 4000)
	return;
    if (s1_server_mode)
	sprintf(server_string, "%s%s = %s\n", server_string, keyword, str);
    else
	sprintf(server_string, "%s, %s = %s", server_string, keyword, str);
}

void
srv_string(keyword, str)
char *keyword, *str;
{
    if( (int) strlen(server_string) > 4000)
	return;

    if (s1_server_mode)
	sprintf(server_string, "%s%s = \"%s\"\n", server_string, keyword, str);
    else
	sprintf(server_string, "%s, %s = \"%s\"", server_string, keyword, str);
}


static void
srvprint(str)
char *str;
{
    int i, len, slen;

#ifdef NOTDEF
    if ((cmdfile != stdin) || (table_file))
	return;
#endif /* NOTDEF */

    len = strlen(server_string);
    if (len > 4000)
	return;

    if (s1_server_mode)
	strcat(server_string, "\"");
    else
	strcat(server_string, ",\"");
    len = strlen(server_string);

    slen = strlen(str);
    for(i=0; i<slen; ++i)
    {
	if(str[i] == '"')
	{
	    server_string[len] = '\\';
	    ++len;
	}

	server_string[len] = str[i];
	++len;
    }

    server_string[len] = '"';
    ++len;

    if (s1_server_mode)
    {
	server_string[len] = '\n';
	++len;
    }

    server_string[len] = '\0';
}

void
srvinit()
{
    if(server_mode == TRUE)
    {
	if(s1_server_mode)
	    strcpy(server_string, "\"OK\"\n");
	else
	    strcpy(server_string, "stat=\"OK\"");
    }
}

#ifdef NOTDEF
void
srvsignal()
{
    /* strcpy(server_string, "SIGNAL: [struct stat=\"OK\""); */
    strcpy(server_string, "[struct stat=\"OK\"");
}
#endif /* NOTDEF */

void
srvflush(str, want_command, want_end)
char *str;
int want_command;  /* flag that "command" keyword is wanted */
int want_end;      /* flag that "end" keyword is wanted */
{
    if ((table_file) || (cmdline_mode))
	return;

    if (server_mode == TRUE)
    {
	if (s1_server_mode)
	{
	    if (server_string[0] != '\0')
	    {
		if (want_command)
		    printf("command = \"%s\" %s", str, server_string);
		else
		    printf("%s", server_string);
	    }
	    if (want_end)
		printf("end = \"%s\"\n", str);
	    fflush(stdout);
	    server_string[0] = '\0';
	}
	else 
	{
	    if (ads_signal)
	    {
		printf("SIGNAL: %s", server_string);
		ads_signal = 0;
	    }
	    else
		printf("[struct cmd = \"%s\", %s]", str, server_string);

	    printf("\n");
	    fflush(stdout);
	}
    }
}

static void
srverror(str)
char *str;
{
    if (s1_server_mode)
    {
	strcpy(server_string, "\"ERROR\" ");
	srvprint(str);
    }
    else
    {
	strcpy(server_string, "stat=\"ERROR\"");
	srv_string("msg", str);
    }
}

void
error_na(str)
char *str;
{
    if (s1_server_mode)
    {
	strcpy(server_string, "\"NA\" ");
	srvprint(str);
    }
    else
    {
#ifdef NOTDEF
	strcpy(server_string, "[struct stat=\"ERROR\"");
	srv_string("msg", str);
#endif /* NOTDEF */
	srverror(str);
    }
}

void
sperror(str, this_errno)
char *str;
int this_errno;
{
    char *error_msg;

    error_msg = strerror(this_errno);
    if (error_msg == NULL)
	sprintf(server_str, "%s:  Unknown error", str);
    else
	sprintf(server_str, "%s:  %s", str, error_msg);
    error1(server_str);
}

#ifdef NO_STRERROR
char *strerror(errnbr)
int errnbr;
{
    extern char *sys_errlist[];
    extern int sys_nerr;

    if (errnbr < sys_nerr)
	return(sys_errlist[errnbr]);
    else
	return("Unknown error");
}
#endif /* NO_STRERROR */

int
cmd(p)
char   *p;
{
    if ((strcmp (p, "y") == 0) || (strcmp (p, "yes") == 0))
	return (YES);

    if ((strcmp (p, "n") == 0) || (strcmp (p, "no") == 0))
	return (NO);

    if (strncmp (p, "al", 2) == 0)
	return (AL);

    if ((strcmp (p, "an") == 0) || (strcmp (p, "annotate") == 0))
	return (AN);

    if (strcmp (p, "asterisk") == 0)
	return (ASTERISK);

    if ((strcmp (p, "ba") == 0) || (strncmp(p, "background", 10) == 0))
	return (BL);

    if (strcmp (p, "black") == 0)
	return (BLACK);

    if (strcmp (p, "blank_color") == 0)
	return (BLANK_COLOR);

    if (strcmp (p, "blank") == 0)
	return (BLANK);

    if (strcmp (p, "blinn") == 0)
	return (BLINN);

    if (strcmp (p, "blinncolor") == 0)
	return (BLINNCOLOR);

    if (strcmp (p, "blue") == 0)
	return (BLUE);

    if (strcmp (p, "border") == 0)
	return (BORDER);

    if (strcmp (p, "box") == 0)
	return (BOX);

    if (strcmp (p, "both") == 0)
	return (BOTH);

    if (strncmp (p, "bo", 2) == 0)
	return (BOTTOM);

    if ((strcmp (p, "ca") == 0) || (strcmp (p, "change_area") == 0))
	return (CA);

    if ((strcmp (p, "cd") == 0) || (strcmp (p, "change_directory") == 0))
	return (CD);

    if ((strcmp (p, "ci") == 0) || (strcmp (p, "current_image") == 0))
	return (CIMG);

    if (strcmp (p, "circle") == 0)
	return (CIRCLE);

    if ((strcmp (p, "cl") == 0) || (strcmp (p, "clip") == 0))
	return (CL);

    if ((strcmp (p, "cm") == 0) || (strcmp (p, "current_map") == 0))
	return (CMAP);

    if (strcmp (p, "color") == 0)
	return (COLOR);

    if (strcmp (p, "conserve_mem") == 0)
	return (CONSERVE_MEM);

    if ((strcmp (p, "co") == 0) || (strncmp (p, "contour", 7) == 0))
	return (CONTOUR);

    if (strcmp (p, "create") == 0)
	return (CREATE);

    if ((strcmp (p, "cr") == 0) || (strcmp (p, "crop") == 0))
	return (CROP);

    if (strcmp (p, "cross") == 0)
	return (CROSS);

    if ((strcmp (p, "cs") == 0) || (strcmp (p, "coord_system") == 0))
	return (CS);

    if ((strcmp (p, "ct") == 0) || (strcmp (p, "color_table") == 0))
	return (CT);

    if ((strcmp (p, "cu") == 0) || (strcmp (p, "cutoff") == 0))
	return (CU);

    if ((strcmp (p, "da") == 0) || (strcmp (p, "define_area") == 0))
	return (DA);

    if ((strcmp (p, "df") == 0) || (strcmp (p, "delete_file") == 0))
	return (DF);

    if ((strcmp (p, "de") == 0) || (strcmp (p, "debug") == 0))
	return (DE);

    if (strcmp (p, "diamond") == 0)
	return (DIAMOND);

    if (strncmp (p, "di", 2) == 0)
	return (DIRR);

    if (strcmp (p, "dot") == 0)
	return (DOT);

    if ((strcmp (p, "dr") == 0) || (strcmp (p, "disk_read") == 0))
	return (DR);

    if (strcmp (p, "echo") == 0)
	return (ECHO1);

    if (strncmp (p, "ec", 2) == 0)
	return (EC);

    if (strcmp (p, "el") == 0)
	return (ELLIPSE);

    if (strcmp (p, "end") == 0)
	return (UNIX);

    if (strncmp (p, "eq", 2) == 0)
	return (EQ);

    if ((strcmp (p, "er") == 0) || (strcmp (p, "erase") == 0))
	return (ER);

    if (strncmp (p, "error", 5) == 0)
	return (ERROR);

    if ((strcmp (p, "ex") == 0) || (strcmp (p, "examine") == 0))
	return (EX);

    if (strcmp (p, "exit") == 0)
	return (UNIX);

    if ((strcmp (p, "fi") == 0) || (strcmp (p, "file") == 0))
	return (FI);

    if (strncmp (p, "fl", 2) == 0)
	return (FL);

    if (strcmp (p, "fullimage") == 0)
	return (FULLIMAGE);

    if (strcmp (p, "fov") == 0)
	return (FOV);

    if ((strcmp (p, "fp") == 0) || (strcmp (p, "find_plate") == 0))
	return (FP);

    if (strncmp (p, "fr", 2) == 0)
	return (FR);

    if ((strcmp (p, "ga") == 0) || (strcmp(p, "gal") == 0) || 
	(strcmp(p, "galactic") == 0))
	return (GA);

    if ((strcmp (p, "gex") == 0) || (strcmp(p, "graphics_examine") == 0))
	return (GEX);

    if (strcmp (p, "gif") == 0)
	return (GIF);

    if ((strcmp (p, "gpick") == 0) || (strcmp(p, "gpi") == 0) || 
	(strcmp(p, "graphics_pick") == 0))
	return (GPICK);

    if ((strcmp (p, "gpoint") == 0) || (strcmp(p, "graphics_point") == 0))
	return (GPOINT);

    if (strcmp (p, "green") == 0)
	return (GREEN);

    if (strcmp (p, "grid") == 0)
	return (GRID);

    if (strcmp (p, "help") == 0)
	return (MAN);

    if ((strcmp (p, "he") == 0) || (strcmp (p, "header") == 0))
	return (HE);

    if ((strcmp (p, "ho") == 0) || (strcmp (p, "histogram_original") == 0))
	return (HO);

    if (strncmp (p, "hi", 2) == 0)
	return (HI);

    if ((strcmp (p, "hs") == 0) || (strcmp (p, "hist_spec") == 0))
	return (HIST_SPEC);

    if ((strcmp (p, "im") == 0) || (strcmp(p, "image") == 0))
	return (IM);

    if (strcmp (p, "imio") == 0)
	return (IMIO);

    if (strcmp (p, "input") == 0)
	return (INPUT);

    if (strcmp (p, "ipl") == 0)
	return (IPL);

    if (strcmp (p, "iraf") == 0)
	return (IRAF);

    if (strcmp (p, "jpeg") == 0)
	return (JPEG);

    if (strncmp (p, "le", 2) == 0)
	return (LEFT);

    if (strncmp (p, "lin", 3) == 0)
	return (LIN);

    if (strcmp (p, "loglog") == 0)
	return (LOGLOG);

    if (strncmp (p, "log", 3) == 0)
	return (LOG);

    if (strcmp (p, "long") == 0)
	return (LONG);

    if (strcmp (p, "lord") == 0)
	return (LORD);

    if (strncmp (p, "ls", 2) == 0)
	return (DIRR);

    if (strncmp (p, "mag", 3) == 0)
	return (MAG);

    if ((strcmp (p, "ma") == 0) || (strcmp(p, "map") == 0))
	return (MA);

    if (strcmp (p, "mark") == 0)
	return (MARK);

    if (strcmp (p, "md") == 0 || (strcmp(p, "memdump") == 0))
	return (MD);

    if (strcmp (p, "mouse") == 0)
	return (MOUSE);

    if (strcmp (p, "movie") == 0)
	return (MOVIE);

    if ((strcmp (p, "movie_prep") == 0) || (strcmp (p, "mp") == 0))
	return (MOVIE_PREP);

    if (strcmp (p, "mo") == 0 || (strcmp(p, "mode") == 0))
	return (MO);

    if (strncmp (p, "noer", 4) == 0)
	return (NO_ERROR);

    if (strcmp (p, "nost") == 0)
	return (NOST);

    if (strcmp (p, "nozp") == 0)
	return (NOZP);

    if (strncmp (p, "of", 2) == 0)
	return (OF);

    if (strncmp (p, "on", 2) == 0)
	return (ONN);

    if (strncmp (p, "or", 2) == 0)
	return (OR);

    if (strcmp (p, "output") == 0)
	return (OUTPUT);

    if (strcmp (p, "pause") == 0)
	return (PAUSE);

    if ((strcmp (p, "pa") == 0) || (strcmp (p, "paint") == 0))
	return (DR);

    if ((strcmp (p, "pb") == 0) || (strcmp (p, "paint_blank") == 0))
	return (PAINT_BLANK);

    if ((strcmp (p, "ph") == 0) || (strcmp(p, "print_history") == 0))
	return (PH);

    if (strncmp (p, "pix", 3) == 0)
	return (PIXCOORD);

    if (strncmp (p, "pi", 2) == 0)
	return (PICK);

    if ((strcmp (p, "plane") == 0) || (strcmp(p, "pl") == 0))
	return (PLANE);

    if (strcmp (p, "point") == 0)
	return (POINT);

    if (strncmp (p, "post", 4) == 0)
	return (POSTSCRIPT);

    if (strcmp (p, "ppm") == 0)
	return (PPM);

    if (strncmp (p, "ppmbin", 6) == 0)
	return (PPMBIN);

    if (strcmp (p, "projtype") == 0)
	return (PROJTYPE);

    if ((strcmp (p, "pwd") == 0) || (strcmp (p, "print_working_dir") == 0))
	return (PWD);

    if (strcmp (p, "quit") == 0)
	return (UNIX);

    if (strncmp (p, "ra", 2) == 0)
	return (RA);

    if (strcmp (p, "red") == 0)
	return (RED);

    if ((strcmp (p, "rep") == 0) || (strcmp (p, "replicate") == 0))
	return (RE);

    if (strcmp (p, "reread") == 0)
	return (REREAD);

    if ((strcmp (p, "rew") == 0) || (strcmp (p, "rewind") == 0))
	return (REW);

    if (strncmp (p, "ri", 2) == 0)
	return (RIGHT);

    if (strcmp (p, "S:") == 0)
	return (SMONGO);

    if (strcmp (p, "s.help") == 0)
	return (MAN);

    if ((strcmp (p, "sc") == 0) || (strcmp (p, "screen") == 0) ||
	(strcmp (p, "scatter_plot") == 0))
	return (SC);

    if ((strcmp (p, "sd") == 0) || (strcmp (p, "screen_dump") == 0))
	return (SD);

    if (strcmp (p, "set") == 0)
	return (SET);

    if (strncmp (p, "sex", 3) == 0)
	return (SEXIGESIMAL);

    if ((strcmp (p, "sh") == 0) || (strncmp (p, "set_history", 11) == 0) || (strcmp(p,"short") == 0))
	return (SH);

    if ((strcmp(p, "sb") == 0) ||(strcmp (p, "sky_box") == 0) || (strcmp(p, "skybox")) == 0)
	return (SKY_BOX);

    if ((strcmp (p, "sl") == 0) || (strcmp (p, "slice") == 0))
	return (SL);

    if (strcmp (p, "sleep") == 0)
	return (SLEEP);

    if (strcmp (p, "ssplit") == 0)
	return (SPLIT);

    if ((strcmp (p, "st") == 0) || (strcmp (p, "stretch") == 0))
	return (ST);

    if ((strcmp (p, "supergalactic") == 0) || (strncmp (p, "sgal", 4) == 0))
	return (SGAL);

    if (strcmp (p, "sync") == 0)
	return (SYNC);

    if ((strcmp (p, "ta") == 0) || (strcmp (p, "take") == 0))
	return (TA);

    if (strncmp (p, "table", 5) == 0)
	return (TABLE);

    if (strcmp (p, "tek") == 0)
	return (TEK);

    if (strncmp (p, "test", 4) == 0)
	return (TEST);

    if (strncmp (p, "to", 2) == 0)
	return (TOP);

    if ((strcmp (p, "tr") == 0) || (strcmp (p, "tape_read") == 0))
	return (TR);

    if (strcmp (p, "triangle") == 0)
	return (TRIANGLE);

    if ((strcmp (p, "tw") == 0) || (strcmp (p, "tape_write") == 0))
	return (TW);

    if (strcmp (p, "u16") == 0)
	return (U16);

    if (strcmp (p, "unix") == 0)
	return (UNIX);

    if ((strcmp (p, "ve") == 0) || (strcmp(p, "vector") == 0))
	return (VE);

    if (strcmp (p, "version") == 0)
	return (VERSION);

    if (strcmp (p, "white") == 0)
	return (WHITE);

    if ((strcmp (p, "we") == 0) || (strcmp (p, "wedge") == 0))
	return (WE);

    if (strncmp (p, "wr", 2) == 0)
	return (WRITE);

    if (strcmp (p, "x") == 0)
	return (X);

    if (strcmp (p, "xy") == 0)
	return (XY);

    if (strncmp (p, "zap", 3) == 0)
	return (ZAP);

    if ((strcmp (p, "zp") == 0) || (strcmp (p, "zoom_pan") == 0))
	return (ZP);

    if ((strcmp (p, "* nil") == 0))
	return (NIL);

    if ((strncmp (p, "#", 1) == 0) || (strcmp(p, "comment") == 0))
	return (NIL);

    if (strncmp (p, "!", 1) == 0)
	return (BANG);

    if (strncmp (p, "?", 1) == 0)
	return (QMARK);

    return (NOTFOUND);
}

/********************************************************************** 
**                                                                   ** 
**  GETINT                                                           ** 
**  Converts a string to an integer with error checking.             ** 
**                                                                   ** 
**  Returns TRUE if integer is clean.                                **
**                                                                   ** 
**  Returns FALSE if there is any funny trash in the string          ** 
**                (also prints error message)                        ** 
**                                                                   ** 
**********************************************************************/ 
 
int getint(str, iret)
char *str;
int *iret;
{
    if (isint(str))
    {
	*iret = atoi(str);
	return(TRUE);
    }
    else
    {
	error1("unexpected character in integer");
	return (FALSE);
    }
}
/********************************************************************** 
**                                                                   ** 
**  ISINT                                                            ** 
**  Checks a string to see if it can be converted to an integer      ** 
**                                                                   ** 
**  Returns TRUE if integer is clean.                                **
**                                                                   ** 
**  Returns FALSE if there is any funny trash in the string          ** 
**                                                                   ** 
**********************************************************************/ 
 
int
isint(str)
char *str;
{
    char *p;

    p = str;
    while (*p == ' ')                /* discard leading spaces */
	p++;
    if ((*p == '+') || (*p == '-'))  /* accept leading sign */
	p++;
    if (*p == '\0')                  /* require at least one digit */
	return(FALSE);
    while (isdigit(*p))              /* next non-digit must be end-of-string */
	p++;
    if (*p != '\0')
    {
	return (FALSE);
    }
    return(TRUE);
}
/********************************************************************** 
**                                                                   ** 
**  ISNBR                                                            ** 
**  Checks a string to see if it looks like a number                 ** 
**                                                                   ** 
**  Returns TRUE if it looks good.                                   **
**                                                                   ** 
**  Returns FALSE if there it starts with funny stuff.               ** 
**                                                                   ** 
**********************************************************************/ 
 
int
isnbr(str)
char *str;
{
    char *p;

    p = str;
    while (*p == ' ')                /* discard leading spaces */
	p++;
    if ((*p == '+') || (*p == '-'))  /* accept leading sign */
	p++;
    if (*p == '.')		     /* accept leading dot */
	p++;
    if (isdigit(*p))                  /* require at least one digit */
	return(TRUE);
    else
	return(FALSE);
}

/********************************************************************** 
**                                                                   ** 
**  ISDBL                                                            ** 
**  Checks a string to see if it converts to a double                ** 
**                                                                   ** 
**  Returns TRUE if it converts exactly.  Stores value in *dbl_ptr   **
**                                                                   ** 
**  Returns FALSE otherwise.                                         ** 
**                                                                   ** 
**********************************************************************/ 
 
int
isdbl(str, dbl_ptr)
char *str;
double *dbl_ptr;
{
    char *ptr;
    double dbl_temp;

    dbl_temp = strtod(str, &ptr);
    if ((ptr == str) || (*ptr != '\0'))
    {
	/* error converting to float - don't change old value */
	return(FALSE);
    }
    else
    {
	*dbl_ptr = dbl_temp;
	return(TRUE);
    }
}
