/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "range.h"
#include "parse.h"
#include "im_proto.h"
#include "fits.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/file.h>
#include <math.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef NO_FLOAT_H
#include <values.h>
#define DBL_MAX MAXDOUBLE
#define FLT_MAX MAXFLOAT
#else
#include <float.h>
#endif


extern int screen_hist[];

extern int fixed_ct;
extern int debug, JUPhardware, graphics, u16, minx, maxx, maxy, miny, req_mode;
extern int debug_val, mode_used;
extern int minx_next, miny_next, frame_advance, control_c;
extern int server_mode, be_quiet, reread_on;
extern int graphics_window;
extern char bunit[], server_str[];
extern int ho_on;
extern int maptype;    /* projection type */
extern int sgn_y, x_offset, y_offset;
extern double iraf_max, iraf_min, bscale;
extern double lastmin, lastmax;
extern int file_coord_sys, output_coord_sys;

extern int clip_top, clip_bottom, clip_left, clip_right;
extern int pixels, lines, naxis3;
extern int x_orig, y_orig;
extern double blank;
extern double   cdelt1, cdelt2, crpix1, crpix2;
extern double   glat, glong;
extern unsigned int win_width, win_height;
extern int frame, bitpix, reqband, band_offset;
extern char plane, fi_name[], filnam[], screen_hist_filnam[];
extern struct hdrstruct filehdr;
extern struct img *curr_img;
extern FILE *session, *debug_file;
extern double slow, shigh;
extern int samp_disp;   /* pixel displacement from left edge to 1st desired */
extern int line_disp;  /* line displacement from top edge to 1st desired line */
extern int ads_server_mode;
extern struct full_ra min_ra, max_ra;
extern double agra_plon, agra_plat, agra_mlon, agra_mlat, agra_su, agra_sv;
extern char agra_proj[];
extern int hist[];
extern double hist_min, hist_binsiz;

static int fd = 0;
static int *pixbuf = NULL;
static int zp = TRUE;          /* flag to do zoom_pan at start of paint */
static int nost = FALSE;       /* flag to do no stretch at start of paint */
static int blank_window = 0;   /* flag for a window with no image pixels */
static int got_agra_scale = 0; /* flag that we got the scaling from agra */

/* function prototypes */
#ifdef _NO_PROTO

static void paint8();
static void paintu16();
static void paint16();
static void paint32();
static void paintm32();
static void paintm64();
static void eq_tbl();
static void deq_tbl();
static double *hist_spec();

#else

static void paint8(int minx, int maxx, int miny, int maxy, 
    int samp_disp, int line_disp, unsigned char *filptr);
static void paintu16(int minx, int maxx, int miny, int maxy, 
    int samp_disp, int line_disp, unsigned short *filptr);
static void paint16(int minx, int maxx, int miny, int maxy, 
    int samp_disp, int line_disp, short *filptr);
static void paint32(int minx, int maxx, int miny, int maxy, 
    int samp_disp, int line_disp, int *filptr);
static void paintm32(int minx, int maxx, int miny, int maxy, 
    int samp_disp, int line_disp, float *filptr);
static void paintm64(int minx, int maxx, int miny, int maxy, 
    int samp_disp, int line_disp, double *filptr);
static void eq_tbl(int tbl[]);
static void deq_tbl(double tbl[]);
static double *hist_spec(void);

#endif /* _NO_PROTO */


void
paint_blank(argc, argv)
int argc;
char *argv[];
{
    int fdtemp, i;
    struct stat stbuf;
    char *fname;

    got_agra_scale = 0;

    if (argc > 1)
    {
	/* see if this is a table file name */
	fname = argv[1];
	fdtemp = im_open(fname, "read");
	if (fdtemp >= 0)
	{
	    /* it's an image file */
	    im_close(fdtemp);
	}
	else
	{
	    /* now, if it's a file, it must be a table file */
	    /* if it's not a file, assume that its a geometry spec */
	    if (!stat(fname, &stbuf))
	    {
		/* it's a file -  must be a table file */
		if (argc < 4)
		{
		    error1("longitude, latitude column names missing");
		    return;
		    
		}
		if (!agra_scale(fname, argv[2], argv[3]))
		    return;


		/* move remaining args down */
		/* pb file.tbl ra dec geom       becomes */
		/* pb geom    */
		for (i = 1; i+3 < argc; i++)
		    argv[i] = argv[i+3];
		argc -= 3;
		got_agra_scale = 1;
	    }
	}

    }
    blank_window = 1;
    fi_name[0] = '\0';  /* no default filename */
    rddisk(argc, argv);
    blank_window = 0;
}


void
rddisk(argc, argv)
int argc;
char *argv[];
{
    int i, j, more_clip;
    struct img *imgp, tmpimg;
    char *labelptr, *geom;
    void *filptr;
    double atbl, sdiff;

    geom = NULL;
    labelptr = NULL;
    enable_control_c();

    for (i = 1; i < argc; i++)
    {
	switch (cmd(argv[i]))
	{
	case NOZP:
	    zp = FALSE;
	    break;
	case ZP:
	    zp = TRUE;
	    break;
	case NOST:
	    nost = TRUE;
	    break;
	case ST:
	    nost = FALSE;
	    break;
	default:
	    if (strncmp(argv[i], "-g", 2) == 0)
	    {
		/* looks like a geometry string */
		if (i + 1 < argc)
		{
		    geom = argv[i + 1];
		    i++;
		    break;
		}
	    }
	    if (argv[i][0] == '=')     /* allow geometry without the -g */
	    {
		geom = argv[i];
		break;
	    }
	    if (strncmp(argv[i], "label", 5) == 0)
	    {
		if (i + 1 < argc)
		{
		    labelptr = argv[i + 1];
		    i++;
		}
		else
		{
		    labelptr = "";
		}
		break;
	    }


	    /* must be a filename */
	    strcpy(fi_name, expand_path(argv[i]));
	    break;
	}
    }

    graphics_window = 0;

    if (blank_window == 1)
    {
	if (fi_name[0] == '\0')
	{
	    if (geom == NULL)
		geom = "=512x512";
	    if (labelptr == NULL)
		labelptr = "";
	    set_header_defaults(&tmpimg);
	    fill_glob(&tmpimg);
	    graphics_window = 1;
	    filnam[0] = '\0';
	    filehdr.tapetype = IMFILE;

	    if (!parse_size(geom, &pixels, &lines))
	    {
		error1("bad geometry specification");
		return;
	    }

	    minx = 0;
	    miny = 0;
	    maxx = pixels -1;
	    maxy = lines - 1;
	    x_offset = 1;
	    y_offset = 1;
	    sgn_y = 1;
	    if (!show_frame(geom, 0, labelptr))
		return;
	    stretch(0, 255);   /* set up color table */

	    if (got_agra_scale)
	    {
		printf("using scaling from agra\n");
		printf(
		"agra_plon, agra_plat, agra_mlon, agra_mlat, agra_su, agra_sv = %f %f %f %f %f %f\n", 
		agra_plon, agra_plat, agra_mlon, agra_mlat, agra_su, agra_sv);
		printf("agra_proj = [%s]\n", agra_proj);
		if (agra_plon >= 0)
		    glong = agra_plon;
		else
		    glong = agra_plon + 360.0;
		glat = agra_plat;
		cdelt1 = - agra_su / win_width;
		cdelt2 = agra_sv / win_height;
		/* FIX THE NEXT TWO using agra_mlon and agra_mlat */
		crpix1 = win_width / 2;
		crpix2 = win_height / 2;
		/* DO THE PROJECTION TYPE */
		if (strcmp(agra_proj, "aitoff") == 0)
		{
		    maptype = AITOFF;
		    file_coord_sys = output_coord_sys;
		}
		else if (strcmp(agra_proj, "gnomonic") == 0)
		{
		    maptype = GNOMONIC;
		    file_coord_sys = output_coord_sys;
		}

	    }

	    imgp = new_img();  /* put into screen mgmt */
	    flush_graphics();
	    wait_for_windowmgr(FALSE);
	    return;
	}
    }

    if (fi_name[0] == '\0')
    {
	error1("no workfile has been specified");
	return;
    }
    if (reread_on)
	force_reread(fi_name);

    fd = wf_open(fi_name, &tmpimg, 1);
    if (fd < 0)
    {
	return;
    }

    if (frame_advance)
	next_frame();
    else
    {
	/* prevent painting into IRAF (imio) frame */
	imgp = peek_FRAME(frame);
	if (imgp != NULL)
	{
	    if (imgp->imio_ok)
		next_frame();
	}
    }

    fill_glob(&tmpimg);
    curr_img = NULL;
    set_file_sys();

    minx = minx_next;
    miny = miny_next;

    /*   see if he wants screen jump before paint */
    if (zp)
    {
	x_orig = minx;
	y_orig = miny;
	initial_zoom();
    }

    if ((pixels == 0) || (lines == 0))
    {
	error1("no pixels in FITS primary data array");
	return;
    }

    maxx = minx + pixels - clip_left - clip_right - 1;
    maxy = miny + lines  - clip_top - clip_bottom - 1;
    x_offset = 1 + clip_left - minx;
    samp_disp = clip_left;

    if (cdelt2 >= 0)
    {
	/* painting up */
	line_disp = clip_bottom;
	sgn_y = 1;
	y_offset = lines - clip_top - maxy;
    }
    else
    {
	/* painting down */
	line_disp = clip_top;
	sgn_y = -1;
	y_offset = lines - clip_bottom + miny;
    }

    if (blank_window == 1)
    {
	graphics_window = 1;
	if (!show_frame(geom, zp, labelptr))
	    return;
	stretch(0, 255);   /* set up color table */
	imgp = new_img();  /* put into screen mgmt */
	flush_graphics();
	wait_for_windowmgr(FALSE);
	return;
    }
    else
    {
	graphics_window = 0;
    }


    if ((reqband > naxis3) || (reqband < 1))
    {
	error1("requested plane is not in file");
	goto paintexit;
    }
    band_offset = reqband - 1;


    if (debug)
    {
	fprintf(debug_file, "x_offset= %d\n", x_offset);
	fprintf(debug_file, 
	    "samp_disp= %d line_disp= %d band_offset = %d\n", 
	    samp_disp, line_disp, band_offset);
	fprintf(debug_file, 
	    "maxy=%d miny=%d maxx=%d minx=%d\n",maxy, miny, maxx, minx);
    }


    /* check if clipping is all within image */
    if (maxy < miny)
    {
	error1("clipping all the lines out of the image");
	goto paintexit;
    }
    if (maxx < minx)
    {
	error1("clipping all the pixels out of the image");
	goto paintexit;
    }

    /* now will it fit onto screen */
    if (JUPhardware)
    {
	if (maxy >= 1024)
	{
	    if(server_mode == FALSE)
		printf("warning: image too tall; cutting off a portion\n");
	    if (sgn_y < 0)  /* if painting down */
	    {
		more_clip = maxy - 1023;   /* how much more to clip out */
		/*
		line_disp += more_clip;
		*/
		y_offset -= more_clip;
	    }
	    maxy = 1023;
	}
	if (maxx >= 1280)
	{
	    if(server_mode == FALSE)
		printf("warning: image too wide; cutting off a portion\n");
	    maxx = 1279;
	}
    }

    if (filehdr.tapetype == IMFILE)
    {
	filptr = imget(fd, fi_name); /* read in entire image */
	if (filptr == NULL)
	{
	    error1("problem reading pixels");
	    im_close(fd);
	    return;
	}
    }

    mode_used = req_mode;

    if (need_hist() || ho_on || (mode_used == EQ))
	    check_hist(fd);

    slow = get_minmax(&min_ra);
    shigh = get_minmax(&max_ra);

    if (slow == shigh)
    {
	/* try again with recomputed histogram and iraf_min, iraf_max */
	check_hist(fd);
	slow = get_minmax(&min_ra);
	shigh = get_minmax(&max_ra);
    }

    if (debug)
	fprintf(debug_file, "stretching from DN=%e to DN=%e\n", slow, shigh);

    if ((server_mode == FALSE) && (!be_quiet) && (mode_used != EQ))
	printf("stretching from %e to %e\n", dn_to_jy(slow), dn_to_jy(shigh));

    if (slow == shigh)
    {
	if (iraf_min == iraf_max)
	{
	    sprintf(server_str, "all non-blank pixels have the same value (%g)", iraf_min);
	    error1(server_str);
	    goto paintexit;
	}
	else if (((iraf_min > iraf_max) && (bscale > 0)) ||
	         ((iraf_min < iraf_max) && (bscale < 0)))
	{
	    if ((minx == maxx) && (miny == maxy))
		error1("cannot compute stretch for a one-pixel image");
	    else
		error1("all pixels are blank");
	    goto paintexit;
	}
	else if (mode_used != EQ)   /* not an error for histogram eq */
	{
	    error1("high and low stretch values are too close");
	    goto paintexit;
	}
    }

    lastmin = dn_to_jy(slow);
    lastmax = dn_to_jy(shigh);

    if (debug_val == 7)
    {
	sdiff = shigh - slow;
	if (mode_used == LIN)
	{
	    for (j = 0; j < 255; j++)
		fprintf(debug_file, 
		    " %d    %g\n", j,   (sdiff / 254) * j + slow);
	}
	else
	{
	    for (j = 0; j < 256; j++)
	    {
		atbl = pow(10., j/254.0);
		fprintf(debug_file, 
		    " %d    %g\n", j, (atbl - 1.) / 9. * sdiff + slow);
	    }
	}
    }

    if (control_c)
	goto paintexit;

    if (!show_frame(geom, zp, labelptr))
	goto paintexit;

    if (!nost)
	stretch(0, 255);   /* set up color table */

    if (graphics)
    {
        wedge(minx + 550, miny, minx + 570, miny + 511);
    }

    for (j=0; j<256; j++)
	screen_hist[j] = 0;
    strcpy(screen_hist_filnam, filnam);

/* now do the pixels */

    pixbuf = (int *) malloc((maxx - minx + 1) * sizeof(int));

    switch (bitpix)
    {
	case 32:
	    paint32(minx, maxx, miny, maxy, samp_disp, line_disp,
		(int *) filptr);
	    break;
	case 16:
	    if (u16)
		paintu16(minx, maxx, miny, maxy, samp_disp, line_disp,
		    (unsigned short *) filptr);
	    else
		paint16(minx, maxx, miny, maxy, samp_disp, line_disp,
		    (short *) filptr);
	    break;
	case 8:
	    paint8(minx, maxx, miny, maxy, samp_disp, line_disp, 
		(unsigned char *) filptr);
	    break;
	case -32:
	    paintm32(minx, maxx, miny, maxy, samp_disp, line_disp,
		(float *) filptr);
	    break;
	case -64:
	    paintm64(minx, maxx, miny, maxy, samp_disp, line_disp,
		(double *) filptr);
	    break;
	default:
	    sprintf(server_str, "illegal bitpix value: %d", bitpix);
	    error1(server_str);
	    goto paintexit;
	    /*NOTREACHED*/
	    break;
    }

    free(pixbuf);
    pixbuf = NULL;
     
    imgp = new_img();  /* put into screen mgmt */
    flush_graphics();
    wait_for_windowmgr(FALSE);

    if ((server_mode == TRUE) && (!be_quiet))
    {
	srv_string("filename", fi_name);
	sprintf(server_str, "%-g", dn_to_jy(slow));
	srv_real("range_min_val", server_str);
	sprintf(server_str, "%-g", dn_to_jy(shigh));
	srv_real("range_max_val", server_str);
	if ((iraf_min != 0.0) && (iraf_max != 0.0))
	{
	    sprintf(server_str, "%-g", dn_to_jy(iraf_min));
	    srv_real("image_min_val", server_str);
	    sprintf(server_str, "%-g", dn_to_jy(iraf_max));
	    srv_real("image_max_val", server_str);
	}
	srv_string("units", bunit);
    }

paintexit:

    if (filehdr.tapetype == IMFILE)
	im_close(fd);
    else
	close(fd);
    fd = 0;
}

static void
paint8(minx, maxx, miny, maxy, samp_disp, line_disp, filptr)
int minx, maxx, miny, maxy, samp_disp, line_disp;
unsigned char *filptr;
{
    int this_errno;
    int rdstat;
    unsigned char *filbuf;
    register int x, runval;
    register unsigned char *run8p;
    register int *bufp, pixval;
    int y, j, local_blank;
    unsigned int read_size;
    off_t file_disp;
    int yend, ystart;
    int tbl[256], tbl1[256];
    int this_val, last_val, i;
    double sdiff, atbl;

    if (FINITE(blank))
    {
	local_blank = blank;
    }
    else
    {
	local_blank = -1;  /* impossible value - (NaN causes traps) */
    }

    if (mode_used == LIN)
    {
	sdiff = shigh - slow;
	for (j=0; j<256; j++)
	{
	    tbl1[j] = (254 / sdiff) * (j - slow);
	    if (tbl1[j] < 0)
		tbl1[j] = 0;
	    if (tbl1[j] > 254)
		tbl1[j] = 254;
	}
    }

    else if ((mode_used == LOG) || (mode_used == LOGLOG))
    {
	sdiff = shigh - slow;
	for (j=0; j<256; j++)
	{
	    if (j <= slow)
		tbl1[j] = 0;
	    else if (j >= shigh)
		tbl1[j] = 254;
	    else
	    {
		if (mode_used == LOG)
		    tbl1[j] = 254 * log10((9 * (j - slow) / sdiff) + 1);
		else
		{
		    /* LOGLOG */
		    atbl = log10((9 * (j - slow) / sdiff) + 1);
		    tbl1[j] = 254 * log10((9.0 * atbl) + 1);
		}

	    }
	}
    }

    else if (mode_used == TRUELOG)
    {
	for (j=0; j<256; j++)
	{
	    if (j <= slow)
		tbl1[j] = 0;
	    else if (j >= shigh)
		tbl1[j] = 254;
	    else
	    {
	        tbl1[j] = 254 * (log10((double)j) - log10(slow)) 
			      / (log10(  shigh  ) - log10(slow));
	    }
	}
    }

    else if (mode_used == EQ)
    {
	eq_tbl(tbl);
	last_val = -1;
	for (j = 0; j <= 255; j++)
	{
	    this_val = tbl[j];
	    if (this_val < 0)
		this_val = 0;
	    else if (this_val > 255)
		this_val = 255;
	    for (i = last_val+1; i <= this_val; i++)
		tbl1[i] = j;
	    last_val = this_val;

	}
	for (i = last_val+1; i <= 255; i++)
	    tbl1[i] = 255;
    }

    if (filehdr.tapetype == IMFILE)
    {
	filptr += (line_disp * pixels + samp_disp) +
	    band_offset * lines * pixels;
    }
    else
    {
	file_disp = filehdr.img_start +  (line_disp * pixels + samp_disp) +
	    band_offset * lines * pixels;
	read_size = maxx - minx + 1;
	filbuf = (unsigned char *) malloc(read_size);
	if (debug)
	    fprintf(debug_file, 
	    "paint8: maxy=%d miny=%d file_disp=%ld\n",maxy, miny, file_disp);
    }

    map_for_AOI(plane);

    if (sgn_y > 0)
    {
	ystart = miny;
	yend = maxy + 1;
    }
    else
    {
	ystart = maxy;
	yend = miny - 1;
    }


    for (y = ystart; y != yend; y+=sgn_y)
    {
	if (filehdr.tapetype == IMFILE)
	{
	    run8p = filptr;
	    filptr += pixels;
	}
	else
	{
	    /*  first read the line from the disk */
	    lseek(fd, file_disp, 0);
	    file_disp += pixels;
	    if ((rdstat = read(fd,filbuf,read_size)) != read_size)
	    {
		if (rdstat == -1)
		{
		    this_errno = errno;
		    sprintf(server_str, "bad read at byte %ld: ", file_disp);
		    sperror(server_str, this_errno);
		    break;
		}
		if (rdstat == 0)
		{
		    sprintf(server_str,
			    "end of file reached at byte %ld", file_disp);
		    error1(server_str);
		    break;
		}
	    }
	    run8p = filbuf;
	}

	bufp = pixbuf;
	x = maxx- minx + 1;
	do
	{
	    runval = *run8p++;

	    if(fixed_ct == TRUE)
	    {
		pixval = runval;
		screen_hist[pixval] += 1;
	    }
	    else
	    {
		if (runval == local_blank)
		{
		    pixval = -1;
		}
		else
		{
		    pixval = tbl1[runval];  /* do the range stretch */
		    screen_hist[pixval] += 1;
		}
	    }
	    *bufp++ = pixval;
	} while (--x);
	write_AOI(minx, y , maxx, y, pixbuf);

	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\npaint interrupted at y = %d\n", y);
	    fprintf(session, "\npaint interrupted at y = %d\n", y);
	    if (sgn_y > 0)
		maxy = y;   /* reset limits for screen mgmt */
	    else
		miny = y;
	    break;
	}
    }

    if (filehdr.tapetype != IMFILE)
	free(filbuf);
    map_normal();   /* reset board mapping to standard configuration */
    image_setup(plane);
}

static void
paintu16(minx, maxx, miny, maxy, samp_disp, line_disp, filptr)
int minx, maxx, miny, maxy, samp_disp, line_disp;
unsigned short *filptr;
{
    int this_errno;
    float floati;
    int rdstat;
    register int x, runval, delta, deltasav;
    register int *bufp, pixval;
    register unsigned short *run16p;
    char *filbuf;
    double sdiff;
    double atbl;
    int y, j;
    int read_size;
    off_t file_disp;
    int yend, ystart;
    int tbl[256];
    double local_blank;

    if (FINITE(blank))
    {
	local_blank = blank;
    }
    else
    {
	local_blank = 1e+30;  /* impossible value - (NaN causes traps) */
    }

    if (mode_used == LIN)
    {
	sdiff = shigh - slow;
	for (j=0; j<255; j++)
	{
	    tbl[j] = (sdiff / 254) * j + slow;
	}
	tbl[255] = 0x7fffffff;
    }

    else if ((mode_used == LOG) || (mode_used == LOGLOG))
    {
	if(debug)
	    fprintf(debug_file, "paintu16: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	sdiff = shigh - slow;
	if(sdiff == 0.)
	    sdiff=1.;

	for (j=0; j<255; ++j)
	{
	    atbl = pow(10., j/254.0);
	    if (mode_used == LOGLOG)
		atbl = pow(10., (atbl - 1.0) / 9.0);
	    floati = (atbl - 1.) / 9. * sdiff + slow;
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == TRUELOG)
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paintu16: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = (j/254.0)*(log10(shigh) - log10(slow)) + log10(slow);

	    floati = pow(10., atbl);
		
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == EQ)
    {
	eq_tbl(tbl);
	sdiff = shigh - slow;
    }


    if (filehdr.tapetype == IMFILE)
    {
	filptr += (line_disp * pixels + samp_disp) +
	    (band_offset * lines * pixels);
    }
    else
    {
	file_disp = filehdr.img_start + 2 * (line_disp * pixels + samp_disp) +
	    (band_offset * lines * pixels) * 2;
	read_size = (maxx - minx +1) * 2;
	filbuf = (char *) malloc(read_size);
    }

    map_for_AOI(plane);

    if (sgn_y > 0)
    {
	ystart = miny;
	yend = maxy + 1;
    }
    else
    {
	ystart = maxy;
	yend = miny - 1;
    }

    if (sdiff >= 0)
	deltasav = 64;
    else
	deltasav = - 64;

    for (y = ystart; y != yend; y+=sgn_y)
    {
	if (filehdr.tapetype == IMFILE)
	{
	    run16p = (unsigned short *) filptr;
	    filptr += pixels;
	}
	else
	{
	    /*  first read the line from the disk */
	    lseek(fd, file_disp, 0);
	    file_disp += 2 * pixels;
	    if ((rdstat = read(fd,filbuf,read_size)) != read_size)
	    {
		if (rdstat == -1)
		{
		    this_errno = errno;
		    sprintf(server_str, 
			"bad read at byte %ld: ", file_disp);
		    sperror(server_str, this_errno);
		    break;
		}
		if (rdstat == 0)
		{
		    sprintf(server_str,
			"end of file reached at byte %ld\n", file_disp);
		    error1(server_str);
		    break;
		}
	    }
	    run16p = (unsigned short *) (filbuf);
	}
	bufp = pixbuf;
	x = maxx - minx + 1;

	do
	{
	    runval = *run16p++;
	    if (runval == local_blank)
	    {
		pixval = -1;  /* set to blank but dont put into histogram */
	    }
	    else
	    {
		pixval = 128;
		delta = deltasav; /* 64 if ra normal, -64 if ra reversed */

		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] >= runval)
		    pixval -= 1;

		screen_hist[pixval] += 1;  /* increment histogram */
	    }
	    *bufp++ = pixval;
	} while (--x);

	write_AOI(minx, y , maxx, y, pixbuf);

	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\npaint interrupted at y = %d\n", y);
	    fprintf(session, "\npaint interrupted at y = %d\n", y);
	    if (sgn_y > 0)
		maxy = y;   /* reset limits for screen mgmt */
	    else
		miny = y;
	    break;
	}
    }

    if (filehdr.tapetype != IMFILE)
	free(filbuf);
    map_normal();   /* reset board mapping to standard configuration */
    image_setup(plane);

}

static void
paint16(minx, maxx, miny, maxy, samp_disp, line_disp, filptr)
int minx, maxx, miny, maxy, samp_disp, line_disp;
short *filptr;
{
    int this_errno;
    float floati;
    int rdstat;
    register int x, runval, delta, deltasav;
    register int *bufp, pixval;
    register short *run16p;
    char *filbuf;
    double sdiff;
    double atbl;
    int y, j, i;
    int read_size;
    off_t file_disp;
    int yend, ystart;
    int tbl[256];
    double *spec_ptr;
    double local_blank;

    if (FINITE(blank))
    {
	local_blank = blank;
    }
    else
    {
	local_blank = 1e+30;  /* impossible value - (NaN causes traps) */
    }

    if (mode_used == LIN)
    {
	sdiff = shigh - slow;
	for (j=0; j<255; j++)
	{
	    tbl[j] = (sdiff / 254) * j + slow;
	}
	tbl[255] = 0x7fffffff;
    }

    else if ((mode_used == LOG) || (mode_used == LOGLOG))
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paint16: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	sdiff = shigh - slow;
	if(sdiff == 0.)
	    sdiff=1.;

	for (j=0; j<255; ++j)
	{
	    atbl = pow(10., j/254.0);
	    if (mode_used == LOGLOG)
		atbl = pow(10., (atbl - 1.0) / 9.0);
	    floati = (atbl - 1.) / 9. * sdiff + slow;
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == TRUELOG)
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paint16: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = (j/254.0)*(log10(shigh) - log10(slow)) + log10(slow);

	    floati = pow(10., atbl);
		
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == EQ)
    {
	eq_tbl(tbl);
	sdiff = shigh - slow;
    }
    else if (mode_used == HIST_SPEC)
    {
	eq_tbl(tbl);
	spec_ptr = hist_spec();
	for (i = 0; i <= 255; i++)
	{
	    tbl[i] = tbl[i] * *spec_ptr++ / 255;
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d\n", i, tbl[i]);
	}

	sdiff = shigh - slow;
    }


    if (filehdr.tapetype == IMFILE)
    {
	filptr += (line_disp * pixels + samp_disp) +
	    (band_offset * lines * pixels);
    }
    else
    {
	file_disp = filehdr.img_start + 2 * (line_disp * pixels + samp_disp) +
	    (band_offset * lines * pixels) * 2;
	read_size = (maxx - minx +1) * 2;
	filbuf = (char *) malloc(read_size);
    }

    map_for_AOI(plane);

    if (sgn_y > 0)
    {
	ystart = miny;
	yend = maxy + 1;
    }
    else
    {
	ystart = maxy;
	yend = miny - 1;
    }

    if (sdiff >= 0)
	deltasav = 64;
    else
	deltasav = - 64;

    for (y = ystart; y != yend; y+=sgn_y)
    {
	if (filehdr.tapetype == IMFILE)
	{
	    run16p = (short *) filptr;
	    filptr += pixels;
	}
	else
	{
	    /*  first read the line from the disk */
	    lseek(fd, file_disp, 0);
	    file_disp += 2 * pixels;
	    if ((rdstat = read(fd,filbuf,read_size)) != read_size)
	    {
		if (rdstat == -1)
		{
		    this_errno = errno;
		    sprintf(server_str,
			"bad read at byte %ld: ", file_disp);
		    sperror(server_str, this_errno);
		    break;
		}
		if (rdstat == 0)
		{
		    sprintf(server_str,
			"end of file reached at byte %ld\n", file_disp);
		    error1(server_str);
		    break;
		}
	    }
	    run16p = (short *) (filbuf);
	}
	bufp = pixbuf;
	x = maxx - minx + 1;

	do
	{
	    runval = *run16p++;
	    if (runval == local_blank)
	    {
		pixval = -1;  /* set to blank but dont put into histogram */
	    }
	    else
	    {
		pixval = 128;
		delta = deltasav; /* 64 if ra normal, -64 if ra reversed */

		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] >= runval)
		    pixval -= 1;

		screen_hist[pixval] += 1;  /* increment histogram */
	    }
	    *bufp++ = pixval;
	} while (--x);

	write_AOI(minx, y , maxx, y, pixbuf);

	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\npaint interrupted at y = %d\n", y);
	    fprintf(session, "\npaint interrupted at y = %d\n", y);
	    if (sgn_y > 0)
		maxy = y;   /* reset limits for screen mgmt */
	    else
		miny = y;
	    break;
	}
    }

    if (filehdr.tapetype != IMFILE)
	free(filbuf);
    map_normal();   /* reset board mapping to standard configuration */
    image_setup(plane);

}

static void
paint32(minx, maxx, miny, maxy, samp_disp, line_disp, filptr)
int minx, maxx, miny, maxy, samp_disp, line_disp;
int *filptr;
{
    int this_errno;
    double real_high, real_low, delta_f, delta_f2i;
    float floati;
    int rdstat;
    register int x, runval, delta, deltasav;
    register int *runp;
    register int *bufp, pixval;
    char *filbuf;
    double sdiff;
    double atbl;
    int y, j;
    int read_size;
    off_t file_disp;
    int yend, ystart;
    int tbl[256];
    double local_blank;

    if (FINITE(blank))
    {
	local_blank = blank;
    }
    else
    {
	local_blank = 1e+30;  /* impossible value - (NaN causes traps) */
    }

    if (mode_used == LIN)
    {
	sdiff = shigh - slow;
	for (j=0; j<255; j++)
	{
	    floati = (sdiff / 254) * j + slow;
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	}
	tbl[255] = 0x7fffffff;
    }
    else if ((mode_used == LOG) || (mode_used == LOGLOG))
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paint32: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	sdiff = shigh - slow;
	if(sdiff == 0.)
	    sdiff=1.;

	for (j=0; j<255; ++j)
	{
	    atbl = pow(10., j/254.0);
	    if (mode_used == LOGLOG)
		atbl = pow(10., (atbl - 1.0) / 9.0);
		
	    floati = (atbl - 1.) / 9. * sdiff + slow;
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == TRUELOG)
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paint32: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = (j/254.0)*(log10(shigh) - log10(slow)) + log10(slow);

	    floati = pow(10., atbl);
		
	    if  (-floati > 0x7fffffff)
		tbl[j] = 0x80000000;
	    else if (floati > 0x7fffffff)
		tbl[j] =     0x7fffffff;
	    else
		tbl[j] = floati;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == LORD)
    {
	sdiff = shigh - slow;

	real_high = dn_to_jy(shigh);
	real_low = dn_to_jy(slow);
	delta_f = log10(real_high - real_low + 1);
	delta_f = delta_f / 254;
	delta_f = pow(10., delta_f);

	delta_f2i = 1;
	for (j = 0; j < 255; j++)
	{
	    tbl[j] = jy_to_dn((real_low - 1) + delta_f2i);
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %d (%f  rounded to %-g)\n", 
		    j, tbl[j], (real_low - 1) + delta_f2i,
		    dn_to_jy(tbl[j]));
	    delta_f2i = delta_f2i * delta_f;
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == EQ)
    {
	eq_tbl(tbl);
	sdiff = shigh - slow;
    }

    if (filehdr.tapetype == IMFILE)
    {
	filptr += (line_disp * pixels + samp_disp) +
	    (band_offset * lines * pixels);
    }
    else
    {
    file_disp = filehdr.img_start + 4 * (line_disp * pixels + samp_disp) +
	(band_offset * lines * pixels) * 4;
	read_size = (maxx - minx +1) * 4;
	filbuf = (char *) malloc(read_size);
    }

    map_for_AOI(plane);

    if (sgn_y > 0)
    {
	ystart = miny;
	yend = maxy + 1;
    }
    else
    {
	ystart = maxy;
	yend = miny - 1;
    }
    if (sdiff >= 0)
	deltasav = 64;
    else
	deltasav = - 64;

    for (y = ystart; y != yend; y+=sgn_y)
    {
	if (filehdr.tapetype == IMFILE)
	{
	    runp = filptr;
	    filptr += pixels;
	}
	else
	{
	    /*  first read the line from the disk */
	    lseek(fd, file_disp, 0);
	    file_disp += 4 * pixels;
	    if ((rdstat = read(fd,filbuf,read_size)) != read_size)
	    {
		if (rdstat == -1)
		{
		    this_errno = errno;
		    sprintf(server_str,
			"bad read at byte %ld: ", file_disp);
		    sperror(server_str, this_errno);
		    break;
		}
		if (rdstat == 0)
		{
		    sprintf(server_str,
			"end of file reached at byte %ld\n", file_disp);
		    error1(server_str);
		    break;
		}
	    }
	    runp = (int *) (filbuf);
	}
	bufp = pixbuf;
	x = maxx - minx + 1;
	do
	{
	    runval = *runp++;
	    if (runval == local_blank)
	    {
		pixval = -1;  /* set to blank but dont put into histogram */
	    }
	    else
	    {
		pixval = 128;
		delta = deltasav; /* 64 if ra normal, -64 if ra reversed */

		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		if (tbl[pixval] >= runval)
		    pixval -= 1;

		screen_hist[pixval] += 1;  /* increment histogram */
	    }
	    *bufp++ = pixval;

	} while (--x);

	write_AOI(minx, y , maxx, y, pixbuf);

	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\npaint interrupted at y = %d\n", y);
	    fprintf(session, "\npaint interrupted at y = %d\n", y);
	    if (sgn_y > 0)
		maxy = y;   /* reset limits for screen mgmt */
	    else
		miny = y;
	    break;
	}
    }

    if (filehdr.tapetype != IMFILE)
	free(filbuf);
    map_normal();   /* reset board mapping to standard configuration */
    image_setup(plane);

}

static void
paintm32(minx, maxx, miny, maxy, samp_disp, line_disp, filptr)
int minx, maxx, miny, maxy, samp_disp, line_disp;
float *filptr;
{
    float *file_disp;
    int deltasav;
    register int x, delta;
    register float runval, *runp;
    register int *bufp, pixval;
    double sdiff;
    double atbl;
    int y, j;
    int yend, ystart;
    double tbl[256];

    sdiff = shigh - slow;
    if(sdiff == 0.)
	sdiff=1.;

    if ((mode_used == LOG) || (mode_used == LOGLOG))
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paintm32: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = pow(10., j/254.0);
	    if (mode_used == LOGLOG)
		atbl = pow(10., (atbl - 1.0) / 9.0);
	    tbl[j] = (atbl - 1.) / 9. * sdiff + slow;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %g (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = FLT_MAX;
    }
    else if (mode_used == TRUELOG)
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paintm32: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = (j/254.0)*(log10(shigh) - log10(slow)) + log10(slow);

	    tbl[j] = pow(10., atbl);
		
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %g (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == EQ)
    {
	deq_tbl(tbl);
    }

    map_for_AOI(plane);

    if (sgn_y > 0)
    {
	ystart = miny;
	yend = maxy + 1;
    }
    else
    {
	ystart = maxy;
	yend = miny - 1;
    }

    if (sdiff > 0)
	deltasav = 64;
    else
	deltasav = - 64;

    file_disp = filptr;
    file_disp += (line_disp * pixels + samp_disp) +
	(band_offset * lines * pixels);
    if (debug)
	fprintf(debug_file, 
	    "maxy=%d miny=%d file_disp=0x%p\n",maxy, miny, (void *) file_disp);

    for (y = ystart; y != yend; y+=sgn_y)
    {
	bufp = pixbuf;
	runp = (float *) (file_disp);
	file_disp += pixels;    /* prepare for next line */
	x = maxx - minx + 1;
	do
	{
	    runval = *runp++;
	    if (!FINITE(runval))
	    {
		pixval = -1;  /* set to blank but dont put into histogram */
	    }
	    else
	    {
		if (mode_used == LIN)
		{
		    runval = (runval - slow) * 254 / sdiff;
		    if (runval < 0)
			pixval = 0;
		    else if (runval > 254)
			pixval = 254;
		    else
			pixval = runval;
		}
		else
		{
		pixval = 128;
		delta = deltasav; /* 64 if ra normal, -64 if ra reversed */

		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] >= runval)
		    pixval -= 1;
		}

		screen_hist[pixval] += 1;  /* increment histogram */
	    }
	    *bufp++ = pixval;
	} while (--x);

	write_AOI(minx, y , maxx, y, pixbuf);

	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\npaint interrupted at y = %d\n", y);
	    fprintf(session, "\npaint interrupted at y = %d\n", y);
	    if (sgn_y > 0)
		maxy = y;   /* reset limits for screen mgmt */
	    else
		miny = y;
	    break;
	}
    }

    map_normal();   /* reset board mapping to standard configuration */
    image_setup(plane);
}

static void
paintm64(minx, maxx, miny, maxy, samp_disp, line_disp, filptr)
int minx, maxx, miny, maxy, samp_disp, line_disp;
double *filptr;
{
    double *file_disp;
    int deltasav;
    register int x, delta;
    register double runval, *runp;
    register int *bufp, pixval;
    double sdiff;
    double atbl;
    int y, j;
    int yend, ystart;
    double tbl[256];

    sdiff = shigh - slow;
    if(sdiff == 0.)
	sdiff=1.;

    if ((mode_used == LOG) || (mode_used == LOGLOG))
    {
	if(debug)
	    fprintf(debug_file, "paintm64: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = pow(10., j/254.0);
	    if (mode_used == LOGLOG)
		atbl = pow(10., (atbl - 1.0) / 9.0);
	    tbl[j] = (atbl - 1.) / 9. * sdiff + slow;
	
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %g (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = FLT_MAX;
    }
    else if (mode_used == TRUELOG)
    {
	if(debug == TRUE)
	    fprintf(debug_file, "paintm64: shigh = %-g, slow = %-g\n", 
		shigh, slow);

	for (j=0; j<255; ++j)
	{
	    atbl = (j/254.0)*(log10(shigh) - log10(slow)) + log10(slow);

	    tbl[j] = pow(10., atbl);
		
	    if (debug_val == 7)
		fprintf(debug_file, "tbl[%d] = %g (%-g)\n", j, tbl[j], atbl);
	}
	tbl[255] = 0x7fffffff;
    }
    else if (mode_used == EQ)
    {
	deq_tbl(tbl);
    }

    map_for_AOI(plane);

    if (sgn_y > 0)
    {
	ystart = miny;
	yend = maxy + 1;
    }
    else
    {
	ystart = maxy;
	yend = miny - 1;
    }

    if (sdiff > 0)
	deltasav = 64;
    else
	deltasav = - 64;

    file_disp = filptr;
    file_disp += (line_disp * pixels + samp_disp) +
	(band_offset * lines * pixels);
    if (debug)
	fprintf(debug_file, 
	    "maxy=%d miny=%d file_disp=0x%p\n",maxy, miny, (void *) file_disp);

    for (y = ystart; y != yend; y+=sgn_y)
    {
	bufp = pixbuf;
	runp = (double *) (file_disp);
	file_disp += pixels;    /* prepare for next line */
	x = maxx - minx + 1;
	do
	{
	    runval = *runp++;
	    if (!FINITE(runval))
	    {
		pixval = -1;  /* set to blank but dont put into histogram */
	    }
	    else
	    {
		if (mode_used == LIN)
		{
		    runval = (runval - slow) * 254 / sdiff;
		    if (runval < 0)
			pixval = 0;
		    else if (runval > 254)
			pixval = 254;
		    else
			pixval = runval;
		}
		else
		{
		pixval = 128;
		delta = deltasav; /* 64 if ra normal, -64 if ra reversed */

		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] < runval)
		    pixval += delta;
		else
		    pixval -= delta;
		delta >>= 1;
		if (tbl[pixval] >= runval)
		    pixval -= 1;
		}

		screen_hist[pixval] += 1;  /* increment histogram */
	    }
	    *bufp++ = pixval;
	} while (--x);

	write_AOI(minx, y , maxx, y, pixbuf);

	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\npaint interrupted at y = %d\n", y);
	    fprintf(session, "\npaint interrupted at y = %d\n", y);
	    if (sgn_y > 0)
		maxy = y;   /* reset limits for screen mgmt */
	    else
		miny = y;
	    break;
	}
    }

    map_normal();   /* reset board mapping to standard configuration */
    image_setup(plane);
}

void
repaintm32(minx1, maxx1, miny1, maxy1, filptr)
int minx1, maxx1, miny1, maxy1;  /* SC coordinates */
				/* samp1 and line1 specify center pixel */
void *filptr;
{
    int samp_disp1, line_disp1, temp;

    pixbuf = (int *) malloc((maxx - minx + 1) * sizeof(int));

    if (maxy1 < miny1)
    {
	temp = maxy1;
	maxy1 = miny1;
	miny1 = temp;
    }

    samp_disp1 = minx1 - minx;
    if (sgn_y > 0)
    {
	line_disp1 = miny1 - miny;
    }
    else
    {
	line_disp1 = maxy - maxy1;
    }
    switch (bitpix)
    {
	case 32:
	    paint32(minx1, maxx1, miny1, maxy1, samp_disp1, line_disp1,
		(int *) filptr);
	    break;
	case 16:
	    if (u16)
		paintu16(minx1, maxx1, miny1, maxy1, samp_disp1, line_disp1,
		    (unsigned short *) filptr);
	    else
		paint16(minx1, maxx1, miny1, maxy1, samp_disp1, line_disp1,
		    (short *) filptr);
	    break;
	case 8:
	    paint8(minx1, maxx1, miny1, maxy1, samp_disp1, line_disp1,
		(unsigned char *) filptr);
	    break;
	case -32:
	    paintm32(minx1, maxx1, miny1, maxy1, samp_disp1, line_disp1,
		(float *) filptr);
	    break;
	case -64:
	    paintm64(minx1, maxx1, miny1, maxy1, samp_disp1, line_disp1,
		(double *) filptr);
	    break;
    }

    free(pixbuf);
    pixbuf = NULL;
}

static void eq_tbl(tbl)
int tbl[];
{
    int goodpix, accum, tblindex, hist_index;
    double next_goal, goodpix_255;

	goodpix = 0;
	for (hist_index = 0; hist_index < HISTSIZ2; hist_index++)
	    goodpix += hist[hist_index];
	goodpix_255 = goodpix / 255.0;

	tblindex = 0;
	tbl[tblindex++] = hist_min;
	next_goal = goodpix_255;
	hist_index = 0;
	accum = 0;
	while (hist_index < HISTSIZ2)
	{
	    if (accum >= next_goal)
	    {
		if (debug_val == 7)
		    printf("RBH setting tbl[%d] to %d\n", tblindex, 
			(int) (hist_index * hist_binsiz + hist_min));
		tbl[tblindex++] = hist_index * hist_binsiz + hist_min;
		next_goal += goodpix_255;
	    }
	    else
	    {
		accum += hist[hist_index++];
	    }
	}
}


static void deq_tbl(tbl)
double tbl[];
{
    int goodpix, accum, tblindex, hist_index;
    double next_goal, goodpix_255;

	goodpix = 0;
	for (hist_index = 0; hist_index < HISTSIZ2; hist_index++)
	    goodpix += hist[hist_index];
	goodpix_255 = goodpix / 255.0;

	tblindex = 0;
	tbl[tblindex++] = hist_min;
	next_goal = goodpix_255;
	hist_index = 0;
	accum = 0;
	while (hist_index < HISTSIZ2)
	{
	    if (accum >= next_goal)
	    {
		if (debug_val == 7)
		    printf("RBH setting tbl[%d] to %d\n", tblindex, 
			(int) (hist_index * hist_binsiz + hist_min));
		tbl[tblindex++] = hist_index * hist_binsiz + hist_min;
		next_goal += goodpix_255;
	    }
	    else
	    {
		accum += hist[hist_index++];
	    }
	}
}


static
double *hist_spec()
{
    int i, j;
    double a, g[256];
    static double h[256];

    for (i = 0; i <= 255; i++)
    {

	a = (erf((i - 127.5) * 3 / 127.5));
	g[i] = 127.5 * (a + 1.0);
#ifdef NOTDEF
	printf("i = %d  a = %f  erf(a) = %f  g[i] = %f\n", i, a, erf(a), g[i]);
#endif /* NOTDEF */
    }

    j = 0;
    i = 0;
    while ((i <= 255) && (j <= 255))
    {
	if (g[j] < i)
	    j++;
	else
	    h[i++] = j;
    }
    h[255] = 255.0;



#ifdef NOTDEF
    for (i = 0; i <= 255; i++)
	printf("i = %d  h[i] = %f\n", i, h[i]);
#endif /* NOTDEF */
    return(h);
}
