/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#define _XOPEN_SOURCE
#include "skyview.h"
#include "fits.h"
#include "img.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <ctype.h>
#include <sys/file.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

extern int mfd, debug, server_mode, vax_bytes;
extern FILE *debug_file;
extern char server_str[];
extern char fi_name[], filnam[];
extern int bitpix, pixels, lines, naxis, naxis3;
extern double blank;
extern double crpix1, crpix2;
extern char bunit[];
extern char ctype1sav[], ctype2sav[], object[];
extern double bscale, b_zero, glong, glat, crval3, cdelt1, cdelt2, twist;
extern int filnbr, tappos;
extern struct img *curr_img;
extern char *hdr_bufp_end, *hdr_buf;  /* used by screen dump and crop also */

void
writetape(argc, argv)
int argc;
char *argv[];
{
    short temp, *pshort;
    struct img tmpimg;
    char prtbuf[80];
    time_t timenow;
    struct tm *tim;
    int fd, bufsize, blocks;
    char timstr[10];
    int i;
    char *bufp1, pixbuf[2880], *filp;
    int rdstat, wrstat, count;
    struct mtop  mstop;
    struct hdrstruct thdr;
    int tmpnbr, nfilnbr;
    char *tapename;

    wait((int *)0);    /* let child processes finish */

    if ((tapename = getenv("STAPE")) == NULL)
	tapename = "/dev/rmt12";

    if (mfd < 0)
    {
	mfd = open(tapename, O_RDWR, 0);
	if (mfd < 0)
	{
	    sperror(tapename, errno);
	    return;
	}
	tappos = 1;
	filnbr = 0;   /* will be incremented by 1 (default) */
    }
    else
    {
	/* must make sure we have it open for write */
	close(mfd);                     /* close without rewind */
	mfd = open(tapename, O_RDWR, 0);
	if (mfd < 0)
	{
	    sperror(tapename, errno);
	    return;
	}
	/* use existing tappos and filnbr */
    }

    if (argc > 3)
    {
	error1("wrong number of arguments to TW command");
	return;
    }
    nfilnbr = filnbr;
    if (argc == 1)
    {
	/*  no arguments */
	nfilnbr++;
    }
    else
    {
	if (argc == 3)
	{
	    if (getint(argv[1], &tmpnbr) == FALSE)
		return;
	    if ((*argv[1] == '+') || (*argv[1] == '-'))
	    {
		nfilnbr += tmpnbr;
	    }
	    else
	    {
		nfilnbr = tmpnbr;
	    }
	    strcpy(fi_name, expand_path(argv[2]));
	}
	if (argc == 2)
	{
	    if (debug)
		fprintf(debug_file, "got 1 argument\n");
	    /* see if its a file number or a workfile name */
	    if ((*argv[1] == '+') || (*argv[1] == '-') || (isdigit(*argv[1])))
	    {
		/* its a number */
		if (debug)
		    fprintf(debug_file, "it is a number\n");
		if (getint(argv[1], &tmpnbr) == FALSE)
		    return;
		if ((*argv[1] == '+') || (*argv[1] == '-'))
		{
		    nfilnbr += tmpnbr;
		}
		else
		{
		    nfilnbr = tmpnbr;
		}
	    }
	    else
	    {
		if (debug)
		    fprintf(debug_file, "it is a filename\n");
		strcpy(fi_name, expand_path(argv[1]));
		nfilnbr++;  /* default to next file */
	    }
	}
    }
    fd = wf_open(fi_name, &tmpimg, 1);
    if (fd < 0)
    {
	return;
    }

    fill_glob(&tmpimg);
    curr_img = NULL;
    set_file_sys();

    filnbr = nfilnbr;   /* (didnt want to change file nbr if file open blew */

    if (debug)
	fprintf(debug_file, 
	"seeking tape filenumber %d   current tappos=%d\n", filnbr, tappos);
    if (filnbr > tappos)
    {
	mstop.mt_op = MTFSF;
	mstop.mt_count = filnbr - tappos;
	wrstat = ioctl(mfd, MTIOCTOP, &mstop);
	if (wrstat == -1)
	{
	    sperror("S error moving mag tape forward", errno);
	    return;
	}
    }
    else    /* filnbr <= tappos */
    {
	if (filnbr != 1)
	{
	    mstop.mt_op = MTBSF;
	    mstop.mt_count = tappos - filnbr + 1;
	    wrstat = ioctl(mfd, MTIOCTOP, &mstop);
	    if (wrstat == -1)
	    {
		sperror("S error moving mag tape backward", errno);
		return;
	    }

	    mstop.mt_op = MTFSF;
	    mstop.mt_count = 1;
	    wrstat = ioctl(mfd, MTIOCTOP, &mstop);
	    if (wrstat == -1)
	    {
		sperror("S error moving mag tape forward one file", errno);
		return;
	    }
	}
    }
    tappos = filnbr;

    lseek(fd, 0L, 0);
    rdstat = read(fd, &thdr, sizeof(thdr));
    if (debug)
	fprintf(debug_file, "disk header read returned %d bytes\n", rdstat);

    bufsize = (thdr.img_start - thdr.hdr_start) / 2880; /* number of blocks */
    bufsize += 3; /* room for more */
    bufsize = bufsize * 2880;
    hdr_buf = malloc(bufsize);
    if (hdr_buf == 0)
    {
	error1("error allocating space for tapewrite");
	return;
    }

    /* now read old FITS header */
    lseek(fd, thdr.hdr_start, 0);
    read (fd, hdr_buf, (int) (thdr.img_start - thdr.hdr_start));

    /* find END keyword */
    if (hdrinit(bufsize) == FALSE)
    {
	error1("tapewrite error: no END in original FITS header");
	return;
    }

    hdrlogical("SIMPLE",'T');
    hdrint("BITPIX",bitpix);
    hdrint("NAXIS",naxis);
    hdrint("NAXIS1",pixels);
    hdrint("NAXIS2",lines);
    hdrint("NAXIS3",naxis3);
    hdrlogical("BLOCKED",'T');
    hdrfloat("BZERO",b_zero);
    hdrfloat("BSCALE",bscale);
    hdrstr("BUNIT",bunit);
    hdrfloat("DATAMAX",thdr.img_max * bscale + b_zero);
    hdrint("BLANK",(int)blank);
    hdrfloat("CRVAL1",glong);
    hdrfloat("CRPIX1",crpix1);
    hdrstr("CTYPE1",ctype1sav);
    if (cdelt1 != 0.0)
	hdrfloat("CDELT1",cdelt1);
    hdrfloat("CRVAL2", glat);
    hdrfloat("CRPIX2", crpix2);
    hdrstr("CTYPE2", ctype2sav);
    if (cdelt2 != 0.0)
	hdrfloat("CDELT2", cdelt2);
    hdrfloat("CROTA2", twist);
    hdrfloat("CRVAL3", crval3);

    timenow = time((time_t *)NULL);
    tim = localtime(&timenow);
    strftime(timstr, 20, "%Y-%m-%d", tim);
    hdrstr("DATE", timstr);
    hdrstr("TELESCOP", "IRAS");
    hdrstr("OBJECT", object);
    comment("TAPE GENERATED BY IPAC JUPITER GRAPHICS PROGRAM");

    filp = filnam + strlen(filnam) - 1;
    while ((filp >= filnam) && (*filp != '/'))   /* search for last slash */
	filp--;
    hdrstr("FILENAME", ++filp);

    /* pad with spaces */
    bufp1 = hdr_bufp_end;
    while ((bufp1 < hdr_buf + bufsize))
	*bufp1++ = ' ';


    if (debug)
    {
	bufp1 = hdr_buf;
	prtbuf[79] = '\0';
	while (bufp1 < hdr_bufp_end)
	{
	    strncpy(prtbuf, bufp1, 79);
	    fprintf(debug_file, "%s\n", prtbuf);
	    bufp1 += 80;
	}
    }
    else
    {
	if(server_mode == FALSE)
	    printf("writing file %s to tape\n", filnam);
	else
	{
	    srv_string("filename", filnam);
	}
    }

    blocks = (hdr_bufp_end - hdr_buf) / 2880;
    if ((blocks * 2880) != (hdr_bufp_end - hdr_buf))
	blocks++;  /* round up to next integral number of blocks */
    for (i = 0, bufp1 = hdr_buf; i < blocks; i++)
    {

	if (!vax_bytes)
	    swab(bufp1, bufp1, 2880);  /* swap bytes */

	wrstat = write(mfd, bufp1, 2880);
	if (wrstat == -1)
	{
	    sperror("S error writing header on mag tape", errno);
	    return;
	}
	bufp1 += 2880;
    }
    free(hdr_buf);

    /* now copy pixels */
    if (debug)
	fprintf(debug_file, "seeking to %ld\n", thdr.img_start);
    lseek(fd, thdr.img_start, 0);

    count = bitpix / 8;   /* bytes per pixel */
    count = count * pixels * lines;   /* total bytes */
    while (count > 0)
    {
	rdstat = read(fd, pixbuf, 2880);
	if (bitpix != 8)
	    swab(pixbuf, pixbuf, 2880);  /* swap bytes */

	    if (vax_bytes)
	    {
		if (bitpix == 32)
		{
		    /* swap words */
		    pshort = (short *) pixbuf;
		    for(i = 0; i < rdstat; i += 4)
		    {
			temp = *pshort;
			*pshort = *(pshort+1);
			*(pshort+1) = temp;
			pshort += 2;
		    }
		}
	    }


	if (count < 2880)
	{
	    for (i=count; i<2880; i++)
		pixbuf[i] = 0;   /* fill with zeros */
	}
	wrstat = write(mfd, pixbuf, 2880);
	if (wrstat == -1)
	{
	    sperror("S error writing pixels on mag tape", errno);
	    return;
	}
	count -= 2880;
    }

    mstop.mt_op = MTWEOF;    /* write 2 EOFs  */
    mstop.mt_count = 2;
    wrstat = ioctl(mfd, MTIOCTOP, &mstop);
    if (wrstat == -1)
    {
	sperror("S error writing eof's on mag tape", errno);
	return;
    }
    mstop.mt_op = MTBSF;   /* backspace over one of them */
    mstop.mt_count = 1;
    wrstat = ioctl(mfd, MTIOCTOP, &mstop);
    if (wrstat == -1)
    {
	sperror("S error backspacing over eof on mag tape", errno);
	return;
    }

    tappos++;
    close(fd);

}
