/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#define _XOPEN_SOURCE
#include "skyview.h"
#include "fits.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <fcntl.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <sys/param.h>
#include <unistd.h>
#include <sys/wait.h>
#define TAPEBUFSIZ  5032  /* next multiple of 4 above 5030 */
#define PIXBUFSIZ 11520   /* worst case 8bit: each byte expands to 4 bytes */
			/* 4 * 2880 = 11520 */

extern int mfd, debug, server_mode, vax_bytes;
extern char server_str[];
extern FILE *session, *debug_file;
extern  char fi_name[];
int filnbr, tappos;

/* function prototypes */
#ifdef _NO_PROTO

static void ascii();
static int opentape();
static int endofhdr();

#else

static void ascii(char *p, int n);
static int opentape(void);
static int endofhdr(char *p);

#endif /* _NO_PROTO */

void
rdtape(argc, argv)
int argc;
char *argv[];
{
    short temp, *pshort;
    int blank, hist_shift;
    int hist[HISTSIZ2+1];
    int ri;
    int i, blksiz, diskfile;
    char *p;
    char tfilnam[MAXPATHLEN];
    char infilnam[MAXPATHLEN];
    char tmpstr[80];
    int infd, tfd = -1;
    int status, rdstat, wrstat;
    struct mtop  mstop;
    char buf[TAPEBUFSIZ];
    struct hdrstruct thdr;
    int j;
    int tmpnbr;
    int bitpix, pixels, lines, totpixels, min, max;
    double bscale, b_zero, datamax, datamin;
    int vicarflag = 0;

    if (argc > 4)
    {
	error1("Too many arguments to TR command");
	return;
    }

    if ((argc == 1) || (isint(argv[argc - 1])))
    {
	/* no args or last arg is not a filename */
	error1("output filename required by TR command");
	return;
    }

    if (strcmp(argv[argc - 1], "vicar") == 0)
    {
	argc--;
	vicarflag = 1;
    }

    /* get output filename */
    strcpy(tfilnam, expand_path(argv[argc - 1]));
    if (debug)
	fprintf(debug_file, "opening output file: %s\n", tfilnam);
    tfd = open(tfilnam, O_RDWR | O_CREAT | O_TRUNC, 0644);
    if (tfd < 0)
    {
	sperror(tfilnam, errno);
	return;
    }

    diskfile = FALSE;
    if (argc == 3)
    {
	if (isint(argv[1]) == FALSE)
	{
	    /* assume an input filename */
	    if (debug)
		fprintf(debug_file, 
		    "TR: input is from a disk file (not mag tape)\n");
	    diskfile = TRUE;
	    strcpy(infilnam, expand_path(argv[1]));
	    infd = open(infilnam, O_RDONLY , 0644);
	    if (infd < 0)
	    {
		sperror(infilnam, errno);
		/* close open output file */
		close(tfd);
		return;
	    }
	    if (vicarflag)
		blksiz = 360;
	    else
		blksiz = 2880;
	}
    }

    if (!diskfile)  /* if mag tape */
    {
	if (debug)
	    fprintf(debug_file, "TR: its mag tape\n");
	wait((int *) 0);   /* let child processes finish */
	if (mfd < 0)
	{
	    mfd = opentape();
	    if (mfd < 0)
	    {
		return;
	    }
	    tappos = 1;
	    filnbr = 0;   /* will be incremented by 1 (default) */
	}

	if (argc == 3)
	{
	    /* argv[1] must be a tape file number */
	    status = getint(argv[1], &tmpnbr);
	    if (status == FALSE)
		return;
	    if ((*argv[1] == '+') || (*argv[1] == '-'))
	    {
		filnbr += tmpnbr;
	    }
	    else
	    {
		filnbr = tmpnbr;
	    }
	}
	else
	{
	    filnbr++;   /* default to next file */
	}


	if (debug)
	    fprintf(debug_file, "seeking tape filenumber %d\n", filnbr);
	if (filnbr > tappos)
	{
	    mstop.mt_op = MTFSF;
	    mstop.mt_count = filnbr - tappos;
	    rdstat = ioctl(mfd, MTIOCTOP, &mstop);
	    if (rdstat == -1)
	    {
		sperror("S error advancing mag tape", errno);
		return;
	    }
	}
	else    /* filnbr <= tappos */
	{
	    if (filnbr == 1)
	    {
		rewind_tape();
		filnbr = 1;    /* got smashed by rewind */
	    }
	    else
	    {
		mstop.mt_op = MTBSF;
		mstop.mt_count = tappos - filnbr + 1;
		rdstat = ioctl(mfd, MTIOCTOP, &mstop);
		if (rdstat == -1)
		{
		    sperror("S error moving mag tape backwards", errno);
		    return;
		}

		mstop.mt_op = MTFSF;
		mstop.mt_count = 1;
		rdstat = ioctl(mfd, MTIOCTOP, &mstop);
		if (rdstat == -1)
		{
		    sperror("S error advancing mag tape", errno);
		    return;
		}
	    }
	}
	tappos = filnbr;
	blksiz = TAPEBUFSIZ;
	infd = mfd;
    }

    for (i=0; i<HISTSIZ2+1; i++)
	hist[i] = 0;

    if (diskfile)

	blksiz = read(infd, buf, blksiz);
    else
	blksiz = first_block(infd, buf, blksiz);
    /*
    if(server_mode == FALSE)
	printf("block size = %d\n",blksiz);
    */
    if (blksiz == -1)
    {
	sperror("read error in TR command", errno);
	return;
    }
    if (blksiz == -2)
    {
	return;
    }
    if (blksiz == 0)
    {
	error1("end of file\n");
	mstop.mt_op = MTBSF;
	mstop.mt_count = 1;
	ioctl(infd, MTIOCTOP, &mstop);
	return;
    }

    thdr.img_max = 0x80000001;
    thdr.img_min = 0x7fffffff;

    switch (blksiz)
    {
    case 2880:
	/* it is a FITS tape */
	datamax = 0;
	datamin = 0;
	bscale = 1;
	b_zero = 0;
	blank = -2000000000;
	thdr.tapetype = INVALID;  /* mark workfile invalid during tape read */
	thdr.hdr_start = sizeof(thdr);
	wrstat = write(tfd, (char *) &thdr, sizeof(thdr));
	if (wrstat < 0)
	{
	    sperror("mt error writing output file: ", errno);
	    if (diskfile)
		close(infd);
	    return;
	}

	wrstat = write(tfd, buf, blksiz);
	if (wrstat < 0)
	{
	    sperror("mt error writing output file: ", errno);
	    if (diskfile)
		close(infd);
	    return;
	}
	if ((parse_fits(buf) != SIMPLE) | (parse_fits(buf+80) != BITPIX))
	{
	    error1(
	    "file does not start with SIMPLE and BITPIX keywords in 80 character fields\n");
	    if (diskfile)
		close(infd);
	    return;
	}

	do
	{
	    p = buf;
	    for (i=0; i<36; i++, p += 80)
	    {
		if (parse_fits(p) == BITPIX)
		{
		    bitpix = atoi(p+10);
		    if ((bitpix != 8) && (bitpix != 16) &&
			(bitpix != 32) && diskfile)
		    {
			error1("cannot TR a non-integer file\n");
			close(infd);
			return;
		    }
		}
		if (parse_fits(p) == BLANK)
		    blank = atoi(p+10);
		if (parse_fits(p) == NAXIS1)
		    pixels = atoi(p+10);
		if (parse_fits(p) == NAXIS2)
		    lines = atoi(p+10);
		if (parse_fits(p) == BSCALE)
		    bscale = atof(p+10);
		if (parse_fits(p) == BZERO)
		    b_zero = atof(p+10);
		if (parse_fits(p) == DATAMAX)
		    datamax = atof(p+10);
		if (parse_fits(p) == DATAMIN)
		    datamin = atof(p+10);
		if (parse_fits(p) == END)
		{
		    if (debug)
			fprintf(debug_file, "got END keyword\n");
		    thdr.img_start = lseek(tfd,0L,1);
		    goto ended;
		}
	    }
	    if (diskfile)
		rdstat = read(infd, buf, blksiz);
	    else
		rdstat = next_block(infd, buf, blksiz);
	    if (rdstat == -1)
	    {
		sperror("read error in TR command", errno);
		return;
	    }

	    wrstat = write(tfd, buf, rdstat);
	    if (wrstat < 0)
	    {
		sperror("mt error writing output file: ", errno);
		if (diskfile)
		    close(infd);
		return;
	    }
	} while (rdstat == blksiz);
    ended:


	max = (datamax - b_zero) / bscale;
	min = (datamin - b_zero) / bscale;
	if ((datamax == 0) && (datamin == 0))
	{
	    switch (bitpix)
	    {
	    case 32:
		hist_shift = 20;
		break;
	    case 16:
		hist_shift = 4;
		break;
	    case 8:
		hist_shift = 0;
		break;
	    }
	    if (debug)
		fprintf(debug_file, "new forced hist_shift = %d\n", hist_shift);
	}
	else
	{
	    if (max > (-min))
		max = max;
	    else
		max = -min;
	    hist_shift = 0;
	    while ((max >> hist_shift) > HISTSIZ)
		++hist_shift;
	    if (debug)
		fprintf(debug_file, "new computed hist_shift = %d\n", hist_shift);
	}

	totpixels = pixels * lines;
	do
	{
	    if (diskfile)
		rdstat = read(infd, buf, blksiz);
	    else
		rdstat = next_block(infd, buf, blksiz);
	    if (rdstat == -1)
	    {
		sperror("S error reading mag tape", errno);
		return;
	    }

	    switch (bitpix)
	    {
	    case 32:

		if (vax_bytes)
		{
		    swab(buf, buf, rdstat);  /* swap bytes */
		    /* swap words */
		    pshort = (short *) buf;
		    for(i = 0; i < rdstat; i += 4)
		    {
			temp = *pshort;
			*pshort = *(pshort+1);
			*(pshort+1) = temp;
			pshort += 2;
		    }
		}
		break;
	    case 16:
		if (vax_bytes)
		{
		    swab(buf, buf, rdstat);  /* swap bytes */
		}
		break;
	    case 8:
		break;
	    }
	    wrstat = write(tfd, buf, rdstat);
	    if (wrstat < 0)
	    {
		sperror("mt error writing output file: ", errno);
		if (diskfile)
		    close(infd);
		return;
	    }

	    ri = rdstat * 8 / bitpix;  /* total pixels in buffer */
	    if (ri > totpixels)
		ri = totpixels;
	    totpixels -= ri;
	    get_hist(bitpix, (void *) buf, ri, &thdr.img_min, 
		&thdr.img_max, (double) blank, hist_shift, hist);
	} while (rdstat == blksiz);
	if (diskfile)
	{
	    close(infd);
	}
	else
	{
	    tappos++;
	}

	thdr.hist_bin_size = 1 << hist_shift;
	thdr.tapetype = FITS;
	thdr.hist_start = lseek(tfd, 0L, 1);
	write(tfd, (char *)hist, sizeof(hist));
	if (wrstat < 0)
	{
	    sperror("mt error writing output file: ", errno);
	    if (diskfile)
		close(infd);
	    return;
	}
	lseek(tfd, 0L, 0);
	wrstat = write(tfd, (char *) &thdr, sizeof(thdr));
	break;

    case 5030:
	if(server_mode == FALSE)
	    printf("starting 5030\n");
	/* it is a blocked VICAR tape */
	thdr.tapetype = INVALID;  /* mark workfile invalid during tape read */
	thdr.hdr_start = sizeof(thdr);
	wrstat = write(tfd, (char *) &thdr, sizeof(thdr));
	blank = -32768;

	if (!vax_bytes)
	    swab(buf, buf, 360);  /* swap bytes */

	ascii(buf,360);
	wrstat = write(tfd, buf, 360);
	memcpy(tmpstr, &buf[24], 8);
	tmpstr[8] = '\0';
	blksiz = atoi(tmpstr);
	lines = atoi(&buf[16]);
	p = buf;
	i = 0;
	while (endofhdr(p) == FALSE)
	{
	    p += 1006;
	    i ++;
	    if (i == 5)
	    {
		rdstat = read(infd, buf, 5030);
		if (rdstat == -1)
		{
		    sperror("S error reading header from mag tape", errno);
		    return;
		}
		p = buf;
		i = 0;
	    }

	    if (!vax_bytes)
		swab(p, p, 360);  /* swap bytes */

	    ascii(p, 360);
	    wrstat = write(tfd, p, 360);
	}
	/* now copy the pixels */
	thdr.img_start = lseek(tfd, 0L, 1);
	hist_shift = HIST_SHIFT;
	for (j=0; j < lines; j++)   /* for each line (each logical block) */
	{
	    p += 1006;
	    i ++;
	    if (i == 5)
	    {
		rdstat = read(infd, buf, 5030);
		if (rdstat == -1)
		{
		    sperror("S error reading pixels from mag tape", errno);
		    return;
		}
		p = buf;
		i = 0;
	    }
	    swab(p, p, blksiz);  /* swap bytes */
	    wrstat = write(tfd, p, blksiz);
	    /* now get histogram and find min, max */
	    ri = blksiz >> 1;  /* pixels = bytes / 2  */
	    get_hist(16, (void *) p, ri, &thdr.img_min, 
		&thdr.img_max, (double) blank, hist_shift, hist);
	} 
	thdr.hist_start = lseek(tfd, 0L, 1);
	write(tfd, (char *) hist, sizeof(hist));
	thdr.hist_bin_size = 1 << hist_shift;
	thdr.tapetype = VICAR;
	lseek(tfd,0L,0);
	wrstat = write(tfd, (char *) &thdr, sizeof(thdr));
	break;

    case 360:
	if(server_mode == FALSE)
	    printf("starting 360\n");
	/* it is an unblocked VICAR tape */
	thdr.tapetype = INVALID;  /* mark workfile invalid during tape read */
	thdr.hdr_start = sizeof(thdr);
	wrstat = write(tfd, (char *) &thdr, sizeof(thdr));
	blank = -32768;

#ifdef NOTDEF
	if (!vax_bytes)
	    swab(buf, buf, 360);  /* swap bytes */
#endif /* NOTDEF */

	ascii(buf, 360);
	memcpy(tmpstr, &buf[24], 8);
	tmpstr[8] = '\0';
	blksiz = atoi(tmpstr);
	if (blksiz != 360)
	{
	    if(server_mode == FALSE)
		printf("new blocksize=%d\n",blksiz);
	    if (diskfile)
	    {
		/* start over, so we are reading even blocks */
		lseek(infd, 0L, 0);
		rdstat = read(infd,buf,blksiz);
		ascii(buf, blksiz);
	    }
	}
	wrstat = write(tfd, buf, blksiz);
	/* the following needs fixing to handle big vicar headers */
	while (endofhdr(buf) == FALSE)
	{
	    {
	    rdstat = read(infd, buf, blksiz);
	    if (rdstat == -1)
	    {
		sperror("S error reading header from mag tape", errno);
		return;
	    }

#ifdef NOTDEF
	    if (!vax_bytes)
		swab(buf, buf, 360);  /* swap bytes */
#endif /* NOTDEF */

	    ascii(buf, blksiz);
	    wrstat = write(tfd, buf, blksiz);
	    }
	}

	/* now copy the pixels */
	thdr.img_start = lseek(tfd, 0L, 1);
	hist_shift = HIST_SHIFT;
	do
	{
	    rdstat = read(infd, buf, blksiz);
	    if (rdstat == -1)
	    {
		sperror("S error reading pixels from mag tape", errno);
		return;
	    }
	    swab(buf, buf, rdstat);  /* swap bytes */
	    write(tfd, buf, rdstat);
	    /* now do histogram and search for max, min */
	    ri = rdstat >> 1;  /* pixels = bytes / 2  */
	    get_hist(16, (void *) buf, ri, &thdr.img_min, 
		&thdr.img_max, (double) blank, hist_shift, hist);
	} while (rdstat == blksiz);
	tappos++;
	thdr.hist_start = lseek(tfd, 0L, 1);
	write(tfd, (char *) hist, sizeof(hist));
	thdr.hist_bin_size = 1 << hist_shift;
	thdr.tapetype = VICAR;
	lseek(tfd,0L,0);
	write(tfd, (char *) &thdr, sizeof(thdr));
	break;

    default:
	sprintf(server_str,
	"unknown tape type (not FITS or VICAR)   block size = %d\n",blksiz);
	error1(server_str);
	return;
	/*NOTREACHED*/
	break;
    }

    close(tfd);
    strcpy(fi_name, tfilnam);  /*  NOTE this may change when tape I/O */
				/*   is forked out */
    if(server_mode == TRUE)
    {
	srv_string("filename", fi_name);
    }
}

static int
endofhdr(p)
/*  look for end of VICAR header in 5 lines beginning at p */
char *p;
{
    int i;

    for (i=0; i<5; i++)
    {
	if (*(p+71) == 'L')
	    return(TRUE);
	p += 72;
    }
    return(FALSE);
}

static void
ascii(p,n)
/* convert n bytes of ebcdic to ascii */
char *p;
int n;
{
    int i, j;
    static int trans[256] = {0,1,2,3,0x9c,0x9,0x86,0x7f,0x97,0x8d,0x8e,0xb,0xc,
	0xd,0xe,0xf,0x10,0x11,0x12,0x13,0x9d,0x85,8,0x87,0x18,0x19,0x92,
	0x8f,0x1c,0x1d,0x1e,0x1f,
	0x80,0x81,0x82,0x83,0x84,0xa,0x17,0x1b,0x88,0x89,0x8a,0x8b,0x8c,5,
	6,7,0x90,0x91,0x16,0x93,0x94,0x95,0x96,4,0x98,0x99,0x9a,0x9b,0x14,0x15,
	0x9e,0x1a,0x20,0xa0,0xa1,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0x5b,
	0x2e,0x3c,0x28,0x2b,0x21,0x26,0xa9,0xaa,0xab,0xac,0xad,0xae,0xaf,
	0xb0,0xb1,0x5d,0x24,0x2a,0x29,0x3b,0x5e,0x2d,0x2f,0xb2,0xb3,0xb4,0xb5,
	0xb6,0xb7,0xb8,0xb9,0x7c,0x2c,0x25,0x5f,0x3e,0x3f,0xba,0xbb,0xbc,0xbd,
	0xbe,0xbf,0xc0,0xc1,0xc2,0x60,0x3a,0x23,0x40,0x27,0x3d,0x22,0xc3,0x61,
	0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,
	0xca,0x6a,0x6b,0x6c,0x6d,0x6e,0x6f,0x70,0x71,0x72,0xcb,0xcc,0xcd,
	0xce,0xcf,0xd0,0xd1,0x7e,0x73,0x74,0x75,0x76,0x77,0x78,0x79,
	0x7a,0xd2,0xd3,0xd4,0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xdb,0xdc,0xdd,
	0xde,0xdf,0xe0,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0x7b,0x41,0x42,
	0x43,0x44,0x45,0x46,0x47,0x48,0x49,0xe8,0xe9,0xea,0xeb,0xec,0xed,
	0x7d,0x4a,0x4b,0x4c,0x4d,0x4e,0x4f,0x50,0x51,0x52,0xee,0xef,
	0xf0,0xf1,0xf2,0xf3,0x5c,0x9f,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5a,
	0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0x30,0x31,0x32,0x33,0x34,0x35,0x36,
	0x37,0x38,0x39,0xfa,0xfb,0xfc,0xfd,0xfe,0xff};

    for (i=0; i<n; i++)
    {
	j = *p & 0xff;
	*p = trans[j];
	p++;
    }
}

void
rewind_tape()
{
    struct mtop  mstop;
    int pid, wrstat, closeit;

    wait((int *) 0);   /* let child processes finish */
    closeit = FALSE;
    if (mfd < 0)
    {
	mfd = opentape();  /* open it just for the rewind */
	closeit = TRUE;
    }
    if (mfd>=0)
    {
	fflush(session);
	pid = fork();
	if (pid == 0)   /* if this is the child */
	{
	    mstop.mt_op = MTREW;
	    mstop.mt_count = 1;
	    wrstat = ioctl(mfd, MTIOCTOP, &mstop);
	    if (wrstat == -1)
	    {
		sperror("S error rewinding mag tape", errno);
	    }
	    exit(0);     /* kill the child */
	}
	tappos = 1;
	filnbr = 0;   /* will be incremented by 1 (default) */
	if (closeit)
	{
	    close(mfd);
	    mfd = -1;
	}
    }
}

static int
opentape()
{
    int tfd;
    char *tapename;

    if ((tapename = getenv("STAPE")) == NULL)
	tapename = "/dev/rmt12";

    tfd = open(tapename, O_RDONLY, 0);
    if (tfd < 0)
    {
	sperror(tapename, errno);
    }
    return(tfd);
}
