/*
Copyright (C) 1996, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* movie.c */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>     /* includes strings.h, sys/time.h, and sys/file.h */
#include <X11/keysym.h>

#include "skyview.h"
#include "img_x.h"
#include "img.h"
#include "fits_table.h"
#include "im_proto.h"
#include "cam2fits.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <math.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/file.h>
#include <ctype.h>
#include <limits.h>

extern int control_c, reqband;
extern int be_quiet;
extern int server_mode;
extern FILE *session;
extern int debug, graphics, frame_advance, frame;
extern unsigned char *image_org;    /* original buffer for xwindow pixels */
extern unsigned char *image_zoomed;  /* buffer for xwindow pixels after zoom */
extern unsigned int win_width, win_height;  /* current dimensions of win */
extern unsigned int width_org, height_org;  /* dimensions of image_org */
extern int win_x_offset, win_y_offset;   /* and translation */
extern struct img *curr_img;
extern Display *display;
extern Window win;
extern XImage *theImage;
extern unsigned width_zoomed, height_zoomed;
extern GC image_gc;

static int zoom_flag;
static struct img *first_frame;
static char *zoom_factor;


/* function prototypes */
#ifdef _NO_PROTO
static void pause_it();
static void do_movie_cube();
#else
static void pause_it(double seconds);
static void do_movie_cube(char *filename);
#endif /* _NO_PROTO* */

/*ARGSUSED*/
void
movie(argc, argv)
int argc;
char *argv[];
{
    struct img *p;
    int last_frame, max_count, frame_index, count;
    unsigned int this_width, this_height;
    double seconds;

    if (!curr_img)
    {
	error1("no current image");
	return;
    }

    if (image_zoomed != NULL)
    {
	this_width = width_zoomed;
	this_height = height_zoomed;
    }
    else
    {
	this_width = width_org;
	this_height = height_org;
    }
    last_frame = max_FRAME();

    if (argc > 1)
	max_count = atoi(argv[1]);
    else
	max_count = 20;

    if (argc > 2)
	seconds = atof(argv[2]);
    else
	seconds = 0;

    enable_control_c();
    for (count = 0; count < max_count; count++)
    {
	for (frame_index = 1; frame_index <= last_frame; frame_index++)
	{
	    p = peek_FRAME(frame_index);
	    if (p)
	    {
		/* switch to alternate image (frame 2) */
		if (p->frameptr->image_zoomed)
		    theImage->data = (char *) p->frameptr->image_zoomed;
		else
		    theImage->data = (char *) p->frameptr->image_org;
		XPutImage(display, win, image_gc, theImage,
		    0,0,0,0, this_width, this_height);
		if (control_c)
		    break;
		pause_it(seconds);
	    }
	}
	if (control_c)
	{
	    if(server_mode == FALSE)
		printf("\nmovie interrupted\n");
	    fprintf(session, "\nmovie interrupted\n");
	    break;
	}
    }

    /* switch back to orig */
    if (image_zoomed != NULL)
	theImage->data = (char *) image_zoomed;
    else
	theImage->data = (char *) image_org;
    XPutImage(display, win, image_gc, theImage,
	0,0,0,0, this_width, this_height);

}

static void
pause_it(seconds)
double seconds;
{
    struct timeval tpstart, tpstop;
    long delta_sec, delta_usec;
    double delta_dsec;

    (void) gettimeofday(&tpstart, (struct timezone *) 0);
#ifdef DEBUG
    printf("start time = %ld seconds and %ld microseconds\n", 
	tpstart.tv_sec, tpstart.tv_usec);
#endif /* DEBUG */
    for (;;)
    {
	(void) gettimeofday(&tpstop, (struct timezone *) 0);
#ifdef DEBUG
	printf("stop time = %ld seconds and %ld microseconds\n", 
	    tpstop.tv_sec, tpstop.tv_usec);
#endif /* DEBUG */
	delta_usec = tpstop.tv_usec - tpstart.tv_usec;
	delta_sec = tpstop.tv_sec - tpstart.tv_sec;
	delta_dsec = delta_sec + (double) delta_usec / 1000000;
	if (delta_usec < 0)
	{
	    delta_sec--;
	    delta_usec += 1000000;
	}
#ifdef DEBUG
	printf("elapsed time = %ld.%06ld seconds  (%f float)\n", 
	    delta_sec, delta_usec, delta_dsec);
#endif /* DEBUG */
	if (delta_dsec > seconds)
	    break;
	if (control_c)
	    break;
    }
}



extern char server_str[];
extern char base_filename[];
extern int starting_index, ending_index, step_size;
int long_header = 0;
int nparms;
int silent = 1;
extern int info_request;
static int header_bytes;
extern int want_resets;



void
movie_prep(argc, argv)
int argc;
char **argv;
{
    int unit = 1;
    char *pt, *quote1, *quote2;
    int nchars;
    char header_filename[80];
    int fd;
    struct img tmpimg;


    first_frame = NULL;

    if (argc < 2)
    {
	/* no filename */
	error1("must supply a filename");
	return;
    }

    if (strcmp(argv[1], "-zp") == 0)
    {
	zoom_flag = 1;
	zoom_factor = argv[2];
	argv+=2;
	argc-=2;
    }
    else
	zoom_flag = 0;

    if (argc < 2)
    {
	/* no filename */
	error1("must supply a filename");
	return;
    }

    if (argc > 2)
	starting_index = atoi(argv[2]);
    else
	starting_index = 1;
    if (argc > 3)
	ending_index = atoi(argv[3]);
    else
	ending_index = INT_MAX;
    if (argc > 4)
	step_size = atoi(argv[4]);
    else
	step_size = 1;


    /* first check if it is a FITS cube */
    fd = wf_open(argv[1], &tmpimg, 1);
    if (fd >= 0)
    {
	if (tmpimg.filehdr.tapetype == IMFILE)
	    im_close(fd);
	else
	    close(fd);

	if ((tmpimg.naxis == 3) && (tmpimg.naxis3 > 0))
	{
	    /* it is a FITS cube */
	    if (ending_index == INT_MAX)
		ending_index = tmpimg.naxis3;
	    do_movie_cube(argv[1]);
	    return;
	}
    }

    /* it is not a FITS cube */
    topenr(unit, argv[1], &nparms);
    if (itstat())
    {
	sprintf(server_str, "problem opening tablefile %s: %s\n", 
	    argv[1], t_error());
	error1(server_str);
	return;
    }
    if (tgettyp(unit) != FITS_BINARY)
    {
	sprintf(server_str, "error:  file %s is not FITS BINARY\n", argv[1]);
	error1(server_str);
	return;
    }

    strcpy(base_filename, "/tmp/");


    tget_primary(unit, &pt, &header_bytes);
    for (;;)
    {
	if (strncmp(pt, "FILENAME", 8) == 0)
	{
		quote1 = strchr(pt+9, '\047');    /* look for ' */
		if (quote1 != NULL)
		{
		    quote2 = strchr(quote1 + 1, '\047');  /* second ' */
		    if (quote2 != NULL)
		    {
			nchars = quote2 - quote1 - 1;
			if (nchars > 79)
			    nchars = 79;
			memcpy(header_filename, quote1 + 1, nchars);
			header_filename[nchars] = '\0';
		    }
		}
	}
	if (strncmp(pt, "END", 3) == 0)
	    break;
	pt += 80;
    }

    if (strncmp(header_filename, "CCIM", 4) == 0)
	do_ccim(unit);
    else if (strncmp(header_filename, "CISP", 4) == 0)
    {
	want_resets = 0;
	do_cisp(unit);
    }
    else if (strncmp(header_filename, "CIER", 4) == 0)
	do_cier(unit);
    else if (strncmp(header_filename, "CMAP", 4) == 0)
	do_cmap(unit);
    else if (strncmp(header_filename, "CMOS", 4) == 0)
	do_cmap(unit);
    /* FILENAME header card not as expected - use input filename */
    else if (strncmp(argv[1], "ccim", 4) == 0)
	do_ccim(unit);
    else if (strncmp(argv[1], "cisp", 4) == 0)
    {
	want_resets = 0;
	do_cisp(unit);
    }
    else if (strncmp(argv[1], "cier", 4) == 0)
	do_cier(unit);
    else if (strncmp(argv[1], "cmap", 4) == 0)
	do_cmap(unit);
    else if (strncmp(argv[1], "cmos", 4) == 0)
	do_cmap(unit);
    else
    {
	error1("ERROR: input file is not in recognized format");
	return;
    }

    tcls(unit);
    printf("\n");
}

static void
do_movie_cube(filename)
char *filename;
{
    char *cmdv[5];
    int cmdc;
    int save_frame_advance, save_reqband, save_frame;
    int dot_count;
    int this_index;

    enable_control_c();

    be_quiet = TRUE;
    save_reqband = reqband;
    save_frame_advance = frame_advance;
    save_frame = frame;  /* BUG FIX */
    frame_advance = TRUE;
    for (this_index = 1; this_index <= ending_index; this_index++)
    {
	if ((this_index < starting_index) || (this_index > ending_index) ||
	    ((this_index - starting_index) % step_size))
	    continue;

	if (first_frame != NULL) /* paint only first frame onto the screen */
	    graphics = 0;
	reqband = this_index;
	cmdv[0] = "paint";
	cmdv[1] = filename;
	cmdc = 2;
	rddisk(cmdc, cmdv);
	if (control_c)
	    return;

	if (zoom_flag)
	{
	    cmdv[0] = "zp";
	    cmdv[1] = zoom_factor;
	    cmdc = 2;
	    zoom_pan(cmdc, cmdv);

	    cmdv[0] = "border";
	    cmdv[1] = "0";
	    cmdc = 2;
	    border(cmdc, cmdv);
	}
	if (first_frame == NULL)
	{
	    first_frame = curr_img;
	    dot_count = 0;
	    printf("collecting frames (one dot per frame)\n.");
	    fflush(stdout);
	}
	else
	{
	    printf(".");
	    dot_count++;
	    if ((dot_count % 50) == 0)
		printf("\n");
	    fflush(stdout);
	}
    }

    if (first_frame != NULL)
    {
	/* make sure that the painted image is the current one */
	fill_glob(first_frame);
    }

    graphics = 1;
    frame_advance = save_frame_advance;
    frame = save_frame;  /* BUG FIX */
    reqband = save_reqband;
    be_quiet = FALSE;
    printf("\n");
}

void
file_done(filename)
char *filename;
{
    char *cmdv[5];
    int cmdc;
    int save_frame_advance;
    static int dot_count;

    be_quiet = TRUE;
    save_frame_advance = frame_advance;
    frame_advance = TRUE;
    if (first_frame != NULL)   /* paint only the first frame onto the screen */
	graphics = 0;
    cmdv[0] = "paint";
    cmdv[1] = filename;
    cmdc = 2;
    rddisk(cmdc, cmdv);

    if (zoom_flag)
    {
	cmdv[0] = "zp";
	cmdv[1] = zoom_factor;
	cmdc = 2;
	zoom_pan(cmdc, cmdv);

	cmdv[0] = "border";
	cmdv[1] = "0";
	cmdc = 2;
	border(cmdc, cmdv);
    }
    unlink(filename);
    if (first_frame == NULL)
    {
	first_frame = curr_img;
	if (!server_mode)
	{
	    dot_count = 0;
	    printf("collecting frames (one dot per frame)\n.");
	    fflush(stdout);
	}
    }
    else
    {
	/* make sure that the painted image is the current one */
	fill_glob(first_frame);  
	if (!server_mode)
	{
	    printf(".");
	    dot_count++;
	    if ((dot_count % 50) == 0)
		printf("\n");
	    fflush(stdout);
	}
    }

    graphics = 1;
    frame_advance = save_frame_advance;
    be_quiet = FALSE;
}

/*ARGSUSED*/
int im_wkey_raw(fid, ptr)
int fid;
char *ptr;
{
    return(0);
}

