/*
Copyright (C) 1992, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* machine dependent code for Jupiter systems graphics hardware */
#include <stdio.h>
#include "skyview.h"
#include "img_jup.h"
#include "img.h"
#include "parse.h"
#include <fcntl.h>

#if defined (VAXJUP) || defined (SUNJUP)
#define CONSOLE       "/dev/console"
#endif VAXJUP or SUNJUP

#ifdef OLDJUP
#define CONSOLE       "/dev/jk01"
#endif OLDJUP

#include "jgio.h"
#include "jconfig.h"

static double   dtr = 1.7453292519943e-2;

JFILE *J12file;
struct framestruct *frameptr;
extern FILE *session;
extern int frame, frame_advance;
int force_graphics = FALSE;   /* flag if user demands graphics */
extern int debug, server, graphics;
int x_zoom, y_zoom;
extern int x_orig, y_orig;    /* values given to last j_zoom function call */
extern int llow_save, hhigh_save;
extern char *image_to_sky();
extern char plane;
int ov_color;
extern double cdelt1;
extern int minx_next, miny_next;
extern struct img *curr_img;
extern char version[];    /* Makefile version date */

init_graphics()
{
    int fd;
    struct jggxattr ja;
    int j_get_result;

    /* see if this user owns the console */
    if ((fd = open(CONSOLE, O_RDONLY, 0644)) >= 0)
	close(fd);
    else
    {
	if (!force_graphics)
	    return(FALSE);
    }

    J12file = jginit(0, 0, CLT8, 1, 0, 4);
    if (J12file == NULL)
    {
	perror("\nFAILURE in Jupiter graphics initialization");
	printf("\7\n");
	return(FALSE);
    }

    j_vpmask(0);         /* turn off clipping on vax */

    /* fetch existing zoom and origin */
    j_get_result = j_get_attr(&ja);
    if (debug)
    {
	printf("j_get_attr result = %d\n", j_get_result);
	printf("xzoom = %f  yzoom = %f  origins: horiz = %d  vert = %d\n", ja.ja_xzoom, ja.ja_yzoom, ja.ja_horg, ja.ja_vorg);
    }
    x_zoom = ja.ja_xzoom;
    y_zoom = ja.ja_yzoom;
    x_orig = ja.ja_horg;
    y_orig = ja.ja_vorg - (1024 / y_zoom) + 1;
    j_zoom(x_orig, y_orig, x_zoom, y_zoom);  /* make sure exactly there */

    return(TRUE);
}

close_graphics()
{
    jfclose(J12file);
}

flush_graphics()
{
    jfflush(J12file);
}

show_frame()
{
    return(TRUE);
}

change_win_name(window_name)
char *window_name;
{
}

write_AOI(x1, y1, x2, y2, bufp)
int x1, y1, x2, y2;
char *bufp;
{
    int nbr;

    nbr = j_area(x1, y1 , x2, y2);
    j_wrt_image(bufp,nbr);
}


/******************************************************************/
/* 								  */
/* MARK_THE_SPOT                                                  */
/* Puts a green cursor mark at x,y                                */
/* Resets graphics to normal image mode (vice overlay mode)
/* 								  */
/******************************************************************/

mark_the_spot(x, y)
int x, y;
{
	    overlay_setup('g');
	    overlay_color(2);
	    j_mov(x - 6, y);
	    j_dva(x + 6, y);

	    j_mov(x, y - 6);
	    j_dva(x, y + 6);

	    j_dca(x, y, 0);	/* turn off cursor */
	    image_setup(plane);
	    jfflush(J12file);
}

set_overlay()
{
    clt_vals r_o[16], g_o[16], b_o[16];

    r_o[0] = 0;
    g_o[0] = 0;
    b_o[0] = 0;
    r_o[1] = 255;
    g_o[1] = 0;
    b_o[1] = 0;
    r_o[2] = 0;
    g_o[2] = 255;
    b_o[2] = 0;
    r_o[3] = 255;
    g_o[3] = 255;
    b_o[3] = 0;
    r_o[4] = 0;
    g_o[4] = 0;
    b_o[4] = 255;
    r_o[5] = 255;
    g_o[5] = 0;
    b_o[5] = 255;
    r_o[6] = 0;
    g_o[6] = 255;
    b_o[6] = 255;
    r_o[7] = 255;
    g_o[7] = 255;
    b_o[7] = 255;
    r_o[8] = 255;
    g_o[8] = 255;
    b_o[8] = 255;
    r_o[9] = 255;
    g_o[9] = 255;
    b_o[9] = 255;
    r_o[10] = 255;
    g_o[10] = 255;
    b_o[10] = 255;
    r_o[11] = 255;
    g_o[11] = 255;
    b_o[11] = 255;
    r_o[12] = 255;
    g_o[12] = 255;
    b_o[12] = 255;
    r_o[13] = 255;
    g_o[13] = 255;
    b_o[13] = 255;
    r_o[14] = 255;
    g_o[14] = 255;
    b_o[14] = 255;
    r_o[15] = 255;
    g_o[15] = 255;
    b_o[15] = 255;

    j_ov_sset(0, 16, r_o, g_o, b_o);
}

off_X_overlay(color, on_off)
char *color;
int on_off;
{
}

off_all_X_overlay(on_off)
int on_off;
{
}

set_overlay_mask(mval)
int mval;
{
    j_ormask(mval);
}

set_clt(J_reds, J_greens, J_blues)
clt_vals *J_reds, *J_greens, *J_blues;
{
    j_cltset(0, 256, J_reds, J_greens, J_blues);
}


set_csr()
{
    int i, j;
    static int cursor[400];

#ifdef OLDJUP   /* old J12 library (when we had 68000) */
      for (i=5; i<18; i++)
	    cursor[20*i+11] = 1;
      for (j=5; j<18; j++)
	    cursor[220+j] = 1;
      cursor[20*11 + 11] = 0;
#endif OLDJUP
#if defined (VAXJUP) || defined (SUNJUP)
      for (i=5; i<18; i++)
	    cursor[20*i+10] = 1;
      for (j=4; j<17; j++)
	    cursor[220+j] = 1;
      cursor[20*11 + 10] = 0;
#endif VAXJUP || SUNJUP
      j_csrset(cursor);
}


/*******************************************************************/
/*                                                                 */
/* ST_BALL uses the trackball to stretch the color table           */
/*                                                                 */
/*******************************************************************/

st_ball()
{
    int dx, dy;
    char comm[20];
    int low, high;
    int x, y;
    int temp, done, what;

    low = llow_save;
    high = hhigh_save;

    printf("set stretch using trackball, then hit CR\n");

    x = 0;
    y = 0;

    done = 0;
    while (!done) 
    { 
	wait_for_event(comm, &what, &x, &y, &dx, &dy, 0);

	switch(what)
	{
	case Button1:
	    temp = low;
	    low = high;
	    high = temp;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    low = 0;
	    high = 255;
	    break;
	case POINTERMOTION:
	    low  += dy;
	    high += dy;
	    low  -= dx;
	    high += dx;
	    break;
	} 

	if (!server)
	{
	    printf("stretch: %d to %d   \r", low, high);
	    fflush(stdout);
	}
	stretch(low,high);
    } 

    printf("\nfinal stretch is %d to %d\n",low,high);
    fprintf(session, "final stretch is %d to %d\n",low,high);
}

/*******************************************************************/
/*                                                                 */
/* FOV_JOY uses the trackball to move the field_of_view graphic    */
/*                                                                 */
/*******************************************************************/

#define POS 0
#define ROT 1

fov_joy(rot_angle, d_coord)
double *rot_angle;
double d_coord[2];
{
    int done, what;
    int x, y;
    int dx, dy;
    int cen_size;
    int x_box_size, y_box_size, max_x_size, max_y_size;
    char comm[20];
    int x1cur, y1cur, x2cur, y2cur;
    int i_coord[2];
    double xdist, ydist;
    int x_center, y_center;
 
    set_csr();


    /* Go into rubber band mode */ 

    overlay_setup('r');
    cen_size = POS;

    get_center_xy(i_coord);
    x_center = i_coord[0];
    y_center = i_coord[1];

    x = x_center;
    y = y_center;

    done = 0;
    while (!done) 
    { 


	j_dca(x_center, y_center, 1);       /* place cursor */
	j_sec(1);                           /* draw graphics */ 
	fov_plot((double) x_center, (double)y_center, *rot_angle, 1);
	jfflush(J12file); 

	wait_for_event(comm, &what, &x, &y, &dx, &dy, 0);

	/* undraw old graphic */
	j_sec(0);      /* draw black vectors  */ 
	fov_plot((double) x_center, (double)y_center, *rot_angle, 1);
	jfflush(J12file); 

	switch(what)
	{
	case Button1:
	    cen_size = POS;
	    x = x_center;   /* avoid a jump */
	    y = y_center;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = ROT;
	    break;
	case POINTERMOTION:

	    if(cen_size == POS)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == ROT)
	    {
		*rot_angle -= (double)dx * dtr;
	    }

	    break;
	} 
    } 

    /* normalize */
    x_center = (x_center + 2560) % 1280;
    y_center = (y_center + 2048) % 1024;

    d_coord[0] = x_center;
    d_coord[1] = y_center;

    if(debug == TRUE)
	printf("fov_joy: coords %g,%g\n", 
	    x_center, y_center);


    j_dca(x,y,0);      /* Turn off cursor */

    overlay_setup('g');
    j_sec(2);      /* draw the vector in  */ 
    fov_plot((double) x_center, (double)y_center, *rot_angle, 0);

    image_setup(plane);   /* reset enabled planes */ 
    jfflush(J12file); 
}

trackball_zoom()
{
    int inc;
    int xc, yc;
    int what, done, update;
    char comm[20], *coord_str;
    int i_coord[2];
    int dx, dy, xdummy = 0, ydummy = 0;


    zoom_setup();   /* turn on all planes */
    printf("set zoom and pan, then hit CR\n");

    done = 0;
    while (!done)
    {
	update = FALSE;
	inc = 0;

	wait_for_event(comm, &what, &xdummy, &ydummy, &dx, &dy, 0);

	switch(what)
	{
	    case Button1:
		if (x_zoom < 16)
		    inc++;
		break;
	    case Button2:
	    case TTYINPUT:
		done = 1;
		break;
	    case Button3:
		if (x_zoom > 1)
		    inc--;
		break;
	    case POINTERMOTION:
		x_orig -= dx;
		y_orig -= dy;
		update = TRUE;
		break;
	}

	/* update origin if zoom changes */

	if(inc != 0)
	{
	    xc = x_orig + 640/x_zoom;
	    yc = y_orig + 512/y_zoom;

	    x_zoom += inc;
	    y_zoom = x_zoom;

	    x_orig = xc - 640/x_zoom;
	    y_orig = yc - 512/y_zoom;

	    update = TRUE;
	}


	/* If anything has changed, update the zoom and origin */

	if(update == TRUE)
	{
	    while(x_orig > 1279) 
		x_orig -= 1280; 
	    while(x_orig < 0)
		x_orig += 1280;

	    while(y_orig > 1023)
		y_orig -= 1024;
	    while(y_orig < 0)
		y_orig += 1024;

	    if(debug == TRUE)
		printf("zoom_pan: x=%d y=%d zoom=%d\n", 
		    x_orig, y_orig, x_zoom);

	    j_zoom(x_orig, y_orig, x_zoom, y_zoom);
	    jfflush(J12file);
	}
    }

    if(debug == TRUE) 
	printf("zoom_pan: zoom=%d, origin %d,%d\n", x_zoom, x_orig, y_orig);

    image_setup(plane);

    center_img();
    i_coord[0] = (x_orig + 640 / x_zoom) % 1280;
    i_coord[1] = (y_orig + 512 / y_zoom) % 1024;
    SC_to_IM(i_coord);

    coord_str = image_to_sky(i_coord);
    printf("Zoom factor %d\nCentered on %s", x_zoom, coord_str);
    fprintf(session, "Zoom factor %d\nCentered on %s", x_zoom, coord_str);
}

do_zoom_pan(argc, tmpzoom, xc, yc, do_expose)
int argc, *xc, *yc;
int do_expose;     /* ignored in Jupiter version */
double *tmpzoom;
{
    if ((*tmpzoom < 1) || (*tmpzoom > 16))
    {
	printf("zoom must be 1 to 16\n");
	*tmpzoom = (double) x_zoom;
	error();
	return;
    }

    if (argc == 2)    /* if he specified only the zoom factor */
    {
	/* get old center x,y  using old zoom factor */
	*xc = x_orig + (640 / x_zoom);
	*yc = y_orig + (512 / y_zoom);
	/* normalize */
	if (*xc < 0)
	    *xc += 1280;
	if (*yc < 0)
	    *yc += 1024;
	*xc = *xc % 1280;
	*yc = *yc % 1024;
    }
    x_zoom = *tmpzoom;
    y_zoom = *tmpzoom;

    /* find new screen origin using new zoom factor */
    x_orig = *xc - (640 / x_zoom);
    y_orig = *yc - (512 / x_zoom);

    /* normalize */
    if (x_orig < 0)
	x_orig += 1280;
    if (y_orig < 0)
	y_orig += 1024;
    x_orig = x_orig % 1280;
    y_orig = y_orig % 1024;

    if (debug == TRUE)
	printf("zoom_pan:  x_orig=%d  y_orig=%d  x_zoom=%d\n",
	x_orig, y_orig, x_zoom);

    zoom_setup();   /* turn on all planes */
    j_zoom(x_orig, y_orig, x_zoom, y_zoom);
    jfflush(J12file);
    image_setup(plane);
}


initial_zoom()
{
    x_zoom = 2;
    y_zoom = 2;
    if (graphics)
    {
	zoom_setup();
	j_zoom(x_orig, y_orig, x_zoom, y_zoom);
	image_setup(plane);
    }
}

/***********************************************************************/
/*                                                                     */
/* GET_CENTER_XY                                                       */
/* Finds screen coords of center of visible screen                     */
/*                                                                     */
/***********************************************************************/

get_center_xy(i_coord)
int i_coord[2];
{
    i_coord[0] = x_orig + (640 / x_zoom);
    i_coord[1] = y_orig + (512 / y_zoom);
    /* normalize */
    i_coord[0] = (i_coord[0] + 2560) % 1280;
    i_coord[1] = (i_coord[1] + 2048) % 1024;
}

/***********************************************************************/
/*                                                                     */
/* CENTER_TO_ORIG                                                      */
/* Converts an x,y of center of screen to x_orig, y_orig               */
/*                                                                     */
/***********************************************************************/

center_to_orig(i_coord)
int i_coord[2];
{
    i_coord[0] = i_coord[0] - (640 / x_zoom);
    i_coord[1] = i_coord[1] - (512 / x_zoom);

    /* normalize */
    i_coord[0] = (i_coord[0] + 2560) % 1280;
    i_coord[1] = (i_coord[1] + 2048) % 1024;
}


get_screen_height()     /* return window height in SC coordinates */
{
    return(1024 / y_zoom);
}

get_screen_width()     /* return window width in SC coordinates */
{
    return(1280 / x_zoom);
}

get_screen_bottom()     /* return bottom edge in SC coordinates */
{
    return(y_orig);
}

get_screen_left()     /* return left edge in SC coordinates */
{
    return(x_orig);
}

overlay_color(i)
int i;
{
    j_sec(i);
    ov_color = i;
}

overlay_mov(x,y)
int x, y;
{
    j_mov(x,y);
}

overlay_draw(x,y)
int x, y;
{
    j_dva(x,y);
}

overlay_dmov(x,y)
double x, y;
{
    j_mov( (int) x, (int) y);
}

overlay_dcircle(radius)
double radius;
{
    j_dfc(radius);
}

overlay_ddraw(x,y)
double x, y;
{
    j_dva( (int) x, (int) y);
}

set_current_window()
{
    if (curr_img == NULL)
	return(0);
    else
	return(1);
}
update_cur_window()
{
}

/******************************************************************/
/*                                                                */
/* FRAME_CMD                                                      */
/* changes the frame, preserving the zoom and center position     */
/*                                                                */
/******************************************************************/


frame_cmd(argc, argv)
int argc;
char *argv[];

{
    int new_frame, old_zoom, old_minx, old_miny;
    int i_coord[2];
    char *coord_str;

    old_zoom = x_zoom;
    old_minx = minx_next;
    old_miny = miny_next;


    if(argc > 3)
    {
	printf("wrong number of arguments to FR command\n");
	error();
	return;
    }


    if (argc == 2)
    {
	if (strncmp(argv[1], "ad", 2) == 0)
	{
	    frame_advance = TRUE;
	    return;
	}
	else if (strncmp(argv[1], "noad", 4) == 0)
	{
	    frame_advance = FALSE;
	    return;
	}
    }

    if(argc >= 2)
    {
	new_frame = atoi(argv[1]);
	if(new_frame >= 0 && new_frame <=4)
	{
	    frame = new_frame;
	}
	else
	{
	    printf("illegal frame number\n");
	    error();
	    return;
	}

	if(argc == 3)    /* change the zoom if he gave one */
	    {
	    x_zoom = atoi(argv[2]);
	    y_zoom = x_zoom;
	    }

	switch(frame)
	{
	case 0:
	    break;
	case 1:
	    minx_next = 0;
	    miny_next = 0;
	    break;

	case 2:
	    minx_next = 640;
	    miny_next = 0;
	    break;

	case 3:
	    minx_next = 0;
	    miny_next = 512;
	    break;

	case 4:
	    minx_next = 640;
	    miny_next = 512;
	    break;
	}


	if(x_zoom != 0)
	{
	    x_orig = x_orig + (640 / old_zoom) - old_minx
		  - (640 /   x_zoom) +     minx_next;

	    y_orig = y_orig + (512 / old_zoom) - old_miny
		  - (512 /   y_zoom) +     miny_next;

	    if(x_orig < 0)
		x_orig += 1280;

	    if(y_orig < 0)
		y_orig += 1024;

	    x_orig = x_orig % 1280;
	    y_orig = y_orig % 1024;
	}
	else     /*  if(x_zoom == 0)  */
	{
	    x_zoom = 2;
	    y_zoom = 2;
	    x_orig = minx_next;
	    y_orig = miny_next;
	}


	if(frame == 0)
	{
	    x_orig = 0;
	    y_orig = 0;
	    x_zoom = 1;
	    y_zoom = 1;
	    minx_next = 0;
	    miny_next = 0;
	}

	zoom_setup();
	j_zoom(x_orig, y_orig, x_zoom, y_zoom);
	image_setup(plane);
	jfflush(J12file);
	center_img();
    }

    i_coord[0] = x_orig + (640 / x_zoom);
    i_coord[1] = y_orig + (512 / y_zoom);
    SC_to_IM(i_coord);
    coord_str = image_to_sky(i_coord);
    printf("frame %d\nzoomed %d,  screen centered on %s\n", frame, x_zoom, coord_str);


}

next_frame()
{
    char buf[10], *cmdv[10];

    frame++;
    if (frame > 4)
	frame = 1;
    cmdv[0] = "frame";
    cmdv[1] = buf;
    sprintf(cmdv[1], "%d", frame);
    frame_cmd(2, cmdv);
}

origin(argc, argv)
int argc;
char *argv[];
{
    if (argc == 1)
    {
	printf ("origin at x=%d  y=%d\n", minx_next, miny_next);
	fprintf (session, "origin at x=%d  y=%d\n", minx_next, miny_next);
	return;
    }
    if (argc == 3)
    {
	if ((*argv[1] == '+') || (*argv[1] == '-'))
	    minx_next = minx_next + atoi(argv[1]);
	else
	    minx_next = atoi(argv[1]);
	if ((*argv[2] == '+') || (*argv[2] == '-'))
	    miny_next = miny_next + atoi(argv[2]);
	else
	    miny_next = atoi(argv[2]);
    }
    else
    {
	printf("wrong number of arguments to OR command\n");
	error();
    }
}

border()
{
}

mag_cmd()
{
    printf("magnifier window only available in X windows version\n");
    error();
}

smongo()
{
}

sync_command()
{
}

do_sync()
{
}

do_sleep(argc, argv)
int argc;
char *argv[];
{
    int n;

    if (getint(argv[1], &n))
	sleep(n);
}


/*********************************************************************/
/* 								     */
/* WIN_to_SC                                                         */
/* converts window coordinates to screen coordinates                 */
/* 								     */
/* This is a nop on Jupiter                                          */
/* 								     */
/*********************************************************************/

WIN_to_SC(i_coord)
int i_coord[2];
{
    if (debug)
    {
	printf("WIN_to_SC: in x=%d, y=%d\n", i_coord[0], i_coord[1]);
	printf("WIN_to_SC: out x=%d, y=%d\n", i_coord[0], i_coord[1]);
    }
}

/*********************************************************************/
/* 								     */
/* dWIN_to_SC                                                        */
/* converts window coordinates to screen coordinates                 */
/* (double precision)
/* 								     */
/* This is a nop on Jupiter                                          */
/* 								     */
/*********************************************************************/

dWIN_to_SC(d_coord)
double d_coord[2];
{
    if (debug)
    {
	printf("dWIN_to_SC: in x=%g, y=%g\n", d_coord[0], d_coord[1]);
	printf("dWIN_to_SC: out x=%g, y=%g\n", d_coord[0], d_coord[1]);
    }
}

/*********************************************************************/
/* 								     */
/* SC_to_WIN                                                         */
/* converts screen coordinates to window coordinates                 */
/* 								     */
/* This is a nop on Jupiter                                          */
/* 								     */
/*********************************************************************/
SC_to_WIN(i_coord)
int i_coord[2];
{
    if (debug)
    {
	printf("SC_to_WIN: in x=%d, y=%d\n", i_coord[0], i_coord[1]);
	printf("SC_to_WIN: out x=%d, y=%d\n", i_coord[0], i_coord[1]);
    }
}

/*********************************************************************/
/* 								     */
/* dSC_to_WIN                                                        */
/* converts screen coordinates to window coordinates                 */
/* (double precision)						     */
/* 								     */
/* This is a nop on Jupiter                                          */
/* 								     */
/*********************************************************************/

dSC_to_WIN(d_coord)
double d_coord[2];
{
    if (debug)
    {
	printf("dSC_to_WIN: in x=%g, y=%g\n", d_coord[0], d_coord[1]);
	printf("dSC_to_WIN: out x=%g, y=%g\n", d_coord[0], d_coord[1]);
    }
}

fill_img_frame()
{
}

fill_glob_frame()
{
}

repaint_x(min_samp, max_samp, min_line, max_line)
int min_samp, max_samp, min_line, max_line;  /* IM coordinates */
{
}

/********************************************************************/
/*                                                                  */
/*  CENTER_IMG                                                      */
/*  Finds the image occupying                                       */
/*  the center of the screen.  If no such image, then the image     */
/*  whose center is closest to the center of the screen.            */
/*  Calls FIND_SC                                                   */
/*  Copies values from the structure to global variables.           */
/*                                                                  */
/*  (used only by the Jupiter routines)                             */
/*                                                                  */
/********************************************************************/

void
center_img()
{
    struct img *p, *psave;
    int x, y, dist, xdist, ydist, distsav, x_cent, y_cent;

    if (head_img.rlink == &head_img)
	return;

    /* first set x,y to center of screen */
    x = (640 / x_zoom + x_orig) % 1280;
    y = (512 / y_zoom + y_orig) % 1024;
    p = find_SC(x,y);
    if (debug)
	fprintf(debug_file, "center_img: x=%d y=%d p=%d\n", x, y, p);
    if (p != NULL)
	return;

    p = head_img.rlink;
    psave = p;
    distsav = 10000000;
    while (p != &head_img)
    {
	x_cent = (p->maxx + p->minx) / 2;
	y_cent = (p->maxy + p->miny) / 2;
	xdist = x - x_cent;
	ydist = y - y_cent;

	/* now take care of wraparound */
	if (xdist > 640)
	    xdist = xdist - 1280;
	if (xdist < -640)
	    xdist = xdist + 1280;
	if (ydist > 512)
	    ydist = ydist - 1024;
	if (xdist < -512)
	    ydist = ydist + 1024;

	dist = xdist*xdist + ydist*ydist;
	if (debug)
	{
	    fprintf(debug_file, 
	    "minx=%d maxx=%d miny=%d maxy=%d\n", p->minx,p->maxx,p->miny,p->maxy);
	    fprintf(debug_file, "dist=%d distsav=%d p=%d\n", dist, distsav, p);
	}
	if (dist < distsav)
	{
	    psave = p;
	    distsav = dist;
	}
	p = p->rlink;
    }
    fill_glob(psave);
    if (debug)
	fprintf(debug_file, "center_img: new imgp=%d\n", psave);
    return;
}
