/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/*  routines which call image access library */

#include "skyview.h"
#include "img.h"
#include "fits.h"
#include "im_proto.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <sys/time.h>
#include <sys/resource.h>


extern int debug, server_mode, conserve_mem, use_projtype, default_u16;
extern FILE *debug_file;
extern int bitpix, blank_set, be_quiet;
extern char server_str[];
extern char version[];          /* Makefile version date */
extern char filnam[], hist_filnam[];
extern double report_blank, blank_flag;

struct pixels {
    struct pixels *llink, *rlink;
    void *pixelptr;
    int dirty;
    double report_blank;
    int nofile;
    char f_name_save[MAXPATHLEN];
    };


static struct pixels head_pix = {&head_pix, &head_pix};

static double   rtd = 180/M_PI;

/* function prototypes */
#ifdef _NO_PROTO

static struct pixels *new_pixels();
static struct pixels *get_pixel_bufs();
static void delete_pixels();
static void delete_all_pixels();
static int delete_oldest_pixels();
static struct pixels *find_pixel_FILNAM();

#else

static struct pixels *new_pixels(void);
static struct pixels *get_pixel_bufs(int totbytes);
static void delete_pixels(struct pixels *p);
static void delete_all_pixels(void);
static int delete_oldest_pixels(struct pixels *save_this_p);
static struct pixels *find_pixel_FILNAM(char *fname);

#endif /* _NO_PROTO */

void
imhdr(fd, imgp)
int fd;
struct img *imgp;
{
    /* process entire im header */
    int axes_reversed;
    double temp;
    double determinant;
    int imstat, file_type;
    char *ptmp, tmpstr[80];
    float fblank;
    char decsign[80];
    double rah,ram,ras, dsign,decd,decm,decs;
    double dec_deg,ra_hours;
    double dtr = M_PI/180;
    int got_cd1_1, got_cd1_2, got_cd2_1, got_cd2_2;
    double cd1_1, cd1_2, cd2_1, cd2_2;
    int got_pc1_1, got_pc1_2, got_pc2_1, got_pc2_2;
    double pc1_1, pc1_2, pc2_1, pc2_2;

    int i,j;
    char keyword[10];

    if (debug)
	fprintf(debug_file, "imhdr:  entry\n");
    /* set defaults */
    set_header_defaults(imgp);
    axes_reversed = 0;

    imstat = im_rkey_i(fd, &file_type, "filetype");
    imstat = im_rkey_i(fd, &imgp->bitpix, "BITPIX");
    imstat = im_rkey_i(fd, &imgp->naxis, "NAXIS");
    imstat = im_rkey_i(fd, &imgp->pixels, "NAXIS1");
    imstat = im_rkey_i(fd, &imgp->lines, "NAXIS2");
    imstat = im_rkey_i(fd, &imgp->naxis3, "NAXIS3");
    imstat = im_rkey_c(fd, imgp->bunit, "BUNIT");
    imstat = im_rkey_d(fd, &imgp->bscale, "BSCALE");
    imstat = im_rkey_d(fd, &imgp->b_zero, "BZERO");
    imstat = im_rkey_d(fd, &imgp->blank, "BLANK");
    if (imstat >= 0)
    {
	imgp->blank_set = TRUE;
	if ((imgp->bitpix < 0) && (file_type != 3) && (server_mode == FALSE))
	{
	    printf("warning:  BLANK header line ignored - \n");
	    printf("          BLANK keyword is illegal in floating point FITS file\n");
	}
    }
    else
    {
	if ((imgp->blank_set) && (server_mode == FALSE) &&
	((imgp->bitpix > 0) || (file_type == 3)))
	{
	    /* no BLANK card - using default */
	    printf("warning:  using BLANK value of %g from 'set' command\n",
		imgp->blank);
	}
    }

    if ((imgp->bitpix == -64) && (file_type != 3))
    {
	im_blankd(&imgp->blank);
	imgp->blank_set = TRUE;
    }
    if ((imgp->bitpix == -32) && (file_type != 3))
    {
	/* Make sure that the float is cast to a double in exactly */
	/* the same way as we will be casting the pixel values. */
	/* This matters only on non-ieee machines.  Ieee machines */
	/* use NaN testing (finite()) for blank pixels */
	im_blank(&fblank);
	imgp->blank = (double) fblank;
	imgp->blank_set = TRUE;
    }

    imgp->report_blank = imgp->blank;  /* save this value for "ci" command */

    if ((imgp->bitpix < 0) && (file_type == 3))
    {
	/* IRAF floating point */
	imgp->blank = blank_flag; /* internal to skyview we use NaN */
    }

    imstat = im_rkey_d(fd, &imgp->glong, "CRVAL1");
    if (imstat < 0)
	imgp->glong = 1e+30;
    imstat = im_rkey_d(fd, &imgp->crpix1, "CRPIX1");
    if (imstat < 0)
	imgp->crpix1 = 0;
    imstat = im_rkey_c(fd, imgp->ctype1, "CTYPE1");
    if (imstat >= 0)
    {
	strncpy(imgp->ctype1sav, imgp->ctype1, 79);
	imgp->ctype1[8] = '\0';
	if (strncmp(imgp->ctype1+4, "-TAN", 4) == 0)
	    imgp->maptype = GNOMONIC;
	if (strncmp(imgp->ctype1+4, "-SIN", 4) == 0)
	    imgp->maptype = ORTHOGRAPHIC;
	if (strncmp(imgp->ctype1+4, "-NCP", 4) == 0)
	    imgp->maptype = NCP;
	if (strncmp(imgp->ctype1+4, "-AIT", 4) == 0)
	    imgp->maptype = AITOFF;
	if (strncmp(imgp->ctype1+4, "-ARC", 4) == 0)
	    imgp->maptype = ARC;
	if (strncmp(imgp->ctype1+4, "-CAR", 4) == 0)
	    imgp->maptype = CAR;
	if (strncmp(imgp->ctype1+4, "-CEA", 4) == 0)
	    imgp->maptype = CEA;
	if (strncmp(imgp->ctype1+4, "-SFL", 4) == 0)
	    imgp->maptype = SFL;
	if (strncmp(imgp->ctype1+4, "-GLS", 4) == 0)
	    imgp->maptype = SFL;
	if (strncmp(imgp->ctype1+4, "----", 4) == 0)
	    imgp->maptype = LINEAR;
	if (strncmp(imgp->ctype1+4, "    ", 4) == 0)
	    imgp->maptype = LINEAR;
	if (imgp->maptype == UNSPECIFIED)
	    imgp->maptype = UNRECOGNIZED;
	ptmp = strchr(imgp->ctype1, '-');
	if (ptmp != 0)
	    *ptmp = '\0';   /* change - to EOS */
	ptmp = strchr(imgp->ctype1, ' ');
	if (ptmp != 0)
	    *ptmp = '\0';   /* change space to EOS */
	if ((strncmp(imgp->ctype1, "DEC", 3) == 0) ||
	(strncmp(imgp->ctype1, "MM", 2) == 0) ||
	(strncmp(imgp->ctype1, "GLAT", 4) == 0) ||
	(strncmp(imgp->ctype1, "LAT", 3) == 0) ||
	(strncmp(imgp->ctype1, "ELAT", 4) == 0))
	{
	    /* flag that axes are reversed */
	    axes_reversed = 1;
	}
    }
    imstat = im_rkey_d(fd, &imgp->cdelt1, "CDELT1");
    imstat = im_rkey_d(fd, &imgp->glat, "CRVAL2");
    if (imstat < 0)
	imgp->glat = 1e+30;
    imstat = im_rkey_d(fd, &imgp->crpix2, "CRPIX2");
    if (imstat < 0)
	imgp->crpix2 = 0;
    imstat = im_rkey_c(fd, imgp->ctype2, "CTYPE2");
    if (imstat >= 0)
    {
	strncpy(imgp->ctype2sav, imgp->ctype2, 79);
	imgp->ctype2[8] = '\0';
	if (strncmp(imgp->ctype2+4, "-TAN", 4) == 0)
	    imgp->maptype = GNOMONIC;
	if (strncmp(imgp->ctype2+4, "-SIN", 4) == 0)
	    imgp->maptype = ORTHOGRAPHIC;
	if (strncmp(imgp->ctype2+4, "-NCP", 4) == 0)
	    imgp->maptype = NCP;
	if (strncmp(imgp->ctype2+4, "-AIT", 4) == 0)
	    imgp->maptype = AITOFF;
	if (strncmp(imgp->ctype2+4, "-ARC", 4) == 0)
	    imgp->maptype = ARC;
	if (strncmp(imgp->ctype2+4, "-CAR", 4) == 0)
	    imgp->maptype = CAR;
	if (strncmp(imgp->ctype2+4, "-CEA", 4) == 0)
	    imgp->maptype = CEA;
	if (strncmp(imgp->ctype1+4, "-SFL", 4) == 0)
	    imgp->maptype = SFL;
	if (strncmp(imgp->ctype1+4, "-GLS", 4) == 0)
	    imgp->maptype = SFL;
	if (strncmp(imgp->ctype2+4, "----", 4) == 0)
	    imgp->maptype = LINEAR;
	if (strncmp(imgp->ctype2+4, "    ", 4) == 0)
	    imgp->maptype = LINEAR;
	if (imgp->maptype == UNSPECIFIED)
	    imgp->maptype = UNRECOGNIZED;
	ptmp = strchr(imgp->ctype2, '-');
	if (ptmp != 0)
	    *ptmp = '\0';   /* change - to EOS */
	ptmp = strchr(imgp->ctype2, ' ');
	if (ptmp != 0)
	    *ptmp = '\0';   /* change space to EOS */
    }
    imstat = im_rkey_d(fd, &imgp->cdelt2, "CDELT2");
    imstat = im_rkey_d(fd, &imgp->twist, "CROTA2");
    imstat = im_rkey_d(fd, &imgp->crval3, "CRVAL3");

    got_cd1_1 = im_rkey_d(fd, &cd1_1, "CD1_1");
    if (got_cd1_1 < 0)
	got_cd1_1 = im_rkey_d(fd, &cd1_1, "CD001001");
    if (got_cd1_1 < 0)
	got_pc1_1 = im_rkey_d(fd, &pc1_1, "PC1_1");

    got_cd1_2 = im_rkey_d(fd, &cd1_2, "CD1_2");
    if (got_cd1_2 < 0)
	got_cd1_2 = im_rkey_d(fd, &cd1_2, "CD001002");
    if (got_cd1_2 < 0)
	got_pc1_2 = im_rkey_d(fd, &pc1_2, "PC1_2");

    got_cd2_1 = im_rkey_d(fd, &cd2_1, "CD2_1");
    if (got_cd2_1 < 0)
	got_cd2_1 = im_rkey_d(fd, &cd2_1, "CD002001");
    if (got_cd2_1 < 0)
	got_pc2_1 = im_rkey_d(fd, &pc2_1, "PC2_1");

    got_cd2_2 = im_rkey_d(fd, &cd2_2, "CD2_2");
    if (got_cd2_2 < 0)
	got_cd2_2 = im_rkey_d(fd, &cd2_2, "CD002002");
    if (got_cd2_2 < 0)
	got_pc2_2 = im_rkey_d(fd, &pc2_2, "PC2_2");
    
    if ((got_cd1_1 <= 0) &&
	(got_cd1_2 <= 0) &&
	(got_cd2_1 <= 0) &&
	(got_cd2_2 <= 0) &&
        (got_pc1_1 > 0) &&
	(got_pc1_2 > 0) &&
	(got_pc2_1 > 0) &&
	(got_pc2_2 > 0))
    {
	cd1_1 = imgp->cdelt1 * pc1_1;
	cd1_2 = imgp->cdelt1 * pc1_2;
	cd2_1 = imgp->cdelt2 * pc2_1;
	cd2_2 = imgp->cdelt2 * pc2_2;
	got_cd1_1 = 1;
	got_cd1_2 = 1;
	got_cd2_1 = 1;
	got_cd2_2 = 1;
    }


    imstat = im_rkey_d(fd, &imgp->iraf_max, "DATAMAX");
    if (imstat >= 0)
	imgp->iraf_max = (imgp->iraf_max - imgp->b_zero) / imgp->bscale;
    imstat = im_rkey_d(fd, &imgp->iraf_min, "DATAMIN");
    if (imstat >= 0)
	imgp->iraf_min = (imgp->iraf_min - imgp->b_zero) / imgp->bscale;
    imstat = im_rkey_d(fd, &imgp->file_equinox, "EPOCH");
    imstat = im_rkey_d(fd, &imgp->file_equinox, "EQUINOX");
    imstat = im_rkey_c(fd, imgp->radecsys, "RADECSYS");
    if (imstat >= 0)
    {
	imgp->radecsys[8] = '\0';
    }
    imstat = im_rkey_c(fd, tmpstr, "PROJTYPE");
    if (imstat >= 0)
    {
	if (use_projtype)
	{
	    if ((imgp->maptype == UNSPECIFIED) ||
		(imgp->maptype == UNRECOGNIZED))
	    {
		imgp->maptype = parse_fits(tmpstr);
		if (imgp->maptype != UNRECOGNIZED)
		    if(server_mode == FALSE)
			printf("using PROJTYPE of %s\n", maptype_string(imgp->maptype));
	    }
	}
    }

    if ((FINITE(imgp->glat)) &&
	(FINITE(imgp->glong)) &&
	(FINITE(imgp->crpix1)) &&
	(FINITE(imgp->crpix2)) &&
	(imgp->maptype != UNSPECIFIED) &&
	(imgp->maptype != UNRECOGNIZED) &&
	(got_cd1_1 > 0) &&
	(got_cd1_2 > 0) &&
	(got_cd2_1 > 0) &&
	(got_cd2_2 > 0))
    {
	if (axes_reversed)
	{
	    temp = imgp->glong;
	    imgp->glong = imgp->glat;
	    imgp->glat = temp;

	    temp = cd2_2;
	    cd2_2 = cd1_2;
	    cd1_2 = cd1_1;
	    cd1_1 = cd2_1;
	    cd2_1 = temp;
	}
	/* save values for Greisen's formulas */
	imgp->using_cd = TRUE;
	imgp->cd1_1 = cd1_1;
	imgp->cd1_2 = cd1_2;
	imgp->cd2_1 = cd2_1;
	imgp->cd2_2 = cd2_2;
	/* invert matrix */
	determinant = cd1_1 * cd2_2 - cd1_2 * cd2_1;
	imgp->dc1_1 = cd2_2 / determinant;
	imgp->dc1_2 = - cd1_2 / determinant;
	imgp->dc2_1 = - cd2_1 / determinant;
	imgp->dc2_2 = cd1_1 / determinant;


	imgp->twist = atan2(-cd2_1, -cd1_1);
	if (fabs(cd1_1) > fabs(cd1_2))
	{
	    imgp->cdelt1 = cd1_1 / cos(imgp->twist);
	    imgp->cdelt2 = cd2_2 / cos(imgp->twist);
	}
	else
	{
	    imgp->cdelt1 = cd2_1 / sin(imgp->twist);
	    imgp->cdelt2 = - cd1_2 / sin(imgp->twist);
	}
	imgp->twist = imgp->twist * rtd;
    }
    else
    {
	if (axes_reversed)
	{
	    temp = imgp->glong;
	    imgp->glong = imgp->glat;
	    imgp->glat = temp;

	    temp = imgp->cdelt1;
	    imgp->cdelt1 = imgp->cdelt2;
	    imgp->cdelt2 = temp;
	    /* dont know what to do with twist */
	    /* will have to wait until I have a sample image */
	}
    }

    imstat = im_rkey_c(fd, imgp->object, "OBJECT");
    if (imstat >= 0)
    {
	imgp->object[8] = '\0';
    }
    imstat = im_rkey_i(fd, &imgp->dskygrid, "DSKYGRID");
    if (imstat >= 0)
    {
	imgp->maptype = ORTHOGRAPHIC;
    }

    /* now do SIRTF distortion corrections */
    if (strncmp(imgp->ctype1sav+8, "-SIP", 4) == 0)
    {
	imgp-> map_distortion = TRUE;
	imgp->a_order = 0.0;
	imgp->b_order = 0.0;


	imstat = im_rkey_i(fd, &imgp->a_order, "A_ORDER");
	if (imstat >= 0)
	{
	    for (i = 0; i <= imgp->a_order; i++)
	    {
		for (j = 0; j <= imgp->a_order; j++)
		{
		    imgp->a[i][j] = 0.0;
		    if (i + j <= imgp->a_order)
		    {
			sprintf(keyword, "A_%d_%d", i, j);
			imstat = im_rkey_d(fd, &imgp->a[i][j], keyword);
		    }
		}
	    }
	}
	imstat = im_rkey_i(fd, &imgp->b_order, "B_ORDER");
	if (imstat >= 0)
	{
	    for (i = 0; i <= imgp->b_order; i++)
	    {
		for (j = 0; j <= imgp->b_order; j++)
		{
		    imgp->b[i][j] = 0.0;
		    if (i + j <= imgp->b_order)
		    {
			sprintf(keyword, "B_%d_%d", i, j);
			imstat = im_rkey_d(fd, &imgp->b[i][j], keyword);
		    }
		}
	    }
	}
	imstat = im_rkey_i(fd, &imgp->ap_order, "AP_ORDER");
	if (imstat >= 0)
	{
	    for (i = 0; i <= imgp->ap_order; i++)
	    {
		for (j = 0; j <= imgp->ap_order; j++)
		{
		    imgp->ap[i][j] = 0.0;
		    if (i + j <= imgp->ap_order)
		    {
			sprintf(keyword, "AP_%d_%d", i, j);
			imstat = im_rkey_d(fd, &imgp->ap[i][j], keyword);
		    }
		}
	    }
	}
	imstat = im_rkey_i(fd, &imgp->bp_order, "BP_ORDER");
	if (imstat >= 0)
	{
	    for (i = 0; i <= imgp->bp_order; i++)
	    {
		for (j = 0; j <= imgp->bp_order; j++)
		{
		    imgp->bp[i][j] = 0.0;
		    if (i + j <= imgp->bp_order)
		    {
			sprintf(keyword, "BP_%d_%d", i, j);
			imstat = im_rkey_d(fd, &imgp->bp[i][j], keyword);
		    }
		}
	    }
	}

    }

    /* now do Digital Sky Survey plate solution cofficients */
    imstat = im_rkey_d(fd, &rah, "PLTRAH");
    if (imstat >= 0)
    {
	/* we've got one of these */
	imgp->maptype = PLATE;
	imstat = im_rkey_d(fd, &ram, "PLTRAM");
	imstat = im_rkey_d(fd, &ras, "PLTRAS");
	ra_hours = rah + (ram / (double)60.0) + (ras / (double)3600.0);
	imgp->plate_ra = ra_hours * 15.0 * dtr;
	imstat = im_rkey_c(fd, decsign, "PLTDECSN");
	if (decsign[0] == '-')
	    dsign = -1.;
	else
	    dsign = 1.;
	imstat = im_rkey_d(fd, &decd, "PLTDECD");
	imstat = im_rkey_d(fd, &decm, "PLTDECM");
	imstat = im_rkey_d(fd, &decs, "PLTDECS");
	dec_deg = dsign * (decd+(decm/(double)60.0)+(decs/(double)3600.0));
	imgp->plate_dec = dec_deg * dtr;
	if (debug)
	{
	    fprintf(debug_file,
	"WCSINIT Plate center RA=%2.0f:%2.0f:%5.3f, Dec=%3.0f:%2.0f:%5.3f\n",
	    rah,ram,ras,dsign*decd,decm,decs);
	}
	imstat = im_rkey_d(fd, &imgp->x_pixel_offset, "CNPIX1");
	imstat = im_rkey_d(fd, &imgp->y_pixel_offset, "CNPIX2");
	imstat = im_rkey_d(fd, &imgp->plt_scale, "PLTSCALE");
	imstat = im_rkey_d(fd, &imgp->x_pixel_size, "XPIXELSZ");
	imstat = im_rkey_d(fd, &imgp->y_pixel_size, "YPIXELSZ");
	imstat = im_rkey_d(fd, &imgp->ppo_coeff[0], "PPO1");
	imstat = im_rkey_d(fd, &imgp->ppo_coeff[1], "PPO2");
	imstat = im_rkey_d(fd, &imgp->ppo_coeff[2], "PPO3");
	imstat = im_rkey_d(fd, &imgp->ppo_coeff[3], "PPO4");
	imstat = im_rkey_d(fd, &imgp->ppo_coeff[4], "PPO5");
	imstat = im_rkey_d(fd, &imgp->ppo_coeff[5], "PPO6");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[0], "AMDX1");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[1], "AMDX2");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[2], "AMDX3");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[3], "AMDX4");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[4], "AMDX5");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[5], "AMDX6");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[6], "AMDX7");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[7], "AMDX8");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[8], "AMDX9");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[9], "AMDX10");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[10], "AMDX11");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[11], "AMDX12");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[12], "AMDX13");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[13], "AMDX14");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[14], "AMDX15");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[15], "AMDX16");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[16], "AMDX17");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[17], "AMDX18");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[18], "AMDX19");
	imstat = im_rkey_d(fd, &imgp->amd_x_coeff[19], "AMDX20");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[0], "AMDY1");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[1], "AMDY2");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[2], "AMDY3");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[3], "AMDY4");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[4], "AMDY5");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[5], "AMDY6");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[6], "AMDY7");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[7], "AMDY8");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[8], "AMDY9");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[9], "AMDY10");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[10], "AMDY11");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[11], "AMDY12");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[12], "AMDY13");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[13], "AMDY14");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[14], "AMDY15");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[15], "AMDY16");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[16], "AMDY17");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[17], "AMDY18");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[18], "AMDY19");
	imstat = im_rkey_d(fd, &imgp->amd_y_coeff[19], "AMDY20");
	
	imgp->crpix1 = 0.5 - imgp->x_pixel_offset;
	imgp->crpix2 = 0.5 - imgp->y_pixel_offset;

	if (imgp->cdelt1 == 0.0)
	{
	    imgp->cdelt1 = - imgp->plt_scale * imgp->x_pixel_size / 1000 / 3600;
	    imgp->cdelt2 = imgp->plt_scale * imgp->y_pixel_size / 1000 / 3600;
	}

    }



    if (imgp->maptype == UNRECOGNIZED)
	if(server_mode == FALSE)
	    printf("Warning:  projection type unrecognized\n");
    imgp->filehdr.tapetype = IMFILE;
    if (debug)
	fprintf(debug_file, "imhdr:  exit\n");
}

void
imhd(id)
int id;
{
    int stat;
    char str[81];

    stat = im_rkey_t(id, str, "1");
    do
    {
	print_header_line(str);
	stat = im_rkey_t(id, str, "+");

    } while (stat > 0);
}


void *imget_nofile(totbytes, f_name)
int totbytes;
char *f_name;
{
    struct pixels *tmppix;

    /* like imget, but creates pixel areas without opening a file */
    tmppix = get_pixel_bufs(totbytes);
    if (tmppix == NULL)
	return(NULL);
    tmppix->nofile = TRUE;  /* lock this into memory - data is not in a file */
    strcpy(tmppix->f_name_save, f_name);
    return(tmppix->pixelptr);
}

void *imget(fd, f_name)
int fd;
char *f_name;
{
    int bytes_per_pixel, totbytes, bitpix1, imstat, totpixels;
    struct pixels *tmppix;
    int same, file_type, do_close;
    struct img tmpimg;


    tmppix = find_pixel_FILNAM(f_name);
    if (tmppix != NULL)
    {
	/* same file */
	/* See if he changed the blank value - if so, reread the file */
	if (FINITE(tmppix->report_blank))
	{
	    if (FINITE(report_blank))
	    {
		if (tmppix->report_blank == report_blank)
		    same = TRUE;
		else
		    same = FALSE;
	    }
	    else
	    {
		same = FALSE;
	    }
	}
	else
	{
	    if (FINITE(report_blank))
		same = FALSE;
	    else
		same = TRUE;
	}
	if (same)
	{
	    if (debug)
		fprintf(debug_file, "imget:  pixels already in memory\n");
	    return(tmppix->pixelptr);   /* already in memory */
	}
	else
	{
	    force_reread(f_name);
	}
    }

    if (fd <= 0)    /* see if we've been handed an open IM workfile */
    {
	/* need to open file */
	fd = wf_open(f_name, &tmpimg, 1);
	if (fd < 0)
	    return(NULL);
	do_close = 1;
    }
    else
    {
	do_close = 0;
    }

    imstat = im_rkey_i(fd, &totpixels, "npix");
    if (imstat < 0)
	return(NULL);

    imstat = im_rkey_i(fd, &bitpix1, "BITPIX");
    if (imstat < 0)
	return(NULL);
    
    switch(bitpix1)
    {
	case 8:
	    bytes_per_pixel = 1;
	    break;
	case 16:
	    bytes_per_pixel = 2;
	    break;
	case 32:
	    bytes_per_pixel = 4;
	    break;
	case -32:
	    bytes_per_pixel = sizeof(float);
	    break;
	case -64:
	    bytes_per_pixel = sizeof(double);
	    break;
    }
    totbytes = totpixels * bytes_per_pixel;

    tmppix = get_pixel_bufs(totbytes);
    if (tmppix == NULL)
	return(NULL);

    imstat = im_rkey_i(fd, &file_type, "filetype");
    if ((bitpix < 0) && (file_type == 3) && 
	(blank_set) && (FINITE(report_blank)))
    {
	/* IRAF floating point with BLANK from header or from "set" command */
	/* Make sure that image access knows about this blank value */
	/* for it to use when mapping blanks to NaN */
	imstat = im_wkey_i(fd, (int)report_blank, "BLANK");
    }

    if ((server_mode == FALSE) && (!be_quiet))
    {
	printf("reading file . .");
	fflush(stdout);
    }
    switch(bitpix1)
    {
	case 8:
	    imstat = im_rpix_c(fd, tmppix->pixelptr, totpixels);
	    break;
	case 16:
	    imstat = im_rpix_s(fd, tmppix->pixelptr, totpixels);
	    break;
	case 32:
	    imstat = im_rpix_i(fd, tmppix->pixelptr, totpixels);
	    break;
	case -32:
	    imstat = im_rpix_r(fd, tmppix->pixelptr, totpixels);
	    break;
	case -64:
	    imstat = im_rpix_d(fd, tmppix->pixelptr, totpixels);
	    break;
    }
    if (do_close)
	im_close(fd);

    if (imstat < 0)
    {
	delete_pixels(tmppix);
	if(server_mode == FALSE)
	    printf("\n");
	return(NULL);
    }
    if ((server_mode == FALSE) && (!be_quiet))
	printf(" done\n");
    strcpy(tmppix->f_name_save, f_name);
    tmppix->report_blank = report_blank;
    return(tmppix->pixelptr);
}

static
struct pixels *get_pixel_bufs(totbytes)
int totbytes;
{
    struct rlimit rlp;
    struct pixels *tmppix;

#ifndef hpux
    /* increase his memory limit - this image may be a big sucker */
    getrlimit(RLIMIT_DATA, &rlp);
    rlp.rlim_cur = rlp.rlim_max;
    setrlimit(RLIMIT_DATA, &rlp);
#endif /* hpux */

    tmppix = new_pixels();   /* get space for "pixels" structure */
    if (tmppix == NULL)
	return(NULL);

    for (;;)
    {
	tmppix->pixelptr = (void *) malloc(totbytes);
	if (tmppix->pixelptr == NULL)
	{
	    /* delete one, if possible, and try again */
	    if (delete_oldest_pixels(tmppix))
	    {
		continue;
	    }
	    else
	    {
		/* nothing else left to delete  */
		if(server_mode == FALSE)
		    printf("cannot obtain memory to read in pixels\n");
		delete_pixels(tmppix);
		return(NULL);
	    }
	}
	else
	{
	    break;  /* got the memory */
	}
    }
    return(tmppix);
}


/********************************************************************/
/*                                                                  */
/*  NEW_PIXELS                                                      */
/*  Place a new entry into the pixels structure.                    */
/*  Return a pointer to the new entry.                              */
/*                                                                  */
/********************************************************************/

static
struct pixels *new_pixels()
{
    static struct pixels *tmppix;
    struct pixels *p, *x;
    struct img *tmpimg;

    if (conserve_mem)
	delete_all_pixels();

    /* discard pixels not in use */
    p = head_pix.rlink;
    while (p != &head_pix)
    {
	if (!p->dirty)
	{
	    tmpimg = peek_FILNAM(p->f_name_save);
	    if (tmpimg == NULL)
	    {
		delete_pixels(p);
		p = &head_pix;   /* start over */

	    }
	}
	p = p->rlink;
    }

    tmppix = (struct pixels *) malloc(sizeof(struct pixels));
    if (tmppix == NULL)
    {
	error1("no memory for pixels structure");
	return(tmppix);
    }
    /* insert to right of head_pix */
    x = &head_pix;
    tmppix->llink = x;
    tmppix->rlink = x->rlink;
    (tmppix->rlink)->llink = tmppix;
    x->rlink = tmppix;

    tmppix->dirty = FALSE;
    tmppix->nofile = FALSE;

    if (debug)
	fprintf(debug_file, "new_pixels: tmppix=0x%p\n", (void *) tmppix);

    return(tmppix);
}

void
change_pixel_name(old_name, new_name, dirty_flag)
char *old_name, *new_name;
int dirty_flag;
{
    struct pixels *tmppix;

    tmppix = find_pixel_FILNAM(old_name);
    if (tmppix != NULL)
    {
	strcpy(tmppix->f_name_save, new_name);
	tmppix->dirty = dirty_flag;
    }
}

static void
delete_all_pixels()
{
    /* discard pixels not in use */
    while (delete_oldest_pixels((struct pixels *) NULL))
	;
}

/********************************************************************/
/*                                                                  */
/*  DELETE_OLDEST_PIXELS                                            */
/*  Deletes oldest entry from the pixels array                      */
/*  Does NOT delete entry with the pointer "save_this_p"            */
/*                                                                  */
/*  Returns TRUE if able to delete something, FALSE if unable.      */
/*                                                                  */
/********************************************************************/

static int
delete_oldest_pixels(save_this_p)
struct pixels *save_this_p;
{
    struct pixels *p;

    p = head_pix.rlink;
    while (p != &head_pix)
    {
	if ((!p->dirty) && (!p->nofile) && (p != save_this_p))
	{
	    delete_pixels(p);
	    return(TRUE);
	}
	p = p->rlink;
    }
    return(FALSE);
}

/********************************************************************/
/*                                                                  */
/*  DELETE_PIXELS                                                   */
/*  Deletes an entry from the pixels array, given the pixel pointer */
/*                                                                  */
/********************************************************************/

static void
delete_pixels(p)
struct pixels *p;
{
    if (debug)
	fprintf(debug_file, "delete_pixels\n");

    if (p->pixelptr != NULL)
	free(p->pixelptr);

    (p->llink)->rlink = p->rlink;
    (p->rlink)->llink = p->llink;
    free(p);

    return;
}

/********************************************************************/
/*                                                                  */
/*  FORCE_REREAD                                                    */
/*  Deletes an entry from the pixels array, given the filename      */
/*                                                                  */
/********************************************************************/

void
force_reread(fname)
char *fname;
{
    struct pixels *tmppix;

    tmppix = find_pixel_FILNAM(fname);
    if (tmppix != NULL)
	delete_pixels(tmppix);
    if (strcmp(hist_filnam, fname) == 0)
	hist_filnam[0] = '\0';       /* invalidate saved histogram */
}

/********************************************************************/
/*                                                                  */
/*  FIND_pixel_FILENAM                                              */
/*  given a filename   returns a pointer to the pixel structure     */
/*  containing the pixels from that file.                           */
/*                                                                  */
/*                                                                  */
/*  If not found, returns NULL.                                     */
/*                                                                  */
/********************************************************************/

static
struct pixels *find_pixel_FILNAM(fname)
char *fname;
{
    struct pixels *p;

    if (debug)
	fprintf(debug_file, "find_pixel_FILNAM: entry  fname =%s=\n", fname);

    p = head_pix.rlink;
    while (p != &head_pix)
    {
	if (strcmp(fname, p->f_name_save) == 0)
	{
	    return(p);
	}
	p = p->rlink;
    }
    return(NULL);
}
