/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* Code for all overlay graphics in image windows */
/* Graphics for non-image windows are done elsewhere */

/* All graphics on images must be repainted upon exposure and with */
/* commands like off, on, and erase overlays */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>     /* includes strings.h, sys/time.h, and sys/file.h */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <setjmp.h>

#include "skyview.h"
#include "parse.h"
#include "img_x.h"
#include "area.h"
#include "jpeglib.h"

#define TRUE 1
#define FALSE 0

#define G_NULL 0
#define G_MARK 1
#define G_STRING 2
#define G_DLINE 3
#define G_DFILLAREA 4
#define G_DCIRCLE 5
#define G_ELLIPSE 6
#define G_DARC 7
#define G_LASTNULL 8


typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    double width;
    double height;
    double angle;
    } G_ellipse;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    int mark_type;
    int mark_size;
    int mark_size2;
    int mark_size3;
    int table_line;
    char *table_filename;
    } G_mark;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    char *string;
    Font g_font;
    double font_size;
    } G_string;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    double x1;
    double y1;
    } G_dline;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    int end;
    } G_dfillarea;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    double radius;
    double angle1;
    double angle_extent;
    } G_darc;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    double radius;
    } G_dcircle;

typedef struct
    {
    int type;
    int overlay_is_on;
    Window g_win;
    GC gc_color;
    int g_bundle;
    double x;
    double y;
    } G_any;

typedef union _G_graphic
    {
    int type;
    G_ellipse g_ellipse;
    G_mark g_mark;
    G_string g_string;
    G_any g_any;
    G_dline g_dline;
    G_dfillarea g_dfillarea;
    G_dcircle g_dcircle;
    G_darc g_darc;
    } G_graphic;

typedef union
{
    INT32 *dmpimg32;
    unsigned char *dmpimg;
} img_struct;

#define MAX_OVERLAYS 15
static GC nographics_GC[MAX_OVERLAYS];

static struct
{
   int  red;
   int  green;
   int  blue;
   char name[80];
}  overlays[MAX_OVERLAYS];

static int noverlays = 0;


typedef unsigned long Pixel;
extern Pixel red_mask, green_mask, blue_mask;
extern XColor exact_defs[];
extern Window win;
extern Colormap theCmap;
extern Display *display;
extern unsigned int depth;
extern GC gc_overlay, gc_red, gc_green, gc_blue, gc_white, gc_black, gcxor;
extern Font default_font_id;
extern Font agra_font_id;
extern double win_zoom;
extern int win_x_offset, win_y_offset;
extern char plane, save_line[];
extern unsigned int win_width, win_height;
extern int debug, graphics, server_mode, s0_server_mode, lines, pixels;
extern char server_str[];
extern FILE *session, *debug_file;
extern int table_linenumber;
extern char *table_nam;
extern double bscale, b_zero;
extern struct area_struct area[MAX_AREAS];
extern char last_coord[2][40];
extern int area_number;
extern int control_c, point_mode;
extern unsigned char *image_org;
extern unsigned int width_org, height_org;

extern int ads_server_mode;

static double graphics_dx, graphics_dy;    /* in SC coordinates */
static int current_bundle = 0;

#define SLOTS 100

static G_graphic *g_graphic;
static int g_repainting = 0;
static int g_lastnull;

/* function prototypes */
#ifdef _NO_PROTO

static void repaint_one_graphic();
static void highlight_graphics_bundle();
static void unhighlight_graphics_bundle();
static void report_mark();
static void xytoimg();
static void dmpellipse();
static int dmpline();
static G_graphic *add_graphic();
static int check_16();
static int find_bundle();
static char *translate_overlay_color();
static void newdmpcircle();
static void set_pixel();

#else

static void repaint_one_graphic(G_graphic *g_g);
static void highlight_graphics_bundle(int bundle);
static void unhighlight_graphics_bundle(int bundle);
static void report_mark(G_graphic *g_g);
static void xytoimg(double xin, double yin, double *xout, double *yout);
static void dmpellipse(int color, img_struct img_ptr,
    unsigned int nx, unsigned int ny, double xcen, double ycen, 
    double half_width, double half_height, double angle);
static int dmpline(int color, img_struct img_ptr,
unsigned int nx, unsigned int ny, double x1, double y1, double x2, double y2);
static G_graphic *add_graphic(void);
static int check_16(double x);
static int find_bundle(int xin, int yin);
static char *translate_overlay_color(char *color);
static void newdmpcircle(int color, img_struct img_ptr,
    unsigned int nx, unsigned int ny, int xcen, int ycen, int radius);
static void set_pixel(int ix, int jy, 
    unsigned int nx, unsigned int ny, int color, img_struct img_ptr);
int WriteGIF(FILE *, unsigned char *, int, int, char *, char *, char *, int, int);


#endif /* _NO_PROTO */

static
G_graphic *add_graphic()
{
    static int g_graphic_slots;
    G_graphic *g_g, *g_next;

    if (g_repainting)
	return (NULL);    /* dont record graphic if this is a playback */

    if (g_graphic == NULL)    /* if this is the very first call */
    {
	/* get initial allotment */
	g_graphic_slots = SLOTS;
	g_graphic = (G_graphic *) malloc(g_graphic_slots * sizeof(G_graphic));
	if (g_graphic == NULL)
	    return (NULL);   /* discard graphics if malloc failed */
	g_graphic->type = G_LASTNULL;    /* mark the end */
	g_lastnull = 0;
    }

    /* see if more room is needed */
    if (g_lastnull + 1 >= g_graphic_slots)
    {
	g_graphic_slots = 2 * g_graphic_slots;   /* double current space */
	g_graphic = (G_graphic *) realloc(g_graphic, 
	    g_graphic_slots * sizeof(G_graphic));
	if (g_graphic == NULL)
	    return (NULL);   /* discard graphics if malloc failed */
    }
    g_g = g_graphic + g_lastnull;
    g_next = g_g + 1;
    g_next->type = G_LASTNULL;
    g_lastnull++;
    return(g_g);
}

void
next_bundle()
{
    current_bundle++;
    if (debug)
	fprintf(debug_file, "next_bundle:  new bundle is %d\n", current_bundle);
}

void
repaint_graphics(dest)      /* repaint graphics for the current window (win) */
			    /* into the drawable "dest" */
Drawable dest;
/* Usually the drawable will be "win" itself, but it may be a pixmap */
/* to be used for a fullimage screen_dump */
{
    GC gc_save;
    G_graphic *g_g;
    Window win_save;
    int win_x_offset_save, win_y_offset_save;
    double win_zoom_save;

    if (g_graphic == NULL)
	return;
    gc_save = gc_overlay;
    win_save = win;
    win_zoom_save = win_zoom;
    win_x_offset_save = win_x_offset;
    win_y_offset_save = win_y_offset;
    if (dest != win)
    {
	win = dest;
	win_zoom = 1.0;
	win_x_offset = 0;
	win_y_offset = 0;
    }
    g_repainting = 1;   /* flag that we are repainting graphics */

    
    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if ((g_g->type != G_NULL) && (g_g->g_any.g_win == win_save) &&
	    (g_g->g_any.overlay_is_on))
	{
	    gc_overlay = g_g->g_any.gc_color;
	    repaint_one_graphic(g_g);
	}
    }
    g_repainting = 0;
    win_x_offset = win_x_offset_save;
    win_y_offset = win_y_offset_save;
    win_zoom = win_zoom_save;
    win = win_save;
    gc_overlay = gc_save;
}

static void
repaint_one_graphic(g_g)
G_graphic *g_g;
{
	    switch(g_g->type)
	    {
	    case G_DLINE:
		overlay_dmov(g_g->g_dline.x, g_g->g_dline.y);
		overlay_ddraw(g_g->g_dline.x1, g_g->g_dline.y1);
		break;
	    case G_DFILLAREA:
		overlay_darea(g_g->g_dfillarea.x, g_g->g_dfillarea.y, 
		   g_g->g_dfillarea.end);
		break;
	    case G_DCIRCLE:
		overlay_dmov(g_g->g_dcircle.x, g_g->g_dcircle.y);
		overlay_dcircle(g_g->g_dcircle.radius);
		break;
	    case G_DARC:
		overlay_dmov(g_g->g_darc.x, g_g->g_darc.y);
		overlay_darc(g_g->g_darc.radius, g_g->g_darc.angle1, 
		    g_g->g_darc.angle_extent);
		break;
	    case G_ELLIPSE:
		overlay_dmov(g_g->g_ellipse.x, g_g->g_ellipse.y);
		overlay_dellipse(g_g->g_ellipse.width, g_g->g_ellipse.height,
		    g_g->g_ellipse.angle);
		break;
	    case G_MARK:
		do_mark(g_g->g_mark.x, g_g->g_mark.y,
		    g_g->g_mark.mark_type,   g_g->g_mark.mark_size,
		    g_g->g_mark.mark_size2,   g_g->g_mark.mark_size3);
		break;
	    case G_STRING:
		overlay_string(g_g->g_string.x, g_g->g_string.y, g_g->g_string.string, g_g->g_string.g_font, g_g->g_string.font_size);
		break;
	    }
}

static void
highlight_graphics_bundle(bundle)  
int bundle;
{
    GC gc_save;
    G_graphic *g_g;

    if (g_graphic == NULL)
	return;
    gc_save = gc_overlay;
    gc_overlay = gc_red;
    g_repainting = 1;   /* flag that we are repainting graphics */

    
    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if ((g_g->type != G_NULL) && (g_g->g_any.g_win == win) &&
	    (g_g->g_any.g_bundle == bundle))
	{

	    repaint_one_graphic(g_g);
	}
    }
    g_repainting = 0;
    gc_overlay = gc_save;
}

static void
unhighlight_graphics_bundle(bundle)  
int bundle;
{
    GC gc_save;
    G_graphic *g_g;

    if (g_graphic == NULL)
	return;
    gc_save = gc_overlay;
    g_repainting = 1;   /* flag that we are repainting graphics */

    
    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if ((g_g->type != G_NULL) && (g_g->g_any.g_win == win) &&
	    (g_g->g_any.g_bundle == bundle))
	{

	    gc_overlay = g_g->g_any.gc_color;
	    repaint_one_graphic(g_g);
	}
    }
    g_repainting = 0;
    gc_overlay = gc_save;
}

void
erase_window_graphics(win1)      /* erase graphics for the window win1 */
Window win1;
{
    G_graphic *g_g, *g_dest;

    if (g_graphic == NULL)
	return;

    /* garbage collect as we go */
    g_g = g_graphic;
    g_dest = g_graphic;
    while (g_g->type != G_LASTNULL)
    {
	if (g_g->g_any.g_win == win1)
	{
	    if (g_g->type == G_STRING)
		free(g_g->g_string.string);
	    g_g++;
	}
	else
	{
	    if (g_dest != g_g)
		memcpy(g_dest, g_g, sizeof(G_graphic));
	    g_g++;
	    g_dest++;
	}
    }
    g_dest->type = G_LASTNULL;
    g_lastnull = g_dest - g_graphic;
}

void
erase_graphics_bundle(bundle)      /* erase graphics in one bundle */
int bundle;
{
    G_graphic *g_g, *g_dest;

    if (g_graphic == NULL)
	return;

    /* garbage collect as we go */
    g_g = g_graphic;
    g_dest = g_graphic;
    while (g_g->type != G_LASTNULL)
    {
	if (g_g->g_any.g_bundle == bundle)
	{
	    if (g_g->type == G_STRING)
		free(g_g->g_string.string);
	    g_g++;
	}
	else
	{
	    if (g_dest != g_g)
		memcpy(g_dest, g_g, sizeof(G_graphic));
	    g_g++;
	    g_dest++;
	}
    }
    g_dest->type = G_LASTNULL;
    g_lastnull = g_dest - g_graphic;
}

void
erase_graphics_color(color)      /* erase graphics color for all windows */
char *color;
{
    G_graphic *g_g, *g_dest;
    GC gc;
    XColor user_color;

    if (g_graphic == NULL)
	return;

    if ((color[1] == 'o') && (color[2] == '\0'))  /* old style */
    {
	if (color[0] == 'a')
	{
	    /* erase all overlays */
	    g_g = g_graphic;
	    while (g_g->type != G_LASTNULL)
	    {
		if (g_g->type == G_STRING)
		    free(g_g->g_string.string);
		g_g++;
	    }
	    free(g_graphic);
	    g_graphic = NULL;
	    delete_all_RGB();
	    return;
	}

	switch(color[0])
	{
	case 'r':
	    gc = gc_red;
	    break;
	case 'g':
	    gc = gc_green;
	    break;
	case 'b':
	    gc = gc_blue;
	    break;
	case 'w':
	    gc = gc_white;
	    break;
	case 'B':
	    gc = gc_black;
	    break;
	}
    }
    else
    {
	if (!graphics)
	    return;

	if (!XParseColor(display, theCmap, color, &user_color))
	{
	    return;
	}
	gc = delete_RGB(&user_color);
	if (gc == NULL)
	    return;
    }

    /* garbage collect as we go */
    g_g = g_graphic;
    g_dest = g_graphic;
    while (g_g->type != G_LASTNULL)
    {
	if (g_g->g_any.gc_color == gc)
	{
	    if (g_g->type == G_STRING)
		free(g_g->g_string.string);
	    g_g++;
	}
	else
	{
	    if (g_dest != g_g)
		memcpy(g_dest, g_g, sizeof(G_graphic));
	    g_g++;
	    g_dest++;
	}
    }
    g_dest->type = G_LASTNULL;
    g_lastnull = g_dest - g_graphic;
}

void
debug_graphics()      /* print a report of graphics */
{
    G_graphic *g_g;

    printf("sizeof(G_graphic) = %lu\n", sizeof(G_graphic));
    if (g_graphic == NULL)
	return;

    g_g = g_graphic;
    do
    {
	switch(g_g->type)
	{
	    case G_DLINE:
		printf("   DLINE:   win=%lx  x = %g  y = %g  x1 = %g  y1 = %g\n",
		    g_g->g_dline.g_win,
		    g_g->g_dline.x,
		    g_g->g_dline.y,
		    g_g->g_dline.x1,
		    g_g->g_dline.y1);
		break;
	    case G_DARC:
		printf("   DARC: win = %lx  x = %g  y = %g  radius = %g angle1 = %g  angle_extent = %g\n",
		    g_g->g_darc.g_win,
		    g_g->g_darc.x,
		    g_g->g_darc.y,
		    g_g->g_darc.radius,
		    g_g->g_darc.angle1,
		    g_g->g_darc.angle_extent);
		break;
	    case G_DCIRCLE:
		printf("   DCIRCLE: win = %lx  x = %g  y = %g  radius = %g\n",
		    g_g->g_dcircle.g_win,
		    g_g->g_dcircle.x,
		    g_g->g_dcircle.y,
		    g_g->g_dcircle.radius);
		break;
	    case G_ELLIPSE:
		printf("   ELLIPSE: win = %lx  x = %g  y = %g  width = %g  height = %g  angle = %g\n",
		    g_g->g_ellipse.g_win,
		    g_g->g_ellipse.x,
		    g_g->g_ellipse.y,
		    g_g->g_ellipse.width,
		    g_g->g_ellipse.height,
		    g_g->g_ellipse.angle);
		break;
	    case G_MARK:
		printf("   MARK:   win = %lx  GC = 0x%p  x = %g  y = %g  type = %d\n",
		    g_g->g_mark.g_win,
		    (void *) g_g->g_mark.gc_color,
		    g_g->g_mark.x,
		    g_g->g_mark.y,
		    g_g->g_mark.mark_type);
		break;
	    case G_DFILLAREA:
		printf("   DFILLAREA: win = %lx  x = %g  y = %g  end = %d\n",
	            g_g->g_dfillarea.g_win,
	            g_g->g_dfillarea.x,
	            g_g->g_dfillarea.y,
	            g_g->g_dfillarea.end);
		break;
	    case G_STRING:
		printf("   STRING:   win = %lx  x = %g  y = %g  string = %s\n",
		    g_g->g_string.g_win,
		    g_g->g_string.x,
		    g_g->g_string.y,
		    g_g->g_string.string);
		break;
	    case G_NULL:
		printf("   NULL\n");
		break;
	    case G_LASTNULL:
		printf("   LASTNULL\n");
		break;
	}
	g_g++;
    } while (g_g->type != G_LASTNULL);
}

static
char *translate_overlay_color(color)
char *color;
{
    char *ret_str;

    ret_str = color;
    if (strcmp(color, "ro") == 0)
	ret_str = "red";
    if (strcmp(color, "go") == 0)
	ret_str = "green";
    if (strcmp(color, "bo") == 0)
	ret_str = "blue";
    if (strcmp(color, "wo") == 0)
	ret_str = "white";
    if (strcmp(color, "Bo") == 0)
	ret_str = "black";
    return(ret_str);
}


void
off_X_overlay(color, on_off)
char *color;
int on_off;
{
    XColor user_color;
    GC tmp_gc;
    G_graphic *g_g;

    if (!XParseColor(display, theCmap, translate_overlay_color(color), 
	&user_color))
    {
	sprintf(server_str, 
	    "color %s is not in X windows color list", color);
	error1(server_str);
	return;
    }
    tmp_gc = lookup_RGB(&user_color);
    if (tmp_gc != NULL)
    {

	if (g_graphic == NULL)
	    return;

	g_g = g_graphic;
	do
	{
	    if (g_g->g_any.gc_color == tmp_gc)
	    {
		g_g->g_any.overlay_is_on = on_off;
	    }
	    g_g++;
	} while (g_g->type != G_LASTNULL);
	expose_all();
    }
}


void
off_all_X_overlay(on_off)
int on_off;
{
    G_graphic *g_g;

    if (g_graphic == NULL)
	return;

    g_g = g_graphic;
    do
    {
	g_g->g_any.overlay_is_on = on_off;
	g_g++;
    } while (g_g->type != G_LASTNULL);
    expose_all();
}

/*ARGSUSED*/
void
set_overlay_mask(dmask)
int dmask;
{
}


/*********************************************************************/ 
/*                                                                   */ 
/*  OVERLAY_DMOV and OVERLAY_DDRAW do moves and draws on             */ 
/*  current image.  Lines are correctly redrawn as image is zoomed.  */ 
/*                                                                   */  
/*  Input: x, y double precision in SC coordinates                   */ 
/*                                                                   */ 
/*********************************************************************/

void
overlay_dmov(x, y)   /* double precision version of overlay_mov */
double x, y;
{
    graphics_dx = x;
    graphics_dy = y;
}

void
overlay_ddraw(x, y)    /* double precision version of overlay_draw */
double x, y;
{
    double d_coord[2], d_coord2[2];
    G_graphic *g_g;

    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_DLINE;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_dline.g_win = win;
	g_g->g_dline.gc_color = gc_overlay;
	g_g->g_dline.g_bundle = current_bundle;
	g_g->g_dline.x = graphics_dx;
	g_g->g_dline.y = graphics_dy;
	g_g->g_dline.x1 = x;
	g_g->g_dline.y1 = y;
    }

    d_coord[0] = graphics_dx;
    d_coord[1] = graphics_dy;
    dSC_to_WIN(d_coord);

    d_coord2[0] = x;
    d_coord2[1] = y;
    dSC_to_WIN(d_coord2);

    if (graphics && check_16(d_coord[0]) && check_16(d_coord[1]) &&
	check_16(d_coord2[0]) && check_16(d_coord2[1]))
        XDrawLine(display, win, gc_overlay, 
	    Round(d_coord[0]), Round(d_coord[1]), 
	    Round(d_coord2[0]), Round(d_coord2[1]));
    graphics_dx = x;
    graphics_dy = y;
}

/* This one optionally draws with XOR and doesn't put into graphics database */
void
overlay_ddraw_n(x, y, xor)    /* double precision version of overlay_draw */
double x, y;
int xor;
{
    double d_coord[2], d_coord2[2];
    GC gc;

    if (xor)
	gc = gcxor;
    else
	gc = gc_overlay;

    d_coord[0] = graphics_dx;
    d_coord[1] = graphics_dy;
    dSC_to_WIN(d_coord);

    d_coord2[0] = x;
    d_coord2[1] = y;
    dSC_to_WIN(d_coord2);

    if (graphics && check_16(d_coord[0]) && check_16(d_coord[1]) &&
	check_16(d_coord2[0]) && check_16(d_coord2[1]))
        XDrawLine(display, win, gc, (int) d_coord[0], (int) d_coord[1], 
	    (int) d_coord2[0], (int) d_coord2[1]);
    graphics_dx = x;
    graphics_dy = y;
}

/* *********************************************************************** */
/* check_16 returns 1 (true) if double x is OK as a 16 bit signed integer */
/* X protocol defines x,y points as 16 bit signed integers and             */
/* does wraparound if handed an int (32 bit) that is outside that range    */
/* *********************************************************************** */
static
int check_16(x)
double x;
{
    if ((x <= 32767) && (x >= -32767))
	return 1;
    else
	return 0;
}

/*********************************************************************/ 
/*                                                                   */ 
/*  OVERLAY_DAREA collects the segments for a filled area and        */ 
/*  when it has them all draws the area.                             */ 
/*                                                                   */  
/*  Input: x, y double precision in SC coordinates                   */ 
/*         end  integer flag (end = 1 => draw the filled area)       */ 
/*                                                                   */ 
/*********************************************************************/

void
overlay_darea(x, y, end)
double x, y;
int end;
{
    double d_coord[2];
    G_graphic *g_g;
    XPoint fill[1024];
    static int fill_count = 0;

    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_DFILLAREA;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_dfillarea.g_win = win;
	g_g->g_dfillarea.gc_color = gc_overlay;
	g_g->g_dfillarea.g_bundle = current_bundle;
	g_g->g_dfillarea.x = x;
	g_g->g_dfillarea.y = y;
	g_g->g_dfillarea.end = end;
    }

    d_coord[0] = x;
    d_coord[1] = y;
    dSC_to_WIN(d_coord);

    fill[fill_count].x = (int) d_coord[0];
    fill[fill_count].y = (int) d_coord[1];
    ++fill_count;

    if(end == 1 && graphics)
    {    
	--fill_count;
        XFillPolygon(display, win, gc_overlay, fill, fill_count,
	   Nonconvex, CoordModeOrigin);

	fill_count = 0;
    }
}


/*********************************************************************/ 
/*                                                                   */ 
/*  OVERLAY_DCIRCLE draws a circle on current image.                 */ 
/*  Circle is correctly resized and centered as image is zoomed.     */ 
/*                                                                   */  
/*  Center comes from previous overlay_dmov() call.                  */ 
/*  Input: radius (double) in SC coordinates                         */ 
/*                                                                   */ 
/*********************************************************************/

void
overlay_dcircle(radius)   /* draw circle */
double radius;
{
    double d_coord[2];
    G_graphic *g_g;

    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_DCIRCLE;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_dcircle.g_win = win;
	g_g->g_dcircle.gc_color = gc_overlay;
	g_g->g_dcircle.g_bundle = current_bundle;
	g_g->g_dcircle.x = graphics_dx;
	g_g->g_dcircle.y = graphics_dy;
	g_g->g_dcircle.radius = radius;
    }

    if (!graphics)
	return;

    d_coord[0] = graphics_dx;
    d_coord[1] = graphics_dy;
    dSC_to_WIN(d_coord);

    radius = radius * win_zoom;
    draw_circle(win, gc_overlay, (int) d_coord[0], (int) d_coord[1], (unsigned) radius);
}

/*********************************************************************/ 
/*                                                                   */ 
/*  OVERLAY_DCIRCLE draws a circle on current image.                 */ 
/*  Circle is correctly resized and centered as image is zoomed.     */ 
/*                                                                   */  
/*  Center comes from previous overlay_dmov() call.                  */ 
/*  Input: radius (double) in SC coordinates                         */ 
/*                                                                   */ 
/*********************************************************************/

void
overlay_darc(radius, angle1, angle_extent)   /* draw arc */
double radius;
double angle1, angle_extent;
{
    double d_coord[2];
    G_graphic *g_g;

    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_DARC;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_darc.g_win = win;
	g_g->g_darc.gc_color = gc_overlay;
	g_g->g_darc.g_bundle = current_bundle;
	g_g->g_darc.x = graphics_dx;
	g_g->g_darc.y = graphics_dy;
	g_g->g_darc.radius = radius;
	g_g->g_darc.angle1 = angle1;
	g_g->g_darc.angle_extent = angle_extent;
    }

    if (!graphics)
	return;

    d_coord[0] = graphics_dx;
    d_coord[1] = graphics_dy;
    dSC_to_WIN(d_coord);

    radius = radius * win_zoom;
    draw_arc(win, gc_overlay, (int) d_coord[0], (int) d_coord[1], (unsigned) radius, angle1, angle_extent);
}


/*********************************************************************/ 
/*                                                                   */ 
/*  OVERLAY_ELLIPSE draws an ellipse on current image.               */ 
/*  Ellipse is correctly resized and centered as image is zoomed.    */ 
/*                                                                   */  
/*  Center comes from previous overlay_mov() call.                   */ 
/*  Input: width, height, (integer) in SC coordinates                */ 
/*         angle (integer degrees)                                   */ 
/*                                                                   */ 
/*********************************************************************/

void
overlay_dellipse(width, height, angle)
double width, height, angle;
{
    double d_coord[2];
    G_graphic *g_g;

    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_ELLIPSE;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_ellipse.g_win = win;
	g_g->g_ellipse.gc_color = gc_overlay;
	g_g->g_ellipse.g_bundle = current_bundle;
	g_g->g_ellipse.x = graphics_dx;
	g_g->g_ellipse.y = graphics_dy;
	g_g->g_ellipse.width = width;
	g_g->g_ellipse.height = height;
	g_g->g_ellipse.angle = angle;
    }

    d_coord[0] = graphics_dx;
    d_coord[1] = graphics_dy;
    dSC_to_WIN(d_coord);

    height = height * win_zoom;
    width = width * win_zoom;

    if(graphics)
        draw_ellipse(win, gc_overlay, (int) d_coord[0], (int) d_coord[1], 
	    (int) width, (int) height, angle);
}

/*********************************************************************/ 
/*                                                                   */ 
/*  OVERLAY_STRING writes text on current image.                     */ 
/*  Text origin is correctly placed as image is zoomed, but the text */ 
/*  size is not changed.  It is tied to the X window font.           */  
/*  The font has been set in label()  (the annotate command)         */  
/*                                                                   */  
/*  Input: x, y (integers) in SC coordinates                         */ 
/*         text  (char *)                                            */ 
/*                                                                   */ 
/*********************************************************************/

void
overlay_string(x, y, text, desired_font_id, font_size)   /* do text */
double x, y;
char *text;
Font desired_font_id;
double font_size;   /* only matters for "agra" font */
{
    double d_coord[2];
    G_graphic *g_g;

    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_STRING;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_string.g_win = win;
	g_g->g_string.gc_color = gc_overlay;
	g_g->g_string.g_bundle = current_bundle;
	g_g->g_string.x = x;
	g_g->g_string.y = y;
	g_g->g_string.g_font = desired_font_id;
	g_g->g_string.font_size = font_size;
	g_g->g_string.string = (char *) malloc(strlen(text) + 1);
	strcpy(g_g->g_string.string , text);
    }

    if (desired_font_id == agra_font_id)
    {
	mlabel(x,  y, text, font_size, 0);
	return;
    }

    d_coord[0] = x;
    d_coord[1] = y;
    dSC_to_WIN(d_coord);

    if(!graphics)
	return;

    if (desired_font_id != default_font_id)
	XSetFont(display, gc_overlay, desired_font_id);

    XDrawString(display, win, gc_overlay, (int) d_coord[0], 
	(int) d_coord[1], text, strlen(text));

    if (desired_font_id != default_font_id)
	XSetFont(display, gc_overlay, default_font_id);
}

/*********************************************************************/ 
/*                                                                   */ 
/*  MARK_THE_SPOT puts a 7x7 cross on the current image.             */ 
/*  The cross center is correctly placed as image is zoomed, but     */ 
/*  the size is always 7x7 pixels.                                   */  
/*                                                                   */  
/*  Input: x, y (integers) in SC coordinates                         */ 
/*                                                                   */ 
/*********************************************************************/

void
mark_the_spot(x, y)
int x, y;
{
    overlay_setup('g');
    overlay_color(2);

    do_mark((double)x, (double)y, CROSS, 7, 0, 0);
    image_setup(plane);
}

/*********************************************************************/ 
/*                                                                   */ 
/*  DO_MARK puts a graphics mark on the current image.               */ 
/*  The mark center is correctly placed as image is zoomed, but      */ 
/*  the size does not change.                                        */  
/*                                                                   */  
/*  Input: x, y (double) in SC coordinates                           */ 
/*         markertype (integer - enumerated)                         */ 
/*         markersize (integer)                                      */ 
/*         markersize2 (integer)                                     */ 
/*         markersize3 (integer)                                     */ 
/*                                                                   */ 
/*********************************************************************/

void
do_mark(x, y, markertype, markersize, markersize2, markersize3)
double x, y;
int markertype, markersize, markersize2, markersize3;
{
    XPoint points[5];
    XSegment segments[5];
    int i_coord[2], height, width;
    double angle, d_coord[2];
    G_graphic *g_g;


    g_g = add_graphic();
    if (g_g != NULL)
    {
	g_g->type = G_MARK;
	g_g->g_any.overlay_is_on = TRUE;
	g_g->g_mark.g_win = win;
	g_g->g_mark.gc_color = gc_overlay;
	g_g->g_mark.g_bundle = current_bundle;
	g_g->g_mark.x = x;
	g_g->g_mark.y = y;
	g_g->g_mark.mark_type = markertype;
	g_g->g_mark.mark_size = markersize;
	g_g->g_mark.mark_size2 = markersize2;
	g_g->g_mark.mark_size3 = markersize3;
	g_g->g_mark.table_line = table_linenumber;
	g_g->g_mark.table_filename = table_nam;
    }

    d_coord[0] = x;
    d_coord[1] = y;
    dSC_to_WIN(d_coord);
    i_coord[0] = Round(d_coord[0]);
    i_coord[1] = Round(d_coord[1]);

    if (!graphics)
	return;

    switch(markertype)
    {
	case DOT:
	    XFillRectangle(display, win, gc_overlay, 
		i_coord[0] - markersize, i_coord[1] - markersize, 
		(unsigned ) (2 * markersize), 
		(unsigned) (2 * markersize));
	    break;
	case CROSS:
	    XDrawLine(display, win, gc_overlay, 
		i_coord[0], i_coord[1] - markersize, 
		i_coord[0], i_coord[1] + markersize);
	    XDrawLine(display, win, gc_overlay, 
		i_coord[0] - markersize, i_coord[1], 
		i_coord[0] + markersize, i_coord[1]);
	    break;
	case ASTERISK:
	    segments[0].x1 = i_coord[0] - markersize;
	    segments[0].y1 = i_coord[1] - markersize;
	    segments[0].x2 = i_coord[0] + markersize;
	    segments[0].y2 = i_coord[1] + markersize;
	    segments[1].x1 = i_coord[0] - markersize;
	    segments[1].y1 = i_coord[1] + markersize;
	    segments[1].x2 = i_coord[0] + markersize;
	    segments[1].y2 = i_coord[1] - markersize;
	    segments[2].x1 = i_coord[0] - markersize;
	    segments[2].y1 = i_coord[1] ;
	    segments[2].x2 = i_coord[0] + markersize;
	    segments[2].y2 = i_coord[1] ;
	    XDrawSegments(display, win, gc_overlay, segments, 3);
	    break;
	case CIRCLE:
	    draw_circle(win, gc_overlay, i_coord[0], i_coord[1], 
		(unsigned) markersize);
	    break;
	case X:
	    segments[0].x1 = i_coord[0] - markersize;
	    segments[0].y1 = i_coord[1] - markersize;
	    segments[0].x2 = i_coord[0] + markersize;
	    segments[0].y2 = i_coord[1] + markersize;
	    segments[1].x1 = i_coord[0] - markersize;
	    segments[1].y1 = i_coord[1] + markersize;
	    segments[1].x2 = i_coord[0] + markersize;
	    segments[1].y2 = i_coord[1] - markersize;
	    XDrawSegments(display, win, gc_overlay, segments, 2);
	    break;
	case BOX:
	    XDrawRectangle(display, win, gc_overlay, 
		i_coord[0] - markersize, i_coord[1] - markersize, 
		(unsigned) (2 * markersize), 
		(unsigned) (2 * markersize));
	    break;
	case DIAMOND:
	    points[0].x = i_coord[0] - markersize;
	    points[0].y = i_coord[1] ;
	    points[1].x = i_coord[0] ;
	    points[1].y = i_coord[1] - markersize;
	    points[2].x = i_coord[0] + markersize;
	    points[2].y = i_coord[1] ;
	    points[3].x = i_coord[0] ;
	    points[3].y = i_coord[1] + markersize;
	    points[4].x = i_coord[0] - markersize;
	    points[4].y = i_coord[1] ;
	    XDrawLines(display, win, gc_overlay, points, 5, 
		CoordModeOrigin);
	    break;
	case TRIANGLE:
	    points[0].x = i_coord[0];
	    points[0].y = i_coord[1] - markersize;
	    points[1].x = i_coord[0] - markersize;
	    points[1].y = i_coord[1] + markersize;
	    points[2].x = i_coord[0] + markersize;
	    points[2].y = i_coord[1] + markersize;
	    points[3].x = i_coord[0];
	    points[3].y = i_coord[1] - markersize;
	    XDrawLines(display, win, gc_overlay, points, 4, 
		CoordModeOrigin);
	    break;
	case ELLIPSE:
	    width = markersize * win_zoom;
	    height = markersize2 * win_zoom;
	    angle = (double) markersize3;
	    draw_ellipse(win, gc_overlay, i_coord[0], i_coord[1], 
		width, height, angle);
	    break;
    }
}


#define DIST_INIT 1000000

int
find_mark(xin,yin)
int xin, yin;     /* integer x,y in WIN coordinates */
{
    G_graphic *g_g, *g_save;
    double x, y, deltax, deltay, dist, dist_save;
    double d_coord[2];

    if (g_graphic == NULL)
    {
	error1("no marks found");
	return(0);
    }
    
    if (debug)
	fprintf(debug_file,
	    "RBH find_mark entry:  xin = %d  yin = %d\n", xin,yin);
    d_coord[0] = xin;
    d_coord[1] = yin;
    dWIN_to_SC(d_coord);
    x = d_coord[0];
    y = d_coord[1];
    dist_save = DIST_INIT;
    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if ((g_g->type == G_MARK) && (g_g->g_any.g_win == win))
	{
	    deltax = g_g->g_mark.x - x;
	    deltay = g_g->g_mark.y - y;
	    dist = deltax * deltax + deltay * deltay;
	    if (debug)
	    {
		fprintf(debug_file,
		    "comparing mark x = %g  y = %g  dist = %g\n", 
		    g_g->g_mark.x, g_g->g_mark.y, dist);
		fprintf(debug_file, "   from file %s  line %d\n",
		    g_g->g_mark.table_filename, g_g->g_mark.table_line);
	    }
	    if (dist < dist_save)
	    {
		/* this mark is closer - save it */
		dist_save = dist;
		g_save = g_g;
	    }
	}
    }
    if (dist_save == DIST_INIT)
    {
	error1("no marks found on current image");
	return(0);
    }
    report_mark(g_save);
    return(1);
}

static void
report_mark(g_g)
G_graphic *g_g;
{
    double d_coord[2];
    char s_coord[2][40];
    char in_line[501], *coord_str;
    FILE *fd;
    int i;
    int coord_sys;
    int jsys;
    double equinox;

    d_coord[0] = g_g->g_mark.x;
    d_coord[1] = g_g->g_mark.y;
    dSC_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    strcpy(last_coord[0], s_coord[0]);
    strcpy(last_coord[1], s_coord[1]);

    if (g_g->g_mark.table_line == 0)   /* if not from a table file */
    {
	fprintf(session, "mark did not come from a table file\n");
	fprintf(session, "mark at %s\n\n", coord_str);

	if (s0_server_mode)
	{
	    if(server_mode == FALSE)
		printf(":0\n");
	}
	else
	{
	    if(server_mode == FALSE)
	    {
	       printf("mark did not come from a table file\n");
	       printf("mark at %s\n\n", coord_str);
	    }
	    else
	    {
	       if (point_mode)
	       {
		   srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	       }
	       else
	       {
		   /* output table file line */
		    printf("%-12.12s %-12.12s %-10.10s %-6d      %-*.*s\n",
			s_coord[0], s_coord[1], coord_name(coord_sys),
			0, MAXPATHLEN+12, MAXPATHLEN+12, "");
	       }
	    }
	}
    }
    else   /* from a table file */
    {
	if (s0_server_mode)
	{
	    if(server_mode == FALSE)
		printf("%s:%d\n",
		    g_g->g_mark.table_filename, g_g->g_mark.table_line);
	}
	else
	{

	    if(server_mode == FALSE)
		printf("mark came from line %d of file %s, printed below:\n",
		    g_g->g_mark.table_line, g_g->g_mark.table_filename);
	    else
	    {
	       if (point_mode)
	       {
		   srv_coord("coord", coord_sys, s_coord, equinox, jsys);
		   srv_string("filename", g_g->g_mark.table_filename);
		   sprintf(server_str, "%d", g_g->g_mark.table_line);
		   srv_real("table_line", server_str);
	       }
	       else if (ads_server_mode)
	       {
		   srv_string("state", "point");
		   srv_coord("coord", coord_sys, s_coord, equinox, jsys);
		   srv_string("filename", g_g->g_mark.table_filename);
		   sprintf(server_str, "%d", g_g->g_mark.table_line);
		   srv_real("table_line", server_str);
	       }
	       else
	       {
		   /* output table file line */
		    printf("%-12.12s %-12.12s %-10.10s %-6d      %-*.*s\n",
			s_coord[0], s_coord[1], coord_name(coord_sys),
			g_g->g_mark.table_line, MAXPATHLEN+12, MAXPATHLEN+12,
			g_g->g_mark.table_filename);
	       }
	    }

	    fprintf(session, "mark came from line %d of file %s, printed below:\n",
		g_g->g_mark.table_line, g_g->g_mark.table_filename);

	    fd = fopen(g_g->g_mark.table_filename, "r");
	    if (fd == NULL)
	    {
		sperror(g_g->g_mark.table_filename, errno);
		return;
	    }
	    i = g_g->g_mark.table_line;
	    while (i--)
	    {
		if (fgets(in_line, 500, fd) == NULL)
		{
		    sprintf(server_str, "error reading file %s\n", 
			g_g->g_mark.table_filename);
		    error1(server_str);
		    fclose(fd);
		    return;
		}
	    }
	    fclose(fd);
	    if(server_mode == FALSE)
		printf("%s", in_line);
	    fprintf(session, "%s", in_line);
	}
    }
}

static int
find_bundle(xin,yin)
int xin, yin;     /* integer x,y in WIN coordinates */
{
    G_graphic *g_g, *g_save;
    double x, y, deltax, deltay, dist, dist_save;
    double d_coord[2];

    if (g_graphic == NULL)
    {
	error1("no graphics found");
	return(0);
    }
    
    if (debug)
	fprintf(debug_file, "RBH find_bundle entry:  xin = %d  yin = %d\n", xin,yin);
    d_coord[0] = xin;
    d_coord[1] = yin;
    dWIN_to_SC(d_coord);
    x = d_coord[0];
    y = d_coord[1];
    dist_save = DIST_INIT;
    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if (g_g->g_any.g_win == win)
	{
	    deltax = g_g->g_any.x - x;
	    deltay = g_g->g_any.y - y;
	    dist = deltax * deltax + deltay * deltay;
	    if (debug)
		fprintf(debug_file,
		    "comparing graphic x = %g  y = %g  dist = %g\n", 
		    g_g->g_any.x, g_g->g_any.y, dist);
	    if (dist < dist_save)
	    {
		/* this graphic is closer - save it */
		dist_save = dist;
		g_save = g_g;
	    }
	}
    }
    if (dist_save == DIST_INIT)
    {
	error1("no graphics found");
	return(0);
    }
    else
    {

	return(g_save->g_any.g_bundle);
    }
}

/******************************************************************/
/* 								  */
/* ERASE_GRAPHICS_TRACKBALL                                       */
/* use trackball to select graphics bundle for erasing            */
/* 								  */
/******************************************************************/


void
erase_graphics_trackball()
{
    int xwin, ywin;
    char anin[MAXLINE];
    int dummy_dx, dummy_dy;
    int what;
    static int bundle, highlighted = FALSE;;

    xwin = 0;  /* keep purify happy */
    ywin = 0;  /* keep purify happy */

    for (;;)
    {
	wait_for_event(anin, &what, &xwin, &ywin, &dummy_dx, &dummy_dy, 0);
	if (highlighted)
	{
	    unhighlight_graphics_bundle(bundle);
	    highlighted = FALSE;
	}
	if(what == Button1)
	{
	    bundle = find_bundle(xwin, ywin);
	    if (bundle)
	    {
		erase_graphics_bundle(bundle);
		expose_all();
	    }
	}
	else if (what == Button2)
	{
	    break;
	}
	else if (what == Button3)
	{
	    bundle = find_bundle(xwin, ywin);
	    if (bundle)
	    {
		highlight_graphics_bundle(bundle);
		highlighted = TRUE;
	    }
	}
	else
	    break;
    }
}


void
gexamine(argc, argv)
int argc;
char **argv;
{
    G_graphic *g_g;
    double d_coord[2];
    int x, y; 
    int INTERIOR=1; 
    char *cmd_name;

    cmd_name = argv[0];

    set_current_window();

    if (g_graphic == NULL)
	return;
    
    enable_control_c();


    if (argc > 1)  /* if he specified an area number */
    {
	if (!switch_area(argv[1]))
	    return;
    }

    if(debug)
	fprintf(debug_file, "graphics_examine:  area number %d \n", area_number);

    if (area[area_number].area_type == 0)
    {
	sprintf(server_str,
	    "graphics_examine error: area %d has not been defined", area_number);
	error1(server_str);
	return;
    }

    if (server_mode == TRUE)
    {
	if (!ads_server_mode)
	{
	    srvflush(argv[0], 1, 0);
	    /* table file header */
	    printf("table\n");
	    printf(
	    "|x           |y           |coord_sys |table_line |filename");
	    printf("%-*.*s|\n", MAXPATHLEN+12, MAXPATHLEN+12, "");
	}
    }

    segment_setup();  /* calculate parameters defining each line */ 

    /* Loop through the marks */

    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if ((g_g->type == G_MARK) && (g_g->g_any.g_win == win))
	{

	    d_coord[0] = g_g->g_mark.x;
	    d_coord[1] = g_g->g_mark.y;
	    dSC_to_IRAF(d_coord);
	    x = Round(d_coord[0]);
	    y = Round(d_coord[1]);
	    x_list(y);     /* list of boundary crossings */
	    if(position(x)==INTERIOR && !on_horizontal(x, y)) 
	    {
		report_mark(g_g);
		srvflush(cmd_name, 0, 0);
		srvinit();
	    }
	}

	if (control_c)
	{
	    error1("graphics_examine interrupted by control c");
	    fprintf(session, "\ngraphics_examine interrupted by control c\n");
	    return;
	}
    }
    if (server_mode == TRUE)
    {
	if (ads_server_mode)
	    srv_string("state", "done");
	else
	    printf("table_end\n");
    }
}


/*********************************************************************/ 
/*                                                                   */ 
/*  S_MONGO does graphic commands coming from Super Mongo, or        */ 
/*  any program which sends commands in the specified format.        */ 
/*  See doc file smongo.s                                            */  
/*                                                                   */  
/*********************************************************************/

void
smongo(argc, argv)
int argc;
char **argv;
{
    double d_coord1[2];
    int i_coord1[2], i_coord2[2];
    static char color = 'w';
    static double ncar_width = 32767;
    static double ncar_height = 32767;
    static double ncar_x_offset = 0;
    static double ncar_y_offset = 0;
    static int getdims = 0;
    int cmdc;
    char *cmdv[10];

    if (save_line[0] == '\0')
	strcpy(save_line, " ");   /* prevent prompt for next command */

    if (argc < 2)
	return;

    set_current_window();

    overlay_setup(color);

    if (strcmp(argv[1], "MOVE") == 0)
    {
	if (win == 0)
	    return;
	d_coord1[0] = (atof(argv[2]) - ncar_x_offset) * pixels / ncar_width;
	d_coord1[1] = (atof(argv[3]) - ncar_y_offset) * lines / ncar_height;
	overlay_dmov(d_coord1[0], d_coord1[1]);
    }
    else if (strcmp(argv[1], "DRAW") == 0)
    {
	if (getdims)
	{
	    /* collecting corner values */
	    getdims--;
	    if (getdims)
	    {
		/* first time thru - we have upper right values */
		ncar_width = atof(argv[2]);   /* save upper right x */
		ncar_height = atof(argv[3]);   /* save upper right y */
		return;
	    }
	    /* second time thru - we have lower left values */
	    ncar_x_offset = atof(argv[2]);
	    ncar_y_offset = atof(argv[3]);
	    ncar_width -= ncar_x_offset;
	    ncar_height -= ncar_y_offset;
	    return;
	}
	if (win == 0)
	    return;
	d_coord1[0] = (atof(argv[2]) - ncar_x_offset) * pixels / ncar_width;
	d_coord1[1] = (atof(argv[3]) - ncar_y_offset) * lines / ncar_height;
	overlay_ddraw(d_coord1[0], d_coord1[1]);
    }
    else if (strcmp(argv[1], "LINE") == 0)
    {
	if (win == 0)
	    return;
	i_coord1[0] = atof(argv[2]) * pixels / 32767;
	i_coord1[1] = atof(argv[3]) * lines / 32767;
	i_coord2[0] = atof(argv[4]) * pixels / 32767;
	i_coord2[1] = atof(argv[5]) * lines / 32767;
	/*
	WIN_to_SC(i_coord1);
	WIN_to_SC(i_coord2);
	*/
	overlay_dmov((double) i_coord1[0], (double) i_coord1[1]);
	overlay_ddraw((double) i_coord2[0], (double) i_coord2[1]);
    }
    else if (strcmp(argv[1], "CHAR") == 0)
    {
	if (strcmp(argv[2], "(null)") != 0)
	{
	if (win == 0)
	    return;
	i_coord1[0] = atof(argv[3]) * pixels / 32767;
	i_coord1[1] = atof(argv[4]) * lines / 32767;
	SC_to_WIN(i_coord1);
	XDrawString(display, win, gc_overlay, i_coord1[0], i_coord1[1], 
	    argv[2], strlen(argv[2]));
	}
    }
    else if (strcmp(argv[1], "COLOR") == 0)
    {
	switch(argv[2][0])
	{
	    case 'r':
	    case 'g':
	    case 'b':
	    case 'w':
	    case 'B':
		color = argv[2][0];
		break;
	}
    }
    else if (strcmp(argv[1], "SIZE") == 0)
    {
	if (win == 0)
	    printf("S: 0 0\n");
	else
	    printf("S: %d %d\n", lines, pixels);
    }
    else if (strcmp(argv[1], "DIMENSIONS") == 0)
    {
	/* next 2 MOVE-DRAW pairs give upper right and lower left */
	/* edges of plot in the coordinates NCAR will be sending */
	getdims = 2;
    }
    else if (strcmp(argv[1], "TEK") == 0)
    {
	for (cmdc = 0; cmdc < argc-1; cmdc++)
	    cmdv[cmdc] = argv[cmdc+1];
	tek_cmd(cmdc, cmdv);
    }
}


/*********************************************************************/ 
/*                                                                   */ 
/*  FOV_JOY uses the mouse to move the field-of-view graphic         */
/*                                                                   */ 
/*********************************************************************/

 
#define POS 0
#define ROT 1

void
fov_joy(rot_angle, d_coord) 
double *rot_angle;
double d_coord[2]; 
{ 
    GC gc_save;
    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int x, y;
    double x_SC, y_SC;                 /* SC coordinates */
    int dx, x_center;         /* WIN coordinates */
    int dy, y_center;
    int cen_size;
    double xdist, ydist;
 

    cen_size = POS;

    XQueryPointer(display, win,
	&root, &child, &root_x, &root_y, 
	&x_center, &y_center, &keys_buttons);

    x = x_center;  /* keep purify happy */
    y = y_center;  /* keep purify happy */

    gc_save = gc_overlay;
    gc_overlay = gcxor;
    g_repainting = 1;   /* dont add to graphics list */

    set_motion_notify(win);
    done = 0;
    while(!done)
    {
	d_coord[0] = x_center;
	d_coord[1] = y_center;
	dWIN_to_SC(d_coord);
	x_SC = d_coord[0];
	y_SC = d_coord[1];

	fov_plot( x_SC, y_SC, *rot_angle, 1);

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	/* undraw old graphics */
	fov_plot( x_SC, y_SC, *rot_angle, 1);

	switch(what)
	{
	case Button1:
	    cen_size = POS;
	    /* jump to old center - better than a snap with 1st motion */
	    XWarpPointer(display, None, win, 0,0,0,0, x_center, y_center);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = ROT;
	    break;
	case POINTERMOTION:
	    if (debug)
		fprintf(debug_file, "fov_joy:  x = %d  y = %d\n", x, y);

	    if(cen_size == POS)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == ROT)
	    {
		xdist = x - x_center;
		ydist = y - y_center;
		if (fabs(xdist) > 30)
		    *rot_angle += .5 * (double)dy/xdist;
		if (fabs(ydist) > 30)
		    *rot_angle += .5 * (double)dx/ydist;
		    
	    }
	    break;
	}

    }
 
    clear_motion_notify();
    gc_overlay = gc_save;
    g_repainting = 0;

    overlay_setup('g');
    overlay_color(2);      /* draw the vector in green  */ 
    next_bundle();
    fov_plot( x_SC, y_SC, *rot_angle, 0);

    image_setup(plane);   /* reset enabled planes */ 
} 


struct my_error_mgr 
{
    struct jpeg_error_mgr pub;    /* "public" fields */
    jmp_buf setjmp_buffer;        /* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
*/

METHODDEF(void)
my_error_exit (j_common_ptr cinfo)
{
    char jpeg_error_message[JMSG_LENGTH_MAX];

    /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
    my_error_ptr myerr = (my_error_ptr) cinfo->err;

    /* format the message. */
    (*cinfo->err->format_message) (cinfo, jpeg_error_message);
    error1(jpeg_error_message);

    /* Return control to the setjmp point */
    longjmp(myerr->setjmp_buffer, 1);
}



/*********************************************************************/ 
/*                                                                   */ 
/*  MEMDUMP  creates GIF file in non-graphics mode                   */
/*                                                                   */ 
/*********************************************************************/

void
memdump(argc, argv)
int argc;
char *argv[];
{
    int           i, j, indexx, indexy, finalindex, nbytes, ix, jy;
    int           colorstyle, numcols;
    int color, borderColor;
    int           markertype, markersize, markersize2, markersize3;
    double        ximg, yimg, ximg1, yimg1, radimg;
    double        widthimg, heightimg, angleimg;
    char          rmap[256], gmap[256], bmap[256];
    char          fname[256];
    FILE         *fp;
    int do_gif;
    struct jpeg_compress_struct cinfo;
    struct my_error_mgr jerr;
    int row_stride;               /* physical row width in image buffer */
    int quality;
    JSAMPARRAY buffer;            /* Output row buffer */
    JSAMPLE *buffer_ptr;
    unsigned char *pixp;
    INT32 *pix32;
    Pixel tempval;
    int redshift, greenshift, blueshift;
    img_struct img_ptr;
    char msg[80];


    G_graphic    *g_g;


    if(graphics)
    {
	error1("Must be in non-graphics mode.");
        return;
    }

    if(argc < 2)
    {
	error1("No output file name given.");
	return;
    }
    if (argc > 2)
    {
	quality = atoi(argv[2]);
	if ((quality <= 0) || (quality > 100))
	{
	    sprintf(msg, 
		"memdump error:  quality of '%s' is not between 0 and 100\n", 
		argv[2]);
	    error1(msg);
	    return;
	}
    }
    else
	quality = 75;

    borderColor = 0;
    if (argc > 3)
       borderColor = 245;
   
    if (strstr(argv[1], ".gif"))
    {
	do_gif = 1;
	if (depth > 8)
	{
	    error1("Cannot do GIF memory dump in 24 plane mode");
	    return;
	}
    }
    else if (strstr(argv[1], ".jp"))
    {
	do_gif = 0;
    }
    else
    {
	error1("md error:  filename must end with .gif or .jpg");
	return;
    }

    strcpy(fname, expand_path(argv[1]));

    if (win == 0)
    {
	error1("No current window");
	return;
    }
	
    nbytes = win_width * win_height;

    if (depth > 8)
    {
	INT32 *image_org32;

	image_org32 = (INT32 *) image_org;

	img_ptr.dmpimg32 = (INT32 *) malloc((size_t) nbytes * 4);

	for(j=0; j<win_height; ++j)
	{
	    for(i=0; i<win_width; ++i)
	    {
	       ix = Round(((double)i + 0.5) / win_zoom - 0.5 + win_x_offset);
	       jy = Round(((double)j + 0.5) / win_zoom - 0.5 + win_y_offset);

	       if(ix >= 0 && ix < width_org && jy >= 0 && jy < height_org)
		  img_ptr.dmpimg32[j*win_width + i] = image_org32[jy*width_org + ix];

	       else
		  img_ptr.dmpimg32[j*win_width + i] = borderColor;
	    }
	}

	/* find out how much to shift each pixval */
	/* i.e how many shifts needed to right justify mask value */
	/* first red */
	tempval = red_mask;
	redshift = 0;
	while ((tempval & 0x1) == 0)
	{
	    redshift++;
	    tempval >>= 1;
	}

	/* next green */
	tempval = green_mask;
	greenshift = 0;
	while ((tempval & 0x1) == 0)
	{
	    greenshift++;
	    tempval >>= 1;
	}

	/* last blue */
	tempval = blue_mask;
	blueshift = 0;
	while ((tempval & 0x1) == 0)
	{
	    blueshift++;
	    tempval >>= 1;
	}

    }
    else
    {
	/* depth == 8 */
	img_ptr.dmpimg = (unsigned char *) malloc((size_t) nbytes);

	for(j=0; j<win_height; ++j)
	{
	    for(i=0; i<win_width; ++i)
	    {
	       ix = Round(((double)i + 0.5) / win_zoom - 0.5 + win_x_offset);
	       jy = Round(((double)j + 0.5) / win_zoom - 0.5 + win_y_offset);

	       if(ix >= 0 && ix < width_org && jy >= 0 && jy < height_org)
		  img_ptr.dmpimg[j*win_width + i] = image_org[jy*width_org + ix];

	       else
		  img_ptr.dmpimg[j*win_width + i] = borderColor;
	    }
	}
    }


    fp = fopen(fname, "wb");

    if(fp == (FILE *) NULL)
    {
	error1("Cannot open output GIF file.");
	return;
    }

    if(g_graphic != NULL)
    {
       if(debug)
	  fprintf(stderr, "\n\nGraphics:\n\n");

       g_g = g_graphic;
       do
       {
	   if (g_g->g_any.g_win == win)
	   {
	   switch(g_g->type)
	   {
	       case G_DLINE:
		   if(debug)
		      printf("DLINE:     win = %lx  GC = 0x%p  x = %g  y = %g  x1 = %g  y1 = %g\n",
			  g_g->g_dline.g_win,
			  (void *) g_g->g_dline.gc_color,
			  g_g->g_dline.x,
			  g_g->g_dline.y,
			  g_g->g_dline.x1,
			  g_g->g_dline.y1);

		   xytoimg(g_g->g_dline.x, g_g->g_dline.y, &ximg, &yimg);
		   xytoimg(g_g->g_dline.x1, g_g->g_dline.y1, &ximg1, &yimg1);

		   color = dmpcolor(g_g->g_dline.gc_color);
		   dmpline(color, img_ptr, win_width, win_height, 
			ximg, yimg, ximg1, yimg1);
	       break;


	       case G_DCIRCLE:
		   if(debug)
		      printf("DCIRCLE:   win = %lx  GC = 0x%p  x = %g  y = %g  radius = %g\n",
			  g_g->g_dcircle.g_win,
			  (void *) g_g->g_dcircle.gc_color,
			  g_g->g_dcircle.x,
			  g_g->g_dcircle.y,
			  g_g->g_dcircle.radius);

		   xytoimg(g_g->g_dcircle.x, g_g->g_dcircle.y, &ximg, &yimg);
		   radimg = g_g->g_dcircle.radius * win_zoom;

		   color = dmpcolor(g_g->g_dline.gc_color);
		   newdmpcircle(color, img_ptr, win_width, win_height,
			  (int) ximg, (int) yimg, (int) radimg);
#ifdef NOTDEF
		   dmpcircle(color, img_ptr, win_width, win_height,
			  ximg, yimg, radimg);
		   dmpellipse(color, img_ptr, win_width, win_height,
		       ximg, yimg, radimg, radimg, 0.0);
#endif /* NOTDEF */

	       break;


	       case G_ELLIPSE:
		   if(debug)
		      printf("ELLIPSE:   win = %lx  GC = 0x%p  x = %g  y = %g  width = %g  height = %g  angle = %g\n",
			  g_g->g_ellipse.g_win,
			  (void *) g_g->g_ellipse.gc_color,
			  g_g->g_ellipse.x,
			  g_g->g_ellipse.y,
			  g_g->g_ellipse.width,
			  g_g->g_ellipse.height,
			  g_g->g_ellipse.angle);

		   xytoimg(g_g->g_ellipse.x, g_g->g_ellipse.y, &ximg, &yimg);

		   widthimg = g_g->g_ellipse.width * win_zoom / 2.;
		   heightimg = g_g->g_ellipse.height * win_zoom / 2.;
		   angleimg = g_g->g_ellipse.angle;

		   color = dmpcolor(g_g->g_dline.gc_color);
		   dmpellipse(color, img_ptr, win_width, win_height,
		       ximg, yimg, widthimg, heightimg, angleimg);

	       break;


	       case G_MARK:
		   if(debug)
		      printf("MARK:      win = %lx  GC = 0x%p  x = %g  y = %g  type = %d size = %d, %d, %d\n",
			  g_g->g_mark.g_win,
			  (void *) g_g->g_mark.gc_color,
			  g_g->g_mark.x,
			  g_g->g_mark.y,
			  g_g->g_mark.mark_type,
			  g_g->g_mark.mark_size,
			  g_g->g_mark.mark_size2,
			  g_g->g_mark.mark_size3);
		   
		   xytoimg(g_g->g_mark.x, g_g->g_mark.y, &ximg, &yimg);

		   color = dmpcolor(g_g->g_dline.gc_color);
				
		   markertype  = g_g->g_mark.mark_type;
		   markersize  = g_g->g_mark.mark_size;
		   markersize2 = g_g->g_mark.mark_size2;
		   markersize3 = g_g->g_mark.mark_size3;

		   switch(markertype)
		   {
		       case DOT:
		       for(jy = -markersize; jy <= markersize; ++jy)
		       {
		           for(ix = -markersize; ix <= markersize; ++ix)
			   {
			       indexx = Round(ximg) + ix;
			       indexy = Round(yimg) + jy;

			       if(indexx >= 0 && indexx <= win_width
			       && indexy >= 0 && indexy <= win_height) 
			       {
			           finalindex = indexy * win_width + indexx;

				   if (depth > 8)
				       img_ptr.dmpimg32[finalindex] = color;
				   else
				       img_ptr.dmpimg[finalindex] = (char) color;
			       }
			   }
		       }
		       break;


		       case CROSS:
		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg, 
			       ximg + markersize, yimg);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg, yimg - markersize, 
			       ximg, yimg + markersize);
		       break;


		       case ASTERISK:
		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg, 
			       ximg + markersize, yimg);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg, yimg - markersize, 
			       ximg, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg - markersize, 
			       ximg + markersize, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg + markersize, yimg - markersize, 
			       ximg - markersize, yimg + markersize);
		       break;


		       case CIRCLE:
			   radimg = markersize * win_zoom;
		           dmpellipse(color, img_ptr, win_width, win_height,
		               ximg, yimg, radimg, radimg, 0.0);
		       break;


		       case X:
		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg - markersize, 
			       ximg + markersize, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg + markersize, yimg - markersize, 
			       ximg - markersize, yimg + markersize);
		       break;


		       case BOX:
		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg - markersize, 
			       ximg + markersize, yimg - markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg + markersize, yimg - markersize,
			       ximg + markersize, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg + markersize, yimg + markersize, 
			       ximg - markersize, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg + markersize, 
			       ximg - markersize, yimg - markersize);
		       break;


		       case DIAMOND:
		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg, 
			       ximg, yimg - markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg, yimg - markersize,
			       ximg + markersize, yimg);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg + markersize, yimg, 
			       ximg, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg, yimg + markersize, 
			       ximg - markersize, yimg);
		       break;


		       case TRIANGLE:
		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg, yimg - markersize, 
			       ximg - markersize, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg - markersize, yimg + markersize,
			       ximg + markersize, yimg + markersize);

		           dmpline(color, img_ptr, 
			       win_width, win_height, 
			       ximg + markersize, yimg + markersize,
			       ximg, yimg - markersize);
		       break;


		       case ELLIPSE:
			   widthimg  = markersize  * win_zoom / 2.;
			   heightimg = markersize2 * win_zoom / 2.;
			   angleimg  = markersize3;

		           dmpellipse(color, img_ptr, win_width, win_height,
		               ximg, yimg, widthimg, heightimg, angleimg);
		       break;
		   }
	       break;


	       case G_DFILLAREA:
		   /* (not implemented */
		   if(debug)
		      printf("DFILLAREA: win = %lx  GC = 0x%p  x = %g  y = %g  end = %d\n",
			  g_g->g_dfillarea.g_win,
			  (void *) g_g->g_dfillarea.gc_color,
			  g_g->g_dfillarea.x,
			  g_g->g_dfillarea.y,
			  g_g->g_dfillarea.end);
	       break;


	       case G_STRING:
		   /* (not implemented */
		   if(debug)
		      printf("STRING:    win = %lx  GC = 0x%p  x = %g  y = %g  string = %s\n",
			  g_g->g_string.g_win,
			  (void *) g_g->g_string.gc_color,
			  g_g->g_string.x,
			  g_g->g_string.y,
			  g_g->g_string.string);
	       break;


	       case G_NULL:
		   if(debug)
		      printf("NULL\n");
	       break;


	       case G_LASTNULL:
		   if(debug)
		      printf("LASTNULL\n");
	       break;
	   }
	   }

	   g_g++;
       } while (g_g->type != G_LASTNULL);
    }

    if(debug)
    {
        printf("\nOverlay colors:\n");
        for(i=0; i<noverlays; ++i)
            printf("%4d: %4d %4d %4d (%s)\n",
                i, overlays[i].red, overlays[i].green, overlays[i].blue,
		overlays[i].name);
    }

    for(i=0; i<noverlays; ++i)
    {
        exact_defs[i+241].red   = overlays[i].red;
        exact_defs[i+241].green = overlays[i].green;
        exact_defs[i+241].blue  = overlays[i].blue;
    }

    if(debug)
    {
        printf("\nColor table:\n");
        for(i=0; i<256; ++i)
            printf("%4d: %4d %4d %4d\n",
                i, exact_defs[i].red, exact_defs[i].green, exact_defs[i].blue);
    }

    for (i = 0; i < 256; i++)
    {
        rmap[i] = (char) exact_defs[i].red;
        gmap[i] = (char) exact_defs[i].green;
        bmap[i] = (char) exact_defs[i].blue;
    }

    if (do_gif)
    {
	numcols    = 256;
	colorstyle =   0;

	WriteGIF(fp, img_ptr.dmpimg, win_width, win_height, 
	    rmap, gmap, bmap, numcols, colorstyle);
    }
    else
    {
	pixp = (unsigned char *) img_ptr.dmpimg;
	pix32 =  img_ptr.dmpimg32;

    /* We have to set up the error handler first, in case the initialization
    * step fails.  (Unlikely, but it could happen if you are out of memory.)
    * This routine fills in the contents of struct jerr, and returns jerr's
    * address which we place into the link field in cinfo.
    */
	    cinfo.err = jpeg_std_error(&jerr.pub);
	    jerr.pub.error_exit = my_error_exit;
	    /* Establish the setjmp return context for my_error_exit to use. */
	    if (setjmp(jerr.setjmp_buffer)) 
	    {
		/* If we get here, the JPEG code has signaled an error.
		* We need to clean up the JPEG object, close the input file
		*/
		jpeg_destroy_compress(&cinfo);
		fclose(fp);
		goto end_md;
	    }
	    /* Now we can initialize the JPEG compression object. */
	    jpeg_create_compress(&cinfo);
	    jpeg_stdio_dest(&cinfo, fp);
  /* Step 3: set parameters for compression */

  /* First we supply a description of the input image.
   * Four fields of the cinfo struct must be filled in:
   */
  cinfo.image_width = win_width; 	/* image width and height, in pixels */
  cinfo.image_height = win_height;
  cinfo.input_components = 3;		/* # of color components per pixel */
  cinfo.in_color_space = JCS_RGB; 	/* colorspace of input image */
  /* Now use the library's routine to set default compression parameters.
   * (You must set at least cinfo.in_color_space before calling this,
   * since the defaults depend on the source color space.)
   */
  jpeg_set_defaults(&cinfo);
  /* Now you can set any non-default parameters you wish to.
   * Here we just illustrate the use of quality (quantization table) scaling:
   */
  jpeg_set_quality(&cinfo, quality, TRUE /* limit to baseline-JPEG values */);

  /* Step 4: Start compressor */

  /* TRUE ensures that we will write a complete interchange-JPEG file.
   * Pass TRUE unless you are very sure of what you're doing.
   */
  jpeg_start_compress(&cinfo, TRUE);

  /* Step 5: while (scan lines remain to be written) */
  /*           jpeg_write_scanlines(...); */

  /* Here we use the library's state variable cinfo.next_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   * To keep things simple, we pass one scanline per call; you can pass
   * more if you wish, though.
   */
  row_stride = win_width * 3;	/* JSAMPLEs per row in image_buffer */
  /* Make a one-row-high sample array that will go away when done with image */
  buffer = (*cinfo.mem->alloc_sarray)
    ((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

  enable_control_c();

  while (cinfo.next_scanline < cinfo.image_height) {
    /* jpeg_write_scanlines expects an array of pointers to scanlines.
     * Here the array is only one element long, but you could pass
     * more than one scanline at a time if that's more convenient.
     */
		buffer_ptr = &buffer[0][0];
	    if (depth > 8)
	    {
		for (i = 0; i < win_width; i++)
		{
		    *buffer_ptr++ = exact_defs[((*pix32) & red_mask)
			>> redshift].red;
		    *buffer_ptr++ = exact_defs[((*pix32) & green_mask)
			>> greenshift].green;
		    *buffer_ptr++ = exact_defs[((*pix32) & blue_mask)
			>> blueshift].blue;
		    pix32++;
		}
	    }
	    else
	    {
		for (i = 0; i < win_width; i++)
		{
		    *buffer_ptr++ = exact_defs[*pixp].red,
		    *buffer_ptr++ = exact_defs[*pixp].green,
		    *buffer_ptr++ = exact_defs[*pixp].blue;
		    pixp++;
		}
	    }
    (void) jpeg_write_scanlines(&cinfo, buffer, 1);
    if (control_c)
	break;
  }

  /* Step 6: Finish compression */

  if (!control_c)
      jpeg_finish_compress(&cinfo);
  /* After finish_compress, we can close the output file. */

  /* Step 7: release JPEG compression object */

  /* This is an important step since it will release a good deal of memory. */
  jpeg_destroy_compress(&cinfo);

  /* And we're done! */
    }

    fclose(fp);
end_md:
    if (depth > 8)
	free(img_ptr.dmpimg32);
    else
	free(img_ptr.dmpimg);
}



static void
xytoimg(xin, yin, xout, yout)
double xin, yin, *xout, *yout;
{
    *xout = (xin - win_x_offset + 0.5) * win_zoom - 0.5;
    *yout = (height_org - yin - win_y_offset - 0.5) * win_zoom - 0.5;
}



int dmpcolor(gc_color)
GC gc_color;
{
    int  icolor;
    int i;

    icolor = 0;
    for(i=1; i<noverlays; ++i)
    {
	if(gc_color == nographics_GC[i])
	{
	    icolor = i;
	    break;
	}
    }

    icolor = icolor + 241;

    if(icolor < 241) icolor = 241;
    if(icolor > 255) icolor = 255;

    if (depth > 8)
	icolor = icolor + (icolor << 8) + (icolor << 16);

    return(icolor);
}




static int
dmpline(color, img_ptr, nx, ny, x1, y1, x2, y2)

int color;
img_struct img_ptr;
unsigned int nx, ny;
double          x1, y1, x2, y2;
{
   int npnts, ix, jy;
   double dx, dy, x, y, xinc, yinc, m, minv;

   npnts = 0;

   x = x1; 
   y = y1;

   dx = x2 - x1;
   dy = y2 - y1;

   if ((dx == 0.0) && (dy == 0.0))
   {
       dx = 0.5;
       dy = 0.5;
   }

   xinc = 1.;
   if(dx < 0)
      xinc = -1.;

   yinc = 1.;
   if(dy < 0)
      yinc = -1.;


   /* Horizontal line */

   if(fabs(dx) >= fabs(dy))
   {
      m = dy/dx;

      for (;;)
      {
	 ix = (int)Round(x);
	 jy = (int)Round(y);

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	 {
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;
	 }

	 ++npnts;

	 x += xinc;
	 y += xinc * m;

	 if(fabs(x-x1) > fabs(dx))
	    break;
      }
      
      return(npnts);
   }


   /* Vertical line */

   else
   {
      minv = dx/dy;
      
      for (;;)
      {
	 ix = (int)Round(x);
	 jy = (int)Round(y);

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	 {
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;
	 }

	 ++npnts;

	 x += yinc * minv;
	 y += yinc;

	 if(fabs(y-y1) > fabs(dy))
	    break;
      }

      return(npnts);
   }
}

static void
newdmpcircle(color, img_ptr, nx, ny, xcen, ycen, radius)
int color;
img_struct img_ptr;
unsigned int             nx, ny;
int          xcen, ycen, radius;
{
    int x, y, delta, del, limit;

    x = 0;
    y = radius;
    delta = 2 * (1 - radius);
    limit = 0;
    set_pixel( x+xcen,  y+ycen, nx, ny, color, img_ptr);
    set_pixel(-x+xcen,  y+ycen, nx, ny, color, img_ptr);
    set_pixel( x+xcen, -y+ycen, nx, ny, color, img_ptr);
    set_pixel(-x+xcen, -y+ycen, nx, ny, color, img_ptr);

    while (y >= limit)
    {
	if (delta < 0)
	{
	    del = 2 * delta + 2 * y -1;
	    if (del > 0)
	    {
		x++;
		y--;
		delta += 2 * x - 2 * y + 2;
	    }
	    else
	    {
		x++;
		delta+= 2 * x + 1;
	    }
	}
	else if (delta > 0)
	{
	    del = 2 * delta - 2 * x - 1;
	    if (del > 0)
	    {
		y--;
		delta += -2 * y + 1;
	    }
	    else
	    {
		x++;
		y--;
		delta += 2 * x - 2 * y + 2;
	    }
	}
	else
	{
	    x++;
	    y--;
	    delta += 2 * x - 2 * y + 2;
	}
	set_pixel( x+xcen,  y+ycen, nx, ny, color, img_ptr);
	set_pixel(-x+xcen,  y+ycen, nx, ny, color, img_ptr);
	set_pixel( x+xcen, -y+ycen, nx, ny, color, img_ptr);
	set_pixel(-x+xcen, -y+ycen, nx, ny, color, img_ptr);
    }
}


static void
set_pixel(ix, jy, nx, ny, color, img_ptr)
int ix, jy;
unsigned int nx, ny;
int color;
img_struct img_ptr;
{
     if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
     {
	if (depth > 8)
	{
	    img_ptr.dmpimg32[jy*nx + ix] = color;
	}
	else
	{
	    img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;
	}
    }
}



#ifdef NOTDEF
static int
dmpcircle(color, img_ptr, nx, ny, xcen, ycen, radius)

int color;
img_struct img_ptr;
unsigned int             nx, ny;
double          xcen, ycen, radius;
{
   int npnts;
   int ix, jy, ixref, jyref;
   double dx, dy, x, y, diff, deltax, deltay;
   double rad;

   npnts = 0;


   /* QUADRANT 1 */

   /* Octant 2 */

   x = xcen + radius; 
   y = ycen;

   dx = radius;
   dy = 0.;

   deltax = 0.;

   for (;;)
   {
      deltax = sqrt(dx*dx - 2*dy - 1.) - dx;

      dy += 1.;
      dx += deltax;

      if(fabs(dy) > fabs(dx))
	 break;

      diff = x - floor(x); 

      if(diff <= 0.5)
      {
	 ix = (int)floor(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;


	 rad = sqrt((floor(x)-xcen)*(floor(x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }
      else
      {
	 ix = (int)ceil(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((ceil (x)-xcen)*(ceil (x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }

      y += 1.;
      x += deltax;

      ++npnts;
   }



   /* Octant 1 */

   x = xcen; 
   y = ycen + radius;

   printf("xcen = %g ycen = %g radius = %g ixref = %d jyref = %d\n", 
       xcen, ycen, radius, ixref, jyref);

   dx = 0.;
   dy = radius;

   deltay = 0.;

   for (;;)
   {
      printf(
	"x = %g  y = %g dx = %g  dy = %g  ix = %d  jy = %d\n",
	x, y, dx, dy, ix, jy);
      printf("radical = %g\n", dy*dy - 2*dx - 1.);
      fflush(stdout);
      deltay = sqrt(dy*dy - 2*dx - 1.) - dy;
      printf("deltay = %g\n", deltay);
      fflush(stdout);

      dx += 1.;
      dy += deltay;

      diff = y - floor(y); 

      if(diff <= 0.5)
      {
	 ix = (int)Round(x);
	 jy = (int)floor(y);
	 if(ix >= ixref && jy <= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (floor(y)-ycen)*(floor(y)-ycen));
      }
      else
      {
	 ix = (int)Round(x);
	 jy = (int)ceil(y);
	 if(ix >= ixref && jy <= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (ceil (y)-ycen)*(ceil (y)-ycen));
      }

      x += 1.;
      y += deltay;

      ++npnts;
   }


   /* QUADRANT 2 */


   /* Octant 3 */

   x = xcen + radius; 
   y = ycen;

   dx = radius;
   dy = 0.;

   deltax = 0.;

   for (;;)
   {
      deltax = sqrt(dx*dx + 2*dy - 1.) - dx;

      dy -= 1.;
      dx += deltax;

      if(fabs(dy) > fabs(dx))
	 break;

      diff = x - floor(x); 

      if(diff <= 0.5)
      {
	 ix = (int)floor(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((floor(x)-xcen)*(floor(x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }
      else
      {
	 ix = (int)ceil(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((ceil (x)-xcen)*(ceil (x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }

      y -= 1.;
      x += deltax;

      ++npnts;
   }


   /* Octant 4 */

   x = xcen; 
   y = ycen - radius;

   dx = 0.;
   dy = -radius;

   deltay = 0.;

   for (;;)
   {
      deltay =  -sqrt(dy*dy - 2*dx - 1.) - dy;

      dx += 1.;
      dy += deltay;

      diff = y - floor(y); 

      if(diff <= 0.5)
      {
	 ix = (int)Round(x);
	 jy = (int)floor(y);
	 if(ix >= ixref && jy >= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (floor(y)-ycen)*(floor(y)-ycen));
      }
      else
      {
	 ix = (int)Round(x);
	 jy = (int)ceil(y);
	 if(ix >= ixref && jy >= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (ceil (y)-ycen)*(ceil (y)-ycen));
      }

      x += 1.;
      y += deltay;

      ++npnts;
   }


   /* QUADRANT 4 */

   /* Octant 7 */

   x = xcen - radius; 
   y = ycen;

   dx = -radius;
   dy = 0.;

   deltax = 0.;

   for (;;)
   {
      deltax = -sqrt(dx*dx - 2*dy - 1.) - dx;

      dy += 1.;
      dx += deltax;

      if(fabs(dy) > fabs(dx))
	 break;

      diff = x - floor(x); 

      if(diff <= 0.5)
      {
	 ix = (int)floor(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((floor(x)-xcen)*(floor(x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }
      else
      {
	 ix = (int)ceil(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((ceil (x)-xcen)*(ceil (x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }

      y += 1.;
      x += deltax;

      ++npnts;
   }



   /* Octant 8 */

   x = xcen; 
   y = ycen + radius;

   dx = 0.;
   dy = radius;

   deltay = 0.;

   for (;;)
   {
      deltay = sqrt(dy*dy + 2*dx - 1.) - dy;

      dx -= 1.;
      dy += deltay;

      diff = y - floor(y); 

      if(diff <= 0.5)
      {
	 ix = (int)Round(x);
	 jy = (int)floor(y);
	 if(ix <= ixref && jy <= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (floor(y)-ycen)*(floor(y)-ycen));
      }
      else
      {
	 ix = (int)Round(x);
	 jy = (int)ceil(y);
	 if(ix <= ixref && jy <= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (ceil (y)-ycen)*(ceil (y)-ycen));
      }

      x -= 1.;
      y += deltay;

      ++npnts;
   }



   /* QUADRANT 3 */


   /* Octant 6 */

   x = xcen - radius; 
   y = ycen;

   dx = -radius;
   dy = 0.;

   deltax = 0.;

   for (;;)
   {
      deltax = -sqrt(dx*dx + 2*dy - 1.) - dx;

      dy -= 1.;
      dx += deltax;

      if(fabs(dy) > fabs(dx))
	 break;

      diff = x - floor(x); 

      if(diff <= 0.5)
      {
	 ix = (int)floor(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((floor(x)-xcen)*(floor(x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }
      else
      {
	 ix = (int)ceil(x);
	 jy = (int)Round(y);
	 ixref = ix;
	 jyref = jy;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((ceil (x)-xcen)*(ceil (x)-xcen) 
		  + (Round(y)-ycen)*(Round(y)-ycen));
      }

      y -= 1.;
      x += deltax;

      ++npnts;
   }


   /* Octant 5 */

   x = xcen; 
   y = ycen - radius;

   dx = 0.;
   dy = -radius;

   deltay = 0.;

   for (;;)
   {
      deltay =  -sqrt(dy*dy + 2*dx - 1.) - dy;

      dx -= 1.;
      dy += deltay;

      diff = y - floor(y); 

      if(diff <= 0.5)
      {
	 ix = (int)Round(x);
	 jy = (int)floor(y);

	 if(ix <= ixref && jy >= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (floor(y)-ycen)*(floor(y)-ycen));
      }
      else
      {
	 ix = (int)Round(x);
	 jy = (int)ceil(y);

	 if(ix <= ixref && jy >= jyref)
	    break;

	 if(ix >= 0 && ix<nx && jy >= 0 && jy<ny)
	    if (depth > 8)
		img_ptr.dmpimg32[jy*nx + ix] = color;
	    else
		img_ptr.dmpimg[jy*nx + ix] = (unsigned char ) color;

	 rad = sqrt((Round(x)-xcen)*(Round(x)-xcen) 
		  + (ceil (y)-ycen)*(ceil (y)-ycen));
      }

      x -= 1.;
      y += deltay;

      ++npnts;
   }

   return(npnts);
}
#endif /* NOTDEF */



static void
dmpellipse(color, img_ptr, nx, ny, xcen, ycen, half_width, half_height, angle)

int color;
img_struct img_ptr;
unsigned int nx, ny;
double          xcen, ycen;
double          half_width, half_height, angle;
{
    int i;

#ifdef NOTDEF
    struct
    {
        short x;
        short y;
    } points[65];
#endif
    XPoint points[65];

    get_ellipse_points(xcen, ycen, half_width, half_height, angle, points);

    for (i=1; i<65; i++)
        dmpline(color, img_ptr, nx, ny, 
	    (double)(points[i-1].x), (double)(points[i-1].y), 
	    (double)(points[ i ].x), (double)(points[ i ].y));
}



GC
get_rgb(color)
char *color;
{
   int   i;
   char  str[81];
   char *red, *green, *blue, *name;
   char  path[256];
   char *envptr;
   GC retval;

   FILE *fp;


   if(noverlays == 0)
   {
      overlays[0].red   = 0;
      overlays[0].green = 0;
      overlays[0].blue  = 0;
      strcpy(overlays[0].name, "transparent");

      nographics_GC[0] = (GC) malloc(1);  
      ++noverlays;
   }


   /* First check to see if we already have this color */

   for(i=1; i<noverlays; ++i)
   {
      if(strcmp(color, overlays[i].name) == 0)
      {
	 return(nographics_GC[i]);
      }
   }

   if(noverlays >= MAX_OVERLAYS)
      return(NULL);


   /* Read through the rgb.txt file, looking for this color name */

   envptr = getenv("RGBFILE");
   if(envptr == (char *) NULL)
      strcpy(path, "/usr/lib/X11/");
   else
      strcpy(path, envptr);

   if(path[strlen(path) -1] != '/')
      strcat(path, "/");
   
   strcat(path, "rgb.txt");

   fp = fopen(path, "r");
   if (fp == NULL)
   {
       fprintf(stderr, "Fatal error:  can't find file rgb.txt\n");    
       exit(1);
   }

   while(fgets(str, 80, fp) != NULL)
   {
    str[strlen(str)-1] = '\0';     /* change newline to null */
    red = strtok(str, " \t");
    green = strtok(NULL, " \t");
    blue = strtok(NULL, " \t");
    name = strtok(NULL, "\t\n");
    while (*name == ' ')
	name++;

      if(strcmp(name, color) == 0)
      {
	 overlays[noverlays].red   = atoi(red);
	 overlays[noverlays].green = atoi(green);
	 overlays[noverlays].blue  = atoi(blue);

	 strcpy(overlays[noverlays].name, name);

	 nographics_GC[noverlays] = (GC) malloc(1);  /* need a unique address */
	 retval = nographics_GC[noverlays];

	 ++noverlays;

	 fclose(fp);
	 return(retval);
      }
   }


   /* Couldn't find the color */

   fclose(fp);
   return(NULL);
}
