/*
Copyright (C) 1992, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include <stdio.h>
#include <math.h>

#include "skyview.h"
#include "parse.h"

#include "jgio.h"
#include "jconfig.h"

#define G_NULL 0
#define G_LINE 1
#define G_CIRCLE 2
#define G_MARK 3
#define G_STRING 4
#define G_DLINE 5
#define G_DCIRCLE 6
#define G_LASTNULL 7


typedef struct
    {
    int type;
    double x;
    double y;
    int mark_type;
    int mark_size;
    int mark_size2;
    int mark_size3;
    int table_line;
    char *table_filename;
    } G_mark;


typedef struct
    {
    int type;
    } G_any;

typedef union _G_graphic
    {
    int type;
    G_mark g_mark;
    G_any g_any;
    } G_graphic;

extern int debug, server;
extern double bscale, b_zero;
extern char *dimage_to_sky();
extern int ov_color;
extern int table_linenumber;
extern char *table_nam;
extern FILE *session;
extern JFILE *J12file;
extern int x_area[100][MAX_AREAS], y_area[100][MAX_AREAS];
extern int area_number;
extern int control_c;

#define SLOTS 100

static G_graphic *g_graphic;
static g_repainting = 0;
static int g_lastnull;

G_graphic *add_graphic()
{
    static G_graphic g_dummy;
    static g_graphic_slots;
    G_graphic *g_g, *g_next;

    if (g_repainting)
	return (&g_dummy);    /* dont record graphic if this is a playback */

    if (g_graphic == NULL)    /* if this is the very first call */
    {
	/* get initial allotment */
	g_graphic_slots = SLOTS;
	g_graphic = (G_graphic *) malloc(g_graphic_slots * sizeof(G_graphic));
	if (g_graphic == NULL)
	    return (&g_dummy);   /* discard graphics if malloc failed */
	g_graphic->type = G_LASTNULL;    /* mark the end */
	g_lastnull = 0;
    }

    /* see if more room is needed */
    if (g_lastnull + 1 >= g_graphic_slots)
    {
	g_graphic_slots = 2 * g_graphic_slots;   /* double current space */
	g_graphic = (G_graphic *) realloc(g_graphic, 
	    g_graphic_slots * sizeof(G_graphic));
	if (g_graphic == NULL)
	    return (&g_dummy);   /* discard graphics if malloc failed */
    }
    g_g = g_graphic + g_lastnull;
    g_next = g_g + 1;
    g_next->type = G_LASTNULL;
    g_lastnull++;
    return(g_g);
}


set_overlay_color()
{
    return(FALSE);
}

save_overlay_color()
{
}

restore_overlay_color()
{
}

good_overlay_color(color)
char *color;
{
    switch(cmd(color))
    {
	case RED:
	case GREEN:
	case BLUE:
	case WHITE:
	    return(TRUE);
	default:
	    return(FALSE);
    }
}

next_bundle()
{
}

debug_graphics()      /* print a report of graphics */
{
    G_graphic *g_g;

    if (g_graphic == NULL)
	return;

    g_g = g_graphic;
    do
    {
	switch(g_g->type)
	{
	    case G_MARK:
		printf("   MARK:   win = %x  x = %d  y = %d\n",
		    g_g->g_mark.x,
		    g_g->g_mark.y);
		break;
	    case G_NULL:
		printf("   NULL\n");
		break;
	    case G_LASTNULL:
		printf("   LASTNULL\n");
		break;
	}
	g_g++;
    } while (g_g->type != G_LASTNULL);
}



do_mark(xdbl, ydbl, markertype, markersize, markersize2, markersize3)
double xdbl, ydbl;
int markertype, markersize, markersize2, markersize3;
{
    int width, height, angle;
    int x, y;
    G_graphic *g_g;


    g_g = add_graphic();
    g_g->type = G_MARK;
    g_g->g_mark.x = x;
    g_g->g_mark.y = y;
    g_g->g_mark.mark_type = markertype;
    g_g->g_mark.mark_size = markersize;
    g_g->g_mark.mark_size2 = markersize2;
    g_g->g_mark.mark_size3 = markersize3;
    g_g->g_mark.table_line = table_linenumber;
    g_g->g_mark.table_filename = table_nam;

    x = round(xdbl);
    y = round(ydbl);
	switch(markertype)
	{
	    case DOT:
		markersize--;
		if (markersize < 1)
		{
		    j_wpx(x, y, ov_color);
		    j_wpx(x, y, 0);     /* Need to send 32 bits when */
					/* boards are mapped for full color */
		}
		else
		{
		    j_mov(x - markersize, y - markersize);
		    j_dfr(x + markersize, y + markersize);
		}
		break;
	    case CROSS:
		j_mov(x - markersize, y);
		j_dva(x + markersize, y);

		j_mov(x, y - markersize);
		j_dva(x, y + markersize);

		j_dca(x, y, 0);	/* turn off cursor */
		break;
	    case ASTERISK:
		j_mov(x - markersize, y - markersize);
		j_dva(x + markersize, y + markersize);
		j_mov(x - markersize, y + markersize);
		j_dva(x + markersize, y - markersize);
		j_mov(x - markersize, y );
		j_dva(x + markersize, y );
		break;
	    case CIRCLE:
		j_mov(x, y);
		j_dfc(markersize);
		break;
	    case X:
		j_mov(x - markersize, y - markersize);
		j_dva(x + markersize, y + markersize);
		j_mov(x - markersize, y + markersize);
		j_dva(x + markersize, y - markersize);
		break;
	    case BOX:
		j_mov(x - markersize, y - markersize);
		j_dva(x - markersize, y + markersize);
		j_dva(x + markersize, y + markersize);
		j_dva(x + markersize, y - markersize);
		j_dva(x - markersize, y - markersize);
		break;
	    case DIAMOND:
		j_mov(x - markersize, y );
		j_dva(x , y - markersize);
		j_dva(x + markersize, y );
		j_dva(x , y + markersize);
		j_dva(x - markersize, y );
		break;
	    case TRIANGLE:
		j_mov(x , y - markersize);
		j_dva(x - markersize, y + markersize);
		j_dva(x + markersize, y + markersize);
		j_dva(x , y - markersize);
		break;
	    case ELLIPSE:
		width = markersize;
		height = markersize2;
		angle = markersize3;
		draw_ellipse(x, y, width, height, angle);
		break;
	}
}


#define DIST_INIT 1000000

find_mark(x,y)
int x, y;
{
    G_graphic *g_g, *g_save;
    int i;
    double deltax, deltay, dist, dist_save;
    FILE *fd;
    char in_line[161], *coord_str;
    double d_coord[2];

    if (g_graphic == NULL)
    {
	printf("no marks found\n");
	return;
    }
    
    if (debug)
	printf("RBH find_mark entry:  x = %d  y = %d\n", x,y);
    dist_save = DIST_INIT;
    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if (g_g->type == G_MARK)
	{
	    deltax = g_g->g_mark.x - x;
	    deltay = g_g->g_mark.y - y;
	    dist = deltax * deltax + deltay * deltay;
	    if (debug)
	    {
		printf("comparing mark x = %g  y = %g  dist = %g\n", 
		    g_g->g_mark.x, g_g->g_mark.y, dist);
		printf("   from file %s  line %d\n",
		    g_g->g_mark.table_filename, g_g->g_mark.table_line);
	    }
	    if (dist < dist_save)
	    {
		/* this mark is closer - save it */
		dist_save = dist;
		g_save = g_g;
	    }
	}
    }
    if (dist_save == DIST_INIT)
    {
	printf("no marks found\n");
    }
    else
    {
	d_coord[0] = g_save->g_mark.x;
	d_coord[1] = g_save->g_mark.y;
	dSC_to_IM(d_coord);
	coord_str = dimage_to_sky(d_coord, 1);

	if (g_save->g_mark.table_line == 0)
	{
	    printf("mark did not come from a table file\n");
	    printf("mark at %s\n\n", coord_str);
	}
	else
	{
	    printf("mark came from line %d of file %s, printed below:\n",
		g_save->g_mark.table_line, g_save->g_mark.table_filename);

	    fd = fopen(g_save->g_mark.table_filename, "r");
	    if (fd == NULL)
	    {
		perror(g_save->g_mark.table_filename);
		return;
	    }
	    i = g_save->g_mark.table_line;
	    while (i--)
	    {
		if (fgets(in_line, 160, fd) == NULL)
		{
		    printf("error reading file %s\n", 
			g_save->g_mark.table_filename);
		    return;
		}
	    }
	    printf("%s", in_line);
	}
    }
}

gexamine(argc, argv)
int argc;
char **argv;
{
    G_graphic *g_g;
    FILE *fd;
    int i;
    double d_coord[2];
    int n, x, y; 
    int INTERIOR=1; 
    char in_line[161], *coord_str;

    set_current_window();

    if (g_graphic == NULL)
	return;
    
    enable_control_c();


    if (argc > 1)  /* if he specified an area number */
    {
	if (!switch_area(argv[1]))
	    return;
    }

    if(debug)
	printf("graphics_examine:  area number %d \n", area_number);

    if (x_area[0][area_number] == 0)
    {
	printf("graphics_examine error: area %d has not been defined\n", area_number);
	error();
	return;
    }

    segment_setup();  /* calculate parameters defining each line */ 

    /* Loop through the marks */

    for (g_g = g_graphic; g_g->type != G_LASTNULL; g_g++)
    {
	if (g_g->type == G_MARK)
	{
	    d_coord[0] = g_g->g_mark.x;
	    d_coord[1] = g_g->g_mark.y;
	    dSC_to_IM(d_coord);
	    x = round(d_coord[0]);
	    y = round(d_coord[1]);
	    x_list(y);     /* list of boundary crossings */
	    if(position(x)==INTERIOR && !on_horizontal(x, y)) 
	    {
		if (g_g->g_mark.table_line == 0) /* if not from a table file */
		{
		    if (server)
		    {
			printf(":0\n");
		    }
		    else
		    {
			coord_str = dimage_to_sky(d_coord, 1);
			printf("mark did not come from a table file\n");
			printf("mark at %s\n\n", coord_str);
		    }
		}
		else   /* from a table file */
		{
		    if (server)
		    {
			printf("%s:%d\n",
			    g_g->g_mark.table_filename, g_g->g_mark.table_line);
		    }
		    else
		    {
			printf("mark came from line %d of file %s, printed below:\n",
			    g_g->g_mark.table_line, g_g->g_mark.table_filename);

			fd = fopen(g_g->g_mark.table_filename, "r");
			if (fd == NULL)
			{
			    perror(g_g->g_mark.table_filename);
			    return;
			}
			i = g_g->g_mark.table_line;
			while (i--)
			{
			    if (fgets(in_line, 160, fd) == NULL)
			    {
				printf("error reading file %s\n", 
				    g_g->g_mark.table_filename);
				return;
			    }
			}
			printf("%s", in_line);
		    }
		}
	    }
	}

	if (control_c)
	{
	    printf("\ngraphics_examine interrupted by control c\n");
	    fprintf(session, "\ngraphics_examine interrupted by control c\n");
	    return;
	}
    }
}
