/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "parse.h"
#include "range.h"
#include "im_proto.h"
#include <stdio.h>
#include <sys/file.h>
#include <math.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>

#ifdef NO_FLOAT_H
#include <values.h>
#define DBL_MAX MAXDOUBLE
#define FLT_MAX MAXFLOAT
#else
#include <float.h>
#endif

#define PARSE_MIN 177
#define PARSE_MAX 178
#define BLKSIZ 512

int ho_on = FALSE, screenflag = FALSE;
struct full_ra min_ho = { RA_SIMPLE, {RA_MIN, 0} , {0,0} } ;
struct full_ra max_ho = { RA_SIMPLE, {RA_MAX, 0} , {0,0} } ;


extern int debug, debug_val, server_mode, graphics, u16, pixels, lines, bitpix;
extern int band_offset;
extern int be_quiet;
extern char server_str[];
extern struct hdrstruct filehdr;
extern struct img *curr_img;
extern int hist[];
extern char filnam[], fi_name[], hist_filnam[];
extern int hist_band;
extern double blank;
extern double hist_min, hist_binsiz, iraf_min, iraf_max;
extern double bscale, b_zero;
extern FILE *session, *debug_file;

static double hist_datamax, hist_datamin;  /* max, min found while doing hist */

/* function prototypes */
#ifdef _NO_PROTO

static void get_hist_IM();

#else

static void get_hist_IM(int fd);

#endif /* _NO_PROTO */

/* get_hist is called only for integer pixels */
void
get_hist(bitpix, buf, ri, min, max, blank, hist_shift, hist1)
int bitpix, *min, *max, hist_shift, hist1[];
double blank;
register int ri;
void *buf;
{
    register int rmin, rmax;
    register short *rp;
    register unsigned short *rpu;
    register int *rpl;
    char *rpc;
    int val;
    double local_blank;

    if (FINITE(blank))
    {
	local_blank = blank;
    }
    else
    {
	local_blank = 1e+30;  /* impossible value - (NaN causes traps) */
    }

    rmax = *max;
    rmin = *min;
    switch(bitpix)
    {
	case 32:
	{
	    rpl = (int *) buf;   /* 32 bit pointer */
	    while (ri > 0)
	    {
		if (*rpl != local_blank)
		{
		    if (*rpl > rmax)
			rmax = *rpl;
		    if (*rpl < rmin)
			rmin = *rpl;

		    if (*rpl >= 0)
		    {
			if ((*rpl >> hist_shift) <= HISTSIZ)
			    hist1[(*rpl >> hist_shift) + HISTSIZ] += 1;
						    /* increment histogram */
			else
			    hist1[HISTSIZ2] +=1;
				/* clip at histogram array bounds */
		    }
		    else
		    {
			if ((*rpl >> hist_shift) >= -HISTSIZ)
			    hist1[(*rpl >> hist_shift) + HISTSIZ] += 1;
						    /* increment histogram */
			else
			    hist1[0] +=1;
				/* clip at histogram array bounds */
		    }
		}
		rpl++;    /* advance 32 bits */
		ri--;
	    }
	    break;
	}
	case 16:
	if (u16)
	{
	    /* fix max, min if negative */
	    if (rmax < 0)
		rmax = 0;
	    if (rmin < 0)
		rmin = 0;
	    rpu = (unsigned short *) buf;
	    while (ri > 0)
	    {
		if (*rpu != local_blank)
		{
		    val = (int) *rpu;
		    if (val > rmax)
			rmax = val;
		    if (val < rmin)
			rmin = val;
		    if ((val >> hist_shift) <= HISTSIZ)
			hist1[(val >> hist_shift) + HISTSIZ] += 1;
						/* increment histogram */
		    else
			hist1[HISTSIZ2] +=1;
			    /* clip at histogram array bounds */
		}
		rpu++;
		ri--;
	    }
	    break;
	}
	else
	{
	    rp = (short *) buf;
	    while (ri > 0)
	    {
		if (*rp != local_blank)
		    {
		    if (*rp > rmax)
			rmax = *rp;
		    if (*rp < rmin)
			rmin = *rp;
		    if (*rp >= 0)
		    {
			if ((*rp >> hist_shift) <= HISTSIZ)
			    hist1[(*rp >> hist_shift) + HISTSIZ] += 1;
						    /* increment histogram */
			else
			    hist1[HISTSIZ2] +=1;
				/* clip at histogram array bounds */
		    }
		    else
		    {
			if ((*rp >> hist_shift) >= -HISTSIZ)
			    hist1[(*rp >> hist_shift) + HISTSIZ] += 1;
						    /* increment histogram */
			else
			    hist1[0] +=1;
				/* clip at histogram array bounds */
		    }
		    }
		rp++;
		ri--;
	    }
	    break;
	}
	case 8:
	{
	    rpc = (char *) buf;
	    while (ri > 0)
	    {
		val = *rpc & 0xff;
		if (val != local_blank)
		    {
		    if (val > rmax)
			rmax = val;
		    if (val < rmin)
			rmin = val;
		    hist1[(val >> hist_shift) + HISTSIZ] += 1;
			/* increment histogram - no need to check bounds */
		    }
		rpc++;
		ri--;
	    }
	    break;
	}
    }
    *max = rmax;
    *min = rmin;
}


static void
get_hist_IM(fd)
int fd;
{
    int n, redo_flag, doing_redo;
    int i, dn_min, dn_max;
    float val;
    int hist_shift;

    float one_percent;        /* for debug messages only */
    int n_values, n_blanks;   /* for debug messages only */
    short *s_filptr, *i16_runp;
    int *i_filptr, *i32_runp;
    char *c_filptr;

    float *file_disp, *filptr;
    register int x ;
    float *runp;
    double d_runval, *d_runp, *d_file_disp;
    double hist_max;
    double local_blank;
    int hist_min_index, hist_max_index;
    int y;
    int goodpix, low_limit, low_sum, high_sum;

    filptr = (float *) imget(fd, filnam);   /* read entire image */
    if (filptr == NULL)
    {
	error1("problem reading pixels");
	return;
    }


    hist_datamax = -DBL_MAX;
    hist_datamin = DBL_MAX;

    if (bitpix == 8)
    {
	if ((server_mode == FALSE) && (!be_quiet))
	{
	    printf("building histogram . . ");
	    fflush(stdout);
	}
	for (i=0; i<=HISTSIZ2; i++) hist[i]=0;
	hist_shift = 0;
	dn_min = 1000;
	dn_max = 0;
	c_filptr = (char *) filptr + band_offset * pixels * lines;
	get_hist(bitpix, (void *) c_filptr, pixels*lines, &dn_min, &dn_max, 
	    blank, hist_shift, hist);
	hist_datamin = dn_min;
	hist_datamax = dn_max;
	iraf_min = hist_datamin;
	iraf_max = hist_datamax;
	hist_binsiz = 1;
	hist_min = -HISTSIZ * hist_binsiz;
	return;
    }

    if ((bitpix == 32) || (bitpix == 16))
    {
	if (FINITE(blank))
	    local_blank = blank;
	else
	    local_blank = 1e+30;  /* impossible value - (NaN causes traps) */

	if ((iraf_max == 0) && (iraf_min == 0))
	{
	    if ((server_mode == FALSE) && (!be_quiet))
	    {
		printf("scanning for min, max . . ");
		fflush(stdout);
	    }
	    dn_max = 0x80000001;
	    dn_min = 0x7fffffff;
	    if (bitpix == 32)
	    {
		i_filptr = (int *) filptr + band_offset * pixels * lines;
		for (y = 0; y < lines; y++)
		{
		    i32_runp = i_filptr;
		    i_filptr += pixels;    /* prepare for next line */
		    x = pixels;
		    do
		    {
			if (*i32_runp != local_blank)
			{
			    if (*i32_runp < dn_min)
				dn_min = *i32_runp;
			    if (*i32_runp > dn_max)
				dn_max = *i32_runp;
			}
			i32_runp++;
		    } while (--x);
		}
	    }
	    else
	    {
		/* bitpix 16 */
		s_filptr = (short *) filptr + band_offset * pixels * lines;
		for (y = 0; y < lines; y++)
		{
		    i16_runp = s_filptr;
		    s_filptr += pixels;    /* prepare for next line */
		    x = pixels;
		    do
		    {
			if (*i16_runp != local_blank)
			{
			    if (*i16_runp < dn_min)
				dn_min = *i16_runp;
			    if (*i16_runp > dn_max)
				dn_max = *i16_runp;
			}
			i16_runp++;
		    } while (--x);
		}
	    }

	    iraf_min = dn_min;
	    iraf_max = dn_max;

	    if ((server_mode == FALSE) && (!be_quiet))
	    {
		printf("done\n");
		fflush(stdout);
	    }

	    if (iraf_min == iraf_max)
		return;

	}

	if ((server_mode == FALSE) && (!be_quiet))
	{
	    printf("building histogram . . ");
	    fflush(stdout);
	}
	hist_min = iraf_min;
	hist_max = iraf_max;
	doing_redo = FALSE;
	for (;;)
	{
	    for (i=0; i<=HISTSIZ2; i++) hist[i]=0;
	    redo_flag = FALSE;
	    hist_binsiz = (hist_max - hist_min) / HISTSIZ2;
	    if (hist_binsiz == 0.0)
		hist_binsiz = 1.0;

	    if (bitpix == 32)
	    {
		i_filptr = (int *) filptr + band_offset * pixels * lines;
		for (y = 0; y < lines; y++)
		{
		    i32_runp = i_filptr;
		    i_filptr += pixels;    /* prepare for next line */
		    x = pixels;
		    do
		    {
			if (*i32_runp != local_blank)
			{
			    i = (*i32_runp - hist_min) / hist_binsiz;
			    if ((i<0) || (i>HISTSIZ2))
				redo_flag = TRUE;   /* hist_min, hist_max were bad */
			    else
			    {
				hist[i] ++;
			    }
			    if (*i32_runp < hist_datamin)
				hist_datamin = *i32_runp;
			    if (*i32_runp > hist_datamax)
				hist_datamax = *i32_runp;
			}
			i32_runp++;
		    } while (--x);
		}
	    }
	    else
	    {
		/* bitpix 16 */
		s_filptr = (short *) filptr + band_offset * pixels * lines;
		for (y = 0; y < lines; y++)
		{
		    i16_runp = s_filptr;
		    s_filptr += pixels;    /* prepare for next line */
		    x = pixels;
		    do
		    {
			if (*i16_runp != local_blank)
			{
			    i = (*i16_runp - hist_min) / hist_binsiz;
			    if ((i<0) || (i>HISTSIZ2))
				redo_flag = TRUE;   /* hist_min, hist_max were bad */
			    else
			    {
				hist[i] ++;
			    }
			    if (*i16_runp < hist_datamin)
				hist_datamin = *i16_runp;
			    if (*i16_runp > hist_datamax)
				hist_datamax = *i16_runp;
			}
			i16_runp++;
		    } while (--x);
		}
	    }

	    iraf_min = hist_datamin;
	    iraf_max = hist_datamax;

    /* check if we got a good spread */

    if (!redo_flag)  /* don't bother checking if we already want a redo */
    {
	    /* see what happens if we lop off top and bottom 0.05% of hist */
	    goodpix = 0;
	    for (i=0; i<HISTSIZ2; i++)
		goodpix += hist[i];
	    low_limit = goodpix * 0.0005;

	    high_sum = 0;
	    for (i = HISTSIZ2; i >= 0; i--)
	    {
		high_sum += hist[i];
		if (high_sum > low_limit)
		    break;
	    }
	    hist_max_index = i;

	    low_sum = 0;
	    for (i = 0; i<HISTSIZ2; i++)
	    {
		low_sum += hist[i];
		if (low_sum > low_limit)
		    break;
	    }
	    hist_min_index = i;


	    if ((hist_max_index - hist_min_index) < HISTSIZ)
	    {
		hist_max = (hist_max_index * hist_binsiz) + hist_min;
		hist_min = (hist_min_index * hist_binsiz) + hist_min;
		redo_flag = TRUE;   /* we can spread it out by factor of 2 */
	    }
    }


	    if ((server_mode == FALSE) && (!be_quiet))
	    {
		printf("done\n");
		fflush(stdout);
	    }

	    if ((!doing_redo) && (redo_flag))
	    {
		if ((server_mode == FALSE) && (!be_quiet))
		{
		    printf("rebuilding histogram . . ");
		    fflush(stdout);
		}
		doing_redo = TRUE;
	    }
	    else
		break;
	}
	if (debug_val == 8)
	{
	    for (i=0; i<=HISTSIZ2; i++)
		if (hist[i] != 0)
		    fprintf(debug_file,
			"hist[%d] = %d\n", i, hist[i]);
	}
	return;
    }

    /* bitpix is -32 or -64 */
    if (iraf_max == iraf_min)
    {
	if (debug)
	    fprintf(debug_file, 
		"entry: iraf_min = %g  iraf_max = %g\n", iraf_min, iraf_max);
	if(server_mode == FALSE)
	{
	    printf("scanning for max min . . ");
	    fflush(stdout);
	}
	if (bitpix == -32)
	{
	    file_disp = (float *) filptr + band_offset * pixels * lines;
	    for (y = 0; y < lines; y++)
	    {
		runp = file_disp;
		file_disp += pixels;    /* prepare for next line */
		x = pixels;
		do
		{
		    d_runval = *runp++;
		    if (FINITE(d_runval))
		    {
			if (d_runval < hist_datamin) 
			    hist_datamin = d_runval;
			if (d_runval > hist_datamax) 
			    hist_datamax = d_runval;
		    }
		} while (--x);
	    }
	}
	else
	{
	    /* bitpix -64 */
	    d_file_disp = (double *) filptr + band_offset * pixels * lines;
	    for (y = 0; y < lines; y++)
	    {
		d_runp = d_file_disp;
		d_file_disp += pixels;    /* prepare for next line */
		x = pixels;
		do
		{
		    d_runval = *d_runp++;
		    if (FINITE(d_runval))
		    {
			if (d_runval < hist_datamin) 
			    hist_datamin = d_runval;
			if (d_runval > hist_datamax) 
			    hist_datamax = d_runval;
		    }
		} while (--x);
	    }
	}

	iraf_min = hist_datamin;
	iraf_max = hist_datamax;
	if ((server_mode == FALSE) && (!be_quiet))
	   printf("done\n");
    }

    if (iraf_min == iraf_max)
	return;

    if ((server_mode == FALSE) && (!be_quiet))
    {
	printf("building histogram . . ");
	fflush(stdout);
    }
    hist_min = iraf_min;
    hist_max = iraf_max;
    doing_redo = FALSE;
redo:


    for (i=0; i<=HISTSIZ2; i++) hist[i]=0;
    redo_flag = FALSE;
    n_values = 0;
    n_blanks = 0;
    hist_binsiz = (hist_max - hist_min) / HISTSIZ2;
    if (hist_binsiz == 0.0)
	hist_binsiz = 1.0;
    if (debug)
    {
	fprintf(debug_file, "\nhist_min = %g  hist_max = %g  hist_binsiz = %g\n", 
	    hist_min, hist_max, hist_binsiz);
    }

    if (bitpix == -32)
    {
	file_disp = (float *) filptr + band_offset * pixels * lines;
	for (y = 0; y < lines; y++)
	{
	    runp = file_disp;
	    file_disp += pixels;    /* prepare for next line */
	    x = pixels;
	    do
	    {
		d_runval = *runp++;

		if (FINITE(d_runval))
		{
		    i = (d_runval - hist_min) / hist_binsiz;
		    if ((i<0) || (i>HISTSIZ2))
			redo_flag = TRUE;   /* hist_min, hist_max were bad */
		    else
		    {
			hist[i] ++;
			n_values ++;
		    }
		    if (d_runval < hist_datamin) 
			hist_datamin = d_runval;
		    if (d_runval > hist_datamax) 
			hist_datamax = d_runval;
		}
		else 
		    n_blanks ++;
	    } while (--x);

	}
    }
    else
    {
	/* bitpix -64 */
	d_file_disp = (double *) filptr + band_offset * pixels * lines;
	for (y = 0; y < lines; y++)
	{
	    d_runp = d_file_disp;
	    d_file_disp += pixels;    /* prepare for next line */
	    x = pixels;
	    do
	    {
		d_runval = *d_runp++;

		if (FINITE(d_runval))
		{
		    i = (d_runval - hist_min) / hist_binsiz;
		    if ((i<0) || (i>HISTSIZ2))
			redo_flag = TRUE;   /* hist_min, hist_max were bad */
		    else
		    {
			hist[i] ++;
			n_values ++;
		    }
		    if (d_runval < hist_datamin) 
			hist_datamin = d_runval;
		    if (d_runval > hist_datamax) 
			hist_datamax = d_runval;
		}
		else 
		    n_blanks ++;
	    } while (--x);

	}
    }

    iraf_min = hist_datamin;
    iraf_max = hist_datamax;

    if ((server_mode == FALSE) && (!be_quiet))
    {
	printf("done\n");
	fflush(stdout);
    }


    /* check if we got a good spread */

    if (!redo_flag)  /* don't bother checking if we already want a redo */
    {
	    /* see what happens if we lop off top and bottom 0.05% of hist */
	    goodpix = 0;
	    for (i=0; i<HISTSIZ2; i++)
		goodpix += hist[i];
	    low_limit = goodpix * 0.0005;

	    high_sum = 0;
	    for (i = HISTSIZ2; i >= 0; i--)
	    {
		high_sum += hist[i];
		if (high_sum > low_limit)
		    break;
	    }
	    hist_max_index = i;

	    low_sum = 0;
	    for (i = 0; i<HISTSIZ2; i++)
	    {
		low_sum += hist[i];
		if (low_sum > low_limit)
		    break;
	    }
	    hist_min_index = i;


	    if ((hist_max_index - hist_min_index) < HISTSIZ)
	    {
		hist_max = (hist_max_index * hist_binsiz) + hist_min;
		hist_min = (hist_min_index * hist_binsiz) + hist_min;
		redo_flag = TRUE;   /* we can spread it out by factor of 2 */
		if (debug)
		{
		    fprintf(debug_file, 
		    "\nrecomputed hist_min = %g  hist_max = %g  hist_binsiz = %g\n", 
			hist_min, hist_max, hist_binsiz);
		}
	    }
    }


    if ((!doing_redo) && (redo_flag))
    {
	/* do it over */
	if(server_mode == FALSE)
	{
	    printf("redoing histogram . . ");
	    fflush(stdout);
	}
	doing_redo = TRUE;
	goto redo;
    }

   if (debug_val == 8)
   {
       fprintf(debug_file, "\\C plmode = histogram\n");
       fprintf(debug_file, "\\R DATAMAX = %g\n",hist_datamax);
       fprintf(debug_file, "\\R DATAMIN = %g\n",hist_datamin);
       fprintf(debug_file, "\\I NVALUES = %d\n",n_values);
       fprintf(debug_file, "\\I NBLANKS = %d\n",n_blanks);

       fprintf(debug_file, "|  i  |  VALUE  |   %%   | cum %% |\n");

       one_percent = 100.0 / n_values;
       val = hist_min + (.5 * hist_binsiz);   /* center of bin 0 */
       n = 0;
       for (i=0; i<=HISTSIZ2; i++) 
       {
	    n += hist[i];
	    if ( hist[i] || i==0 || i==HISTSIZ2)
	    fprintf(debug_file, "%4d  %#10.4G %7.3f %7.3f\n",
		i, val, hist[i]*one_percent, n*one_percent);
	    val += hist_binsiz;
	}
    }
}


/**********************************************************************
**                                                                   ** 
**  CHECK_HIST checks if the histogram in an image workfile is       ** 
**  well distributed.  If it can be spread out by more than a factor ** 
**  of two, the histogram is recomputed.                             ** 
**  On entry, argument fd should be an open image workfile, and its  ** 
**  internal s header must be in structure filehdr. Globals must     ** 
**  have been filled from the image workfile.                        ** 
**                                                                   ** 
**********************************************************************/ 
 
void
check_hist(fd)
int fd;
{
    int max, hist_shift, binsiz;
    int rdstat, totpixels, i, ri;
    char buf[BLKSIZ];

    if (debug)
	fprintf(debug_file, "entering check_hist\n");
    if (fd < 0)
    {
	error1("no workfile");
	return;
    }

    if ((strcmp(hist_filnam, filnam) != 0) || (hist_band != band_offset))
    {
	if (filehdr.tapetype == IMFILE)
	{
	    /* do histogram on image access file */
	    get_hist_IM(fd);
	}
	else
	{
	    /* read histogram from s workfile */
	    lseek(fd, (long) filehdr.hist_start, 0);
	    read(fd,hist, (HISTSIZ2+1) * sizeof(int));
	}
	strcpy(hist_filnam, filnam);
	hist_band = band_offset;
    }
    if (filehdr.tapetype == IMFILE)
    {
	/* make sure max and min haven't been overwritten by img mgmt */
	iraf_min = hist_datamin;
	iraf_max = hist_datamax;
	return;
    }

    if (filehdr.img_max > (-filehdr.img_min))
	max = filehdr.img_max;
    else
	max = -filehdr.img_min;
    hist_shift = 0;
    while ((max >> hist_shift) > HISTSIZ)
	++hist_shift;
    binsiz = 1 << hist_shift;
    if (debug)
    {
	fprintf(debug_file, "check_hist: current hist_bin_size = %x hex\n", 
	    filehdr.hist_bin_size);
	fprintf(debug_file, "check_hist:  max = %d", max);
	fprintf(debug_file, "  possible new hist_shift = %d  new binsiz = %x\n", 
	    hist_shift, binsiz);
    }
    if (binsiz * 2 < filehdr.hist_bin_size)  /* require more than 2x improvement */
    {
	    if ((server_mode == FALSE) && (!be_quiet))
	       printf("rebuilding histogram\n");

	    for (i=0; i<HISTSIZ2+1; i++)
		hist[i] = 0;

	    totpixels = pixels * lines;
	    lseek(fd, (long) filehdr.img_start, 0);
	    do
	    {
		rdstat = read(fd, buf, BLKSIZ);
		if (rdstat == -1)
		{
		    sperror("check_hist error reading pixels ", errno);
		    return;
		}

		ri = rdstat * 8 / bitpix;  /* total pixels in buffer */
		if (ri > totpixels)
		    ri = totpixels;
		totpixels -= ri;
		get_hist(bitpix, (void *) buf, ri, &filehdr.img_min, 
		    &filehdr.img_max,blank,hist_shift,hist);
	    } while (totpixels > 0);

	filehdr.hist_bin_size = 1 << hist_shift;

	/* see if we have write access */
	if (access(filnam, W_OK) != 0)
	{
	    if (debug)
		fprintf(debug_file, 
		    "cannot write histogram - no write access to workfile\n");
	}
	else
	{
	    lseek(fd, (long) filehdr.hist_start, 0);
	    write(fd, hist, (HISTSIZ2+1) * sizeof(int));
	    lseek(fd, 0L, 0);
	    write(fd, &filehdr, sizeof(filehdr));
	}
    }
    /* make it look same as IMFILE results */
    hist_binsiz = filehdr.hist_bin_size;
    hist_min = -HISTSIZ * hist_binsiz;
    iraf_min = filehdr.img_min;
    iraf_max = filehdr.img_max;
}

double nice_step(dy)
double dy;
{
    double dexp;
    int dinc;

    if (dy == 0)
	dy = 1;

    if (dy < 0)
	dy = -dy;

    /* now find a nice step size for labels */
    dexp = 1;
    while (dy >= 10)
    {
	dy = dy / 10;
	dexp = dexp * 10;
    }
    while (dy < 1)
    {
	dy = dy * 10;
	dexp = dexp / 10;
    }
    if (dy > 5)
	dinc = 10;
    else if (dy > 2)
	dinc = 5;
    else
	dinc = 2;
    dy = dinc * dexp;
    return(dy);
}

void
ho_cmd(argc, argv)
int argc;
char **argv;
{
    struct img tmpimg, *save_curr_img;
    int file_d;

    if (fi_name[0] == '\0')
    {
	error1("no workfile has been specified");
	return;
    }

    file_d = wf_open(fi_name, &tmpimg, 1);
    if (file_d >= 0)
    {
	save_curr_img = curr_img;
	fill_glob(&tmpimg);
	check_hist(file_d);
#ifdef NOTDEF
	minx = 0;
	miny = 0;
#endif /* NOTDEF */

	if (argc == 3)
	{
	    if (!parse_minmax(argv[1], &min_ho, PARSE_MIN))
	    {
		error1("bad first argument to HO command");
		return;
	    }
	    if (!parse_minmax(argv[2], &max_ho, PARSE_MAX))
	    {
		error1("bad second argument to HO command");
		return;
	    }
	}
	else
	{
	    min_ho.tag = RA_SIMPLE;
	    min_ho.arg_one.tag = RA_MIN;
	    max_ho.tag = RA_SIMPLE;
	    max_ho.arg_one.tag = RA_MAX;
	}

	screenflag = FALSE;
	if (graphics)
	    create_hist_window();

	curr_img = save_curr_img;
	fill_glob(curr_img);

	if (tmpimg.filehdr.tapetype == IMFILE)
	    im_close(file_d);
	else
	    close(file_d);
    }
}


double dn_to_jy(dn)
double dn;
{
    return(dn * bscale + b_zero);
}

double jy_to_dn(jy)
double jy;
{
    double dn;

    dn = (jy - b_zero) / bscale;
    return(dn);
}

double get_pct(ra_value)
double ra_value;
{
    int sum, goal, i;
    int goodpix;

    goodpix = 0;
    for (i=0; i<HISTSIZ2; i++)
	goodpix += hist[i];
    sum = 0;
    goal = goodpix * (ra_value) / 100;
    i = -1;
    do
    {
	i++;
	sum = sum + hist[i];
    } while (sum < goal);
    if (debug)
	fprintf(debug_file, 
	    "get_pct: goodpix= %d  goal=%d i=%d hist_binsiz=%g\n",
	    goodpix, goal, i, hist_binsiz);
    return( (i + 0.5) * hist_binsiz + hist_min);
}
