/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "im_proto.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <math.h>


/*********************************************************************/
/*                                                                   */
/*  FLUX                                                             */
/*  Extracts the flux from the disk file given the pixel coordinates */
/*  (in IMAGE coordinates)                                           */
/*                                                                   */
/*********************************************************************/


extern int bitpix, u16, minx, maxx, miny, maxy, debug, image_frame;
extern int pixels, lines, band_offset;
extern FILE *debug_file;
extern double blank_flag, blank;
extern double zap_value;
extern int zap_blank;  /* TRUE means use BLANK value for zapping */
extern int blank_set;
extern struct hdrstruct filehdr; 
extern char filnam[], mod_filnam[];
extern struct img *curr_img;
extern int graphics_window;

#ifdef _NO_PROTO
static void do_flux();
#else
static void do_flux(int samp_offset, int line_offset, double fluxes[]);
#endif /* _NO_PROTO */



/* flux_IRAF expects i_coord to be in IRAF coordinate system */
void
flux_IRAF(i_coord, fluxes) 
int i_coord[2];
double fluxes[4];   /* fluxes[0] is returned flux value in DN (double) */
{ 
    int samp_offset, line_offset;      /* 0 based offsets */
    double d_coord[2];

    if (graphics_window)
    {
	fluxes[0] = blank_flag;
	return;
    }

    if (debug)
	fprintf(debug_file, 
	    "flux_IRAF:  entry values x = %d  y = %d\n", 
	    i_coord[0], i_coord[1]);

    /* Check if location on image */ 
    d_coord[0] = i_coord[0];
    d_coord[1] = i_coord[1];
    dIRAF_to_SC(d_coord);

    if (curr_img == NULL || d_coord[0]<minx || d_coord[0]>maxx || 
	d_coord[1]<miny || d_coord[1]>maxy || !FINITE(d_coord[0]) ||
	!FINITE(d_coord[1]) )
    { 
	/* no, it's not on the image */
	fluxes[0] = blank_flag; 
	return; 
    } 

    /* Find the offsets */ 

    samp_offset = i_coord[0] - 1;
    line_offset = i_coord[1] - 1;

    do_flux(samp_offset, line_offset, fluxes);
}

static void
do_flux(samp_offset, line_offset, fluxes) 
int samp_offset, line_offset;
double fluxes[4];   /* fluxes[0] is returned flux value in DN (double) */
{ 
    double flux_val;
    unsigned char valc, *p_char;
    unsigned short *p_ushort;
    short *p_short;
    short vals; 
    int disp, val, *p_int; 
    int bytes; 
    static int fd = -1;
    static char filnam_save[MAXPATHLEN];
    static int code;
    struct img tmpimg;
    float *fptr;
    double *dptr;
    void *v_ptr;

    /* Find the offset displacement in file or array */ 
    disp = samp_offset + 
	   line_offset * pixels + 
	   band_offset * pixels * lines;

    if (filehdr.tapetype == IMFILE)
    {
	if (mod_filnam[0] == '\0')
	    v_ptr = imget(-1, filnam);
	else
	    v_ptr = imget(-1, mod_filnam);
	
	if (v_ptr == NULL)
	{
	    /* bad stuff - cannot read pixels */
	    fluxes[0] = blank_flag;
	    return;
	}

	switch(bitpix)
	{
	case 8:
	    p_char = (unsigned char *) v_ptr;
	    p_char += disp;
	    flux_val = *p_char;
	    if(debug)
		fprintf(debug_file, 
		    "flux:  dn=%g\n", flux_val);
	    if (flux_val == blank)
		flux_val = blank_flag;
	    break;
	case 16:
	    if (u16)
	    {
		p_ushort = (unsigned short *) v_ptr;
		p_ushort += disp;
		flux_val = *p_ushort;
		if(debug)
		    fprintf(debug_file, 
			"flux:  dn=%g\n", flux_val);
		if (flux_val == blank)
		    flux_val = blank_flag;
	    }
	    else
	    {
		p_short = (short *) v_ptr;
		p_short += disp;
		flux_val = *p_short;
		if(debug)
		    fprintf(debug_file, 
			"flux:  dn=%g\n",  flux_val);
		if (flux_val == blank)
		    flux_val = blank_flag;
	    }
	    break;
	case 32:
	    p_int = (int *) v_ptr;
	    p_int += disp;
	    flux_val = *p_int;
	    if(debug)
		fprintf(debug_file, 
		    "flux:  dn=%g\n",  flux_val);
	    if (flux_val == blank)
		flux_val = blank_flag;
	    break;
	case -32:
	    fptr = (float *) v_ptr;
	    fptr += disp;
	    flux_val = *fptr;

	    if(debug)
		fprintf(debug_file, 
		    "float flux:  dn=%g\n",  flux_val);
	    break;
	case -64:
	    dptr = (double *) v_ptr;
	    dptr += disp;
	    flux_val = *dptr;

	    if(debug)
		fprintf(debug_file, 
		    "double flux:  dn=%g\n",  flux_val);
	    break;
	}
    }
    else
    {
	/* it is an s workfile */
	if (fd >= 0)  /* if a file is already open */
	{
	    if (strcmp(filnam_save, filnam) != 0) /* if it is different file */
	    {
		if (code == -32)
		    im_close(fd);
		else
		    close(fd);
		fd = -1;
	    }
	}

	if (fd < 0)
	{
	    fd = wf_open(filnam, &tmpimg, 1);
	    if (fd < 0)
	    {
		/* bad stuff - file has disappeared */
		fluxes[0] = blank_flag;
		return;
	    }
	    if (tmpimg.filehdr.tapetype == IMFILE)
		code = -32;
	    else
		code = bitpix;
	    strcpy(filnam_save, filnam);  /* remember which file fd represents */
	}



	/* Process based on number of bits */ 

	switch (code) 
	{ 
	    case 8: 
	    lseek(fd, disp + filehdr.img_start, 0); 
	    bytes = read(fd,&valc,1); 

	    if (bytes == 1) 
	    { 
		flux_val = (double) valc; 

		if(debug)
		    fprintf(debug_file, 
			"flux:  dn=%d\n", valc);
		if (flux_val == blank)
		    flux_val = blank_flag;
	    } 
	    else 
	    { 
		flux_val = blank_flag;
	    } 
	    break; 


	case 16: 
	    disp = disp * 2; 
	    lseek(fd, disp + filehdr.img_start, 0); 
	    bytes = read(fd,&vals,2); 

	    if (bytes == 2) 
	    { 
		flux_val = (double) vals; 

		if(debug)
		    fprintf(debug_file, 
			"flux:   dn=%d\n", vals);
		if (flux_val == blank)
		    flux_val = blank_flag;
	    } 
	    else 
	    { 
		flux_val = blank_flag;
	    } 
	    break; 

	case 32: 
	    disp = disp * 4; 
	    lseek(fd, disp + filehdr.img_start, 0); 
	    bytes = read(fd,&val,4); 

	    if (bytes == 4) 
	    { 
		flux_val =  (double) val; 

		if(debug)
		    fprintf(debug_file, "flux:  dn=%d\n",  val);
		if (flux_val == blank)
		    flux_val = blank_flag;
	    } 
	    else 
	    { 
		flux_val = blank_flag;
	    } 
	    break; 
	default: 
	    /* -32 and -64 */
	    printf("Internal error;  unexpected bitpix in flux(): %d\n", code); 
	    flux_val = blank_flag;
	    break; 
	}
    }

    if (debug)
	fprintf(debug_file, "flux: output flux_val = %g\n", flux_val);
    fluxes[0] = flux_val;
    return; 
}

void *zap_flux(i_coord)    /* returns pointer to pixels */
int i_coord[2];   /* i_coord is in IRAF coordinate system */
{ 
    int samp_offset, line_offset;      /* 0 based offsets */
    double flux_val;
    unsigned char *p_char;
    short *p_short; 
    unsigned short *p_ushort; 
    int disp, *p_int; 
    int line, samp;
    float *fptr;
    double *dptr;
    void *v_ptr;
    struct timeval currenttime;
    char window_name[200];

    samp = i_coord[0];
    line = i_coord[1];
    if (debug)
	fprintf(debug_file, 
	    "change_flux:  entry values samp = %d  line = %d\n", samp,line);

    /* Find the offsets */ 

    samp_offset = samp - 1;
    line_offset = line - 1;
    if ((line_offset >= lines) || (samp_offset >= pixels) ||
	(line_offset < 0)      || (samp_offset < 0))
    {
	error1("zap error:  location not on image");
	return(NULL);
    }


    /* Find the offset displacement in file (array) */ 
    disp = samp_offset + 
	   line_offset * pixels + 
	   band_offset * pixels * lines;


    if (mod_filnam[0] == '\0')
    {
	/* this is the first change to this image */
	v_ptr = imget(-1, filnam);        /* get the pixels */
    }
    else
    {
	v_ptr = imget(-1, mod_filnam);
    }

    if (v_ptr == NULL)
    {
	error1("cannot get the pixels to modify");
	return(v_ptr);
    }

    if (zap_blank)
	if (blank_set)
	    flux_val = blank;
	else
	{
	    error1("zap error:  blank is undefined");
	    return(NULL);
	}
    else
	flux_val = zap_value;

    /* Process based on number of bits */ 

    switch (bitpix) 
    { 
	case 8: 
	if ((flux_val < 0) || (flux_val > 255))
	{
	    error1("zap value must be 0 - 255 for BITPIX 8");
	    return(NULL);
	}
	p_char = (unsigned char *) v_ptr;
	p_char += disp;
	*p_char = (unsigned char) flux_val;
	break; 

    case 16: 
	if (u16)
	{
	    if ((flux_val < 0) || (flux_val > 65535))
	    {
		error1("zap value must be 0 - 65535 for BITPIX 16 (unsinged)");
		return(NULL);
	    }
	    p_ushort = (unsigned short *) v_ptr;
	    p_ushort += disp;
	    *p_ushort = (unsigned short) flux_val;
	}
	else
	{
	    if ((flux_val < -32768) || (flux_val > 32767))
	    {
		error1("zap value must between -32768 and 32767 for BITPIX 16");
		return(NULL);
	    }
	    p_short = (short *) v_ptr;
	    p_short += disp;
	    *p_short = (short) flux_val;
	}
	break; 

    case 32: 
	if ((flux_val < -2147483648.0) || (flux_val > 2147483647))
	{
	    error1("zap value out of range for BITPIX 32");
	    return(NULL);
	}
	p_int = (int *) v_ptr;
	p_int += disp;
	*p_int = (int) flux_val;
	break; 
    case -32: 
	fptr = (float *) v_ptr;
	fptr += disp;
	*fptr = flux_val;
	break; 
    case -64: 
	dptr = (double *) v_ptr;
	dptr += disp;
	*dptr = flux_val;
	break; 
    } 
    if (mod_filnam[0] == '\0')
    {
	/* this is the first change to this image - get scratch filnam */
	gettimeofday(&currenttime, (struct timezone *) 0);
	sprintf(mod_filnam, "%ld", currenttime.tv_sec);
	fill_img(curr_img);      /* force into img struct for this image */
	change_pixel_name(filnam, mod_filnam, TRUE);
	sprintf(window_name,"%s  (modified) (frame %d)",
	    strrchr(filnam, '/') + 1, image_frame);
	change_win_name(window_name);
    }
    return(v_ptr);
}
