/*
Copyright (C) 1996, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* fitssplit.c */
/* Splits fits files into NAXIS3 separate planes */
#include "im_proto.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/types.h>
#include <time.h>
#include <sys/param.h>

extern char *im_history_line; /* IMage access history line */

int main(argc, argv)
int argc;
char *argv[];
{
    int bitpix1, totpixels, imstat, totbytes, outpixels;
    char errmsg[200];
    void *pixel_ptr;
    int plane;
    char *ptr, strplane[5];
    char fi_name[MAXPATHLEN], outfile[MAXPATHLEN];
    char base_filename[MAXPATHLEN];
    int base_chars;
    int naxis, naxis3, bytes_per_pixel;
    int fd_in[2], fd_out;
    unsigned char *c_pixel_ptr;
    short *s_pixel_ptr;
    int *i_pixel_ptr;
    float *f_pixel_ptr;
    double *d_pixel_ptr;
    char *ct, history_line[80];
    time_t tsec;


    if (argc < 2)
    {
	printf("usage: fitssplit filename\n");
	exit(1);
    }


    tsec = time(0);
    ct = ctime(&tsec);
    sprintf(history_line, "WRITTEN BY FITSSPLIT (%s", ct);
    ptr = strchr(history_line, '\n');
    if (ptr != 0)
	*ptr = ')';
    im_history_line = history_line;

    /* open input file */
    strcpy(fi_name, argv[1]);
    fd_in[0] = im_open(fi_name, "r");
    if (fd_in[0] < 0)
    {
	perror(fi_name);    /* bad stuff - file has disappeared */
	exit(1);
    }

    ptr = strstr(argv[1], ".fit");
    if (ptr == NULL)
    {
	ptr = strstr(argv[1], ".FIT");
    }
    if (ptr == NULL)
    {
	strcpy(base_filename, argv[1]);
    }
    else
    {
	base_chars = ptr - argv[1];
	strncpy(base_filename, argv[1], base_chars);
	base_filename[base_chars] = '\0';
    }



    im_rkey_i(fd_in[0], &totpixels, "npix");
    im_rkey_i(fd_in[0], &bitpix1, "BITPIX");
    im_rkey_i(fd_in[0], &naxis, "NAXIS");
    im_rkey_i(fd_in[0], &naxis3, "NAXIS3");

    if (naxis != 3)
    {
	fprintf(stderr,
	    "fitssplit error:  NAXIS is %d  (must be 3)\n", naxis);
	exit(1);
    }
    if (naxis3 <= 1)
    {
	fprintf(stderr,
	    "fitssplit error:  image file does not have multiple planes\n");
	exit(1);
    }

    /* read the pixels */
    switch(bitpix1)
    {
	case 8:
	    bytes_per_pixel = 1;
	    break;
	case 16:
	    bytes_per_pixel = 2;
	    break;
	case 32:
	    bytes_per_pixel = 4;
	    break;
	case -32:
	    bytes_per_pixel = sizeof(float);
	    break;
	case -64:
	    bytes_per_pixel = sizeof(double);
	    break;
    }
    totbytes = totpixels * bytes_per_pixel;

    pixel_ptr = (void *) malloc(totbytes);
    if (pixel_ptr == NULL)
    {
	fprintf(stderr, "unable to allocate %d byte buffer for pixels\n",
	    totbytes);
	exit(1);
    }

    switch(bitpix1)
    {
	case 8:
	    c_pixel_ptr = (unsigned char *) pixel_ptr;
	    imstat = im_rpix_c(fd_in[0], c_pixel_ptr, totpixels);
	    break;
	case 16:
	    s_pixel_ptr = (short *) pixel_ptr;
	    imstat = im_rpix_s(fd_in[0], s_pixel_ptr, totpixels);
	    break;
	case 32:
	    i_pixel_ptr = (int *) pixel_ptr;
	    imstat = im_rpix_i(fd_in[0], i_pixel_ptr, totpixels);
	    break;
	case -32:
	    f_pixel_ptr = (float *) pixel_ptr;
	    imstat = im_rpix_r(fd_in[0], f_pixel_ptr, totpixels);
	    break;
	case -64:
	    d_pixel_ptr = (double *) pixel_ptr;
	    imstat = im_rpix_d(fd_in[0], d_pixel_ptr, totpixels);
	    break;
    }
    outpixels = totpixels / naxis3;

    for (plane = 1; plane <= naxis3; plane++)
    {
	/* open output file */
	strcpy(outfile, base_filename);
	strcat(outfile, ".pl");
	sprintf(strplane, "%d", plane);
	strcat(outfile, strplane);


	fd_out = im_open(outfile, "w");
	if (fd_out < 0)
	{
	    perror(outfile);
	    im_close(fd_in[0]);
	    exit(1);
	}
	fd_in[1] = 0;
	imstat = im_hdr(fd_out, fd_in, 0);
	if (imstat != 0)
	{
	    fprintf(stderr,
		"fitssplit error:  cannot copy header to output file %s\n",
		outfile);
	    exit(1);
	}
	im_wkey_i(fd_out, 2, "NAXIS");
	/* delete NAXIS3 */
	im_wkey_i(fd_out, 0, "NAXIS3");
	im_wkey_t(fd_out, 0,"delete");

	/* write the pixels */
	switch(bitpix1)
	{
	    case 8:
		imstat = im_wpix_c(fd_out, c_pixel_ptr, outpixels);
		c_pixel_ptr += outpixels;
		break;
	    case 16:
		imstat = im_wpix_s(fd_out, s_pixel_ptr, outpixels);
		c_pixel_ptr += outpixels;
		break;
	    case 32:
		imstat = im_wpix_i(fd_out, i_pixel_ptr, outpixels);
		i_pixel_ptr += outpixels;
		break;
	    case -32:
		imstat = im_wpix_r(fd_out, f_pixel_ptr, outpixels);
		f_pixel_ptr += outpixels;
		break;
	    case -64:
		imstat = im_wpix_d(fd_out, d_pixel_ptr, outpixels);
		d_pixel_ptr += outpixels;
		break;
	}


	/* due to the delayed open in image access, a bad status here       */
	/* may mean that the file was never opened.  In this case the fid   */
	/* inside image access is 0.  Then im_close() will close file 0     */
	/* Thus, we only close the file if the status is good.              */
	if (imstat >= 0)
	{
	    if (im_close(fd_out) != 0)
	    {
		fprintf(stderr, "error closing file %s\n", outfile);
		exit(1);
	    }
	}


#ifdef NOTDEF
	hdrfloat("DATAMIN", (float) 0);
	hdrfloat("DATAMAX", (float) 0);
	hdrfloat("DATAMIN", thdr.img_min * bscale + b_zero);
	hdrfloat("DATAMAX", thdr.img_max * bscale + b_zero);
#endif /* NOTDEF */

	if (imstat != outpixels)
	{
	    sprintf(errmsg, "error writing pixels to new file %s: ", outfile);
	    perror(errmsg);
	    exit(1);
	}
    }
    im_close(fd_in[0]);
    return(0);
}
