/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/
#include "fits_table.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <math.h>
#include <errno.h>

#ifndef SEEK_SET
#define SEEK_SET        0
#endif /* SEEK_SET */

#define MAXFIELDS 150
#define MAXWIDTH 5000   /* max width of a field */
#define MAXTABLELINE 5000
#define TRUE 1
#define FALSE 0


static char zeros[2880];
static int vax_bytes;
static int read_status;
static char *error_msg = "";


/* per file information */

struct file_info {
    struct file_info *llink, *rlink;
    int unit;
    int filetype;
    char *primary_header;
    int header_bytes;
    char *extension_header;
    int extension_bytes;
    int total_width;
    char *input_line;
    int number_of_fields;
    int lines;
    int linenumber;
    char *field_name[MAXFIELDS];
    char field_unit[MAXFIELDS][41];
    int element_count[MAXFIELDS];
    size_t bytes_written;
    int header_written;
    char field_type[MAXFIELDS];
    int field_start[MAXFIELDS];
    int field_width[MAXFIELDS];
    char format[MAXFIELDS][20];
    char *field_value[MAXFIELDS];
    FILE *fp;
    long data_start;
    char mode; /* 'r' or 'w' */
    };

static struct file_info head_file = {&head_file, &head_file};

/* function prototypes */
#ifdef _NO_PROTO

static struct file_info *new_file();
static struct file_info *find_file();
static struct file_info *set_file();
static char *find_keyword();
static void concat_value();
static void delete_table_file();
#ifdef NO_STRERROR
char *strerror();
#endif /* NO_STRERROR */


#else

static struct file_info *new_file(void);
static struct file_info *find_file(int unit);
static struct file_info *set_file(int unit);
static char *find_keyword(struct file_info *this_file, char *keyword);
static void concat_value(char *output_line, struct file_info *output_file,
    char *value, int field_number);
static void delete_table_file(struct file_info *p);
#ifdef NO_STRERROR
char *strerror(int errnbr);
#endif /* NO_STRERROR */


#endif /* _NO_PROTO */


static 
struct file_info *find_file(unit)
int unit;
{
    struct file_info *p;

    p = head_file.rlink;
    while (p != &head_file)
    {
	if (unit == p->unit)
	{
	    return(p);
	}
	p = p->rlink;
    }
    return(NULL);
}
static 
struct file_info *new_file()
{
    struct file_info *x, *tmpstruct;

    tmpstruct = (struct file_info *) malloc(sizeof(struct file_info));
    if (tmpstruct == NULL)
    {
	error_msg = "out of memory";
	return(tmpstruct);
    }
    memset(tmpstruct, 0, sizeof(struct file_info));

    /* insert to right of head_file */
    x = &head_file;
    tmpstruct->llink = x;
    tmpstruct->rlink = x->rlink;
    (tmpstruct->rlink)->llink = tmpstruct;
    x->rlink = tmpstruct;

    return(tmpstruct);
}

static void
delete_table_file(p)
struct file_info *p;
{
    (p->llink)->rlink = p->rlink;
    (p->rlink)->llink = p->llink;
    free(p);
    return;
}

static 
struct file_info *set_file(unit)
int unit;
{
    struct file_info *p;

    p = find_file(unit);
    if (p == NULL)
    {
	p = new_file();
	if (p != NULL)
	{
	    p->unit = unit;
	    p->filetype = IPAC_ASCII;  /* default */
	    p->header_bytes = 0;
	    p->linenumber = 0;
	    p->fp = NULL;
	}
    }
    return(p);
}



void topenr(unit, filename, nparms)
int unit;
char *filename;
int *nparms;
{
    size_t bytes_read;
    char buf[2880];
    char *pt, *quote1, *quote2;
    char *scan_ptr;
    int i, bytes_processed;
    int nchars, pixels;
    char field_spec[80];
    int netlong;
    int field_number = -1;
    unsigned char *byte_ptr;
    struct file_info *current_file;
    char *p, *pout;
    int chars;
    long data_start;
    int data_start_line;

    /* test for byteorder */
    netlong = 0x12345678;
    byte_ptr = (unsigned char *) &netlong;
    if ((byte_ptr[0] == 0x12) && (byte_ptr[1] == 0x34))
    vax_bytes = FALSE;
    else if ((byte_ptr[0] == 0x78) && (byte_ptr[1] == 0x56))
    vax_bytes = TRUE;
    else
    {
	fprintf(stderr, " error:  unrecognized byte order on this machine\n");
	vax_bytes = FALSE;  /* and pray */
    }

    read_status = -1;

    current_file = set_file(unit);
    if (current_file == NULL)
	return;

    current_file->mode = 'r';

    current_file->fp = fopen(filename, "r");
    if (current_file->fp == NULL)
    {
	error_msg = strerror(errno);
	if (error_msg == NULL)
	    error_msg = "Unknown error";
	return;
    }

    /* test the file to see if it is IPAC_ASCII or FITS_BINARY */
    bytes_read = fread(buf, 1, 6, current_file->fp);
    rewind(current_file->fp);
    if (strncmp(buf, "SIMPLE", 6) == 0)
    {

    /* FITS_BINARY */
    current_file->filetype = FITS_BINARY;

    for (;;)
    {
    /* skip through the primary FITS header (but save it in primary_header) */
	bytes_read = fread(buf, 1, 2880, current_file->fp);
	if (bytes_read != 2880)
	{
	    /* we got a short read - must not have an extension header */
	    /* See if we can salvage the primary header */
	    for (pt = current_file->primary_header, bytes_processed = 0;
		bytes_processed < current_file->header_bytes; 
		pt += 80, bytes_processed += 80)
	    {
		if (strncmp(pt, "END   ", 6) == 0)
		{
		    current_file->header_bytes = bytes_processed + 80;
		    current_file->extension_header = NULL;
		    current_file->extension_bytes = 0;
		    read_status = 0;
		    break;
		}
	    }
	    if (read_status != 0)
		error_msg = "no END keyword in primary FITS header";
	    return;
	}
	if (memcmp(buf, "XTENSION", 8) == 0)
	{
	    /* go back through primary header and find END */
	    for (pt = current_file->primary_header; ; pt += 80)
	    {
		if (strncmp(pt, "END   ", 6) == 0)
		{
		    current_file->header_bytes = pt - 
			current_file->primary_header + 80;
		    break;
		}
	    }
	    break;
	}
	if (current_file->header_bytes == 0)
	    current_file->primary_header = (char *) malloc(2880);
	else
	    current_file->primary_header = 
		(char *) realloc(current_file->primary_header, 
		    current_file->header_bytes + 2880);
	    
	memcpy(current_file->primary_header + current_file->header_bytes, 
	    buf, 2880);
	current_file->header_bytes += 2880;
    }

    /* now go through extension header */
    current_file->extension_header = (char *) malloc(2880);
    memcpy(current_file->extension_header, buf, 2880);
    current_file->extension_bytes = 2880;
    pt = buf;
    for (;;)
    {
    for (i = 0; i < 36; i++, pt+=80)
    {
	if (memcmp(pt, "NAXIS1  ", 8) == 0)
	{
	    pixels = atoi(pt+10);
	}
	else if (memcmp(pt, "NAXIS2  ", 8) == 0)
	{
	    current_file->lines = atoi(pt+10);
	}
	else if (memcmp(pt, "TFIELDS ", 8) == 0)
	{
	    current_file->number_of_fields = atoi(pt+10);
	    if (current_file->number_of_fields > MAXFIELDS)
	    {
		error_msg = 
"Tablefile internal error - total number of fields is too large\n  Change constant MAXFIELDS in fits_table.c";
		fprintf(stderr,
		"Tablefile internal error - total number of fields is too large\n");
		fprintf(stderr,
		"   Change constant MAXFIELDS in fits_table.c\n");
		read_status = 1;
		return;
	    }
	}
	else if (memcmp(pt, "END     ", 8) == 0)
	{
	    current_file->total_width = 0;
	    for (i = 1; i <= current_file->number_of_fields; i++)
	    {
		current_file->field_start[i] = current_file->total_width;
		current_file->total_width += current_file->field_width[i];
	    }
	    current_file->input_line = 
		(char *) malloc(current_file->total_width+1);
	    if (current_file->input_line == NULL)
	    {
		error_msg = 
	    "Tablefile internal error - cannot malloc space for input_line";
		fprintf(stderr,
	    "Tablefile internal error - cannot malloc space for input_line\n");
		read_status = 1;
		return;
	    }
	    *nparms = current_file->number_of_fields;
	    /* go back through extension header and find END */
	    for (pt = current_file->extension_header; ; pt += 80)
	    {
		if (strncmp(pt, "END   ", 6) == 0)
		{
		    current_file->extension_bytes = pt - 
			current_file->extension_header + 80;
		    break;
		}
	    }
	    current_file->data_start = ftell(current_file->fp);
	    read_status = 0;
	    return;  /* main exit */
	}
	else if (memcmp(pt, "TTYPE", 5) == 0)
	{
	    field_number = atoi(pt+5);
	    quote1 = strchr(pt+9, '\047');    /* look for ' */
	    if (quote1 != NULL)
	    {
		quote2 = strchr(quote1 + 1, '\047');  /* second ' */
		if (quote2 != NULL)
		{
		    nchars = quote2 - quote1 - 1;
		    current_file->field_name[field_number] = (char *) malloc(9);
		    if (nchars > 8)
		    {
			nchars = 8;
			memcpy(current_file->field_name[field_number], 
			    quote1 + 1, nchars);
			current_file->field_name[field_number][nchars] = '\0';
			fprintf(stderr,
"Tablefile internal error - fieldname %s is longer than 8 characters\n", 
			    current_file->field_name[field_number]);
		    }
		    else
		    {
			memcpy(current_file->field_name[field_number], 
			    quote1 + 1, nchars);
			current_file->field_name[field_number][nchars] = '\0';
		    }
		}
	    }
	}
	else if (memcmp(pt, "TUNIT", 5) == 0)
	{
	    field_number = atoi(pt+5);
	    quote1 = strchr(pt+9, '\047');    /* look for ' */
	    if (quote1 != NULL)
	    {
		quote2 = strchr(quote1 + 1, '\047');  /* second ' */
		if (quote2 != NULL)
		{
		    nchars = quote2 - quote1 - 1;
		    if (nchars > 40)
		    {
			nchars = 40;
			memcpy(current_file->field_unit[field_number], 
			    quote1 + 1, nchars);
			current_file->field_unit[field_number][nchars] = '\0';
			fprintf(stderr,
"Tablefile internal error - fieldunit %s is longer than 40 characters\n", 
			    current_file->field_unit[field_number]);
		    }
		    else
		    {
			memcpy(current_file->field_unit[field_number], 
			    quote1 + 1, nchars);
			current_file->field_unit[field_number][nchars] = '\0';
		    }
		}
	    }
	}
	else if (memcmp(pt, "TFORM", 5) == 0)
	{
	    field_number = atoi(pt+5);
	    quote1 = strchr(pt+9, '\047');    /* look for ' */
	    if (quote1 != NULL)
	    {
		quote2 = strchr(quote1 + 1, '\047');  /* second ' */
		if (quote2 != NULL)
		{
		    nchars = quote2 - quote1 - 1;
		    if (nchars > 79)
			nchars = 79;
		    memcpy(field_spec, quote1 + 1, nchars);
		    field_spec[nchars] = '\0';
		    if (isdigit(field_spec[0]))
		    {
			current_file->element_count[field_number] =
			    (int) strtol(field_spec, &scan_ptr, 10);
		    }
		    else
		    {
			/* no element count */
			current_file->element_count[field_number] = 1;
			scan_ptr = field_spec;
		    }
		    current_file->field_type[field_number] = scan_ptr[0];
		    switch(current_file->field_type[field_number])
		    {
			case 'j':
			    /* make upper case */
			    current_file->field_type[field_number] = 'J';
			    /*FALLTHROUGH*/
			case 'J':
			    /* int */
			    current_file->field_width[field_number] = 
				4 * current_file->element_count[field_number];
			    break;
			case 'i':
			    /* make upper case */
			    current_file->field_type[field_number] = 'I';
			    /*FALLTHROUGH*/
			case 'I':
			    /* short */
			    current_file->field_width[field_number] = 
				2 * current_file->element_count[field_number];
			    break;
			case 'e':
			    /* make upper case */
			    current_file->field_type[field_number] = 'E';
			    /*FALLTHROUGH*/
			case 'E':
			    /* float */
			    current_file->field_width[field_number] = 
				4 * current_file->element_count[field_number];
			    break;
			case 'b':
			    /* make upper case */
			    current_file->field_type[field_number] = 'B';
			    /*FALLTHROUGH*/
			case 'B':
			    /* unsigned byte */
			    current_file->field_width[field_number] = 
				1 * current_file->element_count[field_number];
			    break;
			case 'a':
			    /* make upper case */
			    current_file->field_type[field_number] = 'A';
			    /*FALLTHROUGH*/
			case 'A':
			    /* character */
			    current_file->field_width[field_number] = 
				1 * current_file->element_count[field_number];
			    break;
			case 'd':
			    /* make upper case */
			    current_file->field_type[field_number] = 'D';
			    /*FALLTHROUGH*/
			case 'D':
			    /* double */
			    current_file->field_width[field_number] = 
				8 * current_file->element_count[field_number];
			    break;
		    }
		}
	    }
	    }
	    else if (memcmp(pt, "TDISP", 5) == 0)
	    {
		/* ignor */
		/*EMPTY*/
	    }
	}
	/* read another block of the extension header */
	current_file->extension_header = 
	    (char *) realloc(current_file->extension_header, 
		current_file->extension_bytes + 2880);
	bytes_read = fread(
	    current_file->extension_header + current_file->extension_bytes, 
	    1, 2880, current_file->fp);
	if (bytes_read != 2880)
	    return;
	pt = current_file->extension_header + current_file->extension_bytes;
	current_file->extension_bytes += 2880;
	}
    }
    else
    {
	/* IPAC_ASCII */
	current_file->filetype = IPAC_ASCII;

	current_file->input_line = 
	    (char *) malloc(MAXTABLELINE+1);
	if (current_file->input_line == NULL)
	{
	    error_msg = 
	"Tablefile internal error - cannot malloc space for input_line";
	    read_status = 1;
	    return;
	}
    do
    {
	data_start = ftell(current_file->fp);   /* data may start here */
	data_start_line = current_file->linenumber;
	if (fgets(current_file->input_line, MAXTABLELINE, current_file->fp) == NULL) 
	{
	    fclose(current_file->fp);
	    read_status = -1;
	    return;
	}
	if (strlen(current_file->input_line) == MAXTABLELINE-1)
	{
	    error_msg = "first line of table file longer than MAXTABLELINE";
	    fclose(current_file->fp);
	    read_status = 1;
	    return;
	}
	current_file->linenumber++;
    } while (current_file->input_line[0] == 0x5c);   /* discard comments (backslash lines) */

    /* This is the first non-comment line - It defines the fields */
    current_file->number_of_fields = 0;
    p = strchr(current_file->input_line, '|');
    while (p != NULL)
    {
#ifdef DEBUG
	if (debug)
	    fprintf(debug_file, "field number %d starts at col %d\n", 
	    current_file->number_of_fields + 1, p - current_file->input_line);
#endif /* DEBUG */
	current_file->field_start[current_file->number_of_fields+1] = p - current_file->input_line;
	p = strchr(++p, '|');
	if (p != NULL)
	    current_file->number_of_fields++;
    }

    /* get field names */
    if (current_file->number_of_fields != 0)
    {
	for (i = 0; i < current_file->number_of_fields; i++)
	{
	    p = &current_file->input_line[current_file->field_start[i+1] + 1];
	    chars = current_file->field_start[i+2] - 
		current_file->field_start[i+1] - 1;

	    /* get rid of leading spaces */
	    while ((chars) && (*p == 0x20))
	    {
		p++;
		chars--;
	    }

	    if (!chars)    /* if this fieldname is blank */
		current_file->field_name[i+1] = (char *) malloc(10);
	    else
		current_file->field_name[i+1] = (char *) malloc(chars+1);

	    if (!chars)    /* if this fieldname is blank */
	    {
		sprintf(current_file->field_name[i+1], "field%d", i+1);

	    }
	    else
	    {
		pout = current_file->field_name[i+1];
		while(chars--)
		{
		    *pout++ = *p++;
		}
		/* get rid of trailing spaces */
		while (*--pout == 0x20)
		    ;
		pout++;
		*pout++ = '\0';
	    }
	}

	/* now find the first data */
	do
	{
	    data_start = ftell(current_file->fp);   /* data may start Here */
	    data_start_line = current_file->linenumber;
	    if (fgets(current_file->input_line, MAXTABLELINE, current_file->fp) == NULL) 
		break;
	    current_file->linenumber++;
	} while ((current_file->input_line[0] == 0x5c) || 
	    (current_file->input_line[current_file->field_start[0]] == '|'));

    }
    else
    {
	/* no header line - construct fields from 1st line of data */
	p = strchr(current_file->input_line, '\n');
	if (p)
	    *p = '\0';   /* change newline to null */
	/* delete trailing blanks */
	p = current_file->input_line + strlen(current_file->input_line) - 1;
	while (*p == ' ')
	    *p-- = '\0';

	p = current_file->input_line;
	while (p != NULL)
	{
#ifdef DEBUG
	    printf("field number %d starts at col %d\n", current_file->number_of_fields + 1,
		p - current_file->input_line);
#endif /* DEBUG */
	    current_file->field_start[current_file->number_of_fields+1] = p - current_file->input_line;
	    current_file->field_name[current_file->number_of_fields+1] = 
		(char *) malloc(10);
	    sprintf(current_file->field_name[current_file->number_of_fields+1], 
		"field%d", current_file->number_of_fields+1);
	    current_file->number_of_fields++;
	    while (*p == ' ')   /* discard leading blanks */
		p++;
	    p = strchr(p, ' ');
	    if (p == NULL)
	    {
#ifdef DEBUG
		printf("field number %d starts at col %d\n", current_file->number_of_fields + 1,
		    strlen(current_file->input_line));
#endif /* DEBUG */
		current_file->field_start[current_file->number_of_fields+1] = strlen(current_file->input_line);
	    }
	    if (current_file->number_of_fields > MAXFIELDS)
	    {
		error_msg = "problem looking for fields in first line";
		fclose(current_file->fp);
		read_status = 1;
		return;
	    }
	}
    }
    fseek(current_file->fp, data_start, 0);
    current_file->linenumber = data_start_line;

    for (i = 1; i <= current_file->number_of_fields; i++)
    {
	current_file->field_width[i] = current_file->field_start[i+1] -
	    current_file->field_start[i];
	current_file->field_type[i] = 'A';
#ifdef DEBUG
	printf("field_name[%d] =%s=\n", i, current_file->field_name[i]);
#endif /* DEBUG */
    }

    *nparms = current_file->number_of_fields;
    read_status = 0;
    }
}

void tget_primary(unit, adr, bytes)
int unit;
char **adr;
int *bytes;
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tget_primary:  unit not open";
	read_status = 1;
	return;
    }
    if (current_file->filetype == IPAC_ASCII)
    {
	error_msg = "illegal to call tget_primary with IPAC ASCII file";
	read_status = 1;
	return;
    }

    *adr = current_file->primary_header;
    *bytes = current_file->header_bytes;
    read_status = 0;
}

void tget_extension(unit, adr, bytes)
int unit;
char **adr;
int *bytes;
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tget_extension:  unit not open";
	read_status = 1;
	return;
    }
    if (current_file->filetype == IPAC_ASCII)
    {
	error_msg = "illegal to call tget_extension with IPAC ASCII file";
	read_status = 1;
	return;
    }

    *adr = current_file->extension_header;
    *bytes = current_file->extension_bytes;
    read_status = 0;
}

int tgetrowcount(unit)
int unit;
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgetrowcount:  unit not open";
	read_status = 1;
	return(0);
    }
    if (current_file->filetype == IPAC_ASCII)
    {
	error_msg = "illegal to call tgetrowcount with IPAC ASCII file";
	read_status = 1;
	return(0);
    }

    read_status = 0;
    return(current_file->lines);
}

/* tpinfo - return information about a given column */
void tpinfo(unit, pindex, pname, ptype, pblank, punit, pscale, poffs, pcol, 
    plen, pelements)
int unit;
int pindex;   /* column number (input) */
char pname[]; /* column name (output) */
char *ptype; /* value of parameter type (output) */
char pblank[]; /* string value of parameter 'blank' value (output) */
char punit[]; /* string value of parameter units (output) */
double *pscale; /* real value of parameter scaling value (output) */
double *poffs; /* real value of parameter offset value (output) */
int *pcol; /* value of starting column for this parameter (output) */
int *plen; /* value for number of characters for this parameter (output) */
int *pelements; /* integer value for number of parameter array elements */
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tpinfo:  unit not open";
	read_status = 1;
	return;
    }
    if ((pindex <= 0) || (pindex > current_file->number_of_fields))
    {
	error_msg = "field index out of range";
	read_status = 1;
	return;
    }

    strcpy(pname, current_file->field_name[pindex]);
    *ptype = current_file->field_type[pindex];
    strcpy(punit, current_file->field_unit[pindex]);
    *pcol = current_file->field_start[pindex];
    *plen = current_file->field_width[pindex];
    *pelements = current_file->element_count[pindex];
    read_status = 0;
}

#ifdef NOTDEF
void tpinfo1(unit, pindex, pname, ptype, pcol, plen, pelements)
int unit;
int pindex;   /* column number (input) */
char pname[]; /* column name (output) */
char *ptype; /* value of parameter type (output) */
int *pcol; /* value of starting column for this parameter (output) */
int *plen; /* value for number of characters for this parameter (output) */
int *pelements; /* integer value for number of parameter array elements */
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tpinfo1:  unit not open";
	read_status = 1;
	return;
    }
    strcpy(pname, current_file->field_name[pindex]);
    *ptype = current_file->field_type[pindex];
    *pcol = current_file->field_start[pindex];
    *plen = current_file->field_width[pindex];
    *pelements = current_file->element_count[pindex];
    read_status = 0;
}
#endif /* NOTDEF */

int tblread(unit)
int unit;
{
    int status;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tblread:  unit not open";
	read_status = 1;
	return(0);
    }
    if (current_file->filetype == IPAC_ASCII)
    {
	if (fgets(current_file->input_line, MAXTABLELINE, current_file->fp) == NULL) 
	    read_status = -1;
	else
	{
	    read_status = 0;
	    current_file->linenumber++;
	}
    }
    else
    {
	if (current_file->linenumber >= current_file->lines)
	{
	    read_status = -1;
	}
	else
	{
	    status = fread(current_file->input_line, 1, 
		current_file->total_width, current_file->fp);
	    if (status != current_file->total_width)
	    {
		read_status = -1;  /* EOF */
	    }
	    else
	    {
		read_status = 0;
		current_file->linenumber++;
	    }
	}
    }
    return(current_file->linenumber);
}

void
tseek(unit, recno)
int unit;
int recno;
{
    long pos;
    struct file_info *current_file;
    int status;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tseek:  unit not open";
	read_status = 1;
	return;
    }

    if (current_file->filetype == IPAC_ASCII)
    {
	error_msg = "error in tseek:  illegal to tseek an ascii file";
	read_status = 1;
	return;
    }

    if (recno <=0)
    {
	error_msg = "error in tseek:  negative record number";
	read_status = 1;
	return;
    }

    recno--;  /* records are 1-indexed in the user view */
    current_file->linenumber = recno;
    if (current_file->linenumber >= current_file->lines)
    {
	read_status = -1;  /* EOF */
    }
    else
    {
	/* the arithmetic in the next line may not work on non-unix machines */
	pos = current_file->data_start + recno * current_file->total_width;
	status = fseek(current_file->fp, pos, SEEK_SET);
	if (status == 0)
	    read_status = 0;
	else
	{
	    error_msg = strerror(errno);
	    if (error_msg == NULL)
		error_msg = "Unknown error";
	    read_status = 1;
	}
	/* reset EOF condition if encountered */ 
	if(feof(current_file->fp)) clearerr(current_file->fp);
    }
}

/* return the column number associated with a given name */
void tindp(unit, fieldname, pindex)
int unit;
char *fieldname;
int *pindex;
{
    int i;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tindp:  unit not open";
	read_status = 1;
	return;
    }

    for (i = 1; i <= current_file->number_of_fields; i++)
    {
	if (strncmp(current_file->field_name[i], fieldname, strlen(fieldname)) == 0)
	{
	    read_status = 0;
	    *pindex = i;
	    return;
	}
    }
    if (strncmp("field", fieldname, 5) == 0)
    {
	*pindex = atoi(&fieldname[5]);
	if (*pindex <= current_file->number_of_fields)
	{
	    read_status = 0;
	    return;
	}
    }
    error_msg = "error in tindp:  field not found";
    read_status = 1;
    *pindex = 0;
}

static
char *find_keyword(this_file, keyword)
struct file_info *this_file;
char *keyword;
{
    char *ptr;

    ptr = this_file->primary_header;
    for (;;)
    {
	if (strncmp(ptr, keyword, strlen(keyword)) == 0)
	    return ptr;
	if (strncmp(ptr, "END   ", 6) == 0)
	    return NULL;
	ptr += 80;
    }
}

void t_rkey_i(unit, i_ptr, keyword)
int unit;
int *i_ptr;
char *keyword;
{
    struct file_info *current_file;
    char *ptr, *check_ptr;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in t_rkey_i:  unit not open";
	read_status = 1;
	return;
    }

    ptr = find_keyword(current_file, keyword);
    if (ptr == NULL)
    {
	error_msg = "error in t_rkey_i:  keyword not found";
	read_status = 1;
    }
    else
    {
	ptr = strchr(ptr, '=');
	if (ptr)
	{
	    *i_ptr = (int) strtol(ptr+1, &check_ptr, 10);
	    if (check_ptr == ptr+1)
	    {
		error_msg = "error in t_rkey_i:  value cannot be converted to integer";
		read_status = 1;  /* error */
	    }
	    else
		read_status = 0;
	}
    }

}

void t_rkey_c(unit, c_ptr, keyword)
int unit;
char *c_ptr;
char *keyword;
{
    struct file_info *current_file;
    char *ptr;
    char *quote1, *quote2;
    int nchars;

    read_status = 1;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in t_rkey_c:  unit not open";
	return;   /* error return */
    }

    ptr = find_keyword(current_file, keyword);
    if (ptr)
    {
	ptr = strchr(ptr, '=');
	if (ptr)
	{
	    quote1 = strchr(ptr+1, '\047');    /* look for ' */
	    if (quote1 && (quote1 - ptr < 80))
	    {
		quote2 = strchr(quote1 + 1, '\047');  /* second ' */
		if (quote2)
		{
		    nchars = quote2 - quote1 - 1;
		    if (nchars > 79)
			nchars = 79;
		    memcpy(c_ptr, quote1 + 1, nchars);
		    c_ptr[nchars] = '\0';
		    read_status = 0;
		}
	    }
	}
	if (read_status)
	    error_msg = "error in t_rkey_c:  no string value";
    }
    else
	error_msg = "error in t_rkey_c:  keyword not found";

}

void t_rkey_d(unit, d_ptr, keyword)
int unit;
double *d_ptr;
char *keyword;
{
    struct file_info *current_file;
    char *ptr, *check_ptr;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in t_rkey_d:  unit not open";
	read_status = 1;
	return;
    }

    ptr = find_keyword(current_file, keyword);
    if (ptr == NULL)
    {
	error_msg = "error in t_rkey_d:  keyword not found";
	read_status = 1;
    }
    else
    {
	ptr = strchr(ptr, '=');
	if (ptr)
	{
	    *d_ptr = strtod(ptr+1, &check_ptr);
	    if (check_ptr == ptr+1)
	    {
		error_msg = "error in t_rkey_d:  value cannot convert to double";
		read_status = 1;  /* error */
	    }
	    else
		read_status = 0;
	}
    }

}

void t_rkey_r(unit, f_ptr, keyword)
int unit;
float *f_ptr;
char *keyword;
{
    struct file_info *current_file;
    char *ptr, *check_ptr;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in t_rkey_r:  unit not open";
	read_status = 1;
	return;
    }

    ptr = find_keyword(current_file, keyword);
    if (ptr == NULL)
    {
	error_msg = "error in t_rkey_r:  keyword not found";
	read_status = 1;
    }
    else
    {
	ptr = strchr(ptr, '=');
	if (ptr)
	{
	    *f_ptr = (float) strtod(ptr+1, &check_ptr);
	    if (check_ptr == ptr+1)
	    {
		error_msg = "error in t_rkey_r:  value cannot be converted to real";
		read_status = 1;  /* error */
	    }
	    else
		read_status = 0;
	}
    }

}

void tgetc(unit, pindex, pvalc)
int unit;
int pindex;
char *pvalc;
{
    int col;
    char *p;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgetc:  unit not open";
	read_status = 1;
	return;
    }
    if ((pindex <= 0) || (pindex > current_file->number_of_fields))
    {
	error_msg = "field index out of range";
	read_status = 1;
	return;
    }

    if (current_file->field_type[pindex] == 'A')
    {
	col = current_file->field_start[pindex];
	p = &current_file->input_line[col];
	strncpy(pvalc, p, current_file->field_width[pindex]);
	pvalc[current_file->field_width[pindex]] = '\0';
    }
    else
    {
	error_msg = "error in tgetc:  datum is not string";
	fprintf(stderr,
	"tablefile error in tgetc:  datum is not string\n");
	read_status = 1;
    }
}


void tgeti(unit, pindex, pvali)
int unit;
int pindex;
int *pvali;
{
    int i, col;
    INT16 vals;
    union
    {
	char *ch;
	INT32 *integer;
	INT16 *sht;
	INT8 *byte;
    } dat;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgeti:  unit not open";
	read_status = 1;
	return;
    }

    if ((pindex <= 0) || (pindex > current_file->number_of_fields))
    {
	error_msg = "field index out of range";
	read_status = 1;
	return;
    }

    col = current_file->field_start[pindex];
    dat.ch = &current_file->input_line[col];
    if (current_file->field_type[pindex] == 'J')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	    INT32fetch(dat.integer++, pvali++);
	read_status = 0;
    }
    else if (current_file->field_type[pindex] == 'I')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	{
	    INT16fetch(dat.sht++, &vals);
	    *pvali++ = vals;
	}
	read_status = 0;
    }
    else if (current_file->field_type[pindex] == 'B')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	    *pvali++ = (int) *dat.byte++;
	read_status = 0;
    }
    else
    {
	error_msg = "error in tgeti:  datum is not integer";
	fprintf(stderr,
	"tablefile error in tgeti:  datum is not integer\n");
	read_status = 1;
    }
}

#ifdef NOTDEF
void tgets(unit, pindex, pvals)
/* use tgeti() with same parameters */
#endif /* NOTDEF */

void tgetr(unit, pindex, pvalf)
int unit;
int pindex;
float *pvalf;
{
    int i, col;
    REAL8 dbl_val;
    union
    {
	char *ch;
	REAL4 *flt;
	REAL8 *dbl;
    } dat;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgetr:  unit not open";
	read_status = 1;
	return;
    }

    if ((pindex <= 0) || (pindex > current_file->number_of_fields))
    {
	error_msg = "field index out of range";
	read_status = 1;
	return;
    }

    col = current_file->field_start[pindex];
    dat.ch = &current_file->input_line[col];
    if (current_file->field_type[pindex] == 'D')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	{
	    REAL8fetch(dat.dbl++, &dbl_val);
	    *pvalf++ = (float) dbl_val;
	}
	read_status = 0;
    }
    else if (current_file->field_type[pindex] == 'E')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	    REAL4fetch(dat.flt++, pvalf++);
	read_status = 0;
    }
    else
    {
	error_msg = "error in tgetr:  datum is not floating point";
	fprintf(stderr,
	"tablefile error in tgetr:  datum is not floating point\n");
	read_status = 1;
    }
}

void tgetd(unit, pindex, pval_d)
int unit;
int pindex;
double *pval_d;
{
    int i, col;
    REAL4 flt_val;
    union
    {
	char *ch;
	REAL4 *flt;
	REAL8 *dbl;
    } dat;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgetd:  unit not open";
	read_status = 1;
	return;
    }

    if ((pindex <= 0) || (pindex > current_file->number_of_fields))
    {
	error_msg = "field index out of range";
	read_status = 1;
	return;
    }

    col = current_file->field_start[pindex];
    dat.ch = &current_file->input_line[col];
    if (current_file->field_type[pindex] == 'D')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	    REAL8fetch(dat.dbl++, pval_d++);
	read_status = 0;
    }
    else if (current_file->field_type[pindex] == 'E')
    {
	for (i = 0; i < current_file->element_count[pindex]; i++)
	{
	    REAL4fetch(dat.flt++, &flt_val);
	    *pval_d++ = (double) flt_val;
	}
	read_status = 0;
    }
    else
    {
	error_msg = "error in tgetd:  datum is not floating point";
	fprintf(stderr,
	"tablefile error in tgetd:  datum is not floating point\n");
	read_status = 1;
    }
}


void REAL8fetch(inptr, outptr)
REAL8 *inptr, *outptr;
{
    union
    {
	char *ch;
	REAL4 *flt;
	REAL8 *dbl;
    } in, out;

    in.dbl = inptr;
    out.dbl = outptr;

    if (vax_bytes)
    {
	out.ch[0] = in.ch[7];
	out.ch[1] = in.ch[6];
	out.ch[2] = in.ch[5];
	out.ch[3] = in.ch[4];
	out.ch[4] = in.ch[3];
	out.ch[5] = in.ch[2];
	out.ch[6] = in.ch[1];
	out.ch[7] = in.ch[0];
    }
    else
    {
	out.ch[0] = in.ch[0];
	out.ch[1] = in.ch[1];
	out.ch[2] = in.ch[2];
	out.ch[3] = in.ch[3];
	out.ch[4] = in.ch[4];
	out.ch[5] = in.ch[5];
	out.ch[6] = in.ch[6];
	out.ch[7] = in.ch[7];
    }

}

void REAL4fetch(inptr, outptr)
REAL4 *inptr, *outptr;
{
    union
    {
	char *ch;
	REAL4 *flt;
	REAL8 *dbl;
    } in, out;

    in.flt = inptr;
    out.flt = outptr;

    if (vax_bytes)
    {
	out.ch[0] = in.ch[3];
	out.ch[1] = in.ch[2];
	out.ch[2] = in.ch[1];
	out.ch[3] = in.ch[0];
    }
    else
    {
	out.ch[0] = in.ch[0];
	out.ch[1] = in.ch[1];
	out.ch[2] = in.ch[2];
	out.ch[3] = in.ch[3];
    }
}

void INT32fetch(inptr, outptr)
INT32 *inptr, *outptr;
{
    union
    {
	char *ch;
	INT32 *integer;
	INT16 *sht;
	INT8 *byte;
    } in, out;

    in.integer = inptr;
    out.integer = outptr;

    if (vax_bytes)
    {
	out.ch[0] = in.ch[3];
	out.ch[1] = in.ch[2];
	out.ch[2] = in.ch[1];
	out.ch[3] = in.ch[0];
    }
    else
    {
	out.ch[0] = in.ch[0];
	out.ch[1] = in.ch[1];
	out.ch[2] = in.ch[2];
	out.ch[3] = in.ch[3];
    }

}

void INT16fetch(inptr, outptr)
INT16 *inptr, *outptr;
{
    union
    {
	char *ch;
	INT32 *integer;
	INT16 *sht;
	INT8 *byte;
    } in, out;

    in.sht = inptr;
    out.sht = outptr;

    if (vax_bytes)
    {
	out.ch[0] = in.ch[1];
	out.ch[1] = in.ch[0];
    }
    else
    {
	out.ch[0] = in.ch[0];
	out.ch[1] = in.ch[1];
    }

}

void tcls(unit)
int unit;
{
    int i, status_w, status_c;
    size_t pad_length;
    struct file_info *current_file;
    char msg_buf[100];

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tcls:  unit not open";
	read_status = 1;
	return;
    }
    if ((current_file->mode == 'w') && (current_file->filetype == FITS_BINARY))
    {
	/* pad out to even multiple of 2880 bytes */
	pad_length = 2880 - (current_file->bytes_written % 2880);
	status_w = fwrite(zeros, 1, pad_length, current_file->fp);
	if (status_w != pad_length)
	    status_w = 1;
    }
    status_c = fclose(current_file->fp);
    current_file->fp = 0;

    if (current_file->primary_header != NULL)
	free(current_file->primary_header);
    current_file->primary_header = NULL;
    current_file->header_bytes = 0;

    if (current_file->extension_header != NULL)
	free(current_file->extension_header);
    current_file->extension_header = NULL;
    current_file->extension_bytes = 0;

    if (current_file->input_line != NULL)
	free (current_file->input_line);
    current_file->input_line = NULL;

    for (i = 0; i < current_file->number_of_fields; i++)
    {
	if (current_file->field_value[i] != NULL)
	    free (current_file->field_value[i]);
	current_file->field_value[i] = NULL;
    }

    for (i = 0; i < current_file->number_of_fields; i++)
    {
	if (current_file->field_name[i] != NULL)
	    free (current_file->field_name[i]);
	current_file->field_name[i] = NULL;
    }

    delete_table_file(current_file);


    if (status_w)
    {
	sprintf(msg_buf, "error in tcls flushing buffer: %s", strerror(errno));
	error_msg = msg_buf;
    }
    else if (status_c)
    {
	sprintf(msg_buf, "error in tcls closing file: %s", strerror(errno));
	error_msg = msg_buf;
    }
    else
	read_status = 0;
}

int itstat()
{
    return(read_status);
}

/* t_error returns error message for last error encountered */
char *t_error()
{
    return(error_msg);
}

int tgettyp(unit)
int unit;
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgettyp:  unit not open";
	read_status = 1;
	return(-1);
    }
    read_status = 0;
    return(current_file->filetype);
}

void tfiltyp(unit, newtype)
int unit;
int newtype;
{
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tfiltyp:  unit not open";
	read_status = 1;
	return;
    }
    current_file->filetype = newtype;
    read_status = 0;
}

void topenw(unit, filename)
int unit;
char *filename;
{
    struct file_info *current_file;

    read_status = -1;
    current_file = set_file(unit);
    if (current_file == NULL)
	return;
    if (current_file->fp != NULL)
    {
	printf("Error: unit %d is already open\n", unit);
	return;
    }
    current_file->mode = 'w';
    current_file->bytes_written = 0;
    current_file->number_of_fields = 0;
    current_file->fp = fopen(filename, "w");
    if (current_file->fp == NULL)
    {
	error_msg = strerror(errno);
	if (error_msg == NULL)
	    error_msg = "Unknown error";
	return;
    }
    read_status = 0;
}

void
tblwrit_raw(unit, string)
int unit;
char *string;
{
    struct file_info *output_file;

    output_file = set_file(unit);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tblwrit_raw:  unit not open";
	read_status = 1;
	return;
    }

    fputs(string, output_file->fp);
    read_status = 0;
}

void tlink(unit_in, unit_out)
int unit_in, unit_out;
{
    size_t bytes_read;
    char buf[2880];
    char *pt;
    int i;
    struct file_info *input_file, *output_file;
    char msg_buf[100];

    read_status = 1;

    output_file = set_file(unit_out);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tlink:  output unit not open";
	return;
    }

    input_file = set_file(unit_in);
    if (input_file->fp == NULL)
    {
	error_msg = "error in tlink:  input unit not open";
	return;
    }

    if ((input_file->filetype != FITS_BINARY) || 
	(output_file->filetype != FITS_BINARY))
    {
	error_msg = "error in tlink:  files must be FITS BINARY for linking";
	fprintf(stderr, 
"error in fits_table routine tlink:  files must be FITS BINARY for linking\n");
	return;
    }
    output_file->input_line = input_file->input_line;
    output_file->total_width = input_file->total_width;

    rewind (input_file->fp);
    for (;;)
    {
	/* skip the primary FITS header */
	bytes_read = fread(buf, 1, 2880, input_file->fp);
	if (bytes_read != 2880)
	{
	    sprintf(msg_buf, "error in tlink reading file: %s", 
		strerror(errno));
	    error_msg = msg_buf;
	    return;
	}
	fwrite(buf, 1, 2880, output_file->fp);
	if (memcmp(buf, "XTENSION", 8) == 0)
	    break;
    }

    for (;;)
    {
	for (pt = buf, i = 0; i < 36; i++, pt+=80)
	{
	    if (memcmp(pt, "END     ", 8) == 0)
	    {
		read_status = 0;
		return;
	    }
	}
	/* read another block of the extension header */
	bytes_read = fread(buf, 1, 2880, input_file->fp);
	if (bytes_read != 2880)
	{
	    sprintf(msg_buf, "error in tlink reading file: %s", 
		strerror(errno));
	    error_msg = msg_buf;
	    return;
	}
	fwrite(buf, 1, 2880, output_file->fp);
    }

}

/* return the memory address of a given column number */
void tgetaddr(unit, pindex, adr)
int unit;
int pindex;
char **adr;
{
    int col;
    struct file_info *current_file;

    current_file = set_file(unit);
    if (current_file->fp == NULL)
    {
	error_msg = "error in tgetaddr:  output unit not open";
	read_status = 1;
	return;
    }

    if (current_file->mode == 'w')
    {
	error_msg = "error in tgetaddr:  tgetaddr illegal for an output file";
	read_status = 1;
	return;
    }
    col = current_file->field_start[pindex];
    *adr = &current_file->input_line[col];
    read_status = 0;
}

/* concatenate a value into a field, truncating if necessary */
static
void concat_value(output_line, output_file, value, field_number)
char *output_line;
struct file_info *output_file;
char *value;
int field_number;
{
    int n, old_length;

    old_length = strlen(output_line);

    if (strlen(value) >= (size_t) output_file->field_width[field_number])
    {
	/* no padding spaces needed */
	strncat(output_line, value, 
	    output_file->field_width[field_number]);
	output_line[old_length + output_file->field_width[field_number]] ='\0';
    }
    else
    {
	strcat(output_line, value);
	n = output_file->field_width[field_number] 
	    - (strlen(output_line) - old_length);
	while (n-- > 0)
	    strcat(output_line, " ");
    }
}

void tblwrit(unit)
int unit;
{
    int i, status, do_units;
    struct file_info *output_file;
    char output_line[6500];
    char msg_buf[100];

    read_status = 0;

    output_file = set_file(unit);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tblwrit:  output unit not open";
	read_status = 1;
	return;
    }
    if (output_file->filetype == FITS_BINARY)
    {
	status = fwrite(output_file->input_line, 1, 
	    output_file->total_width, output_file->fp);
	output_file->bytes_written += status;
	if (status != output_file->total_width)
	{
	    sprintf(msg_buf, "error in tblwrit writing file: %s", 
		strerror(errno));
	    error_msg = msg_buf;
	    read_status = 1;
	}
	else
	    read_status = 0;
    }
    else
    {
	if (!output_file->header_written)
	{
	    /* do the header lines */
	    output_line[0] = '|';
	    output_line[1] = '\0';
	    for (i = 1; i <= output_file->number_of_fields; i++)
	    {
		concat_value(output_line, output_file, 
		    output_file->field_name[i], i);
		strcat(output_line, "|");
	    }
	    status = fprintf(output_file->fp, "%s\n", output_line);
	    if (status < 0)
	    {
		sprintf(msg_buf, "error in tblwrit writing file: %s", 
		    strerror(errno));
		error_msg = msg_buf;
		read_status = 1;
		return;
	    }
	    output_file->header_written = 1;

	    /* now see if we need to output the units line */
	    do_units = 0;
	    for (i = 1; i <= output_file->number_of_fields; i++)
	    {
		/* must have a non-space in field_unit */
		if (strspn(output_file->field_unit[i], " ") != 
		    strlen(output_file->field_unit[i]))
		{
		    do_units = 1;
		    break;
		}
	    }
	    if (do_units)
	    {
		output_line[0] = '|';
		output_line[1] = '\0';
		for (i = 1; i <= output_file->number_of_fields; i++)
		{
		    concat_value(output_line, output_file, 
			output_file->field_unit[i], i);
		    strcat(output_line, "|");
		}
		status = fprintf(output_file->fp, "%s\n", output_line);
		if (status < 0)
		{
		    sprintf(msg_buf, "error in tblwrit writing file: %s", 
			strerror(errno));
		    error_msg = msg_buf;
		    read_status = 1;
		    return;
		}
	    }
	    

	}
	output_line[0] = ' ';
	output_line[1] = '\0';
	for (i = 1; i <= output_file->number_of_fields; i++)
	{
	    concat_value(output_line, output_file, 
		output_file->field_value[i], i);
	    strcat(output_line, " ");
	}
	status = fprintf(output_file->fp, "%s\n", output_line);
	if (status < 0)
	{
	    sprintf(msg_buf, "error in tblwrit writing file: %s", 
		strerror(errno));
	    error_msg = msg_buf;
	    read_status = 1;
	}
    }
}

/* tnewp - add a new column to an output file */
void
tnewp(unit, pname, ptype, pblank, punit, pscal, poff, pfmt, plen, pindex)
int unit;
char *pname, *ptype, *pblank, *punit;
double pscal, poff;
char *pfmt;
int plen, *pindex;
{
    struct file_info *output_file;
    char msg_buf[100];

    output_file = set_file(unit);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tnewp:  output unit not open";
	read_status = 1;
	return;
    }
    *pindex = ++(output_file->number_of_fields);
    output_file->field_name[*pindex]  = (char *) malloc(plen+1);
    if (output_file->field_name[*pindex] == NULL)
    {
    fprintf(stderr, "table file error in tnewp:  unable to allocate space\n");
	error_msg = "error in tnewp:  unable to allocate space";
	read_status = 1;
	return;
    }
    strcpy(output_file->field_name[*pindex], pname);
    strcpy(output_file->field_unit[*pindex], punit);
    strcpy(output_file->format[*pindex], pfmt);
    output_file->field_width[*pindex] = plen;
    output_file->field_value[*pindex] = (char *) malloc(plen+1);
    if (output_file->field_value[*pindex] == NULL)
    {
	fprintf(stderr, "table file error in tnewp:  unable to allocate space\n");
	error_msg = "error in tnewp:  unable to allocate space";
	read_status = 1;
	return;
	
    }
    if (plen > MAXWIDTH)
    {
	fprintf(stderr, "table file error in tnewp:  length of field [%s] longer than MAXWIDTH (=%d)\n", pname, MAXWIDTH);
	fprintf(stderr, "    Change MAXWIDTH in file fits_table.c\n");
	read_status = 1;
	return;
    }
    if (strlen(pname) > (size_t) (plen))
    {
	sprintf(msg_buf,
	"error in tnewp:  field name [%s] too long to fit into column header",
	    pname);
	error_msg = msg_buf;
	fprintf(stderr,
"table file error in tnewp:  field name [%s] too long to fit into column header\n", pname);
	read_status = 1;
    }
    read_status = 0;
}

void
tsetc(unit, col, string)
int unit;
int col;
char *string;
{
    struct file_info *output_file;
    char temp[MAXWIDTH];
    char msg_buf[100];

    output_file = set_file(unit);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tsetc:  output unit not open";
	read_status = 1;
	return;
    }
    sprintf(temp, output_file->format[col], string);
    if (strlen(temp) > (size_t) output_file->field_width[col])
    {
	sprintf(msg_buf,
	"table file error in tsetc:  data value [%s] too large for field\n",
	    temp);
	error_msg = msg_buf;
	fprintf(stderr, 
	"table file error in tsetc:  data value [%s] too large for field\n",
	    temp);
	strncpy(output_file->field_value[col], temp,
	    output_file->field_width[col]);
	output_file->field_value[col][output_file->field_width[col] + 1] = '\0';
	read_status = 1;
    }
    else
    {
	strcpy(output_file->field_value[col], temp);
	read_status = 0;
    }
}

void
tseti(unit, col, integer)
int unit;
int col;
int integer;
{
    struct file_info *output_file;
    char temp[MAXWIDTH];
    char msg_buf[100];

    output_file = set_file(unit);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tseti:  output unit not open";
	read_status = 1;
	return;
    }
    sprintf(temp, output_file->format[col], integer);
    if (strlen(temp) > (size_t) output_file->field_width[col])
    {
	sprintf(msg_buf,
	"table file error in tseti:  data value [%s] too large for field\n",
	    temp);
	error_msg = msg_buf;
	fprintf(stderr, 
	"table file error in tseti:  data value [%s] too large for field\n",
	    temp);
	strncpy(output_file->field_value[col], temp,
	    output_file->field_width[col]);
	output_file->field_value[col][output_file->field_width[col] + 1] = '\0';
	read_status = 1;
    }
    else
    {
	strcpy(output_file->field_value[col], temp);
	read_status = 0;
    }
}

void
tsetr(unit, col, real)
int unit;
int col;
double real;
{
    struct file_info *output_file;
    char temp[MAXWIDTH];
    char msg_buf[100];

    output_file = set_file(unit);
    if (output_file->fp == NULL)
    {
	error_msg = "error in tsetr:  output unit not open";
	read_status = 1;
	return;
    }
    sprintf(temp, output_file->format[col], real);
    if (strlen(temp) > (size_t) output_file->field_width[col])
    {
	sprintf(msg_buf,
	"table file error in tsetr:  data value [%s] too large for field\n",
	    temp);
	error_msg = msg_buf;
	fprintf(stderr, 
	"table file error in tsetr:  data value [%s] too large for field\n",
	    temp);
	strncpy(output_file->field_value[col], temp,
	    output_file->field_width[col]);
	output_file->field_value[col][output_file->field_width[col] + 1] = '\0';
	read_status = 1;
    }
    else
    {
	strcpy(output_file->field_value[col], temp);
	read_status = 0;
    }
}


#ifdef TEST
main(argc, argv)
int argc;
char **argv;
{
    int nparms;
    int gpsctkey, p2smkyid;
    int gpsctkey_col, p2smkyid_col, ra_col, dec_col, roll_col, flux_col;
    int unit = 1;
    double ra, dec, roll;
    double flux1[4];
    float *flux_memadr1, *flux_memadr2, *flux_memadr3, *flux_memadr4;
    char *flux_charadr;

    if (argc != 2)
    {
	printf("usage: %s <tablefile>\n", argv[0]);
	exit(1);
    }

    topenr(unit, argv[1], &nparms);
    if (itstat())
    {
	printf("problem opening tablefile %s\n", argv[1]);
    }
    else
    {
	printf("tablefile %s opened successfully\n", argv[1]);
    }

    tindp(unit, "GPSCTKEY", &gpsctkey_col);
    if (itstat())
    {
	printf("cannot find GPSCTKEY column\n");
    }

    tindp(unit, "P2SMKYID", &p2smkyid_col);
    if (itstat())
    {
	printf("cannot find P2SMKYID column\n");
    }

    tindp(unit, "P2SMDEC", &dec_col);
    if (itstat())
    {
	printf("cannot find P2SMDEC column\n");
    }

    tindp(unit, "P2SMRA", &ra_col);
    if (itstat())
    {
	printf("cannot find P2SMRA column\n");
    }

    tindp(unit, "P2SMROL", &roll_col);
    if (itstat())
    {
	printf("cannot find P2SMROL column\n");
    }

    tindp(unit, "P2SMFLX", &flux_col);
    if (itstat())
    {
	printf("cannot find P2SMFLX column\n");
    }
    tgetaddr(unit, flux_col, &flux_charadr);
    flux_memadr1 = (float *) flux_charadr;
    flux_memadr2 = flux_memadr1 + 1;
    flux_memadr3 = flux_memadr1 + 2;
    flux_memadr4 = flux_memadr1 + 3;


    printf(
"GPSCKEY P2SMKYID  P2SMRA P2SMDEC P2SMROL P2SMFLX[1] P2SMFLX[2] P2SMFLX[3] P2SMFLX[4]\n");
    for (;;)
    {
	tblread(unit);
	if (itstat() > 0 )
	{
	    printf("error reading tablefile\n");
	}
	else if (itstat() < 0)
	{
	    printf("EOF reading tablefile\n");
	    break;
	}
	tgeti(unit, gpsctkey_col, &gpsctkey);
	tgeti(unit, p2smkyid_col, &p2smkyid);
	tgetd(unit, ra_col, &ra);
	tgetd(unit, dec_col, &dec);
	tgetd(unit, roll_col, &roll);
	tgetd(unit, flux_col, flux1);  /* alternate method */
	printf("%6d %1d  %9.4f  %9.4f  %9.4f %e %e %e %e\n",
	    gpsctkey, p2smkyid, ra, dec, roll, 
	    *flux_memadr1, *flux_memadr2, *flux_memadr3, *flux_memadr4);
    }
    tcls(unit);
}
#endif /* TEST */
