/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "fits.h"
#include "img.h"
#include "job.h"
#include "im_proto.h"
#include "parse.h"
#include "area.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <signal.h>
#include <ctype.h>
#include <sys/wait.h>
#include <pwd.h>
#include <sys/param.h>
#include <unistd.h>
#include <math.h>

#ifdef NO_FLOAT_H
#include <values.h>
#define DBL_MAX MAXDOUBLE
#define FLT_MAX MAXFLOAT
#else
#include <float.h>
#endif

extern struct hdrstruct filehdr;
extern char org_path[], pathnam[], filnam[], fi_name[], mod_filnam[];
extern int debug, server_mode, captive, area_number;
extern char server_str[];
extern int use_projtype, bitpix, pixels;
extern double crpix1, crpix2;
extern int u16, default_u16;
extern double bscale, b_zero, default_blank, blank, blank_flag;
extern int default_blank_set;
extern int reqband;
extern FILE *cmdfile, *debug_file;
extern FILE *session;
extern struct area_struct area[MAX_AREAS];
extern char *hdr_buf, *hdr_bufp_end;

extern char *im_history_line;  /* IMage access history line */


static char fnamsav[MAXPATHLEN] = { 's', 'e', 's', 's', 'i', 'o', 'n' };
static int session_on = FALSE;


/* function prototypes */
#ifdef _NO_PROTO

static int expand_tilde();
static int gethdr();
static int vicarhdr();
static int fitshdr();

#else

static int expand_tilde(char *outpath, char *inpath);
static int gethdr(int fd, struct img *imgp);
static int vicarhdr(int fd, struct img *imgp);
static int fitshdr(int fd, struct img *imgp);

#endif /* _NO_PROTO */

/**********************************************************************/
/*                                                                    */
/* DIR                                                                */
/* Prints list of files in a directory.                               */
/*                                                                    */
/**********************************************************************/

void
dir(argc, argv)
int argc;
char *argv[];
{
    int i;
    char sysstr[80];

    if(server_mode == TRUE)
    {
       error_na("dir command not allowed in server mode");
       return;
    }

    if (argc == 1)
	printf("files in %s\n", pathnam);

    strcpy(sysstr, "ls");

    for (i=1; i<argc; i++)
    {
	strcat(sysstr, " ");
	strcat(sysstr, argv[i]);
    }

    if (debug)
	fprintf(debug_file, "dir:  passing to system: '%s'\n", sysstr);

    i = system(sysstr);
    if (i != 0)
	printf("dir:  bad return from system   error code = %d\n", i);
}

/*ARGSUSED*/
void
pwd(argc, argv)
int argc;
char *argv[];
{
    char *p, strng[MAXPATHLEN+2];
    char *status_ptr;

    strng[0] = '\0';
    status_ptr = getcwd(strng, MAXPATHLEN+2);
    p = strng;
    if (*p == '\0')
	p = "unreadable";
    if (strncmp(p, "/tmp_mnt", 8) == 0)
	p += 8;
    if(server_mode == FALSE)
       printf("current directory is %s\n", p);
    else
    {
       srv_string("pathname", p);
    }
    fprintf(session, "current directory is %s\n", p);
}

/***********************************************************************/
/*                                                                    */
/* CH_DIR                                                             */
/* Changes default directory for images, take files, etc.             */
/*                                                                    */
/**********************************************************************/

void
ch_dir(argc, argv)
int argc;
char *argv[];
{
    char *p, path_tmp[MAXPATHLEN+2];
    int status;
    char strng[MAXPATHLEN+2];

    if (argc > 2)
    {
	error1("wrong number of arguments to CD command");
	return;
    }

    if (argc == 1)    /* CD alone means login directory */
    {
	/* fake a tilde */
	if (!expand_tilde(path_tmp, "~"))
	{
	    return;
	}
    }
    else if (argv[1][0] == '~')
    {
	/* expand the tilde */
	if (!expand_tilde(path_tmp, argv[1]))
	{
	    return;
	}
    }
    else
	strcpy(path_tmp, argv[1]);

    status = chdir(path_tmp);
    if (status != 0)
    {
	sperror(path_tmp, errno);
	return;
    }
    strng[0] = '\0';
    (void) getcwd(strng, MAXPATHLEN+2);
    p = strng;
    if (*p == '\0')
    {
	error1("cd error:  directory is unreadable");
	return;
    }
    if (strncmp(p, "/tmp_mnt", 8) == 0)
	p += 8;
    strcpy(pathnam, p);
    strcat(pathnam, "/");

    if(server_mode == TRUE)
    {
	srv_string("pathname", pathnam);
    }
}

/***********************************************************************/
/*                                                                    */
/* SDATA_FILENAME                                                     */
/* Returns pointer to filename expanded into a full pathname.         */
/* Handles leading / and ~                                            */
/* Doesn't touch . and ..                                             */
/* Filenames not beginning with / or ~ are searched for in SDATA path */
/*                                                                    */
/**********************************************************************/

char *sdata_filename(name)
char *name;
{
    struct stat stbuf;
    char *cp, *tp;
    char *s_path;
    char path[MAXPATHLEN];
    static char fullpath[MAXPATHLEN];

    if ((name[0] == '/') || (name[0] == '~'))
	return(expand_path(name));

    s_path = getenv("SDATA");
    if (s_path == 0)
	s_path = ":";

    for (cp = s_path; cp && *cp; cp = tp) 
    {
	tp = strchr(cp, ':');
	if (tp) 
	{
	    if (tp == cp) 
	    {
		sprintf(path, "%s", name);
	    }
	    else 
	    {
		sprintf(path, "%.*s/%s", (int) (tp-cp), cp, name);
	    }
	    tp++;
	} 
	else 
	{
	    sprintf(path, "%s/%s", cp, name);
	}

	(void) expand_tilde(fullpath, path);

	if (stat(fullpath, &stbuf) >= 0) 
	{
	    return (fullpath);
	}
    }
    return (NULL);
}


/***********************************************************************/
/*                                                                    */
/* EXPAND_PATH                                                        */
/* Returns pointer to filename expanded into a full pathname.         */
/* Handles leading / and ~                                            */
/* Doesn't touch . and ..                                             */
/*                                                                    */
/**********************************************************************/

char *expand_path(inpath)
char *inpath;
{
    static char path_tmp[MAXPATHLEN];

    if (inpath[0] == '/')
	strcpy(path_tmp, inpath);
    else if (inpath[0] == '~')
	(void) expand_tilde(path_tmp, inpath);
    else
    {
	strcpy(path_tmp, pathnam);
	strcat(path_tmp, inpath);
    }
    return(path_tmp);
}

/***********************************************************************/
/*                                                                    */
/* EXPAND_TILDE                                                       */
/* Copies filename in second arg into first arg, expanding leading ~  */
/* Doesn't touch . and ..                                             */
/* If OK, returns 1                                                   */
/* If errors, returns 0 (and copies the input unchanged to output)    */
/*                                                                    */
/**********************************************************************/

static int
expand_tilde(outpath, inpath)
char *outpath, *inpath;
{
    char *o, *p;
    struct passwd *pw;
    char person[40];

    strcpy(outpath, inpath);

    if (inpath[0] == '~')
    {
	/* expand the tilde */
	for (p = person, o = &inpath[1]; *o && *o != '/'; *p++ = *o++)
		;
	*p = '\0';
	if (person[0] == '\0')
	{
	    p = getenv("HOME");
	    if (p == NULL)
	    {
		error1("HOME environment variable is undefined");
		return(0);
	    }
	    strcpy(outpath, p);
	}
	else
	{
	    pw = getpwnam(person);
	    if (pw == NULL)
	    {
		sprintf(server_str, 
		    "cannot find home directory for %s", person);
		error1(server_str);
		return (0);
	    }
	    strcpy(outpath, pw->pw_dir);
	}
	strcat(outpath, o);
    }
    return(1);
}

void
chgfil(argc, argv)
int argc;
char *argv[];
{
    if (argc == 1)
    {
	if (fi_name[0] == '\0')
	{
	    error1("no workfile has been specified");
	}
	else
	{
	    if(server_mode == FALSE)
		printf("current workfile is %s\n", fi_name);
	    else
	    {
		srv_string("filename", fi_name);
	    }
	}
	fprintf(session, "current workfile is %s\n", fi_name);
	return;
    }
    if (argc != 2)
    {
	error1("wrong number of arguments to FI command");
	return;
    }
    strcpy(fi_name, expand_path(argv[1]));

    if(server_mode == TRUE)
    {
	srv_string("filename", fi_name);
    }
}

/***********************************************************************/
/*                                                                    */
/* WF_OPEN                                                            */
/* Opens an image workfile and returns file descripter.               */
/*    Tries to open for write; if unable, then tries read-only.       */
/*    If failure (no file or no image header) prints error msg and    */
/*    returns negative value.                                         */
/*                                                                    */
/**********************************************************************/

int
wf_open(fname, imgp, give_error_messages)
char fname[];
struct img *imgp;
int give_error_messages;
{
    int file_d;
    char tmp_fname[MAXPATHLEN];

    imgp->frameptr = 0;  /* framestruct not yet allocated */
    strcpy(imgp->filnam, fname);

    strcpy(tmp_fname, fname);
    /* first try to open for write */
    file_d = open(tmp_fname, O_RDWR, 0644);
    if (file_d < 0)
    {
	/* try for read only */
	file_d = open(tmp_fname, O_RDONLY, 0644);
    }
    if (file_d >= 0)
    {
	if (debug)
	    fprintf(debug_file, "successful open on %s\n", tmp_fname);
	file_d = gethdr(file_d, imgp);
	if (file_d < 0)
	    close(-file_d);
	else
	    return(file_d);
    }
    /* first abort on errors other than "file not found" */
    else if (errno != ENOENT)
    {
	if (give_error_messages)
	    sperror("error opening workfile", errno);
	return(file_d);
    }
    /* now try image access routines */
    file_d = im_open(fname, "read");
    if (file_d >= 0)
    {
	if (debug)
	    fprintf(debug_file, "successful im_open on %s\n", tmp_fname);
	imhdr(file_d,imgp);
	return(file_d);
    }

    if (give_error_messages)
    {
	sprintf(server_str, "Unable to open image file %s", tmp_fname);
	error1(server_str);

	if (file_d == -1)
	    if(server_mode == FALSE)
	       printf("   not a proper image file format\n");
    }
    return(file_d);
}


void
delete_file(argc, argv)
int argc;
char *argv[];
{
    char *fname, ans[20];
    int temp_fd;

    if (captive)
    {
	error1("delete file not allowed in captive mode");
	return;
    }

    if (server_mode == TRUE)
    {
	error_na("delete_file not allowed in server mode");
	return;
    }

    if (argc == 2)
    {
	fname = expand_path(argv[1]);

	/* check if file exists and he has write access (stricter than unix) */
	temp_fd = open(fname, O_RDWR | O_NDELAY, 0);
	if (temp_fd < 0)
	{
	    sperror(fname, errno);
	    return;
	}
	close(temp_fd);

	if (cmdfile == stdin)
	{
	   printf("delete file %s? ", fname);
	   fgets(ans, 20, stdin);
	   if (cmd(ans) != YES)
	   {
	       printf("delete_file command aborted\n");
	       return;
	   }
	}
	if (unlink(fname) != 0)
	{
	    sperror(fname, errno);
	}
    }
    else
    {
	error1("wrong number of arguments to delete_file command");
    }
}

/**********************************************************************
**                                                                   ** 
**  GETHDR reads in structure imgp->filehdr, then calls fitshdr or   **
**       vicarhdr to parse the FITS or VICAR header and set 	     **
**       other imgp elements.  					     ** 
**  Returns file descripter, positive for success, negative for fail ** 
**                                                                   ** 
**********************************************************************/ 
 
static int
gethdr(fd, imgp)
int fd;
struct img *imgp;
{
    int rdstat;

    if (debug)
	fprintf(debug_file, "entering gethdr\n");
    lseek(fd, 0L, 0);
    rdstat = read(fd,&imgp->filehdr,sizeof(imgp->filehdr));
    if (rdstat != sizeof(imgp->filehdr))
    {
	return(-fd);
    }
    switch (imgp->filehdr.tapetype)
    {
    case FITS:
	{
	lseek(fd, imgp->filehdr.hdr_start, 0);
	fd = fitshdr(fd, imgp);
	break;
	}
    case VICAR:
	{
	lseek(fd, imgp->filehdr.hdr_start, 0);
	fd = vicarhdr(fd, imgp);
	break;
	}
    default: 
	{
	if (debug)
	    fprintf(debug_file, 
		"gethdr:  bad tapetype = %d\n", imgp->filehdr.tapetype);
	return(-fd);
	/*NOTREACHED*/
	break;
	}
    }
    if (debug)
	fprintf(debug_file, "leaving gethdr\n");
    return(fd);
}

static int
vicarhdr(fd, imgp)
int fd;
struct img *imgp;
{
    char hdr[432], *pt, ctype3[15];
    char mapstring[80];
    int ss, es, sl, el;

    set_header_defaults(imgp);

    if ( read(fd,hdr,432) != 432)
    {
	return(-fd);
    }
    pt = hdr;
    imgp->bitpix = 16;
    imgp->blank = -32768;
    imgp->blank_set = TRUE;
    imgp->naxis3 = 1;
    imgp->lines = atoi(pt+16);
    imgp->pixels = atoi(pt+24) / 2;
    if (memcmp((pt+72), "IRAS", 4) == 0)
    {
	if (debug)
	    fprintf(debug_file, "IRAS VICAR format\n");
	sl = atoi(pt+304);
	ss = atoi(pt+314);
	el = atoi(pt+324);
	es = atoi(pt+334);
	imgp->glat = atof(pt+251);
	imgp->glong = atof(pt+263);
	imgp->cdelt1 = -atof(pt+192);
	if (strncmp(pt+198, "DEGREE", 6) != 0)
	    imgp->cdelt1 = imgp->cdelt1 / 60;
	imgp->bscale = atof(pt+162);
	imgp->b_zero = 0;
	memcpy(imgp->bunit, (pt+170),15);
	imgp->bunit[15] = '\0';
	memcpy(mapstring, (pt+222),8);
	mapstring[8] = '\0';
	imgp->maptype = parse_fits(mapstring);
	memcpy(ctype3, (pt+270),14);
	ctype3[14] = '\0';
	strcpy(imgp->ctype1, "lon");
	strcpy(imgp->ctype1sav, "lon");
	strcpy(imgp->ctype2, "lat");
	strcpy(imgp->ctype2sav, "lat");
	if ((strncmp(ctype3, " (EQUAT", 7) == 0) ||
	    (strncmp(ctype3, " (RTASCDEC", 10) == 0))
	{
	    strcpy(imgp->ctype1, "RA");
	    strcpy(imgp->ctype1sav, "RA");
	    strcpy(imgp->ctype2, "DEC");
	    strcpy(imgp->ctype2sav, "DEC");
	}
	if (strncmp(ctype3, "( GALACTIC", 10) == 0)
	{
	    strcpy(imgp->ctype1, "LON");
	    strcpy(imgp->ctype1sav, "LON");
	    strcpy(imgp->ctype2, "LAT");
	    strcpy(imgp->ctype2sav, "LAT");
	}
	if (strcmp(mapstring, " DEEPSKY" ) == 0)
	{
	    if (debug)
		fprintf(debug_file, "Deepsky plate\n");
	    imgp->maptype = ORTHOGRAPHIC;
	    imgp->twist = atof(pt+374) + 180.0;
	    if (imgp->twist > 360.0)
		imgp->twist = imgp->twist - 360.0;
	    imgp->cdelt2 = -atof(pt+394) / 60.0;
	    if (debug)
		fprintf(debug_file, 
		    "cdelt2=%f twist=%f\n",imgp->cdelt2, imgp->twist);
	}
	else
	{
	    imgp->twist = 0;
	    imgp->cdelt2 = imgp->cdelt1;
	}
    }
    else
    {
	if (debug)
	    fprintf(debug_file, "generic VICAR format\n");
	sl = 0; 
	ss = 0;
	el = imgp->lines - 1;
	es = imgp->pixels - 1; 
	imgp->glat = 0;
	imgp->glong = 0;
	imgp->cdelt1 = 0;
	imgp->cdelt2 = 0;
	imgp->twist = 0;
	imgp->bscale = 1;
	imgp->b_zero = 0;
	strcpy(imgp->bunit, " DN");
	imgp->maptype = UNSPECIFIED;
	strcpy(imgp->ctype1, "lon");
	strcpy(imgp->ctype2, "lat");
	strcpy(imgp->ctype1sav, "lon");
	strcpy(imgp->ctype2sav, "lat");
    }
    imgp->crpix1 = 1 - ss;
    imgp->crpix2 = 1 - sl;
    return(fd);
}

static int
fitshdr(fd, imgp)
int fd;
struct img *imgp;
{
    char hdr[2880], *pt;
    char *ptmp, *quote1, *quote2;
    int rdstat, i, nchars;
    int got_blank = FALSE;
    double rah,ram,ras, dsign,decd,decm,decs;
    double dec_deg,ra_hours;
    double dtr = M_PI/180;
    int got_equinox = FALSE;

    if (debug)
	fprintf(debug_file, "fitshdr: entry\n");

    set_header_defaults(imgp);

    for (;;)
    {
    if ((rdstat = read(fd,hdr,2880)) != 2880)
    {
	if (debug)
	    fprintf(debug_file, "short read %d bytes\n", rdstat);
	return(-fd);
    }

    pt = hdr;
    for (i=0;i<36 ; i++, pt+=80)
    {
	switch(parse_fits(pt))
	{
	    case SIMPLE:
		if ( *(pt+29) != 'T')
		    {
			if(server_mode == FALSE)
			   printf("Warning:  SIMPLE is not T in FITS header\n");
		    }
		break;
	    case BITPIX:
		imgp->bitpix = atoi(pt+10);
		if ((imgp->bitpix != 8) && (imgp->bitpix != 16) &&
		    (imgp->bitpix != 32))
		    return(-fd);  /* die if not a real s workfile */
		break;
	    case NAXIS:
		imgp->naxis = atoi(pt+10);
		if ((imgp->naxis != 2) && (imgp->naxis != 3))
		{
		    if(server_mode == FALSE)
		       printf("warning: naxis is not equal to 2 or 3\n");
		}
		break;
	    case NAXIS1:
		imgp->pixels = atoi(pt+10);
		break;
	    case NAXIS2:
		imgp->lines = atoi(pt+10) ;
		break;
	    case NAXIS3:
		imgp->naxis3 = atoi(pt+10) ;
		break;
	    case BSCALE:
		imgp->bscale = atof(pt+10);
		break;
	    case BZERO:
		imgp->b_zero = atof(pt+10);
		if (strncmp((pt+42), "BIAS LEVEL (GRID REF)", 21) == 0)
		    imgp->base = atof(pt+64);
		break;
	    case BUNIT:
		quote1 = strchr(pt+9, '\047');    /* look for ' */
		if (quote1 != NULL)
		{
		    quote2 = strchr(quote1 + 1, '\047');  /* second ' */
		    if (quote2 != NULL)
		    {
			nchars = quote2 - quote1 - 1;
			if (nchars > 79)
			    nchars = 79;
			memcpy(imgp->bunit, quote1 + 1, nchars);
			imgp->bunit[nchars] = '\0';
		    }
		}
		break;
	    case BLANK:
		imgp->blank = atof(pt+10);
		got_blank = TRUE;
		imgp->blank_set = TRUE;
		break;
	    case CRVAL1:
		imgp->glong = atof(pt+10);
		break;
	    case CRPIX1:
		imgp->crpix1 = atof(pt+10);
		break;
	    case CTYPE1:
		memcpy(imgp->ctype1, pt+11, 8);
		imgp->ctype1[8] = '\0';
		strcpy(imgp->ctype1sav, imgp->ctype1);
		if (strncmp(pt+15, "-TAN", 4) == 0)
		    imgp->maptype = GNOMONIC;
		if (strncmp(pt+15, "-SIN", 4) == 0)
		    imgp->maptype = ORTHOGRAPHIC;
		if (strncmp(pt+15, "-NCP", 4) == 0)
		    imgp->maptype = NCP;
		if (strncmp(pt+15, "-AIT", 4) == 0)
		    imgp->maptype = AITOFF;
		if (strncmp(pt+15, "-ARC", 4) == 0)
		    imgp->maptype = ARC;
		if (strncmp(pt+15, "-CAR", 4) == 0)
		    imgp->maptype = CAR;
		if (strncmp(pt+15, "-CEA", 4) == 0)
		    imgp->maptype = CEA;
		if (strncmp(pt+15, "----", 4) == 0)
		    imgp->maptype = LINEAR;
		if (strncmp(pt+15, "    ", 4) == 0)
		    imgp->maptype = LINEAR;
		if (imgp->maptype == UNSPECIFIED)
		    imgp->maptype = UNRECOGNIZED;
		ptmp = strchr(imgp->ctype1, '-');
		if (ptmp != 0)
		    *ptmp = '\0';   /* change - to EOS */
		ptmp = strchr(imgp->ctype1, ' ');
		if (ptmp != 0)
		    *ptmp = '\0';   /* change space to EOS */
		break;
	    case CDELT1:
		imgp->cdelt1 = atof(pt+10);
		break;
	    case CRVAL2:
		imgp->glat = atof(pt+10);
		break;
	    case CRPIX2:
		imgp->crpix2 = atof(pt+10);
		break;
	    case CTYPE2:
		memcpy(imgp->ctype2, pt+11, 8);
		imgp->ctype2[8] = '\0';
		strcpy(imgp->ctype2sav, imgp->ctype2);
		if (strncmp(pt+15, "-TAN", 4) == 0)
		    imgp->maptype = GNOMONIC;
		if (strncmp(pt+15, "-SIN", 4) == 0)
		    imgp->maptype = ORTHOGRAPHIC;
		if (strncmp(pt+15, "-NCP", 4) == 0)
		    imgp->maptype = NCP;
		if (strncmp(pt+15, "-AIT", 4) == 0)
		    imgp->maptype = AITOFF;
		if (strncmp(pt+15, "-ARC", 4) == 0)
		    imgp->maptype = ARC;
		if (strncmp(pt+15, "-CAR", 4) == 0)
		    imgp->maptype = CAR;
		if (strncmp(pt+15, "-CEA", 4) == 0)
		    imgp->maptype = CEA;
		if (strncmp(pt+15, "----", 4) == 0)
		    imgp->maptype = LINEAR;
		if (strncmp(pt+15, "    ", 4) == 0)
		    imgp->maptype = LINEAR;
		if (imgp->maptype == UNSPECIFIED)
		    imgp->maptype = UNRECOGNIZED;
		ptmp = strchr(imgp->ctype2, '-');
		if (ptmp != 0)
		    *ptmp = '\0';   /* change - to EOS */
		ptmp = strchr(imgp->ctype2, ' ');
		if (ptmp != 0)
		    *ptmp = '\0';   /* change space to EOS */
		break;
	    case CDELT2:
		imgp->cdelt2 = atof(pt+10);
		break;
	    case CROTA2:
		imgp->twist = atof(pt+10); /* not tested - units or direction */
		break;
	    case CRVAL3:
		imgp->crval3 = atof(pt+10);
		break;
	    case DATAMAX:
		imgp->iraf_max = atof(pt+10);
		break;
	    case DATAMIN:
		imgp->iraf_min = atof(pt+10);
		break;
	    case EPOCH:
		/* EQUINOX overrides EPOCH */
		if (!got_equinox)
		    imgp->file_equinox = atof(pt+10);
		break;
	    case EQUINOX:
		imgp->file_equinox = atof(pt+10);
		got_equinox = TRUE;
		break;
	    case RADECSYS:
		memcpy(imgp->radecsys, pt+11, 8);
		imgp->radecsys[8] = '\0';
		break;
	    case PROJTYPE:
		if (use_projtype)
		{
		    if ((imgp->maptype == UNSPECIFIED) || 
			(imgp->maptype == UNRECOGNIZED))
		    {
			imgp->maptype = parse_fits(pt+11);
			if (imgp->maptype != UNRECOGNIZED)
			    if(server_mode == FALSE)
			       printf("using PROJTYPE of %s\n", maptype_string(imgp->maptype));
		    }
		}
		break;
	    case OBJECT:
		memcpy(imgp->object, pt+11, 8);
		imgp->object[8] = '\0';
		break;
	    case DSKYGRID:
		imgp->dskygrid = (int) atof(pt+10);
		imgp->maptype = ORTHOGRAPHIC;
		break;
	    case COMMENT:
		if (strncmp((pt+42), "EST. MEDIAN NOISE", 17) == 0)
		    imgp->noise = atof(pt+61);   /* deepsky noise */
		break;

	    /* now do Digital Sky Survey plate solution cofficients */
	    case PLTRAH:
		rah = atof(pt+10);
		imgp->maptype = PLATE;
		break;
	    case PLTRAM:
		ram = atof(pt+10);
		break;
	    case PLTRAS:
		ras = atof(pt+10);
		break;
	    case PLTDECSN:
		if (*(pt+11) == '-')
		    dsign = -1.;
		else
		    dsign = 1.;
		break;
	    case PLTDECD:
		decd = atof(pt+10);
		break;
	    case PLTDECM:
		decm = atof(pt+10);
		break;
	    case PLTDECS:
		decs = atof(pt+10);
		break;
	    case CNPIX1:
		imgp->x_pixel_offset = atof(pt+10);
		break;
	    case CNPIX2:
		imgp->y_pixel_offset = atof(pt+10);
		break;
	    case PLTSCALE:
		imgp->plt_scale = atof(pt+10);
		break;
	    case XPIXELSZ:
		imgp->x_pixel_size= atof(pt+10);
		break;
	    case YPIXELSZ:
		imgp->y_pixel_size= atof(pt+10);
		break;
	    case PPO1:
		imgp->ppo_coeff[0] = atof(pt+10);
		break;
	    case PPO2:
		imgp->ppo_coeff[1] = atof(pt+10);
		break;
	    case PPO3:
		imgp->ppo_coeff[2] = atof(pt+10);
		break;
	    case PPO4:
		imgp->ppo_coeff[3] = atof(pt+10);
		break;
	    case PPO5:
		imgp->ppo_coeff[4] = atof(pt+10);
		break;
	    case PPO6:
		imgp->ppo_coeff[5] = atof(pt+10);
		break;

	    case AMDX1:
		imgp->amd_x_coeff[0] = atof(pt+10);
		break;
	    case AMDX2:
		imgp->amd_x_coeff[1] = atof(pt+10);
		break;
	    case AMDX3:
		imgp->amd_x_coeff[2] = atof(pt+10);
		break;
	    case AMDX4:
		imgp->amd_x_coeff[3] = atof(pt+10);
		break;
	    case AMDX5:
		imgp->amd_x_coeff[4] = atof(pt+10);
		break;
	    case AMDX6:
		imgp->amd_x_coeff[5] = atof(pt+10);
		break;
	    case AMDX7:
		imgp->amd_x_coeff[6] = atof(pt+10);
		break;
	    case AMDX8:
		imgp->amd_x_coeff[7] = atof(pt+10);
		break;
	    case AMDX9:
		imgp->amd_x_coeff[8] = atof(pt+10);
		break;
	    case AMDX10:
		imgp->amd_x_coeff[9] = atof(pt+10);
		break;
	    case AMDX11:
		imgp->amd_x_coeff[10] = atof(pt+10);
		break;
	    case AMDX12:
		imgp->amd_x_coeff[11] = atof(pt+10);
		break;
	    case AMDX13:
		imgp->amd_x_coeff[12] = atof(pt+10);
		break;
	    case AMDX14:
		imgp->amd_x_coeff[13] = atof(pt+10);
		break;
	    case AMDX15:
		imgp->amd_x_coeff[14] = atof(pt+10);
		break;
	    case AMDX16:
		imgp->amd_x_coeff[15] = atof(pt+10);
		break;
	    case AMDX17:
		imgp->amd_x_coeff[16] = atof(pt+10);
		break;
	    case AMDX18:
		imgp->amd_x_coeff[17] = atof(pt+10);
		break;
	    case AMDX19:
		imgp->amd_x_coeff[18] = atof(pt+10);
		break;
	    case AMDX20:
		imgp->amd_x_coeff[19] = atof(pt+10);
		break;

	    case AMDY1:
		imgp->amd_y_coeff[0] = atof(pt+10);
		break;
	    case AMDY2:
		imgp->amd_y_coeff[1] = atof(pt+10);
		break;
	    case AMDY3:
		imgp->amd_y_coeff[2] = atof(pt+10);
		break;
	    case AMDY4:
		imgp->amd_y_coeff[3] = atof(pt+10);
		break;
	    case AMDY5:
		imgp->amd_y_coeff[4] = atof(pt+10);
		break;
	    case AMDY6:
		imgp->amd_y_coeff[5] = atof(pt+10);
		break;
	    case AMDY7:
		imgp->amd_y_coeff[6] = atof(pt+10);
		break;
	    case AMDY8:
		imgp->amd_y_coeff[7] = atof(pt+10);
		break;
	    case AMDY9:
		imgp->amd_y_coeff[8] = atof(pt+10);
		break;
	    case AMDY10:
		imgp->amd_y_coeff[9] = atof(pt+10);
		break;
	    case AMDY11:
		imgp->amd_y_coeff[10] = atof(pt+10);
		break;
	    case AMDY12:
		imgp->amd_y_coeff[11] = atof(pt+10);
		break;
	    case AMDY13:
		imgp->amd_y_coeff[12] = atof(pt+10);
		break;
	    case AMDY14:
		imgp->amd_y_coeff[13] = atof(pt+10);
		break;
	    case AMDY15:
		imgp->amd_y_coeff[14] = atof(pt+10);
		break;
	    case AMDY16:
		imgp->amd_y_coeff[15] = atof(pt+10);
		break;
	    case AMDY17:
		imgp->amd_y_coeff[16] = atof(pt+10);
		break;
	    case AMDY18:
		imgp->amd_y_coeff[17] = atof(pt+10);
		break;
	    case AMDY19:
		imgp->amd_y_coeff[18] = atof(pt+10);
		break;
	    case AMDY20:
		imgp->amd_y_coeff[19] = atof(pt+10);
		break;
	    case END:
		if (imgp->maptype == PLATE)
		{
		    ra_hours = rah + (ram/(double)60.0) + (ras/(double)3600.0);
		    imgp->plate_ra = ra_hours * 15.0 * dtr;
		    dec_deg = dsign * (decd+(decm/(double)60.0)+(decs/(double)3600.0));
		    imgp->plate_dec = dec_deg * dtr;
		    if (imgp->cdelt1 == 0.0)
		    {
			imgp->cdelt1 = - imgp->plt_scale * 
			    imgp->x_pixel_size / 1000 / 3600;
			imgp->cdelt2 = imgp->plt_scale * 
			    imgp->y_pixel_size / 1000 / 3600;
		    }

		    if (debug)
		    {
			fprintf(debug_file,
			"WCSINIT Plate center RA=%2.0f:%2.0f:%5.3f, Dec=%3.0f:%2.0f:%5.3f\n",
			rah,ram,ras,dsign*decd,decm,decs);
		    }
		}
		if (imgp->iraf_max == 0)
		    imgp->iraf_max = imgp->filehdr.img_max;
		else
		    imgp->iraf_max = (imgp->iraf_max - imgp->b_zero) / imgp->bscale;
		if (imgp->iraf_min == 0)
		    imgp->iraf_min = imgp->filehdr.img_min;
		else
		    imgp->iraf_min = (imgp->iraf_min - imgp->b_zero) / imgp->bscale;
		if (imgp->maptype == UNRECOGNIZED)
		    if(server_mode == FALSE)
		       printf("Warning:  projection type unrecognized\n");
		if ((got_blank == FALSE) && (imgp->blank_set))
		{
		    if (server_mode == FALSE)
		    {
			printf(
			"warning:  using BLANK value of %g from 'set' command\n"
			    , imgp->blank);
		    }
		}
		return(fd);
	}
    }
    }
}

void
set_header_defaults(imgp)
struct img *imgp;
{
    imgp->pixels = 0;
    imgp->lines = 0;
    imgp->naxis3 = 1;
    imgp->twist = 0;
    imgp->crpix1 = 0;
    imgp->crpix2 = 0;
    imgp->glong = 0;
    imgp->glat = 0;
    imgp->cdelt1 = 0;
    imgp->cdelt2 = 0;
    imgp->using_cd = FALSE;
    imgp->cd1_1 = 0;
    imgp->cd1_2 = 0;
    imgp->cd2_1 = 0;
    imgp->cd2_2 = 0;
    imgp->dc1_1 = 0;
    imgp->dc1_2 = 0;
    imgp->dc2_1 = 0;
    imgp->dc2_2 = 0;
    imgp->bscale = 1;
    imgp->b_zero = 0;
    imgp->blank_set = default_blank_set;
    if (imgp->blank_set)
	imgp->blank = default_blank;
    else
	imgp->blank = blank_flag;
    imgp->u16 = default_u16;
    imgp->ctype1[0] = '\0';
    imgp->ctype1sav[0] = '\0';
    imgp->ctype2[0] = '\0';
    imgp->ctype2sav[0] = '\0';
    imgp->object[0] = '\0';
    strcpy(imgp->bunit, " DN");
    imgp->maptype = UNSPECIFIED;
    imgp->map_distortion = FALSE;
    imgp->file_equinox = 1950.0;
    imgp->radecsys[0] = '\0';
    imgp->iraf_max = 0.0;
    imgp->iraf_min = 0.0;
    imgp->band_offset = reqband - 1;
    imgp->imio_ok = FALSE;
    imgp->mod_filnam[0] = '\0';
    imgp->frameptr = 0;
    imgp->graphics_window = 0;
}

void
set_history(argc, argv)
int argc;
char *argv[];
{

    time_t timenow;
    int create;
    int i, start;
    char fname[MAXPATHLEN], *ct;

    start = TRUE;
    create = FALSE;

    for (i=1; i<argc; i++)
    {
	switch(cmd(argv[i]))
	{
	    case CREATE:
		create = TRUE;
		break;
	    case OF:
		start = FALSE;
		break;
	    case ONN:
		start = TRUE;
		break;
	    default:    /* its a filename */
		strcpy(fnamsav, argv[i]);
		break;
	}
    }

    fclose(session);   /* close old session file (may be /dev/null) */

    if (start)
    {
	if (*fnamsav == '/')
	{
	    strcpy(fname, fnamsav);
	}
	else
	{
	    strcpy(fname, org_path);
	    strcat(fname, fnamsav); 
	}
	if (create)
	    session = fopen(fname, "w"); 
	else
	    session = fopen(fname, "a"); 
	if (session == NULL)
	{
	    sprintf(server_str,
		"cannot open session history file %s", fname);
	    error1(server_str);

	    session = fopen("/dev/null", "w");
	    session_on = FALSE;
	}
	else
	{
	    session_on = TRUE;
	    if(server_mode == FALSE)
	       printf("begin logging to %s\n", fname);
	    else
	    {
		srv_string("filename", fname);
	    }
	    timenow = time((time_t *)NULL);
	    ct = ctime(&timenow);
	    fprintf(session, "Skyflux session history starting at %s", ct);
	}
    }
    else
    {
	/* stop session logging */
	session = fopen("/dev/null", "w");
	session_on = FALSE;
    }
}

void
print_history(argc, argv)
int argc;
char *argv[];
{
    FILE *tempfd;
    char fname[MAXPATHLEN], sysstr[75];

    if (server_mode == TRUE)
    {
	error_na("print_history not allowed in server mode");
	return;
    }

    /* construct history filename */
    if (*fnamsav == '/')
    {
	strcpy(fname, fnamsav);
    }
    else
    {
	strcpy(fname, org_path);
	strcat(fname, fnamsav); 
    }

    if (session_on)      /* do we have an open history file */
    {
	/* its open, close it */
	fclose(session);
	/* note: session_on is not reset  */
    }
    else
    {
	/* not open, see if file exists */
	tempfd = fopen(fname, "r"); 
	if (tempfd == NULL)
	{
	    sprintf(server_str,
		"cannot open session history file %s", fname);
	    error1(server_str);
	    return;
	}
	fclose(tempfd);
    }
    strcpy(sysstr, "lpr ");
    if (argc > 1)
    {
	if ((*argv[1] == 't') || (*argv[1] == 'T'))
	    strcpy(sysstr, "cat ");   /* overwrite lpr with cat */
    }
    strcat(sysstr, fname);
    system(sysstr);

    if (session_on)
	session = fopen(fname, "a"); /* reopen history file */

}

void
crop(argc, argv)
int argc;
char *argv[];
{
    float *runp;
    double cnpix1, cnpix2;
    double hist_datamax, hist_datamin;
    double d_runval, *d_runp;
    int val, dn_min, dn_max;
    int ri;
    char *rpc;
    short *rp;
    unsigned short *rpu;
    int *rpl;
    double local_blank;
    time_t timenow;
    char *from_ptr, *p, *ct;
    int totpixels, bitpix1, imstat;
    char *pixel_ptr, history_line[80];
    int i, samples, line;
    int min_samp, max_samp, min_line, max_line;
    char *ppp, buf[2880], *fil_name;
    int disp;
    int bytes, bytes_per_pixel;
    int fd, fd1, fd_in[2];
    int hist_shift, max;
    int hist[HISTSIZ2+1];
    int blocks, rdstat, hdr_bytes, bufsize;
    struct img tmpimg;
    struct hdrstruct thdr;

    if(argc < 2)
    {
	error1("crop command needs a filename");
	return;
    }

    if (argc == 3)  /* if he specified an area number */
    {
	if (!switch_area(argv[1]))
	    return;
    }

    if(debug)
	fprintf(debug_file, "crop:  area number %d \n", area_number);

    if (area[area_number].area_type == 0)
    {
	sprintf(server_str,
	    "crop error: area %d has not been defined", area_number);
	error1(server_str);
	return;
    }
    find_range(&min_samp, &max_samp, &min_line, &max_line);
    /* samp and line are in IRAF coordinate system */
    if (debug)
	fprintf(debug_file, 
	    "min_samp=%d max_samp=%d min_line=%d max_line=%d\n", 
	    min_samp, max_samp, min_line, max_line);

    samples = max_samp - min_samp + 1;

    /* open input file */
    fd = wf_open(filnam, &tmpimg, 1);
    if (fd < 0)
    {
	error1("crop error:  original file has disappeared");
	return;
    }
    if ((tmpimg.naxis > 2) && (tmpimg.naxis3 > 1))
    {
	error1("crop error:  cannot crop a multiplane image");
	return;
    }

    /* get output filename */
    fil_name = expand_path(argv[argc - 1]);

    if (strcmp(fil_name, filnam) == 0)
    {
	error1("crop error:  output file may not be same as input file");
	return;
    }

    if (tmpimg.filehdr.tapetype == IMFILE)
    {
	im_rkey_i(fd, &totpixels, "npix");
	im_rkey_i(fd, &bitpix1, "BITPIX");
	if (mod_filnam[0] == '\0')
	    pixel_ptr = (char *) imget(fd, filnam);  /* get the orig pixels */
	else
	    pixel_ptr = (char *) imget(-1, mod_filnam); /* get the modified pixels */
	if (pixel_ptr == NULL)
	{
	    error1("unable to read pixels from old file");
	    im_close(fd);
	    return;
	}

	fd1 = im_open(fil_name, "w");
	if (fd1 < 0)
	{
	    error1("Cannot open output file");
	    im_close(fd);
	    return;
	}
	fd_in[0] = fd;
	fd_in[1] = 0;
	timenow = time((time_t *) NULL);
	ct = ctime(&timenow);
	sprintf(history_line, "WRITTEN BY SKYVIEW CROP COMMAND (%s", ct);
	p = strchr(history_line, '\n');
	if (p != 0)
	    *p = ')';
	im_history_line = history_line;
	imstat = im_hdr(fd1, fd_in, 0);
	im_history_line = (char *)0;

	imstat = im_wkey_i(fd1, samples, "NAXIS1");
	imstat = im_wkey_i(fd1, max_line - min_line + 1, "NAXIS2");
	imstat = im_wkey_d(fd1, (double) 0.0, "DATAMIN");  /* place holder */
	imstat = im_wkey_d(fd1, (double) 0.0, "DATAMAX");  /* place holder */

	if (tmpimg.maptype == PLATE)
	{
	    cnpix1 = tmpimg.x_pixel_offset + (min_samp - 1);
	    cnpix2 = tmpimg.y_pixel_offset + (min_line - 1);
	    imstat = im_wkey_d(fd1, cnpix1, "CNPIX1");
	    imstat = im_wkey_d(fd1, cnpix2, "CNPIX2");
	}
	else
	{
	    imstat = im_wkey_d(fd1, (double) (1 - min_samp + crpix1), "CRPIX1");
	    imstat = im_wkey_d(fd1, (double) (1 - min_line + crpix2), "CRPIX2");
	}

    if (bitpix > 0)
    {
	dn_max = 0x80000001;
	dn_min = 0x7fffffff;
	if ((bitpix == 16) && (u16))
	    dn_max = 0;
	if (FINITE(blank))
	{
	    local_blank = blank;
	}
	else
	{
	    local_blank = 1e+30; /* impossible value - (NaN causes traps) */
	}
    }
    else
    {
	hist_datamax = -DBL_MAX;
	hist_datamin = DBL_MAX;
    }

    for (line = min_line; line <= max_line; line++)
    {
	disp = (min_samp - 1) + (line - 1) * pixels;
	ri = samples;
	switch(bitpix1)
	{
	    case 8:
		from_ptr = pixel_ptr + disp;
		rpc = (char *) from_ptr;
		while (ri > 0)
		{
		    val = *rpc & 0xff;
		    if (val != local_blank)
		    {
			if (val > dn_max)
			    dn_max = val;
			if (val < dn_min)
			    dn_min = val;
		    }
		    rpc++;
		    ri--;
		}
		imstat = im_wpix_c(fd1, (unsigned char *) from_ptr, samples);
		break;
	    case 16:
		from_ptr = pixel_ptr + disp * 2;
		if (u16)
		{
		    rpu = (unsigned short *) from_ptr;
		    while (ri > 0)
		    {
			if (*rpu != local_blank)
			{
			    val = (int) *rpu;
			    if (val > dn_max)
				dn_max = val;
			    if (val < dn_min)
				dn_min = val;
			}
			rpu++;
			ri--;
		    }
		}
		else
		{
		    rp = (short *) from_ptr;
		    while (ri > 0)
		    {
			if (*rp != local_blank)
			{
			    if (*rp > dn_max)
				dn_max = *rp;
			    if (*rp < dn_min)
				dn_min = *rp;
			}
			rp++;
			ri--;
		    }
		}
		imstat = im_wpix_s(fd1, (short *) from_ptr, samples);
		break;
	    case 32:
		from_ptr = pixel_ptr + disp * 4;
		rpl = (int *) from_ptr;   /* 32 bit pointer */
		while (ri > 0)
		{
		    if (*rpl != local_blank)
		    {
			if (*rpl > dn_max)
			    dn_max = *rpl;
			if (*rpl < dn_min)
			    dn_min = *rpl;
		    }
		    rpl++;    /* advance 32 bits */
		    ri--;
		}
		imstat = im_wpix_i(fd1, (int *) from_ptr, samples);
		break;
	    case -32:
		from_ptr = pixel_ptr + disp * 4;
		runp = (float *) from_ptr;
		while (ri > 0)
		{
		    d_runval = *runp++;
		    if (FINITE(d_runval))
		    {
			if (d_runval < hist_datamin) 
			    hist_datamin = d_runval;
			if (d_runval > hist_datamax) 
			    hist_datamax = d_runval;
		    }
		    runp++;
		    ri--;
		}
		imstat = im_wpix_r(fd1, (float *) from_ptr, samples);
		break;
	    case -64:
		from_ptr = pixel_ptr + disp * 8;
		d_runp = (double *) from_ptr;
		while (ri > 0)
		{
		    d_runval = *d_runp++;
		    if (FINITE(d_runval))
		    {
			if (d_runval < hist_datamin) 
			    hist_datamin = d_runval;
			if (d_runval > hist_datamax) 
			    hist_datamax = d_runval;
		    }
		    d_runp++;
		    ri--;
		}
		imstat = im_wpix_d(fd1, (double *) from_ptr, samples);
		break;
	}
	/* due to the delayed open in image access, a bad status here       */
	/* may mean that the file was never opened.  In this case the fid   */
	/* inside image access is 0.  Then im_close() will close file 0     */
	/* Thus, we only close the file if the status is good.              */
	if (imstat != samples)
	{
	    error1("error writing pixels to new file");
	    im_close(fd);
	    return;
	}
    }

	if (bitpix > 0)
	{
	    imstat = im_wkey_d(fd1, dn_min * bscale + b_zero, "DATAMIN");
	    imstat = im_wkey_d(fd1, dn_max * bscale + b_zero, "DATAMAX");
	}
	else
	{
	    imstat = im_wkey_d(fd1, hist_datamin, "DATAMIN");
	    imstat = im_wkey_d(fd1, hist_datamax, "DATAMAX");
	}

	im_close(fd);
	im_close(fd1);
    }
    else
    {

    /* open output file */
    fd1 = open(fil_name, O_RDWR | O_CREAT | O_TRUNC, 0644);
    if (fd1 < 0)
    {
	sperror(fil_name, errno);
	return;
    }

    for (i=0; i<HISTSIZ2+1; i++)
	hist[i] = 0;

    /* first transfer s header */

    lseek(fd, 0L, 0);
    rdstat = read(fd, &thdr, sizeof(thdr));

    if (thdr.tapetype != FITS)
    {
	/* it's an old VICAR file */
	error1("error: only FITS type files can be cropped");
	return;
    }

    /* it is from a FITS tape */

    if (thdr.img_max > (-thdr.img_min))
	max = thdr.img_max;
    else
	max = -thdr.img_min;
    hist_shift = 0;
    while ((max >> hist_shift) > HISTSIZ)
	++hist_shift;

    thdr.tapetype = INVALID;  /* mark workfile invalid during transfer */
    thdr.img_max = 0x80000001;
    thdr.img_min = 0x7fffffff;
    write(fd1, &thdr, sizeof(thdr));

    /* now read old FITS header */
    bufsize = (thdr.img_start - thdr.hdr_start) / 2880; /* number of blocks */
    bufsize += 2; /* room for more */
    bufsize = bufsize * 2880;
    hdr_buf = malloc(bufsize);
    if (hdr_buf == 0)
    {
	error1("error allocating space for crop");
	return;
    }

    lseek(fd, thdr.hdr_start, 0);
    read (fd, hdr_buf, (int) (thdr.img_start - thdr.hdr_start));

    /* find END keyword */
    if (hdrinit(bufsize) == FALSE)
    {
	error1("crop error: no END in original FITS header");
	return;
    }
    /* fill in header with old values to reserve space in file */
    hdrint("NAXIS1", samples);
    hdrint("NAXIS2", max_line - min_line + 1);
    hdrfloat( "CRPIX1", (double) (1 - min_samp + crpix1));
    hdrfloat( "CRPIX2", (double) (1 - min_line + crpix2));
    /* the next two values will change, but need to reserve space */
    hdrfloat("DATAMIN", thdr.img_min * bscale + b_zero);
    hdrfloat("DATAMAX", thdr.img_max * bscale + b_zero);

    blocks = (hdr_bufp_end - hdr_buf) / 2880;
    if ((blocks * 2880) != (hdr_bufp_end - hdr_buf))
	blocks++;  /* round up to next integral number of blocks */
    hdr_bytes = blocks * 2880;
    ppp = hdr_bufp_end;
    while (ppp < hdr_buf + hdr_bytes)
	*ppp++ = ' ';                /* pad out with spaces */
    write(fd1, hdr_buf, hdr_bytes);

    thdr.img_start = lseek(fd1, 0L, 1);

    bytes_per_pixel = bitpix / 8;
    bytes = samples * bytes_per_pixel;

    for (line = min_line; line <= max_line; line++)
    {
	disp = (min_samp - 1) + (line - 1) * pixels;
	disp = disp * bytes_per_pixel;
	lseek(fd, disp + filehdr.img_start, 0);
	rdstat = read(fd, buf, bytes);
	if (rdstat == bytes)
	{
	    write(fd1, buf, bytes);
	    /* compute histogram */
	    get_hist(bitpix, (void *) buf, samples, &thdr.img_min, 
		&thdr.img_max, blank, hist_shift, hist);
	}
	else
	{
	    error1("cannot read workfile");
	}
    }
    thdr.hist_start = lseek(fd1, 0L, 1);
    write(fd1, hist, sizeof(hist));
    thdr.hist_bin_size = 1 << hist_shift;

    hdrint("NAXIS1", samples);
    hdrint("NAXIS2", max_line - min_line + 1);
    hdrfloat( "CRPIX1", (double) (1 - min_samp + crpix1));
    hdrfloat( "CRPIX2", (double) (1 - min_line + crpix2));
    hdrfloat("DATAMIN", thdr.img_min * bscale + b_zero);
    hdrfloat("DATAMAX", thdr.img_max * bscale + b_zero);
    lseek(fd1, thdr.hdr_start, 0);
    write(fd1, hdr_buf, hdr_bytes);
    free(hdr_buf);

    thdr.tapetype = FITS;
    lseek(fd1, 0L, 0);
    write(fd1, &thdr, sizeof(thdr));
    close(fd);
    close(fd1);
    }
    if (server_mode == TRUE)
	srv_string("filename", fil_name);
}

void
split(argc, argv)
int argc;
char *argv[];
{
    int plane, i, line;
    char strplane[5], *ppp, buf[2880], fil_name[MAXPATHLEN];
    int bytes, bytes_per_pixel;
    int fd, fd1;
    int hist_shift, max;
    int hist[HISTSIZ2+1];
    int blocks, rdstat, hdr_bytes, bufsize;
    struct img tmpimg;
    struct hdrstruct thdr;


    if (argc > 1)
    {
	/* must be a filename */
	if (*argv[1] == '/')
	{
	    strcpy(fi_name, argv[1]);
	}
	else
	{
	    strcpy(fi_name, pathnam);
	    strcat(fi_name, argv[1]);
	}
    }

    if (fi_name[0] == '\0')
    {
	error1("need a filename");
	return;
    }

    /* open input file */
    fd = wf_open(fi_name, &tmpimg, 1);
    if (fd < 0)
    {
	sperror(fi_name, errno);    /* bad stuff - file has disappeared */
	return;
    }
    if (tmpimg.filehdr.tapetype == IMFILE)
    {
	error1("Cannot split non-s files");
	im_close(fd);
	return;
    }

    if (tmpimg.naxis3 <= 1)
    {
	error1("split error:  image file does not have multiple planes");
	return;
    }

    /* first read s header */

    lseek(fd, 0L, 0);
    rdstat = read(fd, &thdr, sizeof(thdr));

    if (thdr.tapetype != FITS)
    {
	error1("split error: header must begin with SIMPLE");
	close(fd);
	return;
    }

    if (thdr.img_max > (-thdr.img_min))
	max = thdr.img_max;
    else
	max = -thdr.img_min;
    hist_shift = 0;
    while ((max >> hist_shift) > HISTSIZ)
	++hist_shift;

    /* now read old FITS header */
    bufsize = (thdr.img_start - thdr.hdr_start) / 2880; /* number of blocks */
    bufsize += 2; /* room for more */
    bufsize = bufsize * 2880;
    hdr_buf = malloc(bufsize);
    if (hdr_buf == 0)
    {
	error1("error allocating space for split");
	return;
    }

    lseek(fd, thdr.hdr_start, 0);
    read (fd, hdr_buf, (int) (thdr.img_start - thdr.hdr_start));

    /* find END keyword */
    if (hdrinit(bufsize) == FALSE)
    {
	error1("split error: no END in original FITS header");
	return;
    }

    hdrint("NAXIS3", 1);
    /* the next two values will change, but need to reserve space */
    hdrfloat("DATAMIN", 0.0);
    hdrfloat("DATAMAX", 0.0);

    blocks = (hdr_bufp_end - hdr_buf) / 2880;
    if ((blocks * 2880) != (hdr_bufp_end - hdr_buf))
	blocks++;  /* round up to next integral number of blocks */
    hdr_bytes = blocks * 2880;
    ppp = hdr_bufp_end;
    while (ppp < hdr_buf + hdr_bytes)
	*ppp++ = ' ';                /* pad out with spaces */

    bytes_per_pixel = tmpimg.bitpix / 8;
    bytes = tmpimg.pixels * bytes_per_pixel;


    for (plane = 1; plane <= tmpimg.naxis3; plane++)
    {

    /* open output file */
    strcpy(fil_name, fi_name);
    strcat(fil_name, ".pl");
    sprintf(strplane, "%d", plane);
    strcat(fil_name, strplane);


    fd1 = open(fil_name, O_RDWR | O_CREAT | O_TRUNC, 0644);
    if (fd1 < 0)
    {
	sperror(fil_name, errno);
	return;
    }

    for (i=0; i<HISTSIZ2+1; i++)
	hist[i] = 0;
    thdr.img_max = 0x80000001;
    thdr.img_min = 0x7fffffff;

    /* first write s header */
    write(fd1, &thdr, sizeof(thdr));

    /* now write old FITS header */
    write(fd1, hdr_buf, hdr_bytes);

    thdr.img_start = lseek(fd1, 0L, 1);

    for (line = 0; line < tmpimg.lines; line++)
    {
	rdstat = read(fd, buf, bytes);
	if (rdstat == bytes)
	{
	    write(fd1, buf, bytes);
	    /* compute histogram */
	    get_hist(tmpimg.bitpix, (void *) buf, tmpimg.pixels, 
		&thdr.img_min, &thdr.img_max, blank, hist_shift, hist);
	}
	else
	{
	    error1("cannot read workfile");
	    close(fd1);
	    close(fd);
	    return;
	}
    }
    thdr.hist_start = lseek(fd1, 0L, 1);
    write(fd1, hist, sizeof(hist));
    thdr.hist_bin_size = 1 << hist_shift;

    hdrfloat("DATAMIN", thdr.img_min * tmpimg.bscale + tmpimg.b_zero);
    hdrfloat("DATAMAX", thdr.img_max * tmpimg.bscale + tmpimg.b_zero);
    lseek(fd1, thdr.hdr_start, 0);
    write(fd1, hdr_buf, hdr_bytes);

    thdr.tapetype = FITS;
    lseek(fd1, 0L, 0);
    write(fd1, &thdr, sizeof(thdr));
    close(fd1);
    }
    free(hdr_buf);
    close(fd);
}


#ifdef NOTDEF 

/*ARGSUSED*/
char *path_base(addon)  /* stuck here to avoid linking with par.a */
char *addon;
{
    static char ret_file[120] = "";
    return(ret_file);
}

#endif /* NOTDEF */




char *maptype_string(m)
int m;
{
    switch(m)
    {
	case GNOMONIC:
	    return("GNOMONIC");
	case ORTHOGRAPHIC:
	    return("ORTHOGRAPHIC");
	case NCP:
	    return("NCP");
	case AITOFF:
	    return("AITOFF");
	case LAMBECYL:
	    return("LAMBECYL");
	case LINEAR:
	    return("LINEAR");
	case PLATE:
	    return("PLATE");
	case ARC:
	    return("ARC");
	case CAR:
	    return("CAR");
	case CEA:
	    return("CEA");
	case SFL:
	    return("SFL");
	case UNSPECIFIED:
	    return("UNSPECIFIED");
	case UNRECOGNIZED:
	    return("UNRECOGNIZED");
	default:
	    sprintf(server_str,
		"ERROR in maptype_string - illegal maptype = %d\n", m);
	    error1(server_str);
	    return("");
    }
}
