/*
 * Copyright(c) 1986 Association of Universities for Research in Astronomy
 * Inc.
 */
/*
Copyright(c) 1986 Association of Universities for Research in Astronomy Inc.

The IRAF software is publicly available, but is NOT in the public domain.
The difference is that copyrights granting rights for unrestricted use and
redistribution have been placed on all of the software to identify its authors.
You are allowed and encouraged to take this software and use it as you wish,
subject to the restrictions outlined below.

Permission to use, copy, modify, and distribute this software and its
documentation is hereby granted without fee, provided that the above copyright
notice appear in all copies and that both that copyright notice and this
permission notice appear in supporting documentation, and that references to
the Association of Universities for Research in Astronomy Inc. (AURA),
the National Optical Astronomy Observatories (NOAO), or the Image Reduction
and Analysis Facility (IRAF) not be used in advertising or publicity
pertaining to distribution of the software without specific, written prior
permission from NOAO.  NOAO makes no representations about the suitability
of this software for any purpose.  It is provided "as is" without express or
implied warranty.

NOAO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL NOAO
BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/



#define _XOPEN_SOURCE
#include <X11/Xlib.h>  /* includes sys/types.h */
#include <X11/Xutil.h>
#include <X11/Xos.h> /* includes sys/types.h strings.h sys/file.h sys/time.h */
#include "skyview.h"
#include "img_x.h"
#include "img.h"
#include "job.h"
#include "parse.h"
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

extern struct framestruct *frameptr;
extern char screen_hist_filnam[], fi_name[], filnam[], server_str[];
extern int screen_hist[], debug_val, imio_on, frame;
extern struct img *curr_img;
extern struct hdrstruct filehdr;
extern double crpix1, crpix2, cdelt1, cdelt2;
extern int minx_next, miny_next, req_mode, mode_used;
extern int bitpix, minx, maxx, miny, maxy, band_offset;
extern int x_offset, sgn_y, y_offset, samp_disp, line_disp, x_orig, y_orig;
extern FILE *debug_file;

extern int pixels, lines;
extern double slow, shigh;
extern int naxis, naxis3;
extern Display *display;
extern int screen;
extern Window win;


#ifdef _NO_PROTO
    static void do_iraf_cmd();
    static int set_reference_frame();
    static decode_frameno();
    static void gio_retcursorval();
    static void get_fbconfig();
    static void set_fbconfig();
    static void get_transformation();
#else
    static void do_iraf_cmd(void);
    static int set_reference_frame(int);
    static int decode_frameno(int);
    static void gio_retcursorval();
    static void get_fbconfig(void);
    static void set_fbconfig(int);
    static void get_transformation(char *);
#endif /* _NO_PROTO */

#define	I_DEVNAME	"/dev/imt1o"
#define	O_DEVNAME	"/dev/imt1i"
#define	OLD_DEVNAME	"/dev/imt1"
#define	IO_TIMEOUT	30
#define	SZ_FIFOBUF	4000

#define	MEMORY		01	/* frame buffer i/o		 */
#define	LUT		02	/* lut i/o			 */
#define	FEEDBACK	05	/* used for frame clears	 */
#define	ZOOMPAN		014	/* not used              	 */
#define	IMCURSOR	020	/* logical image cursor		 */
#define	WCS		021	/* used to set WCS		 */

#define	SZ_IMCURVAL	160
#define	PACKED		0040000
#define	COMMAND		0100000
#define	IIS_READ	0100000
#define	IMC_SAMPLE	0040000

struct iism70
{
    short           tid;
    short           thingct;
    short           subunit;
    short           checksum;
    short           x, y, z;
    short           t;
};

#ifndef min
#define min(a,b)	((a)<(b)?(a):(b))
#endif
#ifndef max
#define max(a,b)	((a)<(b)?(b):(a))
#endif

#define SZ_LINE                 256
#define MAX_FBCONFIG            128     /* max possible frame buf sizes */
#define DEF_NFRAMES             1       /* save memory; only one frame  */
#define DEF_FRAME_WIDTH         512     /* 512 square frame             */
#define DEF_FRAME_HEIGHT        512     /* 512 square frame             */
#define FBCONFIG_1      ".imtoolrc"
#define FBCONFIG_2      "/usr/local/lib/imtoolrc"
#define FBCONFIG_ENV1   "imtoolrc"
#define FBCONFIG_ENV2   "IMTOOLRC"

static int fbconfig = 0; /* current frame buffer config (1st col of imtoolrc) */
int imio_config = -1;
int imio_frame = -1;
int imio_ok = FALSE;
char *imio_filptr;
int imio_wants_char = FALSE;

static int      inpipe, outpipe = 0;
static int      Fb_width = 512, Fb_height = 512;
char     wcsbuf[SZ_WCSBUF];
static double a, b, c, d, tx, ty;
static int cursor_wcs;
int imio_last_x, imio_last_y;
static Window root;
static int root_x, root_y;

static struct job_info *job = NULL;


void
start_imio()
{
    void imio_callback();
    int dummy;

    if (job != NULL)
    {
	return;
    }

    /* open the output pipe */
    /* first open it for read, so that the open for write doesn't fail */
    dummy = open(O_DEVNAME, O_RDONLY | O_NDELAY);
    if (dummy != -1)
    {
	outpipe = open(O_DEVNAME, O_WRONLY | O_NDELAY);
	if (outpipe == -1)
	{
	    sprintf(server_str, 
		"imio failure:  cannot open %s - IRAF imio mode cancelled", 
		O_DEVNAME);
	    error1(server_str);
	    imio_on = 0;  /* shut off this mode */
	    return;
	}
	fcntl(outpipe, F_SETFL, O_WRONLY);  /* clear the O_NDELAY */
	close(dummy); /* it was opened only to prevent failure of O_WRONLY open */
    }

    /* now the input pipe */
    inpipe = open(I_DEVNAME, O_RDWR | O_NDELAY);
    if (inpipe == -1)
    {
	/* no good - try the old fifo name */
	inpipe = open(OLD_DEVNAME, O_RDWR | O_NDELAY);
	if (inpipe == -1)
	{
	    sprintf(server_str, 
		"imio failure:  cannot open %s - IRAF imio mode cancelled", 
		I_DEVNAME);
	    error1(server_str);
	    imio_on = 0;  /* shut off this mode */
	    return;
	}
    }
    fcntl(inpipe, F_SETFL, O_RDONLY);   /* clear the O_NDELAY  */
    get_fbconfig();   /* initialize frame buffer configuration list */

    /* add to skyview callback list */
    job = add_job();   /* get slot */
    strcpy(job->pidname, "imio");
    job->pid = getpid();  /* just fill the slot */
    job->fp = fdopen(inpipe, "r");
    job->imgp = 0;        /* will be filled in by WCS packet */
    job->callback = imio_callback;
    job->valid_callback = 1;

    iraf_ct();   /* set up color table */

}


void
stop_imio()
{
    struct img *savimg, *tmpimg;

    if (job != NULL)
    {
	del_job(job);
	job = NULL;
    }

    if (outpipe != 0)
    {
	close(outpipe);
	outpipe = 0;
    }

    if (inpipe != 0)
    {
	close(inpipe);
	inpipe = 0;
    }

    /* close all imio windows */
    while ((tmpimg = find_last_img()) != NULL)
    {
	if (tmpimg->imio_ok)
	    delete_WIN();
	else
	{
	    savimg = tmpimg;
	    while ((tmpimg = find_previous_img(tmpimg)) != NULL)
	    {
		if (tmpimg->imio_ok)
		{
		    delete_WIN();
		    tmpimg = savimg;
		}
		else
		{
		    savimg = tmpimg;
		}
	    }
	    break;
	}
    }
}

void imio_callback(job)
struct job_info *job;
{
    struct img *imgp_save;

    imgp_save = curr_img;

    if (check_img(job->imgp))
	fill_glob(job->imgp);

    do_iraf_cmd();

    job->imgp = curr_img;   /* save this most recent imio identity */

    if (curr_img != imgp_save)
	fill_glob(imgp_save);

    fflush(stdout);
    fflush(stderr);
}

static void
do_iraf_cmd()
{
    register int    sum, i;
    register short *p;
    int             ndatabytes, nbytes, n;
    static int      errmsg = 0, need_swab = 0;
    struct iism70   iis;
    char            buf[SZ_FIFOBUF];
    int status, save_frame;
    int min_x, min_y, max_x, max_y, xtmp, ytmp;
    int this_frame, totbytes;
    struct img tmpimg, *imgp;
    Window child;
    unsigned int keys_buttons;
    int oldxcur, oldycur;
    double d_coord[2];

    /* get the header */
    status = read(inpipe, (char *) &iis, sizeof(iis));
    if (status  < sizeof(iis))
    {
	fprintf(stderr, 
	    "skyview imio mode error: short read on IIS header - bytes = %d\n",
	    status);
	fprintf(stderr, "tid = 0x%x\n", iis.tid);
	fflush(stderr);
	return;
    }

    if (need_swab)
	swab((char *) &iis, (char *) &iis, sizeof(iis));

    /* do the checksum */
    for (i = 0, sum = 0, p = (short *) &iis; i < 8; i++)
	sum += *p++;
    if ((sum & 0xffff) != 0xffff)
    {
	/* checksum failed - try again with bytes swapped */
	swab((char *) &iis, (char *) &iis, sizeof(iis));
	need_swab = !need_swab;   /* falg that we need to swap bytes */
	for (i = 0, sum = 0, p = (short *) &iis; i < 8; i++)
	    sum += *p++;
	if ((sum & 0xffff) != 0xffff)
	{
	    if (!errmsg++)
	    {
		fprintf(stderr, 
		    "skyview imio mode error: header checksum failed\n");
		fflush(stderr);
	    }
	}
    }

    ndatabytes = -iis.thingct;
    if (!(iis.tid & PACKED))
	ndatabytes *= 2;

    if (debug_val == 5)
    {
	fprintf(debug_file,
	    "subunit=%06o tid=%06o nbytes=%7d x=%06o y=%06o z=%06o t=%06o\n",
	    iis.subunit & 077,
	    iis.tid,
	    ndatabytes,
	    iis.x & 0177777,
	    iis.y & 0177777,
	    iis.z & 0177777,
	    iis.t & 0177777);
	fflush(debug_file);
    }

    switch (iis.subunit & 077)
    {
    case FEEDBACK:
	/* clear frame */
	if (debug_val == 5)
	    fprintf(debug_file,
	    "   FEEDBACK:  erase frame %d\n", decode_frameno(iis.z & 07777));
	status = set_reference_frame(decode_frameno(iis.z & 07777));
	if (status)
	{
	    memset(imio_filptr, 0, lines * pixels);

	    min_x = 0;
	    min_y = 0;
	    max_x = pixels - 1;
	    max_y = lines - 1;
	    repaintm32(min_x, max_x, min_y, max_y, imio_filptr);
	    repaint_x(min_x, max_x, min_y, max_y);

	    /* prepare histogram */
	    for (i=0; i<256; i++)
		screen_hist[i] = 0;
	    strcpy(screen_hist_filnam, filnam);
	}
	break;

    case LUT:
	/* select frame */
	if (iis.subunit & COMMAND)
	{
	    short           x[14];

	    if (read(inpipe, (char *) x, ndatabytes) == ndatabytes)
	    {
		if (need_swab)
		    swab((char *) x, (char *) x, ndatabytes);

		if (debug_val == 5)
		    fprintf(debug_file,
		    "   LUT:  frame %d\n", decode_frameno(x[0]));
		status = set_reference_frame(decode_frameno(x[0]));
		return;
	    }
	    if (debug_val == 5)
		fprintf(debug_file, "   LUT (bad read)\n");
	}
	else
	{
	    if (debug_val == 5)
		fprintf(debug_file, "   LUT (not COMMAND)\n");
	}
	break;

    case MEMORY:
	/* move pixels to/from iraf */
	if (iis.tid & IIS_READ)
	{
	    /* move pixels from skyview to iraf */
	    unsigned char  *fb, *ip;
	    int             nbytes, nleft, n, x, y;
	    time_t            starttime;

	    /* select frame */
	    if (debug_val == 5)
		fprintf(debug_file,
		"   MEMORY read:  frame %d\n", decode_frameno(iis.z & 07777));
	    status = set_reference_frame(decode_frameno(iis.z & 07777));

	    fb = (unsigned char *) imio_filptr;
	    nbytes = ndatabytes;
	    x = iis.x & 01777;
	    y = iis.y & 01777;

	    ip = max(fb, min(fb + Fb_width * Fb_height - nbytes,
			     fb + y * Fb_width + x));
	    if (ip != fb + y * Fb_width + x)
	    {
		fprintf(debug_file,
		    "skyview imio mode error: attempted read out of bounds on buffer\n");
		fprintf(debug_file,
			"read %d bytes at [%d,%d]\n", nbytes, x, y);
	    }

	    if (debug_val == 5)
		fprintf(debug_file, "read %d bytes at x=%d, y=%d\n",
		    nbytes, x, y);

	    /* send the data to iraf */
	    starttime = time((time_t *)0);
	    for (nleft = nbytes; nleft > 0; nleft -= n)
	    {
		n = (nleft < SZ_FIFOBUF) ? nleft : SZ_FIFOBUF;
		if ((n = write(outpipe, ip, n)) <= 0)
		{
		    if (n < 0 || (time((time_t * )0) - starttime > IO_TIMEOUT))
		    {
			fprintf(debug_file, 
			    "skyview imio mode error: timeout sending data\n");
			break;
		    }
		}
		else
		    ip += n;
	    }

	    return;

	}
	else
	{
	    /* move pixels from iraf to skyview */
	    unsigned char  *fb, *op;
	    int             nbytes, nleft, n, x, y;
	    long            starttime;

	    /* select frame */
	    
	    if (debug_val == 5)
		fprintf(debug_file,
		"   MEMORY write:  frame %d\n", decode_frameno(iis.z & 07777));
	    status = set_reference_frame(decode_frameno(iis.z & 07777));
	    if (!status)
		break;
	    if (!imio_ok)
		break;

	    fb = (unsigned char *) imio_filptr;
	    nbytes = ndatabytes;
	    x = iis.x & 07777;
	    y = iis.y & 07777;

	    op = max(fb, min(fb + Fb_width * Fb_height - nbytes,
			     fb + y * Fb_width + x));
	    if (op != fb + y * Fb_width + x)
	    {
	    fprintf(debug_file,
		"skyview imio mode error: attempted write out of bounds on buffer\n");
	    }

	    if (debug_val == 5)
		fprintf(debug_file, "write %d bytes at x=%d, y=%d\n",
		    nbytes, x, y);

	    starttime = time((time_t *)0);
	    for (nleft = nbytes; nleft > 0; nleft -= n)
	    {
		n = (nleft < SZ_FIFOBUF) ? nleft : SZ_FIFOBUF;
		if ((n = read(inpipe, op, n)) <= 0)
		{
		    if (n < 0 || (time((time_t *)0) - starttime > IO_TIMEOUT))
			break;
		}
		else
		{
		    op += n;
		}
	    }

	    xtmp = x;
	    ytmp = y;
	    min_x = xtmp;
	    min_y = ytmp;
	    min_y = lines - 1 - min_y;  /* convert to our SC coords */

	    max_x = xtmp + Fb_width - 1;;
	    max_y = ytmp + nbytes / Fb_width - 1;
	    max_y = lines - 1 - max_y;  /* convert to our SC coords */

	    repaintm32(min_x, max_x, min_y, max_y, fb);
	    repaint_x(min_x, max_x, min_y, max_y);

	    return;
	}
	break;

    case WCS:
	/* WCS (world coord system) is a description of the frame. */
	/* It contains the filename and transformation parameters and text. */
	/* It is sent with the image, and then often requested by iraf */
	/* apparently to confirm that iraf and skyview are talking about */
	/* the same image. */
	if (iis.tid & IIS_READ)
	{
	    /* iraf wants the WCS for a frame */
	    register char  *op;
	    register int    n;
	    char            emsg[SZ_WCSBUF];
	    char           *text;

	    if (debug_val == 5)
		fprintf(debug_file, 
		"   WCS read:  frame %d\n", decode_frameno(iis.z & 07777));
	    for (op = emsg, n = SZ_WCSBUF; --n >= 0;)
		*op++ = 0;

	    this_frame = decode_frameno(iis.z & 07777);
	    status = set_reference_frame(this_frame);
	    if (status)
	    {
		text = wcsbuf;
	    }
	    else
		strcpy(text = emsg, "[NOSUCHFRAME]\n");

	    if (debug_val == 5)
	    {
		fprintf(debug_file, "response to wcs query:\n");
		write(fileno(debug_file), text, SZ_WCSBUF);
	    }

	    write(outpipe, text, SZ_WCSBUF);

	}
	else
	{
	    /* iraf is supplying the WCS for a frame */
	    char            buf[1024];


	    this_frame = decode_frameno(iis.z & 07777);
	    if (debug_val == 5)
		fprintf(debug_file, "   WCS write:  frame %d\n", this_frame);

	    set_fbconfig(iis.t & 077); /* sets fbconfig, Fb_width, Fb_height */

	    if (read(inpipe, buf, ndatabytes) != ndatabytes)
	    {
		fprintf(debug_file, 
		    "skyview imio mode error:  short read on wcs\n");
		return;
	    }

	    if (debug_val == 5)
	    {
		fprintf(debug_file, "set wcs text:\n");
		write(fileno(debug_file), buf, SZ_WCSBUF);
		fprintf(debug_file, "\n");
	    }

	    get_transformation(buf);  /* set a,b,c,d,tx,ty */

	    imgp = find_IMIO(fbconfig, this_frame);
	    if (imgp == NULL)
	    {
		save_frame = frame;  /* save skyview's frame next frame nbr */
		next_frame();
		tmpimg.frameptr = 0;  /* framestruct not yet allocated */
		set_header_defaults(&tmpimg);
		fill_glob(&tmpimg);
		strncpy(wcsbuf, buf, SZ_WCSBUF);

		curr_img = NULL;

		imio_ok = FALSE;
		imio_config = fbconfig;
		imio_frame = this_frame;

		set_file_sys();

		bitpix = 8;
		naxis = 3;
		pixels = Fb_width;
		lines = Fb_height;
		naxis3 = 1;

		cdelt1 = a;
		cdelt2 = d;
		crpix1 = tx;
		crpix2 = ty;

		filehdr.tapetype = IMFILE;
		sprintf(filnam, "/imio:%d:%d", fbconfig, this_frame);
		strcpy(fi_name, filnam);
		totbytes = lines * pixels;
		imio_filptr = (char *) imget_nofile(totbytes, filnam);
		if (imio_filptr == NULL)
		{
		    return;
		}
		memset(imio_filptr, 0, totbytes);

		minx = minx_next;
		miny = miny_next;
		x_orig = minx;
		y_orig = miny;
		initial_zoom();
		maxx = minx + pixels - 1;
		maxy = miny + lines  - 1;
		band_offset = 0;
		x_offset = 1 - minx;
		samp_disp = 0;
		/* painting down */
		line_disp = 0;
		sgn_y = -1;
		y_offset = lines + miny;
		mode_used = req_mode;
		slow = 0;
		shigh = 200;
		imio_last_x = 0;
		imio_last_y = 0;
		if (!show_frame((char *)NULL, TRUE, (char *)NULL))
		    return;
		stretch(0, 255);   /* set up color table */

		imio_ok = TRUE;
		job->imgp = new_img();  /* put into screen mgmt */
		flush_graphics();
		wait_for_windowmgr(FALSE);
		frame = save_frame;
	    }
	    else
	    {
		strncpy(wcsbuf, buf, SZ_WCSBUF);
		strncpy(imgp->wcsbuf, buf, SZ_WCSBUF);
	    }
	}

	return;
	break;

    case IMCURSOR:
	/* do a pick or warp the pointer to a specified point 
	 * Cursor reads may
	 * be either nonblocking (immediate) or blocking, using the keyboard
	 * or mouse to terminate the read, and coordinates may be returned in
	 * either image (world) or frame buffer pixel coordinates.
	 */
	if (iis.tid & IIS_READ)
	{
	    /*
	     * Read the logical image cursor.  In the case of a blocking read
	     * all we do is initiate a cursor read; completion occurs when the
	     * user hits a key or button.
	     */
	    if (iis.tid & IMC_SAMPLE)
	    {
		if (debug_val == 5)
		    fprintf(debug_file, 
		    "   IMCURSOR:  READ in SAMPLE mode (non-blocking)\n");

		d_coord[0] = imio_last_x;
		d_coord[1] = imio_last_y;
		dWIN_to_SC(d_coord);
		dSC_to_IRAF(d_coord);
		gio_retcursorval(d_coord[0], d_coord[1], 
		    imio_frame * 100 + iis.z, 0, "");

	    }
	    else
	    {
		if (debug_val == 5)
		    fprintf(debug_file, 
			"   IMCURSOR:  READ in triggered mode\n");

		XQueryPointer(display, RootWindow(display, screen),
		    &root, &child, &root_x, &root_y,
		    &oldxcur, &oldycur, &keys_buttons);
		XWarpPointer(display, None, win, 0,0,0,0, 
		    imio_last_x, imio_last_y);
		cursor_wcs = iis.z;
		imio_wants_char = TRUE;  /* flag that we want a pick */
	    }

	}
	else
	{
	    if (debug_val == 5)
		fprintf(debug_file, "   IMCURSOR:  WRITE\n");
	}

	return;
	break;

    case ZOOMPAN:
	if (debug_val == 5)
	    fprintf(debug_file, 
	    "   ZOOMPAN: frame %d\n", decode_frameno(iis.z & 07777));
	break;
    default:
	if (debug_val == 5)
	    fprintf(debug_file, "   UNSUPPORTED packet type\n");
	break;
    }

    /* Discard any data following the header. */
    if (!(iis.tid & IIS_READ))
	for (nbytes = ndatabytes; nbytes > 0; nbytes -= n)
	{
	    n = (nbytes < SZ_FIFOBUF) ? nbytes : SZ_FIFOBUF;
	    if ((n = read(inpipe, buf, n)) <= 0)
		break;
	}
}

void
cursor_callback(xwin, ywin, key_pressed)
int xwin, ywin, key_pressed;
{
    double d_coord[2];
    static char *tail_ptr, tail[80];
    static int collecting_tail = FALSE;

    if (key_pressed == ':')
    {
	collecting_tail = TRUE;
	tail_ptr = tail;
	return;
    }

    if (collecting_tail)
    {
	if (key_pressed == '\r')
	{
	    *tail_ptr++ = '\0';
	    key_pressed = ':';
	    collecting_tail = FALSE;
	}
	else if ((key_pressed == '\010') || (key_pressed == '\177'))
	{
	    /* backspace or del */
	    tail_ptr--;
	    if (tail_ptr < tail)
	    {
		collecting_tail = FALSE;
	    }
	    return;
	}
	else if (key_pressed == '\025')
	{
	    /* control-U */
	    collecting_tail = FALSE;
	    return;
	}
	else
	{
	    *tail_ptr++ = key_pressed;
	    return;
	}
    }

    imio_wants_char = FALSE;
    d_coord[0] = xwin;
    d_coord[1] = ywin;
    dWIN_to_SC(d_coord);
    dSC_to_IRAF(d_coord);
    gio_retcursorval(d_coord[0], d_coord[1], imio_frame * 100 + cursor_wcs, key_pressed, tail);
    imio_last_x = xwin;
    imio_last_y = ywin;
    fill_img(curr_img);
    XWarpPointer(display, None, root, 0,0,0,0, root_x, root_y);
}

static void
get_transformation(buf)  /* set a,b,c,d,tx,ty */
char *buf;
{
    char *bufptr;

    bufptr = strchr(buf, '\n');
    if (bufptr)
    {
	sscanf(bufptr+1, "%lf %lf %lf %lf %lf %lf", &a, &b, &c, &d,
	    &tx, &ty);
	if (debug_val == 5)
	    fprintf(debug_file, 
		"a = %f  b = %f  c = %f  d = %f  tx = %f  ty = %f\n",
		a, b, c, d, tx, ty);
    }
    else
    {
	fprintf(stderr, "skyview imio mode error:  bad format in WCS string\n");
	/* stick in some reasonable values */
	a = 1.0;
	b = 0.0;
	c = 0.0;
	d = -1.0;
	tx = 1.0;
	ty = 512.0;
    }
}

static
int set_reference_frame(desired_frame)
int desired_frame;
{
    struct img *tmpimg;

    tmpimg = find_IMIO(fbconfig, desired_frame);
    if (tmpimg == NULL)
    {
	if (debug_val == 5)
	    fprintf(debug_file, "set_reference_frame error: no frame %d\n", 
		desired_frame);
	return(FALSE);
    }
    else
    {
	return(TRUE);
    }
}

static int
decode_frameno(z)
    register int    z;
{
    register int    n;

    /*
     * Get the frame number, encoded with a bit for each frame, 01 is frame 1,
     * 02 is frame 2, 04 is frame 3, and so on.
     */
    if (!z)
	z = 1;
    for (n = 0; !(z & 1); z >>= 1)
	n++;

    return (max(1, n + 1));
}


/*
 * GIO_RETCURSORVAL -- Return the cursor value on the output datastream to the
 * client which requested the cursor read.
 */
static void
gio_retcursorval(wx, wy, wcs, key, strval)
    float           wx, wy;	/* cursor coordinates */
    int             wcs;	/* encoded WCS value */
    int             key;	/* keystroke used as trigger */
    char           *strval;	/* optional string value */
{
    register char  *op;
    register int    n;
    char            curval[SZ_IMCURVAL];
    char            keystr[20];

    for (op = curval, n = SZ_IMCURVAL; --n >= 0;)
	*op++ = 0;


    /* Encode the cursor value. */
    if (key == EOF)
	sprintf(curval, "EOF\n");
    else
    {
	if (isprint(key) && !isspace(key))
	{
	    keystr[0] = key;
	    keystr[1] = '\0';
	}
	else
	    sprintf(keystr, "\\%03o", key);

	sprintf(curval, "%10.3f %10.3f %d %s %s\n",
		wx, wy, wcs, keystr, strval);
    }
    if (debug_val == 5)
	fprintf(debug_file, "%s", curval);

    /* Send it to the client program. */
    write(outpipe, curval, sizeof(curval));
}


void
dIM_to_IRAF(d_coord)
double d_coord[2];
{
    if (imio_ok)  /* if it came from IRAF */
    {
	dIM_to_SC(d_coord);
	dSC_to_IRAF(d_coord);
    }
    else
    {
	d_coord[0] = d_coord[0] + crpix1;
	d_coord[1] = d_coord[1] + crpix2;
    }
}

void
dSC_to_IRAF(d_coord)
double d_coord[2];
{
    double tmp1;

    if (imio_ok)
    {
    get_transformation(wcsbuf);  /* set a,b,c,d,tx,ty */
    d_coord[1] = lines -1 - d_coord[1];  /* flip from SC to IMTOOL coords */
    tmp1       = a * d_coord[0] + c * d_coord[1] + tx;
    d_coord[1] = b * d_coord[0] + d * d_coord[1] + ty;
    d_coord[0] = tmp1;
    }
    else
    {
	d_coord[0] = d_coord[0] + x_offset;
	d_coord[1] = d_coord[1] * sgn_y + y_offset;
    }
}

void
dIRAF_to_SC(d_coord)
double d_coord[2];
{
    double tmp1;

    if (imio_ok)  /* if it came from IRAF */
    {
    get_transformation(wcsbuf);  /* set a,b,c,d,tx,ty */
    tmp1       = (d * (d_coord[0] - tx) - c * (d_coord[1] - ty)) / (a*d - b*c);
    d_coord[1] = (-b * (d_coord[0] - tx) + a * (d_coord[1] - ty)) / (a*d - b*c);
    d_coord[0] = tmp1;
    d_coord[1] = lines -1 - d_coord[1];  /* flip from SC to IMTOOL coords */
    }
    else
    {
	d_coord[0] = d_coord[0] - x_offset;
	d_coord[1] = (d_coord[1] - y_offset ) * sgn_y;
    }
}

void
dIRAF_to_IM(d_coord)
double d_coord[2];
{
    if (imio_ok)  /* if it came from IRAF */
    {
	dIRAF_to_SC(d_coord);
	dSC_to_IM(d_coord);
    }
    else
    {
	d_coord[0] = d_coord[0] - crpix1;
	d_coord[1] = d_coord[1] - crpix2;
    }
}


/* Possible frame buffer sizes. */
struct fbconfig_struct {
    int     nframes;                /* number of frames             */
    int     width;                  /* frame buffer width           */
    int     height;                 /* frame buffer height          */
};

static  struct fbconfig_struct fb_config[MAX_FBCONFIG];


/* GET_FBCONFIG -- Read the IMTOOL startup file to get the set of possible
 * frame buffer sizes.
 *
 * File format:		configno nframes width height [extra fields]
 *	e.g.,			1  2  512  512
 *				2  2  800  800
 *				3  1 1024 1024		# comment
 */
static void
get_fbconfig()
{
	register char	*ip;
	register FILE	*fp;
	int	config, nframes, width, height, i;
	char	lbuf[SZ_LINE+1], *fname;

	/* Initialize the config table. */
	for (i=0;  i < MAX_FBCONFIG;  i++)
	    fb_config[i].nframes = 0;

	/* Attempt to open the config file. */
	fp = NULL;
	if ((fname=getenv(FBCONFIG_ENV1)) || (fname=getenv(FBCONFIG_ENV2)))
	    fp = fopen (fname, "r");
	if (!fp && (fname = getenv ("HOME"))) {
	    sprintf (lbuf, "%s/%s", fname, FBCONFIG_1);
	    fp = fopen (fname = lbuf, "r");
	}
	if (!fp)
	    fp = fopen (fname = FBCONFIG_2, "r");

	/* If cannot find a config file, set up the default configuration. */
	if (!fp) {
	    fb_config[0].nframes = DEF_NFRAMES;
	    fb_config[0].width   = DEF_FRAME_WIDTH;
	    fb_config[0].height  = DEF_FRAME_HEIGHT;
	    return;
	}

	/* Scan the frame buffer configuration file.
	 */
	while (fgets (lbuf, SZ_LINE, fp) != NULL) {
	    /* Skip comment lines and blank lines. */
	    for (ip=lbuf;  *ip == ' ' || *ip == '\t';  ip++)
		;
	    if (*ip == '\n' || *ip == '#')
		continue;
	    if (!isdigit (*ip))
		continue;
	    switch (sscanf (ip, "%d%d%d%d", &config,&nframes,&width,&height)) {
	    case 4:
		break;			/* normal case */
	    case 3:
		height = width;		/* default to square format */
		break;
	    default:
		fprintf (stderr, 
		    "skyview imio mode error: bad config `%s'\n", ip);
		continue;
	    }

	    nframes = max (1, nframes);
	    width   = max (1, width);
	    height  = max (1, height);

	    /* Since the frame buffer is stored in a memory pixrect
	     * (effectively), the line length should be an integral number
	     * of 16 bit words.
	     */
	    if (width & 1) {
		fprintf (stderr, 
		"skyview imio mode warning: fb config %d [%d-%dx%d] - ",
		    config, nframes, width, height);
		fprintf (stderr, "frame width should be even, reset to %d\n",
		    --width);
	    }

	    config = max(1, min(MAX_FBCONFIG, config)) - 1;
	    fb_config[config].nframes = nframes;
	    fb_config[config].width   = width;
	    fb_config[config].height  = height;
	}

	fclose (fp);
}


/* SET_FBCONFIG -- Setup a frame buffer configuration.   */
/* sets fbconfig, Fb_width, Fb_height                    */

static void
set_fbconfig (config)
int	config;   /* zero based */
{
    if (debug_val == 5)
	fprintf (debug_file, "RBH set_fbconfig (%d)\n", config);

    if (config < 0 || config >= MAX_FBCONFIG) 
    {
	fprintf (stderr,
	"skyview imio mode error: no such frame buffer configuration - %d\n",
	    config+1);
	return;
    }
    if (fb_config[config].nframes == 0)
    {
	fprintf (stderr,
	"skyview imio mode error: no such frame buffer configuration - %d\n", 
	    config+1);
	return;
    }

    Fb_width = fb_config[config].width;
    Fb_height = fb_config[config].height;
    fbconfig = config;

    if (debug_val == 5)
	fprintf (debug_file, "Frame buffer configuration %d:  %dx%d\n",
	    fbconfig + 1, Fb_width, Fb_height);
}
