/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "area.h"
#include "parse.h"
#include "im_proto.h"
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>

#ifdef NO_FLOAT_H
#include <values.h>
#define DBL_MAX MAXDOUBLE
#define FLT_MAX MAXFLOAT
#else
#include <float.h>
#endif

extern int area_number, debug, graphics, debug_val, server_mode; 
extern char server_str[];
extern int control_c;
extern struct area_struct area[MAX_AREAS];
extern char filnam[], bunit[], plane, mod_filnam[];
extern int x_offset, minx, maxx, miny, maxy, sgn_y, y_offset;
extern int pixels, lines, reqband, band_offset;
extern double crpix1, crpix2;
extern double cdelt1, cdelt2;
extern double bscale, b_zero;
extern double backgnd_level;
extern FILE *session, *debug_file;
extern struct img *curr_img;
extern int ads_server_mode;
 
static double   dtr = M_PI/180;
static int n_list; 
static int x_cross[101]; 
 
/* function prototypes */
#ifdef _NO_PROTO

static int minmax();

#else

static int minmax(int i, int y);

#endif /* _NO_PROTO */


/********************************************************************** 
**                                                                   ** 
**  EXAMINE_AREA uses the predefined outline of an area to determine ** 
**  basic parameters for the area.                                   ** 
**                                                                   **  
**  These are:    minimum                maximum                     ** 
**                mean (background)      integrated intensity        ** 
**                std deviation          centroid                    ** 
**                                                                   ** 
**********************************************************************/ 

void
examine_area(argc,argv) 

int argc;
char **argv;
 
{
    int min_x,max_x,min_y,max_y;  /* WATCH OUT   in this file x is used */
				     /* for samp and y is used for line */
			    /* ie. x and y are IM coordinates, not SC */
    double z, size1, size2, pixel_area;
    double minimum, maximum; 
    double flux_sum=0., square_sum=0.; 
    double xf_sum=0., yf_sum=0.;
    double x_sum=0., y_sum=0.; 
    double average_flux, stdev; 
    double fluxes[4];
    double x_centroid, y_centroid;	 
    double xf_centroid, yf_centroid;	 
    char funit[80];
    char *p, *coord_str;
    char max_coord_str[80], min_coord_str[80];
    char s_coord_min[2][40], s_coord_max[2][40], s_coord[2][40];
    int coord_sys;
    char temp_filnam[MAXPATHLEN];

    double d_coord[2];
    int i_coord[2];
    int i, x, y, s_len; 
    int INTERIOR=1; 
    int n_average; 
    int init_flag=TRUE;
    int x_minimum, x_maximum;
    int y_minimum, y_maximum;
    int show_mark = FALSE;
    int jsys;
    double equinox;


    set_current_window();
    enable_control_c();

    if(argc > 1)
    {
	if (strcmp(argv[argc-1], "mark") == 0)
	{
	    argc--;
	    show_mark = TRUE;
	}
    }
    if(argc > 1)
    {
	if (!switch_area(argv[1]))
	    return;
    }

    if (area[area_number].area_type == 0)
    {
	error1("examine error:  area has not been defined");
	return;
    }

     if(debug)
	  fprintf(debug_file, "examine:  area number %d  area_type[%d] = %d\n", 
	      area_number, area_number, area[area_number].area_type);

     draw_outline('g', 2);   /* of polygon or circle */ 
 
     find_range( &min_x, &max_x, &min_y, &max_y);     /* min,max for x and y */ 
 
     segment_setup();  /* calculate parameters defining each line */ 

     n_average = 0;
 
    if (show_mark)
    {
	overlay_setup('r');
	overlay_color(1);
	next_bundle();
    }

     /* Loop over the possible area of the image */ 
 
     for(y=min_y; y <= max_y; y++) 
          {
          x_list(y);     /* list of boundary crossings */
 
          for(x=min_x; x <= max_x; x++) 
               {
	       if(debug)
		    fprintf(debug_file, "examine: at position %d,%d\n", x,y);

               if(position(x)==INTERIOR && !on_horizontal(x, y)) 
		    {
		    i_coord[0] = x;
		    i_coord[1] = y;
		    flux_IRAF(i_coord, fluxes);

                    z =  fluxes[0] * bscale + b_zero - backgnd_level; 
		    if(debug)
			fprintf(debug_file, 
			    "examine: flux dn = %g, value = %-g\n",
			    fluxes[0], z);
		    if (show_mark)
		    {
			d_coord[0] = x;
			d_coord[1] = y;
			dIRAF_to_IM(d_coord);
			dIM_to_SC(d_coord);
			do_mark(d_coord[0], d_coord[1], 
			    DOT, 1, 0, 0);
			    
		    }

                    if(!FINITE(fluxes[0]))  
			 continue;

                    if(init_flag == TRUE)
	                 {
		         init_flag=FALSE;
		         minimum=z;
			 x_minimum = x;
			 y_minimum = y;

		         maximum=z;
			 x_maximum = x;
			 y_maximum = y;

			 if(debug)
			     fprintf(debug_file, 
				 "examine: minimum,maximum= %-g %-g\n",
				 minimum,maximum);
		         }
          
                    /* For useable fluxes */ 
 
                    if(z>maximum)   /* Maximum */ 
			 {
                         maximum = z; 
			 x_maximum = x;
			 y_maximum = y;

			 if(debug)
			     fprintf(debug_file, 
				 "examine: maximum= %-g at (%d,%d)\n",
				 maximum,x,y);
                         }
 
                    if(z<minimum)   /* Minimum */ 
			 {
		         minimum = z;  
			 x_minimum = x;
			 y_minimum = y;

			 if(debug)
			      fprintf(debug_file, 
				  "examine: minimum= %-g at (%d,%d)\n",
				  minimum,x,y);
                         }

                    ++n_average; 
 
                    flux_sum += z; 
                    square_sum += z * z; 
 
                    xf_sum += (double)x * z; 
                    yf_sum += (double)y * z; 

 
                    x_sum += (double)x; 
                    y_sum += (double)y; 
		    }
	       if (control_c)
	       {
		   image_setup(plane);
		   error1("examine interrupted by control c");

		   fprintf(session, "\nexamine interrupted by control c\n");
		   return;
	       }
	       }
          }
 
     image_setup(plane);

     if(n_average == 0)  
     {
	  error1("examine error:  no valid pixels in defined area");
	  return;
     }
 
     /* Calculate the mean and standard deviation */ 
 
     average_flux = flux_sum / (double) n_average; 
     stdev = sqrt( (double) (square_sum / n_average  
                            - average_flux * average_flux)  ); 
 
 
     /* Calculate the centroid position */ 
 
     x_centroid = x_sum / (double) n_average; 
     y_centroid = y_sum / (double) n_average; 

     xf_centroid = xf_sum / flux_sum; 
     yf_centroid = yf_sum / flux_sum; 

     /* Calculate the size of a pixel */

     size1 = fabs(cdelt1 * dtr);   /* convert cdelt1 to radians */
     size2 = fabs(cdelt2 * dtr);   /* convert cdelt2 to radians */
     pixel_area = size1 * size2;       /* Area in steradians */

     /* Convert integrated flux */

     if (pixel_area != 0)
	 flux_sum = flux_sum * pixel_area;


     /* Modify the units string */

     strcpy(funit, bunit);
     s_len = strlen(funit);

 /* chop off trailing blanks */
     for(i = s_len - 1; i >= 0; --i)
	  {
	  if(funit[i] != ' ')
	       break;

	  funit[i] = '\0';  
	  }
     s_len = i + 1;



     /* look for a star or slash  */
     for(i = s_len - 1; i >= 0; --i)
	  {
	  if(funit[i] == '*' || funit[i] == '/')
	       break;
	  }
     p = &(funit[i]);


     /* if it ends with /STER etc, chop off the /STER part */
     if(strcasecmp(p, "/STER") == 0 ||
	strcasecmp(p, "/SR")   == 0 ||
	strcasecmp(p, "/ST")   == 0 ||
	strcasecmp(p, "*SR-1") == 0 ||
	strcasecmp(p, "*ST-1") == 0   )
	  {
	  funit[i] = '\0';
	  s_len = i + 1;
	  }
        else
	  {
	  strcat(funit, "*STER");
	  }


 
    /* Print out the information */ 
 
    d_coord[0] = x_maximum;
    d_coord[1] = y_maximum;
    dIRAF_to_IM(d_coord);
    strcpy(max_coord_str, 
	dimage_to_sky(d_coord, s_coord_max, &coord_sys, &equinox, &jsys));
    d_coord[0] = x_minimum;
    d_coord[1] = y_minimum;
    dIRAF_to_IM(d_coord);
    strcpy(min_coord_str, 
	dimage_to_sky(d_coord, s_coord_min, &coord_sys, &equinox, &jsys));

    sprintf(temp_filnam, "%s%s", filnam, mod_filnam[0]?" (modified)":"");

    if(server_mode == FALSE)
    {
	printf(
	"         Area %d Parameters  from file %s\n", area_number, temp_filnam);
	printf(
	"    =========================================================\n\n");
	printf("    Maximum %-g    at %s", maximum, max_coord_str);
	printf("    Minimum %-g    at %s", minimum, min_coord_str);
	printf("\n    Mean    %-g                           \n", average_flux);
	printf("    st dev  %-g                           \n", stdev       );
	if(backgnd_level != 0.)
	{
	    printf("    Using as background %-g         \n", backgnd_level);
	}
	printf(
	    "                                        Flux units %s\n\n", bunit);
    }
    else
    {
	sprintf(server_str, "%d", area_number);
	srv_real("area_number", server_str);
	srv_string("filename", temp_filnam);
	sprintf(server_str, "%-g", maximum);
	if (ads_server_mode)
	{
	    srv_coord_val("max", coord_sys, s_coord_max, equinox, jsys,
		server_str, bunit);
	}
	else
	{
	    srv_string("coord_sys", coord_name(coord_sys));
	    srv_real("max_pixel_value", server_str);
	    srv_real("max_x", s_coord_max[0]);
	    srv_real("max_y", s_coord_max[1]);
	}
	sprintf(server_str, "%-g", minimum);
	if (ads_server_mode)
	{
	    srv_coord_val("min", coord_sys, s_coord_min, equinox, jsys,
		server_str, bunit);
	}
	else
	{
	    srv_real("min_pixel_value", server_str);
	    srv_real("min_x", s_coord_min[0]);
	    srv_real("min_y", s_coord_min[1]);
	}
	sprintf(server_str, "%-g", average_flux);
	srv_real("mean_pixel_value", server_str);
	sprintf(server_str, "%-g", stdev);
	srv_real("std_dev", server_str);
	sprintf(server_str, "%-g", backgnd_level);
	srv_real("background_level", server_str);
	srv_string("units", bunit);
    }

    fprintf(session, "         Area %d Parameters  from file %s\n", 
	area_number, temp_filnam);
    fprintf(session,
	"    =========================================================\n\n");
    fprintf(session, "    Maximum %-g    at %s", maximum, max_coord_str);
    fprintf(session, "    Minimum %-g    at %s", minimum, min_coord_str);
    fprintf(session, "\n    Mean    %-g                 \n", average_flux);
    fprintf(session, "    st dev  %-g                 \n", stdev       );
    if(backgnd_level != 0.)
    {
	fprintf(session, "    Using as background %-g\n", backgnd_level);
    }
    fprintf(session,
	"                                        Flux units %s\n\n", bunit);


 if (pixel_area != 0)
 {
    pixel_area = pixel_area * n_average;

    if(server_mode == FALSE)
    {
	printf("    Flux integral      %-g   %s           \n", flux_sum, funit);
	printf("    Inside %-g Steradian area             \n", pixel_area  );
    }
    else
    {
	srv_string("sum_type", "integral");
	sprintf(server_str, "%-g", flux_sum);
	srv_real("flux_sum", server_str);
	srv_string("sum_units", funit);
	sprintf(server_str, "%-g", pixel_area);
	srv_real("pixel_area", server_str);
    }
    fprintf(session, "    Flux integral      %-g   %s           \n", 
				     flux_sum, funit);
    fprintf(session, "    Inside %-g Steradian area   \n", pixel_area  );
 }
 else
 {
    if(server_mode == FALSE)
	printf("    Flux sum           %-g   %s           \n", 
				     flux_sum, bunit);
    else
    {
	srv_string("sum_type", "sum");
	sprintf(server_str, "%-g", flux_sum);
	srv_real("flux_sum", server_str);
	srv_string("sum_units", bunit);
    }
    fprintf(session, "    Flux sum           %-g   %s           \n", 
				     flux_sum, bunit);
 }

    if(server_mode == FALSE)
    {
	printf("    (%d pixels)                           \n\n", n_average   );
    }
    else
    {
	sprintf(server_str, "%d", n_average);
	srv_real("pixel_count", server_str);
    }
    fprintf(session, "    (%d pixels)                 \n\n", n_average   );

    d_coord[0] = x_centroid;
    d_coord[1] = y_centroid;
    dIRAF_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);


    if(server_mode == FALSE)
    {
	printf("    Centroid                              %s\n\n",coord_str);
    }
    else if (ads_server_mode)
    {
	srv_coord("centroid", coord_sys, s_coord, equinox, jsys);
    }
    else
    {
	srv_real("centroid_x", s_coord[0]);
	srv_real("centroid_y", s_coord[1]);
    }
    fprintf(session, "    Centroid                    %s\n\n",coord_str);

    d_coord[0] = xf_centroid;
    d_coord[1] = yf_centroid;
    dIRAF_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);


    if(server_mode == FALSE)
    {
	printf("    Flux weighted centroid                %s\n\n",coord_str);
	printf("    ==========================================================\n");
    }
    else if (ads_server_mode)
    {
	srv_coord("flux_weighted_centroid", coord_sys, s_coord, equinox, jsys);
    }
    else
    {
	srv_real("flux_weighted_centroid_x", s_coord[0]);
	srv_real("flux_weighted_centroid_y", s_coord[1]);
    }
    fprintf(session, "    Flux weight centroid        %s\n\n",coord_str);
    fprintf(session,
	"    ==========================================================\n");
 
 
}
 
 
 
/* FIND_RANGE  
** finds the minimum and maximum x and y in the area definition table  
** (used to limit the range searched for interior points) 
*/ 
 
void
find_range(min_x, max_x, min_y, max_y) 
int *min_x, *max_x, *min_y, *max_y; 
{
    int i, n_area;
    double rad, d_coord[2]; 
    double x, y, xmin, ymin, xmax, ymax;

    if(area[area_number].area_type == POLYGON)
    { 
	n_area = area[area_number].vertices; 

	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];
	dIM_to_IRAF(d_coord);
	xmin = xmax = d_coord[0];
	ymin = ymax = d_coord[1];
	for(i=2; i<=n_area; ++i) 
	{ 
	    d_coord[0] = area[area_number].x[i];
	    d_coord[1] = area[area_number].y[i];
	    dIM_to_IRAF(d_coord);
	    x = d_coord[0];
	    y = d_coord[1];

	    if(x < xmin)
		xmin = x; 

	    if(x > xmax)
		xmax = x; 

	    if(y < ymin)
		ymin = y; 

	    if(y > ymax)
		ymax = y; 
	} 
	*min_x = ceil(xmin);
	*max_x = floor(xmax);
	*min_y = ceil(ymin);
	*max_y = floor(ymax);
    } 
 
    else if(area[area_number].area_type == CIRCLE)
    { 
	rad = area[area_number].radius;

	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];

	if(debug)
	    fprintf(debug_file, "find_range: circle  %g %g %g\n", 
	    d_coord[0], d_coord[1], rad);
	
	dIM_to_IRAF(d_coord);

	*min_x = ceil(d_coord[0] - rad);
	*max_x = floor(d_coord[0] + rad);

	*min_y = ceil(d_coord[1] - rad);
	*max_y = floor(d_coord[1] + rad);
    } 
 
    else if(area[area_number].area_type == ELLIPSE)
    { 
	if (area[area_number].x[2] > area[area_number].y[2])
	    rad = area[area_number].x[2] / 2;
	else
	    rad = area[area_number].y[2] / 2;

	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];
	dIM_to_IRAF(d_coord);
	x = d_coord[0];
	y = d_coord[1];

	if(debug)
	    fprintf(debug_file, "find_range: ellipse %g %g %g\n", x,y,rad);

	*min_x = ceil(x - rad);
	*max_x = floor(x + rad);

	*min_y = ceil(y - rad);
	*max_y = floor(y + rad);
    } 

    /* clip at image boundary */
    if (*min_x  < 1)
	*min_x = 1;
    if (*min_y  < 1)
	*min_y = 1;

    if (*max_x > pixels)
	*max_x = pixels;
    if (*max_y > lines)
	*max_y = lines;


 
    if(debug)
	fprintf(debug_file, 
	    "find_range:  x %d to %d, y %d to %d \n", 
	    *min_x, *max_x, *min_y, *max_y);
}
 
 
 
/* SEGMENT_SETUP 
** determines the coefficients of a parameterized characterization of 
** each line segment.  The parameter is 0 at the first point and 1 at 
** the second point.  Therefore, any y value which corresponds to a 
** value of the parameter outside this range represents a horizontal 
** line that does not cross the segment. 
*/ 
 
static double xa[101], xb[101], ya[101], yb[101]; 
static double sin_angle, cos_angle;
 
void
segment_setup() 
 
{
    int n_area, i;
    double x, x2, y, y2; 
    double angle, d_coord[2];
 
    if(area[area_number].area_type == POLYGON)
    {
	n_area = area[area_number].vertices; 

	for(i=1; i<n_area; ++i) 
	{
	    d_coord[0] = area[area_number].x[i]; 
	    d_coord[1] = area[area_number].y[i]; 
	    dIM_to_IRAF(d_coord);
	    x = d_coord[0];
	    y = d_coord[1];

	    d_coord[0] = area[area_number].x[i+1]; 
	    d_coord[1] = area[area_number].y[i+1]; 
	    dIM_to_IRAF(d_coord);
	    x2 = d_coord[0];
	    y2 = d_coord[1];

	    xa[i] = x; 
	    xb[i] = x2-x; 

	    ya[i] = y; 
	    yb[i] = y2-y; 

	    if(debug)
		fprintf(debug_file, 
		    "segment_setup: line %d  x = %gt + %g, y = %gt + %g \n",
		    i, xb[i], xa[i], yb[i], ya[i]);

	}
    }
    else if (area[area_number].area_type == ELLIPSE)
    {
	angle = area[area_number].x[3];
	if (cdelt2 < 0)
	    angle = 180 - angle;
	sin_angle = sin(angle * dtr);
	cos_angle = cos(angle * dtr);
    }
}
 
 
 
/* X_LIST 
** uses the line parameters defined by 'segment_setup' and the  
** current y-value to determine a list of the boundary crossings 
*/ 
 
void
x_list(y) 
 
int y; 
 
{ 
    double height, width;
    int n_area, i, j;
    double t, dx, dy, xc, yc, rad; 
    double half_height, half_width, width2, height2;
    int x, inside, old_inside, orig_inside;
    double x_rot, y_rot, xc_rot, yc_rot;
    double d_x_temp; 
    double d_x_cross[101];
    double d_coord[2];

    if(area[area_number].area_type == CIRCLE)
    {
	rad = area[area_number].radius; 

	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];

	dIM_to_IRAF(d_coord);

	xc = d_coord[0];
	yc = d_coord[1];

	dy = (double) y - yc; 
	if ((rad*rad - dy*dy) > 0)   /* RBH fix */
	    dx = sqrt( rad*rad - dy*dy ); 
	else
	    dx = 0;

	x_cross[0] = floor(xc - dx);
	x_cross[1] = ceil(xc + dx);

	n_list = 2; 
	if(x_cross[0] == x_cross[1])
	    n_list = 0;

	if(debug)
	{
	    fprintf(debug_file, "x_list:   y=%d \n",y);
	    for(i=0; i<n_list; ++i)
		fprintf(debug_file, "          %d  %d \n", i, x_cross[i]);
	}

	return;
    }

    else if(area[area_number].area_type == ELLIPSE)
    { 
	width = area[area_number].x[2];
	height = area[area_number].y[2];
	if (height > width)
	    dx = (height / 2) + 2;   /* 1 for truncation, 1 to make sure */
				     /*    we get outside the ellipse */
	else
	    dx = (width / 2) + 2;

	half_height = height / 2;
	half_width = width / 2;

	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];
	dIM_to_IRAF(d_coord);
	xc = d_coord[0];
	yc = d_coord[1];

	/* Now we test each point along the y = yc line                */
	/* to see where it crosses in and out of the ellipse.          */
	/* Formula for ellipse is                                      */
	/*   [(x-xc) * (x-xc) / (half_width * half_width)]  +          */
	/*   [(y-yc) * (y-yc) / (half_height * half_height)]  = 1      */

	n_list = 0;
	width2 = half_width * half_width;
	height2 = half_height * half_height;
	xc_rot =  xc * cos_angle + yc * sin_angle;
	yc_rot = -xc * sin_angle + yc * cos_angle;
	x_rot =  xc * cos_angle + y * sin_angle;
	y_rot = -xc * sin_angle + y * cos_angle;
	/* following is a boolean expression */
	orig_inside = ((x_rot-xc_rot) * (x_rot-xc_rot) / width2  + 
	 	      (y_rot-yc_rot) * (y_rot-yc_rot) / height2  ) < 1;
	old_inside = orig_inside;
	/* first go in negative direction */
	for (x = xc; x >= xc - dx; x--)
	{
	    x_rot =  x * cos_angle + y * sin_angle;
	    y_rot = -x * sin_angle + y * cos_angle;
	    inside = ((x_rot-xc_rot) * (x_rot-xc_rot) / width2  + 
	              (y_rot-yc_rot) * (y_rot-yc_rot) / height2 )  < 1;
	    if (old_inside != inside)
	    {
		/* we got a crossing */
		if (inside)
		{
		    x_cross[n_list] = x+1;  /* move back outside */
		    n_list++;
		    old_inside = inside;
		}
		else
		{
		    x_cross[n_list] = x;
		    n_list++;
		    if (n_list == 2)
		    {
			/* we got our 2 crossings (but in wrong order) */
			x_cross[1] = x_cross[0];
			x_cross[0] = x;
			return;
		    }
		    else
			break;
		}
	    }
	}

	old_inside = orig_inside;
	/* now go in positive direction */
	for (x = xc + 1; x <= xc + dx; x++)
	{
	    x_rot =  x * cos_angle + y * sin_angle;
	    y_rot = -x * sin_angle + y * cos_angle;
	    inside = ((x_rot-xc_rot) * (x_rot-xc_rot) / width2  + 
	              (y_rot-yc_rot) * (y_rot-yc_rot) / height2 )  < 1;
	    if (old_inside != inside)
	    {
		/* we got a crossing */
		if (inside)
		{
		    x_cross[n_list] = x-1;   /* move back outside */
		    n_list++;
		    old_inside = inside;
		}
		else
		{
		    x_cross[n_list] = x;
		    n_list++;
		    break;    /* this has to be the second crossing */
		}
	    }
	}

	if(debug)
	{
	    fprintf(debug_file, "x_list (ellipse):   y=%d \n",y);
	    for(i=0; i<n_list; ++i)
		fprintf(debug_file, "          %d  %d \n", i, x_cross[i]);
	}
	return;
    }


    else if(area[area_number].area_type == POLYGON)
    {
	n_area = area[area_number].vertices; 

	n_list = 0;

	for(i=1; i<n_area; ++i) 
	{ 
	    if ((y == ya[i]) && (yb[i] == 0.0))
	    {
		/* it's a horizontal - use both endpoints */
		d_x_cross[n_list] = xa[i];
		n_list++;
		d_x_cross[n_list] = xa[i] + xb[i];
		n_list++;
	    }
	    else if (yb[i] != 0.0)
	    { 
		t = (double) (y - ya[i]) / (double) yb[i] ; 

		if(t>=0. && t<=1.) 
		{
		    d_x_cross[n_list] = xa[i] + xb[i] * t;
		    ++n_list; 
		}
	    }
	}
    }

    /* Sort the list */

    for(i = n_list-2; i >= 0; --i)
    {
	for(j = 0; j <= i; ++j)
	{
	    if(d_x_cross[j] > d_x_cross[j+1])
	    {
		d_x_temp       = d_x_cross[j];
		d_x_cross[j]   = d_x_cross[j+1];
		d_x_cross[j+1] = d_x_temp;
	    }
	}
    }


    for(j = 0; j < n_list-1; ++j)
    {
	x_cross[j] = Round(d_x_cross[j]);
    }

    /* Compress the list by removing duplicate endpoints */
    /* unless the point is the min-max of both segments  */

    for(j = 0; j < n_list-1; ++j)
    {
	if((x_cross[j] == x_cross[j+1]) && !minmax(j, y))
	{
	    for(i = j+1; i < n_list-1; ++i)
	    {
		x_cross[i] = x_cross[i+1];
		d_x_cross[i] = d_x_cross[i+1];
	    }
	    --n_list;
	}
    }

    for(j = 0; j < n_list-1; j+=2)
    {
	x_cross[j] = floor(d_x_cross[j]);
	x_cross[j+1] = ceil(d_x_cross[j+1]);
    }

    if(debug)
    {
	fprintf(debug_file, "x_list:   y=%d \n",y);
	for(i=0; i<n_list; ++i)
	    fprintf(debug_file, "          %d  %d \n", i, x_cross[i]);
    }

} 
 
 
/* ROUND 
** converts a double precision number to the nearest integer 
*/ 
  
int
Round(x) 
 
double x; 
  
{ 
     if( x - (double) floor(x) < 0.5)
          return(floor(x)); 
     else 
          return(ceil(x)); 
} 


#define NEITHER 999

/* MINMAX
** determines whether a point is a minimum or maximum of two line segments
*/ 
  
static int
minmax(i, y)
int i, y;
{
    int iseg, seg[2]; 
    int j, n_area, x;
    int horiz;
    static int horiz_entry;

    n_area = area[area_number].vertices; 
    iseg=0;
    horiz = NEITHER;
    x = x_cross[i];

    for(j = 1; j < n_area; ++j)
    {
	/* if at one end of a horizontal line (t=1 end) */
	if(x == xa[j]+xb[j] && y == ya[j]+yb[j] && yb[j] == 0) 
	{
	    if (horiz != NEITHER)
		return(TRUE);   /* joining point of two horiz lines */

	    if (xb[j] > 0)
		horiz = RIGHT;
	    else
		horiz = LEFT;
	}

	/* if at other end of the horizontal line (t=0 end) */
	else if(x == xa[j] && y == ya[j] && yb[j] == 0) 
	{
	    if (horiz != NEITHER)
		return(TRUE);   /* joining point of two horiz lines */

	    if (xb[j] > 0)
		horiz = LEFT;
	    else
		horiz = RIGHT;
	}


	/* if at low end of this line */
	/* if x is at t=0 end and line slopes up toward t=1 */
	else if(x == xa[j]  && y == ya[j] && ya[j] < ya[j]+yb[j])
	{
	    seg[iseg++] = 0; 
	}

	/* if at high end of this line */
	/* if x is at t=0 end and line slopes down toward t=1 */
	else if(x == xa[j]  && y == ya[j] && ya[j] > ya[j]+yb[j])
	{
	    seg[iseg++] = 1; 
	}

	/* if at low end of this line */
	/* if x is at t=1 end and line slopes up toward t=0 */
	else if(x == xa[j]+xb[j]  && y == ya[j] + yb[j] && ya[j]+yb[j] < ya[j])
	{
	    seg[iseg++] = 0; 
	}

	/* if at high end of this line */
	/* if x is at t=1 end and line slopes down toward t=0 */
	else if(x == xa[j]+xb[j]  && y == ya[j] + yb[j] && ya[j]+yb[j] > ya[j])
	{
	    seg[iseg++] = 1; 
	}

	if(iseg > 1)
	    break;
    }

    if (horiz == LEFT)
    {
	horiz_entry = seg[0];   /* save entry direction */
	return(TRUE);
    }
    if (horiz == RIGHT)
    {
	seg[iseg++] = horiz_entry;   /* retrieve direction from which */
				    /* we entered this horizontal line */
    }

    if (iseg == 0)
	return(TRUE);   /* we got no vertices  - it's crossover lines */

    /* if it's a local max (high end of both line segments), or */
    /*    it's a local min (low end of both line segments) */
    if(seg[0] == seg[1])
	return(TRUE);

    return(FALSE);
}

 
 
 
/* POSITION 
** determines whether a given x value is inside or outside 
** the area. 
*/ 
 
int
position(x) 
int x; 
{ 
     int i;
 
     for(i=0; i<n_list-1; i+=2)
	  {
	  if(x > x_cross[i] && x < x_cross[i+1])
 	        return(TRUE);
	  }

     return(FALSE);
} 
 


/* ON_HORIZONTAL
** determines whether a given position is on a horizontal line
*/ 
 
int
on_horizontal(x, y) 
int x, y; 
{ 
    int j;
    int n_area;

    if(area[area_number].area_type == POLYGON)
    {
	n_area = area[area_number].vertices; 
	for(j = 1; j < n_area; ++j)
	{
	    if(y == ya[j] && yb[j] == 0)
	    {
		if(   (x >= xa[j] && x <= xa[j]+xb[j])
		    || (x >= xa[j]+xb[j] && x <= xa[j]))

		return(TRUE);
	    }
	}
    }
    return(FALSE);
}

void
slice(argc,argv)
int argc;
char **argv;
{
    double d_coord[2], d_coord2[2];
    char s_coord[2][40], *coord_str;
    char s_coord2[2][40];
    int coord_sys;
    char *sl_fname;
    FILE *slice_file;
    int argx, i_coord[2], x, y, x2, y2, xx, yy;
    int x_min, y_min, x_max, y_max;
    double fluxes[4];
    double oldscale, dnmax, dnmin, *dn;
    int bias;
    double a, scale, fmin, fmax;
    struct img *imgp, *imgp2;
    double z, stdev, average_flux, flux_sum = 0.0, square_sum = 0.0;
    int n_average = 0;
    int jsys;
    double equinox;

    enable_control_c();

    /* process args:  */
    /* Possible argument combinations:  */
    /* 1  slice	                         */
    /* 2  slice	 FNAME                   */
    /* 5  slice	       x y x y           */
    /* 6  slice	 FNAME x y x y           */
    /* 6  slice	       x y x y oldscale  */
    /* 7  slice	 FNAME x y x y oldscale  */

    oldscale = 0;

    /* first process filename  */

    switch(argc)
    {
	case 1:   /* bare command */
	    sl_fname = expand_path("slice");
	    break;
	case 2:   /* filename only */
	    sl_fname = expand_path(argv[1]);
	    break;
	case 5:   /* coordinates, but no filename */
	    sl_fname = expand_path("slice");
	    argx = 1;
	    break;
	case 6:   /* filename and coordinates */
	       /* or coordinates and oldscale */
	    /* test if begins with a coordinate */
	    strcpy(s_coord[0], argv[1]);
	    strcpy(s_coord[1], argv[2]);
	    if (str_to_coord(s_coord, d_coord))
	    {
		sl_fname = expand_path("slice");
		argx = 1;
		oldscale = atof(argv[5]);
	    }
	    else
	    {
		sl_fname = expand_path(argv[1]);
		argx = 2;
	    }
	    break;
	case 7:   /* filename and coordinates and oldscale */
	    sl_fname = expand_path(argv[1]);
	    argx = 2;
	    oldscale = atof(argv[6]);
	    break;
	default:
	    error1("wrong number of arguments to slice command");
	    return;
    }
    if ((slice_file = fopen(sl_fname, "w")) == NULL)
    {
	sprintf(server_str, "Can't open slice file '%s'", sl_fname);
	error1(server_str);
	return;
    }

    if (argc < 5)
    {
	sl_vector(d_coord, d_coord2);
    }
    else
    {
	strcpy(s_coord[0], argv[argx]);
	strcpy(s_coord[1], argv[argx + 1]);
	strcpy(s_coord2[0], argv[argx + 2]);
	strcpy(s_coord2[1], argv[argx + 3]);

	(void) sky_to_image(s_coord, d_coord);
	dIM_to_SC(d_coord);
	(void) sky_to_image(s_coord2, d_coord2);
	dIM_to_SC(d_coord2);
    }
    x = Round(d_coord[0]);
    y = Round(d_coord[1]);
    x2 = Round(d_coord2[0]);
    y2 = Round(d_coord2[1]);

    if(debug)
	fprintf(debug_file, 
	    "slice:  x = %d y = %d  x2 = %d y2 = %d\n", x, y, x2, y2);

    imgp = find_SC(x, y);
    if (debug)
	fprintf(debug_file, "slice: imgp = 0x%p\n", (void *) imgp);
    imgp2 = find_SC(x2, y2);
    if (debug)
	fprintf(debug_file, "slice: imgp2 = 0x%p\n", (void *) imgp2);
    if ((imgp == NULL) || (imgp2 == NULL))
    {
	error1("end points must lie on an image");
	fclose(slice_file);
	return;
    }
    else if (imgp != imgp2) 
    {
	error1("end points fall on different images");
	fclose(slice_file);
	return;
    }

    /* plot the vector */
    overlay_setup('g');
    overlay_color(2);
    next_bundle();
    overlay_dmov((double) x, (double) y);
    overlay_ddraw((double) x2, (double) y2);
    image_setup(plane);
    flush_graphics();

    /* initialize dnmin, dnmax */
    dnmin = DBL_MAX;
    dnmax = -DBL_MAX;
    x_min = x;
    x_max = x;
    y_min = y;
    y_max = y;
    next_bundle();

    if (abs(x - x2) > abs(y - y2))
    {
	/* plot along x axis */
	if (x > x2)
	{
	    xx = x;  /* exchange values */
	    x = x2;
	    x2 = xx;
	    yy = y;
	    y = y2;
	    y2 = yy;
	}

	dn = (double *) malloc((x2+1) * sizeof(double));

	a = ( (double) y2 - y) / (x2 - x);
	if(debug)
	    fprintf(debug_file, 
		"x = %d y = %d  x2 = %d y2 = %d a = %e\n", x, y, x2, y2, a);

	for(xx = x; xx <= x2; xx++)
	{
	    if (control_c)
	    {
		error1("slice interrupted by control c");
		fprintf(session, "\nslice interrupted by control c\n");
		fclose(slice_file);
		free(dn);
		return;
	    }

	    yy = Round(a * (xx - x) + y);
	    d_coord[0] = xx;
	    d_coord[1] = yy;
	    dSC_to_IRAF(d_coord);
	    i_coord[0] = Round(d_coord[0]);
	    i_coord[1] = Round(d_coord[1]);
	    flux_IRAF(i_coord, fluxes);
	    dn[xx] = fluxes[0];

	    if (FINITE(dn[xx]))
	    {
		if (fluxes[0] < dnmin)
		{
		    dnmin = fluxes[0];
		    x_min = xx;
		    y_min = yy;
		}

		if (fluxes[0] > dnmax)
		{
		    dnmax = fluxes[0];
		    x_max = xx;
		    y_max = yy;
		}

	        z = fluxes[0] * bscale + b_zero - backgnd_level;
		++n_average;
		flux_sum += z;
		square_sum += z * z;

		d_coord[0] = xx;
		d_coord[1] = yy;
		dSC_to_IM(d_coord);
		fprintf(slice_file, "%-14.6e %-10.10s at %s", z, bunit,
		       dimage_to_sky(d_coord, s_coord, &coord_sys, 
			   &equinox, &jsys));
		if(debug)
		{
		    fprintf(debug_file, "%-14.6e %s at %s", z, bunit,
			dimage_to_sky(d_coord, s_coord, &coord_sys,
			    &equinox, &jsys));
		}
	    }
	}

	if (debug)
	    fprintf(debug_file, 
		"slice: dnmin = %g  dnmax = %g\n", dnmin, dnmax);

	if (dnmax == dnmin)
	    scale = get_screen_height() / 2;
	else
	    scale = get_screen_height() / ((float) (dnmax - dnmin) * 2);
	    /* half screen high */
	if (oldscale != 0)
	    scale = oldscale;
	bias = get_screen_bottom() - dnmin * scale + 1;
	if (debug)
	    fprintf(debug_file, "slice: scale = %g  bias = %d\n", scale, bias);

	overlay_setup('r');
	overlay_color(1);  /* red */
	overlay_dmov((double) x, (dn[x] * scale + bias));
	for(xx = x; xx <= x2; xx++)
	{
	    if (!FINITE(dn[xx]))
		overlay_ddraw((double) xx, (dnmin * scale + bias));
	    else
		overlay_ddraw((double) xx, (dn[xx] * scale + bias));
	}
	image_setup(plane);
    }
    else
    {
	/* plot along y axis */
	if (y > y2)
	{
	    xx = x;  /* exchange values */
	    x = x2;
	    x2 = xx;
	    yy = y;
	    y = y2;
	    y2 = yy;
	}

	dn = (double *) malloc((y2+1) * sizeof(double));

	a = ( (double) x2 - x) / (y2 - y);
	if(debug)
	    fprintf(debug_file, 
		"x = %d y = %d  x2 = %d y2 = %d a = %e\n", x, y, x2, y2, a);

	for(yy = y; yy <= y2; yy++)
	{
	    if (control_c)
	    {
		error1("slice interrupted by control c");
		fprintf(session, "\nslice interrupted by control c\n");
		fclose(slice_file);
		free(dn);
		return;
	    }

	    xx = Round(a * (yy - y) + x);
	    d_coord[0] = xx;
	    d_coord[1] = yy;
	    dSC_to_IRAF(d_coord);
	    i_coord[0] = Round(d_coord[0]);
	    i_coord[1] = Round(d_coord[1]);
	    flux_IRAF(i_coord, fluxes);
	    dn[yy] = fluxes[0];

	    if (FINITE(dn[yy]))
	    {
		if (fluxes[0] < dnmin)
		{
		    dnmin = fluxes[0];
		    x_min = xx;
		    y_min = yy;
		}

		if (fluxes[0] > dnmax)
		{
		    dnmax = fluxes[0];
		    x_max = xx;
		    y_max = yy;
		}

	        z = fluxes[0] * bscale + b_zero - backgnd_level;
		++n_average;
		flux_sum += z;
		square_sum += z * z;

		d_coord[0] = xx;
		d_coord[1] = yy;
		dSC_to_IM(d_coord);
		fprintf(slice_file, "%-14.6e %-10.10s at %s", z, bunit,
		       dimage_to_sky(d_coord, s_coord, &coord_sys, 
		       &equinox, &jsys));
		if(debug)
		{
		    fprintf(debug_file, "%-14.6e %s at %s", z, bunit,
			dimage_to_sky(d_coord, s_coord, &coord_sys,
			&equinox, &jsys));
		}
	    }
	}

	if (debug)
	    fprintf(debug_file, 
		"slice: dnmin = %g  dnmax = %g\n", dnmin, dnmax);

	if (dnmax == dnmin)
	    scale = get_screen_width() / 2;
	else
	    scale = get_screen_width() / ((float) (dnmax - dnmin) * 2); 
	    /* 640 pixels wide  */
	if (oldscale != 0)
	    scale = oldscale;
	bias = get_screen_left() - dnmin * scale + 1;
	if (debug)
	    fprintf(debug_file, "slice: scale = %g  bias = %d\n", scale, bias);

	overlay_setup('r');
	overlay_color(1);  /* red */
	overlay_dmov((dn[y] * scale + bias), (double) y);
	for(yy = y; yy <= y2; yy++)
	{
	    if (!FINITE(dn[yy]))
	    overlay_ddraw((dnmin * scale + bias), (double) yy);
	    else
	    overlay_ddraw((dn[yy] * scale + bias), (double) yy);
	}
	image_setup(plane);
    }

    free(dn);
    fclose(slice_file);
    fmin = dnmin * bscale + b_zero - backgnd_level;
    fmax = dnmax * bscale + b_zero - backgnd_level;
    d_coord[0] = x_min;
    d_coord[1] = y_min;
    dSC_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    if(server_mode == FALSE)
       printf("\nFlux minimum %-g %s at %s", fmin, bunit, coord_str);
    else
    {
	sprintf(server_str,"%-g", fmin);
	if (ads_server_mode)
	{
	    srv_coord_val("min", coord_sys, s_coord, equinox, jsys,
		server_str, bunit);
	}
	else
	{
	    srv_real("min_pixel_value", server_str);
	    srv_string("units", bunit);
	    srv_string("coord_sys", coord_name(coord_sys));
	    srv_real("min_x", s_coord[0]);
	    srv_real("min_y", s_coord[1]);
	}
    }
    fprintf(session, "\nFlux minimum %-g %s at %s", fmin, bunit, coord_str);
    d_coord[0] = x_max;
    d_coord[1] = y_max;
    dSC_to_IM(d_coord);
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    if(server_mode == FALSE)
       printf("\nFlux maximum %-g %s at %s", fmax, bunit, coord_str);
    else
    {
	sprintf(server_str,"%-g", fmax);
	if (ads_server_mode)
	{
	    srv_coord_val("max", coord_sys, s_coord, equinox, jsys,
		server_str, bunit);
	}
	else
	{
	    srv_real("max_pixel_value", server_str);
	    srv_real("max_x", s_coord[0]);
	    srv_real("max_y", s_coord[1]);
	}
    }
    fprintf(session, "\nFlux maximum %-g %s at %s", fmax, bunit, coord_str);

    average_flux = flux_sum / n_average;
    stdev = sqrt(square_sum / n_average - average_flux * average_flux);
    if(server_mode == FALSE)
       printf("st dev= %g\n", stdev);
    else
    {
	sprintf(server_str,"%g", stdev);
	srv_real("std_dev", server_str);
    }
    fprintf(session, "stdev= %g\n", stdev);
}

void
scatter_plot(argc, argv)
int argc;
char *argv[];
{
    FILE *scatter_file;
    struct img *save_curr_img, tmpimg;
    double flux_x, flux_x2, flux_y, flux_y2;
    double oldbscale, oldbzero, bscale2, bzero2, slope, intercept;
    int fd, image, elts;
    char tmp_name[MAXPATHLEN], sign, sc_fname[MAXPATHLEN];
    char oldbunit[80], s_coord[2][40];
    int coord_sys;
    int indig, infile, outfile;
    int stash_count;
    double *stash0, *stash1, *stashtemp, *stashtemp2;
    int min_samp, max_samp, min_line, max_line;
    double dnmax[2], dnmin[2];
    double fluxes[4], oldflux;
    int i_coord[2];
    int x, y; 
    int jsys;
    double equinox;
    double d_coord[2];
    int INTERIOR=1; 

    if (!graphics)
	return;

    enable_control_c();

    switch(argc)
    {
	case 2:   /* sc file */
	    indig = 0;
	    infile = 1;
	    outfile = 0;
	    break;
	case 3:   /* sc n file OR sc file outfile */
	    if (isdigit(argv[1][0]))
	    {
		indig = 1;
		infile = 2;
		outfile = 0;
	    }
	    else
	    {
		indig = 0;
		infile = 1;
		outfile = 2;
	    }
	    break;
	case 4:   /* sc n file outfile */
	    indig = 1;
	    infile = 2;
	    outfile = 3;
	    break;
	default:
	    error1("scatter_plot command needs a filename");
	    return;

    }

    if (indig)  /* if he specified an area number */
    {
	if (!switch_area(argv[indig]))
	    return;
    }

    if(debug)
	fprintf(debug_file, "scatter_plot:  area number %d \n", area_number);

    if (area[area_number].area_type == 0)
    {
	sprintf(server_str, "scatter_plot error: area %d has not been defined", area_number);
	error1(server_str);
	return;
    }

    /* now test if his file exists */
    strcpy(tmp_name, expand_path(argv[infile]));
    fd = wf_open(tmp_name, &tmpimg, 1);
    if (fd < 0)
    {
	return;
    }
    if (tmpimg.filehdr.tapetype == IMFILE)
	im_close(fd);
    else
	close(fd);

    /* open the output file */
    if (outfile)
    {
	strcpy(sc_fname, expand_path(argv[outfile]));
    }
    else
    {
	strcpy(sc_fname, expand_path("scatter"));
    }
    if (debug)
	fprintf(debug_file, "scatter_plot: opening output file %s\n", sc_fname);
    if ((scatter_file = fopen(sc_fname, "w")) == NULL)
    {
	sprintf(server_str, "Can't open scatter file '%s'", sc_fname);
	error1(server_str);
	return;
    }

    draw_outline('g', 2);   /* of polygon or circle */ 

    find_range( &min_samp, &max_samp, &min_line, &max_line);

    elts = (max_samp - min_samp + 1) * (max_line - min_line + 1);
    if (debug)
	fprintf(debug_file, 
	    "scatter_plot: elts = %d  size = %lu\n", elts, sizeof(fluxes[0]));
    stash0 = (double *) calloc(2 * elts, sizeof(fluxes[0]));
    if (stash0 == NULL)
    {
	error1("scatter_plot error: calloc failed");
	fclose(scatter_file);
	return;
    }
    stashtemp = stash0;
    stashtemp2 = stash0;
    stash_count = 0;

    segment_setup();  /* calculate parameters defining each line */ 

    /* Loop over the possible area of the image */ 

    for (image = 0; image <= 1; image++)
    {

	dnmax[image] = -DBL_MAX;
	dnmin[image] = DBL_MAX;

	if (image == 1)
	{
	    save_curr_img = curr_img;
	    oldbscale = bscale;
	    oldbzero = b_zero;
	    strcpy(oldbunit, bunit);

	    if (debug)
		fprintf(debug_file, 
		    "scatter_plot: opening file %s\n", tmp_name);
	    fd = wf_open(tmp_name, &tmpimg, 1);
	    /* we already checked if it exists */

	    fill_glob(&tmpimg);   /* get globals */
	    /* restore some globals from current image */
	    x_offset = save_curr_img->x_offset;
	    crpix1 = save_curr_img->crpix1;
	    minx = save_curr_img->minx;
	    maxx = save_curr_img->maxx;
	    miny = save_curr_img->miny;
	    maxy = save_curr_img->maxy;
	    sgn_y = save_curr_img->sgn_y;
	    y_offset = save_curr_img->y_offset;
	    crpix2 = save_curr_img->crpix2;
	    band_offset = reqband - 1;       /* I hope this is reasonable */

	    set_file_sys();
	    if (tmpimg.filehdr.tapetype == IMFILE)
		im_close(fd);
	    else
		close(fd);
	    bscale2 = bscale;
	    bzero2 = b_zero;
	    stash1 = stashtemp;
	    stashtemp2 = stash0; /* pointer to values already got */
	}

	for(y=min_line; y <= max_line; y++) 
	{ 
	    x_list(y);     /* list of boundary crossings */ 

	    for(x=min_samp; x <= max_samp; x++) 
	    {
		if(debug_val == 9)
		    fprintf(debug_file, 
			"scatter_plot: at position %d,%d\n", x,y);

		if(position(x)==INTERIOR && !on_horizontal(x, y)) 
		{
		    i_coord[0] = x;
		    i_coord[1] = y;
		    flux_IRAF(i_coord, fluxes);
		    *stashtemp++ = fluxes[0];  /* save it in array */
		    stash_count++;
		    oldflux = *stashtemp2++;  /* flux from 1st file */
		    if(debug_val == 9)
			fprintf(debug_file, 
			    "scatter_plot: flux dn = %g\n", fluxes[0]);

		    if(!FINITE(fluxes[0]))  
			continue;

		    if ((image == 1) && (FINITE(oldflux)))
		    {
			d_coord[0] = i_coord[0];
			d_coord[1] = i_coord[1];
			dIRAF_to_IM(d_coord);

			/* output to file */
			fprintf(scatter_file, "%-14.6e %s %-14.6e %s at %s",
			    oldflux * oldbscale + oldbzero, oldbunit,
			    fluxes[0] * bscale + b_zero, bunit,
			    dimage_to_sky(d_coord, s_coord, &coord_sys,
			    &equinox, &jsys));
			if(debug_val == 9)
			{
			    fprintf(debug_file, "%-14.6e %s %-14.6e %s at %s",
			    oldflux * oldbscale + oldbzero, oldbunit,
			    fluxes[0] * bscale + b_zero, bunit,
			    dimage_to_sky(d_coord, s_coord, &coord_sys,
			    &equinox, &jsys));
			}
		    }

		    /* For useable fluxes */ 

		    if(fluxes[0] > dnmax[image])   /* Maximum */ 
		    {
			dnmax[image] = fluxes[0];

			if(debug)
			     fprintf(debug_file, 
				 "scatter_plot: dnmax= %g\n", dnmax[image]);
		    }

		    if(fluxes[0] < dnmin[image])   /* Minimum */ 
		    {
			dnmin[image] = fluxes[0];

			if(debug)
			    fprintf(debug_file, 
				"scatter_plot: dnmin= %g\n", dnmin[image]);
		    }
		}
		if (control_c)
	        {
		    error1("scatter_plot interrupted by control c");
		    fprintf(session, "\nscatter_plot interrupted by control c\n");
		    free(stash0);
		    curr_img = save_curr_img;
		    fill_glob(curr_img);
		    fclose(scatter_file);
		    return;
	        }
	    }
	} 
    }
    fclose(scatter_file);
    curr_img = save_curr_img;
    fill_glob(curr_img);
    if (debug)
	fprintf(debug_file, "scatter_plot: done collecting points\n");
    if ((dnmax[0] == -DBL_MAX) || (dnmax[1] == -DBL_MAX))
    {
	error1("scatter_plot error:  no valid fluxes in defined area");
	free(stash0);
	return;
    }
    scatter_sub(stash0, stash1, dnmin, dnmax, stash_count, &flux_x, &flux_y,
	&flux_x2, &flux_y2);    /* do machine dependent part */

    free(stash0);
    if (debug)
	fprintf(debug_file, 
	"scatter_plot:  flux_x = %g flux_y = %g flux_x2 = %g flux_y2 = %g\n", 
	flux_x, flux_y, flux_x2, flux_y2);
    flux_x =  flux_x * bscale + b_zero;
    flux_x2 = flux_x2 * bscale + b_zero;
    flux_y =  flux_y * bscale2 + bzero2;
    flux_y2 = flux_y2 * bscale2 + bzero2;
    if (debug)
	fprintf(debug_file, 
	"scatter_plot:  flux_x = %g flux_y = %g flux_x2 = %g flux_y2 = %g\n", 
	flux_x, flux_y, flux_x2, flux_y2);
    if (flux_x2 == flux_x)
	slope = 1;
    else
	slope = (flux_y2 - flux_y) / (flux_x2 - flux_x);
    intercept = flux_y - flux_x * slope;
    sign = ((intercept < 0) ? ' ' : '+');
    if(server_mode == FALSE)
    {
       printf("scatter_plot results:\n");
       printf("  flux from %s =\n", tmp_name);
       printf("   %g * (flux from %s) %c%g\n", slope, filnam, sign, intercept);
    }
    else
    {
       srv_string("filename1", tmp_name);
       srv_string("filename2", filnam);
       sprintf(server_str, "%g", slope);
       srv_real("slope", server_str);
       sprintf(server_str, "%c%g", sign, intercept);
       srv_real("intercept", server_str);
    }
    fprintf(session, "scatter_plot results:\n");
    fprintf(session, "  flux from %s =\n", tmp_name);
    fprintf(session, "   %g * (flux from %s) %c%g\n", slope, filnam, sign, intercept);
}
