/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "job.h"
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <sys/wait.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>

extern int server_mode;
extern char server_str[];


static struct job_info head_job = {&head_job, &head_job};

#ifdef _NO_PROTO
void del_job();
void error1();
void sperror();
#else
void del_job(struct job_info *p);
void error1(char *str);
void sperror(char *str, int this_errno);
#endif /* _NO_PROTO */


/* signal handling interface */

#ifdef NOTDEF
struct sigvec vec;
#define NO_RUPTS	{ vec.sv_handler = SIG_IGN; sigvec(SIGINT, &vec, 0); }
#define OK_RUPTS	{ vec.sv_handler = SIG_DFL; sigvec(SIGINT, &vec, 0); }
#define NO_CTLZ 	{ vec.sv_handler = SIG_IGN; sigvec(SIGTSTP, &vec, 0); }
#endif /* NOTDEF */
#define NO_RUPTS	{ signal(SIGINT,  SIG_IGN); }
#define NO_CTLZ		{ signal(SIGINT,  SIG_IGN); }

                        /* with pipe()'s                              */
                        /* parent writes to fildes[1],                */
			/* child read()'s from fildes[0]              */


/* fork another job with read and write pipes to its stdout and stdin */
/* Returns 1 for success, 0 for failure */
int
duplex_pipe(argv, readfd, writefd, jobptr)	
char *argv[];
FILE **readfd, **writefd;
struct job_info **jobptr;
{
	int this_errno;
	int fildes[2], fildes1[2];
	struct job_info *job;
	FILE *tmpfd;
	int save_errno;
	int status;
	int cont_char;

	/* get an open slot for pid */
	job = add_job();
	*jobptr = job;
	strcpy(job->pidname, argv[0]);

	if (pipe(fildes) == -1)                   /* set-up pipe */
	{
	    this_errno = errno;
	    sprintf(server_str,
		"could not create pipe when forking %s", argv[0]);
	    sperror(server_str, this_errno);
	    return(0);
	}
	if (pipe(fildes1) == -1)                   /* set-up 2nd pipe */
	{
	    this_errno = errno;
	    sprintf(server_str,
		"could not create pipe when forking %s", argv[0]);
	    sperror(server_str, this_errno);
	    return(0);
	}


	if ((job->pid= fork()) == 0)        /* fork off child */
	{
		NO_RUPTS;
		NO_CTLZ;

		/* now we want stdin to point to the pipe */
		fclose(stdin);    /* Trust me, it works. (frees up file 0) */
		if (dup(fildes[0]) == -1)  /* dup gets file 0 (stdin) */
			/* thus hooking stdin to the read half of the pipe */
		{
		    this_errno = errno;
		    sprintf(server_str,
			"could not dup file descripter forking %s", argv[0]);
		    sperror(server_str, this_errno);
		    exit(1);
		}
		close(fildes[0]);/* no longer need this reference to the pipe */
		close(fildes[1]);  /* child wont be writing to this pipe */

		/* same with stdout */
		fclose(stdout);
		dup(fildes1[1]);
		close(fildes1[1]);
		close(fildes1[0]);

		/* Note:  ptrace is used for an immediate indication of   */
		/* whether the exec is successful.  If it is successful,  */
		/* the parent will be awakened from the waitpid           */
		/* If the exec fails, the parent will be awakened from    */
		/* the waitpid by the exit(242).  By testing for the 242  */
		/* the parent can distinguish between wuccess and failure */
		ptrace(0, 0, 0, 0);

		execvp(argv[0], argv); 
		save_errno = errno;
		sprintf(server_str,
		    "could not exec %s", argv[0]);
		sperror(server_str, save_errno);

		tmpfd = fdopen(1, "w"); /* need a FILE* descriptor to stdout */
		fprintf(tmpfd, 
		    "duplex_pipe error: Could not exec %s: %s", 
		    argv[0], strerror(save_errno));
		exit(242);   /* magic exit code for failed exec */
	}
	if (job->pid < 0)
	{
	    save_errno = errno;
	    sprintf(server_str, "could not fork %s", argv[0]);
	    sperror(server_str, save_errno);
	    return(0);
	}
	close(fildes[0]);   /* parent wont read on this one */
	close(fildes1[1]);   /* parent wont write on this one */
	*writefd = fdopen(fildes[1], "w"); /* parent needs a FILE* descriptor */
	job->fpout = *writefd;
	*readfd = fdopen(fildes1[0], "r"); /* parent needs a FILE* descriptor */
	job->fp = *readfd;

	waitpid(job->pid, &status, 0);  /* wait for child to start */
	if ((WIFEXITED(status)) && (WEXITSTATUS(status) == 242))
	{
	    /* Special exit code for failed exec    */
	    /* Child code in duplex_pipe has printed a diagnostic */
	    while (!feof(job->fp))
	    {
		cont_char = getc(job->fp);
		if (cont_char != EOF)
		    putchar(cont_char);
	    }
	    printf("\n");
	    del_job(job);   /* free up the slot */
	    return(0);
	}

	ptrace(7, job->pid, 1, 0);  /* resume execution of child */
	if(!server_mode)
	   printf("%s started as background job  (pid = %d)\n", argv[0], job->pid);
	return(1);
}

void 
check_forks()
{
    int pid1;
    int status;
    struct job_info *job, *next_job;

    pid1 = waitpid((pid_t)-1, &status, WNOHANG);
    if (pid1 > 0)
    {
	job = find_job(pid1);
	if (job != NULL)
	{
	    if (!WIFEXITED(status))
	    {
		if (!server_mode)
		    printf(
		    "process %s aborted   termination code 0x%x  (pid = %d)\n",
		    job->pidname, status, job->pid);
		del_job(job);   /* free up the slot */
	    }

	    else if (WEXITSTATUS(status))
	    {
		if (!server_mode)
		    printf(
		    "process %s exited with non-zero code (%d) (pid = %d)\n",
		    job->pidname, WEXITSTATUS(status), job->pid);
		del_job(job);   /* free up the slot */
	    }

	    else
	    {
		if (!server_mode)
		    printf("process %s completed   (pid = %d)\n",
			job->pidname, job->pid);
		if (job->valid_callback)
		    job->callback(job);
		/* callback may have done del_job()  */
		job = find_job(pid1);   /* see if it is still there */
		if (job != NULL)
		    del_job(job);   /* free up the slot */
	    }
	}
    }

    /* now, since the above may miss the termination if we are     */
    /* suspended (like doing a !ps), we look for the disappearance */
    /* of the pid.						   */
    job = head_job.rlink;
    while (job != &head_job)
    {
	/* kill(pid, 0) does not kill, but does verify the pid */
	next_job = job->rlink;
	if (kill(job->pid, 0))
	{
	    if(!server_mode)
	       printf("process %s completed   (pid = %d)   status unknown\n",
		job->pidname, job->pid);
	    del_job(job);   /* free up the slot */
	}
	job = next_job;
    }
}



struct job_info *add_job()
{
    struct job_info *new_job, *x;

    new_job = (struct job_info *) malloc(sizeof(struct job_info));
    if (new_job == NULL)
    {
	error1("skyview: error allocating space in job table");
	return(new_job);
    }
    /* insert to right of head_job */
    x = &head_job;
    new_job->llink = x;
    new_job->rlink = x->rlink;
    (new_job->rlink)->llink = new_job;
    x->rlink = new_job;

    new_job->fp = NULL;
    new_job->valid_callback = 0;
    return(new_job);
}

void
del_job(p)
struct job_info *p;
{
    (p->llink)->rlink = p->rlink;
    (p->rlink)->llink = p->llink;
    free(p);
}

#ifdef NOTDEF
print_all_pid()
{
    struct job_info *p;

    p = head_job.rlink;
    while (p != &head_job)
    {
	printf("pid = %d\n", p->pid);
	p = p->rlink;
    }
    printf("\n");
}
#endif /* NOTDEF */

struct job_info *find_job(pid)
int pid;
{
    struct job_info *p;

    p = find_first_job();
    while (p != NULL)
    {
	if (p->pid == pid)
	    return(p);
	p = find_next_job(p);
    }
    return( (struct job_info *) 0);
}

struct job_info *find_first_job()
{
    struct job_info *psave_local;

    psave_local = head_job.rlink;
    if (psave_local == &head_job)
	return( (struct job_info *) 0);
    else
	return(psave_local);
}

struct job_info *find_next_job(psave_local)
struct job_info *psave_local;
{
    psave_local = psave_local->rlink;
    if (psave_local == &head_job)
	return( (struct job_info *) 0);
    else
	return(psave_local);
}

void
add_callbacks(fdset)
fd_set *fdset;
{
    struct job_info *p;

    p = find_first_job();
    while (p != NULL)
    {
	if (p->fp != NULL)
	    FD_SET(fileno(p->fp), fdset);
	p = find_next_job(p);
    }
}

void
do_callback(fdset)
fd_set *fdset;
{
    struct job_info *p;

    p = find_first_job();
    while (p != NULL)
    {
	if (p->fp != NULL)
	{
	    if (FD_ISSET(fileno(p->fp), fdset))
	    {
		FD_CLR(fileno(p->fp), fdset);  /* prevent repeated calls */
		if (p->valid_callback)
		    p->callback(p);

		/* now start over through job list since callback may */
		/* have done del_job(p), rendering p invalid */
		p = find_first_job();
		continue;
	    }
	}
	p = find_next_job(p);
    }
}
