/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* machine dependent code for SUN SPARCstation running X windows */ 

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>     /* includes strings.h, sys/time.h, and sys/file.h */
#include <X11/keysym.h>
#include <X11/cursorfont.h>

#include "skyview.h"
#include "parse.h"
#include "img_x.h"
#include "img.h"
#include "area.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <sys/types.h>

#define SMALL 1
#define OK 0

extern FILE *session, *debug_file;
extern int s1_server_mode;
extern Window sc_win;
extern Window cur_window; /* window last receiving tty or button input */
extern int debug, graphics;
extern int server_mode;
extern char server_str[];
extern int minx, maxx, miny, maxy;
extern char last_coord[2][40];
extern char plane;


static char Id[] = "@(#)define_x.c   RBH        11/07/95>";

extern int area_number;
extern struct area_struct area[MAX_AREAS];
extern double cdelt1;

static int sc_exposed;
static short *sc_data = NULL;
static unsigned int sc_width, sc_height;
static double circle_x_vals[16];
static double circle_y_vals[16];
static double   dtr = M_PI/180; 

extern Display *display;
extern int screen;

extern Atom wm_delete_window;

extern GC gcxor, gc_blue, gc_green;
#define CENTER 0
#define SIZE   1
#define ANGLE  2


/*  -- everything below here appears both globally and in image structure -- */
extern Window win;
extern unsigned int win_width, win_height;
extern double win_zoom;
extern char filnam[], mod_filnam[];
extern int image_frame;


/* function prototypes */
#ifdef _NO_PROTO

static void draw_line();
static void sc_vector();
static void create_sc_window();
static void make_unit_circle();

#else

static void draw_line(Window box_win, GC gc, 
    int xx0, int yy0, int xx1, int yy1);
static void sc_vector(int i_coord[], int i_coord2[], Window wind, 
    unsigned int wind_width, unsigned int wind_height);
static void create_sc_window(void);
static void make_unit_circle(void);

#endif /* _NO_PROTO */

void
set_box(xx1, yy1, xx2, yy2)
double *xx1, *yy1, *xx2, *yy2;
{
    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int x, y;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    int cen_size;
    int x_box_size, y_box_size, max_x_size, max_y_size;
    double d_coord[2]; 
 
    if (!set_current_window())
    {
	error1("no images currently being displayed");

	*xx1 = 0;
	*yy1 = 0;
	*xx2 = 0;
	*yy2 = 0;
	return;
    }

    cen_size = CENTER;

    XQueryPointer(display, win,
	&root, &child, &root_x, &root_y, 
	&x_center, &y_center, &keys_buttons);

    x = x_center;
    y = y_center;

    x_size = win_width;
    y_size = win_height;

    max_x_size = x_size;
    max_y_size = y_size;

    x_box_size = x_size / 8;
    y_box_size = y_size / 8;

    set_motion_notify(win);
    done = 0;
    while(!done)
    {
	if (debug)
	{
	    fprintf(debug_file, "drawing box x_center = %d y_center = %d",
		x_center, y_center);
	    fprintf(debug_file, " x_box_size = %d y_box_size = %d\n",
		x_box_size, y_box_size);
	}
	draw_box(win, gcxor, x_center, y_center, x_box_size, y_box_size);

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	/* undraw old line */
	draw_box(win, gcxor, x_center, y_center, x_box_size, y_box_size);

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    /* jump to old center - better than a snap with 1st motion */
	    XWarpPointer(display, None, win, 0,0,0,0, x_center, y_center);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    if (debug)
		fprintf(debug_file, "set_box:  x = %d  y = %d\n", x, y);

	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		x_box_size += dx;
		y_box_size += dy;
	    }


	    /* Limit the box to these limits */

	    if(x_box_size < 0)
	    x_box_size = 0;

	    if(x_box_size > max_x_size)
	    x_box_size = max_x_size;

	    if(y_box_size < 0)
	    y_box_size = 0;

	    if(y_box_size > max_y_size)
	    y_box_size = max_y_size;

	    break;
	}

    }
 
    clear_motion_notify();
 

	       /* RBH Must Check range on x,y */

    d_coord[0] = x_center - x_box_size / 2;
    d_coord[1] = y_center - y_box_size / 2;
    dWIN_to_SC(d_coord);
    *xx1 = d_coord[0];
    *yy1 = d_coord[1];

    d_coord[0] = x_center + x_box_size / 2;
    d_coord[1] = y_center + y_box_size / 2;
    dWIN_to_SC(d_coord);
    *xx2 = d_coord[0];
    *yy2 = d_coord[1];
}

/* CIRCLE 
** uses the trackball to determine a circle to be used as 
** the outline of an area to be examined 
*/ 
 
void
circle(radius) 
double radius;
{ 
    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int rad, drad; 
    double d_coord[2]; 
    int x, y;
    int dx, x_center;
    int dy, y_center;
    int cen_size;
    int max_rad;
    struct img *imgp;
 
    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }
 
     if(radius == -1.)
     {
	 /* no radius specified */
	 rad = win_height / 8;
	 max_rad = win_height / 2;
     }
     else
     {
	 /* Translate the specified radius in arcmin to pixels */

	 if (cdelt1 == 0)
	     rad = radius * win_zoom;
	 else
	     rad = (radius / (60.0 * fabs(cdelt1))) * win_zoom;
     }

     if(debug == TRUE)
     {
	 fprintf(debug_file, "circle: input radius = %-g\n", radius);
	 fprintf(debug_file, "circle: equals %d pixels\n", rad);
	 fprintf(debug_file, "circle: (cdelt1 = %-g arcmin)\n", cdelt1);
     }


     /* Go into rubber band mode */ 
 
     cen_size = CENTER;

    XQueryPointer(display, win,
	&root, &child, &root_x, &root_y, 
	&x_center, &y_center, &keys_buttons);

    x = x_center;  /* to make purify happy */
    y = y_center;  /* to make purify happy */


    set_motion_notify(win);
    done = 0;
    while(!done)
    {
	if (debug)
	    fprintf(debug_file, 
		"drawing circle x_center = %d y_center = %d rad = %d\n",
		x_center, y_center, rad);

	draw_circle(win, gcxor, x_center, y_center, 
	    (unsigned int) rad);

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	/* undraw old circle */
	draw_circle(win, gcxor, x_center, y_center, 
	    (unsigned int) rad );

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    /* jump to old center - better than a snap with 1st motion */
	    XWarpPointer(display, None, win, 0,0,0,0, x_center, y_center);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    if (radius == -1.)
		cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    if (debug)
		fprintf(debug_file, "circle:  x = %d  y = %d\n", x, y);

	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		drad = dx;
		if(dy*dy > dx*dx)
		drad = dy;

		rad += drad;

		/* Limit the radius */

		if(rad < 0)
		    rad = 0;

		if(rad > max_rad)
		    rad = max_rad;
	    }
	}

    } 
    clear_motion_notify();

    d_coord[0] = x_center;
    d_coord[1] = y_center;
    dWIN_to_SC(d_coord);
 
 
    /* First find image on which the center lies */
    if (debug)
	fprintf(debug_file, "circle: looking for center x= %d y= %d\n",
	    (int) d_coord[0], (int) d_coord[1]);
    imgp = find_SC((int) d_coord[0], (int) d_coord[1]);
    if (imgp == NULL)
    {
	error1("center must lie on an image");
	return;
    }
    if (debug)
	fprintf(debug_file, "circle: image came from %s\n", imgp->filnam);

    /* Load area definition arrays */ 

    area[area_number].area_type = CIRCLE;
    area[area_number].radius = rad / win_zoom;

    dSC_to_IM(d_coord);
    area[area_number].x[1] = d_coord[0]; 
    area[area_number].y[1] = d_coord[1]; 

    describe_area(area_number);
} 

/* ELLIPSE
** uses the trackball to determine an ellipse to be used as 
** the outline of an area to be examined 
*/ 
 
void
ellipse()
{ 
    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int x, y;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    int cen_size;
    int max_x_size, max_y_size;
    double d_coord[2]; 
    double angle;
    int width, height;
    double xdist, ydist;
 
    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    cen_size = CENTER;

    XQueryPointer(display, win,
	&root, &child, &root_x, &root_y, 
	&x_center, &y_center, &keys_buttons);

    x = x_center;
    y = y_center;

    x_size = win_width;
    y_size = win_height;

    max_x_size = x_size;
    max_y_size = y_size;

    width = x_size / 4;
    height = y_size / 4;
    angle = 0;

    set_motion_notify(win);
    done = 0;
    while(!done)
    {
	draw_ellipse(win, gcxor, x_center, y_center, 
	    width , height , angle);

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	/* undraw old line */
	draw_ellipse(win, gcxor, x_center, y_center, 
	    width , height , angle);

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    /* jump to old center - better than a snap with 1st motion */
	    XWarpPointer(display, None, win, 0,0,0,0, x_center, y_center);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    if (cen_size == SIZE)
		cen_size = ANGLE;
	    else
		cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    switch(cen_size)
	    {
		case CENTER:
		    x_center = x;
		    y_center = y;
		    break;

		case SIZE:
		    width += dx;
		    height += dy;
		    break;

		case ANGLE:
		    xdist = x - x_center;
		    ydist = y - y_center;
		    if (fabs(xdist) > 30)
			angle += 50 * (double)dy/xdist;
		    if (fabs(ydist) > 30)
			angle += 50 * (double)dx/ydist;
		    break;
	    }


	    /* Limit the box to these limits */

	    if(width < 0)
	    width = 0;

	    if(width > max_x_size)
	    width = max_x_size;

	    if(height < 0)
	    height = 0;

	    if(height > max_y_size)
	    height = max_y_size;

	    break;
	}

    }
 
    clear_motion_notify();


    area[area_number].area_type = ELLIPSE;

    d_coord[0] = x_center;             /* center */    
    d_coord[1] = y_center;
    dWIN_to_SC(d_coord);
    dSC_to_IM(d_coord);
    area[area_number].x[1] = d_coord[0]; 
    area[area_number].y[1] = d_coord[1]; 

    area[area_number].x[2] = width / win_zoom;
    area[area_number].y[2] = height / win_zoom;

    area[area_number].x[3] = angle;

    describe_area(area_number);
} 

 
/* POLYGON 
** uses the trackball to define a set of connected points to 
** be used to outline an area to be examined 
*/ 
 
void
polygon() 
{ 
    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int n_area;
    double d_coord[2]; 
    int x, y, oldx0, oldy0;
    int xcur, ycur;
    int dummy_dx, dummy_dy;
    struct img *imgp, *imgp_sav;
    char *coord_str, s_coord[2][40];
    int coord_sys;
    int jsys;
    double equinox;

    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }

    n_area = 0;
    imgp_sav = NULL;


    XQueryPointer(display, win,
	&root, &child, &root_x, &root_y, 
	&xcur, &ycur, &keys_buttons);

    oldx0=xcur; 
    oldy0=ycur; 

    x = xcur;   /* make purify happy */
    y = ycur;   /* make purify happy */


    set_motion_notify(win);
    done = 0;
    while (!done) 
    {
 
	draw_line(win, gcxor, oldx0, oldy0, xcur, ycur);

	wait_for_event(anin, &what, &x, &y, &dummy_dx, &dummy_dy, 0);

	/* undraw old line */
	draw_line(win, gcxor, oldx0, oldy0, xcur, ycur);

	switch(what)
	{
	case Button1:

	    d_coord[0] = xcur;
	    d_coord[1] = ycur;
	    dWIN_to_SC(d_coord);
	    imgp = find_SC((int) d_coord[0], (int) d_coord[1]);

	    if (imgp_sav == NULL)
	    {
		imgp_sav = imgp;  /* save first vertex info */
	    }

	    if (imgp_sav == NULL)
	    {
		if(server_mode == FALSE)
		   printf("\7first vertex must lie on an image\n");
	    }
	    else if ((imgp != imgp_sav) && (imgp != NULL))
	    {
		fill_glob(imgp_sav);   /* restore globals for first image */
		if(server_mode == FALSE)
		   printf("\7vertices fall on multiple images\n");
	    }
	    else
	    {
		draw_line(win, gc_green, oldx0, oldy0, xcur, ycur);
		d_coord[0] = xcur;
		d_coord[1] = ycur;
		dWIN_to_SC(d_coord);
		dSC_to_IM(d_coord);
		++n_area; 
		area[area_number].x[n_area] = d_coord[0]; 
		area[area_number].y[n_area] = d_coord[1]; 

		coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys,
		    &equinox, &jsys);
		if(server_mode == FALSE)
		   printf(" vertex# %d  at %s\n", n_area, coord_str);
		fprintf(session, " vertex# %d  at %s\n", n_area, coord_str);
		if(debug)
		    fprintf(debug_file, 
			"polygon:  vertex#%d  xcur=%d, ycur=%d\n",
			n_area, xcur, ycur);

		oldx0=xcur;
		oldy0=ycur;
	    }
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    break;
	case POINTERMOTION:

	    xcur = x;
	    ycur = y;

	    if(n_area == 0) 
	    { 
		oldx0=xcur; 
		oldy0=ycur; 
	    } 
	} 
    }
    clear_motion_notify();


    /*  complete the polygon by copying the 1st vertex to a new last vertex */
    ++n_area; 
    area[area_number].x[n_area] = area[area_number].x[1]; 
    area[area_number].y[n_area] = area[area_number].y[1]; 

    area[area_number].area_type = POLYGON;
    area[area_number].vertices = n_area; 

    describe_area(area_number);
} 
 
/*********************************************************************/ 
/*                                                                   */ 
/*  DRAW_LINE   draws a line     on the current image.               */ 
/*  The line   does not change as image is zoomed.                   */ 
/*                                                                   */  
/*  Input: x, y, x, y (integer) in WIN coordinates                   */ 
/*         gc  (graphics context)                                    */ 
/*         box_win (window id)                                       */ 
/*                                                                   */ 
/*********************************************************************/

static void
draw_line(box_win, gc, xx0, yy0, xx1, yy1)
Window box_win;
GC gc;
int xx0, yy0, xx1, yy1;
{
    if (graphics)
	XDrawLine(display, box_win, gc, xx0, yy0, xx1, yy1);
}


/*********************************************************************/ 
/*                                                                   */ 
/*  DRAW_CIRCLE draws a circle   on the current image.               */ 
/*  The circle does not change as image is zoomed.                   */ 
/*                                                                   */  
/*  Input: x, y (integer) in WIN coordinates                         */ 
/*         radius (integer) in WIN coordinates                       */ 
/*         gc  (graphics context)                                    */ 
/*         circle_win (window id)                                    */ 
/*                                                                   */ 
/*********************************************************************/

void
draw_circle(circle_win, gc, x, y, radius)
Window circle_win;
GC gc;
int x, y;
unsigned int radius;
{
    if (graphics)
	XDrawArc(display, circle_win, gc, x - (int) radius, y - (int) radius,
	    2 * radius, 2 * radius, 0, 360*64);
}


/*********************************************************************/ 
/*                                                                   */ 
/*  DRAW_ARC draws an arc   on the current image.                    */ 
/*  The arc does not change as image is zoomed.                      */ 
/*                                                                   */  
/*  Input: x, y (integer) in WIN coordinates                         */ 
/*         radius (integer) in WIN coordinates                       */ 
/*         starting angle (degrees)                                  */ 
/*         extent of angle (degrees)                                 */ 
/*         gc  (graphics context)                                    */ 
/*         circle_win (window id)                                    */ 
/*                                                                   */ 
/*********************************************************************/

void
draw_arc(arc_win, gc, x, y, radius, angle1, angle_extent)
Window arc_win;
GC gc;
int x, y;
double angle1, angle_extent;
unsigned int radius;
{
    if (graphics)
	XDrawArc(display, arc_win, gc, x - (int) radius, y - (int) radius,
	    2 * radius, 2 * radius, angle1*64, angle_extent*64);
}

/*********************************************************************/ 
/*                                                                   */ 
/*  DRAW_ELLIPSE draws an ellipse  on the current image.             */ 
/*  The ellipse does not change as image is zoomed.                  */ 
/*                                                                   */  
/*  Input: x, y (integer) in WIN coordinates                         */ 
/*         height, width (integer) in WIN coordinates                */ 
/*         angle in degrees                                          */ 
/*         gc  (graphics context)                                    */ 
/*         circle_win (window id)                                    */ 
/*                                                                   */ 
/*********************************************************************/

void
draw_ellipse(ellipse_win, gc, x, y, width, height, angle)
Window ellipse_win;
GC gc;
int x, y;
double angle;
int width, height;
{
    XPoint points[65];

    if (!graphics)
	return;

#ifdef NOTDEF
    XDrawArc(display, ellipse_win, gc, x - width/2, y - height/2,
	width, height, 0, 360*64);
#endif
    
    get_ellipse_points((double) x, (double) y, (double) (width/2),
    (double) (height/2), angle, points);
    XDrawLines(display, ellipse_win, gc, 
	points, 65, CoordModeOrigin);
}


/* the next two functions in their original form carried the message: */
/* Copyright:	1989 Smithsonian Astrophysical Observatory
 *		You may do anything you like with this file except remove
 *		this copyright.  The Smithsonian Astrophysical Observatory
 *		makes no representations about the suitability of this
 *		software for any purpose.  It is provided "as is" without
 *		express or implied warranty.
 * Modified:	{0} Michael VanHilst	initial version		   9 May 1989
 *		{n} <who> -- <does what> -- <when>
 */

void get_ellipse_points(x, y, half_width, half_height, angle, points)
double x, y, half_width, half_height, angle;
XPoint *points;
{
    double sine_angle, cos_angle;
    double xval, yval;
    register int i;
    register int x1, x2, y1, y2;
    static int init_done = FALSE;

    if (!init_done)
    {
	make_unit_circle();
	init_done = TRUE;
    }

    sine_angle = -sin(angle * dtr);
    cos_angle = cos(angle * dtr);

    for(i = 0; i < 16; i++) 
    {
	xval = circle_x_vals[i] * half_width;
	yval = circle_y_vals[i] * half_height;
	x1 = (xval * cos_angle)  - (yval * sine_angle);
	y1 = (xval * sine_angle) + (yval * cos_angle);
	x2 = (xval * cos_angle)  + (yval * sine_angle);
	y2 = (xval * sine_angle) - (yval * cos_angle);
	points[i].x = x + x1;
	points[i].y = y + y1;
	points[32+i].x = x - x1;
	points[32+i].y = y - y1;
	points[31-i].x = x + x2;
	points[31-i].y = y + y2;
	points[63-i].x = x - x2;
	points[63-i].y = y - y2;
    }
    points[64].x = points[0].x;
    points[64].y = points[0].y;
}

static void
make_unit_circle ()
{
    int i;
    double angle, delta_angle;

    delta_angle = M_PI / 32.0;
    angle = delta_angle / 2.0;
    for(i = 0; i <= 16; i++) 
    {
	circle_x_vals[i] = -sin(angle);
	circle_y_vals[i] = -cos(angle);
	angle += delta_angle;
    }
}

/*********************************************************************/ 
/*                                                                   */ 
/*  DRAW_BOX draws a rectangle   on the current image.               */ 
/*  The box does not change as image is zoomed.                      */ 
/*                                                                   */  
/*  Input: x, y (integer) in WIN coordinates                         */ 
/*         box_width, box_height (integer) in WIN coordinates        */ 
/*         gc  (graphics context)                                    */ 
/*         box_win (window id)                                       */ 
/*                                                                   */ 
/*********************************************************************/

void
draw_box(box_win, gc, x, y, box_width, box_height)
Window box_win;
GC gc;
int x, y, box_width, box_height;
{
    XPoint corner[5];
    int half_width, half_height;

    if (!graphics)
	return;

    half_width = box_width / 2;
    half_height = box_height / 2;
    corner[0].x = x - half_width;
    corner[0].y = y - half_height;
    corner[1].x = x + half_width;
    corner[1].y = y - half_height;
    corner[2].x = x + half_width;
    corner[2].y = y + half_height;
    corner[3].x = x - half_width;
    corner[3].y = y + half_height;
    corner[4].x = x - half_width;
    corner[4].y = y - half_height;
    XDrawLines(display, box_win, gc, corner, 5, CoordModeOrigin);

}


/* DRAW_OUTLINE 
** uses the definition of an area to draw the outline 
*/ 
 
void
draw_outline(overlay_plane, color) 
int overlay_plane;
int color;
{       
    int i; 
    int area_type; 
    double d_coord[2];


    if (color == 0)
	return;       /* no need to erase graphics in X (they were xor) */

    overlay_setup(overlay_plane);
    overlay_color(color);
    next_bundle();

    area_type = area[area_number].area_type; 

    if(area_type==POLYGON)                    /* Draw a polygon */ 
    {
	for(i=1; i <= area[area_number].vertices; ++i) 
	{ 
	    d_coord[0] = area[area_number].x[i];
	    d_coord[1] = area[area_number].y[i];
	    dIM_to_SC(d_coord);

	    if(debug)
		fprintf(debug_file, 
		    "draw_outline: polygon point#%d  x=%g, y=%g \n", 
			i, d_coord[0], d_coord[1]);

	    if(i==1) 
		overlay_dmov(d_coord[0], d_coord[1]); 
	    else
		overlay_ddraw(d_coord[0], d_coord[1]); 
	} 
    }

    else if(area_type==CIRCLE)                /* Draw a circle */ 
    { 
	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];
	dIM_to_SC(d_coord);

	if(debug)
	    fprintf(debug_file, 
		"draw_outline:  circle center (%g,%g), radius %g \n", 
		d_coord[0], d_coord[1], area[area_number].radius);

	overlay_dmov(d_coord[0], d_coord[1]); 
	overlay_dcircle(area[area_number].radius); 
    } 


    else if(area_type==ELLIPSE)
    {
	d_coord[0] = area[area_number].x[1];
	d_coord[1] = area[area_number].y[1];
	dIM_to_SC(d_coord);

	if(debug)
	    fprintf(debug_file, 
	    "draw_outline:  ellipse center (%g,%g), width %g height %g angle %g\n",
		d_coord[0], d_coord[1],
		area[area_number].x[2], area[area_number].y[2], 
		area[area_number].x[3]);

	overlay_dmov(d_coord[0], d_coord[1]); 
	overlay_dellipse(area[area_number].x[2], area[area_number].y[2], 
	    area[area_number].x[3]);
    }
}


/******************************************************************/
/*                                                                */
/* TRACKBALL_PICK                                                 */
/* uses the mouse to define a point for flux extraction           */
/*                                                                */
/* Returns position x,y in SC coordinates                         */
/* Returns position winx,winy in WIN coordinates                  */
/* Loads globals for the image                                    */
/*                                                                */
/******************************************************************/
 

int
trackball_pick(x, y, winx, winy) 
int *x, *y;
int *winx, *winy;
{ 
    char anin[MAXLINE];
    struct img *savimg, *tmpimg;
    int dummy_dx, dummy_dy;
    int i_coord[2], what;

    i_coord[0] = 0;  /* make purify happy */
    i_coord[1] = 0;  /* make purify happy */

    wait_for_event(anin, &what, &i_coord[0], &i_coord[1], &dummy_dx, &dummy_dy,
	0);

    if(what == Button1)
    {
	*winx = i_coord[0];
	*winy = i_coord[1];

	/* find most recent image in this window with given x,y */

	/* coords are in WIN space as they must be when img struct changes */
	tmpimg = find_WIN(cur_window); /* find last image in this window */
	if (tmpimg == NULL)
	    return(FALSE);  /* clicking button 1 on any non-image window */
			    /* stops the pick   */
	savimg = tmpimg;

	WIN_to_SC(i_coord);
	while ((tmpimg!=NULL) && 
	    ( (i_coord[0] < minx) || (i_coord[0] > maxx) || 
	      (i_coord[1] < miny) || (i_coord[1] > maxy) ))
	{
	    SC_to_WIN(i_coord);
	    /* look for another image in this window */
	    tmpimg = find_WIN_again(tmpimg);
	    /* if not found - go with the last image painted into this window */
	    if (tmpimg == NULL)
	    {
		fill_glob(savimg);
	    }
	    WIN_to_SC(i_coord);
	}
	*x = i_coord[0];
	*y = i_coord[1];
	return(TRUE);
    }
    else
    {
	return(FALSE);
    }
} 

void
trackball_mark()

{
    /* undocumented trackball mark */
    /*  - like pick, but with subpixel accuracy, and no flux report */

    int status, i_coord[2], xcur, ycur, win_x, win_y;
    double d_coord[2];
    int markertype, markersize, markercolor;
    char overlay_plane, *coord_str;
    char s_coord[2][40];
    int coord_sys;
    char tmp_filnam[MAXPATHLEN + 12];
    int jsys;
    double equinox;

    markertype = CROSS;
    markersize = 7;
    markercolor = 2;
    overlay_plane = 'g';

    /* for trackball mark, start cursor in center of visible screen */
    get_center_xy(i_coord);
    xcur = i_coord[0];
    ycur = i_coord[1];

    if (s1_server_mode)
    {
	srvflush("mark", 1, 0);
	/* table file header */
	printf("table\n");
	printf(
	"|x           |y           |coord_sys |");
	printf("frame |filename");
	printf("                                                     ");
	printf("                                                     |\n");
    }

    for (;;)
    {
	status = trackball_pick(&xcur, &ycur, &win_x, &win_y);
	if (status == FALSE)
	    break;
	d_coord[0] = win_x;
	d_coord[1] = win_y;
	dWIN_to_SC(d_coord);
	overlay_setup(overlay_plane);
	overlay_color(markercolor);
	next_bundle();
	do_mark(d_coord[0], d_coord[1], markertype, markersize, 0, 0);
	image_setup(plane);
	dSC_to_IM(d_coord);
	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
	strcpy(last_coord[0], s_coord[0]);
	strcpy(last_coord[1], s_coord[1]);
	if(server_mode == FALSE)
	   printf("Mark at %s\n", coord_str);
	else
	{
	    if (s1_server_mode)
	    {
		/* output table file line */
		strcpy(tmp_filnam, filnam);
		if (mod_filnam[0] != '\0')
		    strcat(tmp_filnam, " (modified)");
		printf("%-12.12s %-12.12s %-10.10s %-3d    %-110.110s\n",
		    s_coord[0], s_coord[1], coord_name(coord_sys),
		    image_frame, tmp_filnam);
	    }
	    else
	    {
		/* ads server mode */
		srv_string("state", "point");
		srv_coord("coord", coord_sys, s_coord, equinox, jsys);
		sprintf(server_str, "%d", image_frame);
		srv_real("frame", server_str);
		sprintf(server_str, "%s%s",
		    filnam, mod_filnam[0]?" (modified)":"");
		srv_string("filename", server_str);

		srvflush("mark", 0, 0);
		srvinit();
	    }
	}
    }
    if (server_mode)
    {
	if (s1_server_mode)
	{
	    printf("table_end\n");
	}
	else
	{
	    srv_string("state", "done");
	}
    }
}



/* SL_VECTOR
** uses the trackball to determine a vector to be used in the slice
** command
*/                                             
 
void
sl_vector(d_coord, d_coord2) 
double d_coord[2], d_coord2[2]; 
{ 
    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int x, y;
    int x1cur, y1cur, x2cur, y2cur;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    int cen_size;
    int x_box_size, y_box_size;
 
    if (!set_current_window())
    {
	error1("no images currently being displayed");
	d_coord[0] = 0;
	d_coord[1] = 0;
	d_coord2[0] = 0;
	d_coord2[1] = 0;
	return;
    }

     cen_size = CENTER;

    XQueryPointer(display, win,
	&root, &child, &root_x, &root_y, 
	&x_center, &y_center, &keys_buttons);

     x_size = win_width;
     y_size = win_height;

     x_box_size = x_size / 8;
     y_box_size = y_size / 8;

    x = x_center;  /* make purify happy */
    y = y_center;  /* make purify happy */

    set_motion_notify(win);
    done = 0;
    while(!done)
    {
	x1cur = x_center - x_box_size / 2;
	y1cur = y_center - y_box_size / 2;

	x2cur = x_center + x_box_size / 2;
	y2cur = y_center + y_box_size / 2;

	draw_line(win, gcxor, x1cur, y1cur, x2cur, y2cur);

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	/* undraw old line */
	draw_line(win, gcxor, x1cur, y1cur, x2cur, y2cur);

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    /* jump to old center - better than a snap with 1st motion */
	    XWarpPointer(display, None, win, 0,0,0,0, x_center, y_center);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    if (debug)
		fprintf(debug_file, "sl_vector:  x = %d  y = %d\n", x, y);

	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		x_box_size += dx;
		y_box_size -= dy;
	    }
	    break;
	}

    }
 
    clear_motion_notify();

    d_coord[0] = x1cur;
    d_coord[1] = y1cur;
    dWIN_to_SC(d_coord);

    d_coord2[0] = x2cur;
    d_coord2[1] = y2cur;
    dWIN_to_SC(d_coord2);
} 

/* sc_vector does the vector in the scatter_plot command in X windows */
static void
sc_vector(i_coord, i_coord2, wind, wind_width, wind_height) 
Window wind;
int i_coord[2], i_coord2[2]; 
unsigned wind_width, wind_height;
{ 

    char anin[MAXLINE];
    unsigned int keys_buttons;
    int done, what;
    Window root, child;
    int root_x, root_y;
    int x, y;
    int x1cur, y1cur, x2cur, y2cur;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    int cen_size;
    int x_box_size, y_box_size;
 

     cen_size = CENTER;

    XQueryPointer(display, wind,
	&root, &child, &root_x, &root_y, 
	&x_center, &y_center, &keys_buttons);

     x_size = wind_width;
     y_size = wind_height;

     x_box_size = x_size / 8;
     y_box_size = - (y_size / 8);

    x = x_center;  /* make purify happy */
    y = y_center;  /* make purify happy */

    set_motion_notify(wind);
    done = 0;
    while(!done)
    {
	x1cur = x_center - x_box_size / 2;
	y1cur = y_center - y_box_size / 2;

	x2cur = x_center + x_box_size / 2;
	y2cur = y_center + y_box_size / 2;

	draw_line(wind, gcxor, x1cur, y1cur, x2cur, y2cur);

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	/* undraw old line */
	draw_line(wind, gcxor, x1cur, y1cur, x2cur, y2cur);

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    /* jump to old center - better than a snap with 1st motion */
	    XWarpPointer(display, None, wind, 0,0,0,0, x_center, y_center);
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    if (debug)
		fprintf(debug_file, "sc_vector:  x = %d  y = %d\n", x, y);

	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		x_box_size += dx;
		y_box_size -= dy;
	    }
	    break;
	}

    }
 
    clear_motion_notify();

    i_coord[0] = x1cur;
    i_coord[1] = y1cur;

    i_coord2[0] = x2cur;
    i_coord2[1] = y2cur;
 
    XDrawLine(display, sc_win, gc_green, x1cur, y1cur, x2cur, y2cur);
} 

void
scatter_sub(stashtemp, stashtemp1, dnmin, dnmax, stash_count, x, y, 
    x2, y2)
double *stashtemp, *stashtemp1, dnmin[2], dnmax[2];
int stash_count;
double *x, *y, *x2, *y2;
{
    int i_coord[2], i_coord2[2];
    int i, xcur, ycur, bias0, bias1;
    double scale0, scale1;
    int x_orig = 0, y_orig = 0;
    short *sc_data_p;

    create_sc_window();

    if (dnmax[0] == dnmin[0])
	dnmax[0] = dnmin[0] + 1;
    scale0 = (sc_width - 20) / (dnmax[0] - dnmin[0]);
    bias0 = x_orig - dnmin[0] * scale0 + 7;
    if (dnmax[1] == dnmin[1])
	dnmax[1] = dnmin[1] + 1;
    scale1 = (sc_height - 20) / (dnmax[1] - dnmin[1]);
    bias1 = y_orig - dnmin[1] * scale1 + 7;
    if (debug)
    {
	fprintf(debug_file, 
	    "scatter_plot: scale0 = %g  bias0 = %d\n", scale0, bias0);
	fprintf(debug_file,
	    "scatter_plot: scale1 = %g  bias1 = %d\n", scale1, bias1);
    }

    /* prepare a block of memory for saving graphics	*/
    /* format:  n x y x y x y ... X Y X Y		*/
    /*  where n = number of crosses			*/
    /*        x y = window coords of a cross 		*/
    /*        X Y = endpoint of user drawn line		*/
    if (sc_data != NULL)
	free(sc_data);
    sc_data = (short *) malloc((stash_count+5) * sizeof(short));
    sc_data_p = sc_data + 1;
    *sc_data = 0;

    for (i = 0; i < stash_count / 2; i++)
    {
	if (debug)
	    fprintf(debug_file, 
		"scatter_plot: first image = %g second image = %g\n", 
		*stashtemp, *stashtemp1);
	if ((FINITE(*stashtemp)) && (FINITE(*stashtemp1)))
	{
	    xcur = *stashtemp * scale0 + bias0;
	    ycur = *stashtemp1 * scale1 + bias1;
	    ycur = sc_height - ycur;   /* upside down */
	    if (debug)
		fprintf(debug_file, "xcur = %d  ycur = %d\n", xcur, ycur);

	    /* save cross locations */
	    (*sc_data)++;   /* increment count */
	    *sc_data_p++ = xcur;
	    *sc_data_p++ = ycur;

	    XDrawLine(display, sc_win, gc_blue, xcur-3, ycur, xcur+3, ycur);
	    XDrawLine(display, sc_win, gc_blue, xcur, ycur-3, xcur, ycur+3);
	}
	stashtemp++;
	stashtemp1++;
    }

    sc_vector(i_coord, i_coord2, sc_win, sc_width, sc_height);
    /* save the end points of the line */
    *sc_data_p++ = i_coord[0];;
    *sc_data_p++ = i_coord[1];;
    *sc_data_p++ = i_coord2[0];;
    *sc_data_p++ = i_coord2[1];;

    *x = i_coord[0];
    *y = sc_height - i_coord[1];
    *x2 = i_coord2[0];
    *y2 = sc_height - i_coord2[1];

    if (debug)
	fprintf(debug_file, 
	    "scatter_sub:  x = %g y = %g  x2 = %g y2 = %g\n", *x, *y, *x2, *y2);
    *x = (*x - bias0) / scale0;
    *x2 = (*x2 - bias0) / scale0;
    *y = (*y - bias1) / scale1;
    *y2 = (*y2 - bias1) / scale1;
}

static void
create_sc_window()
{
    XSizeHints size_hints;
    int x = 0, y = 0;            /* window position */
    char *window_name;
    char *icon_name;
    int cmdc;
    char **cmdv;
    char anin[MAXLINE];
    int what, dummy_dx, dummy_dy;

    if (sc_win != 0)
    {
	XClearWindow(display, sc_win);
	return;
    }
    window_name = "scatter_plot";
    icon_name = window_name;
    sc_width = 400;
    sc_height = 300;

    sc_win = new_window(x, y, sc_width, sc_height);

    /* initialize size hint property for window manager */
    size_hints.flags = PSize | PMinSize;
    size_hints.x = x;
    size_hints.y = y;
    size_hints.width = sc_width;
    size_hints.height = sc_height;
    size_hints.min_width = 50;
    size_hints.min_height = 50;
    cmdc = 0;
    cmdv = NULL;

    /* set Properties for window manager (always before mapping) */
    XSetStandardProperties(display, sc_win, window_name, icon_name,
	(Pixmap) 0, cmdv, cmdc, &size_hints);

#ifdef WithdrawnState
    (void) XSetWMProtocols (display, sc_win, &wm_delete_window, 1);
#endif /* WithdrawnState */

    /* Select event types wanted */
    XSelectInput(display, sc_win, KeyPressMask | ExposureMask |
	ButtonPressMask | StructureNotifyMask);

    /* Display window */
    XMapWindow(display, sc_win);

    sc_exposed = FALSE;
    do
    {
	wait_for_event(anin, &what, &x, &y, &dummy_dx, &dummy_dy, 0);
    } while (!sc_exposed);
}


void
sc_expose()
{
    int i, x, y, x2, y2, xcur, ycur;
    short *sc_data_p;

    if ((sc_exposed == TRUE) && (sc_data != NULL))
    {
	/* this is a re-expose - repaint the old graphics */
	sc_data_p = sc_data + 1;
	for (i = 0; i < *sc_data; i++)
	{
	    xcur = *sc_data_p++;
	    ycur = *sc_data_p++;
	    XDrawLine(display, sc_win, gc_blue, xcur-3, ycur, xcur+3, ycur);
	    XDrawLine(display, sc_win, gc_blue, xcur, ycur-3, xcur, ycur+3);
	}
	x = *sc_data_p++;
	y = *sc_data_p++;
	x2 = *sc_data_p++;
	y2 = *sc_data_p++;
	XDrawLine(display, sc_win, gc_green, x, y, x2, y2);
    }
    sc_exposed = TRUE;
}


void
sc_destroy()
{
    if (sc_win != 0)
    {
	XDestroyWindow(display, sc_win);
	sc_win = 0;
    }
    if (sc_data != NULL)
    {
	free(sc_data);
	sc_data = NULL;
    }
}

void
sc_configure(width, height)
unsigned int width, height;
{
    sc_width = width;
    sc_height = height;
}
