/*
Copyright (C) 1992, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/* machine dependent code for Jupiter systems graphics hardware */
#include <stdio.h>
#include <math.h>
#include "skyflux.h"
#include "img_jup.h"
#include "img.h"
#include "parse.h"

#include "jgio.h"
#include "jconfig.h"

#define CENTER 0
#define SIZE 1
#define ANGLE 2

typedef struct
{
    short x,y;
} XPoint;

JFILE *J12file;
extern FILE *session;
extern int debug ;
extern int blank, x_zoom, y_zoom, x_orig, y_orig;
extern char *image_to_sky();
extern char plane;
extern double cdelt1;
extern int area_number; 
extern int x_area[100][MAX_AREAS], y_area[100][MAX_AREAS]; 

static double   dtr = 1.7453292519943e-2; 

set_box(xx1, yy1, xx2, yy2)
int *xx1, *yy1, *xx2, *yy2;
{
    int i_coord[2], done, what;
    int x, y;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    int cen_size;
    int x_box_size, y_box_size, max_x_size, max_y_size;
    char comm[20];
 
 
     /* Go into rubber band mode */ 
 
    cen_size = CENTER;

    get_center_xy(i_coord);
    x_center = i_coord[0];
    y_center = i_coord[1];
    x = x_center;
    y = y_center;

    x_size = 1280 / x_zoom;
    y_size = 1024 / x_zoom;

    max_x_size = x_size;
    max_y_size = y_size;

    x_box_size = x_size / 8;
    y_box_size = y_size / 8;

    done = 0;
    while (!done) 
    { 

	draw_box(1, x_center, y_center, x_box_size, y_box_size);

	wait_for_event(comm, &what, &x, &y, &dx, &dy, 0);

	/* undraw old box */
	draw_box(0, x_center, y_center, x_box_size, y_box_size);

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    x = x_center;   /* avoid a jump */
	    y = y_center;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = SIZE;
	    break;
	case POINTERMOTION:

	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		x_box_size += dx;
		y_box_size += dy;
	    }

	    if(debug == TRUE)
		printf("box: dx  dy = %d %d %d %d\n", x, dx, y, dy);



	    /* Limit the box to these limits */

	    if(x_box_size < 0)
		x_box_size = 0;

	    if(x_box_size > max_x_size)
		x_box_size = max_x_size;

	    if(y_box_size < 0)
		y_box_size = 0;

	    if(y_box_size > max_y_size)
		y_box_size = max_y_size;

	    break;
	} 
    } 


    *xx1 = x_center - x_box_size / 2;
    *yy1 = y_center - y_box_size / 2;
    *xx2 = x_center + x_box_size / 2;
    *yy2 = y_center + y_box_size / 2;
    /* normalize */
    *xx1 = (*xx1 + 2560) % 1280;
    *yy1 = (*yy1 + 2048) % 1024;
    *xx2 = (*xx2 + 2560) % 1280;
    *yy2 = (*yy2 + 2048) % 1024;
}


/* CIRCLE 
** uses the trackball to determine a circle to be used as 
** the outline of an area to be examined 
*/ 
 
circle(radius) 

double radius;

{ 
    char anin[MAXLINE], *coord_str;
    int done, what, cen_size;
    int i_coord[2]; 
    int x, y, rad, max_rad, drad;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    struct img *imgp;


    x_size = 1280 / x_zoom;
    y_size = 1024 / x_zoom;
    x_center = x_orig + x_size / 2;
    y_center = y_orig + y_size / 2;
    x = x_center;
    y = y_center;

    /* Translate the radius in arcmin to pixels */

    if(radius == -1.)
    {
	/* no radius specified */
	rad = x_size / 8;
	max_rad = y_size / 2;
    }
    else
    {
	if (cdelt1 == 0)
	    rad = radius;
	else
	    rad = .5 + (radius / (60.0 * fabs(cdelt1)));  /* round */
    }

    if(debug == TRUE)
    {
	printf("circle: input radius = %-g\n", radius);
	printf("circle: equals %d pixels\n", rad);
	printf("circle: (cdelt1 = %-g arcmin)\n", cdelt1);
    }

    /* Go into rubber band mode */ 

    cen_size = CENTER;

    done = 0;
    while (!done) 
    { 
	j_sec(1);            /* draw new circle */ 
	j_mov(x_center, y_center);
	j_dfc(rad);                            
	jfflush(J12file); 

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	j_sec(0);                     /* draw black circle */ 
	j_mov(x_center, y_center);    /* erasing the old circle */ 
	j_dfc(rad);                            

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    x = x_center;   /* avoid a jump */
	    y = y_center;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    if(radius == -1.)
		cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		drad = dx;
		if(dy*dy > dx*dx)
		    drad = dy;

		rad += drad;
	    }


	    /* Limit the radius */

	    if(rad < 0)
		rad = 0;

	    if(rad > max_rad)
		rad = max_rad;

	    break;
	} 
    } 

    jfflush(J12file); 

    x_center  = (x_center  + 2560) % 1280;
    y_center  = (y_center  + 2048) % 1024;

    /* First find image on which the center lies */
    if (debug)
	printf("circle: looking for x_center= %d y_center= %d\n",
	    x_center, y_center);
    imgp = find_SC(x_center, y_center);
    if (imgp == NULL)
    {
	error();
	printf("center must lie on an image\n");
	return;
    }
    if (debug)
	printf("circle: image came from %s\n", imgp->filnam);

    /* Load area definition arrays */ 

    x_area[0][area_number]=CIRCLE;    /* flag for circle */ 
    y_area[0][area_number]=rad;  /* radius */ 

    i_coord[0] = x_center;             /* center */    
    i_coord[1] = y_center;
    SC_to_IM(i_coord);
    x_area[1][area_number]=i_coord[0]; 
    y_area[1][area_number]=i_coord[1]; 

    coord_str = image_to_sky(i_coord);
    if (cdelt1 == 0)
    {
	printf("Circle of radius %d pixels\nCentered at %s\n", rad, coord_str);
	fprintf(session, "Circle of radius %d pixels\nCentered at %s\n", rad, coord_str);
    }
    else
    {
	printf("Circle of radius %-g arcmin\nCentered at %s\n", rad * 60.0 * fabs(cdelt1), coord_str);
	fprintf(session, "Circle of radius %-g arcmin\nCentered at %s\n", rad * 60.0 * fabs(cdelt1), coord_str);
    }
} 
 
/* ELLIPSE
** uses the trackball to determine an ellipse to be used as 
** the outline of an area to be examined 
*/ 

ellipse()
{
    char anin[MAXLINE];
    int done, what, cen_size;
    int i_coord[2]; 
    int x, y, max_x_size, max_y_size;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    struct img *imgp;
    double xdist, ydist, angle;
    int width, height;


    x_size = 1280 / x_zoom;
    y_size = 1024 / x_zoom;
    x_center = x_orig + x_size / 2;
    y_center = y_orig + y_size / 2;
    x = x_center;
    y = y_center;

    max_x_size = x_size;
    max_y_size = y_size;

    width = x_size / 4;
    height = y_size / 4;
    angle = 0;


    /* Go into rubber band mode */ 

    cen_size = CENTER;

    done = 0;
    while (!done) 
    {
	j_sec(1);            /* draw new ellipse */ 
	draw_ellipse(x_center, y_center, width, height, (int)angle);
	jfflush(J12file); 

	wait_for_event(anin, &what, &x, &y, &dx, &dy, 0);

	j_sec(0);                     /* draw black ellipse */ 
	draw_ellipse(x_center, y_center, width, height, (int)angle);
					/* erasing the old circle */ 

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    x = x_center;   /* avoid a jump */
	    y = y_center;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    if (cen_size == SIZE)
		cen_size = ANGLE;
	    else
		cen_size = SIZE;
	    break;
	case POINTERMOTION:
	    switch(cen_size)
	    {
		case CENTER:
		    x_center = x;
		    y_center = y;
		    break;

		case SIZE:
		    width += dx;
		    height += dy;
		    break;
		case ANGLE:
		    xdist = x = x_center;
		    ydist = y = y_center;
		    if (fabs(xdist) > 30)
			angle += 50 * (double)dy/xdist;
		    if (fabs(ydist) > 30)
			angle += 50 * (double)dx / ydist;
		    break;
	    }


	    /* Limit the size */

	    if (width < 0)
		width = 0;

	    if (width > max_x_size)
		width = max_x_size;

	    if (height < 0)
		height = 0;

	    if (height > max_y_size)
		height = max_y_size;

	    break;
	} 
    }

    jfflush(J12file); 

    x_center  = (x_center  + 2560) % 1280;
    y_center  = (y_center  + 2048) % 1024;

    /* First find image on which the center lies */
    if (debug)
	printf("ellipse: looking for x_center= %d y_center= %d\n",
	    x_center, y_center);
    imgp = find_SC(x_center, y_center);
    if (imgp == NULL)
    {
	error();
	printf("center must lie on an image\n");
	return;
    }
    if (debug)
	printf("ellipse: image came from %s\n", imgp->filnam);

    /* Load area definition arrays */ 

    x_area[0][area_number]=ELLIPSE;

    i_coord[0] = x_center;             /* center */    
    i_coord[1] = y_center;
    SC_to_IM(i_coord);
    x_area[1][area_number]=i_coord[0]; 
    y_area[1][area_number]=i_coord[1]; 

    x_area[2][area_number]=width;
    y_area[2][area_number]=height; 

    describe_area(area_number);
}
 
/* POLYGON 
** uses the trackball to define a set of connected points to 
** be used to outline an area to be examined 
*/ 
 
polygon() 
{ 
    int done, what;
    int n_area, i_coord[2]; 
    int x, y, oldx0, oldy0;
    int xcur, ycur;
    int dx, x_size;
    int dy, y_size;
    struct img *imgp, *imgp_sav;
    char comm[MAXLINE], *coord_str;

    set_csr();


    /* Go into rubber band mode */ 

    n_area = 0;
    imgp_sav = NULL;

    x_size = 1280 / x_zoom;
    y_size = 1024 / x_zoom;
    xcur = x_orig + x_size / 2;
    ycur = y_orig + y_size / 2;

    x = xcur;
    y = ycur;

    oldx0 = xcur;
    oldy0 = ycur;

    done = 0;
    while (!done) 
    { 

	j_dca(xcur, ycur, 1);      /* place  cursor     */ 

	j_sec(1);            /* draw the new vector */ 
	j_mov(oldx0,oldy0); 
	j_dva(xcur,ycur); 

	jfflush(J12file); 

	wait_for_event(comm, &what, &x, &y, &dx, &dy, 0);

	j_sec(0);            /* draw a black vector */ 
	j_mov(oldx0,oldy0);  /* erasing the line on */ 
	j_dva(xcur,ycur);    /* the screen          */ 
 
	switch(what)
	{
	case Button1:

	    i_coord[0] = (xcur + 2560) % 1280;
	    i_coord[1] = (ycur + 2048) % 1024;

	    imgp = find_SC(i_coord[0], i_coord[1]);

	    if (imgp_sav == NULL)
	    {
		imgp_sav = imgp;  /* save first vertex info */
	    }

	    if (imgp_sav == NULL)
	    {
		error();
		printf("first vertex must lie on an image\n");
	    }

	    else if ((imgp != imgp_sav) && (imgp != NULL))
	    {
		error();
		printf("vertices fall on multiple images\n");
	    }

	    else
	    {
		j_sec(1);            /* draw the new vector */ 
		j_mov(oldx0,oldy0); 
		j_dva(xcur,ycur); 

		SC_to_IM(i_coord);
		++n_area; 
		x_area[n_area][area_number]=i_coord[0]; 
		y_area[n_area][area_number]=i_coord[1]; 

		coord_str = image_to_sky(i_coord);
		printf(" vertex# %d  at %s\n", n_area, coord_str);
		fprintf(session, " vertex# %d  at %s\n", n_area, coord_str);
		if(debug == TRUE)
		    printf("polygon:  vertex#%d  xcur=%d, ycur=%d\n",
			n_area, xcur, ycur);

		oldx0=xcur;
		oldy0=ycur;
	    }
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    break;
	case POINTERMOTION:
	    xcur = x;
	    ycur = y;
	    if(n_area == 0) 
	    { 
		oldx0=xcur; 
		oldy0=ycur; 
	    } 
	    break;
	} 
    }

    j_dca(xcur,ycur,0);      /* Turn off cursor */

    /*  complete the polygon by copying the 1st vertex to a new last vertex */
    ++n_area; 
    x_area[n_area][area_number] = x_area[1][area_number];
    y_area[n_area][area_number] = y_area[1][area_number];

    x_area[0][area_number]=POLYGON;      /* Code for polygon */ 
    y_area[0][area_number]=n_area; 
 
    jfflush(J12file); 
} 

draw_ellipse(x, y, width, height, angle)
int x, y, angle;
int width, height;
{
    XPoint points[65];
    int i, npoints;
    double sin_angle, cos_angle;

    angle = 180-angle;     /* Jupiter has y=0 at bottom */
    sin_angle = sin(angle * dtr);
    cos_angle = cos(angle * dtr);
    npoints = make_ellipse((double) x, (double) y, (double) (width/2),
    (double) (height/2), sin_angle, cos_angle, points);
    
    j_mov(points[0].x, points[0].y);
    for (i = 0; i < npoints; i++)
	j_dva(points[i].x, points[i].y);
    jfflush(J12file);
}

draw_box(color, x, y, x_size, y_size)
int color, x, y, x_size, y_size;
{
    int x1, x2, y1, y2;
    j_sec(color); 

    x1 = x - x_size / 2;
    y1 = y - y_size / 2;
    x2 = x + x_size / 2;
    y2 = y + y_size / 2;

    /* normalize */
    x1 = (x1 + 2560) % 1280;
    y1 = (y1 + 2048) % 1024;
    x2 = (x2 + 2560) % 1280;
    y2 = (y2 + 2048) % 1024;
    j_mov(x1,y1);
    j_dva(x2,y1);
    j_dva(x2,y2); 
    j_dva(x1,y2); 
    j_dva(x1,y1); 
    jfflush(J12file); 
}

/* DRAW_OUTLINE 
** uses the definition of an area to draw the outline 
*/ 
 
draw_outline(overlay_plane, color) 
 
{       
    int i,x,y,rad; 
    int area_type, n_area; 
    int i_coord[2];
    int width, height, angle;


    overlay_setup(overlay_plane);
    j_sec(color);

    area_type=x_area[0][area_number]; 
    n_area=y_area[0][area_number]; 

    if(area_type==POLYGON)                    /* Draw a polygon */ 
    {
	for(i=1; i<=n_area; ++i) 
	{
	    i_coord[0] = x_area[i][area_number]; 
	    i_coord[1] = y_area[i][area_number]; 
	    IM_to_SC(i_coord);
	    x = i_coord[0];
	    y = i_coord[1];

	    if(debug == TRUE)
		printf("draw_outline: polygon point#%d  x=%d, y=%d \n",
		    i, x, y);

	    if(i==1) 
		j_mov(x,y); 

	    j_dva(x,y); 
	}
    }

    else if(area_type==CIRCLE)                /* Draw a circle */ 
    {
	rad=y_area[0][area_number]; 
	i_coord[0] = x_area[1][area_number]; 
	i_coord[1] = y_area[1][area_number]; 
	IM_to_SC(i_coord);
	x = i_coord[0];
	y = i_coord[1];

	if(debug == TRUE)
	    printf("draw_outline:  circle center (%d,%d), radius %d \n",
		x, y, rad);

	j_mov(x,y); 
	j_dfc(rad); 
    }

    else if(area_type==ELLIPSE)
    {
	rad=y_area[0][area_number]; 
	i_coord[0] = x_area[1][area_number]; 
	i_coord[1] = y_area[1][area_number]; 
	IM_to_SC(i_coord);
	x = i_coord[0];
	y = i_coord[1];
	width = x_area[2][area_number];
	height = y_area[2][area_number];
	angle = x_area[3][area_number];

	if(debug)
	    printf("draw_outline:  ellipse center (%d,%d), width %d height %d angle %d\n",
		x, y, width, height, angle);

	draw_ellipse(x, y, width, height, angle);
    }
    jfflush(J12file);

} 


/******************************************************************/
/* 								  */
/* TRACKBALL_PICK                                                 */
/* uses the trackball to define a point                           */
/* Returns TRUE if point is good, 				  */
/* FALSE if user ended the operation 		  	  	  */
/* 								  */
/* winx and winy are same as x and y in Jupiter version           */
/* 								  */
/******************************************************************/

trackball_pick(x, y, winx, winy)
int *x, *y, *winx, *winy;
{
    int done, what;
    int dx, dy;
    char comm[20];
    int xcur, ycur;
    struct img *imgp;


    set_csr();

    overlay_setup('r');

    /* Pick the point */

    xcur = *x;
    ycur = *y;

    done = 0;
    while (!done) 
    { 

	j_dca(xcur, ycur, 1);      /* place  cursor     */ 
	jfflush(J12file);
 
	wait_for_event(comm, &what, &xcur, &ycur, &dx, &dy, 0);

	switch(what)
	{
	case Button1:
	    *x = xcur;
	    *y = ycur;
	    *winx = xcur;
	    *winy = ycur;

	    imgp = find_SC(xcur, ycur);  /* load the right globals */

	    if ((debug == TRUE) && (imgp != NULL))
		printf("pick: center pixel came from %s\n", imgp->filnam);

	    done = 1;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    break;
	case POINTERMOTION:

		if (debug)
		{
		    printf("trackball_pick: x dx y dy = %d %d %d %d\n", xcur, dx, ycur, dy);
		    fflush(stdout);
		}
		break;
	}
    }

    if (what == Button1)
	return(TRUE);	/* indicate that an x,y pair is available */
    j_dca(*x, *y, 0);	/* turn off cursor */
    image_setup(plane);
    jfflush(J12file);
    return (FALSE);	/* indicate that he wants to end picks */
}

trackball_mark()
{
    printf("Too few arguments for mark command\n");
    error();
}


/* SL_VECTOR
** uses the trackball to determine a vector to be used in the slice
** command
*/                                             
 
sl_vector(i_coord, i_coord2) 

int i_coord[2], i_coord2[2]; 

{ 
    int done, what;
    int x, y;
    int dx, x_center, x_size;
    int dy, y_center, y_size;
    int cen_size;
    int x_box_size, y_box_size, max_x_size, max_y_size;
    char comm[20];
    int x1cur, y1cur, x2cur, y2cur;
 
    set_csr();


    /* Go into rubber band mode */ 

    overlay_setup('r');
    cen_size = CENTER;

    x_size = 1280 / x_zoom;
    y_size = 1024 / x_zoom;
    x_center = x_orig + x_size / 2;
    y_center = y_orig + y_size / 2;

    max_x_size = x_size;
    max_y_size = y_size;

    x = x_center;
    y = y_center;
    x_box_size = x_size / 8;
    y_box_size = y_size / 8;

    done = 0;
    while (!done) 
    { 

	x1cur = x_center - x_box_size / 2;
	y1cur = y_center - y_box_size / 2;
	x2cur = x_center + x_box_size / 2;
	y2cur = y_center + y_box_size / 2;

	j_dca(x_center, y_center, 1);      /* place  cursor     */ 
 
	j_sec(1);            /* draw the new vectors*/ 
	j_mov(x1cur,y1cur); 
	j_dva(x2cur,y2cur); 
	jfflush(J12file); 

	wait_for_event(comm, &what, &x, &y, &dx, &dy, 0);

	/* undraw old line */
	j_sec(0);      /* draw black vectors  */ 
	j_mov(x1cur,y1cur);  /* erasing the lines on*/ 
	j_dva(x2cur,y2cur);  /* the screen          */ 

	switch(what)
	{
	case Button1:
	    cen_size = CENTER;
	    x = x_center;   /* avoid a jump */
	    y = y_center;
	    break;
	case Button2:
	case TTYINPUT:
	    done = 1;
	    break;
	case Button3:
	    cen_size = SIZE;
	    break;
	case POINTERMOTION:

	    if(cen_size == CENTER)
	    {
		x_center = x;
		y_center = y;
	    }

	    if(cen_size == SIZE)
	    {
		x_box_size += dx;
		y_box_size += dy;
	    }

	    /* Limit the box to these limits */

	    if(x_box_size > max_x_size)
		x_box_size = max_x_size;

	    if(y_box_size > max_y_size)
		y_box_size = max_y_size;

	    if(x_box_size < -max_x_size)
		x_box_size = -max_x_size;

	    if(y_box_size < -max_y_size)
		y_box_size = -max_y_size;

	    break;
	} 
    } 

 

    i_coord[0] = x1cur;
    i_coord[1] = y1cur;

    i_coord2[0] = x2cur;
    i_coord2[1] = y2cur;

    if(debug == TRUE)
	printf("sl_vector: coords (%d,%d) - (%d,%d)\n", 
	    x1cur, y1cur, x2cur, y2cur);


    j_dca(x,y,0);      /* Turn off cursor */

    overlay_setup('g');
    j_sec(2);      /* draw the vector in  */ 
    j_mov(x1cur,y1cur);  /* green               */ 
    j_dva(x2cur,y2cur);  

    image_setup(plane);   /* reset enabled planes */ 
    jfflush(J12file); 
} 

scatter_sub(stashtemp, stashtemp1, dnmin, dnmax, stash_count, x, y, 
    x2, y2)
double *stashtemp, *stashtemp1, dnmin[2], dnmax[2];
int stash_count;
double *x, *y, *x2, *y2;
{
    int i_coord[2], i_coord2[2];
    int i, xcur, ycur, bias0, bias1;
    double scale0, scale1;

    if (dnmax[0] == dnmin[0])
	dnmax[0] = dnmin[0] + 1;
    scale0 = 1000 / ( (float) x_zoom * (dnmax[0] - dnmin[0])); /* 1024 means full screen */
    bias0 = x_orig - dnmin[0] * scale0 + 7;
    if (dnmax[1] == dnmin[1])
	dnmax[1] = dnmin[1] + 1;
    scale1 = 900 / ( (float) x_zoom * (dnmax[1] - dnmin[1])); /* 1024 means full screen */
    bias1 = y_orig - dnmin[1] * scale1 + 7;
    if (debug)
    {
	printf("scatter_sub: scale0 = %g  bias0 = %d\n", scale0, bias0);
	printf("scatter_sub: scale1 = %g  bias1 = %d\n", scale1, bias1);
    }
    overlay_setup('b');
    overlay_color(4);  /* blue */
    for (i = 0; i < stash_count / 2; i++)
    {
	if (debug)
	    printf("scatter_sub: first image = %g second image = %g\n", *stashtemp, *stashtemp1);
	if ((*stashtemp != (double) blank) && (*stashtemp1 != (double) blank))
	{
	    xcur = *stashtemp * scale0 + bias0;
	    ycur = *stashtemp1 * scale1 + bias1;
	    if (debug)
		printf("xcur = %d  ycur = %d\n", xcur, ycur);
	    if (xcur >= 1280)
		xcur -= 1280;
	    if (ycur >= 1024)
		ycur -= 1024;
	    overlay_mov(xcur-3, ycur);
	    overlay_draw(xcur+3, ycur);

	    overlay_mov(xcur, ycur-3);
	    overlay_draw(xcur, ycur+3);
	}
	stashtemp++;
	stashtemp1++;
    }
    jfflush(J12file);
    image_setup(plane);

    sl_vector(i_coord, i_coord2);

    *x = i_coord[0];
    *y = i_coord[1];
    *x2 = i_coord2[0];
    *y2 = i_coord2[1];

    if (debug)
	printf("scatter_sub:  x = %g y = %g  x2 = %g y2 = %g\n", *x, *y, *x2, 
	*y2);
    if (*x < x_orig)
	*x += 1280;
    if (*x > x_orig + 1280 / x_zoom)
	*x -= 1280;
    if (*y < y_orig)
	*y += 1024;
    if (*y > y_orig + 1024 / y_zoom)
	*y -= 1024;
    if (*x2 < x_orig)
	*x2 += 1280;
    if (*x2 > x_orig + 1280 / x_zoom)
	*x2 -= 1280;
    if (*y2 < y_orig)
	*y2 += 1024;
    if (*y2 > y_orig + 1024 / y_zoom)
	*y2 -= 1024;
    *x = (*x - bias0) / scale0;
    *x2 = (*x2 - bias0) / scale0;
    *y = (*y - bias1) / scale1;
    *y2 = (*y2 - bias1) / scale1;
    if (debug)
	printf("scatter_sub:  scaled x = %g y = %g  x2 = %g y2 = %g\n", *x, *y, *x2, 
	*y2);
}
