/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/**********************************************************************
**                                                                   ** 
**  DEFINE_AREA allows the user to define an area either as a box,   ** 
**  a circle, or a closed polygon.  The definition is used later by  ** 
**  the function 'examine_area'.                                     ** 
**                                                                   ** 
**********************************************************************/ 

 
#include "skyview.h"
#include "img.h"
#include "parse.h"
#include "area.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <string.h>

#define CENTER 0
#define SIZE   1

                /***********************************/ 
                /*                                 */ 
                /* Area types:                     */ 
                /*             BOX                 */ 
		/*             CIRCLE              */ 
		/*             ELLIPSE             */ 
		/*             POLYGON             */ 
                /*             LIST                */ 
                /*             OUTLINE             */ 
                /*                                 */ 
                /***********************************/ 

    /******************************************************/ 
    /* Syntax options:                                    */ 
    /*   da [n]                                           */ 
    /*   da [n] bo                                        */ 
    /*   da [n] po                                        */ 
    /*   da [n] ci [radius [lon lat]]                     */ 
    /*   da [n] li                                        */ 
    /*   da [n] el [lon lat semiminor semimajor angle]    */ 
    /*   da [n] ol lon_min lon_max lat_min lat_max        */ 
    /******************************************************/ 
 
extern int debug, graphics;
extern int server_mode;
extern char server_str[];
extern int pixels, lines;
extern char plane;
extern int output_coord_sys;
extern double cdelt1;
extern FILE *session, *debug_file;
extern FILE *cmdfile;
extern struct img *curr_img;
extern int ads_server_mode;

int area_number=0; 
struct area_struct area[MAX_AREAS];

/* function prototypes */
#ifdef _NO_PROTO
static void poly_list();
static void box();
#else
static void poly_list(void);
static void box(void);
#endif /* _NO_PROTO */

void
define_area(argc,argv) 
 
int argc; 
char **argv; 
 
{
    static int area_type=CIRCLE; 
    int n, mark, type, arg_bo, arg_ci, arg_el, arg_ol, i, old_area_type;
    double radius, height, width, angle, d_coord[2];
    char s_coord[2][40];
    int new_area_number;

    if (curr_img == NULL)
    {
	error1("no current image");
	return;
    }

    new_area_number = area_number;
    old_area_type = area_type;
    radius = -1.;    /* flag that no radius has been specified yet */
    arg_ci = 0;      /* which argv is "ci" */
    arg_el = 0;      /* which argv is "el" */
    arg_ol = 0;      /* which argv is "ol" */
    arg_bo = 0;      /* which argv is "ol" */
    mark = TRUE;

    if(argc >= 2)
    {
	if (strcmp(argv[argc-1], "nomark") == 0)
	{
	    mark = FALSE;
	    argc--;
	}
    }

    if(argc >= 2)
    { 
	type = FALSE;

	if(strncmp(argv[1],"bo",2) == 0) /* Check first arg */
	{
	    type = TRUE;
	    area_type=BOX; 
	    arg_bo = 1;
	}

	if(strncmp(argv[1],"ol",2) == 0)
	{
	    type = TRUE;
	    area_type=OUTLINE; 
	    arg_ol = 1;
	}

	if(strncmp(argv[1],"ci",2) == 0)
	{
	    type = TRUE;
	    area_type=CIRCLE; 
	    arg_ci = 1;
	}

	if(strncmp(argv[1],"el",2) == 0)
	{
	    type = TRUE;
	    area_type=ELLIPSE; 
	    arg_el = 1;
	}

	if(strncmp(argv[1],"po",2) == 0)
	{
	    type = TRUE;
	    area_type=POLYGON; 
	}

	if(strncmp(argv[1],"li",2) == 0)
	{
	    type = TRUE;
	    area_type=LIST; 
	}

	if (type == FALSE)   /* if first arg was not an area_type, then */
	/* it must be an area_number */
	{
	    if (getint(argv[1], &new_area_number) == FALSE)  /* area number */
		return;
	    if (new_area_number < 0 || new_area_number >= MAX_AREAS)
	    {
		error1("area number out of range");
		return;
	    }
	}

	if(debug == TRUE)
	    fprintf(debug_file,
		"define: after arg #1 area_type = %d, new_area_number= %d\n",
		area_type, new_area_number);
    } 



    if ((argc >= 3 ) && (type == FALSE)) /* look at second argument */ 
    {

	if(strncmp(argv[2],"bo",2) == 0)
	{
	    type = TRUE;
	    area_type=BOX; 
	    arg_bo = 2;
	}

	if(strncmp(argv[2],"ol",2) == 0)
	{
	    type = TRUE;
	    area_type=OUTLINE; 
	    arg_ol = 2;
	}

	if(strncmp(argv[2],"ci",2) == 0)
	{
	    type = TRUE;
	    area_type=CIRCLE; 
	    arg_ci = 2;
	}

	if(strncmp(argv[2],"el",2) == 0)
	{
	    type = TRUE;
	    area_type=ELLIPSE; 
	    arg_el = 2;
	}

	if(strncmp(argv[2],"po",2) == 0)
	{
	    type = TRUE;
	    area_type=POLYGON; 
	}

	if(strncmp(argv[2],"li",2) == 0)
	{
	    type = TRUE;
	    area_type=LIST; 
	}

	if (type == FALSE)
	{
	    error1("bad parameter to DA command");
	    return;
	}

	if(debug)
	    fprintf(debug_file, 
		"define: after arg #1 area_type = %d, new_area_number= %d\n",
		area_type, new_area_number);
    }

    area_number = new_area_number;

    if ((arg_bo != 0) && (argc > arg_bo + 1))
    {
	/* box */
	/* da [n] bo x1 y1 x2 y2  */
	if (argc != arg_bo + 5)
	{
	    fprintf(debug_file, 
		"define:  need 4 values:  'da bo x1 y1 x2 y2'\n");
	    return;
	}
	area[area_number].area_type=POLYGON;
	area[area_number].vertices = 5;


	/* Lower left */

	strcpy(s_coord[0], argv[arg_bo + 1]);
	strcpy(s_coord[1], argv[arg_bo + 2]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[1] = d_coord[0];
	area[area_number].y[1] = d_coord[1];

	area[area_number].x[5] = area[area_number].x[1];
	area[area_number].y[5] = area[area_number].y[1];



	/* Lower right */

	strcpy(s_coord[0], argv[arg_bo + 3]);
	strcpy(s_coord[1], argv[arg_bo + 2]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[2] = d_coord[0];
	area[area_number].y[2] = d_coord[1];



	/* Upper right */

	strcpy(s_coord[0], argv[arg_bo + 3]);
	strcpy(s_coord[1], argv[arg_bo + 4]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[3] = d_coord[0];
	area[area_number].y[3] = d_coord[1];



	/* Upper left */

	strcpy(s_coord[0], argv[arg_bo + 1]);
	strcpy(s_coord[1], argv[arg_bo + 4]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[4] = d_coord[0];
	area[area_number].y[4] = d_coord[1];

	if (mark)
	    draw_outline('g', 2);

	describe_area(area_number);
	return;
    }

    if(arg_ol != 0)
    {
	/* outline */
	/* da [n] ol x1 x2 y1 y2  */
	if (argc != arg_ol + 5)
	{
	    fprintf(debug_file, 
		"define:  need 4 values:  'da ol x1 x2 y1 y2'\n");
	    return;
	}
	area[area_number].area_type=POLYGON;
	area[area_number].vertices = 5;


	/* Lower left */

	strcpy(s_coord[0], argv[arg_ol + 1]);
	strcpy(s_coord[1], argv[arg_ol + 3]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[1] = d_coord[0];
	area[area_number].y[1] = d_coord[1];

	area[area_number].x[5] = area[area_number].x[1];
	area[area_number].y[5] = area[area_number].y[1];



	/* Lower right */

	strcpy(s_coord[0], argv[arg_ol + 2]);
	strcpy(s_coord[1], argv[arg_ol + 3]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[2] = d_coord[0];
	area[area_number].y[2] = d_coord[1];



	/* Upper right */

	strcpy(s_coord[0], argv[arg_ol + 2]);
	strcpy(s_coord[1], argv[arg_ol + 4]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[3] = d_coord[0];
	area[area_number].y[3] = d_coord[1];



	/* Upper left */

	strcpy(s_coord[0], argv[arg_ol + 1]);
	strcpy(s_coord[1], argv[arg_ol + 4]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[4] = d_coord[0];
	area[area_number].y[4] = d_coord[1];

	if (mark)
	    draw_outline('g', 2);
	return;
    }

    if(arg_ci != 0 && argc > arg_ci + 1)  /*  da ... ci [radius ...]  */
	    				/* if something follows the ci */
    {
	radius = atof(argv[arg_ci + 1]);
	if (radius < 0)
	{
	    error1("define_area:  radius may not be negative");
	    return;
	}

	if(debug)
	    fprintf(debug_file, "define: circle input radius %-g\n", radius);
    }
     

    if(arg_ci != 0 && argc == arg_ci + 4)   /*  da ... ci radius lat lon  */
			/* if exactly 3 items follow the ci */
    {
	if ((cdelt1 == 0) || 
	    (output_coord_sys == IM) ||
	    (output_coord_sys == SC) ||
	    (output_coord_sys == IRAF))
	{
	    area[area_number].radius = radius;
	}
	else
	{
	    area[area_number].radius = radius / (60.0 * fabs(cdelt1));
	}

	area[area_number].area_type = CIRCLE;

	strcpy(s_coord[0], argv[arg_ci + 2]);
	strcpy(s_coord[1], argv[arg_ci + 3]);
	(void) sky_to_image(s_coord, d_coord);

	area[area_number].x[1] = d_coord[0];
	area[area_number].y[1] = d_coord[1];

	describe_area(area_number);

	if (mark)
	    draw_outline('g', 2);
	return;
    }

    if(arg_el != 0 && argc > arg_el + 1)  /* da ... el x y width height angle */
	    				/* if something follows the el */
    {

	if (argc != arg_el + 6)
	{
	    /* exactly 5 items must follow the el */
	    error1("define_area ellipse:  wrong number of arguments");
	    return;
	}
	area[area_number].area_type = ELLIPSE;

	strcpy(s_coord[0], argv[arg_el + 1]);
	strcpy(s_coord[1], argv[arg_el + 2]);
	(void) sky_to_image(s_coord, d_coord);
	area[area_number].x[1] = d_coord[0];
	area[area_number].y[1] = d_coord[1];

	width = atof(argv[arg_el + 3]);
	height = atof(argv[arg_el + 4]);
	if ((cdelt1 != 0) && 
	    (output_coord_sys != IM) &&
	    (output_coord_sys != SC) &&
	    (output_coord_sys != IRAF))
	{
	    width = (width/ (60.0 * fabs(cdelt1)));
	    height = (height/ (60.0 * fabs(cdelt1)));
	}
	area[area_number].x[2] = width;
	area[area_number].y[2] = height;

	angle = atof(argv[arg_el + 5]);
	area[area_number].x[3] = angle;

	describe_area(area_number);

	if (mark)
	    draw_outline('g', 2);

#ifdef NOTDEF
	ellipse_to_polygon(area_number);   /* RBH TEST */
#endif

	return;
    }


    if ((!graphics) && (area_type != LIST))
	return;  /* don't try interactive stuff */
    
    overlay_setup('r');

    switch(area_type) 
    { 
	case(BOX):      /* Draw a box */   
	    box(); 
	    break; 

	case(CIRCLE):      /* Draw a circle */ 
	    circle(radius); 
	    break; 

	case(ELLIPSE):      /* Draw an ellipse */ 
	    ellipse();
	    break; 

	case(POLYGON):      /* Draw a polygon */ 
	    polygon(); 
	    break; 

	case(LIST):      /* Take coord list from std input */ 
	    poly_list(); 
	    break; 

	default: 
	    /* this should never happen */
	    error1("define error:  unrecognized area type");
	    break; 
    }

    if(area[area_number].area_type == POLYGON)
    {
	/* compress out degenerate lines (zero length) */
	n = 1;
	for (i = 2; i <= area[area_number].vertices; i++)
	{
	    /* discard degenerate lines (zero length) */
	    if ((area[area_number].x[i] != area[area_number].x[n]) || 
		(area[area_number].y[i] != area[area_number].y[n]))
	    {
		/* it's unique - copy it over */
		n++;
		area[area_number].x[n] = area[area_number].x[i];
		area[area_number].y[n] = area[area_number].y[i];
	    }
	}
	area[area_number].vertices = n;
    }
           
    if(debug)
    {
	fprintf(debug_file, "define:  area number %d\n", area_number);

	if(area[area_number].area_type == POLYGON)
	{
	    fprintf(debug_file, 
		"define:  polygon   %d points\n", area[area_number].vertices);

	    for(i=1; i<=area[area_number].vertices; ++i)
		fprintf(debug_file, "define:  %d  %g  %g\n",
		    i, area[area_number].x[i], area[area_number].y[i]);
	}

	if(area[area_number].area_type == CIRCLE)
	{
	    fprintf(debug_file,
		"define:  circle,  radius %g\n", area[area_number].radius);

	    fprintf(debug_file, "define:           center (%g,%g)\n",
		area[area_number].x[1], area[area_number].y[1]);
	}
    }

    /* erase his red scratchings */
    draw_outline('r', 0); 

    if (mark)
	draw_outline('g', 2); 

    image_setup(plane);   /* reset enabled planes */ 

    if (area_type == LIST)
	area_type = old_area_type;  /* poly list is not sticky */
}
 
 
void
describe_area(area_num)
int area_num;
{
    double d_coord[2];
    double rad;
    char *coord_str;
    double angle;
    double width, height;
    char s_coord[2][40];
    int coord_sys;
    int i, jsys;
    double equinox;

    sprintf(server_str, "%d", area_num);
    srv_real("area_number", server_str);

    switch(area[area_num].area_type)
    {
    case CIRCLE:
	rad = area[area_number].radius;  /* radius */ 

	d_coord[0] = area[area_num].x[1]; 
	d_coord[1] = area[area_num].y[1]; 
	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);

	if ((cdelt1 == 0) ||
	    (output_coord_sys == IM) ||
	    (output_coord_sys == SC) ||
	    (output_coord_sys == IRAF))
	{
	    if(server_mode == FALSE)
	       printf("Circle of radius %g pixels\nCentered at %s\n", 
		rad, coord_str);
	    else
	    {
		srv_string("area_type", "circle");
		sprintf(server_str, "%g", rad);
		srv_real("radius", server_str);
		srv_string("units", "pixels");
		srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	    }
	    fprintf(session, "Circle of radius %g pixels\nCentered at %s\n", 
		rad, coord_str);
	}
	else
	{
	    if(server_mode == FALSE)
	       printf("Circle of radius %-g arcmin\nCentered at %s\n", 
		rad * 60.0 * fabs(cdelt1), coord_str);
	    else
	    {
		srv_string("area_type", "circle");
		sprintf(server_str, "%-g", rad * 60.0 * fabs(cdelt1));
		srv_real("radius", server_str);
		srv_string("units", "arcmin");
		srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	    }

	    fprintf(session, "Circle of radius %-g arcmin\nCentered at %s\n", 
		rad * 60.0 * fabs(cdelt1), coord_str);
	}
	break;
    case ELLIPSE:
	width = area[area_num].x[2]; 
	height = area[area_num].y[2]; 
	angle = area[area_num].x[3]; 

	d_coord[0] = area[area_num].x[1]; 
	d_coord[1] = area[area_num].y[1]; 
	coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
	if ((cdelt1 != 0) && 
	    (output_coord_sys != IM) &&
	    (output_coord_sys != SC) &&
	    (output_coord_sys != IRAF))
	{
	    if(server_mode == FALSE)
	       printf("Ellipse of width %-g height %-g  arcmin  angle %g\nCentered at %s\n",
		width * 60.0 * fabs(cdelt1),
		height * 60.0 * fabs(cdelt1), angle, coord_str);
	    else
	    {
		srv_string("area_type", "ellipse");
		sprintf(server_str, "%-g", width * 60.0 * fabs(cdelt1));
		srv_real("width", server_str);
		sprintf(server_str, "%-g", height * 60.0 * fabs(cdelt1));
		srv_real("height", server_str);
		sprintf(server_str, "%-g", angle);
		srv_real("angle", server_str);
		srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	    }

	    fprintf(session, "Ellipse of width %-g height %-g  arcmin  angle %g\nCentered at %s\n",
		width * 60.0 * fabs(cdelt1),
		height * 60.0 * fabs(cdelt1), angle, coord_str);
	}
	else
	{
	    if(server_mode == FALSE)
	       printf("Ellipse of width %g height %g  pixels  angle %g\nCentered at %s\n",
		width, height, angle, coord_str);
	    else
	    {
		srv_string("area_type", "ellipse");
		sprintf(server_str, "%-g", width);
		srv_real("width", server_str);
		sprintf(server_str, "%-g", height);
		srv_real("height", server_str);
		sprintf(server_str, "%-g", angle);
		srv_real("angle", server_str);
		srv_coord("coord", coord_sys, s_coord, equinox, jsys);
	    }

	    fprintf(session, "Ellipse of width %g height %g  pixels  angle %g\nCentered at %s\n",
		width, height, angle, coord_str);
	}
	break;
    case POLYGON:
	if (ads_server_mode)
	{
	    srv_string("area_type", "polygon");
	    srv_start_array("vertex");
	    for (i = 1; i < area[area_num].vertices; i++)
	    {
		d_coord[0] = area[area_num].x[i]; 
		d_coord[1] = area[area_num].y[i]; 
		coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
		srv_coord(NULL, coord_sys, s_coord, equinox, jsys);
	    }
	    srv_end_array(area[area_num].vertices-1);
	}
	break;
    }
}

#ifdef NOTDEF
ellipse_to_polygon(area_num)
int area_num;
{
    double x, y, width, height, angle, sin_angle, cos_angle;
    int n, i;
    struct
    {
	short x;
	short y;
    } points[65];

    width = area[area_num].x[2];
    height = area[area_num].y[2];
    angle = area[area_num].x[3];

    x = area[area_num].x[1];
    y = area[area_num].y[1];
    get_ellipse_points(x, y, width/2, height/2, angle, points);

    area[area_num].area_type = POLYGON; 
    n = 1;
    area[area_num].x[n] = points[0].x;
    area[area_num].y[n] = points[0].y; 
    for (i = 1; i < 65; i++)
    {
	/* discard degenerate lines (zero length) */
	/*
	if ((points[i].x != points[i-1].x) || (points[i].y != points[i-1].y))
	*/
	if (1)
	{
	    n++;
	    area[area_num].x[n] = points[i].x;
	    area[area_num].y[n] = points[i].y; 
	    if(server_mode == FALSE)
	       printf(" %d   %d\n", 
		points[i].x, points[i].y);
	}
    }
    area[area_number].vertices = n; 
}
#endif /* NOTDEF */

 
/* BOX  
** uses trackball to determine a box outlining an area  
** to be examined 
*/                                             
 
static void
box() 
{
    int i;
    double d_coord[2]; 
    double x1cur, y1cur, x2cur, y2cur;
    char *coord_str, s_coord[2][40];
    int coord_sys;
    struct img *imgp_sav, *imgp[4];
    int jsys;
    double equinox;


    set_box(&x1cur, &y1cur, &x2cur, &y2cur);

    /* check if multiple images */
    /* First find image on which each vertex lies */
    imgp[0] = find_SC( (int) x1cur,  (int) y1cur);
    imgp[1] = find_SC( (int) x1cur,  (int) y2cur);
    imgp[2] = find_SC( (int) x2cur,  (int) y1cur);
    imgp[3] = find_SC( (int) x2cur,  (int) y2cur);
    /* now find a non-null image on which a vertex lies */
    imgp_sav = NULL;
    for (i=0; i<4; i++)
    {
	if (imgp_sav == NULL)
	    imgp_sav = imgp[i];
	if ((imgp[i] != NULL) && (debug))
	    fprintf(debug_file, "box: image came from %s\n", imgp[i]->filnam);
    }
    if (imgp_sav == NULL)
    {
	error1("at least one vertex must lie on an image");
	return;
    }
    for (i=0; i<4; i++)
    {
	if ((imgp[i] != NULL) && (imgp[i] != imgp_sav))
	{
	    error1("vertices fall on multiple images");
	    return;
	}
    }
    /* The globals are now set for this image */

    /* Load up the area definition arrays */ 

    area[area_number].area_type = POLYGON;
    area[area_number].vertices = 5;  /* number of points */ 

    d_coord[0] = x1cur;
    d_coord[1] = y1cur;
    dSC_to_IM(d_coord);
    area[area_number].x[1] = d_coord[0]; 
    area[area_number].y[1] = d_coord[1]; 
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    if(server_mode == FALSE)
	printf("Box \nFrom %s", coord_str);
    else if (!ads_server_mode)
    {
	srv_string("area_type", "box");
	srv_coord("coord", coord_sys, s_coord, equinox, jsys);
    }
    fprintf(session, "Box \nFrom %s", coord_str);

    d_coord[0] = x2cur;
    d_coord[1] = y1cur;
    dSC_to_IM(d_coord);
    area[area_number].x[2] = d_coord[0];
    area[area_number].y[2] = d_coord[1];

    d_coord[0] = x2cur;
    d_coord[1] = y2cur;
    dSC_to_IM(d_coord);
    area[area_number].x[3] = d_coord[0]; 
    area[area_number].y[3] = d_coord[1]; 
    coord_str = dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    if(server_mode == FALSE)
	printf("To %s", coord_str);
    else if (!ads_server_mode)
    {
	srv_real("x1", s_coord[0]);
	srv_real("y1", s_coord[1]);
    }

    fprintf(session, "To %s", coord_str);

    d_coord[0] = x1cur;
    d_coord[1] = y2cur;
    dSC_to_IM(d_coord);
    area[area_number].x[4] = d_coord[0]; 
    area[area_number].y[4] = d_coord[1]; 

    d_coord[0] = x1cur;
    d_coord[1] = y1cur;
    dSC_to_IM(d_coord);
    area[area_number].x[5] = d_coord[0]; 
    area[area_number].y[5] = d_coord[1]; 

    if (ads_server_mode)
	describe_area(area_number);
}
 
 
 
/* POLY_LIST 
** reads the list of coordinate pairs from the standard input 
** and outlines the area defined for later examination 
*/ 
 
static void
poly_list() 
 
{
    int i, cmdc; 
    char s_coord[2][40], *cmdv[10];
    double d_coord[2];

    i = 0;

    for (;;)
    {
	do
	{
	    do
	    {
		if (cmdfile == stdin)
		    if(server_mode == FALSE)
			printf("define_area");
	    } while (getcmd(&cmdc, cmdv) == NIL);
	    if ((cmdc != 2) && (cmdv[0][0] != 'e'))
	    {
		if(server_mode == FALSE)
		    printf("define_area error: need exactly 2 items on line\n");
	    }
	} while ((cmdc != 2) && (cmdv[0][0] != 'e'));
	strcpy(s_coord[0], cmdv[0]);
	if (cmdc == 2)
	    strcpy(s_coord[1], cmdv[1]);

	if(s_coord[0][0] == 'e')   /* early termination */
	    break;

	if(i == 0)
	{
	    if(strcmp(s_coord[0], "ci") == 0)
	    {
		area[area_number].area_type = CIRCLE; 
		if (cdelt1 == 0)
		    area[area_number].radius = atof(s_coord[1]);
		else
		    area[area_number].radius = (atof(s_coord[1]) / (60.0 * fabs(cdelt1)));  /* radius of circle */

		i = 1;
	    }

	    if(strcmp(s_coord[0], "po") == 0 ||
	    strcmp(s_coord[0], "bo") == 0   )
	    {
		area[area_number].area_type = POLYGON; 
		area[area_number].vertices = atoi(s_coord[1]);  /* number of points */
		i = 1;
	    }

	    if(area[area_number].area_type == 0)
	    {
		if(server_mode == FALSE)
		    printf("Unacceptable area type (try ci, bo or po)\n");
	    }
	}

	else
	{
	    (void) sky_to_image(s_coord, d_coord);

	    area[area_number].x[i] = d_coord[0];
	    area[area_number].y[i] = d_coord[1];

	    ++i;
	}

	if(area[area_number].area_type == CIRCLE && i > 1)
	    break;

	if(area[area_number].area_type == POLYGON && i > area[area_number].vertices)
	    break;
    }



    /* Fix things at the end if necessary */

    --i;

    if(area[area_number].area_type == POLYGON)
    {
	if(area[area_number].vertices != i)
	    area[area_number].vertices = i;

	if(area[area_number].x[1] != area[area_number].x[i] ||
	    area[area_number].y[1] != area[area_number].y[i]   )
	{
	    ++i;
	    area[area_number].vertices = i;

	    area[area_number].x[i] = area[area_number].x[1];
	    area[area_number].y[i] = area[area_number].y[1];
	}
    }
}
 
 
 
/* CHANGE_AREA command */

void
change_area(argc,argv) 
int argc; 
char **argv; 
      
{ 
    if(argc > 2) 
    {
	error1("wrong number of arguments to change_area");
	return;
    }

    if(argc == 2) 
    {
	if (!switch_area(argv[1]))
	    return;

	draw_outline('g', 2);
    }
    if(server_mode == FALSE)
	printf("current area = %d\n", area_number);
    else
    {
	sprintf(server_str, "%d", area_number);
	srv_real("area_number", server_str);
    }
} 

/**********************************************************************/
/* SWITCH_AREA                                                        */
/* resets the area number                                             */
/* returns TRUE if OK,                                                */
/*         FALSE if area not changed (error message has been printed  */
/*                                    and error() has been called)    */
/**********************************************************************/

int
switch_area(instr)
char *instr;
{
     int n;

     if (!getint(instr, &n))
	return(FALSE);
 
     if(n < 0 || n >= MAX_AREAS) 
     {
	 sprintf(server_str,"area number not in range 0 - %d",MAX_AREAS-1);
	 error1(server_str);
	 return(FALSE);
     }

     if (area[n].area_type == 0)
     {
	sprintf(server_str, "area %d has not been defined", n);
	error1(server_str);
	return(FALSE);
     }
     area_number = n;
     return(TRUE);
}
