/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

/***********************************************************************/
/*  Here is how this all works:                                        */
/*    The contour plot is produced by external program cmap.           */
/*    Cmap takes all of its input from the command line.               */
/*    Its postscript output is to file /var/tmp/skyview_contourXXXXXX  */
/*    Since cmap takes some time to execute, skyview sets up a         */
/*    callback for its completion and returns control to the user.     */
/*                                                                     */
/*    The grid is produced by external program agra.                   */
/*    Agra solicits lines of input with a prompt ">"                   */
/*    Its postscript output is to file /var/tmp/skyview_gridXXXXXX     */
/*    Since agra runs fairly rapidly, skyview hangs waiting for it     */
/*    to exit.                                                         */
/*                                                                     */
/*    The user can request output to a Postscript printer (via command */
/*    contour) or to overlay the current image (via command contourovl)*/
/*    If he wants overlay: the grid routine and the contour routine    */
/*    act independently. Each plots its Postscript file on the image,  */
/*    and then deletes the file.                                       */
/*    If the user wants the output to go to the printer, the           */
/*      Postscript output is always produces by agra.  The user can    */
/*      request either contours or a grid or both.  If contours are    */
/*      requested then cmap is first run, with output to an agra       */
/*      vector file.  Then agra is run.  Agra creates the grids, if    */
/*      requested, inputs the contour vector file, if requested, adds  */
/*      a box at the edge of the image, and outputs a Postscript file. */
/*      The resultant Postscript output is redirected to the printer.  */
/*                                                                     */
/***********************************************************************/

#include "skyview.h"
#include "img.h"
#include "job.h"
#include "parse.h"
#include "fits.h"
#include "im_proto.h"
#include "svclib.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>

extern char fi_name[], filnam[], mod_filnam[];
extern int server_mode, debug;
extern int file_coord_sys, output_coord_sys;
extern int maptype;
extern int pixels, lines;
extern double cdelt1, cdelt2, twist;
extern double crpix1, crpix2;
extern double glat, glong;
extern struct hdrstruct filehdr;
extern double	plate_ra;	/* Right ascension of plate center */
extern double	plate_dec;	/* Declination of plate center */
extern double	x_pixel_offset;	/* X pixel offset of image lower right */
extern double	y_pixel_offset;	/* Y pixel offset of image lower right */
extern double	plt_scale; /* plate scale arc sec per mm */
extern double	x_pixel_size;	/* X pixel_size */
extern double	y_pixel_size;	/* Y pixel_size */
extern double	ppo_coeff[6];
extern double	amd_x_coeff[20]; /* X coefficients for plate model */
extern double	amd_y_coeff[20]; /* Y coefficients for plate model */
extern int    file_jsys, output_jsys;
extern double file_equinox, output_equinox;
extern int output_use_image_equinox;



extern unsigned int width_org, height_org;
extern double win_zoom;
extern FILE *debug_file;
extern char server_str[];
extern struct img *curr_img;

double agra_plon, agra_plat, agra_mlon, agra_mlat, agra_su, agra_sv;
char agra_proj[100];

static char grid_file[100];
static char join_file[300];
static int overlay_mode;
static int grid_arg;
static int gridonly;
static char im_filename[300];
static int contour_pid = 0;
static int contourps = 0;
static char contourps_file[300];
static char vector_file[100];

static double   rtd = 180/M_PI;

/* function prototypes */
#ifdef _NO_PROTO

static void cmap_callback();
static void run_agra();
static int send_agra();
static char *send_agra1();
static char *get_agra_response();
static int good_coordsys();
static int wait_for_glitch();
static void print_it();

#else

static void cmap_callback(struct job_info *job);
static void run_agra(int use_vector_file, int generate_grid);
static int send_agra(char *agra_cmd, FILE *writefd, FILE *readfd);
static char *send_agra1(char *agra_cmd, FILE *writefd, FILE *readfd);
static char *get_agra_response(FILE *fd);
static int good_coordsys(int coordsys);
static int wait_for_glitch(FILE *fd);
static void print_it(char *filename);
static double labelscale;
static int override_labelscale = 0;

#endif /* _NO_PROTO */

/*ARGSUSED*/
void grid_cmd(argc, argv)
int argc;
char *argv[];
{
    if (!set_current_window())
    {
	error1("no images currently being displayed");
	return;
    }
    if (argc > 1)
    {
	labelscale = atof(argv[1]);
	override_labelscale = 1;
    }
    else
    {
	override_labelscale = 0;
    }
    overlay_mode = 1;
    im_filename[0] = '\0';
    grid_arg = -1;
    gridonly = 1;
    vector_file[0] = '\0';
    run_agra(0, 1);
}


static
void run_agra(use_vector_file, generate_grid)
int use_vector_file, generate_grid;
{
    int fd;
    char *cmdv[5];
    int cmdc;
    FILE *readfd, *writefd;
    struct job_info *job;
    double d_coord[2];
    int coord_sys;
    char s_coord[2][40];
    double xsize, ysize;
    char agra_cmd[600];
    char jchar;
    char temp_str[100];
    double equinox;
    int jsys;


#ifdef NOTDEF
    struct img tmpimg, *save_imgp;
    int fd;

    if (im_filename[0] != '\0')
    {
	/* fetch parameters from file (pixels, lines, )  */
	fd = wf_open(im_filename, &tmpimg, 1);
	if (fd < 0)
	{
	    sprintf(server_str, 
		"Unable to open file: %s", im_filename);
	    error1(server_str);
	    return;
	}
	if (filehdr.tapetype == IMFILE)
	    im_close(fd);
	else
	    close(fd);

	save_imgp = curr_img;
	fill_glob(&tmpimg);
	set_file_sys();
	d_coord[0] = ((double)pixels / 2.) - 0.5;
	d_coord[1] = ((double)lines / 2.) - 0.5;

	/* now simulate dSC_to_IM(d_coord)  */
	d_coord[0] = d_coord[0] + 1 - crpix1;
	if (cdelt2 >= 0)
	{
	    d_coord[1] = d_coord[1] + 1 - crpix2;
	}
	else
	{
	    d_coord[1] = - d_coord[1] + lines - crpix2;
	}

	if (maptype == PLATE)
	{
	    xsize = fabs((double) pixels *
		plt_scale * x_pixel_size / 1000 / 3600);
	    ysize = fabs((double) lines *
		plt_scale * y_pixel_size / 1000 / 3600);
	}
	else
	{
	    xsize = fabs((double) pixels * cdelt1);
	    ysize = fabs((double) lines * cdelt2);
	}

    }
    else
#endif /* NOTDEF */
    {
	d_coord[0] = ((double)width_org / 2.) - 0.5;
	d_coord[1] = ((double)height_org / 2.) - 0.5;
	dSC_to_IM(d_coord);

	if (maptype == PLATE)
	{
	    xsize = fabs((double) width_org *
		plt_scale * x_pixel_size / 1000 / 3600);
	    ysize = fabs((double) height_org *
		plt_scale * y_pixel_size / 1000 / 3600);
	}
	else
	{
	    xsize = fabs((double) width_org * cdelt1);
	    ysize = fabs((double) height_org * cdelt2);
	}
    }

    if (!good_coordsys(file_coord_sys))
    {
	sprintf(server_str, "cannot draw grid of image with no coordinates");
	error1(server_str);
	goto grid_error1;
    }

    /* take the coord sys from the cs command */
    if (!good_coordsys(output_coord_sys))
    {
	sprintf(server_str, "cannot draw grid in %s coordinates", 
	    coord_name(output_coord_sys));
	error1(server_str);
	goto grid_error1;
    }



    /* temporarily use file's coord sys to get coords for mcent command */
    save_coord_sys();
    cmdv[1] = coord_name(file_coord_sys);
    if ((file_coord_sys == EQ) || (file_coord_sys == EC))
    {
	jchar = 'B';
	if ((file_jsys == 0) || (file_jsys == 13))
	    jchar = 'J';
	sprintf(temp_str, "%c%f", jchar, file_equinox);
	cmdv[2] = temp_str;
	cmdv[3] = "output";
	cmdc = 4;
    }
    else
    {
	cmdv[2] = "output";
	cmdc = 3;
    }
    set_coord_sys3(cmdc, cmdv, 1);
    dimage_to_sky(d_coord, s_coord, &coord_sys, &equinox, &jsys);
    str_to_coord(s_coord, d_coord);
    restore_coord_sys();



    strcpy(grid_file, "/var/tmp/skyview_gridXXXXXX");
    /*
    (void) mktemp(grid_file);
    */
    fd = mkstemp(grid_file);
    if (fd == -1)
    {
	sprintf(server_str, 
	    "Unable to create temp file for grid");
	error1(server_str);
    }
    else
    {
	close (fd);
    }


    cmdv[0] = "agra";
    cmdv[1] = "-x";    /* suppress X windows output from agra */
    cmdv[2] = (char *) 0;

    if (duplex_pipe(cmdv, &readfd, &writefd, &job))
    {
	job->fp = readfd;
	job->imgp = curr_img;
	job->callback = 0;    /* register callback */
	job->valid_callback = 0;
    }
    else
    {
	sprintf(server_str, 
	    "Unable to start background job: %s", "agra");
	error1(server_str);
    }
    if (wait_for_glitch(readfd))
	goto grid_error;

    if ((file_coord_sys == EQ) || (file_coord_sys == EC))
	sprintf(agra_cmd, "pcs %s %c %6.1f\n", coord_name(file_coord_sys),
	    (file_jsys == 1 || file_jsys == 3)?'B':'J',
	    file_equinox);
    else
	sprintf(agra_cmd, "pcs %s\n", coord_name(file_coord_sys));
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    sprintf(agra_cmd, "proj %s\n", maptype_string(maptype));
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    sprintf(agra_cmd, "pcent %11.6f %11.6f\n", glong, glat);
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    sprintf(agra_cmd, "mcent %11.6f %11.6f\n", d_coord[0], d_coord[1]);
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    sprintf(agra_cmd, "size %-g %-g\n", xsize, ysize);
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    if ((output_coord_sys == EQ) || (output_coord_sys == EC))
	sprintf(agra_cmd, "gcs %s %c %6.1f\n", coord_name(output_coord_sys),
	    (output_jsys == 1 || output_jsys == 3)?'B':'J',
	    output_equinox);
    else
	sprintf(agra_cmd, "gcs %s\n", coord_name(output_coord_sys));

    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    if (cdelt1 > 0)
    {
	sprintf(agra_cmd, "flipx\n");
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
    }

    if (maptype == PLATE)
    {
	sprintf(agra_cmd, "platescale %-g\n", plt_scale);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, "platera %11.6f\n", plate_ra * rtd);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, "platedec %11.6f\n", plate_dec * rtd);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, "platesize %14.7e %14.7e\n", x_pixel_size, y_pixel_size);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, "plateoff %-g %-g\n", x_pixel_offset, y_pixel_offset);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, "naxis %-d %-d\n", pixels, lines);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, 
	"platex  %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e\n",
	    amd_x_coeff[0],
	    amd_x_coeff[1],
	    amd_x_coeff[2],
	    amd_x_coeff[3],
	    amd_x_coeff[4],
	    amd_x_coeff[5],
	    amd_x_coeff[6],
	    amd_x_coeff[7],
	    amd_x_coeff[8],
	    amd_x_coeff[9],
	    amd_x_coeff[10],
	    amd_x_coeff[11],
	    amd_x_coeff[12],
	    amd_x_coeff[13],
	    amd_x_coeff[14],
	    amd_x_coeff[15],
	    amd_x_coeff[16],
	    amd_x_coeff[17],
	    amd_x_coeff[18],
	    amd_x_coeff[19]
	);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, 
	"platey  %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e\n",
	    amd_y_coeff[0],
	    amd_y_coeff[1],
	    amd_y_coeff[2],
	    amd_y_coeff[3],
	    amd_y_coeff[4],
	    amd_y_coeff[5],
	    amd_y_coeff[6],
	    amd_y_coeff[7],
	    amd_y_coeff[8],
	    amd_y_coeff[9],
	    amd_y_coeff[10],
	    amd_y_coeff[11],
	    amd_y_coeff[12],
	    amd_y_coeff[13],
	    amd_y_coeff[14],
	    amd_y_coeff[15],
	    amd_y_coeff[16],
	    amd_y_coeff[17],
	    amd_y_coeff[18],
	    amd_y_coeff[19]
	);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
	sprintf(agra_cmd, 
	"platep  %20.13e %20.13e %20.13e %20.13e %20.13e %20.13e\n",
	    ppo_coeff[0],
	    ppo_coeff[1],
	    ppo_coeff[2],
	    ppo_coeff[3],
	    ppo_coeff[4],
	    ppo_coeff[5]
	);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
    }
    else
    {
	sprintf(agra_cmd, "rotation %11.6f\n", -twist);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
    }


    if (generate_grid)
    {
	if (override_labelscale)
	{
	    sprintf(agra_cmd, "labelscale %f\n", labelscale);
	    if (send_agra(agra_cmd, writefd, readfd))
		goto grid_error;
	}
	sprintf(agra_cmd, "grid\n");
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
    }

    if (use_vector_file)
    {
	/* we have a cmap which must be merged */
	sprintf(agra_cmd, "contour %s\n", vector_file);
	if (send_agra(agra_cmd, writefd, readfd))
	    goto grid_error;
    }

    sprintf(agra_cmd, "border\n");
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    sprintf(agra_cmd, "print %s\n", grid_file);
    if (send_agra(agra_cmd, writefd, readfd))
	goto grid_error;

    if (debug)
	fprintf(debug_file, "quit\n");
    fprintf(writefd, "quit\n");
    fflush(writefd);

    fclose(writefd);
    fclose(readfd);
    waitpid(job->pid, (int *) NULL, 0);
    del_job(job);
#ifdef NOTDEF
    if (im_filename[0] != '\0')
    {
	if (save_imgp == NULL)
	    curr_img = NULL;
	else
	    fill_glob(save_imgp);  /* refill global with current image stuff */
    }
#endif /* NOTDEF */

    if (use_vector_file)
	unlink(vector_file);


    if (overlay_mode)
    {
	cmdv[0] = "post";
	cmdv[1] = grid_file;
	cmdc = 2;
	post_cmd(cmdc, cmdv);
	unlink(grid_file);
    }
    else 
    {
	/* grid only - just send it to printer (or a file) */
	print_it(grid_file);
    }
    return;

grid_error:
    fclose(writefd);
    fclose(readfd);
    waitpid(job->pid, (int *) NULL, 0);
    del_job(job);
grid_error1:
#ifdef NOTDEF
    if (im_filename[0] != '\0')
    {
	if (save_imgp == NULL)
	    curr_img = NULL;
	else
	    fill_glob(save_imgp);  /* refill global with current image stuff */
    }
#endif /* NOTDEF */
    if (vector_file[0] != '\0')
	unlink(vector_file);
}

static void
print_it(filename)
char *filename;
{
    char sysstr[MAXPATHLEN];
    char *sprinter, printer_arg[30];
    int i;

	/* just send it to printer (or a file) */
	if (contourps)
	{
	    /* output to a postscript file */
	    if (contourps_file[0] != '\0')
	    {
		/* he gave an output filename */
		sprintf(sysstr, "mv %s %s",
		    filename, contourps_file);
		if (debug)
		    fprintf(debug_file, "passing to system: [%s]\n", sysstr);
		i = system(sysstr);
		if (i != 0)
		{
		    if (i == 1)
			sperror(sysstr, errno);
		    else
			printf("mv exited with status %d\n", WEXITSTATUS(i));
		}
		if (!server_mode)
		    printf("plot is in file %s\n", contourps_file);
	    }
	    else
	    {
		if (!server_mode)
		    printf("plot is in file %s\n", filename);
	    }
	}
	else
	{
	    /* output to printer */
	    sprinter = getenv("SPRINTER");
	    if (sprinter == NULL)
		printer_arg[0] = '\0';
	    else
		sprintf(printer_arg, "-P%s", sprinter);
	    sprintf(sysstr, "lpr %s %s; rm -f %s", 
		printer_arg, filename, filename);
	    if (debug)
		fprintf(debug_file, "passing to system: [%s]\n", sysstr);
	    i = system(sysstr);
	    if (i != 0)
	    {
		if (i == 1)
		    sperror(sysstr, errno);
		else
		    printf("lpr exited with status %d\n", WEXITSTATUS(i));
		unlink(filename);
	    }
	}
}


static int 
good_coordsys(coordsys)
int coordsys;
{
    if ((coordsys == EQ) ||
	(coordsys == EC) ||
	(coordsys == SGAL) ||
	(coordsys == GA))
	return(1);
    else
	return(0);
}

/*  send_agra  */
/*     send a command to agra and wait for agra's glitch */
/*  returns 0 for OK, 							*/
/*          1 for agra error - agra returned something other than "> "   */
static int
send_agra(agra_cmd, writefd, readfd)
char *agra_cmd;
FILE *writefd, *readfd;
{
    if (debug)
	fputs(agra_cmd, debug_file);
    fputs(agra_cmd, writefd);
    fflush(writefd);
    return (wait_for_glitch(readfd));
}


/* wait_for_glitch 							*/
/*  returns 0 for OK, 							*/
/*          1 for agra error - agra returned something other than "> "   */
static int
wait_for_glitch(fd)
FILE *fd;
{
    char ch;
    int glitch_flag;
    static char error_msg[100];
    static int msg_length = 0;
    int retval;

    retval = 1;
    glitch_flag = 0;
    while (!feof(fd))
    {
	fread(&ch, 1, 1, fd);
	if (ch == '>')
	    glitch_flag = 1;  /* wait for space */
	else if ((ch == ' ') && (glitch_flag))
	{
	    if (error_msg[0] == '\0')
	    {
		retval = 0;
	    }
	    else
	    {
		sprintf(server_str, 
		    "grid error from agra: %s", error_msg);
		error1(server_str);
		error_msg[0] = '\0';
		msg_length = 0;
		retval = 1;
	    }
	    break;
	}
	else 
	{
	    if (msg_length < 99)
	    {
		error_msg[msg_length++] = ch;
		error_msg[msg_length] = '\0';
	    }
	}
    }
    if ((retval == 1) && (error_msg[0] != '\0'))
    {
	sprintf(server_str, 
	    "grid error from agra: %s", error_msg);
	error1(server_str);
	error_msg[0] = '\0';
	msg_length = 0;
    }
    return(retval);
}


void cplot(argc, argv)
int argc;
char *argv[];
{
    int fd;
    int i, j;
    char *cmdv[20];
    int cmdc = 0;
    FILE *readfd, *flfile;
    struct job_info *job;
    char *p, first_arg[80];
    double dummy_dbl;
    char contour_file[100];


    if ((strcmp(argv[0], "co") == 0) || 
	(strcmp(argv[0], "contour") == 0) ||
	(strcmp(argv[0], "contourps") == 0) ||
	(strcmp(argv[0], "contour_plot") == 0))
	    overlay_mode = 0;
    else if (strcmp(argv[0], "contourovl") == 0)
	overlay_mode = 1;
    else
    {
	sprintf(server_str, 
	    "contour error:  command_name %s not recognized", argv[0]);
	error1(server_str); 
	return;
    }
    if (strcmp(argv[0], "contourps") == 0)
    {
	contourps = 1;
	if ((argc >= 3) && (strcmp(argv[1], "-o") == 0))
	{
	    strcpy(contourps_file, argv[2]);
	    argc -=2;
	    argv +=2;
	}
	else
	    contourps_file[0] = '\0';
    }
    else
	contourps = 0;

    strcpy(contour_file, "/var/tmp/skyview_contourXXXXXX");
    /*
    (void) mktemp(contour_file);
    */
    fd = mkstemp(contour_file);
    if (fd == -1)
    {
	sprintf(server_str, 
	    "Unable to create temp file for contour plot");
	error1(server_str);
    }
    else
    {
	close (fd);
    }
    cmdv[cmdc++] = "cmap";
    cmdv[cmdc++] = "-o";
    cmdv[cmdc++] = contour_file;

    strcpy(vector_file, "/var/tmp/skyview_vectorXXXXXX");
    /*
    (void) mktemp(vector_file);
    */
    fd = mkstemp(vector_file);
    if (fd == -1)
    {
	sprintf(server_str, 
	    "Unable to create temp file for contour plot");
	error1(server_str);
    }
    else
    {
	close (fd);
    }
    cmdv[cmdc++] = "-v";
    cmdv[cmdc++] = vector_file;

    /* check if first argument is a filename */
    im_filename[0] = '\0';
    if (argc > 1)
    {
	strncpy(first_arg, argv[1], 78);
	p = strchr(first_arg, '%');
	if (p != NULL)
	{
	    if (*(p+1) == '\0')    /* if % is followed by null */
		*p='\0';           /* change % to null */
	}
    }
    if ( (argc <= 1) || (isdbl(first_arg, &dummy_dbl)) ||
	(strcmp(argv[1], "lin") == 0) ||
	(strcmp(argv[1], "linear") == 0) ||
	(strcmp(argv[1], "stretch") == 0) ||
	(strcmp(argv[1], "grid") == 0) ||
	(strcmp(argv[1], "nogrid") == 0) ||
	(strcmp(argv[1], "gridonly") == 0))
    {
	/* first arg missing or it looks like a number or reserved word */
	/* so use current filename */

	if (mod_filnam[0] != '\0')
	{
	    error1("contour_plot error:  must first write modified file");
	    return;
	}
	if (strlen(filnam) != 0)
	{
	    cmdv[cmdc++] = filnam;
	}
	else if (strlen(fi_name) != 0)
	{
	    cmdv[cmdc++] = fi_name;
	}
	else
	{
	    error1("contour_plot error:  no default filename available");
	    return;
	}

	j = 1;    /* start with arg 1 */
    }
    else
    {
	/* looks like a filename */
	cmdv[cmdc++] = expand_path(argv[1]);
	strcpy(im_filename, expand_path(argv[1]));
	j = 2;   /* start with arg after filename */
    }

    /* tak on the rest of the args */
    grid_arg = 0;
    gridonly = 0;
    for (i = j; i < argc; i++)
    {
	if (strcmp(argv[i], "grid") == 0)
	{
	    grid_arg = i;
	    break;
	}
	else if (strcmp(argv[i], "gridonly") == 0)
	{
	    gridonly = 1;
	    grid_arg = i;
	    break;
	}
	else if (strcmp(argv[i], "nogrid") == 0)
	{
	    if (!server_mode)
		printf(
		"nogrid option is no longer valid - it is now the default\n");
	}
	else
	{
	    cmdv[cmdc++] = argv[i];
	}
    }

    if (overlay_mode)
    {
	if (!set_current_window())
	{
	    error1("no images currently being displayed");
	    return;
	}
    }
    if (!gridonly)
    {
	cmdv[cmdc] = (char *) 0;

	if (debug)
	{
	    fprintf(debug_file, "RBH forking:");
	    for (i = 0; i < cmdc; i++)
		fprintf(debug_file, " %s", cmdv[i]);
	    fprintf(debug_file, "\n");
	}

	/* start cmap */
	if (duplex_pipe(cmdv, &readfd, &flfile, &job))
	{
	    fclose(flfile);
	    job->fp = readfd;
	    contour_pid = job->pid;
	    job->imgp = curr_img;
	    job->callback = cmap_callback;    /* register callback */
	    job->valid_callback = 1;
	    strcpy(job->job_argfilname, contour_file);
	    if(server_mode == FALSE)
		printf("contour computing in background\n");
	    /* will eventually call do_post(readfd, 2) */
	}
	else
	{
	    sprintf(server_str, 
		"Unable to start background job: %s", "cmap");
	    error1(server_str);
	}
    }

    if (grid_arg != 0)
    {
	if ((overlay_mode) || (gridonly))
	    run_agra(0, 1);
	else
	{
	    /* wait to call run_agra (until cmap done) */
	}
    }
}



static void
cmap_callback(job)
struct job_info *job;
{
    int remapping;
    int fd;
    struct img *imgp_save, tmpimg;
    char *cmdv[3];
    int cmdc;
    int status;
    pid_t wait_return;
    char contour_file[100];
    int cont_char;

    strcpy(contour_file, job->job_argfilname);
    if (overlay_mode)
    {
	wait_return = waitpid(job->pid, &status, WNOHANG);
	if ((wait_return == job->pid) && 
	    (WIFEXITED(status)) && (WEXITSTATUS(status) != 0))
	{
	    /* cmap died  and has already printed a diagnostic */
	    while (!feof(job->fp))
	    {
		cont_char = getc(job->fp);
		if (cont_char != EOF)
		    putchar(cont_char);
	    }
	    printf("\n");
	    fclose(job->fp);
	    del_job(job);
	    contour_pid = 0;
	    return;
	}

	imgp_save = curr_img;
	if (check_img(job->imgp))
	{
	    fill_glob(job->imgp);

	    /* now see if contours need remapping (reprojection) */
	    if (im_filename[0] == '\0')
	    {
		remapping = 0;  /* needs no remapping if no second image */
	    }
	    else if (!good_coordsys(file_coord_sys))
	    {
		remapping = 0;  /* needs no remapping if displayed image */
				/* has no projection */
	    }
	    else 
	    {
		/* look at second image's projection */
		fd = wf_open(im_filename, &tmpimg, 1);
		if (fd < 0)
		{
		    return;   /* this should never happen */
		}
		else
		{
		    fill_glob(&tmpimg);
		    if (!good_coordsys(file_coord_sys))
			remapping = 0;  /* needs no remapping if second */
					/* image has no projection */
		    else
			remapping = 1;
		    fill_glob(job->imgp);  /* restore globals */
		    if (tmpimg.filehdr.tapetype == IMFILE)
			im_close(fd);
		    else
			close(fd);
		}
	    }

	    if (remapping)
	    {
		/* needs remapping */
		unlink(contour_file);
		run_agra(1, 0);
	    }
	    else
	    {
		/* needs no remapping if no second image or */
		/* if either image has no projection */
		unlink(vector_file);
		cmdv[0] = "post";
		cmdv[1] = contour_file;
		cmdc = 2;
		post_cmd(cmdc, cmdv);
		unlink(contour_file);
		draw_border();
	    }
	    fill_glob(imgp_save);
	}
	while (!feof(job->fp))
	    getc(job->fp);
	fclose(job->fp);
	wait_return = waitpid(job->pid, &status, 0);
	if ((wait_return == -1) && (errno == ECHILD))
	{
	    /*EMPTY*/
	    /* no such child */
	}
	else
	{
	if ((status != 0) && (!server_mode))
	    printf("contour exited with non-zero status code = 0x%x\n", status);
	}
	del_job(job);
	contour_pid = 0;
    }
    else
    {
	/* output to a the printer (or a file) */
	while (!feof(job->fp))
	    getc(job->fp);
	fclose(job->fp);
	wait_return = waitpid(job->pid, &status, 0);
	if ((wait_return == -1) && (errno == ECHILD))
	{
	    /*EMPTY*/
	    /* no such child */
	}
	else
	{
	if ((status != 0) && (!server_mode))
	    printf("contour exited with non-zero status code = 0x%x\n", status);
	}
	del_job(job);
	contour_pid = 0;
	if ((grid_arg == 0) && (im_filename[0] == '\0'))
	{
	    /* no remapping or concatenation needed */
	    unlink(vector_file);
	    print_it(contour_file);
	}
	else
	{
	    unlink(contour_file);

	    if (grid_arg != 0)
	    {
		/* concatenation needed */
		run_agra(1, 1);
	    }
	    else
	    {
		/* remapping of contours needed */
		run_agra(1, 0);
	    }
	}
    }
}

void kill_contour_job()
{
    struct job_info *tmp_job;

    if (contour_pid != 0)
    {
	tmp_job = find_job(contour_pid);
	if (tmp_job != NULL)
	{
	    if (!server_mode)
		printf("Killing contour job\n");
	    if (kill(contour_pid, SIGKILL) == 0)
	    {
		unlink(grid_file);
		unlink(tmp_job->job_argfilname);
		unlink(join_file);
	    }
	}
    }
}

/* agra_scale asks for a good image dimensions and projection */
/*   given a table file with lat, lon */
int agra_scale(filename, lon, lat)
char *filename, *lon, *lat;
{
    char *cmdv[3];
    FILE *readfd, *writefd;
    struct job_info *job;
    char agra_cmd[600];
    char *agra_answer;
    char val[256];
    int retval;

    retval = 0;

    cmdv[0] = "agra";
    cmdv[1] = "-s";
    cmdv[2] = (char *) 0;

    if (duplex_pipe(cmdv, &readfd, &writefd, &job))
    {
	job->fp = readfd;
	job->imgp = curr_img;
	job->callback = 0;    /* register callback */
	job->valid_callback = 0;
    }
    else
    {
	sprintf(server_str, 
	    "Unable to start background job: %s", "agra");
	error1(server_str);
    }

    sprintf(agra_cmd, "file %s\n", filename);
    agra_answer = send_agra1(agra_cmd, writefd, readfd);
    if (agra_answer == NULL)
    {
	goto agra_error;
    }

    sprintf(agra_cmd, "header\n");
    agra_answer = send_agra1(agra_cmd, writefd, readfd);
    if (agra_answer == NULL)
    {
	goto agra_error;
    }
    else
    {
	printf("RBH got from agra [%s]\n", agra_answer);
	if (svc_val(agra_answer, "return.count", val))
	{
	    if (atoi(val) < 2)
	    {
		sprintf(server_str, 
		    "not enough columns in table file %s", filename);
		error1(server_str);
		goto agra_error;
	    }
	}
    }

    sprintf(agra_cmd, "read x %s\n", lon);
    agra_answer = send_agra1(agra_cmd, writefd, readfd);
    if (agra_answer == NULL)
    {
	goto agra_error;
    }

    sprintf(agra_cmd, "read y %s\n", lat);
    agra_answer = send_agra1(agra_cmd, writefd, readfd);
    if (agra_answer == NULL)
    {
	goto agra_error;
    }

    sprintf(agra_cmd, "scale x y\n");
    agra_answer = send_agra1(agra_cmd, writefd, readfd);
    if (agra_answer == NULL)
    {
	goto agra_error;
    }
    else
    {
	if (svc_val(agra_answer, "return.plon", val))
	    agra_plon = atof(val);
	if (svc_val(agra_answer, "return.plat", val))
	    agra_plat = atof(val);
	if (svc_val(agra_answer, "return.mlon", val))
	    agra_mlon = atof(val);
	if (svc_val(agra_answer, "return.mlat", val))
	    agra_mlat = atof(val);
	if (svc_val(agra_answer, "return.su", val))
	    agra_su = atof(val);
	if (svc_val(agra_answer, "return.sv", val))
	    agra_sv = atof(val);
	svc_val(agra_answer, "return.proj", agra_proj);
    }

    if (debug)
	fprintf(debug_file, "quit\n");
    fprintf(writefd, "quit\n");
    retval = 1;

agra_error:
    fclose(writefd);
    fclose(readfd);
    waitpid(job->pid, (int *) NULL, 0);
    del_job(job);
    return(retval);
}

/*  send_agra1  */
/*     send a command to agra and wait for agra's glitch */
/*  returns string from agra if  OK, 				*/
/*          NULL for agra error (agra returned something other than "> "   */
static
char *send_agra1(agra_cmd, writefd, readfd)
char *agra_cmd;
FILE *writefd, *readfd;
{
    if (debug)
	fputs(agra_cmd, debug_file);
    fputs(agra_cmd, writefd);
    fflush(writefd);
    return (get_agra_response(readfd));
}

static
char *get_agra_response(fd)
FILE *fd;
{
    static char str[SVC_STRLEN];
    char *retval;
    char val[256];

    if (fgets(str, SVC_STRLEN, fd))
    {
	if(str[strlen(str) - 1] == '\n')
	    str[strlen(str) - 1] = '\0';
	if (svc_val(str, "stat", val))
	{
	    if (strncmp(val, "ERROR", 5) == 0)
	    {
		if (svc_val(str, "return", val))
		sprintf(server_str, "agra error: %s", val);
		error1(server_str);
		retval = NULL;
	    }
	    else
		retval = str;
	}
	else
	{
	    error1("cannot get status from agra");
	    retval = NULL;
	}
    }
    else
    {
	error1("error receiving from agra");
	retval = NULL;
    }
    return(retval);
}
