/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "img.h"
#include "parse.h"
#include "fits.h"
#include "wcs.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>


extern char     ctype1[];
extern int      debug;
extern int graphics_window;
extern int      server_mode, s1_server_mode;
extern char     server_str[];
extern int      x_offset, sgn_y, y_offset;
extern double   cdelt1, cdelt2, crpix1, crpix2;
extern double cd1_1, cd1_2, cd2_1, cd2_2;
extern double dc1_1, dc1_2, dc2_1, dc2_2;
extern int using_cd;
extern double   twist, glat, glong;
extern int      maptype;
extern FILE    *session, *debug_file;
extern struct img *curr_img;
extern int maxy, miny;
extern double win_zoom;

extern int map_distortion;    /* flag for SIRTF distortion corrections */
extern int a_order, ap_order, b_order, bp_order;   /* distortion corrections */
extern double a[5][5], ap[5][5], b[5][5], bp[5][5]; /* distortion corr.*/

char last_coord[2][40];

int  output_coord_sys = EQ;
double output_equinox = 1950.0;
int output_use_image_equinox = 1;
int  output_jsys = 1;	/* 1 = equ Besselian */
			/* 0 = equ Julian */
			/* 3 = ecl B */
			/* 13= ecl Julian */
			/* 2 = gal */
			/* 4 = sgal */
static int  input_coord_sys = EQ;
double input_equinox = 1950.0;
static int input_use_image_equinox = 1;
int  input_jsys = 1;

int    file_coord_sys;
double file_equinox;
int    file_jsys;
char radecsys[80];

static int  save_input_coord_sys;
static double save_input_equinox;
static int  save_input_jsys;
static int save_output_use_image_equinox;

static int  save_output_coord_sys;
static double save_output_equinox;
static int  save_output_jsys;
static int save_input_use_image_equinox;

#define DEFAULT_SEX 2
static int desired_sex = DEFAULT_SEX;
static int sex = TRUE;

#ifdef _NO_PROTO
    static void conform_jsys();
    static void set_sex();
#else
    int arcfwd(const double, const double, struct prjprm *, double *, double *);
    int arcrev(const double, const double, struct prjprm *, double *, double *);
    static void conform_jsys(int *this_jsys);
    static void set_sex(int choice);
    double dispos(double *dra0, double *decd0, double *dra, double *decd, 
	double *dist);

#endif /* _NO_PROTO */

static double   dtr = M_PI/180;
static double   rtd = 180/M_PI;


int set_desired_sex(sex_str)
char *sex_str;
{
    if (strcmp(sex_str, "default") == 0)
	desired_sex = DEFAULT_SEX;
    else if (strcmp(sex_str, "off") == 0)
    {
	desired_sex = DEFAULT_SEX;
	set_sex(FALSE);
	desired_sex = FALSE;
    }
    else if (strcmp(sex_str, "on") == 0)
    {
	desired_sex = DEFAULT_SEX;
	set_sex(TRUE);
	desired_sex = TRUE;
    }
    else
    {
	error1("set sexigesimal argument must be 'on', 'off', or 'default'\n");
	return(0);
    }
    return(1);
}

static void
set_sex(choice)
int choice;
{
    if (desired_sex == DEFAULT_SEX)
	sex = choice;
}



/***********************************************************************/
/* */
/* SET_COORD_SYSTEM                                                   */
/* sets the output coordinate system and determines the system        */
/* used in the file.                                                  */
/* */
/**********************************************************************/

void
set_coord_system(argc, argv)
int  argc;
char **argv;
{
    int silent;

    silent = 0;
    if (argc > 1)
    {
	if (strcmp(argv[argc-1], "silent") == 0)
	{
	    silent = 1;
	    argc--;
	}
    }
    set_coord_sys3(argc, argv, silent);
}

void
set_coord_sys3(argc, argv, silent)
int  argc;
char **argv;
int silent;
{
    static char output_coord_str[20] = "equatorial";
    static char input_coord_str[20] = "equatorial";
    static char output_equinox_str[100] = "image";
    static char input_equinox_str[100] = "image";
    double new_equinox;
    char equinox_char, new_equinox_str[100];
    int good_flag, got_equinox, cmd_result, which, new_coord_sys;
    int new_use_image_equinox;

    /* Set the output coordinate system according to the command arguments */

    good_flag = 1;
    new_coord_sys = -1;
    got_equinox = 0;
    which = BOTH;

    if (argc > 1)
    {

	switch (cmd(argv[1]))
	{
	case EQ:
	    new_coord_sys = EQ;
	    set_sex(TRUE);
	    break;
	case EC:
	    new_coord_sys = EC;
	    set_sex(FALSE);
	    break;
	case GA:
	    new_coord_sys = GA;
	    set_sex(FALSE);
	    break;
	case SGAL:
	    new_coord_sys = SGAL;
	    set_sex(FALSE);
	    break;
	case IM:
	    new_coord_sys = IM;
	    set_sex(FALSE);
	    break;
	case XY:
	    new_coord_sys = XY;
	    set_sex(FALSE);
	    break;
	case SC:
	    new_coord_sys = SC;
	    set_sex(FALSE);
	    break;
	case IRAF:
	    new_coord_sys = IRAF;
	    set_sex(FALSE);
	    break;
	default:
	    error1("unrecognized coordinate system");
	    /* dont change anything */
	    good_flag = 0;
	    break;
	}
	if ((argc > 2) && good_flag)
	{
	    if (strcmp(argv[2], "image") == 0)
	    {
		new_use_image_equinox = 1;
		/* fudge in some values */
		/*   - they will be changed by set_file_sys() */
		strcpy(new_equinox_str, "image");
		new_equinox = 1950.0;
		got_equinox = 1;
	    }
	    else if (parse_equinox(argv[2], new_equinox_str, &new_equinox,
		&equinox_char))
	    {
		if ((new_coord_sys != EQ) && (new_coord_sys != EC))
		{
		    error1("equinox has meaning only for equatorial or ecliptic");
		    good_flag = 0;
		}
		else
		{
		    new_use_image_equinox = 0;
		    got_equinox = 1;
		}
	    }
	    else
	    {
		cmd_result = cmd(argv[2]);
		switch (cmd_result)
		{
		    case INPUT:
		    case OUTPUT:
		    case BOTH:
			which = cmd_result;
			break;
		    default:
			error1("unable to convert equinox to a number");
			good_flag = 0;
			break;
		}
	    }
	}
	if ((argc > 3) && good_flag)
	{
	    cmd_result = cmd(argv[3]);
	    switch (cmd_result)
	    {
		case INPUT:
		case OUTPUT:
		case BOTH:
		    which = cmd_result;
		    break;
		default:
		    error1("last argument must be 'input', 'output', or 'both'");
		    good_flag = 0;
		    break;
	    }
	}

	/* now, if all checks have passed, set the real variables */
	if (good_flag)
	{
	    if ((which == BOTH) || (which == OUTPUT))
	    {
		output_coord_sys = new_coord_sys;
		strcpy(output_coord_str, coord_name(output_coord_sys));
		if (got_equinox)
		{
		    strcpy(output_equinox_str, new_equinox_str);
		    output_equinox = new_equinox;
		    output_use_image_equinox = new_use_image_equinox;
		}
		switch(new_coord_sys)
		{
		    case EQ:
			if (output_equinox_str[0] == 'J')
			    output_jsys = 0;
			else
			    output_jsys = 1;
			break;
		    case EC:
			if (output_equinox_str[0] == 'J')
			    output_jsys = 13;
			else
			    output_jsys = 3;
			break;
		    case GA:
			output_jsys = 2;
			break;
		    case SGAL:
			output_jsys = 4;
			break;
		}
	    }

	    if ((which == BOTH) || (which == INPUT))
	    {
		input_coord_sys = new_coord_sys;
		strcpy(input_coord_str, coord_name(input_coord_sys));
		if (got_equinox)
		{
		    strcpy(input_equinox_str, new_equinox_str);
		    input_equinox = new_equinox;
		    input_use_image_equinox = new_use_image_equinox;
		}
		switch(new_coord_sys)
		{
		    case EQ:
			if (input_equinox_str[0] == 'J')
			    input_jsys = 0;
			else
			    input_jsys = 1;
			break;
		    case EC:
			if (input_equinox_str[0] == 'J')
			    input_jsys = 13;
			else
			    input_jsys = 3;
			break;
		    case GA:
			input_jsys = 2;
			break;
		    case SGAL:
			input_jsys = 4;
			break;
		}
	    }
	}
    }

    /* Set the file coordinate system according to the file header */
    set_file_sys();

    /* now do the report */
    if (silent)
	return;

    if(server_mode == FALSE)
    {
	if ((output_coord_sys == EQ) || (output_coord_sys == EC))
	{
	    if (output_use_image_equinox)
	    {
		printf(
		"Positions will be output using %s coordinates with equinox\n",
		    output_coord_str);
		printf("  determined by image file.\n");
	    }
	    else
		printf(
		    "Positions will be output using %s %s coordinates.\n", 
		    output_coord_str, output_equinox_str);
	}
	else
	    printf(
		"Positions will be output using %s coordinates.\n", 
		output_coord_str);
    }
    else
    {
	srv_string("coord_sys", output_coord_str);
	if ((output_coord_sys == EQ) || (output_coord_sys == EC))
	    srv_string("equinox", output_equinox_str);
    }

    if ((output_coord_sys == EQ) || (output_coord_sys == EC))
    {
	if (output_use_image_equinox)
	{
	    fprintf(session,
	    "Positions will be output using %s coordinates with equinox\n",
		output_coord_str);
	    fprintf(session, "  determined by image file.\n");
	}
	else
	    fprintf(session,
		"Positions will be output using %s %s coordinates.\n", 
		output_coord_str, output_equinox_str);
    }
    else
	fprintf(session,
	    "Positions will be output using %s coordinates.\n", output_coord_str);


    if(server_mode == FALSE)
    {
	if ((input_coord_sys == EQ) || (input_coord_sys == EC))
	{
	    if (input_use_image_equinox)
	    {
		printf(
		"Input positions interpreted in %s coordinates with equinox\n", 
		    input_coord_str);
		printf("  determined by image file.\n");
	    }
	    else
		printf(
		    "Input positions interpreted in %s %s coordinates.\n\n", 
		    input_coord_str, input_equinox_str);
	}
	else
	    printf(
		"Input positions interpreted in %s coordinates.\n\n", 
		input_coord_str);
    }
    else
    {
	srv_string("in_coord_sys", input_coord_str);
	if ((input_coord_sys == EQ) || (input_coord_sys == EC))
	    srv_string("in_equinox", input_equinox_str);
    }

    if ((input_coord_sys == EQ) || (input_coord_sys == EC))
    {
	if (input_use_image_equinox)
	{
	    fprintf(session,
	    "Input positions interpreted in %s coordinates with equinox\n", 
		input_coord_str);
	    fprintf(session, "  determined by image file.\n");
	}
	else
	    fprintf(session,
		"Input positions interpreted in %s %s coordinates.\n\n", 
		input_coord_str, input_equinox_str);
    }
    else
	fprintf(session,
	    "Input positions interpreted in %s coordinates.\n\n", input_coord_str);


}


void save_coord_sys()
{
    save_output_coord_sys = output_coord_sys;
    save_output_equinox = output_equinox;
    save_output_jsys = output_jsys;
    save_output_use_image_equinox = output_use_image_equinox;
    save_input_coord_sys = input_coord_sys;
    save_input_equinox = input_equinox;
    save_input_jsys = input_jsys;
    save_input_use_image_equinox = input_use_image_equinox;
}
void restore_coord_sys()
{
    char *cmdv[10];
    int cmdc;
    char jchar;
    char temp_str[100];

    cmdv[1] = coord_name(save_output_coord_sys);
    if ((save_output_coord_sys == EQ) || (save_output_coord_sys == EC))
    {
	if (save_output_use_image_equinox)
	{
	    cmdv[2] = "image";
	}
	else
	{
	    if ((save_output_jsys == 0) || (save_output_jsys == 13))
		jchar = 'J';
	    else
		jchar = 'B';
	    sprintf(temp_str, "%c%6.1f", jchar, save_output_equinox);
	    cmdv[2] = temp_str;
	}
	cmdv[3] = "output";
	cmdc = 4;
    }
    else
    {
	cmdv[2] = "output";
	cmdc = 3;
    }
    set_coord_sys3(cmdc, cmdv, 1);

    cmdv[1] = coord_name(save_input_coord_sys);
    if ((save_input_coord_sys == EQ) || (save_input_coord_sys == EC))
    {
	if (save_input_use_image_equinox)
	{
	    cmdv[2] = "image";
	}
	else
	{
	    if ((save_input_jsys == 0) || (save_input_jsys == 13))
		jchar = 'J';
	    else
		jchar = 'B';
	    sprintf(temp_str, "%c%6.1f", jchar, save_input_equinox);
	    cmdv[2] = temp_str;
	}
	cmdv[3] = "input";
	cmdc = 4;
    }
    else
    {
	cmdv[2] = "input";
	cmdc = 3;
    }
    set_coord_sys3(cmdc, cmdv, 1);
}

char *coord_name(coord_sys)
int coord_sys;
{
    switch (coord_sys)
    {
	case EQ:
	    return("equatorial");
	case EC:
	    return("ecliptic");
	case GA:
	    return("galactic");
	case SGAL:
	    return("supergalactic");
	case IM:
	    return("image");
	case XY:
	    return("xy");
	case SC:
	    return("screen");
	case IRAF:
	    return("iraf");
	default:
	    return("unknown");
    }
}



/**********************************************************************/
/* */
/* SET_FILE_SYS                                                       */
/* sets file_coord_sys based on first 2 chars of ctype1               */
/* */
/**********************************************************************/


void
set_file_sys()
{
    char s[3];

    /* Set the file coordinate system according to the file header */

    if (graphics_window)
	return;

    s[0] = ctype1[0];
    s[1] = ctype1[1];
    s[2] = '\0';


    file_coord_sys = IM;

    if (strcmp(s, "RA") == 0)
	file_coord_sys = EQ;

    if (strcmp(s, "DE") == 0)
	file_coord_sys = EQ;

    if (strcmp(s, "LL") == 0)
	file_coord_sys = EQ;

    if (strcmp(s, "GL") == 0)
	file_coord_sys = GA;

    if (strcmp(s, "LO") == 0)
	file_coord_sys = GA;

    if (strcmp(s, "EL") == 0)
	file_coord_sys = EC;
    
    if (maptype == PLATE)
	file_coord_sys = EQ;

    switch(file_coord_sys)
    {
	case EQ:
	    if (strncmp(radecsys, "FK4", 3) == 0)
		file_jsys = 1;
	    else if (strncmp(radecsys, "FK5", 3) == 0)
		file_jsys = 0;
	    else if (file_equinox < 2000.0)
		file_jsys = 1;
	    else
		file_jsys = 0;
	    break;
	case EC:
	    if (strncmp(radecsys, "FK4", 3) == 0)
		file_jsys = 3;
	    else if (strncmp(radecsys, "FK5", 3) == 0)
		file_jsys = 13;
	    else if (file_equinox < 2000.0)
		file_jsys = 3;
	    else
		file_jsys = 13;
	    break;
	case GA:
	    file_jsys = 2;
	    break;
	case SGAL:
	    file_jsys = 4;
	    break;
    }

    if (output_use_image_equinox)
    {
	output_equinox = file_equinox;
	conform_jsys(&output_jsys);
    }

    if (input_use_image_equinox)
    {
	input_equinox = file_equinox;
	conform_jsys(&input_jsys);
    }

    if (debug)
	fprintf(debug_file,
"set_file_sys: ctype1 = [%s];  interpreted as [%s];  file_coord_sys set to %s\n",
	    ctype1, s, coord_name(file_coord_sys));
}

/* force a jsys to be same {Besselian or Julian} as file_jsys */
static void 
conform_jsys(this_jsys)
int *this_jsys;
{
    if ((file_jsys == 1) || (file_jsys == 3))
    {
	/* it's Besselian */
	if (*this_jsys == 0)
	    *this_jsys = 1;
	else if (*this_jsys == 13)
	    *this_jsys = 3;
    }
    else if ((file_jsys == 0) || (file_jsys == 13))
    {
	/* it's Julian */
	if (*this_jsys == 1)
	    *this_jsys = 0;
	else if (*this_jsys == 3)
	    *this_jsys = 13;
    }
    else
    {
	/* image is jsys= 2 (galactic) or 4 (supergalactic) */
	if (file_equinox < 2000.0)
	{
	    /* make it Besselian */
	    if (*this_jsys == 0)
		*this_jsys = 1;
	    else if (*this_jsys == 13)
		*this_jsys = 3;
	}
	else
	{
	    /* make it Julian */
	    if (*this_jsys == 1)
		*this_jsys = 0;
	    else if (*this_jsys == 3)
		*this_jsys = 13;
	}
    }
}


/****************************************************************/
/*                                                              */
/* IMAGE_TO_SKY                                                 */
/* takes an (samp, line) pair of image coordinates and formats  */
/* coordinates in the system currently chosen.                  */
/* Input coords (i_coord) are not altered.                      */
/* Returns a string pointer to the formatted output ready for   */
/* printing.                                                    */
/* Also returns s_coord, string versions of the two values,     */
/* lon, lat, without any extraneous characters.                 */
/* Also returns coord_sys, the integer representation of the    */
/* coordinate system used (SC, IM, IRAF, EQ, EC, GA, etc)       */
/*                                                              */
/*                                                              */
/****************************************************************/


char 
*dimage_to_sky(d_coord, s_coord, coord_sys, equinox, jsys)
/* uses floating pt arguments */
double d_coord[2];
char s_coord[2][40];
int *coord_sys;
double *equinox;
int *jsys;
{
#ifdef NOTDEF
    double paout;
    int ierr;
#endif /* NOTDEF */
    double          fsamp, fline;
    double          coord[2], rpp1, rpp2, xx, yy, xxx, yyy;
    double          rad, beta, delta, temp, rtwist;
    double          lat, lon;
    double          glatr, glongr;
    static char     result[80];
    char   pcode[4];
    struct celprm cel;
    double phi, theta;
    struct prjprm prj;
    int status, i, j;
    double asin_arg;
    double fsamp_correction;
    double fline_correction;
    double twopi, sins, coss, dt, da;


    s_coord[0][0] = '\0';
    s_coord[1][0] = '\0';
    *coord_sys = -1;  /* illegal value */
    *equinox = -1.0;   /* illegal value */
    *jsys = -1;        /* illegal value */

    fsamp = d_coord[0];
    fline = d_coord[1];
    lat = 0;
    lon = 0;

    if (debug == TRUE)
    {
	fprintf(debug_file, "dimage_to_sky: in fsamp, fline = %-g, %-g\n", fsamp, fline);
    }

    if (curr_img == NULL)
    {
	if(server_mode == FALSE)
	   strcpy(result, " undefined location\n");
	else
	   strcpy(result, "undefined");
	return (result);
    }

    if ((output_coord_sys == XY) ||
	(file_coord_sys != IM &&
	 output_coord_sys != IRAF &&
	 output_coord_sys != IM &&
	 output_coord_sys != SC))	/* Calculate the latitudinal and */
    {				        /* longitudinal positions in the */
					/* files coordinate system       */

	if (debug)
	    fprintf(debug_file, "maptype = %s\n", maptype_string(maptype));

	switch(maptype)
	{
	case GNOMONIC:
	    if (map_distortion)
	    {
		/* apply SIRTF distortion corrections */
		fsamp_correction = 0.0;
		for (i = 0; i <= a_order; i++)
		{
		    for (j = 0; j <= a_order; j++)
		    {
			if (i + j <= a_order)
			{
			    if (debug)
			    {
				printf("i = %d j = %d  a[%d][%d] = %f   this contributiln = %f\n",
				i, j, i, j, a[i][j], 
				a[i][j] * pow(fsamp, i) * pow(fline, j));
			    }
			    fsamp_correction += 
				a[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		if (debug == TRUE)
		    fprintf(debug_file,
		    "dimage_to_sky: fsamp correction = %f\n", fsamp_correction);

		fline_correction = 0.0;
		for (i = 0; i <= b_order; i++)
		{
		    for (j = 0; j <= b_order; j++)
		    {
			if (i + j <= b_order)
			{
			    fline_correction += 
				b[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		fsamp += fsamp_correction;
		fline += fline_correction;
		if (debug == TRUE)
		    fprintf(debug_file,
		    "dimage_to_sky: fline correction = %f\n", fline_correction);

	    }
	    if (using_cd)
	    {
		xx = -(cd1_1 * fsamp + cd1_2 * fline) * dtr;
		yy = -(cd2_1 * fsamp + cd2_2 * fline) * dtr;
	    }
	    else
	    {
		rpp1 = cdelt1 * dtr;	/* radians per pixel */
		rpp2 = cdelt2 * dtr;	/* radians per pixel */
		xx = -fsamp * rpp1;
		yy = -fline * rpp2;

		rtwist = twist * dtr;	/* convert to radians */
		temp = xx * cos(rtwist) - yy * sin(rtwist);	/* do twist */
		yy = xx * sin(rtwist) + yy * cos(rtwist);
		xx = temp;
	    }

	    delta = atan(sqrt(xx * xx + yy * yy));

	    if ((xx == 0.0) && (yy == 0.0))
		yy = 1.0;  /* avoid domain error in atan2 */
	    beta = atan2(-xx, yy);
	    glatr = glat * dtr;
	    glongr = glong * dtr;
	    lat = asin(-sin(delta) * cos(beta) * cos(glatr) + cos(delta) * sin(glatr));
	    xxx = sin(glatr) * sin(delta) * cos(beta) + cos(glatr) * cos(delta);
	    yyy = sin(delta) * sin(beta);
	    lon = glongr + atan2(yyy, xxx);

	    lat = lat * rtd;
	    lon = lon * rtd;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: GNOMONIC %14.9f %14.9f\n", lon, lat);
	    break;

	case ORTHOGRAPHIC:
	    if (map_distortion)
	    {
		/* apply SIRTF distortion corrections */
		fsamp_correction = 0.0;
		for (i = 0; i <= a_order; i++)
		{
		    for (j = 0; j <= a_order; j++)
		    {
			if (i + j <= a_order)
			{
			    fsamp_correction += 
				a[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		if (debug == TRUE)
		    fprintf(debug_file,
		    "dimage_to_sky: fsamp correction = %f\n", fsamp_correction);

		fline_correction = 0.0;
		for (i = 0; i <= b_order; i++)
		{
		    for (j = 0; j <= b_order; j++)
		    {
			if (i + j <= b_order)
			{
			    fline_correction += 
				b[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		fsamp += fsamp_correction;
		fline += fline_correction;
		if (debug == TRUE)
		    fprintf(debug_file,
		    "dimage_to_sky: fline correction = %f\n", fline_correction);

	    }

	    if (using_cd)
	    {
		xx = (cd1_1 * fsamp + cd1_2 * fline) * dtr;
		yy = (cd2_1 * fsamp + cd2_2 * fline) * dtr;
	    }
	    else
	    {
		rpp1 = cdelt1 * dtr;	/* radians per pixel */
		rpp2 = cdelt2 * dtr;	/* radians per pixel */
		xx = fsamp * rpp1;
		yy = fline * rpp2;

		rtwist = twist * dtr;	/* convert to radians */
		temp = xx * cos(rtwist) - yy * sin(rtwist);	/* do twist */
		yy = xx * sin(rtwist) + yy * cos(rtwist);
		xx = temp;
	    }

	    glatr = glat * dtr;
	    glongr = glong * dtr;
	    if ((1 - xx * xx - yy * yy) < 0)
	    {
		if(server_mode == FALSE)
		   strcpy(result, " undefined location\n");
		else
		   strcpy(result, "undefined");
		return (result);
	    }
	    rad = sqrt(1 - xx * xx - yy * yy);
	    lat = asin(yy * cos(glatr) + sin(glatr) * rad);
#ifdef NOTDEF
	    lon = glongr + atan(xx / (cos(glatr) * rad - yy * sin(glatr)));
#endif /* NOTDEF */
	    lon = glongr + atan2(xx, (cos(glatr) * rad - yy * sin(glatr)));

	    lat = lat * rtd;
	    lon = lon * rtd;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: ORTHOGRAPHIC %-g %-g\n", lon, lat);
	    break;


	case NCP:
	    rpp1 = cdelt1 * dtr;	/* radians per pixel */
	    rpp2 = cdelt2 * dtr;	/* radians per pixel */
	    xx = fsamp * rpp1;
	    yy = fline * rpp2;

	    rtwist = twist * dtr;	/* convert to radians */
	    temp = xx * cos(rtwist) - yy * sin(rtwist);	/* do twist */
	    yy = xx * sin(rtwist) + yy * cos(rtwist);
	    xx = temp;

	    glatr = glat * dtr;
	    glongr = glong * dtr;

	    lon = glongr + atan(xx / (cos(glatr) - yy * sin(glatr)));
	    lat = copysign( acos((cos(glatr) - yy * sin(glatr)) / 
		cos(lon - glongr)),glatr);

	    lat = lat * rtd;
	    lon = lon * rtd;

	    if (debug == TRUE)
		fprintf(debug_file, "dimage_to_sky: NCP %-g %-g\n", lon, lat);
	    break;

	case LAMBECYL:
	    rpp2 = -cdelt2 * dtr;	/* radians per pixel */

	    lon = glong + fsamp * cdelt1;
	    if (fabs(fline * rpp2) > 1.0)
	    {
		error1("dimage_to_sky: failure in LAMBECYL deprojection");

		if(server_mode == FALSE)
		   strcpy(result, " undefined location\n");
		else
		   strcpy(result, "undefined");
		return (result);
	    }
	    else
	    {
		lat = -asin(fline * rpp2);
	    }

	    lat = glat + lat * rtd;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: LAMBECYL %-g %-g\n", lon, lat);
	    break;

	case AITOFF:
	    if (debug)
		fprintf(debug_file, "starting AITOFF\n");

	    rpp1 = cdelt1 * dtr;	/* radians per pixel */
	    rpp2 = cdelt2 * dtr;

	    xx = (fsamp / 2.) * rpp1;
	    yy = (fline / 2.) * rpp2;

	    if (((4.- xx * xx - 4.* yy * yy) < 2) || (glat != 0.0))
	    {
		if(server_mode == FALSE)
		   strcpy(result, " undefined location\n");
		else
		   strcpy(result, "undefined");
		return (result);
	    }
	    temp = sqrt(4.- xx * xx - 4.* yy * yy);

	    lat = asin(temp * yy);
	    asin_arg = (temp * xx / (2.* cos(lat)));
	    if (asin_arg > 1.0)
		asin_arg = 1.0;
	    if (asin_arg < -1.0)
		asin_arg = -1.0;
	    lon = 2 * asin(asin_arg);
#ifdef NOTDEF
	    /* IRAS AITOFF formula doesn't allow glat != 0.0 */
	    lat = glat + lat * rtd;
#endif /* NOTDEF */
	    lat = lat * rtd;
	    lon = glong + lon * rtd;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: AITOFF   %-g %-g\n", lon, lat);
	    break;

	case PLATE:
	    /* DSS Plate stuff */
	    platepos (fsamp, fline, curr_img, &lon, &lat);
	    if (debug == TRUE)
		fprintf(debug_file, "lon = %f  lat = %f\n", lon, lat);
	    break;

	case CAR:
	    rtwist = - twist * dtr;	/* convert to radians */
	    temp = fsamp * cos(rtwist) - fline * sin(rtwist);	/* do twist */
	    fline = fsamp * sin(rtwist) + fline * cos(rtwist);
	    fsamp = temp;

	    xx = fsamp * cdelt1;
	    yy = fline * cdelt2;

	    strcpy(pcode, "CAR");

	    /* Initialize projection parameters. */
	    prj.flag = 0;
	    prj.r0 = 0.0;
	    for (j = 0; j < 10; prj.p[j++] = 0.0);

	    /* Set reference angles for the native grid. */
	    cel.ref[0] =   glong;
	    cel.ref[1] =   glat;
	    cel.ref[2] = 999.0;  /* should look for LONGPOLE in header */
	    cel.ref[3] = 999.0;
	    cel.flag   = 0;

	    status = celrev(pcode, xx, yy, &prj, &phi, &theta,
		&cel, &lon, &lat);
	    
	    if (lon < 0.0)
		lon += 360.0;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: CAR  %-g %-g\n", lon, lat);
	    break;

	case CEA:
	    rtwist = - twist * dtr;	/* convert to radians */
	    temp = fsamp * cos(rtwist) - fline * sin(rtwist);	/* do twist */
	    fline = fsamp * sin(rtwist) + fline * cos(rtwist);
	    fsamp = temp;

	    xx = fsamp * cdelt1;
	    yy = fline * cdelt2;

	    strcpy(pcode, "CEA");

	    /* Initialize projection parameters. */
	    prj.flag = 0;
	    prj.r0 = 0.0;
	    for (j = 0; j < 10; prj.p[j++] = 0.0);

	    prj.p[1]= 1.0;  /* default */

	    /* Set reference angles for the native grid. */
	    cel.ref[0] =   glong;
	    cel.ref[1] =   glat;
	    cel.ref[2] = 999.0;  /* should look for LONGPOLE in header */
	    cel.ref[3] = 999.0;
	    cel.flag   = 0;

	    status = celrev(pcode, xx, yy, &prj, &phi, &theta,
		&cel, &lon, &lat);

	    if (lon < 0.0)
		lon += 360.0;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: CEA  %-g %-g\n", lon, lat);
	    break;

	case SFL:
	    rtwist = - twist * dtr;	/* convert to radians */
	    temp = fsamp * cos(rtwist) - fline * sin(rtwist);	/* do twist */
	    fline = fsamp * sin(rtwist) + fline * cos(rtwist);
	    fsamp = temp;

	    xx = fsamp * cdelt1;
	    yy = fline * cdelt2;

	    strcpy(pcode, "SFL");

	    /* Initialize projection parameters. */
	    prj.flag = 0;
	    prj.r0 = 0.0;
	    for (j = 0; j < 10; prj.p[j++] = 0.0);

	    /* Set reference angles for the native grid. */
	    cel.ref[0] =   glong;
	    cel.ref[1] =   glat;
	    cel.ref[2] = 999.0;  /* should look for LONGPOLE in header */
	    cel.ref[3] = 999.0;
	    cel.flag   = 0;

	    status = celrev(pcode, xx, yy, &prj, &phi, &theta,
		&cel, &lon, &lat);
	    
	    if (lon < 0.0)
		lon += 360.0;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: SFL  %-g %-g\n", lon, lat);
	    break;

	case ARC:
	    rtwist = - twist * dtr;	/* convert to radians */
	    temp = fsamp * cos(rtwist) - fline * sin(rtwist);	/* do twist */
	    fline = fsamp * sin(rtwist) + fline * cos(rtwist);
	    fsamp = temp;

	    glatr = glat * dtr;
	    glongr = glong * dtr;

	    xx = fsamp * cdelt1 * dtr;
	    yy = fline * cdelt2 * dtr;

	    twopi = M_PI * 2;
	    sins = xx * xx + yy * yy;
	    if (sins>=twopi*twopi/4.0)
	    {
		if (debug)
		    printf("problem in ARC projection\n");
	    }
	    sins = sqrt(sins);
	    coss = cos (sins);
	    if (sins!=0.0) sins = sin (sins) / sins;
	    else
		sins = 1.0;
	    dt = yy * cos(glatr) * sins + sin(glatr) * coss;
	    if ((dt>1.0) || (dt<-1.0))
	    {
		if (debug)
		    printf("problem (1) in ARC projection\n");
	    }
	    lat = asin (dt);
	    da = coss - dt * sin(glatr);
	    dt = xx * sins * cos(glatr);
	    if ((da==0.0) && (dt==0.0))
	    {
		if (debug)
		    printf("problem (2) in ARC projection\n");
	    }
	    lon = glongr + atan2 (dt, da);

	    lat = lat * rtd;
	    lon = lon * rtd;


	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: ARC  %-g %-g\n", lon, lat);
	    break;
	case LINEAR:
	case UNSPECIFIED:
	case UNRECOGNIZED:
	    rtwist = - twist * dtr;	/* convert to radians */
	    temp = fsamp * cos(rtwist) - fline * sin(rtwist);	/* do twist */
	    fline = fsamp * sin(rtwist) + fline * cos(rtwist);
	    fsamp = temp;

	    lon = glong + fsamp * cdelt1;
	    lat = glat + fline * cdelt2;

	    if (debug == TRUE)
		fprintf(debug_file,
		    "dimage_to_sky: LINEAR  %-g %-g\n", lon, lat);
	    break;
	}
    }

    coord[0] = lon;
    coord[1] = lat;

    *coord_sys = output_coord_sys;
    if ((file_coord_sys == IM   ) &&
	(output_coord_sys      != IRAF ) && 
	(output_coord_sys      != SC   ) && 
	(output_coord_sys      != XY   ))
    {
	*coord_sys = IM;
    }

    switch (*coord_sys)		/* Convert to the output coordinate system */
    {

    case (IRAF):		/* We want IRAF coords      */
	coord[0] = d_coord[0];
	coord[1] = d_coord[1];
	dIM_to_IRAF(coord);

	sprintf(result, "IRAF pixel = %g   IRAF line = %g\n", coord[0], coord[1]);
	sprintf(s_coord[0], "%g", coord[0]);
	sprintf(s_coord[1], "%g", coord[1]);
	break;

    case (IM):			/* We want image coords      */
	sprintf(result, "samp = %g   line = %g\n", fsamp, fline);
	sprintf(s_coord[0], "%g", fsamp);
	sprintf(s_coord[1], "%g", fline);
	break;

    case (XY):			/* We want XY */
	sprintf(result, "x = %g   y = %g\n", lon, lat);
	sprintf(s_coord[0], "%g", lon);
	sprintf(s_coord[1], "%g", lat);
	break;

    case (SC):			/* We want screen coords      */
	coord[0] = d_coord[0];
	coord[1] = d_coord[1];
	dIM_to_SC(coord);
	sprintf(result, "x = %g   y = %g\n", coord[0], coord[1]);
	sprintf(s_coord[0], "%g", coord[0]);
	sprintf(s_coord[1], "%g", coord[1]);
	break;


    case (EQ):			/* We want equatorial coords */

	jcnvc2(file_jsys, file_equinox, coord[0], coord[1],
	    output_jsys, output_equinox, &coord[0], &coord[1], 0.0);

	coord_to_string(coord, s_coord);
	sprintf(result, "RA = %s  Dec = %s\n", s_coord[0], s_coord[1]);
	*equinox = output_equinox;
	*jsys = output_jsys;
	break;



    case (EC):			/* We want ecliptic coords   */

	jcnvc2(file_jsys, file_equinox, coord[0], coord[1],
	    output_jsys, output_equinox, &coord[0], &coord[1], 0.0);
#ifdef NOTDEF
	jcnvp2(file_jsys, 0.0, 0.0, 0.0, file_equinox, coord[0], coord[1],
	    0.0, output_jsys, output_equinox, &paout, &coord[0], &coord[1], &ierr);
#endif /* NOTDEF */

	coord_to_string(coord, s_coord);
	sprintf(result, "Elong = %s  Elat = %s\n", s_coord[0], s_coord[1]);
	*equinox = output_equinox;
	*jsys = output_jsys;
	break;



    case (GA):			/* We want galactic coords   */

	jcnvc2(file_jsys, file_equinox, coord[0], coord[1],
	    output_jsys, output_equinox, &coord[0], &coord[1], 0.0);

	coord_to_string(coord, s_coord);
	sprintf(result, "Glong = %s  Glat = %s\n", s_coord[0], s_coord[1]);
	*equinox = output_equinox;
	*jsys = output_jsys;
	break;

    case (SGAL):			/* We want supergalactic coords   */

	jcnvc2(file_jsys, file_equinox, coord[0], coord[1],
	    output_jsys, output_equinox, &coord[0], &coord[1], 0.0);

	coord_to_string(coord, s_coord);
	sprintf(result, "SGALlong = %s  SGALlat = %s\n", s_coord[0], s_coord[1]);
	*equinox = output_equinox;
	*jsys = output_jsys;
	break;
    }
    return (result);
}


/************************************************************************/
/*									*/
/* dIM_to_SC								*/
/* converts image coordinates to screen coordinates (double precision)	*/
/*									*/
/************************************************************************/

void
dIM_to_SC(d_coord)
double d_coord[2];
{

    if (debug)
	fprintf(debug_file,
	    "dIM_to_SC: in samp=%g, line=%g\n", d_coord[0], d_coord[1]);

    d_coord[0] = d_coord[0] - x_offset + crpix1;
    d_coord[1] = (d_coord[1] - y_offset + crpix2) * sgn_y;

    if (debug)
	fprintf(debug_file,
	    "dIM_to_SC: out x=%g, y=%g\n", d_coord[0], d_coord[1]);
}


/*********************************************************************/
/* 								     */
/* dSC_to_IM                                                         */
/* converts screen coordinates to image coordinates (double precision*/
/* 								     */
/*********************************************************************/

void
dSC_to_IM(d_coord)
double d_coord[2];
{
    if (debug)
	fprintf(debug_file,
	    "dSC_to_IM: in x=%g, y=%g\n", d_coord[0], d_coord[1]);

    d_coord[0] = d_coord[0] + x_offset - crpix1;
    d_coord[1] = d_coord[1] * sgn_y + y_offset - crpix2;

    if (debug)
	fprintf(debug_file, 
	    "dSC_to_IM: out samp=%g, line=%g\n", d_coord[0], d_coord[1]);
}



#ifdef NOTDEF
/*********************************************************************/
/* 								     */
/* EC_to_EQ                                                          */
/* converts ecliptic coordinates to equatorial (both in degrees)     */
/* 								     */
/*********************************************************************/

void
EC_to_EQ(coord)
double coord[2];
{
    int             i, j;
    double          ra, dec, lat, lon, x[3], xp[3];

    static double   A[3][3] = {1.000000000, 0.000000000, 0.000000000,
			       0.000000000, 0.917436945, -0.397881203,
			       0.000000000, 0.397881203, 0.917436945};

    if (debug == TRUE)
	fprintf(debug_file, "EC_to_EQ: in %-g %-g\n", coord[0], coord[1]);

    lon = coord[0] * dtr;
    lat = coord[1] * dtr;

    x[0] = cos(lon) * cos(lat);
    x[1] = sin(lon) * cos(lat);
    x[2] = sin(lat);

    for (i = 0; i < 3; ++i)
    {
	xp[i] = 0.;
	for (j = 0; j < 3; ++j)
	{
	    xp[i] += A[i][j] * x[j];
	}
    }

    if (xp[2] > 1.)
	xp[2] = 1.;

    if (xp[2] < -1.)
	xp[2] = -1.;

    dec = asin(xp[2]);
    ra = atan2(xp[1], xp[0]);

    if (ra < 0.)
	ra += 2.* M_PI;

    coord[0] = ra / dtr;
    coord[1] = dec / dtr;

    if (debug == TRUE)
	fprintf(debug_file, "EC_to_EQ: out %-g %-g\n", coord[0], coord[1]);

}



/*********************************************************************/
/* 								     */
/* GA_to_EQ                                                          */
/* converts galactic coordinates to equatorial (both in degrees)     */
/* 								     */
/*********************************************************************/

void
GA_to_EQ(coord)
double coord[2];
{
    int             i, j;
    double          ra, dec, lat, lon, x[3], xp[3];

    static double   A[3][3] = {-0.066988740, 0.492728470, -0.867600820,
			       -0.872755770, -0.450346960, -0.188374600,
			       -0.483538920, 0.744584640, 0.460199790};

    if (debug == TRUE)
	fprintf(debug_file, "GA_to_EQ: in %-g %-g\n", coord[0], coord[1]);

    lon = coord[0] * dtr;
    lat = coord[1] * dtr;

    x[0] = cos(lon) * cos(lat);
    x[1] = sin(lon) * cos(lat);
    x[2] = sin(lat);

    for (i = 0; i < 3; ++i)
    {
	xp[i] = 0.;
	for (j = 0; j < 3; ++j)
	{
	    xp[i] += A[i][j] * x[j];
	}
    }

    if (xp[2] > 1.)
	xp[2] = 1.;

    if (xp[2] < -1.)
	xp[2] = -1.;

    dec = asin(xp[2]);
    ra = atan2(xp[1], xp[0]);

    if (ra < 0.)
	ra += 2.* M_PI;

    coord[0] = ra * rtd;
    coord[1] = dec * rtd;

    if (debug == TRUE)
	fprintf(debug_file, "GA_to_EQ: out %-g %-g\n", coord[0], coord[1]);
}


/*********************************************************************/
/* 								     */
/* EQ_to_EC                                                          */
/* converts equatorial coordinates to ecliptic (both in degrees)     */
/* 								     */
/*********************************************************************/

void
EQ_to_EC(coord)
double coord[2];
{
    int             i, j;
    double          ra, dec, lat, lon, x[3], xp[3];

    static double   A[3][3] = {1.000000000, 0.000000000, 0.000000000,
			       0.000000000, 0.917436945, 0.397881203,
			       0.000000000, -0.397881203, 0.917436945};

    if (debug == TRUE)
	fprintf(debug_file, "EQ_to_EC: in %-g %-g\n", coord[0], coord[1]);

    ra = coord[0] * dtr;
    dec = coord[1] * dtr;

    x[0] = cos(ra) * cos(dec);
    x[1] = sin(ra) * cos(dec);
    x[2] = sin(dec);

    for (i = 0; i < 3; ++i)
    {
	xp[i] = 0.;
	for (j = 0; j < 3; ++j)
	{
	    xp[i] += A[i][j] * x[j];
	}
    }

    if (xp[2] > 1.)
	xp[2] = 1.;

    if (xp[2] < -1.)
	xp[2] = -1.;

    lat = asin(xp[2]);
    lon = atan2(xp[1], xp[0]);

    if (lon < 0.)
	lon += 2.* M_PI;

    coord[0] = lon * rtd;
    coord[1] = lat * rtd;

    if (debug == TRUE)
	fprintf(debug_file, "EQ_to_EC: out %-g %-g\n", coord[0], coord[1]);
}


/*********************************************************************/
/* 								     */
/* GA_to_EC                                                          */
/* converts galactic coordinates to ecliptic (both in degrees)       */
/* 								     */
/*********************************************************************/

void
GA_to_EC(coord)
double coord[2];
{
    if (debug == TRUE)
	fprintf(debug_file, "GA_to_EC: in %-g %-g\n", coord[0], coord[1]);

    GA_to_EQ(coord);
    EQ_to_EC(coord);

    if (debug == TRUE)
	fprintf(debug_file, "GA_to_EC: out %-g %-g\n", coord[0], coord[1]);
}



/*********************************************************************/
/* 								     */
/* EQ_to_GA                                                          */
/* converts equatorial coordinates to galactic (both in degrees)     */
/* 								     */
/*********************************************************************/

void
EQ_to_GA(coord)
double coord[2];
{
    int             i, j;
    double          ra, dec, lat, lon, x[3], xp[3];

    static double   A[3][3] = {-0.066988740, -0.872755770, -0.483538920,
			       0.492728470, -0.450346960, 0.744584640,
			       -0.867600820, -0.188374600, 0.460199790};

    if (debug == TRUE)
	fprintf(debug_file, "EQ_to_GA: in %-g %-g\n", coord[0], coord[1]);

    ra = coord[0] * dtr;
    dec = coord[1] * dtr;

    x[0] = cos(ra) * cos(dec);
    x[1] = sin(ra) * cos(dec);
    x[2] = sin(dec);

    for (i = 0; i < 3; ++i)
    {
	xp[i] = 0.;
	for (j = 0; j < 3; ++j)
	{
	    xp[i] += A[i][j] * x[j];
	}
    }

    if (xp[2] > 1.)
	xp[2] = 1.;

    if (xp[2] < -1.)
	xp[2] = -1.;

    lat = asin(xp[2]);
    lon = atan2(xp[1], xp[0]);

    if (lon < 0.)
	lon += 2.* M_PI;

    coord[0] = lon * rtd;
    coord[1] = lat * rtd;

    if (debug == TRUE)
	fprintf(debug_file, "EQ_to_GA: out %-g %-g\n", coord[0], coord[1]);

}



/*********************************************************************/
/* 								     */
/* EC_to_GA                                                          */
/* converts ecliptic coordinates to galactic (both in degrees)       */
/* 								     */
/*********************************************************************/

void
EC_to_GA(coord)
double coord[2];
{
    if (debug == TRUE)
	fprintf(debug_file, "EC_to_GA: in %-g %-g\n", coord[0], coord[1]);

    EC_to_EQ(coord);
    EQ_to_GA(coord);

    if (debug == TRUE)
	fprintf(debug_file, "EC_to_GA: in %-g %-g\n", coord[0], coord[1]);
}

#endif



/**********************************************************************/
/* 								     */
/* COORD_TO_STRING                                                    */
/* converts the coordinates to strings to be printed out.  The        */
/* variable 'sex' is used to decide whether to output in decimal      */
/* degrees or sexigesimal.                                            */
/* 								     */
/**********************************************************************/


void
coord_to_string(coord, s_coord)
double          coord[2];
char            s_coord[2][40];
{
    char            sign;
    int             hours, min, degs;
    double          dsec, c_hours, c_degs;

    if (debug == TRUE)
	fprintf(debug_file, 
	    "coord_to_string: lon,lat %-.8g %-.8g \n", coord[0], coord[1]);

    if ((sex == TRUE) && (s1_server_mode == FALSE))  /* server gets decimal */
    {
	if (coord[0] < 0.)
	    sign = '-';
	else
	    sign = ' ';

	/* Convert longitude */
	if (output_coord_sys == EQ)
	{
	    c_hours = coord[0] / 15;	/* use hours, min, sec */
	    if (c_hours < 0)
		c_hours += 24;	/* no negative hours */
	    c_hours += .005 / 3600;   /* round up one-half of .01 sec */
	}
	else
	{
	    c_hours = fabs(coord[0]);	/* use deg, min, sec */
	    c_hours += .05 / 3600;   /* round up one-half .1 sec */
	}

	hours = c_hours;
	min = (c_hours - hours) * 60.;
	dsec = (c_hours - hours - min / 60.) * 3600.;

	if (output_coord_sys == EQ)
	{
	    /* remove the rounding (f format does it) */
	    dsec -= .005;
	    if (dsec < 0.0)
		dsec = 0.0;
	    sprintf(s_coord[0], "%dh%02dm%05.2fs", hours, min, dsec);
	}
	else
	{
	    /* remove the rounding (f format does it) */
	    dsec -= .05;
	    if (dsec < 0.0)
		dsec = 0.0;
	    sprintf(s_coord[0], "%c%dd%02dm%04.1fs", sign, hours, min, dsec);
	}



	/* Convert latitude */
	if (coord[1] < 0.)
	    sign = '-';
	else
	    sign = ' ';

	c_degs = fabs(coord[1]);/* Convert latitude     */
	c_degs += .05 / 3600;   /* round up one-half of .1 sec */

	degs = c_degs;
	min = (c_degs - degs) * 60.;
	dsec = (c_degs - degs - min / 60.) * 3600.;

	/* remove the rounding (f format does it) */
	dsec -= .05;
	if (dsec < 0.0)
	    dsec = 0.0;
	sprintf(s_coord[1], "%c%dd%02dm%04.1fs", sign, degs, min, dsec);
    }

    else			/* if sex == FALSE  */
    {
	sprintf(s_coord[0], "%f", coord[0]);
	sprintf(s_coord[1], "%f", coord[1]);
    }
}



/**********************************************************************/
/* 								      */
/* SKY_TO_IMAGE                                                       */
/* Converts the input string to image coordinates (samp,line).        */
/* Uses the current image if the resulting samp,line is visible       */
/* on the screen.  Otherwise, finds an image with the coordinates     */
/* visible and fills the global variables from that image structure.  */
/* 								      */
/* With success, returns a pointer to the img structure from which    */
/* the global variables have been filled.                             */
/* 								      */
/* With failure (no visible image with specified coordinates),        */
/* returns NULL pointer and leaves global variables as at entry.      */
/* 								      */
/* The variable 'sex' is set by the called function str_to_coord      */
/* to reflect whether the input was read as decimal                   */
/* degrees or sexigesimal.                                            */
/* 								      */
/*  On entry:                                                         */
/*     s_coord[0] = lon (string)                                      */
/*     s_coord[1] = lat (string)                                      */
/*                                                                    */
/*  On exit:                                                          */
/*     Returns image pointer.                                         */
/*     d_coord2[0] = samp (double precision)                          */
/*     d_coord2[1] = line (double precision)                          */
/**********************************************************************/


struct img * sky_to_image(s_coord, d_coord2)
char s_coord[2][40];
double d_coord2[2];
{
    double          d_coord[2];
    struct img     *imgp;
    int             coord_sys;
    int i_coord[2];

    strcpy(last_coord[0], s_coord[0]);
    strcpy(last_coord[1], s_coord[1]);

    imgp = NULL;

    coord_sys = input_coord_sys;
    if ((file_coord_sys == IM   ) &&
	(coord_sys      != IRAF ) && 
	(coord_sys      != SC   ) && 
	(coord_sys      != XY   ))
    {
	coord_sys = IM;
    }

    switch (coord_sys)
    {
    case (IRAF):		/* We have IRAF coords */
	d_coord2[0] = atof(s_coord[0]);
	d_coord2[1] = atof(s_coord[1]);
	imgp = find_IRAF(d_coord2);/* may return NULL */
	/* convert to IM coords */
	dIRAF_to_IM(d_coord2);
	break;

    case (IM):			/* We have image coords */
	d_coord2[0] = atof(s_coord[0]);
	d_coord2[1] = atof(s_coord[1]);
	i_coord[0] = Round(d_coord2[0]);
	i_coord[1] = Round(d_coord2[1]);
	imgp = find_IM(i_coord);/* may return NULL */
	break;

    case (SC):			/* We have screen coords */
	d_coord2[0] = atof(s_coord[0]);
	d_coord2[1] = atof(s_coord[1]);
	i_coord[0] = Round(d_coord2[0]);
	i_coord[1] = Round(d_coord2[1]);
	imgp = find_SC(i_coord[0], i_coord[1]);	/* may return NULL */
	dSC_to_IM(d_coord2);
	break;

    case (XY):			/* We have XY */
	if (str_to_coord(s_coord, d_coord))
	{
	    imgp = find_COORD(d_coord, d_coord2);
	}
	break;

    case (EQ):			/* We have equatorial coords */
	if (str_to_coord(s_coord, d_coord))
	{
	    jcnvc2(input_jsys, input_equinox, d_coord[0], d_coord[1],
		file_jsys, file_equinox, &d_coord[0], &d_coord[1], 0.0);

	    imgp = find_COORD(d_coord, d_coord2);
	}
	break;

    case (EC):			/* We have ecliptic coords   */
	if (str_to_coord(s_coord, d_coord))
	{
	    jcnvc2(input_jsys, input_equinox, d_coord[0], d_coord[1],
		file_jsys, file_equinox, &d_coord[0], &d_coord[1], 0.0);

	    imgp = find_COORD(d_coord, d_coord2);
	}
	break;

    case (GA):			/* We have galactic coords   */
	if (str_to_coord(s_coord, d_coord))
	{
	    jcnvc2(input_jsys, input_equinox, d_coord[0], d_coord[1],
		file_jsys, file_equinox, &d_coord[0], &d_coord[1], 0.0);

	    imgp = find_COORD(d_coord, d_coord2);
	}
	break;
    case (SGAL):			/* We have galactic coords   */
	if (str_to_coord(s_coord, d_coord))
	{
	    jcnvc2(input_jsys, input_equinox, d_coord[0], d_coord[1],
		file_jsys, file_equinox, &d_coord[0], &d_coord[1], 0.0);

	    imgp = find_COORD(d_coord, d_coord2);
	}
	break;
    }
    return (imgp);
}

/**********************************************************************/
/* 								      */
/* DEPROJECT                                                          */
/* Converts from lat,lon to line,samp using global variables.         */
/* Input:  coord[0] = lon (degrees)                                   */
/*         coord[1] = lat (degrees)                                   */
/* Output: d_coord[0] = samp  (double precision)                      */
/*         d_coord[1] = line  (double precision)                      */
/* 								      */
/**********************************************************************/


void
deproject(coord, d_coord)
double coord[2], d_coord[2];
{
    double          lat, lon;
    double          rpp1, rpp2, lat0, lon0;
    double          aa, ff1, ff2, rho;
    double          fline, fsamp, rtwist, temp;
    char   pcode[4];
    struct celprm cel;
    double phi, theta;
    struct prjprm prj;
    int status, i, j;
    double xx,yy;
    double asin_arg;
    double fsamp_correction, fline_correction;
    double distance, distance_deg;
    double plate_ra_deg, plate_dec_deg;

    lon = coord[0] * dtr;
    lat = coord[1] * dtr;


    /* Deproject to image coordinates */

    switch(maptype)
    {
    case GNOMONIC:
	if (debug)
	    fprintf(debug_file, 
		"deproject: GNOMONIC lat,lon = %-g,%-g\n", lat, lon);

	rpp1 = -cdelt1 * dtr;
	rpp2 = -cdelt2 * dtr;

	lon0 = glong * dtr;
	lat0 = glat * dtr;

	aa = cos(lat) * cos(lon - lon0);
	ff1 = 1./ (sin(lat0) * sin(lat) + aa * cos(lat0));
	ff2 = 1./ (sin(lat0) * sin(lat) + aa * cos(lat0));

	if (ff1 < 0)
	{
	    /* we're more than 90 degrees from projection center */
	    /* return values which cannot be on any image */
	    fline = 1e+8;
	    fsamp = 1e+8;
	}
	else
	{
	    fline = -ff2 * (cos(lat0) * sin(lat) - aa * sin(lat0));
	    fsamp = -ff1 * cos(lat) * sin(lon - lon0);

	    if (using_cd)
	    {
		temp = -(dc1_1 * fsamp + dc1_2 * fline) * rtd;
		fline = -(dc2_1 * fsamp + dc2_2 * fline) * rtd;
		fsamp = temp;
	    }
	    else
	    {
		/* do the twist */
		rtwist = twist * dtr;	/* convert to radians */
		temp = fsamp * cos(rtwist) + fline * sin(rtwist);
		fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
		fsamp = temp;

		fsamp = (fsamp / rpp1);	/* now apply cdelt */
		fline = (fline / rpp2);
	    }

	    if (map_distortion)
	    {
		/* apply SIRTF distortion corrections */
		fsamp_correction = 0.0;
		for (i = 0; i <= ap_order; i++)
		{
		    for (j = 0; j <= ap_order; j++)
		    {
			if (i + j <= ap_order)
			{
			    fsamp_correction += 
				ap[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		if (debug == TRUE)
		    fprintf(debug_file,
			"deproject: fsamp correction = %f\n", fsamp_correction);

		fline_correction = 0.0;
		for (i = 0; i <= bp_order; i++)
		{
		    for (j = 0; j <= bp_order; j++)
		    {
			if (i + j <= bp_order)
			{
			    fline_correction += 
				bp[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		fsamp += fsamp_correction;
		fline += fline_correction;
		if (debug == TRUE)
		    fprintf(debug_file,
			"deproject: fline correction = %f\n", fline_correction);

	    }
	}

	break;

    case ORTHOGRAPHIC:
	if (debug)
	    fprintf(debug_file, 
		"deproject: ORTHOGRAPHIC lat,lon = %-g,%-g\n", lat, lon);

	rpp1 = -cdelt1 * dtr;
	rpp2 = -cdelt2 * dtr;

	lon0 = glong * dtr;
	lat0 = glat * dtr;

	fsamp = -cos(lat) * sin(lon - lon0);
	fline = -sin(lat) * cos(lat0) + cos(lat) * sin(lat0) * cos(lon - lon0);

	if (using_cd)
	{
	    temp = -(dc1_1 * fsamp + dc1_2 * fline) * rtd;
	    fline = -(dc2_1 * fsamp + dc2_2 * fline) * rtd;
	    fsamp = temp;
	}
	else
	{
	    /* do the twist */
	    rtwist = twist * dtr;	/* convert to radians */
	    temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	    fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	    fsamp = temp;

	    fsamp = (fsamp / rpp1);	/* now apply cdelt */
	    fline = (fline / rpp2);
	}

	    if (map_distortion)
	    {
		/* apply SIRTF distortion corrections */
		fsamp_correction = 0.0;
		for (i = 0; i <= ap_order; i++)
		{
		    for (j = 0; j <= ap_order; j++)
		    {
			if (i + j <= ap_order)
			{
			    fsamp_correction += 
				ap[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		if (debug == TRUE)
		    fprintf(debug_file,
			"deproject: fsamp_correction = %f\n", fsamp_correction);

		fline_correction = 0.0;
		for (i = 0; i <= bp_order; i++)
		{
		    for (j = 0; j <= bp_order; j++)
		    {
			if (i + j <= bp_order)
			{
			    fline_correction += 
				bp[i][j] * pow(fsamp, i) * pow(fline, j);
			}
		    }
		}
		fsamp += fsamp_correction;
		fline += fline_correction;
		if (debug == TRUE)
		    fprintf(debug_file,
			"deproject: fline correction = %f\n", fline_correction);

	    }

	break;


    case NCP:
	if (debug)
	    fprintf(debug_file, 
		"deproject: NCP lat,lon = %-g,%-g\n", lat, lon);

	rpp1 = -cdelt1 * dtr;
	rpp2 = -cdelt2 * dtr;

	lon0 = glong * dtr;
	lat0 = glat * dtr;

	fsamp = -cos(lat) * sin(lon - lon0);
	fline = (-cos(lat0) + cos(lat) * cos(lon - lon0)) / sin(lat0);

	/* do the twist */
	rtwist = twist * dtr;	/* convert to radians */
	temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	fsamp = temp;

	fsamp = (fsamp / rpp1);	/* now apply cdelt */
	fline = (fline / rpp2);

	break;

    case LAMBECYL:
	if (debug)
	    fprintf(debug_file, 
		"deproject: LAMBECYL lat,lon = %-g,%-g\n", lat, lon);

	rpp1 = cdelt1 * dtr;
	rpp2 = cdelt2 * dtr;

	lon0 = glong * dtr;

	fline = (1 / rpp2 * sin(lat));
	if (lon - lon0 > M_PI)
	    lon = lon - 2 * M_PI;
	if (lon0 - lon > M_PI)
	    lon = lon + 2 * M_PI;
	fsamp = ((lon - lon0) / rpp1);

	break;

    case AITOFF:
	if (debug)
	    fprintf(debug_file,
		"deproject: AITOFF   lat,lon = %-g,%-g\n", lat, lon);

	rpp1 = cdelt1 * dtr;
	rpp2 = cdelt2 * dtr;

	lon0 = glong * dtr;
	lat0 = glat * dtr;

	/* get delta-lon in range -180 to +180 */
	if ((lon - lon0) > M_PI)
	    lon -= 2 * M_PI;
	if ((lon - lon0) < -M_PI)
	    lon += 2 * M_PI;

	rho = acos(cos(lat) * cos((lon - lon0) / 2.));
	if ((rho < 0.0001) && (rho > -0.0001))
	{
	    theta = 0;
	}
	else
	{
	    asin_arg = cos(lat) * sin((lon - lon0) / 2.) / sin(rho);
	    if (asin_arg > 1.0)
		asin_arg = 1.0;
	    if (asin_arg < -1.0)
		asin_arg = -1.0;
	    theta = asin(asin_arg);
	}

	fsamp = 4./ rpp1 * sin(rho / 2.) * sin(theta);
	fline = 2./ rpp2 * sin(rho / 2.) * cos(theta);

	if (lat < 0.)
	    fline = -fline;

	break;

    case PLATE :
	/* DSS Plate stuff */

	/* avoid mirror image at opposite point on sphere */

	plate_ra_deg = curr_img->plate_ra * rtd;
	plate_dec_deg = curr_img->plate_dec * rtd;
	(void) dispos(&coord[0], &coord[1],
	    &plate_ra_deg, &plate_dec_deg, &distance);
	distance_deg = distance / 60.0;

	/* error if point is more than 90 degrees from plate center */
	if (distance_deg > 90.0)
	{
	    /* return values which cannot be on any image */
	    fline = 1e+8;
	    fsamp = 1e+8;
	}
	else
	{
	    amdinv(curr_img, lon, lat, 0.0, 0.0, &fsamp, &fline);
	}
	break;

    case CAR:
	if (debug)
	    fprintf(debug_file, 
	    "deproject: CAR lat,lon (degrees) = %-g,%-g\n", coord[0], coord[1]);
	strcpy(pcode, "CAR");

	/* Initialize projection parameters. */
	prj.flag = 0;
	prj.r0 = 0.0;
	for (j = 0; j < 10; prj.p[j++] = 0.0);

	/* Set reference angles for the native grid. */
	cel.ref[0] =   glong;
	cel.ref[1] =   glat;
	cel.ref[2] = 999.0;  /* should look for LONGPOLE in header */
	cel.ref[3] = 999.0;
	cel.flag   = 0;

	status = celfwd(pcode, coord[0], coord[1], &cel, &phi, &theta, 
	    &prj, &xx, &yy);

	fsamp = xx / cdelt1;
	fline = yy / cdelt2;

	/* do the twist */
	rtwist = - twist * dtr;	/* convert to radians */
	temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	fsamp = temp;

	break;

    case CEA:
	if (debug)
	    fprintf(debug_file, 
	    "deproject: CEA lat,lon (degrees) = %-g,%-g\n", coord[0], coord[1]);
	strcpy(pcode, "CEA");

	/* Initialize projection parameters. */
	prj.flag = 0;
	prj.r0 = 0.0;
	for (j = 0; j < 10; prj.p[j++] = 0.0);

	prj.p[1]= 1.0;  /* default */

	/* Set reference angles for the native grid. */
	cel.ref[0] =   glong;
	cel.ref[1] =   glat;
	cel.ref[2] = 999.0;  /* should look for LONGPOLE in header */
	cel.ref[3] = 999.0;
	cel.flag   = 0;

	status = celfwd(pcode, coord[0], coord[1], &cel, &phi, &theta, 
	    &prj, &xx, &yy);

	fsamp = xx / cdelt1;
	fline = yy / cdelt2;

	/* do the twist */
	rtwist = - twist * dtr;	/* convert to radians */
	temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	fsamp = temp;

	break;

    case SFL:
	if (debug)
	    fprintf(debug_file, 
	    "deproject: SFL lat,lon (degrees) = %-g,%-g\n", coord[0], coord[1]);
	strcpy(pcode, "SFL");

	/* Initialize projection parameters. */
	prj.flag = 0;
	prj.r0 = 0.0;
	for (j = 0; j < 10; prj.p[j++] = 0.0);

	/* Set reference angles for the native grid. */
	cel.ref[0] =   glong;
	cel.ref[1] =   glat;
	cel.ref[2] = 999.0;  /* should look for LONGPOLE in header */
	cel.ref[3] = 999.0;
	cel.flag   = 0;

	status = celfwd(pcode, coord[0], coord[1], &cel, &phi, &theta, 
	    &prj, &xx, &yy);

	fsamp = xx / cdelt1;
	fline = yy / cdelt2;

	/* do the twist */
	rtwist = - twist * dtr;	/* convert to radians */
	temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	fsamp = temp;

	break;

    case ARC:
	if (debug)
	    fprintf(debug_file, 
	    "deproject: ARC lat,lon (degrees) = %-g,%-g\n", coord[0], coord[1]);

	rpp1 = -cdelt1 * dtr;
	rpp2 = -cdelt2 * dtr;

	lon0 = glong * dtr;
	lat0 = glat * dtr;

	fsamp = sin(lon-lon0) * cos (lat);
	fline = sin (lat) * sin(lat0) + cos (lat) * cos(lat0) * cos(lon-lon0);
	if (fline <-1.0) fline = -1.0;
	if (fline >1.0) fline = 1.0;
	fline = acos (fline);
	if (fline!=0)
	    fline = fline / sin(fline);
	else
	    fline = 1.0;
	fsamp = fsamp * fline;
	fline = (sin (lat) * cos(lat0) - cos (lat) * sin(lat0) * cos(lon-lon0)) * fline;

	fsamp = - fsamp;
	fline = - fline;

	/* do the twist */
	rtwist = twist * dtr;	/* convert to radians */
	temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	fsamp = temp;

	fsamp = (fsamp / rpp1);	/* now apply cdelt */
	fline = (fline / rpp2);

	break;

    case LINEAR:
    case UNSPECIFIED:
    case UNRECOGNIZED:
	if (debug)
	    fprintf(debug_file, 
		"deproject: LINEAR  lon,lat = %-g,%-g\n", lon, lat);

	fsamp = ((lon / dtr - glong) / cdelt1);
	fline = ((lat / dtr - glat) / cdelt2);

	/* do the twist */
	rtwist = - twist * dtr;	/* convert to radians */
	temp = fsamp * cos(rtwist) + fline * sin(rtwist);
	fline = -fsamp * sin(rtwist) + fline * cos(rtwist);
	fsamp = temp;

	break;
    }

    d_coord[0] = fsamp;
    d_coord[1] = fline;

    if (debug)
	fprintf(debug_file, 
	    "     deproject:     fsamp,fline = %g,%g\n", fsamp, fline);

}



/**********************************************************************/
/* 								      */
/* STR_TO_COORD                                                       */
/* converts the string read in to coordinates.  The variable 'sex'    */
/* is set to reflect whether the input was read as decimal            */
/* degrees or sexigesimal.                                            */
/* 								      */
/* Returns TRUE if everything went ok, FALSE if something wrong       */
/* with the string.  Even with a bad string, it tries to return       */
/* something sensible.                                                */
/* 								      */
/**********************************************************************/


int
str_to_coord(s_coord, coord)
char s_coord[2][40];
double coord[2];
{
    int             i, j, sign;
    int             s_length, s_index, t_index;
    int             s_ind_old, found;
    int             hour_flag, sexa[2], retval;
    double          deghr, min, sec;
    char            temp[40], *p;
    int sexx;

    if ((!isnbr(s_coord[0])) || (!isnbr(s_coord[1])))
    {
	if (debug)
	    fprintf(debug_file, 
		"str_to_coord: %s or %s not a number\n", s_coord[0], s_coord[1]);
	return (FALSE);
    }

    /* Scan each coordinate to see if it is */
    /* given in sexigesimal */


    sexx = FALSE;
    sexa[0] = FALSE;
    sexa[1] = FALSE;



    for (j = 0; j < 2; j++)
    {
	p = s_coord[j];
	while (*p != '\0')
	{
	    if ((*p == 'h') || (*p == 'd') || (*p == 'm') || (*p == 's'))
	    {
		sexx = TRUE;
		sexa[j] = TRUE;
	    }
	    ++p;
	}
    }
    set_sex(sexx);


    if (sexx == FALSE)		/* If not sexigesimal, do conversion */
    {
	coord[0] = atof(s_coord[0]);
	coord[1] = atof(s_coord[1]);
	retval = TRUE;

	if (debug)
	    fprintf(debug_file,
		"str_to_coord: nosex  %-g  %-g\n", coord[0], coord[1]);
    }

    else			/* If sexigesimal, convert */
    {

	/* LONGITUDE and LATITUDE */
	for (j = 0; j < 2; j++)
	{
	    if (sexa[j] == FALSE)	/* if no h,d,m,or s, assume deg */
	    {
		coord[j] = atof(s_coord[j]);
		/* but flag as error since he gave only one in sexagesimal */
		retval = FALSE;	/* but flag as error */
	    }
	    else
	    {
		retval = TRUE;	/* assume true for now */

		sign = 1;

		s_length = strlen(s_coord[j]);

		for (i = 0; i < s_length; ++i)
		{
		    if (s_coord[j][i] == '-')
		    {
			s_coord[j][i] = ' ';
			sign = -1;
		    }
		}

		t_index = 0;	/* Process the degrees or hours */
		s_index = 0;

		s_ind_old = s_index;
		found = FALSE;
		hour_flag = FALSE;

		for (;;)
		{
		    if (s_coord[j][s_index] == 'h')
		    {
			found = TRUE;
			hour_flag = TRUE;
			++s_index;
			break;
		    }

		    else
		    if (s_coord[j][s_index] == 'd')
		    {
			found = TRUE;
			++s_index;
			break;
		    }

		    else
		    if (s_coord[j][s_index] == '\0')
			break;

		    else
		    {
			temp[t_index] = s_coord[j][s_index];
			++s_index;
			++t_index;
		    }
		}


		temp[t_index] = '\0';

		if (found == TRUE)
		{
		    deghr = atof(temp);
		}
		else
		{
		    deghr = 0;
		    s_index = s_ind_old;
		}



		t_index = 0;	/* Process the minutes */
		s_ind_old = s_index;
		found = FALSE;

		for (;;)
		{
		    if (s_coord[j][s_index] == 'm')
		    {
			found = TRUE;
			++s_index;
			break;
		    }

		    else
		    if (s_coord[j][s_index] == '\0')
			break;

		    else
		    {
			temp[t_index] = s_coord[j][s_index];

			++s_index;
			++t_index;
		    }
		}

		temp[t_index] = '\0';

		if (found == TRUE)
		{
		    min = atof(temp);
		}
		else
		{
		    min = 0;
		    s_index = s_ind_old;
		}



		t_index = 0;	/* Process the seconds */
		s_ind_old = s_index;
		found = FALSE;

		for (;;)
		{
		    if (s_coord[j][s_index] == 's')
		    {
			found = TRUE;
			break;
		    }

		    else
		    if (s_coord[j][s_index] == '\0')
			break;

		    else
		    {
			temp[t_index] = s_coord[j][s_index];

			++s_index;
			++t_index;
		    }
		}

		temp[t_index] = '\0';

		if (found == TRUE)
		{
		    sec = atof(temp);
		}
		else
		    sec = 0;

		coord[j] = sign * (deghr + min / 60.+ sec / 3600.);

		if (hour_flag == TRUE)
		    coord[j] = coord[j] * 15;

		if ((min > 60) || (sec > 60))
		    retval = FALSE;

		if (debug == TRUE)
		    fprintf(debug_file, 
		    "str_to_coord: coord[%d]  %s    %d*%-g %-g %-g    %-.8g\n",
			   j, s_coord[j], sign, deghr, min, sec, coord[j]);

	    }

	}
    }
    return (retval);
}
