/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "parse.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#define ON   1
#define OFF  0

typedef unsigned char clt_vals;

extern char current_clt[];

int fixed_ct = FALSE;

extern int debug, graphics, debug_val;
extern int server_mode;
extern char server_str[];
extern FILE *session, *debug_file;
extern char plane;


static int rr[256], gg[256], bb[256];
static clt_vals J_reds[256], J_greens[256], J_blues[256];
static clt_vals zeros[256];

static int *clt_ptr;
static int clt0[]=  {
		      0,   0,   0,   0,
		    255, 255, 255, 255
		    -1, -1, -1, -1
		    };
static int clt1[]=  {
		    255,   0,   0,   0,
		      0, 255, 255, 255
		    -1, -1, -1, -1
		    };
static int clt2[]=  {
		    0, 0, 0, 0,
		    17, 98, 0, 180,
		    34, 180, 0, 98,
		    51, 255, 0, 0,
		    68, 246, 88, 0,
		    85, 220, 181, 0,
		    102, 181, 220, 0,
		    119, 88, 245, 0,
		    136, 0, 255, 0,
		    153, 0, 235, 98,
		    170, 0, 180, 180,
		    187, 0, 98, 235,
		    204, 0, 0, 255,
		    221, 85, 128, 255,
		    238, 170, 192, 255,
		    255, 255, 255, 255,
		    -1, -1, -1, -1
		    };
static int clt3[]=  {
		      0,   0,   0,   0,
		     25,   0,   0, 150,
		     40,   0,   0, 255,
		     55,   0, 125, 225,
		     70,   0, 196, 196,
		     90,   0, 226, 125,
		    110,   0, 255,   0,
		    130, 150, 255,   0,
		    150, 225, 225,   0,
		    175, 255, 150,   0,
		    200, 255,   0,   0,
		    225, 196,   0, 196,
		    255, 255, 255, 255,
		    -1, -1, -1, -1
		    };
static int clt4[]=  {
		    0, 0, 0, 0,
		    17, 0, 0, 95,
		    34, 0, 20, 175,
		    51, 0, 63, 250,
		    68, 0, 112, 255,
		    85, 0, 162, 175,
		    102, 0, 210, 90,
		    119, 20, 255, 30,
		    136, 135, 240, 0,
		    153, 210, 205, 0,
		    170, 240, 135, 0,
		    187, 255, 20, 0,
		    204, 240, 0, 0,
		    221, 255, 85, 85,
		    238, 255, 170, 170,
		    255, 255, 255, 255,
		    -1, -1, -1, -1
		    };
static int clt5[]=  {
		    0,   255, 255, 255,
		    17,  255, 170, 170,
		    34,  255,  85,  85,
		    51,  240,   0,   0,
		    68,  255,  20,   0,
		    85,  240, 135,   0,
		    102, 210, 205,   0,
		    119, 135, 240,   0,
		    136,  20, 255,  30,
		    153,   0, 210,  90,
		    170,   0, 162, 175,
		    187,   0, 112, 255,
		    204,   0,  63, 250,
		    221,   0,  20, 175,
		    238,   0,   0,  95,
		    255,   0,   0,   0,
		    -1, -1, -1, -1
		    };
static int clt6[]=  {
		    0, 0, 0, 0,
		    17, 0, 0, 47,
		    34, 0, 0, 95,
		    51, 0, 10, 140,
		    68, 0, 20, 175,
		    85, 0, 41, 212,
		    102, 0, 63, 250,
		    119, 0, 83, 252,
		    136, 0, 112, 255,
		    153, 0, 137, 212,
		    170, 0, 162, 175,
		    187, 20, 255, 30,
		    204, 210, 205, 0,
		    221, 240, 135, 0,
		    238, 240, 0, 0,
		    255, 255, 255, 255,
		    -1, -1, -1, -1
		    };
static int clt7[]=  {
		    0, 255, 0, 0,
		    17, 255, 32, 32,
		    34, 255, 64, 64,
		    51, 255, 96, 96,
		    68, 255, 128, 128,
		    85, 255, 160, 160,
		    102, 255, 192, 192,
		    119, 255, 224, 224,
		    136, 224, 224, 255,
		    153, 192, 192, 255,
		    170, 160, 160, 255,
		    187, 128, 128, 255,
		    204, 96, 96, 255,
		    221, 64, 64, 255,
		    238, 32, 32, 255,
		    255, 0, 0, 255,
		    -1, -1, -1, -1
		    };

static int r=ON, g=ON, b=ON, ro=ON, go=ON, bo=ON, wo=ON, Bo=ON;
static int coff=0; 
int llow_save=0, hhigh_save=255;

/* function prototypes */
#ifdef _NO_PROTO

static void read_clt_file();
static void color_table();
static void set_color();

#else

static void read_clt_file(FILE *fclt, int *dn, int *red, int *green, int *blue);
static void color_table(void);
static void set_color(int dn, int r, int g, int b);

#endif /* _NO_PROTO */

/********************************************************************/
/*                                                                  */
/* COLOR_SETUP loads a gray scale lookup table                      */
/*                                                                  */
/********************************************************************/

void
color_setup()

{
    int i;

    strcpy(current_clt, "0");

    for(i=0; i<256; ++i)
    {
	rr[i] = i;
	gg[i] = i;
	bb[i] = i;

	J_reds[i] = i;
	J_greens[i] = i;
	J_blues[i] = i;
    }
}



/********************************************************************/
/*                                                                  */
/* CHGTBL reads in the color table from the pointed to file         */
/*                                                                  */
/********************************************************************/

void
chgtbl(argc, argv)
int argc;
char *argv[];

{
    double offset, gamma;
    int i, inc, clt_nbr;
    int dn, red, green, blue, old_dn, old_red, old_green, old_blue;
    char *fi_name;

    FILE *fclt;

    /* Check for information request */

    if(argc == 1)
    {
	if(server_mode == FALSE)
	   printf("Current color lookup table is %s\n", current_clt);
	else
	{
	    srv_string("color_table", current_clt);
	}

	return;
    }

    if (strcmp(argv[1], "iraf") == 0)
    {
	iraf_ct();
	return;
    }

    gamma = 1;
    fixed_ct = FALSE;
    if(argc > 2)
    {
        fixed_ct = TRUE;
	for(i=0; i<255; ++i)
	{
	    rr[i] = 0;
	    gg[i] = 0;
	    bb[i] = 0;
	}

	gamma = atof(argv[2]);

	if(gamma <= 0.)
	   gamma = 1.;
	
	if(gamma < 0.01)
	   gamma = 0.01;
	
	if(gamma > 10.)
	   gamma = 10.;
	
	gamma = 1./gamma;
    }

    /* Open the color lookup table file */


    fi_name = sdata_filename(argv[1]);
    if (fi_name == NULL)
    {
	fclt = NULL;
    }
    else
    {
	fclt = fopen(fi_name, "r");
    }
    if(fclt == NULL)
    {
	/* look for an internal color table (0 - 7) */
	clt_ptr = NULL;
	if (isint(argv[1]))
	{
	    clt_nbr = atoi(argv[1]);
	    switch(clt_nbr)
	    {
		case 0:
		    clt_ptr = clt0;
		    break;
		case 1:
		    clt_ptr = clt1;
		    break;
		case 2:
		    clt_ptr = clt2;
		    break;
		case 3:
		    clt_ptr = clt3;
		    break;
		case 4:
		    clt_ptr = clt4;
		    break;
		case 5:
		    clt_ptr = clt5;
		    break;
		case 6:
		    clt_ptr = clt6;
		    break;
		case 7:
		    clt_ptr = clt7;
		    break;
		default:
		    break;
	    }
	}
	if (clt_ptr == NULL)
	{
	    sprintf(server_str, "color table %s not found", argv[1]);
	    error1(server_str);
	    return;
	}
	fi_name = argv[1];
	fclt = stdin;  /* flag as internal color table */
    }

    strcpy(current_clt, fi_name);

    /* Read in the data, interpolating as we go */

    read_clt_file(fclt, &dn, &red, &green, &blue);

    if(fixed_ct == TRUE && dn >= 0 && dn <= 255)
    {
        rr[dn] = (int)(255. * pow(((double)   red / 255.), gamma));
        gg[dn] = (int)(255. * pow(((double) green / 255.), gamma));
        bb[dn] = (int)(255. * pow(((double)  blue / 255.), gamma));
    }

    for (;;)
    {
	old_dn = dn;
	old_red = red;
	old_green = green;
	old_blue = blue;

	read_clt_file(fclt, &dn, &red, &green, &blue);

	if(debug)
	{
	    fprintf(debug_file, 
		"color: input dn=%d, red=%d, green=%d, blue=%d\n",
		dn, red, green, blue);
	}

        if(fixed_ct == TRUE && dn >= 0 && dn <= 255)
        {
            rr[dn] = (int)(255. * pow(((double)   red / 255.), gamma));
            gg[dn] = (int)(255. * pow(((double) green / 255.), gamma));
            bb[dn] = (int)(255. * pow(((double)  blue / 255.), gamma));
	    continue;
        }

	if (dn < 0)   /* if EOF */
	    break;

	inc = 1;
	if(dn < old_dn)
	    inc = -1;

	for(i = old_dn; i != dn; i += inc)
	{
	    if(i>=0 && i<=255)
	    {
		offset = (double) (i-old_dn) / (dn-old_dn);
		rr[i] = old_red   + (int) (offset*(red   - old_red  ));
		gg[i] = old_green + (int) (offset*(green - old_green));
		bb[i] = old_blue  + (int) (offset*(blue  - old_blue ));

		if(debug_val == 7)
		{
		    fprintf(debug_file,
			"color: new dn=%d, red=%d, green=%d, blue=%d\n",
			i, rr[i], gg[i], bb[i]);
		}
	    }
	}

    }

    if(old_dn>=0 && old_dn<=255)
    {
	rr[old_dn] = old_red;
	gg[old_dn] = old_green;
	bb[old_dn] = old_blue;
    }

    if (fclt != stdin)
	fclose(fclt);

    if(fixed_ct == TRUE)
	stretch(0, 255);
    else
	stretch(llow_save, hhigh_save);

    if(server_mode == TRUE)
    {
	srv_string("color_table", current_clt);
    }
}

/* IRAF_CT  */
/* set color table as IRAF expects it */
void
iraf_ct()
{
    int dn;

    fixed_ct = TRUE;
    for (dn = 0; dn < 200; dn++)
    {
        rr[dn] = dn;
        gg[dn] = dn;
        bb[dn] = dn;
    }
    set_color(200, 255, 255, 255);
    set_color(201, 255, 255, 255);
    set_color(202,   0,   0,   0);
    set_color(203, 255, 255, 255);
    set_color(204, 255,   0,   0);
    set_color(205,   0, 255,   0);
    set_color(206,   0,   0, 255);
    set_color(207, 255, 255,   0);
    set_color(208,   0, 255, 255);
    set_color(209, 255,   0, 255);
    set_color(210, 255, 127,  80);
    set_color(211, 176,  48,  96);
    set_color(212, 255, 165,   0);
    set_color(213, 240, 230, 140);
    set_color(214, 218, 112, 214);
    set_color(215,  64, 224, 208);
    set_color(216, 238, 130, 238);
    set_color(217, 245, 222, 179);
    stretch(0, 255);
}

static void
set_color(dn, r, g, b)
int dn, r, g, b;
{
    rr[dn] = r;
    gg[dn] = g;
    bb[dn] = b;
}



/********************************************************************/
/*                                                                  */
/* READ_CLT_FILE                                                    */
/* reads one line of the file containing the clt information        */
/*                                                                  */
/*  Returns dn = -1 to flag EOF                                     */
/*                                                                  */
/*   Skips lines beginning with #           			    */
/*   Discards longer than MAXSTRLEN characters        		    */
/*                                                                  */
/********************************************************************/
#define MAXSTRLEN 255 /* Maximum allowable line length */

static void
read_clt_file(fclt, dn, red, green, blue)
FILE *fclt;
int *dn, *red, *green, *blue;
{
    char linebuffer[MAXSTRLEN+2]; /* number of chars plus \n\0 */

    if (fclt == stdin)
    {
	/* internal table file */
	*dn = *clt_ptr++;
	*red = *clt_ptr++;
	*green = *clt_ptr++;
	*blue = *clt_ptr++;
    }
    else
    {
	while (fgets(linebuffer, MAXSTRLEN+1, fclt) !=  (char *) NULL )
	{
	    if(linebuffer[strlen(linebuffer) -1] != '\n')
	    {
		if (debug) 
		    fprintf(debug_file, "Line Too Long... Flushing tail\n");

		/* flush remainder of long line */
		while(!feof(fclt) && (fgetc(fclt) != '\n')); 
	    }
	    else if(*linebuffer != '#')    /* if not a comment line */
	    {
		sscanf(linebuffer, "%d %d %d %d\n", dn, red, green, blue);
		return;
	    }
	}
	*dn = -1;    /* flag end of file */
    }
}


/********************************************************************/
/*                                                                  */
/* CUTOFF                                                           */
/* sets up a cutoff level for the color table (in image display     */
/* memory DN)                                                       */
/*                                                                  */
/********************************************************************/

void
cutoff(argc, argv)
int argc;
char **argv;

{
     if(argc != 1)
     {
	 coff = atoi(argv[1]);
	 stretch(llow_save, hhigh_save);
     }
     
     if(server_mode == FALSE)
	 printf("cutoff is at %d\n", coff);
     else
     {
	 sprintf(server_str, "%d", coff);
	 srv_real("cutoff", server_str);
     }
}



/********************************************************************/
/*                                                                  */
/* ON                                                               */
/* turns on display of the red, green, blue or all images           */
/* or the red, green, blue, or white overlay planes                 */
/*                                                                  */
/********************************************************************/

void
on(argc, argv)
int argc;
char **argv;

{
    int i;
    char *p;

    if (!graphics)
	return;

    /*  Determine which planes are to be turned on */

    if(argc == 1)
    {
	r = ON;
	g = ON;
	b = ON;
	ro = ON;
	go = ON;
	bo = ON;
	wo = ON;
	Bo = ON;
	off_all_X_overlay(1);
    }

    for(i = 1; i < argc; ++i)
    {
	p = argv[i];

	if(strcmp(p,"a") == 0)
	{
	    r = ON;
	    g = ON;
	    b = ON;
	}

	else if(strcmp(p,"ao") == 0)
	{
	    ro = ON;
	    go = ON;
	    bo = ON;
	    wo = ON;
	    Bo = ON;
	    off_all_X_overlay(1);
	    continue;
	}

	else if(strcmp(p,"r") == 0)
	    r = ON;

	else if(strcmp(p,"g") == 0)
	    g = ON;

	else if(strcmp(p,"b") == 0)
	    b = ON;

	else if(strcmp(p,"ro") == 0)
	{
	    ro = ON;
	    off_X_overlay(p, 1);
	}

	else if(strcmp(p,"go") == 0)
	{
	    go = ON;
	    off_X_overlay(p, 1);
	}

	else if(strcmp(p,"bo") == 0)
	{
	    bo = ON;
	    off_X_overlay(p, 1);
	}

	else if(strcmp(p,"wo") == 0)
	{
	    wo = ON;
	    off_X_overlay(p, 1);
	}

	else if(strcmp(p,"Bo") == 0)
	{
	    Bo = ON;
	    off_X_overlay(p, 1);
	}
	else
	    off_X_overlay(p, 1);
    }

    color_table();
}




/********************************************************************/
/*                                                                  */
/* OFF                                                              */
/* turns off display of the red, green, blue or all images          */
/* or the red, green, blue, or white overlay planes                 */
/*                                                                  */
/********************************************************************/


void
off(argc, argv)
int argc;
char **argv;
{
    int i;
    char *p;

    if (!graphics)
	return;

    /*  Determine which planes are to be turned off */

    if(argc == 1)
    {
	r = OFF;
	g = OFF;
	b = OFF;
	ro = OFF;
	go = OFF;
	bo = OFF;
	wo = OFF;
	Bo = OFF;
	off_all_X_overlay(0);
    }

    for(i = 1; i < argc; ++i)
    {
	p = argv[i];

	if(strcmp(p,"a") == 0)
	{
	    r = OFF;
	    g = OFF;
	    b = OFF;
	}

	else if(strcmp(p,"ao") == 0)
	{
	    ro = OFF;
	    go = OFF;
	    bo = OFF;
	    wo = OFF;
	    Bo = OFF;
	    off_all_X_overlay(0);
	    continue;
	}

	else if(strcmp(p,"r") == 0)
	    r = OFF;

	else if(strcmp(p,"g") == 0)
	    g = OFF;

	else if(strcmp(p,"b") == 0)
	    b = OFF;

	else if(strcmp(p,"ro") == 0)
	{
	    ro = OFF;
	    off_X_overlay(p, 0);
	}

	else if(strcmp(p,"go") == 0)
	{
	    go = OFF;
	    off_X_overlay(p, 0);
	}

	else if(strcmp(p,"bo") == 0)
	{
	    bo = OFF;
	    off_X_overlay(p, 0);
	}

	else if(strcmp(p,"wo") == 0)
	{
	    wo = OFF;
	    off_X_overlay(p, 0);
	}

	else if(strcmp(p,"Bo") == 0)
	{
	    Bo = OFF;
	    off_X_overlay(p, 0);
	}
	else
	    off_X_overlay(p, 0);
    }
     
    color_table();
}



/********************************************************************/
/*                                                                  */
/* COLOR_TABLE                                                      */
/* uses the color table entries (J_reds[] etc) calculated elsewhere */
/* to fill the lookup tables.                                       */
/* Sets up the POMs for four overlay colors                         */
/* (red, green, blue, and white)                                    */
/*                                                                  */
/********************************************************************/

static void
color_table()
{
    int i, mval;

    zoom_setup();   /* enable all boards to avoid psychadelics */

    set_overlay();
    mval = (Bo << 4) + (wo << 3) + (bo << 2) + (go << 1) + ro;
    set_overlay_mask(mval);

    for (i=0; i<256; i++)
	zeros[i] = 0;

    zeros[255] = 255;
    J_reds[255] = 255;
    J_greens[255] = 255;
    J_blues[255] = 255;

    mval = (r << 2) + (g << 1) + b;

    switch(mval)
	{
	case 0:  /* all off */
	    set_clt(zeros, zeros, zeros);
	    break;

	case 1:  /* blue only */
	    set_clt(zeros, zeros, J_blues);
	    break;

	case 2:  /* green only */
	    set_clt(zeros, J_greens, zeros);
	    break;

	case 3:  /* green and blue */
	    set_clt(zeros, J_greens, J_blues);
	    break;

	case 4:  /* red only */
	    set_clt(J_reds, zeros, zeros);
	    break;

	case 5:  /* red and blue */
	    set_clt(J_reds, zeros, J_blues);
	    break;

	case 6:  /* red and green */
	    set_clt(J_reds, J_greens, zeros);
	    break;

	case 7:  /* red, green and blue */
	    set_clt(J_reds, J_greens, J_blues);
	    break;
        }

    image_setup(plane);  /* restore previous board mapping */
    flush_graphics();
}



/**********************************************************************/
/*                                                                    */
/* STRETCH                                                            */
/* interpolates the color table to produce a stretched one            */
/* only the enabled planes tables are stretched                       */
/* Uses rr[], gg[], and bb[]                                          */
/* Sets J_reds[], J_greens[], and J_blues[], then calls color_table   */
/* to put these values into the hardware.                             */
/*                                                                    */
/**********************************************************************/


void
stretch(llow, hhigh)
int llow, hhigh;

{
     int i, ind, dind;

     llow_save = llow;
     hhigh_save = hhigh;

     dind = hhigh - llow;
     if(dind == 0)
	  dind = 1;

     switch(plane)
	  {
	  case 'a':   /* all planes  red, green, blue  */

		    r = ON;
		    g = ON;
		    b = ON;

		    for (i=0; i<256; ++i)
			 {
			 ind = 255 * (i - llow) / dind;

			 if(ind < 0) 
			      ind = 0;

			 if(ind > 255) 
			      ind = 255;

			 J_reds[i]   = rr[ind];
			 J_greens[i] = gg[ind];
			 J_blues[i]  = bb[ind];

			 if(i < coff)
			     {
			     J_reds[i]   = 0;
			     J_greens[i] = 0;
			     J_blues[i]  = 0;
			     }
			 }

		    break;


	  case 'r':   /* red plane only */

		    r = ON;

		    for (i=0; i<256; ++i)
			 {
			 ind = 255 * (i - llow) / dind;

			 if(ind < 0) 
			      ind = 0;

			 if(ind > 255) 
			      ind = 255;

			 J_reds[i]   = rr[ind];

			 if(i < coff)
			     {
			     J_reds[i]   = 0;
			     }
			 }

		    break;


	  case 'g':   /* green plane only */

		    g = ON;

		    for (i=0; i<256; ++i)
			 {
			 ind = 255 * (i - llow) / dind;

			 if(ind < 0) 
			      ind = 0;

			 if(ind > 255) 
			      ind = 255;

			 J_greens[i] = gg[ind];

			 if(i < coff)
			     {
			     J_greens[i] = 0;
			     }
			 }

		    break;


	  case 'b':   /* blue plane only  */

		    b = ON;

		    for (i=0; i<256; ++i)
			 {
			 ind = 255 * (i - llow) / dind;

			 if(ind < 0) 
			      ind = 0;

			 if(ind > 255) 
			      ind = 255;

			 J_blues[i]  = bb[ind];

			 if(i < coff)
			     {
			     J_blues[i]  = 0;
			     }
			 }

		    break;
	  }


     color_table(); 
}




/*******************************************************************/
/*                                                                 */
/* ST_COLOR stretches the color table                              */
/*                                                                 */
/*******************************************************************/

void
st_cmd(argc, argv)
int argc;
char *argv[];
{
    int low, high;

    if(fixed_ct == TRUE)
    {
        error1("Can't stretch a fixed color table");
        return;
    }

    if (argc == 3)  /* The stretch has been specified */
    {
	if ((getint(argv[1], &low)) && (getint(argv[2], &high)))
	stretch(low, high);

	if(server_mode == TRUE)
	{
	   sprintf(server_str, "%d", low);
	   srv_real("stretch_min", server_str);
	   sprintf(server_str, "%d", high);
	   srv_real("stretch_max", server_str);
	}

	return;
    }

    if (argc == 1)  /* The stretch is to be by trackball */
    {
	st_ball();
	return;
    }

    error1("wrong number of arguments to ST command");
}
