/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/
#include "fits_table.h"
#include "im_proto.h"
#include "cam2fits.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#define UNINTERLEAVED

#define MAXFIELDS 50
#define MAXTABLELINE 500

static char ID[] = "@(#)cam2fits.c   RBH        06/27/96";

char base_filename[1024];
int starting_index = 1, ending_index = INT_MAX, step_size = 1;
extern char server_str[];
extern int long_header;
extern int nparms;
extern int silent;
extern int control_c;
int cube = 0;
int info_request = 0;
int header_bytes;
int want_resets = 1;


#ifdef _NO_PROTO
static void get_col();
static void do_long_header();
#else
static void get_col(int unit, char *column_name, int *column_index);
static void do_long_header(int unit, int fdout);
#endif /* _NO_PROTO */


void
do_generic(unit)
int unit;
{
    double pscale, poffs;
    char pblank[1], punit[1];
    char pname[80];
    char ptype;
    int pcol, plen, pelements;
    int header_total, data_total;
    int header_raw_total, data_raw_total;
    char output_filename[80];
    int fdout, imstat;
    int bitpix_col, naxis_col, naxis1_col, naxis2_col, naxis3_col;
    int bscale_col, b_zero_col, array_col;
    int blank_col, bunit_col;
    int blank, bitpix, naxis, naxis1, naxis2, naxis3, index;
    double bscale, b_zero;
    char bunit[13];
    char *array_charadr;

    get_col(unit, "BITPIX", &bitpix_col);
    get_col(unit, "NAXIS", &naxis_col);
    get_col(unit, "NAXIS1", &naxis1_col);
    get_col(unit, "NAXIS2", &naxis2_col);
    get_col(unit, "NAXIS3", &naxis3_col);
    get_col(unit, "BSCALE", &bscale_col);
    get_col(unit, "BZERO", &b_zero_col);
    get_col(unit, "BLANK", &blank_col);
    get_col(unit, "BUNIT", &bunit_col);

    get_col(unit, "ARRAY", &array_col);
    if (array_col == 0)
    {
	error1("unable to continue without ARRAY column\n");
	return;
    }
    tgetaddr(unit, array_col, &array_charadr);
    tpinfo(unit, array_col, pname, &ptype, pblank, punit, &pscale, &poffs,
	&pcol, &plen, &pelements);

    if (info_request)
    {
	header_raw_total = 960 + header_bytes - 320 + nparms * 80 - 880;
	header_total = header_raw_total;
	if ((header_total % 2880) != 0)
	    header_total = ((header_total / 2880) +1) * 2880;
	data_raw_total = 1024 * 4;
	data_total = data_raw_total;
	if ((data_total % 2880) != 0)
	    data_total = ((data_total / 2880) +1) * 2880;
#ifdef DEBUG
	printf("primary and fixed part of extended header = %d bytes\n",
	    header_bytes);
	printf(
	    "each file has %d field header lines * 80 = %d\n",
	    nparms, nparms * 80);
	printf( "total header is %d\n", header_raw_total);
	printf("  header rounded up to %d\n", header_total);
	printf("data array is %d bytes\n", data_raw_total);
	printf("  data rounded up to %d\n", data_total);
#endif /* DEBUG */
	printf(
"program will produce %d image files, each approximately %d bytes long\n",
	    tgetrowcount(unit), header_total + data_total);

	exit(0);
    }

    if (cube)
    {
	error1(
    "cube option only implemented for CIER, CISP, CCIM, CMAP, and CMOS files");
	return;
    }

#ifdef DEBUG

    printf(
"INDEX BITPIX NAXIS NAXIS1 NAXIS2 BSCALE BZERO POLARISE DATAREF\n");
#endif /* DEBUG */

    for (index = 1; index <= ending_index; index++)
    {
	tblread(unit);
	if (itstat() > 0 )
	{
	    error1("error reading tablefile");
	    exit(1);
	}
	else if (itstat() < 0)
	{
	    break;
	}
	    tgeti(unit, bitpix_col, &bitpix);
	    tgeti(unit, naxis_col, &naxis);
	    tgeti(unit, naxis1_col, &naxis1);
	    tgeti(unit, naxis2_col, &naxis2);
	    if (naxis3_col != 0)
		tgeti(unit, naxis3_col, &naxis3);
	    tgetd(unit, bscale_col, &bscale);
	    tgetd(unit, b_zero_col, &b_zero);
	    tgeti(unit, blank_col, &blank);
	    tgetc(unit, bunit_col, bunit);

	if ((index < starting_index) || (index > ending_index) || 
	    ((index - starting_index) % step_size))
	    continue;
#ifdef DEBUG
	printf("%d %d  %d %d %d  %e  %e\n", 
	    index, bitpix, naxis, naxis1, naxis2, bscale, b_zero);
#endif /* DEBUG */

#ifdef NOTDEF
	array_adr = (int *) array_charadr;
	for (i = 0; i < 16; i++)
	{
	    printf("array[%d] = %9.3e\n", i, (double) *array_adr++);
	}
#endif /* NOTDEF */

	sprintf(output_filename, "%s_%04d.fits", base_filename, index);

	if (!silent)
	    printf("opening file [%s]\n", output_filename);

	fdout = im_open(output_filename, "w");
	if (fdout < 0)
	{
	    perror(output_filename);
	    exit(1);
	}

	if (bitpix == 0)
	    bitpix = 32;

	/* override bitpix mistake */
	if (ptype == 'J')
	    bitpix = 32;
	else if (ptype == 'I')
	    bitpix = 16;
	imstat = im_wkey_i(fdout, bitpix, "BITPIX");
	if (naxis == 0)
	    naxis = 2;
	imstat = im_wkey_i(fdout, naxis, "NAXIS");
	if (naxis1 == 0)
	    naxis1 = 32;
	imstat = im_wkey_i(fdout, naxis1, "NAXIS1");
	if (naxis2 == 0)
	    naxis2 = 32;
	imstat = im_wkey_i(fdout, naxis2, "NAXIS2");
	if (naxis3_col != 0)
	    imstat = im_wkey_i(fdout, naxis3, "NAXIS3");
	imstat = im_wkey_d(fdout, bscale, "BSCALE");
	imstat = im_wkey_d(fdout, b_zero, "BZERO");
	imstat = im_wkey_i(fdout, blank, "BLANK");
	imstat = im_wkey_c(fdout, bunit, "BUNIT");

	if (long_header)
	    do_long_header(unit, fdout);

	if (ptype == 'J')
	    imstat = im_wpix_i(fdout, (int *) array_charadr, pelements);
	else if (ptype == 'I')
	    imstat = im_wpix_s(fdout, (short *) array_charadr, pelements);
	else
	{
	    error1("unimplemented bitpix");
	}
	if (imstat < 0)
	{
	    sprintf(server_str,
		"error writing pixels to file %s", output_filename);
	    error1(server_str);
	    return;
	}

	imstat = im_close(fdout);
	file_done(output_filename);
	if (control_c)
	    return;
    }

}

void
do_ccim(unit)
int unit;
{
    int header_total, data_total;
    int header_raw_total, data_raw_total;
    char output_filename[80];
    int fdout, imstat;
    int bitpix_col, naxis_col, naxis1_col, naxis2_col;
    int bscale_col, b_zero_col, array_col, index_col, channel_col;
    int blank_col, bunit_col, type_col;
    int blank, bitpix, naxis, naxis1, naxis2, index;
    double bscale, b_zero;
    char type[20];
    char bunit[13];
    char channel[5];
    char *array_charadr;
    int *array_adr;
#ifdef DEBUG
    int polarise_col, dataref_col;
    char polarise[80];
    char dataref[80];
#endif /* DEBUG */

    get_col(unit, "INDEX", &index_col);
    get_col(unit, "CHANNEL", &channel_col);
    get_col(unit, "BITPIX", &bitpix_col);
    get_col(unit, "NAXIS", &naxis_col);
    get_col(unit, "NAXIS1", &naxis1_col);
    get_col(unit, "NAXIS2", &naxis2_col);
    get_col(unit, "BSCALE", &bscale_col);
    get_col(unit, "BZERO", &b_zero_col);
    get_col(unit, "BLANK", &blank_col);
    get_col(unit, "BUNIT", &bunit_col);
    get_col(unit, "TYPE", &type_col);

    get_col(unit, "ARRAY", &array_col);
    tgetaddr(unit, array_col, &array_charadr);

    if (info_request)
    {
	header_raw_total = 960 + header_bytes - 320 + nparms * 80 - 880;
	header_total = header_raw_total;
	if ((header_total % 2880) != 0)
	    header_total = ((header_total / 2880) +1) * 2880;
	data_raw_total = 1024 * 4;
	data_total = data_raw_total;
	if ((data_total % 2880) != 0)
	    data_total = ((data_total / 2880) +1) * 2880;
#ifdef DEBUG
	printf("primary and fixed part of extended header = %d bytes\n",
	    header_bytes);
	printf(
	    "each file has %d field header lines * 80 = %d\n",
	    nparms, nparms * 80);
	printf( "total header is %d\n", header_raw_total);
	printf("  header rounded up to %d\n", header_total);
	printf("data array is %d bytes\n", data_raw_total);
	printf("  data rounded up to %d\n", data_total);
#endif /* DEBUG */
	printf(
"program will produce %d image files, each approximately %d bytes long\n",
	    tgetrowcount(unit), header_total + data_total);

	exit(0);
    }

    if (cube)
    {
	sprintf(output_filename, "%s_cube.fits", base_filename);
	if (!silent)
	    printf("opening file [%s]\n", output_filename);
	fdout = im_open(output_filename, "w");
	if (fdout < 0)
	{
	    perror(output_filename);
	    exit(1);
	}
	imstat = im_wkey_i(fdout, 32, "BITPIX");
	imstat = im_wkey_i(fdout, 3, "NAXIS");
	naxis1 = 32;
	imstat = im_wkey_i(fdout, naxis1, "NAXIS1");
	naxis2 = 32;
	imstat = im_wkey_i(fdout, naxis2, "NAXIS2");
	if (ending_index == INT_MAX)
	    ending_index = tgetrowcount(unit);

	if (ending_index > tgetrowcount(unit))
	{
	    sprintf(server_str, 
		"ending index greater than number of rows in file (%d)",
		tgetrowcount(unit));
	    error1(server_str);
	    return;
	}
	if (starting_index > ending_index)
	{
	    error1("starting index must be less than ending index");
	    return;
	}
	imstat = im_wkey_i(fdout, 
	    (ending_index - starting_index) / step_size + 1, "NAXIS3");
    }

#ifdef DEBUG
    get_col(unit, "POLARISE", &polarise_col);
    get_col(unit, "DATAREF", &dataref_col);

    printf(
"INDEX BITPIX NAXIS NAXIS1 NAXIS2 BSCALE BZERO POLARISE DATAREF\n");
#endif /* DEBUG */

    for (;;)
    {
	tblread(unit);
	if (itstat() > 0 )
	{
	    error1("error reading tablefile");
	    exit(1);
	}
	else if (itstat() < 0)
	{
	    break;
	}
	    tgeti(unit, index_col, &index);
	    tgetc(unit, channel_col, channel);
	    tgeti(unit, bitpix_col, &bitpix);
	    tgeti(unit, naxis_col, &naxis);
	    tgeti(unit, naxis1_col, &naxis1);
	    tgeti(unit, naxis2_col, &naxis2);
	    tgetd(unit, bscale_col, &bscale);
	    tgetd(unit, b_zero_col, &b_zero);
	    tgeti(unit, blank_col, &blank);
	    tgetc(unit, bunit_col, bunit);
	    tgetc(unit, type_col, type);

	if ((index < starting_index) || (index > ending_index) || 
	    ((index - starting_index) % step_size))
	    continue;
#ifdef DEBUG
	tgetc(unit, polarise_col, polarise);
	tgetc(unit, dataref_col, dataref);
	printf("%d %d  %d %d %d  %e  %e %s %s\n", 
	    index, bitpix, naxis, naxis1, naxis2, bscale, b_zero, polarise, 
	    dataref);
#endif /* DEBUG */

#ifdef NOTDEF
	array_adr = (int *) array_charadr;
	for (i = 0; i < 16; i++)
	{
	    printf("array[%d] = %9.3e\n", i, (double) *array_adr++);
	}
#endif /* NOTDEF */

	array_adr = (int *) array_charadr;

	if (bitpix == 0)
	    bitpix = 32;
	if (naxis == 0)
	    naxis = 2;
	if (naxis1 == 0)
	    naxis1 = 32;
	if (naxis2 == 0)
	    naxis2 = 32;

	if (!cube)
	{
	    sprintf(output_filename, "%s_%04d.fits", base_filename, index);

	    if (!silent)
		printf("opening file [%s]\n", output_filename);

	    fdout = im_open(output_filename, "w");
	    if (fdout < 0)
	    {
		perror(output_filename);
		exit(1);
	    }

	    imstat = im_wkey_i(fdout, bitpix, "BITPIX");
	    imstat = im_wkey_i(fdout, naxis, "NAXIS");
	    imstat = im_wkey_i(fdout, naxis1, "NAXIS1");
	    imstat = im_wkey_i(fdout, naxis2, "NAXIS2");
	    imstat = im_wkey_d(fdout, bscale, "BSCALE");
	    imstat = im_wkey_d(fdout, b_zero, "BZERO");
	    imstat = im_wkey_i(fdout, blank, "BLANK");
	    imstat = im_wkey_c(fdout, bunit, "BUNIT");
	    imstat = im_wkey_c(fdout, channel, "CHANNEL");
	    imstat = im_wkey_i(fdout, index, "INDEX");
	    imstat = im_wkey_c(fdout, type, "TYPE");

	    if (long_header)
		do_long_header(unit, fdout);
	}

	imstat = im_wpix_i(fdout, array_adr, naxis1*naxis2);
	if (imstat < 0)
	{
	    sprintf(server_str,
		"error writing pixels to file %s", output_filename);
	    error1(server_str);
	    return;
	}

	if (!cube)
	{
	    imstat = im_close(fdout);
	    file_done(output_filename);
	    if (control_c)
		return;
	}
    }
    if (cube)
	imstat = im_close(fdout);

}

void
do_cisp(unit)
int unit;
{
    int header_total, data_total;
    int header_raw_total, data_raw_total;
    char output_filename[80];
    int fdout, imstat;
    int index;
    int cispdata_col;
    char *cispdata_charadr;
    short *cispdata_adr;
    int cisprese_col;
    char *cisprese_charadr;
    short *cisprese_adr;

    get_col(unit, "CISPDATA", &cispdata_col);
    tgetaddr(unit, cispdata_col, &cispdata_charadr);

    get_col(unit, "CISPRESE", &cisprese_col);
    tgetaddr(unit, cisprese_col, &cisprese_charadr);

    if (info_request)
    {
	header_raw_total = 960 + header_bytes - 320 + nparms * 80 + 6540;
	header_total = header_raw_total;
	if ((header_total % 2880) != 0)
	    header_total = ((header_total / 2880) +1) * 2880;
	data_raw_total = 1024 * 2;
	data_total = data_raw_total;
	if ((data_total % 2880) != 0)
	    data_total = ((data_total / 2880) +1) * 2880;
#ifdef DEBUG
	printf("primary and fixed part of extended header = %d bytes\n",
	    header_bytes);
	printf(
	    "each file has %d field header lines * 80 = %d\n",
	    nparms, nparms * 80);
	printf( "total header is %d\n", header_raw_total);
	printf("  header rounded up to %d\n", header_total);
	printf("data array is %d bytes\n", data_raw_total);
	printf("  data rounded up to %d\n", data_total);
#endif /* DEBUG */
	printf(
"program will produce %d image files, each approximately %d bytes long\n",
	    tgetrowcount(unit), header_total + data_total);
	printf(
"           and %d reset image files, each approximately %d bytes long\n",
	    tgetrowcount(unit), header_total + data_total);

	exit(0);
    }

    if (cube)
    {
	sprintf(output_filename, "%s_cube.fits", base_filename);
	if (!silent)
	    printf("opening file [%s]\n", output_filename);
	fdout = im_open(output_filename, "w");
	if (fdout < 0)
	{
	    perror(output_filename);
	    exit(1);
	}
	imstat = im_wkey_i(fdout, 16, "BITPIX");
	imstat = im_wkey_i(fdout, 3, "NAXIS");
	imstat = im_wkey_i(fdout, 32, "NAXIS1");
	imstat = im_wkey_i(fdout, 32, "NAXIS2");
	if (ending_index == INT_MAX)
	    ending_index = tgetrowcount(unit);

	if (ending_index > tgetrowcount(unit))
	{
	    sprintf(server_str, 
		"ending index greater than number of rows in file (%d)",
		tgetrowcount(unit));
	    error1(server_str);
	    return;
	}
	if (starting_index > ending_index)
	{
	    error1("starting index must be less than ending index");
	    return;
	}
	imstat = im_wkey_i(fdout, 
	    (ending_index - starting_index) / step_size + 1, "NAXIS3");
    }

    for (index = 1;; index++)
    {
	tblread(unit);
	if (itstat() > 0 )
	{
	    error1("error reading tablefile");
	    exit(1);
	}
	else if (itstat() < 0)
	{
	    break;
	}
	if ((index < starting_index) || (index > ending_index) || 
	    ((index - starting_index) % step_size))
	    continue;
#ifdef NOTDEF
	cispdata_adr = (short *) cispdata_charadr;
	cisprese_adr = (short *) cisprese_charadr;
	for (i = 0; i < 16; i++)
	{
	    printf("cispdata[%d] = %9.3e\n", i, (double) *cispdata_adr++);
	}
	for (i = 0; i < 16; i++)
	{
	    printf("cisprese[%d] = %9.3e\n", i, (double) *cisprese_adr++);
	}
#endif /* NOTDEF */

	cispdata_adr = (short *) cispdata_charadr;
	cisprese_adr = (short *) cisprese_charadr;

	if (!cube)
	{
	    sprintf(output_filename, "%s_%04d.fits", base_filename, index);
	    if (!silent)
		printf("opening file [%s]\n", output_filename);
	    fdout = im_open(output_filename, "w");
	    if (fdout < 0)
	    {
		perror(output_filename);
		exit(1);
	    }

	    imstat = im_wkey_i(fdout, 16, "BITPIX");
	    imstat = im_wkey_i(fdout, 2, "NAXIS");
	    imstat = im_wkey_i(fdout, 32, "NAXIS1");
	    imstat = im_wkey_i(fdout, 32, "NAXIS2");

	    if (long_header)
		do_long_header(unit, fdout);
	}

	imstat = im_wpix_s(fdout, cispdata_adr, 1024);
	if (imstat < 0)
	{
	    sprintf(server_str,
		"error writing pixels to file %s", output_filename);
	    error1(server_str);
	    return;
	}
	if (!cube)
	{
	    imstat = im_close(fdout);
	    file_done(output_filename);
	    if (control_c)
		return;
	}

	if ((want_resets) && (!cube))   /* dont do cube for resets */
	{
	    sprintf(output_filename, "%s_reset_%04d.fits", 
		base_filename, index);
	    if (!silent)
		printf("opening file [%s]\n", output_filename);
	    fdout = im_open(output_filename, "w");
	    if (fdout < 0)
	    {
		perror(output_filename);
		exit(1);
	    }

	    imstat = im_wkey_i(fdout, 16, "BITPIX");
	    imstat = im_wkey_i(fdout, 2, "NAXIS");
	    imstat = im_wkey_i(fdout, 32, "NAXIS1");
	    imstat = im_wkey_i(fdout, 35, "NAXIS2");

	    if (long_header)
		do_long_header(unit, fdout);

	    imstat = im_wpix_s(fdout, cisprese_adr, 1122);
	    if (imstat < 0)
	    {
		sprintf(server_str,
		    "error writing pixels to file %s", output_filename);
		error1(server_str);
		return;
	    }
	    imstat = im_close(fdout);
	    file_done(output_filename);
	    if (control_c)
		return;
	}
    }
    if (cube)
	imstat = im_close(fdout);
}

void
do_cier(unit)
int unit;
{
    int header_total, data_total;
    int header_raw_total, data_raw_total;
    char output_filename[80];
    int fdout, imstat;
    int index;
    int f2imag_col;
    char *f2imag_charadr;
    short *f2imag_adr;
    int f2imorig, f2dataty, f2datade, f2improc;
    int f2imorig_col, f2dataty_col, f2datade_col, f2improc_col;

    get_col(unit, "F2IMORIG", &f2imorig_col);
    get_col(unit, "F2DATATY", &f2dataty_col);
    get_col(unit, "F2DATADE", &f2datade_col);
    get_col(unit, "F2IMPROC", &f2improc_col);

    get_col(unit, "F2IMAG", &f2imag_col);
    tgetaddr(unit, f2imag_col, &f2imag_charadr);

    if (info_request)
    {
	header_raw_total = 960 + header_bytes - 320 + nparms * 80 + 12480;
	header_total = header_raw_total;
	if ((header_total % 2880) != 0)
	    header_total = ((header_total / 2880) +1) * 2880;
	data_raw_total = 1122 * 2;
	data_total = data_raw_total;
	if ((data_total % 2880) != 0)
	    data_total = ((data_total / 2880) +1) * 2880;
#ifdef DEBUG
	printf("primary and fixed part of extended header = %d bytes\n",
	    header_bytes);
	printf(
	    "each file has %d field header lines * 80 = %d\n",
	    nparms, nparms * 80);
	printf( "total header is %d\n", header_raw_total);
	printf("  header rounded up to %d\n", header_total);
	printf("data array is %d bytes\n", data_raw_total);
	printf("  data rounded up to %d\n", data_total);
#endif /* DEBUG */
	printf(
"program will produce %d image files, each approximately %d bytes long\n",
	    tgetrowcount(unit), header_total + data_total);

	exit(0);
    }

    if (cube)
    {
	sprintf(output_filename, "%s_cube.fits", base_filename);
	if (!silent)
	    printf("opening file [%s]\n", output_filename);
	fdout = im_open(output_filename, "w");
	if (fdout < 0)
	{
	    perror(output_filename);
	    exit(1);
	}
	for (;;)     /* look for the first SW or LW */
	{
	    tblread(unit);
	    if (itstat() != 0 )
	    {
		error1("error reading tablefile\n");
		exit(1);
	    }
	    tgeti(unit, f2imorig_col, &f2imorig);
	    tgeti(unit, f2dataty_col, &f2dataty);
	    tgeti(unit, f2datade_col, &f2datade);
	    tgeti(unit, f2improc_col, &f2improc);
	    /* check if "pattern" or "memory dump"  */
	    if ((f2datade != 6) && (f2datade != 5))
	    {
		/* this is SW or LW */
		/* use it to define size of each plane in cube */
		break;
	    }
	}

	tseek(unit, 1);  /* set to reread this record */

	imstat = im_wkey_i(fdout, 16, "BITPIX");
	imstat = im_wkey_i(fdout, 3, "NAXIS");
	if (f2imorig)
	{
	    /* LW */
	    imstat = im_wkey_i(fdout, 32, "NAXIS1");
	    imstat = im_wkey_i(fdout, 35, "NAXIS2");
	}
	else
	{
	    /* SW or "pattern" */
	    imstat = im_wkey_i(fdout, 34, "NAXIS1");
	    imstat = im_wkey_i(fdout, 33, "NAXIS2");
	}
	if (ending_index == INT_MAX)
	    ending_index = tgetrowcount(unit);

	if (ending_index > tgetrowcount(unit))
	{
	    sprintf(server_str, 
		"ending index greater than number of rows in file (%d)",
		tgetrowcount(unit));
	    error1(server_str);
	    return;
	}
	if (starting_index > ending_index)
	{
	    error1("starting index must be less than ending index");
	    return;
	}
	imstat = im_wkey_i(fdout, 
	    (ending_index - starting_index) / step_size + 1, "NAXIS3");
    }

#ifdef DEBUG
    printf(
"F2IMORIG  F2DATATY  F2DATADE  F2IMPROC\n");
#endif /* DEBUG */

    for (index = 1;; index++)
    {
	tblread(unit);
	if (itstat() > 0 )
	{
	    error1("error reading tablefile");
	    exit(1);
	}
	else if (itstat() < 0)
	{
	    break;
	}
	if ((index < starting_index) || (index > ending_index) || 
	    ((index - starting_index) % step_size))
	    continue;

	if (!cube)    /* dont change values if doing a cube */
	{
	    tgeti(unit, f2imorig_col, &f2imorig);
	    tgeti(unit, f2dataty_col, &f2dataty);
	    tgeti(unit, f2datade_col, &f2datade);
	    tgeti(unit, f2improc_col, &f2improc);
	}
#ifdef DEBUG
	printf("%d %d %d %d\n", 
	    f2imorig, f2dataty, f2datade, f2improc);
#endif /* DEBUG */

#ifdef NOTDEF
	f2imag_adr = (short *) f2imag_charadr;
	for (i = 0; i < 16; i++)
	{
	    printf("f2imag[%d] = %9.3e\n", i, (double) *f2imag_adr++);
	}
#endif /* NOTDEF */

	f2imag_adr = (short *) f2imag_charadr;

	if (!cube)
	{
	    sprintf(output_filename, "%s_%04d.fits", base_filename, index);
	    if (!silent)
		printf("opening file [%s]\n", output_filename);
	    fdout = im_open(output_filename, "w");
	    if (fdout < 0)
	    {
		perror(output_filename);
		exit(1);
	    }

	    imstat = im_wkey_i(fdout, 16, "BITPIX");
	    imstat = im_wkey_i(fdout, 2, "NAXIS");
	    if (f2imorig)
	    {
		/* LW */
		imstat = im_wkey_i(fdout, 32, "NAXIS1");
		imstat = im_wkey_i(fdout, 35, "NAXIS2");
	    }
	    else
	    {
		/* SW or "pattern" */
		imstat = im_wkey_i(fdout, 34, "NAXIS1");
		imstat = im_wkey_i(fdout, 33, "NAXIS2");
	    }

	    if (long_header)
		do_long_header(unit, fdout);

	if ((f2imorig == 1) || (f2datade == 6) || (f2datade == 5))
	{
	    /* LW or pattern or memory dump */
	    imstat = im_wpix_s(fdout, f2imag_adr, 1122);
	}
	else
	{
	    /* SW */
#ifdef UNINTERLEAVED
	/*
	printf("skipping uninterleaving - using raw pixel order\n");
	*/
	imstat = im_wpix_s(fdout, f2imag_adr, 1122);
#else
	    imstat = 0;
	    for (i = 1; i <= 31; i+=2)
		imstat |= im_wpix_s(fdout, f2imag_adr + (i-1) * 34 , 34);
	    for (i = 2; i <= 32; i+=2)
		imstat |= im_wpix_s(fdout, f2imag_adr + (i-1) * 34 , 34);
	    for (i = 33; i <= 33; i++)
		imstat |= im_wpix_s(fdout, f2imag_adr + (i-1) * 34 , 34);
#endif /* UNINTERLEAVED */

	}
	}
	else
	{
	    /* writing a cube - number of pixels is unchanging */
	    if (f2imorig)
	    {
		/* LW  32x35 */
		imstat = im_wpix_s(fdout, f2imag_adr, 1120);
	    }
	    else
	    {
		/* SW or "pattern"  34x33 */
		imstat = im_wpix_s(fdout, f2imag_adr, 1122);
	    }
	}

	if (imstat < 0)
	{
	    sprintf(server_str,
		"error writing pixels to file %s", output_filename);
	    error1(server_str);
	    return;
	}
	if (!cube)
	{
	    imstat = im_close(fdout);
	    file_done(output_filename);
	    if (control_c)
		return;
	}
    }
    if (cube)
	imstat = im_close(fdout);
}

void
do_cmap(unit)
int unit;
{
    int header_total, data_total;
    int header_raw_total, data_raw_total;
    char output_filename[80];
    int fdout, imstat;
    int bitpix_col, naxis_col, naxis1_col, naxis2_col, blank_col;
    int bscale_col, b_zero_col, polarise_col, array_col, index_col;
    int bunit_col, ctype1_col, crval1_col, crpix1_col, cdelt1_col;
    int crota2_col, ctype2_col, crval2_col, crpix2_col, cdelt2_col;
    int cd001001_col, cd002002_col, cd001002_col, cd002001_col;
    int channel_col, type_col;
    int bitpix, naxis, naxis1, naxis2, index, blank;
    double bscale, b_zero;
    char polarise[5], type[20];
    char bunit[13];
    char ctype1[9], ctype2[9], channel[5];
    double crval1, crpix1, cdelt1;
    double crval2, crpix2, cdelt2, crota2;
    double cd001001, cd002002, cd002001, cd001002;
    char *array_charadr;
    int *array_adr;

    get_col(unit, "CHANNEL", &channel_col);
    get_col(unit, "INDEX", &index_col);
    get_col(unit, "BITPIX", &bitpix_col);
    get_col(unit, "NAXIS", &naxis_col);
    get_col(unit, "NAXIS1", &naxis1_col);
    get_col(unit, "NAXIS2", &naxis2_col);
    get_col(unit, "BSCALE", &bscale_col);
    get_col(unit, "BZERO", &b_zero_col);
    get_col(unit, "BLANK", &blank_col);
    get_col(unit, "BUNIT", &bunit_col);
    get_col(unit, "CTYPE1", &ctype1_col);
    get_col(unit, "CRVAL1", &crval1_col);
    get_col(unit, "CRPIX1", &crpix1_col);
    get_col(unit, "CDELT1", &cdelt1_col);
    get_col(unit, "CTYPE2", &ctype2_col);
    get_col(unit, "CRVAL2", &crval2_col);
    get_col(unit, "CRPIX2", &crpix2_col);
    get_col(unit, "CDELT2", &cdelt2_col);
    get_col(unit, "CROTA2", &crota2_col);
    get_col(unit, "CD001001", &cd001001_col);
    get_col(unit, "CD002002", &cd002002_col);
    get_col(unit, "CD001002", &cd001002_col);
    get_col(unit, "CD002001", &cd002001_col);
    get_col(unit, "POLARISE", &polarise_col);
    get_col(unit, "TYPE", &type_col);

    get_col(unit, "ARRAY", &array_col);
    tgetaddr(unit, array_col, &array_charadr);

    if (info_request)
    {
	header_raw_total = 2000 + header_bytes - 320 + nparms * 80 - 1920;
	header_total = header_raw_total;
	if ((header_total % 2880) != 0)
	    header_total = ((header_total / 2880) +1) * 2880;
	/* now get number of pixels */
	/* assume that all rows are the same size */
	tblread(unit);
	if (itstat() == 0 )
	{
	    tgeti(unit, naxis1_col, &naxis1);
	    tgeti(unit, naxis2_col, &naxis2);
	}
	else
	{
	    printf("EOF reading tablefile:  0 images in file\n");
	    exit(1);
	}
	data_raw_total = naxis1*naxis2 * 4;
	data_total = data_raw_total;
	if ((data_total % 2880) != 0)
	    data_total = ((data_total / 2880) +1) * 2880;
#ifdef DEBUG
	printf("primary and fixed part of extended header = %d bytes\n",
	    header_bytes);
	printf(
	    "each file has %d field header lines * 80 = %d\n",
	    nparms, nparms * 80);
	printf( "total header is %d\n", header_raw_total);
	printf("  header rounded up to %d\n", header_total);
	printf("data array is %d bytes\n", data_raw_total);
	printf("  data rounded up to %d\n", data_total);
#endif /* DEBUG */
	printf(
"program will produce %d image files, each approximately %d bytes long\n",
	    tgetrowcount(unit), header_total + data_total);

	exit(0);
    }

    if (cube)
    {
	sprintf(output_filename, "%s_cube.fits", base_filename);
	if (!silent)
	    printf("opening file [%s]\n", output_filename);
	fdout = im_open(output_filename, "w");
	if (fdout < 0)
	{
	    perror(output_filename);
	    exit(1);
	}
	tblread(unit);
	if (itstat() != 0 )
	{
	    error1("error reading tablefile\n");
	    exit(1);
	}
	tgeti(unit, bitpix_col, &bitpix);
	tgeti(unit, naxis1_col, &naxis1);
	tgeti(unit, naxis2_col, &naxis2);
	tgeti(unit, blank_col, &blank);
	tgetd(unit, cdelt2_col, &cdelt2);
	tseek(unit, 1);  /* set to reread this record */
	if (bitpix == 0)
	    bitpix = 32;
	imstat = im_wkey_i(fdout, bitpix, "BITPIX");
	imstat = im_wkey_i(fdout, 3, "NAXIS");
	imstat = im_wkey_i(fdout, naxis1, "NAXIS1");
	imstat = im_wkey_i(fdout, naxis2, "NAXIS2");

	if (ending_index == INT_MAX)
	    ending_index = tgetrowcount(unit);

	if (ending_index > tgetrowcount(unit))
	{
	    sprintf(server_str, 
		"ending index greater than number of rows in file (%d)",
		tgetrowcount(unit));
	    error1(server_str);
	    return;
	}
	if (starting_index > ending_index)
	{
	    error1("starting index must be less than ending index");
	    return;
	}
	imstat = im_wkey_i(fdout, 
	    (ending_index - starting_index) / step_size + 1, "NAXIS3");
	imstat = im_wkey_i(fdout, blank, "BLANK");
	imstat = im_wkey_d(fdout, cdelt2, "CDELT2");
    }

#ifdef DEBUG
    printf(
"CHANNEL INDEX BITPIX NAXIS NAXIS1 NAXIS2 BSCALE BZERO POLARISE cd001001 ..\n");
#endif /* DEBUG */
    for (;;)
    {
	tblread(unit);
	if (itstat() > 0 )
	{
	    error1("error reading tablefile\n");
	    break;
	}
	else if (itstat() < 0)
	{
	    break;
	}
	tgetc(unit, channel_col, channel);
	tgeti(unit, index_col, &index);
	tgeti(unit, bitpix_col, &bitpix);
	tgeti(unit, naxis_col, &naxis);
	tgeti(unit, naxis1_col, &naxis1);
	tgeti(unit, naxis2_col, &naxis2);
	tgetd(unit, bscale_col, &bscale);
	tgetd(unit, b_zero_col, &b_zero);
	tgeti(unit, blank_col, &blank);
	tgetc(unit, bunit_col, bunit);
	tgetc(unit, ctype1_col, ctype1);
	tgetd(unit, crval1_col, &crval1);
	tgetd(unit, crpix1_col, &crpix1);
	tgetd(unit, cdelt1_col, &cdelt1);
	tgetc(unit, ctype2_col, ctype2);
	tgetd(unit, crval2_col, &crval2);
	tgetd(unit, crpix2_col, &crpix2);
	tgetd(unit, cdelt2_col, &cdelt2);
	tgetd(unit, crota2_col, &crota2);
	tgetd(unit, cd001001_col, &cd001001);
	tgetd(unit, cd002002_col, &cd002002);
	tgetd(unit, cd001002_col, &cd001002);
	tgetd(unit, cd002001_col, &cd002001);
	tgetc(unit, polarise_col, polarise);
	tgetc(unit, type_col, type);

	if ((index < starting_index) || (index > ending_index) || 
	    ((index - starting_index) % step_size))
	    continue;

#ifdef DEBUG
	printf("%s %d %d  %d %d %d  %e  %e %s %e %e %e %e\n", 
	    channel, index, bitpix, naxis, naxis1, naxis2, 
	    bscale, b_zero, polarise,
	    cd001001, cd002002, cd002001, cd001002);
#endif /* DEBUG */
#ifdef NOTDEF
	array_adr = (int *) array_charadr;
	for (i = 0; i < 16; i++)
	{
	    printf("array[%d] = %9.3e\n", i, (double) *array_adr++);
	}
#endif /* NOTDEF */

	array_adr = (int *) array_charadr;

	if (!cube)
	{
	    sprintf(output_filename, "%s_%04d.fits", base_filename, index);
	    if (!silent)
		printf("opening file [%s]\n", output_filename);
	    fdout = im_open(output_filename, "w");
	    if (fdout < 0)
	    {
		perror(output_filename);
		exit(1);
	    }

	    if (bitpix == 0)
		bitpix = 32;
	    imstat = im_wkey_i(fdout, bitpix, "BITPIX");
	    if (naxis == 0)
		naxis = 2;
	    imstat = im_wkey_i(fdout, naxis, "NAXIS");
	    imstat = im_wkey_i(fdout, naxis1, "NAXIS1");
	    imstat = im_wkey_i(fdout, naxis2, "NAXIS2");
	    imstat = im_wkey_d(fdout, bscale, "BSCALE");
	    imstat = im_wkey_d(fdout, b_zero, "BZERO");
	    imstat = im_wkey_i(fdout, blank, "BLANK");
	    imstat = im_wkey_c(fdout, bunit, "BUNIT");
	    imstat = im_wkey_c(fdout, ctype1, "CTYPE1");
	    imstat = im_wkey_d(fdout, crval1, "CRVAL1");
	    imstat = im_wkey_d(fdout, crpix1, "CRPIX1");
	    imstat = im_wkey_d(fdout, cdelt1, "CDELT1");
	    imstat = im_wkey_c(fdout, ctype2, "CTYPE2");
	    imstat = im_wkey_d(fdout, crval2, "CRVAL2");
	    imstat = im_wkey_d(fdout, crpix2, "CRPIX2");
	    imstat = im_wkey_d(fdout, cdelt2, "CDELT2");
	    imstat = im_wkey_d(fdout, crota2, "CROTA2");
	    imstat = im_wkey_d(fdout, cd001001, "CD001001");
	    imstat = im_wkey_d(fdout, cd002002, "CD002002");
	    imstat = im_wkey_d(fdout, cd001002, "CD001002");
	    imstat = im_wkey_d(fdout, cd002001, "CD002001");
	    imstat = im_wkey_c(fdout, channel, "CHANNEL");
	    imstat = im_wkey_i(fdout, index, "INDEX");
	    imstat = im_wkey_c(fdout, type, "TYPE");

	    if (long_header)
		do_long_header(unit, fdout);
	}

	imstat = im_wpix_i(fdout, array_adr, naxis1*naxis2);
	if (imstat < 0)
	{
	    sprintf(server_str,
		"error writing pixels to file %s", output_filename);
	    error1(server_str);
	    return;
	}

	if (!cube)
	{
	    imstat = im_close(fdout);
	    file_done(output_filename);
	    if (control_c)
		return;
	}

    }
    if (cube)
	imstat = im_close(fdout);
}


static void
get_col(unit, column_name, column_index)
int unit;
char *column_name;
int *column_index;
{
    tindp(unit, column_name, column_index);
    if (itstat())
    {
	sprintf(server_str,
	    "cannot find %s column", column_name);
	error1(server_str);
	*column_index = 0;
    }
}

static void do_long_header(unit, fdout)
int unit;
int fdout;
{
    double pscale, poffs;
    char pblank[1], punit[1];
    char *pt;
    int header_bytes, extension_bytes;
    char buf[81];
    int i, j;
    char pname[80];
    char ptype;
    int pcol, plen, pelements;
    int int_value;
    char str_value[80];
    char *array_charadr;
    double dbl_value;
    REAL4 val32, *ptr32; 
    REAL8 val64, *ptr64; 
    INT8 int8, *iptr8;
    INT16 int16, *iptr16;
    INT32 int32, *iptr32;

    /* do primary header first */
    tget_primary(unit, &pt, &header_bytes);
    for ( ; ; pt+= 80)
    {
	/* filter out certain lines */
	if (strncmp(pt, "SIMPLE", 6) == 0)
	    continue;
	if (strncmp(pt, "BITPIX", 6) == 0)
	    continue;
	if (strncmp(pt, "NAXIS", 5) == 0)
	    continue;
	if (strncmp(pt, "NAXIS1", 6) == 0)
	    continue;
	if (strncmp(pt, "NAXIS2", 6) == 0)
	    continue;
	if (strncmp(pt, "EXTEND", 6) == 0)
	    continue;
	if (strncmp(pt, "EXTNAME", 7) == 0)
	    continue;
	strncpy(buf, pt, 80);
	buf[80] = '\0';
	if (strncmp(pt, "END",3) == 0)
	    break;
	(void) im_wkey_raw(fdout, buf);
    }

    /* now the extension header */
    tget_extension(unit, &pt, &extension_bytes);
    for ( ; ; pt+= 80)
    {
	/* filter out certain lines */
	if (strncmp(pt, "BITPIX", 6) == 0)
	    continue;
	if (strncmp(pt, "NAXIS", 5) == 0)
	    continue;
	if (strncmp(pt, "NAXIS1", 6) == 0)
	    continue;
	if (strncmp(pt, "NAXIS2", 6) == 0)
	    continue;
	if (strncmp(pt, "EXTNAME", 7) == 0)
	    continue;
	if (strncmp(pt, "XTENSION", 8) == 0)
	    continue;
	if (strncmp(pt, "XTENSION", 8) == 0)
	    continue;
	if (strncmp(pt, "PCOUNT", 6) == 0)
	    continue;
	if (strncmp(pt, "GCOUNT", 6) == 0)
	    continue;
	if (strncmp(pt, "TFIELDS", 7) == 0)
	    continue;
	if (strncmp(pt, "TFORM", 5) == 0)
	    continue;
	if (strncmp(pt, "TTYPE", 5) == 0)
	    continue;
	if (strncmp(pt, "TUNIT", 5) == 0)
	    continue;
	if (strncmp(pt, "TDISP", 5) == 0)
	    continue;
	if (strncmp(pt, "TZERO", 5) == 0)
	    continue;
	strncpy(buf, pt, 80);
	buf[80] = '\0';
	if (strncmp(pt, "END",3) == 0)
	    break;
	(void) im_wkey_raw(fdout, buf);
    }

    /* now do the fields in the binary part of the file */
    for (i = 1; i <= nparms; i++)
    {
	tpinfo(unit, i, pname, &ptype, pblank, punit, &pscale, &poffs,
	    &pcol, &plen, &pelements);

	if ((strncmp(pname, "ARRAY", 5) == 0) ||
	    (strncmp(pname, "F2IMAG", 6) == 0) ||
	    (strncmp(pname, "CISPDATA", 8) == 0) ||
	    (strncmp(pname, "CISPRESE", 8) == 0) ||
	    (strncmp(pname, "BITPIX", 6) == 0) ||
	    (strncmp(pname, "NAXIS", 5) == 0) ||
	    (strncmp(pname, "NAXIS1", 6) == 0) ||
	    (strncmp(pname, "NAXIS2", 6) == 0) ||
	    (strncmp(pname, "BSCALE", 6) == 0) ||
	    (strncmp(pname, "BZERO", 5) == 0) ||
	    (strncmp(pname, "BLANK", 5) == 0) ||
	    (strncmp(pname, "BUNIT", 5) == 0) ||
	    (strncmp(pname, "CTYPE1", 5) == 0) ||
	    (strncmp(pname, "CRVAL1", 5) == 0) ||
	    (strncmp(pname, "CRPIX1", 5) == 0) ||
	    (strncmp(pname, "CDELT1", 5) == 0) ||
	    (strncmp(pname, "CTYPE2", 5) == 0) ||
	    (strncmp(pname, "CRVAL2", 5) == 0) ||
	    (strncmp(pname, "CRPIX2", 5) == 0) ||
	    (strncmp(pname, "CDELT2", 5) == 0) ||
	    (strncmp(pname, "CHANNEL", 7) == 0) ||
	    (strncmp(pname, "INDEX", 5) == 0) ||
	    (strncmp(pname, "TYPE", 4) == 0) ||
	    (strncmp(pname, "CD001001", 8) == 0) ||
	    (strncmp(pname, "CD002002", 8) == 0) ||
	    (strncmp(pname, "CD001002", 8) == 0) ||
	    (strncmp(pname, "CD002001", 8) == 0) ||
	    (strncmp(pname, "CROTA2", 5) == 0))
	    continue;
	switch(ptype)
	{
	    case 'B':
		if (pelements > 1)
		{
		    tgetaddr(unit, i, &array_charadr);
		    iptr8 = (INT8 *) array_charadr;
		    for (j = 1; j <= pelements; j++)
		    {
			int8 = *iptr8++;
			(void) im_wkey_i(fdout, int8, pname);
		    }
		}
		else
		{
		    tgeti(unit, i, &int_value);
		    (void) im_wkey_i(fdout, int_value, pname);
		}
		break;

	    case 'I':
		if (pelements > 1)
		{
		    tgetaddr(unit, i, &array_charadr);
		    iptr16 = (INT16 *) array_charadr;
		    for (j = 1; j <= pelements; j++)
		    {
			INT16fetch(iptr16++, &int16);
			(void) im_wkey_i(fdout, (int) int16, pname);
		    }
		}
		else
		{
		    tgeti(unit, i, &int_value);
		    (void) im_wkey_i(fdout, int_value, pname);
		}
		break;
	    case 'J':
		if (pelements > 1)
		{
		    tgetaddr(unit, i, &array_charadr);
		    iptr32 = (INT32 *) array_charadr;
		    for (j = 1; j <= pelements; j++)
		    {
			INT32fetch(iptr32++, &int32);
			(void) im_wkey_i(fdout, int32, pname);
		    }
		}
		else
		{
		    tgeti(unit, i, &int_value);
		    (void) im_wkey_i(fdout, int_value, pname);
		}
		break;
	    case 'E':
		if (pelements > 1)
		{
		    tgetaddr(unit, i, &array_charadr);
		    ptr32 = (REAL4 *) array_charadr;
		    for (j = 1; j <= pelements; j++)
		    {
			REAL4fetch(ptr32++, &val32);
			(void) im_wkey_d(fdout, (double) val32, pname);
		    }
		}
		else
		{
		    tgetd(unit, i, &dbl_value);
		    (void) im_wkey_d(fdout, dbl_value, pname);
		}
		break;
	    case 'D':
		if (pelements > 1)
		{
		    tgetaddr(unit, i, &array_charadr);
		    ptr64 = (REAL8 *) array_charadr;
		    for (j = 1; j <= pelements; j++)
		    {
			REAL8fetch(ptr64++, &val64);
			(void) im_wkey_d(fdout, val64, pname);
		    }
		}
		else
		{
		    tgetd(unit, i, &dbl_value);
		    (void) im_wkey_d(fdout, dbl_value, pname);
		}
		break;
	    case 'A':
		tgetc(unit, i, str_value);
		(void) im_wkey_c(fdout, str_value, pname);
		break;
	    default:
		sprintf(server_str,
		    "internal error - header type %c not recognized", ptype);
		error1(server_str);
		return;

	}
    }
}
