/*
Copyright (C) 1995, California Institute of Technology.
U.S. Government Sponsorship under NASA Contract NAS7-918 is acknowledged.
*/

#include "skyview.h"
#include "parse.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <sys/file.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/param.h>
#include <fcntl.h>

static char aliasname[MAXPATHLEN];     /* name of alias file */
extern char last_coord[2][40];
static char *ptri, *ptro;
static char **aliases = NULL;
static char **alias_defs = NULL;
extern FILE *cmdfile, *debug_file;
extern int server_mode;
extern char server_str[];

extern int debug;

/* function prototypes */
#ifdef _NO_PROTO

static int find_command();
static int sub_alias();

#else

static int find_command(char *com);
static int sub_alias(char *com);

#endif /* _NO_PROTO */

void
load_aliases(name)
char *name;
{
    int fd, i, newlines;
    off_t size;
    static char *aliasbuf = NULL;
    char *p;

    if(server_mode == TRUE)   /* no aliases in server mode */
	return;

    strcpy(aliasname, name);
    fd = open(aliasname, O_RDONLY);
    if (fd != -1)
    {
	/* get file size */
	size = lseek(fd, 0L, SEEK_END);
	(void) lseek(fd, 0L, L_SET);

	if (aliasbuf != NULL)
	    free(aliasbuf);

	aliasbuf = malloc(size);
	if (aliasbuf == NULL)
	{
	    error1("error allocating memory for alias file");

	    close(fd);
	    return;
	}
	read(fd, aliasbuf, size);
	p = aliasbuf;
	newlines = 0;
	/* count aliases and definitions (number of newlines) */
	while (p < aliasbuf + size)
	{
	    if (*p++ == '\n')
		newlines++;
	}
	if (aliases != NULL)
	    free(aliases);

	/* need one char pointer to each alias and definition */
	/*    (plus 1 for NULL and 1 for a possible new alias definition */
	aliases = (char **) malloc(((newlines/3)+2) * sizeof(char *));
	if (aliases == NULL)
	{
	    error1("error allocating memory for aliases");

	    free(aliasbuf);
	    aliasbuf = NULL;
	    close(fd);
	    return;
	}

	if (alias_defs != NULL)
	    free(alias_defs);

	alias_defs = (char **) malloc(((newlines/3)+2) * sizeof(char *));
	if (alias_defs == NULL)
	{
	    error1("error allocating memory for alias definitions");

	    free(aliasbuf);
	    aliasbuf = NULL;
	    free(aliases);
	    aliases = NULL;
	    close(fd);
	    return;
	}
	p = aliasbuf;
	i = 0;
	while (i < newlines / 3)
	{
	    p = (char *) strchr(p, '\n');
	    *p++ = '\0';
	    aliases[i] = p;

	    p = (char *) strchr(p, '\n');
	    *p++ = '\0';
	    alias_defs[i] = p;

	    p = (char *) strchr(p, '\n');
	    *p++ = '\0';

	    i++;
	}
	aliases[i] = NULL;     /* mark end of aliases */
	alias_defs[i] = NULL;
	close(fd);
    }
    else
    {
	/* no alias file - get space for the NULL and 1 new def */
	aliases = (char **) malloc(2 * sizeof(char *));
	alias_defs = (char **) malloc(2 * sizeof(char *));
	aliases[0] = NULL;     /* mark end of aliases */
	alias_defs[0] = NULL;
    }
}


/*
// ALIAS
// Inserts an alias definition into the alias file
*/

void
alias(argc, argv)
int argc;
char **argv;

{
    int j, alias_found;
    char ans[20];
    FILE *ft;

    if(server_mode == TRUE)
    {
	error_na("alias commmand not allowed in server mode");
	return;
    }

    if(argc > 3)
    {
	error1("Too many arguments to ALIAS command.");
	return;
    }


    if(argc == 1)                 /* Print out the aliases */
    {
	printf("\nCurrent Aliases\n"); 
	printf("===============\n"); 

	j = 0;
	while(aliases[j] != NULL) 
	{
	    printf("%-10s       %s\n", aliases[j], alias_defs[j]);
	    j++;
	}
	printf("\n"); 
	return;
    }


    else if(argc == 2)                 /* Remove an alias */
    {
	j = 0;
	while(aliases[j] != NULL) 
	{
	    if (strcmp(aliases[j], argv[1]) == 0)
	    {
		/* found the alias - pack it out */
		do	
		{
		    aliases[j] = aliases[j+1];
		    alias_defs[j] = alias_defs[j+1];
		    j++;
		} while(aliases[j] != NULL);     /* move the NULL also */
		break;
	    }
	    else
		j++;
	}
    }


    else if(argc == 3)                 /* Add a new alias */
    {
	if ((cmdfile == stdin) && (cmd(argv[1]) != NOTFOUND))
	{
	    /* he is not in a take file and he is redefining an s command */
	    printf(
	      "%s is a skyview command;  do you really want to redefine it? ",
	       argv[1]);

	    fgets(ans, 20, stdin);
	    if (cmd(ans) != YES)
	    {
		printf("alias command aborted\n");
		return;
	    }
	}
	alias_found = FALSE;
	j = 0;
	while(aliases[j] != NULL) 
	{
	    if (strcmp(aliases[j], argv[1]) == 0)
	    {
		/* this is a current alias - redefine it */
		alias_found = TRUE;
		alias_defs[j] = argv[2];    /* caution: characters are not */
					    /* yet in aliasbuf      */
		break;
	    }
	    j++;
	}
	if(alias_found == FALSE)
	{
	    aliases[j] = argv[1];
	    alias_defs[j] = argv[2];
	    aliases[j+1] = NULL;
	    alias_defs[j+1] = NULL;
	}
    }

    /* write the file out */
    ft = fopen(aliasname, "w+");
    if (ft == NULL)
    {
	sprintf(server_str, "cannot open the alias file: %s", aliasname);
	error1(server_str);
	return;
    }
    j = 0;
    while(aliases[j] != NULL) 
    {
	fprintf(ft, "\n%s\n%s\n", aliases[j], alias_defs[j]);
	j++;
    }
    fclose(ft);

    /* and read it back in to get all chars into aliasbuf */
    load_aliases(aliasname);
}



/*
// EXPAND_ALIAS
// Takes a string as input, looks up and substitutes any aliases
// from the alias file, and outputs the expanded string
*/

void
expand_alias(in, out)
char *in, *out;
{
    char com[512], *com_ptr;
    int a_f, alias_found;

    if (strchr(in, '%') != NULL)
	get_param(in);    /* expand takefile parameters %1 %2 etc. */
    do
    {
	if(debug)
	    fprintf(debug_file,
		"expand_alias:   input =%s=  length = %ld\n", in, strlen(in));

	ptri = in;
	ptro = out;
	alias_found = FALSE;

	while(find_command(com) == TRUE)      /* get 1 word into com */
	{
	    if (strcmp(com, "$") == 0)
	    {
		strcpy(com, last_coord[0]);
		strcat(com, " ");
		strcat(com, last_coord[1]);
	    }
	    a_f = sub_alias(com);      /* replace com with its expansion */
	    if (a_f == TRUE)
		alias_found = TRUE;
	    com_ptr = com;
	    while (*com_ptr != '\0')
		*ptro++ = *com_ptr++;  /* copy in this word or alias */
	    if ((ptro - out) > 450)
	    {
		error1(
		"overflow in alias expansion - alias is probably recursive");

		out[0] = '\0';
		return;
	    }
	}

	if (alias_found == TRUE)
	    strcpy(in, out);

	if(debug)
	    fprintf(debug_file,
		"expand_alias:  output =%s=  length = %ld\n", out, strlen(out));
    } while(alias_found == TRUE);

}


    /*
    // FIND_COMMAND
    // extracts the next word from the input line and puts it into com[]
    // Leading spaces, semicolons, and quoted strings are not put
    // into com[], but are copied from input[] to output[].
    // Returns FALSE if the input line is exhausted (places null at end
    // of output[].
    */

static int
find_command(com)
char *com;
{
	int i;

    restart:
	if(*ptri == '\0')
	{
	    *ptro = '\0';
	    return(FALSE);
	}

    while( *ptri == ' ' 
	|| *ptri == ';' 
	|| *ptri == ',')
	{
	    *ptro++ = *ptri++;  /* copy spaces, semis, commas */
	}

    if(*ptri == '"')
    {
	*ptro++ = *ptri++;        /* copy the quote */

	while(*ptri != '"' && *ptri != '\0')
	{
	    *ptro++ = *ptri++;      /* copy up to ending quote */
	}
	*ptro++ = '"';               /* tak on the ending quote */

	if(*ptri == '"')
	    ++ptri;			/* move past the ending quote */

	goto restart;
    }

    i=0;
    for (;;)
    {

	if(    *ptri == ' ' 
	    || *ptri == ';' 
	    || *ptri == ','
	    || *ptri == '\0')
	{
	    com[i] = '\0';
	    break;
	}
	else
	{
	    com[i] = *ptri++; /* copy chars into com until space, semi,
					comma, or null */
	    ++i;
	}
    }
    return(TRUE);
}



/*
// SUB_ALIAS
// Look through the alias file to see if the current word (com)
// has an alias. If so replace com with its alias
*/

static int
sub_alias(com)
char *com;
{
    int i, j, alias_found, len;

    alias_found = FALSE;            /* Search for an alias */
    j = 0;

    while(aliases[j] != NULL) 
    {
	if(strcmp(com, aliases[j]) == 0)
	{
	    alias_found = TRUE;

	    if(debug == TRUE)
		printf("sub_alias: %s becomes %s\n", com, alias_defs[j]);

	    len = strlen(alias_defs[j]);
	    for(i=0; i<len; ++i)
		if(alias_defs[j][i] != '"')
		{
		    com[i] = alias_defs[j][i];
		}
	    com[i] = '\0';
	    break;
	}
	j++;
    }
    return(alias_found);
}
