/*
 * Coordinate Conversion Calculator
 * Version  1.7
 * 
 * Copyright Infrared Processing and Analysis Center  1991
 * All Rights Reserved
 * California Institute of Technology
 *
 * <rick@ipac.caltech.edu>
 * <xiuqin@ipac.caltech.edu>
 */
/* #Copyright (C) 1992, California Institute of Technology. */
/* #U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* #is acknowledged. */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>
#if defined(USE_STDLIB)
#include <stdlib.h>
#endif

#include "coord.h"

void dd2sex(double ra, double dec, char *cra, char *cdec);
int check_lat(char *s1, char *s2);

int check(lon, lat)
double  lon, lat;

{
   int status;

   status = 0;

   if (lon > 360.0 || lon < 0.0)
      status = 1;
   else 
      status = 0;


   if (lat > 90.0 || lat < -90.0)
      {
      if (status == 0)
	 status = 2;
      else 
	 status = 3;
      }

   return(status);
}

int check_f(s)
char *s;
{
   int   len, i;

   len = strlen(s);
   while (s[len] ==' ')
      len--;

   i = 0;
   while (s[i] == ' ' || s[i] =='+')
      i++;

   while (isdigit(s[i]) != 0)
      i++;

   if (s[i] == '.')
      i++;

   while (isdigit(s[i]) != 0)
      i++;

   if (s[i] =='e' || s[i] =='E')
      i++;
   
   while (isdigit(s[i]) != 0)
      i++;

   if (i == len)
      return(TRUE);
   else
      return(FALSE);
}

/*
//////////////////////////////////////////////////////////////////////
c                                                                    /
c     SUBSTR finds the substrings in string                          /
c                                                                    /
/     A string is divided into substrings by ' '(space) or ':'(colon)/
/     If no characters(non_space) between two colons, the substring  /
/  is a null string.                                                 /
c/////////////////////////////////////////////////////////////////////
*/
void substr(string, subst, nsub)
char  *string, **subst;
int   *nsub;
{
   char *p;
   int num;
   int i;

   p = string;
   num = 0;

/*     Find the Beginning of the First Substring        */
   i = 0;
   while ( *p ==' ')
      p++;
   *(subst+num) = p;

   while (*p != '\0')
   {
      while (*p != ' ' &&  *p != ':' && *p != '\0') 
        p++;
      if (*p == ' ' || *p == ':')
         {
	 *p = '\0';
	 p++;
	 }
      while (*p == ' ' )/*|| *p == ':')*/
	 p++;
      num++;
      *(subst + num) = p;
   }
   *nsub=num;
}     
      
int check_lon(s1, s2)
char *s1, *s2;
{

   char   *subst[10];
   char   tmph[40], tmpd[40], tmpm[40], tmps[40], strtmp[40];
   char   chr, *p;
   int    status, ideg, syntax;
   int    i, leng;
   int    isign, nsub, nh, nm, ns;
   double  scale, deg, hr, min, sec;
   double  lon, lat;


   status = SEX;
   ideg = FALSE;
   deg = 0;
   hr  = 0;
   min = 0;
   sec = 0;

   strcpy(tmpd , "0");
   strcpy(tmph , "0");
   strcpy(tmpm , "0");
   strcpy(tmps , "0");


/*    Determine the Sign on the RA                             */

   isign = 1;
   leng = strlen(s1);

   i = 0;
   while (s1[i] == ' ')
      i++; 
   if (s1[i] == '-') {
      isign = -1;
      s1[i] = ' ';
      }
   else if (s1[i] == '+')
      s1[i] = ' ';

/*     Find the positions of the 'h' (or 'd'), 'm', and 's'     */

   nh = -1;
   nm = -1;
   ns = -1;
   leng = strlen(s1);
   for(i = 0; i<=leng-1; i++)
      {
      chr = s1[i];
      if (chr == 'h'  ||  chr == 'H') 
	 {
	 nh = i;
/*	 nm = leng + 1;
	 ns = leng + 1;
*/
         }
      else if (chr == 'd' ||  chr == 'D') 
	 {
	 ideg = TRUE;
	 nh = i;
/*	 nm = leng + 1;
	 ns = leng + 1;
*/
         }

      if (chr == 'm' || chr == 'M') 
	 nm = i;
      if (chr == 's' || chr == 'S') 
	 ns = i;
      }

/*     Extract the Integer Hours, Minutes, and Seconds    */

   if (nh > 0) 
      {
      p = s1;
      strcpy(tmph , p);
      tmph[nh] ='\0';
      }
   if (nm-1 > nh) 
      {
      p = s1+nh+1;
      strcpy(tmpm , p);
      tmpm[nm-nh-1] = '\0';
      }
   if (nm == -1 && ns-1 > nh)
      {
      p = s1+nh+1;
      strcpy(tmps , p);
      tmps[ns-nh-1] ='\0';
      }
   else if(ns-1 > nm) 
      {
      p = s1+nm+1;
      strcpy(tmps , p);
      tmps[ns-nm-1] ='\0';
      }



/*     If There are No 'hms', Find the Substrings             */

   nsub = 0;

   if (nh == -1 && nm == -1 && ns == -1) 
      {
      substr(s1, subst, &nsub);
       if (nsub >= 3) 
	  {
	      strcpy(tmph , subst[0]);
	      strcpy(tmpm , subst[1]);
	      strcpy(tmps , subst[2]);
	  }

       if (nsub == 2) 
	  {
	      strcpy(tmph , subst[0]);
	      strcpy(tmpm , subst[1]);
	  }

       if (nsub == 1) 
	  {
	      status = DEG;
	      strcpy(tmpd , subst[0]);
	  }
       }
    else if (nm == -1 && ns == -1)
       {
       substr( (s1+nh+1), subst, &nsub);
       if (nsub == 2) 
	  {
	      strcpy(tmpm , subst[0]);
	      strcpy(tmps , subst[1]);
	  }

       if (nsub == 1) 
	      strcpy(tmpm , subst[0]);
       }
    else if (ns == -1)
       {
       substr( (s1+nm+1), subst, &nsub);
       if (nsub == 1) 
	      strcpy(tmps , subst[0]);
       }

    if (status == DEG)
       syntax = check_f(tmpd);
    else
       {
       syntax = check_f(tmps);
       if (syntax ==TRUE)
	  syntax = check_f(tmpm);
       if (syntax ==TRUE)
	  syntax = check_f(tmph);
       }

      if (syntax == FALSE)
         return(JUNK);


    deg = atof(tmpd);
    hr =  atof(tmph);
    min = atof(tmpm);
    sec = atof(tmps);
     
    if (status == DEG)
       lon = isign*deg;
    else
       {
       scale = 15.;
       if (ideg  ==  TRUE) 
	 scale = 1.;
       lon = isign * scale * (hr + (min/60.) + (sec/3600.));
       }

    while (lon < 0.0)
       lon += 360.0;
    while (lon > 360.0)
       lon -= 360.0;
    if (status == DEG)
       {
       sprintf(s2, "%gd", lon);
       }
    else
       {
       lat = 0.0;
       dd2sex(lon, lat, s2, strtmp);
       }
    return(status);
}

int syscode(csys) 
char *csys;
{
   int  code;
   char wcsys[80];

   strcpy(wcsys, csys);
   code =FALSE;
   wcsys[2] = '\0';
   if (strcasecmp(wcsys, "ec")==0)
      code=ECL;
   else if (strcasecmp(wcsys, "eq")==0)
      code=EQU;
   else if (strcasecmp(wcsys, "ga")==0)
      code=GAL;
   else if (strcasecmp(wcsys, "su") ==0)
      code=SGAL;
   return(code);
}


void cinitial(p)
struct COORD *p;
{
   p->lon=0.0;
   p->lat=0.0;
   *(p->clon)='\0';
   *(p->clat)='\0';
   strcpy(p->csys, "Equatorial");
   p->epoch=1950.;
}

int check_coord(strin, strout, len, flag)
char *strin, *strout;
int len, flag;
{
   int    status;
   char   tmpin[40], tmpout[40];

   strcpy(tmpin, strin);
   if (flag == LON)
      status = check_lon(tmpin, tmpout);
   else
      status = check_lat(tmpin, tmpout);

   if (strlen(tmpout) > len)
      return(JUNK);
   else
      {
      strcpy(strout, tmpout);
      return(status);
      }
}




int check_lat(s1, s2)
char *s1, *s2;
{
   char   *subst[10];
   char   tmpd[40], tmpm[40], tmps[40], strtmp[40];
   char   chr, *p;
   int    syntax, status;
   int    i, leng;
   int    isign, nsub, nd, nm, ns;
   double  deg, min, sec;
   double  lon, lat;



   status = SEX;
   deg = 0;
   min = 0;
   sec = 0;
   strcpy(tmpd , "0");
   strcpy(tmpm , "0");
   strcpy(tmps , "0");


/*     Determine the Sign on the Dec                         */

   isign = 1;
   leng = strlen(s1);
   i = 0;
   while( s1[i] ==' ')
      i++;
   if (s1[i] == '-') {
      isign = -1;
      s1[i] = ' ';
      }
   else if (s1[i] == '+')
      s1[i] = ' ';

/*     Find the positions of the  'd', 'm', and 's'     */

   nd = -1;
   nm = -1;
   ns = -1;
   leng = strlen(s1);
   for(i = 0; i<=leng-1; i++)
      {
      chr = s1[i];
      if (chr == 'd' ||  chr == 'D') 
	 {
	 nd = i;
/*	 nm = leng + 1;
	 ns = leng + 1;
*/
         }

      if (chr == 'm' || chr == 'M') 
	 nm = i;
      if (chr == 's' || chr == 'S') 
	 ns = i;
      }

/*     Extract the Integer Hours, Minutes, and Seconds    */

   if (nd > 0) 
      {
      p=s1;
      strcpy(tmpd , s1);
      tmpd[nd] ='\0';
      }
   if (nm-1 > nd) 
      {
      p =s1+nd+1;
      strcpy(tmpm , p);
      tmpm[nm-nd-1] = '\0';
      }
   if (nm == -1 && ns-1 > nd)
      {
      p =s1+nd+1;
      strcpy(tmps , p);
      tmps[ns-nd-1] ='\0';
      }
   else if(ns-1 > nm) 
      {
      p =s1+nm+1;
      strcpy(tmps , p);
      tmps[ns-nm-1] ='\0';
      }


/*     If There are No 'hms', Find the Substrings             */

   nsub = 0;
   if (nd == -1 && nm == -1 && ns == -1) 
      {
      substr(s1, subst, &nsub);
       if (nsub >= 3) 
	  {
	      strcpy(tmpd , subst[0]);
	      strcpy(tmpm , subst[1]);
	      strcpy(tmps , subst[2]);
	  }

       if (nsub == 2) 
	  {
	      strcpy(tmpd , subst[0]);
	      strcpy(tmpm , subst[1]);
	  }
       if (nsub == 1) 
	   {
	   strcpy(tmpd , subst[0]);
	   status = DEG;
	   }

       }
    else if (nm == -1 && ns == -1)
       {
       substr( (s1+nd+1), subst, &nsub);
       if (nsub == 2) 
	  {
	      strcpy(tmpm , subst[0]);
	      strcpy(tmps , subst[1]);
	  }

       if (nsub == 1) 
	      strcpy(tmpm , subst[0]);
       }
    else if (ns == -1)
       {
       substr( (s1+nm+1), subst, &nsub);
       if (nsub == 1) 
	      strcpy(tmps , subst[0]);
       }


    if (status == DEG)
       syntax = check_f(tmpd);
    else
       {
       syntax = check_f(tmps);
       if (syntax ==TRUE)
	  syntax = check_f(tmpm);
       if (syntax ==TRUE)
	  syntax = check_f(tmpd);
       }

      if (syntax == FALSE)
         return(JUNK);

    if (status == DEG)
       lat = isign*atof(tmpd);
    else
       {
       deg = atof(tmpd);
       min = atof(tmpm);
       sec = atof(tmps);
       lat = isign * (deg + (min/60.) + (sec/3600.));
       }

    if (lat >90.0 || lat < -90.0)
       return(JUNK);
    else if (status == DEG)
       sprintf(s2, "%gd", lat);
    else
    {
       lon = 0.0;
       dd2sex(lon, lat, strtmp, s2);
    }

    return(status);
       
}



/*//////////////////////////////////////////////////////////////////////
/                                                                     /
/     determines the sexigesimal equivalent of the input ra and dec   /
/     input  : ra, dec     decimal degrees                            /
/     output : cra, cdec   sexigesimal expression                     /
/                                                                     /
/     requirement for input :                                         /
/             0 <= ra  <= 360                                         /
/           -90 <= dec <= +90                                         /
/                                                                     /
//////////////////////////////////////////////////////////////////////*/

void dd2sex(ra, dec, cra, cdec)
double  ra, dec;
char    *cra, *cdec;

{
      char   sign;
      double res;
      int    rah, ram, ras, rass, decd, decm, decs, decss;



/*     RA                      */

      res = ra + 0.05/3600.0;
      res=res/15.0;

      rah=(int)res;
      res=(res-(double) rah)*60.0;
      ram=(int)res;
      res=(res-(double)ram)*60.0;
      ras=(int) res;
      res = res - (double) ras;
      rass = (int) (res*10);

      if(ras  == 60) 
	 {
          ram=ram+1;
          ras=0;
	  }

      if(ram == 60)
	  {
          rah=rah+1;
          ram=0;
          }


/*     Dec    */

      sign='+';
      if (dec < 0.0) 
	 sign='-';
      res=fabs(dec);
      res = res + 0.05/3600.0;
      decd=(int) res;
      res=(res-(double) decd)*60.0;
      decm=(int) res;
      res=(res-(double) decm)*60.0;
      decs=(int) res;
      res = res - (double)decs;
      decss = (int) (res*10.0);

      if(decs == 60)
	  {
          decm=decm+1;
          decs=0;
	  }
      

      if(decm == 60)
	 {
          decd=decd+1;
          decm=0;
         }

      sprintf(cra, "%02dh%02dm%02d.%1ds", rah, ram, ras, rass);
      sprintf(cdec, "%c%02dd%02dm%02d.%1ds", sign, decd, decm, decs, decss);
}
void eclequ(lam, beta, rad, decd)                             
double lam, beta, *rad, *decd;
{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double x1, y1, z1, x2, y2, z2;                          
   double lamr, betar, rar, decr;                          
                                                                       
      lamr = lam*dtor;                                           
      betar = beta*dtor;                                           
                                                                       
      x1 = cos(betar)*cos(lamr);      
      y1 = cos(betar)*sin(lamr);                        
      z1 = sin(betar);                                                

      x2 = x1;                                                           
      y2 = cose*y1-sine*z1;                                           
      z2 = sine*y1+cose*z1;                                             

      if (fabs(z2) > 1.0)
	 z2 = copysign(1.0, z2);

      if (fabs(z2) == 1.0)
	 rar = 0;
      else
	 rar = atan2(y2, x2);                                              
      decr =  asin(z2);                                                 
                                                                     
      *rad = rar * rtod;                                                  
      *decd = decr * rtod;                                                

      if(*rad < 0.) 
	 *rad = 360. + *rad;                                  
                                                                       
}                                                              
void eclgal(lam, beta, ld, bd)                             
double lam, beta, *ld, *bd;
{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double x1, y1, z1;
   double x2, y2, z2;                          
   double x3, y3, z3;
   double lamr, betar, lr, br;                          

/*   from Ecliptic   to   Equtorial                         */ 

      lamr = lam*dtor;                                           
      betar = beta*dtor;                                           
                                                                       
      x1 = cos(betar)*cos(lamr);      
      y1 = cos(betar)*sin(lamr);                        
      z1 = sin(betar);                                                

      x2 = x1;                                                           
      y2 = cose*y1-sine*z1;                                           
      z2 = sine*y1+cose*z1;                                             

      if (fabs(z2) > 1.0)
	 z2 = copysign(1.0, z2);

      if (fabs(z2) == 1.0) 
         {
	 x2 = 0;
	 y2 = 0;
	 }

/*  from Equtorial  to   Galactic                         */

      x3 = -.06698874*x2 - .87275577*y2 - .48353892*z2;            
      y3 =  .49272847*x2 - .45034696*y2 + .74458464*z2;             
      z3 = -.86760082*x2 - .18837460*y2 + .46019979*z2;             

      if (fabs(z3) >1)
	 z3 = copysign(1.0, z3);

      br = asin(z3);
      if (fabs(z3) ==1.0)
	 lr = 0.;
      else
	 lr = atan2(y3, x3);

      *ld = lr *rtod;
      *bd = br * rtod;
      if (*ld < 0.0)
	 *ld += 360.0;
}
void eclsgal(lam, beta, sl, sb)
double lam, beta, *sl, *sb;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double  lamr, betar;
   double  slr, sbr; 
   double  x1, y1, z1;
   double  x2, y2, z2;

   lamr = lam*dtor;
   betar = beta*dtor;

   x1 = cos(betar)*cos(lamr);
   y1 = cos(betar)*sin(lamr);
   z1 = sin(betar);

   x2 = 0.38298536*x1 + 0.6514861*y1 + 0.65489547*z1;
   y2 = -0.897243031*x1 + 0.093723959*y1 + 0.4314751*z1; 
   z2 = 0.219720634*x1 -  0.752849043*y1 + 0.620436267*z1;

   if (fabs(z2) > 1.0)
      z2 = copysign(1.0, z2);

   if (fabs(z2) == 1.0)
      slr = 0.0;
   else
      slr = atan2(y2,x2);
   sbr = asin(z2);

   *sl = slr*rtod;
   *sb = sbr*rtod;
   if (*sl < 0.0)
      *sl += 360.0;
}

void equecl(rad, decd, lam, beta)                             
double  rad, decd, *lam, *beta;
{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double x1, y1, z1, x2, y2, z2;   
   double rar, decr, lamr, betar;
                                                                       
      rar =   rad*dtor;                                           
      decr  = decd*dtor;                                           
                                                                       
      x1 = cos(decr)*cos(rar);    
      y1 = cos(decr)*sin(rar);                        
      z1 = sin(decr);                                                

      x2 = x1;
      y2 = cose*y1 + sine*z1 ;                                           
      z2 = -sine*y1 + cose*z1 ;                                            

      if (fabs(z2) > 1.0)
	 z2 = copysign(1.0, z2);

      if (fabs(z2) == 1.0)
	 lamr = 0;
      else 
	 lamr = atan2(y2, x2);                                              
      betar =  asin(z2);         
                                                                     
      *lam = lamr * rtod;                                                
      *beta = betar * rtod;                                                  

      if(*lam < 0.) 
	 *lam = 360. + *lam;                                  
                                                                       
}                                                              
void equgal(rad, decd, ld, bd)
double  rad, decd, *ld, *bd;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double  rar, decr, lr, br; 
   double  x1, y1, z1;
   double  x2, y2, z2;


   rar = rad * dtor;
   decr = decd * dtor;

   x1 = cos(decr)*cos(rar);
   y1 = cos(decr)*sin(rar);
   z1 = sin(decr);

   x2 = -.06698874*x1 - .87275577*y1 - .48353892*z1;            
   y2 =  .49272847*x1 - .45034696*y1 + .74458464*z1;             
   z2 = -.86760082*x1 - .18837460*y1 + .46019979*z1;             

   if (fabs(z2) >1)
      z2 = copysign(1.0, z2);

   if (fabs(z2) ==1.0)
      lr = 0.;
   else
      lr = atan2(y2, x2);
   br = asin(z2);

   *ld = lr *rtod;
   *bd = br * rtod;
   if (*ld < 0.0)
      *ld += 360.0;
}
void equsgal(rad, decd, sl, sb)
double  rad, decd, *sl, *sb;
{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double x1, y1, z1;
   double x2, y2, z2;   
   double x3, y3, z3;   
   double rar, decr;
   double  slr, sbr; 
                                                                       
/*   From  Equtorial  To Ecliptic                          */

   rar =   rad*dtor;                                           
   decr  = decd*dtor;                                           
								    
   x1 = cos(decr)*cos(rar);    
   y1 = cos(decr)*sin(rar);                        
   z1 = sin(decr);                                                

   x2 = x1;
   y2 = cose*y1 + sine*z1 ;                                           
   z2 = -sine*y1 + cose*z1 ;                                            

   if (fabs(z2) > 1.0)
      z2 = copysign(1.0, z2);
   if (fabs(z2) == 1.0)
      {
      x2 = 0.0;
      y2 = 0.0;
      }   

/*    From  Ecliptic  To  Super Galactic                         */

   x3 = 0.38298536*x2 + 0.6514861*y2 + 0.65489547*z2;
   y3 = -0.897243031*x2 + 0.093723959*y2 + 0.4314751*z2; 
   z3 = 0.219720634*x2 -  0.752849043*y2 + 0.620436267*z2;

   if (fabs(z3) > 1.0)
      z3 = copysign(1.0, z3);

   if (fabs(z3) == 1.0)
      slr = 0.0;
   else
      slr = atan2(y3,x3);
   sbr = asin(z3);

   *sl = slr*rtod;
   *sb = sbr*rtod;
   if (*sl < 0.0)
      *sl += 360.0;
}

void galecl(ld, bd, lam, beta)
double  ld, bd, *lam, *beta;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double  lr, br, lamr, betar; 
   double  x1, y1, z1;
   double  x2, y2, z2;
   double  x3, y3, z3;

/*     From Galactc   to   Equtorial                  */

   lr = ld * dtor;
   br = bd * dtor;

   x1 = cos(br)*cos(lr);
   y1 = cos(br)*sin(lr);
   z1 = sin(br);

   x2  = -.06698874*x1 + .49272847*y1 - .86760082*z1;             
   y2  = -.87275577*x1 - .45034696*y1 - .18837460*z1;             
   z2  = -.48353892*x1 + .74458464*y1 + .46019979*z1; 

   if (fabs(z2) >1)
      z2 = copysign(1.0, z2);

   if (fabs(z2) ==1.0)
      {
      x2 = 0.;
      y2 = 0.;
      }

/*      From  Equtorial     to    Ecliptic                  */

      x3 = x2;
      y3 = cose*y2 + sine*z2 ;                                           
      z3 = -sine*y2 + cose*z2 ;                                            

      if (fabs(z3) > 1.0)
	 z2 = copysign(1.0, z3);

      if (fabs(z3) == 1.0)
	 lamr = 0;
      else 
	 lamr = atan2(y3, x3);                                              
      betar =  asin(z3);         
                                                                     
      *lam = lamr * rtod;                                                
      *beta = betar * rtod;                                                  

      if(*lam < 0.) 
	 *lam = 360. + *lam;                                  
                                                                       
}                                                              
void galequ(ld, bd, rad, decd)
double  ld, bd, *rad, *decd;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double  lr, br, rar, decr; 
   double  x1, y1, z1;
   double  x2, y2, z2;


   lr = ld * dtor;
   br = bd * dtor;

   x1 = cos(br)*cos(lr);
   y1 = cos(br)*sin(lr);
   z1 = sin(br);

   x2  = -.06698874*x1 + .49272847*y1 - .86760082*z1;             
   y2  = -.87275577*x1 - .45034696*y1 - .18837460*z1;             
   z2  = -.48353892*x1 + .74458464*y1 + .46019979*z1; 

   if (fabs(z2) >1)
      z2 = copysign(1.0, z2);

   decr = asin(z2);
   if (fabs(z2) ==1.0)
      rar = 0.;
   else
      rar = atan2(y2, x2);

   *rad = rar *rtod;
   *decd = decr * rtod;
   if (*rad < 0.0)
      *rad += 360.0;
}
void galsgal(ld, bd, sl, sb)
double  ld, bd, *sl, *sb;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double  lr, br, slr, sbr; 
   double  x1, y1, z1;
   double  x2, y2, z2;
   double  x3, y3, z3;
   double  x4, y4, z4;

/*     From Galactc   to   Equtorial                  */

   lr = ld * dtor;
   br = bd * dtor;

   x1 = cos(br)*cos(lr);
   y1 = cos(br)*sin(lr);
   z1 = sin(br);

   x2  = -.06698874*x1 + .49272847*y1 - .86760082*z1;             
   y2  = -.87275577*x1 - .45034696*y1 - .18837460*z1;             
   z2  = -.48353892*x1 + .74458464*y1 + .46019979*z1; 

   if (fabs(z2) >1)
      z2 = copysign(1.0, z2);

   if (fabs(z2) ==1.0)
      {
      x2 = 0.;
      y2 = 0.;
      }

/*      From  Equtorial     to    Ecliptic                  */

      x3 = x2;
      y3 = cose*y2 + sine*z2 ;                                           
      z3 = -sine*y2 + cose*z2 ;                                            

      if (fabs(z3) > 1.0)
	 z2 = copysign(1.0, z3);

      if (fabs(z3) == 1.0)
	 {
	 x3 = 0.0;
	 y3 = 0.0;
	 }

/*       From   Ecliptic   TO  Super  Galactic       */

   x4 = 0.38298536*x3 + 0.6514861*y3 + 0.65489547*z3;
   y4 = -0.897243031*x3 + 0.093723959*y3 + 0.4314751*z3; 
   z4 = 0.219720634*x3 -  0.752849043*y3 + 0.620436267*z3;

   if (fabs(z4) > 1.0)
      z4 = copysign(1.0, z4);

   if (fabs(z4) == 1.0)
      slr = 0.0;
   else
      slr = atan2(y4,x4);
   sbr = asin(z4);

   *sl = slr*rtod;
   *sb = sbr*rtod;
   if (*sl < 0.0)
      *sl += 360.0;
}

/*************************************************************************
*    This version was translated from the fortran source code on cyber at *
*      :user.iras649.ipacx.hgtprc_f                                       *
*                                                                         * 
*    A correction was made to avoid the error in the old version:         *
*      When epoch1 == epoch2 == 0, in the old version there will a error  * 
*      message "atan2 range error" from system. In tis version, since the *
*      epoch1 == epoch2, there will be no epoch conversion. The output    *
*      values ra2, dec2 are the same as the input values ra1, dec1.       *
*                                                                         * 
*   input : ra1, dec1, epoch1, epoch2                                     *
*   output: ra2, dec2                                                     *
*                                                                         * 
***************************************************************************/

void hgtprc(ra1, dec1, epoch1, epoch2, ra2, dec2)
double ra1, dec1, epoch1, epoch2, *ra2, *dec2;
{
   double pi2 = 6.283185307179586;
   double ep1 = 0.0, ep2 = 0.0;
   double cdr = 0.017453292519943;
   double x1[3], x2[3], r[3][3], t, st, a, b, c, csr;
   double sina, sinb, sinc, cosa, cosb, cosc;
   int i, j;

/*   COMPUTE INPUT DIRECTION COSINES       */

   a = cos(dec1);
   x1[0] = a * cos(ra1);
   x1[1] = a * sin(ra1);
   x1[2] = sin(dec1);



/*     SET UP ROTATION MATRIX (R)     */

   if (ep1 == epoch1 && ep2 == epoch2)
      {
      *ra2 = ra1;
      *dec2 = dec1;
      }
   else
      {
      csr = cdr/3600.0; 
      t = 0.001 * (epoch2 - epoch1);
      st = 0.001 * (epoch1 - 1900.0);
      a = csr * t * (23042.53 + st * (139.75 + 0.06 * st) + 
	  t * (30.23 - 0.27 * st + 18.0*t));
      b = csr * t * t * (79.27 + 0.66 * st + 0.32 * t) + a;
      c = csr * t * (20046.85 - st * (85.33 + 0.37 * st) + 
	  t * (-42.67 - 0.37 * st - 41.8 * t));
      sina = sin(a);
      sinb = sin(b);
      sinc = sin(c);
      cosa = cos(a);
      cosb = cos(b);
      cosc = cos(c);
      r[0][0] = cosa*cosb*cosc - sina*sinb;
      r[0][1] = -cosa*sinb - sina*cosb*cosc;
      r[0][2] = -cosb*sinc;
      r[1][0] = sina*cosb + cosa*sinb*cosc;
      r[1][1] = cosa*cosb - sina*sinb*cosc;
      r[1][2] = -sinb*sinc;
      r[2][0] = cosa*sinc;
      r[2][1] = -sina*sinc;
      r[2][2] = cosc;

/*     PERFORM THE ROTATION TO GET THE DIRECTION COSINES AT EPOCH2  */

      for (i=0; i<3; i++)
	 {
	 x2[i] = 0.0;
	 for (j=0; j<3; j++)
	    x2[i] = x2[i] + r[i][j] * x1[j];
	 }
    
      *ra2 = atan2(x2[1], x2[0]);
      if (*ra2 < 0.0) 
	 *ra2 = pi2 + *ra2;
      *dec2 = asin(x2[2]);
      }
}
/* ************************************************************************
* jprech.c  version that call hgtprc (herget precession)                  *
*     all arguments double precision                                      *
*     rain,decin,raout,decout   decimal degrees.                          *
*     epoch1,epoch2             year A.D.  (e.g. 1900., 1950.)            *
*									  *
***************************************************************************/

void jprech(epoch1,rain,decin,epoch2,raout,decout)
double rain,decin,*raout,*decout;
double epoch1, epoch2;
{
   double ra1,dec1,ra2,dec2;
   double dtor=0.01745329251994329,rtod=57.29577951308233;        

   ra1 = rain * dtor;
   dec1 = decin * dtor;

   hgtprc(ra1,dec1,epoch1,epoch2,&ra2,&dec2);

   *raout = ra2 * rtod;
   *decout = dec2 * rtod;

   if(*raout < 0.0) *raout += 360.0;
   if(*raout > 360.0) *raout -= 360.0;
}
/**************************************************************************
*    Check the validity of Longitude and Latitude                         *
*    0   <= Longitude <=360                                               *
*    -90 <= Latitude  <= +90                                              *
*    if Longitude is invalid, adjust it                                   *
*    Return value:  0   both are valid values                             *
*                   1   longitude is invalid, and adjusted                *
*                   2   latitude is invalid                               *
*                   3   both are invalid values, longitude adjusted       *
***************************************************************************/  

int range(lon, lat)
double *lon, *lat;

{
    int   status, i;
    double lonin, latin;

    lonin = *lon;
    latin = *lat;

    status = check(lonin, latin);

    if (status ==1 || status == 3)
    {
       while (lonin < 0.0)
	  if (lonin < 360.0)
	     lonin += 360.0;
       if (lonin > 360.0)
	  {
	  i = (int) (lonin/360.0);
	  lonin = lonin - i*360.0;
	  }
    }
    *lon = lonin;
    *lat = latin;
    return(status);
}

/*****************************************************************************
*    Converts the sexgicimal longitude and latitude to the decimal longitude
*    and latitude.                                                           *  *    input :  sexgicimal longitude and latitude                              *
*             "xxhxxmxx.xs", "xx xx xx.x",  "xx:xx:xx.x",  "xxxxxx"          *
*             "xxdxxmxx.xs", "xx xx xx.x",  "xx:xx:xx.x",  "xxxxxx"          *
*    output:  decimal longtitude and latitude                                *
*             xxx.xxxx,  xxxx.xxxx                                           *
*    return value :
*             0  both are converted successfully                             *
*             1  syntax error in longitude                                   *
*             2  syntax error in latitude                                    *
*             3  syntax error in both                                        *
*****************************************************************************/
int sex2dd(cra, cdec, ra, dec)

double *ra, *dec;
char *cra, *cdec;

{   
   char   crain[40], cdecin[40], *subst[10];
   char   tmph[40], tmpd[40], tmpm[40], tmps[40];
   char   chr, *p;
   int    ideg, syntax, lonstatus, latstatus;
   double  scale, hr, deg, min, sec;
   int    i, leng;
   int    isign, nsub, nh, nd, nm, ns;


   strcpy(crain , cra);
   strcpy(cdecin , cdec);

/*     RA ----------------------------------                       */

   ideg = FALSE;
   hr  = 0;
   min = 0;
   sec = 0;

   strcpy(tmph , "0");
   strcpy(tmpm , "0");
   strcpy(tmps , "0");


/*    Determine the Sign on the RA                             */

   isign = 1;
   leng = strlen(crain);

   for(i = 0; i<=leng-1; i++)
   {
      if (crain[i] == '-') 
	 {
	 isign = -1;
	 crain[i] = ' ';
	 }
      else if (crain[i] == '+') 
	 crain[i] = ' ';
   }

/*     Find the positions of the 'h' (or 'd'), 'm', and 's'     */

   nd = 0;
   nh = -1;
   nm = -1;
   ns = -1;
   leng = strlen(crain);
   for(i = 0; i<=leng-1; i++)
      {
      chr = crain[i];
      if (chr == 'h'  ||  chr == 'H') 
	 {
	 nh = i;
/*	 nm = leng + 1;
	 ns = leng + 1;
*/
         }
      else if (chr == 'd' ||  chr == 'D') 
	 {
	 ideg = TRUE;
	 nh = i;
/*	 nm = leng + 1;
	 ns = leng + 1;
*/
         }

      if (chr == 'm' || chr == 'M') 
	 nm = i;
      if (chr == 's' || chr == 'S') 
	 ns = i;
      }

/*     Extract the Integer Hours, Minutes, and Seconds    */

   if (nh > 0) 
      {
      p = crain;
      strcpy(tmph , p);
      tmph[nh] ='\0';
      }
   if (nm-1 > nh) 
      {
      p = crain+nh+1;
      strcpy(tmpm , p);
      tmpm[nm-nh-1] = '\0';
      }
   if(ns-1 > nm) 
      {
      p = crain+nm+1;
      strcpy(tmps , p);
      tmps[ns-nm-1] ='\0';
      }



/*     If There are No 'hms', Find the Substrings             */
   nsub = 0;

   if (nh == -1 && nm == -1 && ns == -1) 
      {
      substr(crain, subst, &nsub);
       if (nsub >= 3) 
	  {
	      strcpy(tmph , subst[0]);
	      strcpy(tmpm , subst[1]);
	      strcpy(tmps , subst[2]);
	  }

       if (nsub == 2) 
	  {
	      strcpy(tmpm , subst[0]);
	      strcpy(tmps , subst[1]);
	  }

       if (nsub == 1) 
	      strcpy(tmps , subst[0]);
       }
    else if (nm == -1 && ns == -1)
       {
       substr( (crain+nh+1), subst, &nsub);
       if (nsub == 2) 
	  {
	      strcpy(tmpm , subst[0]);
	      strcpy(tmps , subst[1]);
	  }

       if (nsub == 1) 
	      strcpy(tmpm , subst[0]);
       }
    else if (ns == -1)
       {
       substr( (crain+nm+1), subst, &nsub);
       if (nsub == 1) 
	      strcpy(tmps , subst[0]);
       }
/*     If the Number is Too Big, Assume +hhmmss.s         */

    syntax = TRUE;
    leng = strlen(tmps);
    for (i = 0; i <= leng-1; i++) 
       if (isdigit(tmps[i]) == 0 && tmps[i] !=' ' && tmps[i] !='.')
	  syntax = FALSE;

    leng = strlen(tmpm);
    if (syntax == TRUE)
    for (i = 0; i <= leng-1; i++) 
       if (isdigit(tmpm[i]) == 0 && tmpm[i] !=' ' && tmpm[i] !='.')
	  syntax = FALSE;

    leng = strlen(tmph);
    if (syntax == TRUE)
    for (i = 0; i <= leng-1; i++) 
       if (isdigit(tmph[i]) == 0 && tmph[i] !=' ' && tmph[i] !='.')
	  syntax = FALSE;
    
    lonstatus = syntax;
    if (syntax == TRUE)
       {
	 sec = atof(tmps);
	 if (sec >  60.) 
	    {
	    i = (int) (sec/10000.);
	    hr = (double) i;
	    i = (int) ((sec - hr*10000.) / 100.);
	    min = (double) i;
	    sec = sec - hr*10000. - min*100.;
	    }
	 else
	    {
	    hr =  atof(tmph);
	    min = atof(tmpm);
	    sec = atof(tmps);
	    }
        } 

/*     Generate the RA in Decimal Degrees                    */

      scale = 15.;
      if (ideg  ==  TRUE) 
	 scale = 1.;
      *ra = isign * scale * (hr + (min/60.) + (sec/3600.));


/*     DEC -------------------------------------             */


      deg = 0;
      min = 0;
      sec = 0;
      strcpy(tmpd , "0");
      strcpy(tmpm , "0");
      strcpy(tmps , "0");


/*     Determine the Sign on the Dec                         */

   isign = 1;
   leng = strlen(cdecin);

   for(i = 0; i<=leng-1; i++)
   {
      if (cdecin[i] == '-') 
	 {
	 isign = -1;
	 cdecin[i] = ' ';
	 }
      else if (cdecin[i] == '+') 
	 cdecin[i] = ' ';
   }

/*     Find the positions of the 'h' (or 'd'), 'm', and 's'     */

   nd = -1;
   nm = -1;
   ns = -1;
   leng = strlen(cdecin);
   for(i = 0; i<=leng-1; i++)
      {
      chr = cdecin[i];
      if (chr == 'd' ||  chr == 'D') 
	 {
	 nd = i;
/*	 nm = leng + 1;
	 ns = leng + 1;
*/
         }

      if (chr == 'm' || chr == 'M') 
	 nm = i;
      if (chr == 's' || chr == 'S') 
	 ns = i;
      }

/*     Extract the Integer Hours, Minutes, and Seconds    */

   if (nd > 0) 
      {
      p=cdecin;
      strcpy(tmpd , cdecin);
      tmpd[nd] ='\0';
      }
   if (nm-1 > nd) 
      {
      p =cdecin+nd+1;
      strcpy(tmpm , p);
      tmpm[nm-nd-1] = '\0';
      }
   if(ns-1 > nm) 
      {
      p =cdecin+nm+1;
      strcpy(tmps , p);
      tmps[ns-nm-1] ='\0';
      }


/*     If There are No 'hms', Find the Substrings             */

   nsub = 0;
   if (nd == -1 && nm == -1 && ns == -1) 
      {
      substr(cdecin, subst, &nsub);
       if (nsub >= 3) 
	  {
	      strcpy(tmpd , subst[0]);
	      strcpy(tmpm , subst[1]);
	      strcpy(tmps , subst[2]);
	  }

       if (nsub == 2) 
	  {
	      strcpy(tmpm , subst[0]);
	      strcpy(tmps , subst[1]);
	  }
       if (nsub == 1) 
	   strcpy(tmps , subst[0]);

       }
    else if (nm == -1 && ns == -1)
       {
       substr( (cdecin+nd+1), subst, &nsub);
       if (nsub == 2) 
	  {
	      strcpy(tmpm , subst[0]);
	      strcpy(tmpm , subst[0]);
	      strcpy(tmps , subst[1]);
	  }

       if (nsub == 1) 
	      strcpy(tmpm , subst[0]);
       }
    else if (ns == -1)
       {
       substr( (cdecin+nm+1), subst, &nsub);
       if (nsub == 1) 
	      strcpy(tmps , subst[0]);
       }

/*     If the Number is Too Big, Assume +hhmmss.s         */


/*     If the Number is Too Big, Assume +hhmmss.s         */

    syntax = TRUE;
    leng = strlen(tmps);
    for (i = 0; i <= leng-1; i++) 
       if (isdigit(tmps[i]) == 0 && tmps[i] !=' ' && tmps[i] !='.')
	  syntax = FALSE;

    leng = strlen(tmpm);
    if (syntax == TRUE)
    for (i = 0; i <= leng-1; i++) 
       if (isdigit(tmpm[i]) == 0 && tmpm[i] !=' ' && tmpm[i] !='.')
	  syntax = FALSE;

    leng = strlen(tmpd); 
    if (syntax == TRUE)
    for (i = 0; i <= leng-1; i++) 
       if (isdigit(tmpd[i]) == 0 && tmpd[i] !=' ' && tmpd[i] !='.')
	  syntax = FALSE;
    
    latstatus = syntax;
    if (syntax == TRUE)
       {
       sec = atof(tmps);
	 if (sec >  60.) 
	    {
	    i = (int) (sec/10000.);
	    deg = (double) i;
	    i = (int) ((sec - deg*10000.) / 100.);
	    min= (double) i;
	    sec = sec - deg*10000. - min*100.;
	    }
	 else
	    {
	    deg = atof(tmpd);
	    min = atof(tmpm);
	    sec = atof(tmps);
	    }
         }


/*     Generate the DEC in Decimal Degrees              */

      *dec = isign * (deg + (min/60.) + (sec/3600.));

/*   returned value reflects the syntax of RA and DEC    */

      if (lonstatus == FALSE && latstatus == FALSE)
      	 return(3);
      else if (lonstatus == FALSE)
	 return(1);
      else if (latstatus == FALSE)
	 return(2);
      else return(0);
}

void sgalecl(sl, sb,lam, beta)
double sl, sb, *lam, *beta;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double  slr, sbr; 
   double  lamr, betar;
   double  x1, y1, z1;
   double  x2, y2, z2;

   slr = sl*dtor;
   sbr = sb*dtor;

   x1 = cos(sbr)*cos(slr);
   y1 = cos(sbr)*sin(slr);
   z1 = sin(sbr);

   x2 = 0.382985359336*x1 - 0.897243031677*y1 + 0.219720633951*z1;
   y2 = 0.651486099226*x1 + 0.093723959195*y1 - 0.752849042942*z1; 
   z2 = 0.654895469487*x1 + 0.431475100662*y1 + 0.620436267152*z1;

   if (fabs(z2) > 1.0)
      z2 = copysign(1.0, z2);

   if (fabs(z2) == 1.0)
      lamr = 0.0;
   else
      lamr = atan2(y2,x2);
   betar = asin(z2);

   *lam = lamr*rtod;
   *beta = betar*rtod;
   if (*lam < 0.0)
      *lam += 360.0;
}

void sgalequ(sl, sb, ra, dec)
double sl, sb, *ra, *dec;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double  slr, sbr; 
   double  rar, decr;
   double  x1, y1, z1;
   double  x2, y2, z2;
   double  x3, y3, z3;

/*   From Sup Galactic To  Ecliptic    */

   slr = sl*dtor;
   sbr = sb*dtor;

   x1 = cos(sbr)*cos(slr);
   y1 = cos(sbr)*sin(slr);
   z1 = sin(sbr);

   x2 = 0.382985359336*x1 - 0.897243031677*y1 + 0.219720633951*z1;
   y2 = 0.651486099226*x1 + 0.093723959195*y1 - 0.752849042942*z1; 
   z2 = 0.654895469487*x1 + 0.431475100662*y1 + 0.620436267152*z1;

   if (fabs(z2) > 1.0)
      z2 = copysign(1.0, z2);

   if (fabs(z2) == 1.0)
      {
      x2 = 0.0;
      y2 = 0.0;
      }
                                                                       
/*   From Ecliptic To Equtorial       */

      x3 = x2;                                                           
      y3 = cose*y2-sine*z2;                                           
      z3 = sine*y2+cose*z2;                                             

      if (fabs(z3) > 1.0)
	 z3 = copysign(1.0, z3);

      if (fabs(z3) == 1.0)
	 rar = 0;
      else
	 rar = atan2(y3, x3);                                              
      decr =  asin(z3);                                                 
                                                                     
      *ra = rar * rtod;                                                  
      *dec = decr * rtod;                                                

      if(*ra < 0.) 
	 *ra = 360. + *ra;                                  
                                                                       
}                                                              
void sgalgal(sl, sb, ld, bd)
double sl, sb, *ld, *bd;

{
   double dtor=0.01745329251994329,rtod=57.29577951308233;        
   double cose=0.9174369452102114,sine=0.3978812028273710;        

   double  slr, sbr; 
   double  lr, br;
   double  x1, y1, z1;
   double  x2, y2, z2;
   double  x3, y3, z3;
   double  x4, y4, z4;

/*    From Super Galactic To  Ecliptic          */

   slr = sl*dtor;
   sbr = sb*dtor;

   x1 = cos(sbr)*cos(slr);
   y1 = cos(sbr)*sin(slr);
   z1 = sin(sbr);

   x2 = 0.382985359336*x1 - 0.897243031677*y1 + 0.219720633951*z1;
   y2 = 0.651486099226*x1 + 0.093723959195*y1 - 0.752849042942*z1; 
   z2 = 0.654895469487*x1 + 0.431475100662*y1 + 0.620436267152*z1;

   if (fabs(z2) > 1.0)
      z2 = copysign(1.0, z2);

   if (fabs(z2) == 1.0)
      {
      x2 = 0.0;
      y2 = 0.0;
      }


/*   from Ecliptic   to   Equtorial                         */ 

      x3 = x2;                                                           
      y3 = cose*y2-sine*z2;                                           
      z3 = sine*y2+cose*z2;                                             

      if (fabs(z3) > 1.0)
	 z3 = copysign(1.0, z3);

      if (fabs(z3) == 1.0) 
         {
	 x3 = 0;
	 y3 = 0;
	 }

/*  from Equtorial  to   Galactic                         */

      x4 = -.06698874*x3 - .87275577*y3 - .48353892*z3;            
      y4 =  .49272847*x3 - .45034696*y3 + .74458464*z3;             
      z4 = -.86760082*x3 - .18837460*y3 + .46019979*z3;             

      if (fabs(z4) >1)
	 z4 = copysign(1.0, z4);

      br = asin(z4);
      if (fabs(z4) ==1.0)
	 lr = 0.;
      else
	 lr = atan2(y4, x4);

      *ld = lr *rtod;
      *bd = br * rtod;
      if (*ld < 0.0)
	 *ld += 360.0;
}

/*****************************************************************************
*                                                                            *
*	    struct COORD                                                     *
*		     {                                                       *
*		     double lon;         (longitude)                         *
*		     double lat;         (latitude)                          *
*		     char   clon[80];    (sexgicimal longitude)              *
*		     char   clat[80];    (sexgicimal latitude)               *
*		     char   csys[80];    (coordinate system)                 *
*		     double epoch;       (epoch year)                        *
*	        	};                                                   *
*    input   *inp                                                            *
*    output  *outp                                                           *
*    return value                                                            *
*       0   conversion finished successfully                                 *
*           if decimal longitude is out of range,                            *
*	    it is converted back to range                                    *  
*       1   syntax error in sexgicimal lontitude                             *
*       2   syntax error in sexgicimal latitude or                           *
*           decimal latitude out or range [-90, +90]                         *
*       3   both 1 and 2                                                     *
*****************************************************************************/
int cconv(inp, outp)
struct COORD *inp, *outp;

{
    double lonin, latin, lonout, latout;
    char   clonin[80], clatin[80], clonout[80], clatout[80];
    double epochin, epochout;
    char   *csysin, *csysout;
   
    int    sysconv, epochconv, issex, status; 
    int    syscodein, syscodeout;


    lonin=inp->lon;
    latin=inp->lat;
    strcpy(clonin,inp->clon);
    strcpy(clatin,inp->clat);
    epochin=inp->epoch;
    epochout=outp->epoch;
    csysin=inp->csys;
    csysout=outp->csys;


    syscodein=syscode(csysin);
    syscodeout=syscode(csysout);

    issex = FALSE;
    epochconv = FALSE;
       
    status = 0;
    if (strcmp(clonin,  "")!=0)
       {
       issex = TRUE;
       status = sex2dd(clonin, clatin, &lonin, &latin);
       inp->lon=lonin;
       }

    if (strcmp(clatin, "")!=0 )
       {
       issex = TRUE;
       status = sex2dd(clonin, clatin, &lonin, &latin);
       inp->lat=latin;
       }

    if (issex == TRUE)
       {
       lonin = inp->lon;
       latin = inp->lat;
       }

    if (status != 0)
       return(status);
    else 
       {
       status = range(&lonin, &latin);
       if (status !=0 && status != 1)
	  return(status);
       else 
	  {
	  status = 0;
	  dd2sex(lonin, latin, clonin, clatin);
	  }
       }

    if (syscodein != syscodeout)
       sysconv=TRUE;
    else 
       sysconv=FALSE;

    if  (sysconv == TRUE)
       {
       switch(syscodein){
       case ECL:
	     if (syscodeout==EQU)
		{
		eclequ(lonin, latin, &lonout, &latout);
		if (epochout != 1950.0)
		   {
		   lonin = lonout;
		   latin = latout;
		   epochconv = TRUE;
		jprech(1950.0, lonin, latin, epochout, &lonout, &latout);
		   }
		}
             else if (syscodeout == GAL) 
		eclgal(lonin, latin, &lonout, &latout);
	     else  /*  SGAL */
		eclsgal(lonin, latin, &lonout, &latout);
             break;
       case EQU:
	     if (epochin != 1950.0)
		{
		jprech(epochin, lonin, latin, 1950.0, &lonout, &latout);
		epochconv = TRUE;
		lonin = lonout;
		latin = latout;
		}
	     if (syscodeout==ECL)
		equecl(lonin, latin, &lonout, &latout);
             else if(syscodeout == GAL)  
		equgal(lonin, latin, &lonout, &latout);
	     else  /*  SGAL  */
		equsgal(lonin, latin, &lonout, &latout);
             break;
       case GAL:
	     if (syscodeout==ECL)
		galecl(lonin, latin, &lonout, &latout);
             else if (syscodeout == EQU)  
		{
		galequ(lonin, latin, &lonout, &latout);
		if (epochout != 1950.0)
		   {
		   lonin = lonout;
		   latin = latout;
		   epochconv = TRUE;
		jprech(1950.0, lonin, latin, epochout, &lonout, &latout);
		   }
	        }
	     else  /*   SGAL   */
		galsgal(lonin, latin, &lonout, &latout);
	     break;
       case SGAL:
	     if (syscodeout==ECL)
		sgalecl(lonin, latin, &lonout, &latout);
             else if (syscodeout == EQU)  
		{
		sgalequ(lonin, latin, &lonout, &latout);
		if (epochout != 1950.0)
		   {
		   lonin = lonout;
		   latin = latout;
		   epochconv = TRUE;
		jprech(1950.0, lonin, latin, epochout, &lonout, &latout);
		   }
	        }
	     else  /*   GAL   */
		sgalgal(lonin, latin, &lonout, &latout);
	     break;
         }
       }
   else if (epochin != epochout && syscodein == EQU)
      {
      jprech(epochin, lonin, latin, epochout, &lonout, &latout);
      epochconv = TRUE;
      }

    if (sysconv == TRUE || epochconv == TRUE)
       {
       (*outp).lon=lonout;
       (*outp).lat=latout;
       dd2sex(lonout, latout, clonout, clatout);
       strcpy(outp->clon, clonout);
       strcpy(outp->clat, clatout);
       }
     else 
       {
        outp->lon = lonin;
	outp->lat = latin;
        strcpy(outp->clon, clonin);
        strcpy(outp->clat, clatin);
        }
     return(status);
}


