#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "contour.h"

#define  FALSE    0
#define  TRUE    !FALSE
#define  FOREVER  TRUE

void init_contours();
int contour(double **z, int nx, int ny, int x, int y, int direction, 
    double clevel, double blank);
int contseg(double **z, int nx, int ny, int x, int y, int direction, int cw, 
    double clevel, double blank);
void contour_start(double x, double y);
void contour_point(double x, double y);


int CPOLY_MAXC = 0;
int CPOLY_MAXCINC = 128;

int CPOLY_MAXPINC = 1024;

int CPOLY_CURRENT;

int CPOLY_SAVE = 0;
int CPOLY_SAVEINC = 10000;

CPOLY *conts;

int xoff[] = {-1, -1,  0,  1,  1,  1,  0, -1};
int yoff[] = { 0,  1,  1,  1,  0, -1, -1, -1};

int nsave, *xsave, *ysave;


/**************************************************/
/*                                                */
/* CONTOURS -- Finds the contours at level clevel */
/*                                                */
/**************************************************/

CPOLY *contours(z, nx, ny, clevel, blank)

double **z;
int nx, ny;
double clevel, blank;
{
   int i, j, n;
   int  x,  y,  direction;
   int use;

   direction  =  0;

   nsave = 0;
   if(xsave == (int *)(NULL))
   {
      CPOLY_SAVE = CPOLY_SAVEINC;
      xsave = (int *) malloc((unsigned)(CPOLY_SAVE * sizeof(int)));
      if (xsave == NULL)
      {
	 fprintf(stderr, "Can't allocate required memory\n");
	 exit(1);
      }
      ysave = (int *) malloc((unsigned)(CPOLY_SAVE * sizeof(int)));
      if (ysave == NULL)
      {
	 fprintf(stderr, "Can't allocate required memory\n");
	 exit(1);
      }
   }

   init_contours();

   /* We have to handle the special cases (contours that intersect the   */
   /* edge) first, to get them out of the way so they don't confuse      */
   /* things.  If we hit them while processing some interior curve, we   */
   /* might follow it to the edge and then stop prematurely              */

   /* Because of the nature of these contours, a lot of the checking     */
   /* that we must do for the interior points is not necessary           */


   /* First the bottom */

   j = 0;
   for(i=0; i<nx-1; ++i)
   {
      x = i + 1;
      y = j;
      direction = 0;

      if(z[j][i] < clevel && z[j][i+1] >= clevel)
	 contour(z, nx, ny, x, y, direction, clevel, blank);
   }


   /* Then the top */

   j = ny - 1;
   for(i=0; i<nx-1; ++i)
   {
      x = i;
      y = j;
      direction = 4;

      if(z[j][i+1] < clevel && z[j][i] >= clevel)
	 contour(z, nx, ny, x, y, direction, clevel, blank);
   }


   /* Then the right side */

   i = nx - 1;
   for(j=0; j<ny-1; ++j)
   {
      x = i;
      y = j+1;
      direction = 6;

      if(z[j][i] < clevel && z[j+1][i] >= clevel)
	 contour(z, nx, ny, x, y, direction, clevel, blank);
   }


   /* And finally the left side */

   i = 0;
   for(j=0; j<ny-1; ++j)
   {
      x = i;
      y = j;
      direction = 2;

      if(z[j+1][i] < clevel && z[j][i] >= clevel)
	 contour(z, nx, ny, x, y, direction, clevel, blank);
   }


   /*  Now we can do the interior */

   for(j=1; j<ny-1; ++j)
   {
      for(i=1; i<nx-1; ++i)
      {
	 /* See if we go across the contour moving from this point to    */
	 /* the next one horizontally (i.e., in x)                       */

	 x = i + 1;
	 y = j;
	 direction  =  0;


	 /* Check the pair for a contour crossing */

	 if(z[j][i] < clevel && z[j][i+1] >= clevel)
	 {

	    /* Check to see if point has already been used on a contour */

	    use = TRUE;
	    for(n=0; n<nsave; ++n)
	    {
	       
	       if(x==xsave[n] && y==ysave[n])
		  use = FALSE;
	    }

	    if(use == FALSE)
	       continue;

	    if(nsave == CPOLY_SAVE)
	    {
	       CPOLY_SAVE += CPOLY_SAVEINC;
	       xsave = (int *) realloc((char *) xsave,
			(unsigned)(CPOLY_SAVE * sizeof(int)));
	       ysave = (int *) realloc((char *) ysave,
			(unsigned)(CPOLY_SAVE * sizeof(int)));
	    }

	    xsave[nsave] = x;
	    ysave[nsave] = y;
	    ++nsave;

	    contour(z, nx, ny, x, y, direction, clevel, blank);
	 }
      }
   }

   return(conts);
}



/****************************************************************/
/*                                                              */
/* CONTOUR -- Collects the data for a whole contour curve       */
/*                                                              */
/****************************************************************/

int contour(z, nx, ny, x, y, direction, clevel, blank)

int  x,  y;
int nx, ny;
int direction;

double **z;
double clevel, blank;
{
   int i, cw, stat, nconts, npts1, npts;
   double xtemp, ytemp;

   cw = TRUE;
   stat = contseg(z, nx, ny, x, y, direction, cw, clevel, blank);

   nconts = conts->npoly;
   npts1 = conts->poly[nconts-1]->npts;

   if(stat < 0)
   {
      cw = FALSE;
      contseg(z, nx, ny, x, y, direction, cw, clevel, blank);
   }
   else
      return(0);


   /* Need to sort the two subsets of the line segment found          */
   /* First reverse the order of the first subset, then the whole set */

   npts = conts->poly[nconts-1]->npts;

   for(i=0; i<npts1/2; ++i)
   {
      xtemp = conts->poly[nconts-1]->pnt[i]->x;
      ytemp = conts->poly[nconts-1]->pnt[i]->y;
      
      conts->poly[nconts-1]->pnt[i]->x 
	 = conts->poly[nconts-1]->pnt[npts1-i-1]->x;

      conts->poly[nconts-1]->pnt[i]->y 
	 = conts->poly[nconts-1]->pnt[npts1-i-1]->y;

      conts->poly[nconts-1]->pnt[npts1-i-1]->x = xtemp;
      conts->poly[nconts-1]->pnt[npts1-i-1]->y = ytemp;
   }

   for(i=0; i<npts/2; ++i)
   {
      xtemp = conts->poly[nconts-1]->pnt[i]->x;
      ytemp = conts->poly[nconts-1]->pnt[i]->y;
      
      conts->poly[nconts-1]->pnt[i]->x 
	 = conts->poly[nconts-1]->pnt[npts-i-1]->x;

      conts->poly[nconts-1]->pnt[i]->y 
	 = conts->poly[nconts-1]->pnt[npts-i-1]->y;

      conts->poly[nconts-1]->pnt[npts-i-1]->x = xtemp;
      conts->poly[nconts-1]->pnt[npts-i-1]->y = ytemp;
   }

   return(0);
}




/****************************************************************/
/*                                                              */
/* CONTSEG -- Collects the data for one contour curves segment  */
/*                                                              */
/****************************************************************/

int contseg(z, nx, ny, x, y, direction, cw, clevel, blank)

int  x,  y;
int nx, ny;
int direction, cw;

double **z;
double clevel, blank;
{
   int start_x, start_y, start_dir;

   int dx, dy;
   int x_neighbor, y_neighbor;

   double xmap, ymap;
   double frac;


   /* Remember starting point (we have to stop when we get back here) */

   start_x = x;
   start_y = y;
   start_dir = direction;


   /* Where's the neighboring point we want to check first? */

   dx = xoff[direction];
   dy = yoff[direction];


   /* Move to the starting point of this contour */

   if(dx == 0)
   {
      frac = (z[y][x] - clevel) / (z[y][x] - z[y+dy][x]);
      xmap = x;
      ymap = frac * (double)dy + (double)y;
   }
   else
   {
      frac = (z[y][x] - clevel) / (z[y][x] - z[y][x+dx]);
      xmap = frac * (double)dx + (double)x;
      ymap = y;
   }
   
   if(cw == TRUE)
      contour_start(xmap, ymap);


   /* Walk around the contour */

   while(FOREVER)
   {
      
      /* Scan the points (clockwise) looking for another */
      /* contour crossing                                */

      if(cw == TRUE)
      {
	 ++direction;
	 if(direction >= 8) 
	    direction -= 8;
      }
      else
      {
	 --direction;
	 if(direction < 0) 
	    direction += 8;
      }
      
      dx = xoff[direction];
      dy = yoff[direction];

      x_neighbor = x + dx;
      y_neighbor = y + dy;



      /* Check to see if we hit the edge */

      if( x_neighbor < 0 || x_neighbor > nx-1 ||
          y_neighbor < 0 || y_neighbor > ny-1   )
      {
	 return(0);
      }


      /* Stop if we encounter a "blank" area */

      if(z[y_neighbor][x_neighbor] == blank)
	 return(-1);


      /* Areas of NaN are blank too                    */
      /* (partly because z==blank fails if blank = NaN */

      if(isnan(z[y_neighbor][x_neighbor]))
	 return(-1);


      /* When we find a contour crossing ... */

      if((clevel - z[y_neighbor][x_neighbor]) > 0)
      {
	 if((int)(direction/2.)*2 == direction)
	 {
	    if(dx == 0)
	    {
	       xmap = x;
	       frac = (z[y][x]-clevel) / (z[y][x]-z[y+dy][x]);
	       ymap = frac * (double)dy + (double)y;
	    }

	    else
	    {
	       frac = (z[y][x]-clevel) / (z[y][x]-z[y][x+dx]);
	       xmap = frac * (double)dx + (double)x;
	       ymap = y;
	    }

	    contour_point(xmap, ymap);

	    if(nsave == CPOLY_SAVE)
	    {
	       CPOLY_SAVE += CPOLY_SAVEINC;
	       xsave = (int *) realloc((char *) xsave,
			(unsigned)(CPOLY_SAVE * sizeof(int)));
	       ysave = (int *) realloc((char *) ysave,
			(unsigned)(CPOLY_SAVE * sizeof(int)));
	    }

	    xsave[nsave] = x;
	    ysave[nsave] = y;
	    ++nsave;
	 }
      }
      
      else
      {
	 direction += 4;
	 x = x_neighbor;
	 y = y_neighbor;
      }

      if(x==start_x && y==start_y && direction==start_dir)
	 return(0);
   }
}



/**************************************************/
/*                                                */
/* INIT_CONTOURS -- Initialize the contours,      */
/*                 allocate space if first time   */
/*                                                */
/**************************************************/

void init_contours()

{
   int i, j;

   if(conts == (CPOLY *)(NULL))
   {
      CPOLY_MAXC = CPOLY_MAXCINC;

      conts 
	 = (CPOLY *) 
	   malloc((unsigned)(sizeof(CPOLY)));
      if (conts == NULL)
      {
	 fprintf(stderr, "Can't allocate required memory\n");
	 exit(1);
      }

      conts->npoly = 0;
      conts->poly  
	 = (CPOLYN **) 
	   malloc((unsigned)(CPOLY_MAXC * sizeof(CPOLYN *)));

      if (conts->poly == NULL)
      {
	 fprintf(stderr, "Can't allocate required memory\n");
	 exit(1);
      }

      for(i=0; i<CPOLY_MAXC; ++i)
      {
	 conts->poly[i] 
	    = (CPOLYN *) 
	      malloc((unsigned)(sizeof(CPOLYN)));
         if (conts->poly[i] == NULL)
	 {
	    fprintf(stderr, "Can't allocate required memory\n");
	    exit(1);
	 }

	 conts->poly[i]->npts = 0;
	 conts->poly[i]->maxpts = CPOLY_MAXPINC;

	 conts->poly[i]->pnt 
	    = (CPOLY_PNT **) 
	      malloc((unsigned)(CPOLY_MAXPINC * sizeof(CPOLY_PNT *)));
         if (conts->poly[i]->pnt == NULL)
	 {
	    fprintf(stderr, "Can't allocate required memory\n");
	    exit(1);
	 }

	 for(j=0; j<CPOLY_MAXPINC; ++j)
	 {
	    conts->poly[i]->pnt[j] 
	       = (CPOLY_PNT *) 
		 malloc((unsigned)(sizeof(CPOLY_PNT)));
	    if (conts->poly[i]->pnt[j] == NULL)
	    {
	       fprintf(stderr, "Can't allocate required memory\n");
	       exit(1);
	    }
	 }
      }
   }

   conts->npoly = 0;
}




/**************************************************/
/*                                                */
/* CONTOUR_START -- Begins a new contour          */
/*                                                */
/**************************************************/

void contour_start(x, y)

double x, y;
{
   int i, j;

   if(conts == (CPOLY *)(NULL))
      init_contours();

   CPOLY_CURRENT = conts->npoly;

   if(conts->npoly == CPOLY_MAXC)
   {
      CPOLY_MAXC += CPOLY_MAXCINC;

      conts->poly  
	 = (CPOLYN **) 
	   realloc((char *) conts->poly,
	      (unsigned)(CPOLY_MAXC * sizeof(CPOLYN *)));

      for(i=conts->npoly; i<CPOLY_MAXC; ++i)
      {
	 conts->poly[i] 
	    = (CPOLYN *) 
	      malloc((unsigned)(sizeof(CPOLYN)));
         if (conts->poly[i] == NULL)
	 {
	    fprintf(stderr, "Can't allocate required memory\n");
	    exit(1);
	 }

	 conts->poly[i]->npts = 0;
	 conts->poly[i]->maxpts = CPOLY_MAXPINC;

	 conts->poly[i]->pnt 
	    = (CPOLY_PNT **) 
	      malloc((unsigned)(CPOLY_MAXPINC * sizeof(CPOLY_PNT *)));
         if (conts->poly[i]->pnt == NULL)
	 {
	    fprintf(stderr, "Can't allocate required memory\n");
	    exit(1);
	 }

	 for(j=0; j<CPOLY_MAXPINC; ++j)
	 {
	    conts->poly[i]->pnt[j] 
	       = (CPOLY_PNT *) 
		 malloc((unsigned)(sizeof(CPOLY_PNT)));
	    if (conts->poly[i]->pnt[j] == NULL)
	    {
	       fprintf(stderr, "Can't allocate required memory\n");
	       exit(1);
	    }
	 }
      }
   }

   ++(conts->npoly);
   conts->poly[CPOLY_CURRENT]->npts = 1;
   conts->poly[CPOLY_CURRENT]->pnt[0]->x = x;
   conts->poly[CPOLY_CURRENT]->pnt[0]->y = y;
}




/**************************************************/
/*                                                */
/* CONTOUR_POINT -- Adds a new point to a contour */
/*                                                */
/**************************************************/

void contour_point(x, y)

double x, y;
{
   int j, n;

   n = conts->poly[CPOLY_CURRENT]->npts;

   if(n == conts->poly[CPOLY_CURRENT]->maxpts)
   {
      conts->poly[CPOLY_CURRENT]->maxpts += CPOLY_MAXPINC;

      conts->poly[CPOLY_CURRENT]->pnt 
	 = (CPOLY_PNT **) 
	   realloc((char *) conts->poly[CPOLY_CURRENT]->pnt,
	    (unsigned)(conts->poly[CPOLY_CURRENT]->maxpts 
			* sizeof(CPOLY_PNT *)));

      for(j=n; j<conts->poly[CPOLY_CURRENT]->maxpts ; ++j)
      {
	 conts->poly[CPOLY_CURRENT]->pnt[j] 
	    = (CPOLY_PNT *) 
	      malloc((unsigned)(sizeof(CPOLY_PNT)));
	 if (conts->poly[CPOLY_CURRENT]->pnt[j] == NULL)
	 {
	    fprintf(stderr, "Can't allocate required memory\n");
	    exit(1);
	 }
      }
   }
	   
   conts->poly[CPOLY_CURRENT]->pnt[n]->x = x;
   conts->poly[CPOLY_CURRENT]->pnt[n]->y = y;
   ++conts->poly[CPOLY_CURRENT]->npts;
}
