/* rwc.c */
/* Copyright (C) 1992, California Institute of Technology. */
/* U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* is acknowledged. */
/*
891103: change CDELT's of 0 to 1
        fix bug: initialize q_type for n==naxes
*/

#include <ctype.h>
#include <string.h>
#include "im.h"
#include "rwc.h"
#include "map_struct.h"

#define MAGIC_NUM 928351

enum val_type_enum {
   T_illegal,
   T_ambig,
   T_absolute_pixel,
   T_relative_pixel,
   T_1d_rwc,
   T_geometric,
   T_none };

static enum val_type_enum determine_value_type();


int
rwc_fid(r,fid)
struct rwc_struct *r;
int fid;
/* returns:
    0 = OK
   -9 = cannot read header info from file 
*/
{
   int stat;
   int n;

/* get NAXIS */
   stat = im_rkey_i(fid,&r->n_axes,"NAXIS");
   if (stat == -3) return(-9);

/* get values from header */
   for (n=1; n<=r->n_axes; n++) {
      r->ctype[n][0] = '\0';
      im_rkey_c(fid, r->ctype[n],im_key("CTYPE",n));

      im_rkey_i(fid,&r->naxis[n],im_key("NAXIS",n));

      r->cdelt[n] = 1.0;
      im_rkey_d(fid,&r->cdelt[n],im_key("CDELT",n));
      if(r->cdelt[n]==0.0) r->cdelt[n] = 1.0;

      r->crval[n] =  0.0;
      im_rkey_d(fid,&r->crval[n],im_key("CRVAL",n));

      r->crpix[n] =  1.0;
      im_rkey_d(fid,&r->crpix[n],im_key("CRPIX",n));

      r->crota[n] =  0.0;
      im_rkey_d(fid,&r->crota[n],im_key("CROTA",n));
   }

r->magic = MAGIC_NUM;

/**********
for (n=1; n<=r->n_axes; n++) {
printf("rwc_fid: ctype[%d] = {%s}\n", n, r->ctype[n]);
printf("rwc_fid: naxis[%d] = %d\n",   n, r->naxis[n]);
printf("rwc_fid: cdelt[%d] = %f\n",   n, r->cdelt[n]);
printf("rwc_fid: crpix[%d] = %f\n",   n, r->crpix[n]);
printf("rwc_fid: crval[%d] = %f\n",   n, r->crval[n]);
printf("rwc_fid: crota[%d] = %f\n",   n, r->crota[n]);
}
***************/

return(0);

}


int
rwcf(r,string,skips,takes)
struct rwc_struct *r;
char string[];
int skips[];
int takes[];
/* returns:
   0 = OK
  -1 = too many axes specified ( > NAXIS)
  -2 = illegal header info
  -3 = illegal geometric coordinate syntax
  -9 = bad rwc_block
  -11 = illegal real world or pixel coordinate syntax in 1st field
  -12 = illegal real world or pixel coordinate syntax in 2nd field
  -13 to -19  etc.
  -21 to -29  (Nth field) ambigous coordinate (no @ or .)
*/
{
   int n;
   int stat;
   char *c;
   char *f;

   int n_fields;
   char field[10][40];

   char fld_ctype[16];
   enum val_type_enum fld_type1;
   char fld_val1[80];
   char fld_sep1;
   enum val_type_enum fld_type2;
   char fld_val2[80];
   char fld_sep2;
   enum val_type_enum fld_type3;
   char fld_val3[80];

   double put_val1;
   double put_val2;
   double put_val3;

   enum q_type_enum {Q_none,Q_lo_hi,Q_mid_size} q_type[10];
   double q_val1[10];
   double q_val2[10];

   struct map_struct map_block;
   int axis;
   int axis2; /* second axis used for 2-D transforms only */
   int ax;
   double tmp;
   

   if (r->magic != MAGIC_NUM) return(-9);

   /* initialize */
   for (n=1; n<=r->n_axes; n++) q_type[n] = Q_none;
   axis = 1;

   /* spilt string into fields (by commas) */
   n_fields = 0;
   for(c=string; *c; ) {
      n_fields++;
      f = field[n_fields];
      while (*c && (*c != ',')) *f++ = *c++;
      *f = '\0';
      if (*c) c++;
   }

/**************
for(n=1; n<=n_fields; n++) printf("rwcf: field[%d] {%s}\n",n,field[n]);
**************/


   /* process fields */
   for (n=1; n<=n_fields; n++) {

      /* split up into sub-fields */
      for (c=field[n], f=fld_ctype;
       isalpha(*c) || (*c=='-' && (*(c+1)=='-' || isalpha(*(c+1)) ));)
       *f++ = *c++;
      *f = '\0';
      if (fld_ctype[1]=='\0') switch(fld_ctype[0]) {
         case 'R': field[n][0] = fld_ctype[0] = 'r';
		    /*FALLTHROUGH*/
         case 'r': c = field[n]; break;
         case 'E': field[n][0] = fld_ctype[0] = 'e';
		    /*FALLTHROUGH*/
         case 'e': c = field[n]; break;
         case 'G': field[n][0] = fld_ctype[0] = 'g';
		    /*FALLTHROUGH*/
         case 'g': c = field[n]; break;
      }

      for (f=fld_val1; *c && *c!=':' && *c!='^';) *f++ = *c++;
      *f = '\0';
      fld_sep1 = *c;
      if (*c) {
         for (c++, f=fld_val2; *c && *c!='^';) *f++ = *c++;
         *f = '\0';
         fld_sep2 = *c;
      }
         if (*c) {
         for (c++, f=fld_val3; *c;) *f++ = *c++;
         *f = '\0';
      }


/***********************
printf("rwcf: [%d] {%s} {%s}",n,fld_ctype,fld_val1);
if (fld_sep1) printf(" {%c} {%s}",fld_sep1,fld_val2);
if (fld_sep2) printf(" {%c} {%s}",fld_sep2,fld_val3);
printf("\n");
***********************/


      /* pre-process (determine type and convert values from ascii) */
      fld_type1 = determine_value_type(fld_ctype,fld_val1,&put_val1);
      if (fld_type1 == T_illegal) return(-10-n);
      if (fld_type1 == T_ambig) return(-20-n);

      if (fld_sep1) fld_type2 = determine_value_type(0,fld_val2,&put_val2);
      else          fld_type2 = T_none;
      if (fld_type2 == T_illegal) return(-10-n);
      if (fld_type2 == T_ambig) return(-20-n);

      if (fld_sep2) fld_type3 = determine_value_type(0,fld_val3,&put_val3);
      else          fld_type3 = T_none;
      if (fld_type3 == T_illegal) return(-10-n);
      if (fld_type3 == T_ambig) return(-20-n);

      /* select correct axis */
      if (fld_type1 == T_geometric) {
         for (ax=1; ax<=r->n_axes; ax++) {
            if (strncmp(r->ctype[ax],"RA--",4) == 0
             ||  strcmp(r->ctype[ax],"RA")     == 0
             || strncmp(r->ctype[ax],"GLON",4) == 0
             || strncmp(r->ctype[ax],"ELON",4) == 0) break;
         }
         if (ax > r->n_axes) return(-1);
         axis = ax;
         for (ax=axis+1; ax<=r->n_axes; ax++) {
            if (strncmp(r->ctype[ax],"DEC-",4) == 0
             ||  strcmp(r->ctype[ax],"DEC")    == 0
             || strncmp(r->ctype[ax],"GLAT",4) == 0
             || strncmp(r->ctype[ax],"ELAT",4) == 0) break;
         }
         if (ax > r->n_axes) return(-1);
         axis2 = ax;
      }

      else if(fld_ctype[0]) {
         for (ax=1; ax<=r->n_axes; ax++) 
           if (strcmp(r->ctype[ax],fld_ctype) == 0) break;
         axis = ax;
      }

/*********   printf("axis = %d\n",axis); ***********/

      /* process if 2-D rwc */
      if (fld_type1 == T_geometric) {
         stat = map_set(&map_block,
                        r->ctype[axis],r->ctype[axis2],
                        r->cdelt[axis],r->cdelt[axis2],
                        r->crval[axis],r->crval[axis2],
                        r->crpix[axis],r->crpix[axis2],
                        r->crota[axis],r->crota[axis2]);
         if (stat) return(-2);

         stat = mapf_string(&map_block,fld_val1,&q_val1[axis],&q_val1[axis2]);
         if (stat) return(-3);

         q_type[axis] = q_type[axis2] = Q_mid_size;

         /* set q_val2[axis] */
         switch (fld_type2) {
            case T_none: q_val2[axis] = 1.0; break;
            case T_relative_pixel:
            case T_absolute_pixel: q_val2[axis] = put_val2; break;
            case T_1d_rwc: 
             tmp = r->cdelt[axis];
             if (tmp) q_val2[axis] = put_val2 / tmp;
             else     q_val2[axis] = 1.0;
             break;
	    default:
	     break;
         }

         /* set q_val2[axis2] */
         if (fld_type3 == T_none) {
            fld_type3 = fld_type2;
            put_val3=put_val2;
            }
         switch (fld_type3) {
            case T_none: q_val2[axis2] = 1.0; break;
            case T_relative_pixel:
            case T_absolute_pixel: q_val2[axis2] = put_val3; break;
            case T_1d_rwc: 
             tmp = r->cdelt[axis2];
             if (tmp) q_val2[axis2] = put_val3 / tmp;
             else     q_val2[axis2] = 1.0;
             break;
	    default:
	     break;
         }

         axis = axis2+1;
      }

      /* process if 1-D rwc */
      else {
         if (axis > r->n_axes) return(-1);

         switch(fld_type1) {
            case T_none:
             q_val1[axis] = 1.0;
             break;
            case T_absolute_pixel:
             q_val1[axis] = put_val1;
             break;
            case T_relative_pixel:
             q_val1[axis] = put_val1 + r->crpix[axis];
             break;
            case T_1d_rwc:
             if (r->cdelt[axis]==0) q_val1[axis] = 1.0;
             else q_val1[axis] = (put_val1 - r->crval[axis])
                                / r->cdelt[axis] + r->crpix[axis];
             break;
	    default:
	     break;
         }

         switch(fld_sep1) {

            case '\0':
             if (fld_type1 == T_none) q_type[axis] = Q_none;
             else q_type[axis] = Q_lo_hi;
             q_val2[axis] = q_val1[axis];
             break;

            case ':':
             q_type[axis] = Q_lo_hi;
             switch(fld_type2) {
                case T_none:
                 q_val2[axis] = 32768.0;
                 break;
                case T_absolute_pixel:
                 if (fld_type1 != T_relative_pixel) q_val2[axis] = put_val2;
                 else q_val2[axis] = put_val2 + r->crpix[axis];
                 break;
                case T_relative_pixel:
                 q_val2[axis] = put_val2 + r->crpix[axis];
                 break;
                case T_1d_rwc:
                 if (r->cdelt[axis]==0) q_val2[axis] = 1.0;
                 else q_val2[axis] = (put_val2 - r->crval[axis])
                                    / r->cdelt[axis] + r->crpix[axis];
                 break;
                default:
                 break;
             }
             break;

            case '^':
             q_type[axis] = Q_mid_size;
             if (fld_type1==T_none) q_val1[axis] = r->naxis[axis] / 2.0 + 0.5;
             switch(fld_type2) {
                case T_none:
                 q_val2[axis] = 1.0;
                 break;
                case T_absolute_pixel:
                case T_relative_pixel:
                 q_val2[axis] = put_val2;
                 break;
                case T_1d_rwc:
                 if (r->cdelt[axis]==0) q_val2[axis] = 1.0;
                 else q_val2[axis] = put_val2 / r->cdelt[axis];
                 break;
                default:
                 break;
             }
             break;
         }
         axis ++;
      }

   }

   /* post-process (set skips and takes) */
   takes[0] = r->n_axes;
   for (n=1; n<=r->n_axes; n++) {
      switch(q_type[n]) {
         case Q_none:
          skips[n] = 0;
          takes[n] = 32767;
          break;
         case Q_lo_hi:
          if (q_val1[n] > q_val2[n]) { /* swap if order reversed */
             tmp = q_val1[n];
             q_val1[n] = q_val2[n];
             q_val2[n] = tmp;
          }
          skips[n] = q_val1[n] - 1.0 +.5;
          takes[n] = q_val2[n] +.5;
          takes[n] -= skips[n];
          break;
         case Q_mid_size:
          if (q_val2[n] < 0) q_val2[n] = -q_val2[n];
          takes[n] = q_val2[n] +.5;
          if (takes[n] <= 0) takes[n] = 1;
          skips[n] = q_val1[n] - 1.0 +.5;
          skips[n] -= (takes[n] /2);
          break;
      } /* end switch */

/*************
printf("%d %d %f %f",n,q_type[n],q_val1[n],q_val2[n]);
printf(" = %d %d\n",skips[n],takes[n]);
*************/
      
   }

   return(0);

}



static enum val_type_enum determine_value_type(ctype,val_text,put_val)
char ctype[];
char val_text[];
double *put_val;
{
   char *c;
   int n;
   enum val_type_enum type;
   static int found_at_sign;

   if (ctype) {
      found_at_sign = 0;

   /* see if geometric coordinates */
      if (ctype[1]=='\0') switch(ctype[0]) {
         case 'r': case 'e': case 'g': return(T_geometric);
      }
      for (c=val_text; *c; c++) if (*c=='h') return(T_geometric);
   }

   else {
      for (c=val_text; *c; c++) {
         if (*c=='h' || *c=='d') {
            n = map_hms(val_text,put_val);
            if (n <0) return(T_illegal);
            return(T_1d_rwc);
         }
      }
   }

   if (*val_text == '@') {
      val_text++;
      found_at_sign = 1;
   }

   if (*val_text == '\0') return(T_none);
   
   type = T_1d_rwc;

   c = val_text;

   if (found_at_sign) {
      if (*c=='+' || *c=='-') type = T_relative_pixel;
      else                    type = T_absolute_pixel;
   }

   if (type == T_1d_rwc) { /* be sure it is floating if rwc */
      while (*c && (*c!='.' && *c!='e' && *c!='E')) c++;
      if (*c=='\0') return(T_ambig);
   }

   n = sscanf(val_text,"%lf%c",put_val,c);
   if (n!=1) return(T_illegal);

   return(type);
}
