/* rpix.c  (read pixels) */
/* %W% %E% %U% */
/* #Copyright (C) 1992, California Institute of Technology. */
/* #U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* #is acknowledged. */

#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>
#include <math.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include "im.h"

#ifdef NOTDEF
#ifndef finite
extern int      finite(double);
#endif /* finite */
#endif /* NOTDEF */

#ifdef ALPHA
#include <file.h>
#include <unixio.h>
#endif /* ALPHA */

struct fid_struct *f;

extern int im_rawflag;
static int read_bytes();

/******* *******
 return codes:
-3 = bad fid
-2 = bad file status (open for read, error, etc.)
-1 = wrong data type
 0 = no more pixels to read
 n = ok: number of pixels read
******* *******/

/*---------------------------------------------------------*/

static float *pix_ptr;
static double *dbl_ptr;

/*---------------------------------------------------------*/

int
im_rpix_i(fid,buffer,n_pix)
int fid;
int buffer[];
int n_pix;
{
   int n_bytes;

   if (IM_set_fid(fid)) return(-3);

   if (f->val_bitpix != 32) return(-2);
   
   if (n_pix < 0) return(IM_r_rewind(f->cur_window));
   n_bytes = read_bytes((unsigned char *)buffer,n_pix*4);
   if (n_bytes <= 0) return(n_bytes);
   if (f->file_type == Type_fits && !im_rawflag ) 
       IM_swap_4((unsigned char *)buffer,n_bytes);
   return(n_bytes/4);
}


/*---------------------------------------------------------*/

int
im_rpix_s(fid,buffer,n_pix)
int fid;
short int buffer[];
int n_pix;
{
   int n_bytes;
   int cur_byte_save, io_offset_save, fd, pix_fid_save;
   char *fi_name, system_string[100];
   static char temp_filename[100];
   int status;

   if (IM_set_fid(fid)) return(-3);

   if (f->val_bitpix != 16) return(-2);

   if (n_pix < 0) return(IM_r_rewind(f->cur_window));
   io_offset_save = f->io_offset;                 /* for compressed case */
   cur_byte_save = f->cur_window->cur_byte_read;  /* for compressed case */
   n_bytes = read_bytes((unsigned char *)buffer,n_pix*2);
   if (n_bytes == -2)
   {
	/* short read - see if it's a compressed file */
	/* see if name ends with .H */
	fi_name = f->pix_filename;
	if (strncmp(&fi_name[strlen(fi_name) - 2], ".H", 2) != 0)
	    return(-2);
	if (temp_filename[0] == '\0')
	{
	    strcpy(temp_filename, "/usr/tmp/skyview_decompressXXXXXX");
	    /*
	    (void) mktemp(temp_filename);
	    */
	    fd = mkstemp(temp_filename);
	    if (fd == -1)
	    {
		/*
		"Unable to create temp file for uncompressing pixels");
		*/
		return(-2);
	    }
	    else
	    {
		close (fd);
	    }
	}
	sprintf(system_string, "hdecomp -o fits < %s > %s", fi_name,
	    temp_filename);
	status = system(system_string);
	if (status != 0)
	    return(-2);
	fd = open(temp_filename, O_RDONLY, 0);
	if (fd >= 0)
	{
	    pix_fid_save = f->unix_pix_fid;    /* save fid */
	    /* fake the two values */
	    f->unix_pix_fid = fd;
	    f->io_offset = io_offset_save;
	    f->cur_window->cur_byte_read = cur_byte_save;
	    n_bytes = read_bytes((unsigned char *)buffer,n_pix*2);
	    close(fd);
	    unlink(temp_filename);  /* discard scratch file */
	    f->unix_pix_fid = pix_fid_save;    /* restore fid */
	}
   }

   if (n_bytes <= 0) return(n_bytes);
   if (f->file_type == Type_fits && !im_rawflag) 
       IM_swap_2((unsigned char *)buffer,n_bytes);
   return(n_bytes/2);
}

/*---------------------------------------------------------*/


int
im_rpix_c(fid,buffer,n_pix)
int fid;
unsigned char buffer[];
int n_pix;
{
   if (IM_set_fid(fid)) return(-3);

   if (f->val_bitpix != 8) return(-2);
   
   if (n_pix < 0) return(IM_r_rewind(f->cur_window));
   n_pix = read_bytes((unsigned char *)buffer,n_pix);
   return(n_pix);
}

/*---------------------------------------------------------*/


int
im_rpix_r(fid,buffer,n_pix)
int fid;
float buffer[];
int n_pix;
{
   union {double *dbl; float *flt; int *hex;} temp;
   int i;
   int n_bytes;

   unsigned char *i_8;
   short int *i_16;
   int *i_32;
   float value_NaN;

   /* RBH ADDED */
   if (f->val_bitpix == -64)
      return(-2);  /* the buffer won't have enough room for double prec pix */
   /* END RBH ADDED */

   if (IM_set_fid(fid)) return(-3);

   if (n_pix < 0) return(IM_r_rewind(f->cur_window));

   n_bytes = n_pix * f->bytes_per_pix;
   n_bytes = read_bytes((unsigned char *)buffer,n_bytes);
   if (n_bytes <= 0) return(n_bytes);

   n_pix = n_bytes / f->bytes_per_pix;

   im_blank(&value_NaN);


   switch(f->val_bitpix) {

      case 8:
       i_8 = (unsigned char *) buffer + n_pix-1;
       if (f->val_blank_flag) for (i=n_pix-1; i>=0; i--) {
          if (*i_8 == f->val_blank) buffer[i] = value_NaN;
          else buffer[i] = *i_8;
          i_8--;
       }
       else for (i=n_pix-1; i>=0; i--) buffer[i] = *i_8--;
       break;

      case 16:
       if (f->file_type == Type_fits ) IM_swap_2((unsigned char *)buffer,n_bytes);
       i_16 = (short int *) buffer + n_pix-1;
       if (f->val_blank_flag) for (i=n_pix-1; i>=0; i--) {
          if (*i_16 == f->val_blank) {
             buffer[i] = value_NaN;
          }
          else buffer[i] = *i_16;
          i_16--;
       }
       else for (i=n_pix-1; i>=0; i--) buffer[i] = *i_16--;
       break;

      case 32:
       if (f->file_type == Type_fits ) IM_swap_4((unsigned char *)buffer,n_bytes);
       i_32 = (int *) &buffer[n_pix-1];
       if (f->val_blank_flag) for (i=n_pix-1; i>=0; i--) {
          if (*i_32 == f->val_blank) buffer[i] = value_NaN;
          else buffer[i] = *i_32;
          i_32--;
       }
       else for(i=n_pix-1; i>=0; i--) buffer[i] = *i_32--;
       break;

      case -32:
       if (f->file_type == Type_fits ) IM_swap_4((unsigned char *)buffer,n_bytes);
       for(temp.flt=buffer+n_pix-1; temp.flt>=buffer; temp.flt--) {
	    if ( !FINITE(*temp.flt))  *temp.flt = value_NaN;
       }
       /* RBH ADDED */
       if ((f->file_type == Type_iraf) && (f->val_blank_flag))
            for(temp.flt=buffer+n_pix-1; temp.flt>=buffer; temp.flt--) {
                 if (*temp.flt == f->val_blank) *temp.flt = value_NaN;
       /* END RBH ADDED */
       }
       break;

       default:
	 return(-2); /* Returning double prec. pixels in floats is hard */
   }

 if(f->val_bitpix > 0){
   if (f->val_scale != 1.0) {
      for(pix_ptr = buffer; pix_ptr<buffer+n_pix; pix_ptr++)
       *pix_ptr *= f->val_scale;
   }

   if (f->val_offset) {
      for(pix_ptr = buffer; pix_ptr<buffer+n_pix; pix_ptr++)
       *pix_ptr += f->val_offset;
   }
 }
   return(n_pix);
}

/*---------------------------------------------------------*/


int
im_rpix_d(fid,buffer,n_pix)
int fid;
double buffer[];
int n_pix;
{
   union {double *dbl; float *flt; int *hex;} temp;
   int i;
   int n_bytes;

   unsigned char *i_8;
   short int *i_16;
   int *i_32;
   double value_NaN;

   if (IM_set_fid(fid)) return(-3);

   if (n_pix < 0) return(IM_r_rewind(f->cur_window));

   n_bytes = n_pix * f->bytes_per_pix;
   n_bytes = read_bytes((unsigned char *)buffer,n_bytes);
   if (n_bytes <= 0) return(n_bytes);

   n_pix = n_bytes / f->bytes_per_pix;

   im_blankd(&value_NaN);


   switch(f->val_bitpix) {

      case 8:
       i_8 = (unsigned char *) buffer + n_pix-1;
       if (f->val_blank_flag) for (i=n_pix-1; i>=0; i--) {
          if (*i_8 == f->val_blank) buffer[i] = value_NaN;
          else buffer[i] = *i_8;
          i_8--;
       }
       else for (i=n_pix-1; i>=0; i--) buffer[i] = *i_8--;
       break;

      case 16:
       if (f->file_type == Type_fits ) IM_swap_2((unsigned char *)buffer,n_bytes);
       i_16 = (short int *) buffer + n_pix-1;
       if (f->val_blank_flag) for (i=n_pix-1; i>=0; i--) {
          if (*i_16 == f->val_blank) {
             buffer[i] = value_NaN;
          }
          else buffer[i] = *i_16;
          i_16--;
       }
       else for (i=n_pix-1; i>=0; i--) buffer[i] = *i_16--;
       break;

      case 32:
       if (f->file_type == Type_fits ) IM_swap_4((unsigned char *)buffer,n_bytes);
       i_32 = (int *) buffer +n_pix-1;
       if (f->val_blank_flag) for (i=n_pix-1; i>=0; i--) {
          if (*i_32 == f->val_blank) buffer[i] = value_NaN;
          else buffer[i] = *i_32;
          i_32--;
       }
       else for(i=n_pix-1; i>=0; i--) buffer[i] = *i_32--;
       break;

      case -32:
       if (f->file_type == Type_fits ) IM_swap_4((unsigned char *)buffer,n_bytes);
       temp.flt = (float *) buffer +n_pix-1;
       for(i=n_pix-1; i>=0; i--){
	    if ( !FINITE(*temp.flt)) buffer[i] = value_NaN;
	    else buffer[i] = *temp.flt;
	    temp.flt--;
       }

       /* RBH ADDED */
       if ((f->file_type == Type_iraf) && (f->val_blank_flag))
       {
            for(i=n_pix-1; i>=0; i--)
                 if (buffer[i] == f->val_blank) buffer[i] = value_NaN;
       }
       /* END RBH ADDED */
       break;

      case -64:
       if (f->file_type == Type_fits ) IM_swap_8((unsigned char *)buffer,n_bytes);
       for(temp.dbl=buffer+n_pix-1; temp.dbl>=buffer; temp.dbl--) {
	    if (!FINITE(*temp.dbl))  *temp.dbl = value_NaN;
       }
       /* RBH ADDED */
       if ((f->file_type == Type_iraf) && (f->val_blank_flag))
       {
	   for(temp.dbl=buffer+n_pix-1; temp.dbl>=buffer; temp.dbl--) {
		if (*temp.dbl == f->val_blank) *temp.dbl = value_NaN;
	   }
       }
       break;
   }

  if(f->val_bitpix > 0){
   if (f->val_scale != 1.0) {
      for(dbl_ptr = buffer; dbl_ptr<buffer+n_pix; dbl_ptr++)
       *dbl_ptr *= f->val_scale;
   }

   if (f->val_offset) {
      for(dbl_ptr = buffer; dbl_ptr<buffer+n_pix; dbl_ptr++)
       *dbl_ptr += f->val_offset;
   }
 }
   return(n_pix);
}

/*---------------------------------------------------------*/

static int read_bytes(buffer,n_bytes_to_do)
unsigned char *buffer;
int n_bytes_to_do;
{
   struct window_struct *w;
   int n_bytes_done;
   int actual_bytes;
   int now_bytes;
   int dim;
   int i;

   w = f->cur_window;

   n_bytes_done = 0;
   if (n_bytes_to_do > w->total_bytes_left_to_read)
     n_bytes_to_do = w->total_bytes_left_to_read;

   while (n_bytes_to_do) {

      if (f->unix_pix_fid)
       lseek(f->unix_pix_fid,f->io_offset+w->cur_byte_read,0);
      if (n_bytes_to_do > w->line_bytes_left_to_read)
        now_bytes = w->line_bytes_left_to_read;
        else now_bytes = n_bytes_to_do;
      if (f->unix_pix_fid)
       actual_bytes = read(f->unix_pix_fid,buffer,now_bytes);
      else actual_bytes = fread(buffer,1,now_bytes,stdin);
      if (actual_bytes < now_bytes) return(-2);

      buffer += actual_bytes;
      n_bytes_done += actual_bytes;
      n_bytes_to_do -= actual_bytes;
      w->cur_byte_read += actual_bytes;
      w->line_bytes_left_to_read -= actual_bytes;
   
      if (w->line_bytes_left_to_read <=0) {
         w->cur_byte_read += w->skip_bytes;
         w->line_bytes_left_to_read = w->line_bytes;

         w->left_to_read[2] --;
         for (i=2; (i<f->val_naxis[0]) && (w->left_to_read[i]==0); i++) {
            if (i==2) dim = f->bytes_per_pix;
            dim *= f->val_naxis[i-1];
            w->cur_byte_read += (f->val_naxis[i] - w->naxis_take[i]) * dim;
            w->left_to_read[i] = w->naxis_take[i];
            w->left_to_read[i+1] --;
         }
      }

   }
 
   w->total_bytes_left_to_read -= n_bytes_done;
   return(n_bytes_done);

}

/*---------------------------------------------------------*/

int
IM_r_rewind(w)
struct window_struct *w;
{
   int i;
   w->cur_byte_read = w->start_byte;
   w->total_bytes_left_to_read = w->total_bytes;
   w->line_bytes_left_to_read = w->line_bytes;
   for (i=2; i < f->val_naxis[0]; i++) w->left_to_read[i] = w->naxis_take[i];
   return(w->total_bytes / f->bytes_per_pix);
}
