/* rkey.c  (read keyword value) */
/* %W% %E% %U% */
/* #Copyright (C) 1992, California Institute of Technology. */
/* #U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* #is acknowledged. */

#include "im.h"
#include <string.h>

/*
return codes:
-4 = bad value
-3 = bad fid
-2 = bad call
-1 = can't find keyword
 0 = ok, (value available, but no actual card)
>0 = ok, (pointer to card)
*/

extern struct fid_struct *f;

#ifdef _NO_PROTO
static find_kwd();
static adjust_for_windows();
#else
static int find_kwd(char slot[], char keyword[]);
static int adjust_for_windows(char keyword[], char value[]);
#endif /* _NO_PROTO */


/*---------------------------------------------------------*/

int
im_rkey_i(fid,ptr,keyword)
int fid;
int *ptr;
char keyword[];
{
   int stat;
   char slot[80];

   if (ptr == 0) return(-2);

   if (IM_set_fid(fid)) return(-3);

   if (strncmp(keyword,"npix",4) == 0) {
      if (f->cur_window==0) {*ptr = f->n_pixels; return(0);}
      *ptr = f->cur_window->total_bytes / f->bytes_per_pix;
      return(0);
   }

   if (strncmp(keyword,"filetype",8) == 0) {
      switch(f->file_type)
      {
	  case Type_unknown:
	      *ptr = 0;
	      break;
	  case Type_fits:
	      *ptr = 1;
	      break;
	  case Type_zz:
	      *ptr = 2;
	      break;
	  case Type_iraf:
	      *ptr = 3;
	      break;
	  case Type_S:
	      *ptr = 4;
	      break;
      }
      return(0);
   }

   stat = find_kwd(slot,keyword);
   if (stat < 0) return(stat);

   if (kwd_isvalid(keyword) == 0) return -4;
   adjust_for_windows(keyword,slot);
   
   if (sscanf(slot,"%d",ptr) != 1) return(-4);

   return(stat);

}

/*---------------------------------------------------------*/

int
im_rkey_l(fid,ptr,keyword)
int fid;
int *ptr;
char keyword[];
{
   int stat;
   char slot[80];

   if (ptr == 0) return(-2);

   if (IM_set_fid(fid)) return(-3);

   stat = find_kwd(slot,keyword);
   if (stat < 0) return(stat);

   if (! kwd_isvalid(keyword)) return -4;
   switch (slot[19]) {
      case 'T': *ptr = 1; break;
      default:  *ptr = 0;
   };
   return(stat);

}

/*---------------------------------------------------------*/

int
im_rkey_r(fid,ptr,keyword)
int fid;
float *ptr;
char keyword[];
{
   int stat;
   char slot[80];

   if (ptr == 0) return(-2);

   if (IM_set_fid(fid)) return(-3);

   stat = find_kwd(slot,keyword);
   if (stat < 0) return(stat);

   if (! kwd_isvalid(keyword)) return -4;
   adjust_for_windows(keyword,slot);

   if (sscanf(slot,"%f",ptr) != 1) return(-4);

   return(stat);

}

/*---------------------------------------------------------*/

int
im_rkey_d(fid,ptr,keyword)
int fid;
double *ptr;
char keyword[];
{
   int stat;
   char slot[80];

   if (ptr == 0) return(-2);

   if (IM_set_fid(fid)) return(-3);

   stat = find_kwd(slot,keyword);
   if (stat < 0) return(stat);

   if (! kwd_isvalid(keyword)) return -4;
   adjust_for_windows(keyword,slot);

   if (sscanf(slot,"%lf",ptr) != 1) return(-4);
   return(stat);

}

/*---------------------------------------------------------*/

int
im_rkey_c(fid,ptr,keyword)
int fid;
char *ptr;
char keyword[];
{
   int stat;
   char slot[80];
   char *c;

   if (ptr == 0) return(-2);

   if (IM_set_fid(fid)) return(-3);

   stat = find_kwd(slot,keyword);
   if (stat < 0) return(stat);

   if (! kwd_isvalid(keyword)) return -4;
   if (slot[0] != '\'') return(-1);
   for (c=slot+1; *c != '\''; c++) if (*c == '\0') return(-1);
   /* strip off trailing blanks */
   /* for (c=c-1; *c==' ';c--); */
   /* *(c+1) = '\0';            */
   *c = '\0';
   strcpy(ptr,slot+1);

   return(stat);

}

/*---------------------------------------------------------*/

int
im_rkey_t(fid,ptr,keyword)
int fid;
char ptr[81];
char keyword[];
{
   struct header_struct *h;
   char *c, *card;
   int stat;
   int retval;
   int flag = 0; /* set !=0 if in a quoted string */

   if (ptr == 0) return(-2);

   if (IM_set_fid(fid)) return(-3);
   h = f->cur_header;

   switch(keyword[0]) {

      case '/': /* comment field */
       card = h->text;
       c = card+10;
       while (*c!='\0' && (*c!='/' || flag==1) && c<card+77) {
          if (*c == '\'') {if (flag) flag=0; else flag=1;}
          c++;
       }
       if (*c != '/') return(-1);
       strcpy(ptr,c+1);
       return(2);

      case '=': /* current card */
       h = f->cur_header;
       retval = 2;
       break;
 
      case '+': /* next card */
       if (h == 0) return(-1);
       h = h->next;
       retval = 3;
       break;
 
      case '-': /* previous card */
       if (h == 0) return(-1);
       h = h->prev;
       retval = 1;
       break;
 
      case '\0': /* last card */
       h = f->last_header;
       if (f->cur_header == h)
	   retval = 2;
       else
	   retval = 3;
       break;
 
      case '1': /* first card */
       h = f->first_header;
       if (f->cur_header == h)
	   retval = 2;
       else
	   retval = 1;
       break;

      default: /* keyword lookup */
       stat = IM_card_find(keyword);
       if (stat > 0)
	   h = f->cur_header;
       else
	   h = 0;
       retval = stat;

   } /* end switch */

   if (h == 0) return(-1);

   strncpy(ptr,h->text,80);
   ptr[80] = '\0'; /* guarantee a null at end of string */
   f->cur_header = h;

   if (! kwd_isvalid(keyword)) return -4;
   adjust_for_windows(ptr,ptr+10);

   return(retval);

}


/*----------------------------------------------------------------*/

static int find_kwd(slot,keyword)
char slot[80];
char keyword[];
/* =-1 if not found, >0 found,  <-1 if illegal */
{
   char *c, *s;
   int stat;

   stat = IM_card_find(keyword);
   if (stat>0) {
      for (c=f->cur_header->text+10,s=slot; *c;) *s++ = *c++;
      *s = '\0';
   }
   return (stat);
}



/*---------------------------------------------------------*/
static int adjust_for_windows(keyword,value)
 /* returns: =0 if substitution;   =1 if not */
char keyword[];
char value[];
{
   int n;
   char *c;
   float fval;

   if (f->cur_window==0 || f->cur_window==f->windows) return(1);

   if (strncmp(keyword,"NAXIS",5) == 0) { /* adjust for windows */
      n = keyword[5] - '0';
      if (n>=1 && n<=9) {
         sprintf(value+0,"%20d",f->cur_window->naxis_take[n]);
         return(0);
      }
   }
 
   else if (strncmp(keyword,"CRPIX",5) == 0) { /* adjust for windows */
      n = keyword[5] - '0';
      if (n>=1 && n<=9) {
         sscanf(value+0,"%f",&fval);
         fval -= (float) f->cur_window->naxis_skip[n];
         sprintf(value+0," %19.10G",fval);
         for (c=value+0; *c && *c!='.'; c++);
         /* add decimal if none */
         if (*c=='\0') {for (c=value+2; *c; c++) *(c-1)= *c; *(c-1)='.';}
         return(0);
       }
    }

    return(1);
}
