/* open.c */
/* %W% %E% %U% */
/* contains im_open and its subroutines */
/* Copyright (C) 1992,1993, California Institute of Technology.   */
/* U.S. Government Sponsorship under NASA Contract NAS7-918  */
/* is acknowledged.                                          */

/* Sep 98, RBH: discard HDR$/ as well as HDR$ in pix filename */
/* Aug 97, RBH: put date into V_IM for "what" command */
/* Sep 95, RBH: close file when format is bad */
/* Jan 93, add support for IRAF type 7, FITS BITPIX=-64      */

#include "im.h"
#include "irafhdr.h"
#include <sys/file.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <unistd.h>

#ifdef ALPHA
#include <file.h>
#include <unixio.h>
#endif /* ALPHA */

struct fid_struct *f;

static int unix_mode;

char *V_IM = "@(#)im.a 08/21/97";

/* __________________________________________ */

static void parse_filename(full_name,file_name,subset_text)
char *full_name;
char *file_name;
char *subset_text;
/* seperate actual file name from subset specification */
{
   char *i, *j;

   for (i=full_name, j=file_name; *i&&(*i!=','); i++, j++) *j = *i;
   *j = '\0';

   j = subset_text;
   if (*i == ',') for (i++; *i; i++, j++) *j = *i;
   *j = '\0';
}


/* __________________________________________ */

static void iraf_to_ascii(strng,nn)
char *strng;
int nn;
/* translates the "double" iraf text to real ASCII and adds a null */
{
   char *ptri, *ptro;

   /* Modified by David Shupe 01-Nov-1996 to make it work on Linux */
   if (im_byteorder() == 1)
       for (ptri=strng+1, ptro=strng+0; ptri < strng+nn; ptri += 2, ptro++) 
	 *ptro = *ptri;
   else
       for (ptri=strng+0, ptro=strng+0; ptri < strng+nn; ptri += 2, ptro++)
	 *ptro = *ptri;

   *ptro = '\0';
}

/* __________________________________________ */


static int get_iraf_header()
{

   struct IMHDR ih;
   char buff[2880];
   char *c, *cc;
   char *c_iraf, *c_ascii;
   char ctemp[16];
   int n;

   /* read header data */
   n = read(f->unix_hdr_fid,ih.im_magic,sizeof(ih));
   if (n != sizeof(ih)) return(-1);

   /* check validity */
   iraf_to_ascii(ih.im_magic,SZ_IMMAGIC);
   if (strcmp(ih.im_magic,"imhdr")) return(-1);

   /* set BITPIX */
   switch(ih.im_pixtype) {
      case 3: n =  16; break;
      case 4:
      case 5: n =  32; break;
      case 6: n = -32; break;
      case 7: n = -64; break;
      default: return(-1);
   }
   im_wkey_i(f->id,n,"BITPIX");

   /* set NAXIS's */
   im_wkey_i(f->id,ih.im_ndim,"NAXIS");
   for (n=1; n<=ih.im_ndim; n++) im_wkey_i(f->id, ih.im_len[n-1], im_key("NAXIS",n));
   f->line_skip_pix = ih.im_physlen[0] - ih.im_len[0];

   /* set BSCALE & BZERO */
   if (ih.ct_bscale &&(ih.ct_bscale != 1.0)) im_wkey_r(f->id,ih.ct_bscale,"BSCALE");
   if (ih.ct_bzero)  im_wkey_r(f->id,ih.ct_bzero,"BZERO");

   /* OBJECT */
   if(ih.im_title[0]) {
      iraf_to_ascii(ih.im_title,SZ_IMTITLE);
      im_wkey_c(f->id,ih.im_title,"OBJECT");
   }

   /* set DATAMIN & DATAMAX & NBLANKS */
   if (ih.im_min || ih.im_max) {
      im_wkey_r(f->id,ih.im_min,"DATAMIN");
      im_wkey_r(f->id,ih.im_max,"DATAMAX");
   }
   if (ih.im_nbpix)  im_wkey_i(f->id,ih.im_nbpix,"NBLANKS");

   /* additional header cards */
   (void) lseek(f->unix_hdr_fid,SZ_HDR,0);
   while ((n = read(f->unix_hdr_fid,buff,162))) {
      if (n == 162) {
         iraf_to_ascii(buff,n);
         buff[80] = '\0';
         im_wkey_t(f->id,buff,"add");
      }
   }

   /* "history" cards */
   strcpy(buff,"HISTORY ");
   c_iraf = ih.im_history+1;
   while (*c_iraf) {
      for (c_ascii=buff+8; *c_iraf != '\n' ; c_ascii++, c_iraf+=2)
            *c_ascii = *c_iraf;
      *c_ascii = '\0';
      c_iraf += 2;
      im_wkey_t(f->id,buff,"add");
   }

   /* close the hdr file */
   close(f->unix_hdr_fid);

   /* open pixel file */
   iraf_to_ascii(ih.im_pixfile,SZ_IMPIXFILE);
   cc = ih.im_pixfile;
   for (c=cc; *c; c++) if (*c == '!') cc = c+1; /* skip remote ID */
   if (strncmp(cc,"HDR$/",5)==0)  cc += 5; /* skip HDR$/ if present RBH 9-10-98*/
   if (strncmp(cc,"HDR$",4)==0)  cc += 4; /* skip HDR$ if present */
   n = -1; /* n indexes the last / in the path */
   if (*cc != '/') { /* get path from name of header file */
      for(c=f->hdr_filename ; *c; c++) if (*c=='/') n = c - f->hdr_filename;
      if (n >= 0) strncpy(f->pix_filename,f->hdr_filename,n+1);
   }
   strcpy(f->pix_filename+n+1,cc);

   if ((int)strlen(cc) > 63) return(-1);
   else f->unix_pix_fid = open(f->pix_filename,unix_mode,0);
   if (f->unix_pix_fid < 0) return(-2);
   n = read(f->unix_pix_fid,ctemp,SZ_IMMAGIC);
   iraf_to_ascii(ctemp,SZ_IMMAGIC);
   if (strcmp(ctemp,"impix")) return(-1);
      
   f->io_offset = SZ_PIXHDR;
   return(0);
}

/* __________________________________________ */

static int get_zz_header()
{
   int n;
   FILE *stream_fid;
   char buffer[82];
   char *c;

   /* read header info */
   close(f->unix_hdr_fid);
   stream_fid = fopen(f->hdr_filename,"r");
   while (fgets(buffer,82,stream_fid)) {
      if (buffer[8] != '=' && buffer[9] != '=' &&
       strncmp(buffer,"       ",7)  > 0 &&
       strncmp(buffer,"COMMENT",7) != 0 &&
       strncmp(buffer,"HISTORY",7) != 0) {
         continue;
      }
      for (c=buffer; *c; c++) if (*c == '\n') *c = '\0';
      if (c > buffer+81) {
         buffer[79] = '\0';
      }
      IM_card_add(buffer);
   }
   fclose(stream_fid);

   /* open pixel file */
   strcpy(f->pix_filename,f->hdr_filename);
   n = strlen(f->pix_filename);
   if (strcmp(f->pix_filename+n-4,".zzh") == 0) f->pix_filename[n-4] = '\0';
   strcat(f->pix_filename,".zzp");
   f->unix_pix_fid = open(f->pix_filename,unix_mode,0);
   if (f->unix_pix_fid < 0) return(-2);

   f->magic = IM_MAGIC;
   return(0);
}

/* __________________________________________ */

static int get_fits_header()
{
   int n;
   int i;
   int need_simple; /* need check for SIMPLE=T */
   char *c, *j;
   char buff[2880];

   need_simple = 1;
   for (;;) {
      if (f->unix_hdr_fid) n = read(f->unix_hdr_fid,buff,2880);
      else n = fread(buff,1,2880,stdin);
      if (n != 2880) return(-1);
      f->io_offset += 2880;
      for (c=buff; c<buff+2880; c += 80) {
         /* check if END card */
         if (strncmp(c,"END     ",8)==0) {
            f->unix_pix_fid = f->unix_hdr_fid;
            strcpy(f->pix_filename,f->hdr_filename);
            f->magic = IM_MAGIC;
            return(0);
         }
         /* strip off trailing blanks */
         for(j=c+79; *j==' ' && j>=c; j--);
         if (j!=c+79) *(j+1) = '\0';
         /* check if SIMPLE card */
         if (need_simple) { 
            need_simple = 0;
            if (strncmp(c,"SIMPLE ",7) || *(c+29)!='T') return(-1);
         }
         i = IM_card_add(c);
	 if (i < 0) 
	 {
	     return(-1); /* illegal card */
	 }
      }
   }
}

/* __________________________________________ */

static int get_s_header()
{
   char *c, *j;
   char buff[2880];
   struct S_S {
      int tapetype;
      int hdr_start;
      int img_start;
   } s_struct;

   read(f->unix_hdr_fid,&s_struct,sizeof(s_struct));
   if (s_struct.tapetype != 46) return(-1);
   lseek(f->unix_hdr_fid,s_struct.hdr_start,0);

   for (;;) {
      read(f->unix_hdr_fid,buff,2880);
      for (c=buff; c<buff+2880; c += 80) {
         /* check if END card */
         if (strncmp(c,"END     ",8)==0) {
            f->io_offset = s_struct.img_start;
            f->unix_pix_fid = f->unix_hdr_fid;
            strcpy(f->pix_filename,f->hdr_filename);
            f->magic = IM_MAGIC;
            return(0);
         }
         /* strip off trailing blanks */
         for(j=c+79; *j==' ' && j>=c; j--);
         if (j!=c+79) *(j+1) = '\0';
         IM_card_add(c);
      }
   }
}

int
im_open(in_filename,mode)
char in_filename[];
char mode[];
/* -1 if illegal file format 
   -2 if can't open
   -3 if bad call */
{

   int i, n;
   char buff[2880];
   int *buff_int_ptr = (int *) buff;
   char *c;
   int stat;
   int take1[10], skip1[10]; /* for default window 1 */

   c = V_IM; /* to insure version is linked in for programs */

   f = new_fid();

   f->file_type = Type_unknown;
   f->val_scale = 1.0;
   for (i=1; i<MAX_WINDOWS; i++) f->windows[i].total_bytes = -1; 
                                            /* mark window as not defined */

   strncpy(f->user_filename,in_filename,MAXPATHLEN);
   parse_filename(in_filename,f->hdr_filename,f->subset_text);

   switch(mode[0]) {

      case 'r':
       if (strcmp(f->hdr_filename,"-" ) == 0) {
          if (mode[1] != '-') return(-3);
          f->unix_hdr_fid = 0;
          f->unix_pix_fid = 0;
          f->file_type = Type_fits;
	  /*FALLTHROUGH*/
       }
      case 'u':
       if (mode[0] == 'r') {
          f->mode = 1; /* read */
          unix_mode = O_RDONLY; /* read only */
       }
       else {
         f->mode = 3; /* update */
         unix_mode = O_RDWR; /* read&write */
       }

       if (f->file_type == Type_unknown) {
          f->unix_hdr_fid = open(f->hdr_filename,unix_mode,0);
          n = strlen(f->hdr_filename);
          if (f->unix_hdr_fid > 0) {
             if      (strcmp(&f->hdr_filename[n-4],".imh") == 0)
              f->file_type = Type_iraf;
             else if (strcmp(&f->hdr_filename[n-4],".zzh" ) == 0)
              f->file_type = Type_zz;
             else if (strcmp(&f->hdr_filename[n-5],".fits") == 0)
              f->file_type = Type_fits;
             else if (strcmp(&f->hdr_filename[n-2],".s") == 0)
              f->file_type = Type_S;
          }
       }

       if (f->unix_hdr_fid < 0) {
          strcpy(f->hdr_filename+n,".imh");
          f->unix_hdr_fid = open(f->hdr_filename,unix_mode,0);
          if (f->unix_hdr_fid >0) f->file_type = Type_iraf;
       }

       if (f->unix_hdr_fid < 0) {
          strcpy(f->hdr_filename+n,".zzh");
          f->unix_hdr_fid = open(f->hdr_filename,unix_mode,0);
          if (f->unix_hdr_fid >0) f->file_type = Type_zz;
       }

       if (f->unix_hdr_fid < 0) {
          strcpy(f->hdr_filename+n,".fits");
#ifdef ALPHA

	  f->unix_hdr_fid = open(in_filename, unix_mode, 0666, "RFM=FIX","MRS=2880","BLS=2880");
#else
          f->unix_hdr_fid = open(f->hdr_filename,unix_mode,0);
#endif /* ALPHA */
          if (f->unix_hdr_fid >0) f->file_type = Type_fits;
       }

       if (f->unix_hdr_fid < 0) {
          strcpy(f->hdr_filename+n,".fit");
#ifdef ALPHA

	  f->unix_hdr_fid = open(in_filename, unix_mode, 0666, "RFM=FIX","MRS=2880","BLS=2880");
#else
          f->unix_hdr_fid = open(f->hdr_filename,unix_mode,0);
#endif /* ALPHA */
          if (f->unix_hdr_fid >0) f->file_type = Type_fits;
       }

       if (f->unix_hdr_fid < 0) {
          strcpy(f->hdr_filename+n,".s");
          f->unix_hdr_fid = open(f->hdr_filename,unix_mode,0);
          if (f->unix_hdr_fid >0) f->file_type = Type_S;
       }
       if (f->unix_hdr_fid < 0) /* still ! */
	  return -2;

       if (f->file_type == Type_unknown) { /* determine file type from file */
          n = read(f->unix_hdr_fid,buff,16);
          if      (strncmp(buff,"SIMPLE",6) == 0) f->file_type = Type_fits;
          else if (strncmp(buff,"BITPIX",6) == 0) f->file_type = Type_zz;
          else if (strncmp(buff,"NAXIS" ,5) == 0) f->file_type = Type_zz;
          else if (buff[1]=='i' && buff[3]=='m' && buff[5]=='h' &&
                   buff[7]=='d' && buff[9]=='r')  f->file_type = Type_iraf;
          else if (*buff_int_ptr == 46) f->file_type = Type_S;
          else 
	  {
	      close(f->unix_hdr_fid);  /* RBH added */
	      return(-1);
	  }
          lseek(f->unix_hdr_fid,0,0);
       }

       f->magic = IM_MAGIC;

       switch(f->file_type) {
          case Type_fits: stat = get_fits_header(); break;
          case Type_zz: stat = get_zz_header();  break;
          case Type_iraf: stat = get_iraf_header(); break;
          case Type_S: stat = get_s_header(); break;
	  case Type_unknown: stat = -2; break;
       }

       if (stat < 0) return(stat);

       if (f->unix_hdr_fid == 0) f->subset_text[0] = '\0';
       stat = IM_setw1(f ,skip1,take1);
       if (stat < 0) return(-4);

       im_dwin(f->id, 1,take1,skip1); /* initialize window 1 */
       im_swin(f->id, 1); /* select window 1 */

       break;


      case 'w':
       if (strcmp(f->hdr_filename,"-" ) == 0) if (mode[1] != '-') return(-2);

	/* RBH addition start */
	n = strlen(f->hdr_filename);
	if      (strcmp(&f->hdr_filename[n-4],".imh") == 0)
	    f->file_type = Type_iraf;
	else if (strcmp(&f->hdr_filename[n-4],".zzh" ) == 0)
	    f->file_type = Type_zz;
	else if (strcmp(&f->hdr_filename[n-5],".fits") == 0)
	    f->file_type = Type_fits;
	else if (strcmp(&f->hdr_filename[n-2],".s") == 0)
	    f->file_type = Type_S;
       /* RBH addition end */

       f->need_wpix_setup = 1;
       f->mode = 2; /* (write) */
       f->magic = IM_MAGIC;
       return(f->id);

      default:
       return(-3);
   }

   return(f->id);

}

