/* misc.c  (misc. internal subroutines) */
/* %W% %E% %U% */
/* Copyright (C) 1992,1993 California Institute of Technology.   */
/* U.S. Government Sponsorship under NASA Contract NAS7-918      */
/* is acknowledged.                                              */
/* Jan 1993, add bitpix = -64 Iraftype=7    <rick@ipac.caltech.edu */

#include "im.h"
#include "irafhdr.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#ifndef MSDOS
#include <sys/file.h>
#include <unistd.h>
#endif /* MSDOS */

#ifdef ALPHA
#include <file.h>
#include <unixio.h>
#endif /* ALPHA */

extern int errno; /* UNIX error number */

struct fid_struct *f;

static struct IMHDR ih;
static int iraf_time;
static int iraf_hist_index;

static int put_iraf_card();
static int put_iraf_history_card();
static void ascii_to_iraf();
static void write_fits_card();
static struct fid_struct head_fid = {&head_fid, &head_fid};

int im_rawflag = 0;

/*---------------------------------------------------------*/

struct fid_struct *new_fid()
{
    struct fid_struct *tmpfid, *x;
    static int next_id = 1;

    tmpfid = (struct fid_struct *) calloc(1,sizeof(struct fid_struct));
    if (tmpfid == NULL)
    {
	/* in trouble now */
	fprintf(stderr, "Cannot malloc space in new_fid()\n");
	exit(1);
    }
    /* insert to right of head_fid */
    x = &head_fid;
    tmpfid->llink = x;
    tmpfid->rlink = x->rlink;
    (tmpfid->rlink)->llink = tmpfid;
    x->rlink = tmpfid;

    tmpfid->id = next_id++;
    return(tmpfid);
}

/*---------------------------------------------------------*/

void
destroy_fid(p)
struct fid_struct *p;
{
    (p->llink)->rlink = p->rlink;
    (p->rlink)->llink = p->llink;
    free(p);
    f = NULL;
}


/*---------------------------------------------------------*/

struct fid_struct *find_fid(id)
int id;
{
    struct fid_struct *p;

    p = head_fid.rlink;
    while (p != &head_fid)
    {
	if (p->id == id)
	{
	    return(p);
	}
	p = p->rlink;
    }
    return(NULL);
}


/*---------------------------------------------------------*/

int
IM_set_fid(id)
int id;
/* check fid and set set f to it if ok */
{
   f = find_fid(id);
   if (f == NULL)
       return(-3);

   if (f->magic != IM_MAGIC) return(-3);

   return(0);
}


/*---------------------------------------------------------*/


int
IM_write_hdr()
/* write the current header info into the file */
/* return:  0=normal    -2=error(messsage sent already)   */
{
   struct header_struct *h;
   int naxis_flag;
   FILE *stream_fid;
   char tbuf[200];
   int n;
   int imh_len;
   char *c, *cc;


   switch(f->file_type) {

      case Type_fits:
       if (f->io_offset != ((f->n_headers+3+35)/36)*2880) return(-2);
       f->unix_hdr_fid = f->unix_pix_fid;
       if (f->unix_hdr_fid)
        lseek(f->unix_hdr_fid,0,0); /* rewind the file */
       write_fits_card ("SIMPLE  =                    T");
       naxis_flag=0;
       for (h=f->first_header; h; h=h->next) {
          /* add BLOCKED and EXTEND keywords after NAXIS keywords */
          if (strncmp(h->text,"NAXIS",5) ==0) naxis_flag ++;
          else if (naxis_flag > 0) {
             write_fits_card
 ("EXTEND  =                    T / TAPE MAY HAVE STANDARD FITS EXTENSIONS");
#ifdef NOTDEF /* BLOCKED is deprecated in V100-1.0 of FITS Std. rje 29Mar94 */
             write_fits_card
 ("BLOCKED =                    T / TAPE MAY BE BLOCKED IN MULTIPLES OF 2880");
#endif
             naxis_flag = -99999;
          }
               
          write_fits_card(h->text);
       }
       write_fits_card("END");
       write_fits_card(0); /* pad out */
       break;


      case Type_zz:
       stream_fid = fopen(f->hdr_filename,"w");
       for (h=f->first_header; h; h=h->next) {
          fputs(h->text,stream_fid);
          fputs("\n",stream_fid);
       }
       fclose(stream_fid);
       break;


      case Type_iraf:
       iraf_time = get_iraf_time();

       if (f->mode == 2) { /* (write mode) */
          f->unix_hdr_fid = open(f->hdr_filename,O_RDWR|O_CREAT|O_TRUNC,0664);
          for (n=0; n<sizeof(ih); n++) ih.im_magic[n]   = '\0';

          for(cc=c=f->hdr_filename; *c; c++) if (*c == '/') cc=c+1;
          ascii_to_iraf(ih.im_hdrfile,cc,SZ_IMHDRFILE);
          for(cc=c=f->pix_filename; *c; c++) if (*c == '/') cc=c+1;
          ascii_to_iraf(ih.im_pixfile,"HDR$",8);
          ascii_to_iraf(ih.im_pixfile+8,cc,SZ_IMPIXFILE-8);

          ih.im_ctime = iraf_time;
          ih.im_pixoff = (SZ_PIXHDR / 2) +1;
          ih.im_hgmoff = ih.im_pixoff + (f->n_pixels * f->bytes_per_pix / 2);
          ih.im_blist  = ih.im_hgmoff + 4096;
       }
       else { /* (update mode) */
          f->unix_hdr_fid = open(f->hdr_filename,O_RDWR,0);
          read(f->unix_hdr_fid,&ih,sizeof(ih));
          ih.im_ndim = ih.im_max = ih.im_min = ih.ct_bscale = ih.ct_bzero = 0;
          for (n=0; n<IM_MAXDIM; n++) ih.im_len[n] = ih.im_physlen[n] = 0;
          for (n=0; n<SZ_IMTITLE; n++) ih.im_title[n]   = '\0';
          for (n=0; n<SZ_IMHIST;  n++) ih.im_history[n] = '\0';
       }
       if (f->unix_hdr_fid < 0) return(-2);

       /* fill in info */
          ih.im_mtime = iraf_time;

       /* fill in from header card info */
       lseek(f->unix_hdr_fid,SZ_HDR,0);
       iraf_hist_index = 0;
       for (h=f->first_header; h; h=h->next) {
         if (put_iraf_card(h->text) < 0) return(-2);
       }

       /* add a couple of nulls at end of file */
       tbuf[0] = tbuf[1] = '\0';
       write(f->unix_hdr_fid,tbuf,2);
       imh_len = lseek(f->unix_hdr_fid,0,SEEK_CUR);
       ih.im_hdrlen = imh_len / 4;

       /* write .imh file header info */
       ascii_to_iraf(ih.im_magic,"imhdr",SZ_IMMAGIC);
       lseek(f->unix_hdr_fid,0,0);
       write(f->unix_hdr_fid,&ih,sizeof(ih));
       close(f->unix_hdr_fid);

       /* write .pix file header info */
       ascii_to_iraf(ih.im_magic,"impix",SZ_IMMAGIC);
       lseek(f->unix_pix_fid,0,0);
       write(f->unix_pix_fid,&ih,SZ_PIXHDR);
 
       break;


      default: /* ERROR */
       return(-3);

   }; /* end switch filetype */

   return(0);
}


static int put_iraf_card(card)
char *card;
/* return:  0=OK   -2=illegal BITPIX for IRAF  */
{
   int i, n;
   char *c;
   char tbuf[2880];

   switch(card[0]) {

      case 'B':
       if (strncmp(&card[1],"ITPIX  ",7) == 0) {
          sscanf(&card[10],"%d",&n);
          switch(n) {
             case  16: ih.im_pixtype = 3; break;
             case  32: ih.im_pixtype = 5; break;
             case -32: ih.im_pixtype = 6; break;
             case -64: ih.im_pixtype = 7; break;
             default:  ih.im_pixtype = 0; return(-2);
          }
          return(0);
       }

       if (strncmp(&card[1],"SCALE  ",7) == 0) {
          sscanf(&card[10],"%f",&ih.ct_bscale);
          return(0);
       }

       if (strncmp(&card[1],"ZERO   ",7) == 0) {
          sscanf(&card[10],"%f",&ih.ct_bzero);
          if (ih.ct_bscale == 0) ih.ct_bscale = 1.0;
          return(0);
       }

       break;

      case 'D':

       if (strncmp(&card[1],"ATAMIN ",7) == 0) {
          sscanf(&card[10],"%f",&ih.im_min);
          ih.im_limtime = iraf_time;
          return(0);
       }

       if (strncmp(&card[1],"ATAMAX ",7) == 0) {
          sscanf(&card[10],"%f",&ih.im_max);
          ih.im_limtime = iraf_time;
          return(0);
       }

       if (strncmp(&card[1],"ATE    ",7) == 0) {
          if (put_iraf_history_card(card+10) < 0) break;
          return(0);
       }

       break;

      case 'H':

       if (strncmp(&card[1],"ISTORY ",7) == 0) {
          if (put_iraf_history_card(card+8) < 0) break;
          return(0);
       }

       break;
   
      case 'N':

       if (strncmp(&card[1],"AXIS",4) == 0) {
          if (card[6]!=' ') break;
          sscanf(&card[10],"%d",&n);
          if (card[5] == ' ') {
             ih.im_ndim = n;
             return(0);
          }
          else i = card[5] - '0';
          if (i<1 || i>7) break;
          ih.im_len[i-1] = ih.im_physlen[i-1] = n;
          return(0);
       } /* end if NAXIS */

       if (strncmp(&card[1],"BLANKS ",7) == 0) {
          sscanf(&card[10],"%ld",&ih.im_nbpix);
          return(0);
       }

       break;

      case 'O':

       if (strncmp(&card[1],"BJECT  ",7) == 0) {
          for(c=card+12; *c!='\''; c++);
          *c = '\0';
          ascii_to_iraf(ih.im_title,card+11,SZ_IMTITLE);
          return(0);
       }

       if (strncmp(&card[1],"RIGIN  ",7) == 0) {
          if (put_iraf_history_card(card+10) < 0) break;
          return(0);
       }

       break;

   } /* end switch card[0] */

   /* pad with blanks & write out card at end of header */
   for (i=strlen(card); i<80; i++) card[i]=' ';
   card[80]='\n';
   ascii_to_iraf(tbuf,card,162);
   write(f->unix_hdr_fid,tbuf,162);

   return(0);
}


static int put_iraf_history_card(text)
char text[];
/* adds text to history area if room;  returns -1 if not */
{
   int len_str;

   len_str = 2 * (strlen(text)+1);
   if (iraf_hist_index + len_str >= SZ_IMHIST) return(-1);
   ascii_to_iraf(&ih.im_history[iraf_hist_index],text,len_str);
   iraf_hist_index += len_str;
   ih.im_history[iraf_hist_index-1] = '\n';
   ih.im_history[iraf_hist_index-0] = '\0';
   ih.im_history[iraf_hist_index+1] = '\0';
   return(0);

}


static void ascii_to_iraf(iraf_area,text,n_iraf)
char iraf_area[];
char text[];
int n_iraf;
{
   int i;
   char *c;

   for (i=0; i<n_iraf; i++) iraf_area[i] = '\0';
   /* Modified by David Shupe 01-Nov-1996 to work on Linux */
   if (im_byteorder() == 1)
       for (c=text, i=1; *c; c++, i += 2) iraf_area[i] = *c;
   else
       for (c=text, i=0; *c; c++, i += 2) iraf_area[i] = *c;

}


static void write_fits_card(text)
char text[];
{
   static char buff[2880]; /* for forming blocks of header cards */
   static char *card = buff; /* pointer into buffer */

   char *c, *t;

   if (text) {
      for (c=card, t=text; *t && c<card+80; c++, t++) *c = *t;
      while (c < card+80) *c++ = ' ';
      card += 80;
      if (card < buff+2880) return;
   }

   if (card == buff) return;
   for (c=card; c < buff+2880;) *c++ = ' ';
   if (f->unix_hdr_fid) write(f->unix_hdr_fid,buff,2880);
   else fwrite(buff,1,2880,stdout);
   card = buff;
   return;
}



void
IM_swap_4(buffer,n)
unsigned char buffer[];
int n;
/*  1234  to 4321  */
{
   unsigned char *i;
   unsigned char tmp;

   if (im_byteorder() == 1) return;
   for(i=buffer; i<buffer+n; i += 4) {
      tmp = *i; 
      *i = *(i+3);
      *(i+3) = tmp;
      tmp = *(i+1); 
      *(i+1) = *(i+2);
      *(i+2) = tmp;
   }
   return;
}

void
IM_swap_8(buffer,n)
unsigned char buffer[];
int n;
/*     12345678  to 43218765   !! is this correct ?!? */
/* NO, 12345678  to 87654321   is correct ! */
{
   unsigned char *i;
   unsigned char tmp;

   if (im_byteorder() == 1) return;
   for(i=buffer; i<buffer+n; i += 8) {
      tmp = *i; 
      *i = *(i+7);
      *(i+7) = tmp;
      tmp = *(i+1); 
      *(i+1) = *(i+6);
      *(i+6) = tmp;
      tmp = *(i+2); 
      *(i+2) = *(i+5);
      *(i+5) = tmp;
      tmp = *(i+3); 
      *(i+3) = *(i+4);
      *(i+4) = tmp;
   }
   return;
}

/*---------------------------------------------------------*/

void
IM_swap_2(buffer,n)
unsigned char buffer[];
int n;
{
   unsigned char *i;
   unsigned char tmp;

   if (im_byteorder() == 1) return;
   for(i=buffer; i<buffer+n; i += 2) {
      tmp = *i; 
      *i = *(i+1);
      *(i+1) = tmp;
   }
   return;
}

int
im_byteorder()
{
   int im_fitsorder;
   long netlong;
   unsigned char *byte_ptr;

#ifdef ALPHA
    return(0);
#else /* not ALPHA */
   /* test for byteorder */
   netlong = 0x12345678;
   byte_ptr = (unsigned char *) &netlong;
   if ((byte_ptr[0] == 0x12) && (byte_ptr[1] == 0x34))
       im_fitsorder = 1; 
   else if ((byte_ptr[0] == 0x78) && (byte_ptr[1] == 0x56))
       im_fitsorder = 0; 
   else
   {
       fprintf(stderr, 
	   "fatal error:  unrecognized byte order on this machine\n");
       exit(1);
   }
   return (im_fitsorder);
#endif /* ALPHA */
}
/* 
 * set treatment of "raw" pixels from FITS format files
 *  if v  then read pixel bytes [i,s,c types] as they appear in the file
 *	  I.E. Don't swap to machine order
 *  else  do swap bytes to host order
 */
int im_raw(v)
int v;
{
    int l;
    l = im_rawflag;
    if(v) im_rawflag = 1;
    else
	  im_rawflag = 0;

    return(l);
}

