/*   mapset.c   */
/* Copyright (C) 1992, California Institute of Technology. */
/* U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* is acknowledged. */

/* creates a map_block given all of the header information in binary */

#include <math.h>
#include <string.h>
#include "map_struct.h"

int map_set
  (p,ctype1,ctype2,cdelt1,cdelt2,crval1,crval2,crpix1,crpix2,crota1,crota2)
struct map_struct *p;
char   ctype1[];
char   ctype2[];
double cdelt1;
double cdelt2;
double crval1;
double crval2;
double crpix1;
double crpix2;
double crota1;
double crota2;
/* returns:
   0 = OK
  -1 = illegal ctype coordinate type
  -2 = inconsistent ctype coordinates
  -3 = inconsistent ctype projection types
  -4 = illegal center coordinate specified (crval)
  -5 = one of the cdelts is 0
  -6 = unequal crota's (not implemented currently)
*/
  

/* ===================================================
FITS ctypes:

done:
   ---- linear
   (or blank)
   -TAN Gnomonic
   -SIN Orthographic
   -AIT Aitoff
   -ARC arc
   -POL Polar

coming soon:
   -NCP NCP
   -STG stereographic
   -GLS global sinusoidal (Sanson Flamsteed)
   -MER Mercator

===================================================
*/

{
/* save numeric values */
   p->lon0 = crval1 * RAD_PER_DEG;
   p->lat0 = crval2 * RAD_PER_DEG;
   if (cdelt1 == 0.0 || cdelt2 == 0.0) return(-5);
   p->x_scale = DEG_PER_RAD / cdelt1;
   p->y_scale = DEG_PER_RAD / cdelt2;
   p->x0 = crpix1;
   p->y0 = crpix2;
   p->twist_angle1 = crota1 * RAD_PER_DEG;
   p->twist_angle2 = crota2 * RAD_PER_DEG;

   if (crota1 && crota1!=crota2) return(-6); /* currently not allowed */


/* figgure out coordinate type */
   p->map_coord_type = ' ';

   if (strncmp("RA--",ctype1,4)==0 || strcmp("RA",ctype1)==0 ) {
      if (strncmp("DEC",ctype2,3)!=0) return(-2);
      p->map_coord_type = 'r';
   }

   else if (strncmp("ELON",ctype1,4) ==0) {
      if (strncmp("ELAT",ctype2,4) !=0) return(-2);
      p->map_coord_type = 'e';
   }

   else if (strncmp("GLON",ctype1,4) ==0) {
      if (strncmp("GLAT",ctype2,4) !=0) return(-2);
      p->map_coord_type = 'g';
   }

   if (p->map_coord_type == ' ') return(-1);

/* figgure out projection type */

   if      ((int)strlen(ctype1) <= 4) p->proj_type = P_linear;
   else if (strncmp("----",ctype1+4,4) ==0) p->proj_type = P_linear;

   else if (strncmp("-AIT",ctype1+4,4) ==0) {
      if (strncmp("-AIT",ctype2+4,4) !=0) return(-3);
      if (p->lat0 != 0) return(-4);
      p->proj_type = P_AIT;
   }

   else if (strncmp("-TAN",ctype1+4,4) ==0) {
      if (strncmp("-TAN",ctype2+4,4) !=0) return(-3);
      p->sin_lat0 = sin(p->lat0);
      p->cos_lat0 = cos(p->lat0);
      p->proj_type = P_TAN;
   }

   else if (strncmp("-SIN",ctype1+4,4) ==0) {
      if (strncmp("-SIN",ctype2+4,4) !=0) return(-3);
      p->sin_lat0 = sin(p->lat0);
      p->cos_lat0 = cos(p->lat0);
      p->proj_type = P_SIN;
   }

   else if (strncmp("-ARC",ctype1+4,4) ==0) {
      if (strncmp("-ARC",ctype2+4,4) !=0) return(-3);
      p->sin_lat0 = sin(p->lat0);
      p->cos_lat0 = cos(p->lat0);
      p->proj_type = P_ARC;
   }

   else if (strncmp("-POL",ctype1+4,4) ==0) {
      if (strncmp("-POL",ctype2+4,4) !=0) return(-3);
      if (crval2 != 90.0 && crval2 != -90.0) return(-4);
      p->proj_type = P_POL;
   }

   else {
      if (strcmp(ctype1+4,ctype2+4) !=0) return(-3);
      p->sin_lat0 = sin(p->lat0);
      p->cos_lat0 = cos(p->lat0);
      p->proj_type = P_custom;
   }


/* return if all OK */
   p->magic = MAGIC_NUMBER;
   return(0);

}
