/*   mapi.c   */
/* Copyright (C) 1992, California Institute of Technology. */
/* U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* is acknowledged. */

/* inverse (x,y  ->  lat,lon) map projection subroutines */

#include <math.h>
#include "im.h"
#include "map_struct.h"

/* function prototypes */
#ifdef _NO_PROTO
static void mapi_AIT();
static void mapi_TAN();
static void mapi_SIN();
static void mapi_ARC();
static void mapi_POL();
#else
static void mapi_AIT(struct map_struct *m);
static void mapi_TAN(struct map_struct *m);
static void mapi_SIN(struct map_struct *m);
static void mapi_ARC(struct map_struct *m);
static void mapi_POL(struct map_struct *m);
#endif /* _NO_PROTO */

/*-------------------------------------------*/
int mapi(p,coord_type,in_x,in_y,out_lon,out_lat)
struct map_struct *p; /* from map_set */
char coord_type; /* r g e */
double in_x; /* pseudo degrees */
double in_y; /* pseudo degrees */
double *out_lon; /* degrees */
double *out_lat; /* degrees */
/* return: 0=OK  -1=bad map_struct -2=bad coord_type -3=bad x,y */
{
   double save_x;
   double cos_twist, sin_twist;
   int stat;

/* be sure this ia a legal map_struct */
   if (p->magic != MAGIC_NUMBER) return(-1);

/* remove scale and offset to get x,y */
   p->x = (in_x - p->x0) / p->x_scale;
   p->y = (in_y - p->y0) / p->y_scale;

/* remove twist */
   if (p->twist_angle2) {
      cos_twist = cos(p->twist_angle2);
      sin_twist = sin(p->twist_angle2);
      save_x = p->x;
      p->x = p->x * cos_twist - p->y * sin_twist;
      p->y = p->y * cos_twist + save_x * sin_twist;
   }

   switch(p->proj_type) {
      case P_AIT: mapi_AIT(p); break;
      case P_TAN: mapi_TAN(p); break;
      case P_SIN: mapi_SIN(p); break;
      case P_ARC: mapi_ARC(p); break;
      case P_POL: mapi_POL(p); break;
      case P_custom: if (p->custom_mapi) {(*(p->custom_mapi))(p); break;}
	   /* BUG ? - fallthrough   - RBH */
      case P_linear: p->dlon = p->x; p->lat = p->y + p->lat0; break;
   }; /* end switch */

/* convert to degrees */
   *out_lon = (p->dlon + p->lon0) * DEG_PER_RAD;
   if (*out_lon >  360.0) *out_lon -= 360.0;
   if (*out_lon <    0.0) *out_lon += 360.0;
   *out_lat = p->lat * DEG_PER_RAD;

/* do coordinate conversion if needed */
   if(coord_type != ' ' && p->map_coord_type != coord_type) {
      stat=map_coord
       (p->map_coord_type,*out_lon,*out_lat,coord_type,out_lon,out_lat);
      if (stat<0) return(-2);
   }

/* return */
   return(0);

}


/*-------------------------------------------*/
static void
mapi_AIT(m)
struct map_struct *m;
{
   double t, a;

   t = 4.0 - 0.25*m->x*m->x - m->y*m->y;
   if (t < 0) return;
   a = sqrt(t);
   t = 0.5*a*m->y;
   if (t > 1.0) t = 1.0;
   if (t < -1.0) t = -1.0;
   m->lat = asin(t);
   t = cos(m->lat);
   if (t != 0.0) {
      t = 0.25 * a * m->x / t;
      if (t > 1.0) t = 1.0;
      if (t < -1.0) t = -1.0;
      m->dlon = 2.0 * asin(t);
   }
   else m->dlon = 0.0;
}



/*-------------------------------------------*/
static void
mapi_TAN(m)
struct map_struct *m;
{
   double t;

   t = m->cos_lat0 - m->y * m->sin_lat0;
   m->dlon = atan2(m->x,t);
   if (t == 0) m->lat = 90.0;
   m->lat = atan( cos(m->dlon) * (m->y*m->cos_lat0+m->sin_lat0) / t);
}



/*-------------------------------------------*/
static void
mapi_SIN(m)
struct map_struct *m;
{
   double t, delta;

   t = 1.0 - m->x*m->x - m->y*m->y;
   if (t < 0.0) return;
   delta = sqrt(t);
   t = m->y * m->cos_lat0 + delta * m->sin_lat0;
   if (t >  1.0) t =  1.0;
   if (t < -1.0) t = -1.0;
   m->lat = asin (t);
   t = m->cos_lat0 * delta - m->y * m->sin_lat0;
   m->dlon = atan2(m->x , t);
}





/*-------------------------------------------*/
static void
mapi_ARC(m)
struct map_struct *m;
{

   double theta, sintheta, f;
   double t;

   theta = sqrt(m->x*m->x + m->y*m->y);
   sintheta = sin(theta);
   if (theta) f = sintheta / theta;
   else       f = 0;
   t = f * m->y * m->cos_lat0 + cos(theta) * m->sin_lat0;
   if (t >  1.0) t =  1.0;
   if (t < -1.0) t = -1.0;
   m->lat = asin (t);
   t = cos(m->lat);
   if (t) t = f * m->x / cos(m->lat);
   if (t >  1.0) t =  1.0;
   if (t < -1.0) t = -1.0;
   m->dlon = asin (t);
}



/*-------------------------------------------*/
static void
mapi_POL(m)
struct map_struct *m;
{
   if (m->lat0 > 0) {
      m->dlon = atan2(m->x,-m->y);
      m->lat = m->lat0 - hypot(m->x,m->y);
   }
   else {
      m->dlon = atan2(m->x,m->y);
      m->lat = hypot(m->x,m->y) + m->lat0;
   }
}

