/*   mapf.c   */
/* Copyright (C) 1992, California Institute of Technology. */
/* U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* is acknowledged. */

/*********************************************************
  forward (lon,lat  to x,y) map transformations
*********************************************************/

#include <math.h>
#include <stdlib.h>
#include "im.h"
#include "map_struct.h"

/* function prototypes */
#ifdef _NO_PROTO
static void mapf_AIT();
static void mapf_TAN();
static void mapf_SIN();
static void mapf_ARC();
static void mapf_POL();
#else
static void mapf_AIT(struct map_struct *m);
static void mapf_TAN(struct map_struct *m);
static void mapf_SIN(struct map_struct *m);
static void mapf_ARC(struct map_struct *m);
static void mapf_POL(struct map_struct *m);
#endif /* _NO_PROTO */

/*-------------------------------------------------------*/
int mapf(p,coord_type,in_lon,in_lat,out_x,out_y)
struct map_struct *p; /* from map_set */
int coord_type; /* r g e */
double in_lon; /* degrees */
double in_lat; /* degrees */
double *out_x; /* pseudo degrees */
double *out_y; /* pseudo degrees */
/* return: 0=OK   -1=bad map_struct   -2=bad coord type */
{
   double save_x;
   double cos_twist, sin_twist;
   int stat;

/* be sure this ia a legal map_struct */
   if (p->magic != MAGIC_NUMBER) return(-1);

/* do coordinate conversion if needed */
   if(coord_type != ' ' && p->map_coord_type != coord_type) {
      stat=map_coord(
       coord_type,in_lon,in_lat,p->map_coord_type,&in_lon,&in_lat);
      if (stat < 0) return(-2);
   }

/* get dlon and lat in radians */
   p->dlon = in_lon*RAD_PER_DEG - p->lon0;
   if      (p->dlon < -PI) p->dlon += TWO_PI; /* get it between -PI and +PI */
   else if (p->dlon >  PI) p->dlon -= TWO_PI;
   p->lat = in_lat * RAD_PER_DEG;

/* do map projection */
   switch(p->proj_type) {
      case P_AIT: mapf_AIT(p); break;
      case P_TAN: mapf_TAN(p); break;
      case P_SIN: mapf_SIN(p); break;
      case P_ARC: mapf_ARC(p); break;
      case P_POL: mapf_POL(p); break;
      case P_custom: if(p->custom_mapf) {(*(p->custom_mapf))(p); break;}
	  /* BUG ? fallthrough  - RBH */
      case P_linear: p->x = p->dlon; p->y = p->lat - p->lat0; break;
   }; /* end switch */

/* apply twist angle(s) */
   if (p->twist_angle2) {
      cos_twist = cos(-(p->twist_angle2));
      sin_twist = sin(-(p->twist_angle2));
      save_x = p->x;
      p->x = p->x * cos_twist - p->y * sin_twist;
      p->y = p->y * cos_twist + save_x * sin_twist;
   }

/* apply scale and offset */
   *out_x = p->x * p->x_scale + p->x0;
   *out_y = p->y * p->y_scale + p->y0;

/*
printf("p->proj_type = %d\n",p->proj_type);

printf("p->y = %f\n",p->y);
printf("p->y_scale = %f\n",p->y_scale);
printf("p->y0 = %f\n",p->y0);
printf("*out_y = %f\n",*out_y);

printf("p->x = %f\n",p->x);
printf("p->x_scale = %f\n",p->x_scale);
printf("p->x0 = %f\n",p->x0);
printf("*out_x = %f\n",*out_x);
*/

   return(0);

}

/*-------------------------------------------------------*/
static void
mapf_AIT(m)
struct map_struct *m;
{
   double coslat, rho, sintheta, theta, f;

   coslat = cos(m->lat);
   rho = acos ( coslat * cos(m->dlon/2.0) );
   if (fabs(rho) < 0.000001) sintheta = 0.0;
   else sintheta = coslat * sin(m->dlon/2.0) / sin(rho);
   if (sintheta > 1.0) sintheta = 1.0;
   if (sintheta < -1.0) sintheta = -1.0;
   theta = asin(sintheta);
   f = sin(rho/2.0);
   m->x = f * sintheta   * 4.0;
   m->y = f * cos(theta) * 2.0;
   if (m->lat < 0) m->y = - m->y;
}


/*-------------------------------------------------------*/
static void
mapf_TAN(m)
struct map_struct *m;
{
   double coslat, coslatcosdlon, f;

   coslat = cos(m->lat);

/*
printf("m->lat = %f\n",m->lat);
printf("m->dlon = %f\n",m->dlon);
printf("coslat = %f\n",coslat);
*/

   coslatcosdlon = coslat * cos(m->dlon);
   f = m->sin_lat0 * sin(m->lat) + m->cos_lat0 * coslatcosdlon;
   if (f >= 0) {if (f <  0.000001) f =  0.000001;}
   else        {if (f > -0.000001) f = -0.000001;}
   m->x = coslat * sin(m->dlon) / f;

/*
printf("sin() = %f\n",sin(m->dlon));
printf("f = %f\n",f);
printf("m->x = %f\n",m->x);
*/

   m->y = (m->cos_lat0 * sin(m->lat) - m->sin_lat0 * coslatcosdlon) / f;
}


/*-------------------------------------------------------*/
static void
mapf_SIN(m)
struct map_struct *m;
{
   double coslat;

   coslat = cos(m->lat);
   m->x = coslat * sin(m->dlon);
   m->y = m->cos_lat0 * sin(m->lat) - m->sin_lat0 * coslat * cos(m->dlon);
}



/*-------------------------------------------*/
static void
mapf_ARC(m)
struct map_struct *m;
{
   double theta, f;
   double sinlat, coslat, cosdlon;

   sinlat = sin(m->lat);
   coslat = cos(m->lat);
   cosdlon = cos(m->dlon);
   theta = acos(sinlat * m->sin_lat0 + coslat * m->cos_lat0 * cosdlon);
   if (theta) f = theta / sin(theta);
   else       f = 0;
   m->x = f * coslat * sin(m->dlon);
   m->y = f * (sinlat * m->cos_lat0 -  coslat * m->sin_lat0 * cosdlon);
}


/*-------------------------------------------------------*/
static void
mapf_POL(m)
struct map_struct *m;
{
   double dist;
   if (m->lat0 > 0) {
      dist = m->lat0 - m->lat;
      m->y = - dist * cos(m->dlon);
   }
   else {
      dist = m->lat - m->lat0;
      m->y = dist * cos(m->dlon);
   }
   m->x =   dist * sin(m->dlon);
}
