/*   coord.c   ---  coordinate conversion subroutine set  */
/* Copyright (C) 1992, California Institute of Technology. */
/* U.S. Government Sponsorship under NASA Contract NAS7-918 */
/* is acknowledged. */


#include "im.h"
#include <math.h>

#ifndef PI
#ifdef M_PI
#define PI M_PI
#else
#define PI 3.14159265358979323846
#endif
#endif

#define RAD_PER_DEG (PI/180.0)
#define DEG_PER_RAD (180.0/PI)

static double m_rg[10] = { 0.0,
    -0.066988739, -0.872755757, -0.483538931,
     0.492728466, -0.450346972,  0.744584624,
    -0.867600811, -0.188374611,  0.460199781 };

static double m_gr[10] = { 0.0,
    -0.066988739,  0.492728466, -0.867600811,
    -0.872755757, -0.450346972, -0.188374611,
    -0.483538931,  0.744584624,  0.460199781 };

static double m_re[10] = { 0.0,
     1.000000000,  0.000000000,  0.000000000,
     0.000000000,  0.917436945,  0.397881203,
     0.000000000, -0.397881203,  0.917436945 };

static double m_er[10] = { 0.0,
     1.000000000,  0.000000000,  0.000000000,
     0.000000000,  0.917436945, -0.397881203,
     0.000000000,  0.397881203,  0.917436945 };

static double m_eg[10] = { 0.0,
    -0.066988739, -0.993089427, -0.096363369,
     0.492728466, -0.116908724,  0.862294038,
    -0.867600811,  0.010283015,  0.497154998 };

static double m_ge[10] = { 0.0,
    -0.066988739,  0.492728466, -0.867600811,
    -0.993089427, -0.116908724,  0.010283015,
    -0.096363369,  0.862294038,  0.497154998 };

int
map_coord(in_type,in_lon,in_lat,out_type,out_lon,out_lat)
int in_type; /* r g e */
double in_lon; /* degrees */
double in_lat; /* degrees */
int out_type; /* r g e */
double *out_lon; /* degrees 0 to 360 */
double *out_lat; /* degrees  -90 to +90 */
/* returns 0 if OK,  returns -2 if illegal type selected */
{
   double *m;
   double in_lon_r, in_lat_r;
   double x_hat, y_hat, z_hat;
   double x_prime, y_prime, z_prime;

   switch(in_type) {
      case 'r':
       if      (out_type == 'r') m = 0;
       else if (out_type == 'g') m = m_rg;
       else if (out_type == 'e') m = m_re;
       else return(-2);
       break;
      case 'g':
       if      (out_type == 'r') m = m_gr;
       else if (out_type == 'g') m = 0;
       else if (out_type == 'e') m = m_ge;
       else return(-2);
       break;
      case 'e':
       if      (out_type == 'r') m = m_er;
       else if (out_type == 'g') m = m_eg;
       else if (out_type == 'e') m = 0;
       else return(-2);
       break;
      default: return(-2);
   }; /* end switch */

   if (m == 0) {
      *out_lon = in_lon;
      *out_lat = in_lat;
      return(0);
   }

      
   in_lon_r = in_lon * RAD_PER_DEG;
   in_lat_r = in_lat * RAD_PER_DEG;

   x_hat = cos(in_lon_r) * cos(in_lat_r);
   y_hat = sin(in_lon_r) * cos(in_lat_r);
   z_hat =                 sin(in_lat_r);

   x_prime = x_hat * m[1] + y_hat * m[2] + z_hat * m[3] ;
   y_prime = x_hat * m[4] + y_hat * m[5] + z_hat * m[6] ;
   z_prime = x_hat * m[7] + y_hat * m[8] + z_hat * m[9] ;

   *out_lon = atan2(y_prime,x_prime) * DEG_PER_RAD;
   if (*out_lon < 0) *out_lon += 360.0;
   if      (z_prime >  1.0) *out_lat =  90.0;
   else if (z_prime < -1.0) *out_lat = -90.0;
   else *out_lat = asin(z_prime) * DEG_PER_RAD;

   return(0);

}  /********  end map_coord  **********/
