#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <signal.h>
#include "svclib.h"

#define TRUE    1
#define FALSE   0


static int  pid[SVC_MAXSVC] = {SVC_MAXSVC*0};
static int  running[SVC_MAXSVC];

static int  fdin[SVC_MAXSVC][2],  fdout[SVC_MAXSVC][2];
static FILE *toexec[SVC_MAXSVC], *fromexec[SVC_MAXSVC];

static char svcname[SVC_MAXSVC][128];
static char sigfunc[SVC_MAXSVC][128];
static char quitstr[SVC_MAXSVC][128];


/* local function prototypes */
#ifdef _NO_PROTO
    static void svc_check();
    static int getargs ();
#else
    static void svc_check(void);
    static int getargs (char *cmd, char *cmdv[]);
#endif _NO_PROTO


static void
svc_check()
{
   int i, npid;

   for(i=0; i<SVC_MAXSVC; ++i)
   {
      running[i] = FALSE;

      npid = pid[i];

      if(npid > 0)
      {
	 if (!kill(npid, 0))
	    running[i] = TRUE;
      }
   }
#ifdef NOTDEF
   printf("RBH svc_check:  pid[0] = %d  running[0] = %d\n", pid[0], running[0]);
#endif NOTDEF
}



int svc_init(svcstr)

char *svcstr;
{
   int i, npid, index, cmdc;
   char str[SVC_STRLEN];

   char path[SVC_STRLEN], *cmdv[SVC_STRLEN];

   if((int)strlen(svcstr) < SVC_STRLEN - 1)
      strcpy(str, svcstr);
   else
      return(SVC_ERROR);

   cmdc = getargs(str, cmdv);
   cmdv[cmdc] = (char *) NULL;

   if(cmdc < 1)
      return(SVC_ERROR);

   strcpy(path, cmdv[0]);


   /* Find an open slot in the service table */

   index = -1;
   for(i=0; i<SVC_MAXSVC; ++i)
   {
      if(pid[i] == 0)
      {
	 index = i;
	 break;
      }
   }

   if(index < 0)
      return(SVC_ERROR);

   strcpy(svcname[index], "");
   strcpy(sigfunc[index], "");
   strcpy(quitstr[index], "");
   running[index] = TRUE;


   /* Ignore child termination signals */

   signal(SIGCHLD, SIG_IGN);


   /* Set up pipes and start child */

   (void) pipe(fdin[index]);
   (void) pipe(fdout[index]);

   if((npid=fork()) == 0)                                       /* CHILD */
   {
      close( fdin[index][1]);
      close(fdout[index][0]);

      (void) dup2( fdin[index][0], 0);
      (void) dup2(fdout[index][1], 1);

      execvp(path, cmdv);
      fprintf(stderr,"svclib: could not start %s\n", path);
      perror(path);
      exit(1);
   }
   else                                                         /* PARENT */
   {
      close( fdin[index][0]);
      close(fdout[index][1]);

      toexec[index]   = fdopen( fdin[index][1], "a");
      fromexec[index] = fdopen(fdout[index][0], "r");

      pid[index] = npid;
   }

   /* Return index number */

   return(index);
}



int svc_register(index, name, sig, quit)

int   index;
char *name, *sig, *quit;

{
   if(pid[index] != 0)
   {
      strcpy(svcname[index], name);
      strcpy(sigfunc[index], sig);
      strcpy(quitstr[index], quit);

      return(SVC_ERROR);
   }
   else
      return(SVC_OK);
}



int svc_close(index)

int index;
{

   if(pid[index] != 0)
   {
      if((int)strlen(quitstr[index]) > 0)
	 fprintf(toexec[index], "%s\n", quitstr[index]);

      (void) fflush(  toexec[index]);
      (void) fflush(fromexec[index]);

      (void) fclose(  toexec[index]);
      (void) fclose(fromexec[index]);

      (void) close( fdin[index][1]);
      (void) close(fdout[index][0]);

      svc_check();
      if(running[index])
	 (void) kill(pid[index], SIGKILL);

      pid[index] = 0;
      running[index] = FALSE;
      
      return(SVC_OK);
   }
   else
      return(SVC_ERROR);
}
      
   

int svc_send(index, cmd)

int   index;
char *cmd;
{
   svc_check();

   if(!running[index])
   {
      svc_close(index);

      return(SVC_ERROR);
   }

   fprintf(toexec[index], "%s\n", cmd);
   fflush(toexec[index]);

   return(SVC_OK);
}



char *svc_receive(index)

int index;
{
   static char str[SVC_STRLEN];


   if(fgets(str, SVC_STRLEN, fromexec[index]) != NULL)
   {
      svc_check();

      if(!running[index])
	 svc_close(index);

      if(str[strlen(str) - 1] == '\n')
	 str[strlen(str) - 1] = '\0';

      return(str);
   }
   else
   {
      svc_check();

      if(!running[index])
	 svc_close(index);

      return((char *)NULL);
   }
}



static int getargs (cmd, cmdv)

char   *cmd;        /* input argument - command string to be parsed */
char   *cmdv[];	    /* output argument - array of strings (cmd parameters) */

{
   char   *ptmp;
   int     i, len, cmdc;	    
   int     in_quotes;


   /* Scan the command for commas, semicolons, etc. */

   in_quotes = FALSE;
   len = (int)strlen (cmd);

   for (i = 0; i < len; i++)
   {
      if (isprint (cmd[i]))
      {
	 if (cmd[i] == '"')
	    in_quotes = !in_quotes;    /* toggle quoting on/off          */

	 if (in_quotes == FALSE)       /* if this isn't a quoted string  */
	 {
	    if (cmd[i] == ',')	       /* change commas to spaces        */
	       cmd[i] = ' ';

	    if(cmd[i] == ';')          /* change semicolon to null       */
	       cmd[i] = '\0';
	 }

	 if (cmd[i] == '\0')	       /* if this is a null */
	    break;		       /* stop the scan     */
      }

      else
	 cmd[i] = ' ';                 /* change non-printables to blank */
   }


   /* Collect pointers to each argument for this command */

   ptmp = cmd;
   cmdc = 0;

   while (*ptmp == ' ')	               /* discard spaces */
      ptmp++;

   while (*ptmp != '\0')	       /* while some of string is left */
   {

      if (*ptmp == '"')                /* If this is a quote */
      {
	 *ptmp = '\0';                 /* change it to a NULL */

	 ptmp++;                       /* move past quote */
	 cmdv[cmdc] = ptmp;	       /* pointer to this arg  */
	 ++cmdc;

	 while (*ptmp != '"' && *ptmp != '\0')
	    ptmp++;                    /* search for ending quote */

	 if (*ptmp == '"')
	 {
	    *ptmp = '\0';              /* change quote to a NULL */
	    ++ptmp;
	 }
      }

      else
      {
	 cmdv[cmdc] = ptmp;            /* pointer to this arg     */
	 ++cmdc;
      }
      
      while ((*ptmp != ' ') && (*ptmp != '\0'))
	 ptmp++;                       /* Find the next space */

      if (*ptmp == ' ')
	 *ptmp++ = '\0';               /* change space to null and advance */

      while (*ptmp == ' ')
	 ptmp++;                       /* discard additional spaces */
   }

   return (cmdc);
}



SVC *svc_struct(instr)

char *instr;
{
   int i, len, inquote, blev;
   char *str, *p, *pend, *begin, *end;
   char *sb, *se, *key, *peq, *val, *kend;

   SVC *svc;


    if (instr == NULL)
      return((SVC *)NULL);

   /* Allocate initial space for structure */

   svc = (SVC *) malloc(sizeof(SVC));

   svc->count  = 0;
   svc->nalloc = SVC_SVCCNT;

   svc->key = (char **) malloc(svc->nalloc * sizeof(char *));
   svc->val = (char **) malloc(svc->nalloc * sizeof(char *));

   for(i=0; i<svc->nalloc; ++i)
   {
      svc->key[i] = (char *) malloc(SVC_STRLEN * sizeof(char));
      svc->val[i] = (char *) malloc(SVC_STRLEN * sizeof(char));
   }



   /* Strip off header and trailer */

   len = (int)strlen(instr);
   str = (char *) malloc((len + 1) * sizeof(char));
   strcpy(str, instr);

   p  = str;
   pend = str+len-1;

   for (;;)
   {
      if(*p == ' ')
      {
	 *p = '\0';
	 ++p;
      }
      else
	 break;
   }

   for (;;)
   {
      if(*pend == ' ')
      {
	 *pend = '\0';
	 --pend;
      }
      else
	 break;
   }

   if(     !strncmp(p, "[struct ", 8))
      p += 8;
   else if(!strncmp(p, "[array ",  7))
      p += 7;
   else
      return((SVC *)NULL);
   
   if(*pend != ']')
      return((SVC *)NULL);
   
   *pend = '\0';



   /* Now step through the key = val (or val, val, ... sets for array) */

   blev = 0;
   begin = p;
   end = p;

   len = (int)strlen(p);

   /* Loop over structure elements */

   for (;;)
   {
      /* Search for closing comma */

      inquote = FALSE;
      for (;;)
      {
	 if(!inquote && blev == 0 && *end == ',')
	    break;

	 if(*end == '\0')
	    break;

	 if(end > p + len)
	    break;

	 if(*end == '"' && *(end-1) != '\\')
	    inquote = !inquote;

	 if(!inquote && *end == '[')
	    ++blev;

	 if(!inquote && *end == ']')
	    --blev;
	 
	 ++end;
      }
      if(inquote)
	 return((SVC *)NULL);

      *end = '\0';


      /* Take the key = val expression apart */

      sb = begin;
      se = end;

      for (;;)
      {
	 if(*sb == ' ')
	 {
	    *sb = '\0';
	    ++sb;
	 }
	 else
	    break;
      }

      for (;;)
      {
	 if(*se == ' ')
	 {
	    *se = '\0';
	    --se;
	 }
	 else
	    break;
      }

      blev = 0;
      inquote = FALSE;
      key = sb;
      val = sb;

      /* Find '=' (if any) */

      peq = (char *) NULL;

      for (;;)
      {
	 if(!inquote && blev == 0 && *val == '=')
	 {
	    peq = val;
	    *peq = '\0';
	    ++val;
	    break;
	 }

	 if(*val == '"' && *(val-1) != '\\')
	    inquote = !inquote;

	 if(val >= se)
	    break;

	 if(!inquote && *val == '[')
	    ++blev;

	 if(!inquote && *val == ']')
	    --blev;
	 
	 ++val;
      }
      if(inquote)
	 return((SVC *)NULL);

      /* Discard trailing spaces from key */
      kend = key + strlen(key) - 1;
      for (;;)
      {
	if (*kend == ' ')
	{
	    *kend = '\0';
	    --kend;
	}
	else
	    break;
      }

      /* Skip leading blanks in val */

      for (;;)
      {
	 if(*val == ' ')
	 {
	    *val = '\0';
	    ++val;
	 }
	 else
	    break;
      }


      /* Remove external quotes from key, val */

      if(strlen(key) > (size_t) 1)
      {
	 if(key[0] == '"' && key[strlen(key) - 1] == '"')
	 {
	    key[strlen(key) - 1] = '\0';
	    ++key;
	 }
      }

      if(strlen(val) > (size_t) 1)
      {
	 if(val[0] == '"' && val[strlen(val) - 1] == '"')
	 {
	    val[strlen(val) - 1] = '\0';
	    ++val;
	 }
      }


      /* Assign the key, val to the return structure */

      if(peq)
      {
	 strcpy(svc->key[svc->count], key);
	 strcpy(svc->val[svc->count], val);
      }
      else
      {
	 sprintf(svc->key[svc->count], "%-d", svc->count);
	 strcpy(svc->val[svc->count], key);
      }


      /* If necessary, allocate more space for structure */

      ++svc->count;
      if(svc->count >= svc->nalloc)
      {
	 svc->nalloc += SVC_SVCCNT;

	 svc->key = (char **) realloc(svc->key, svc->nalloc * sizeof(char *));
	 svc->val = (char **) realloc(svc->val, svc->nalloc * sizeof(char *));

	 for(i=svc->nalloc - SVC_SVCCNT; i<svc->nalloc; ++i)
	 {
	    svc->key[i] = (char *) malloc(SVC_STRLEN * sizeof(char));
	    svc->val[i] = (char *) malloc(SVC_STRLEN * sizeof(char));
	 }
      }


      /* Go on to the next substructure */

      begin = end + 1;
      end = begin;

      if(end >= p + len)
	 break;
   }

   free(str);
   return(svc);
}




void svc_free(svc)

SVC *svc;
{
   int i, nalloc;
   nalloc = svc->nalloc;

   for(i=0; i<nalloc; ++i)
   {
      free(svc->key[i]);
      free(svc->val[i]);
   }

   free(svc->key);
   free(svc->val);

   free(svc);
   svc = (SVC *) NULL;   /* RBH: this has no effect */
}




char *svc_val(structstr, key, val)

char *structstr, *key, *val;
{
   int  i, len;
   char subkey[SVC_STRLEN], tail[SVC_STRLEN], subval[SVC_STRLEN];

   SVC *sv;

   strcpy(subkey, key);
   len = strlen(subkey);

   for(i=0; i<len; ++i)
   {
      if(subkey[i] == '.' || subkey[i] == '[')
      {
	 subkey[i] = '\0';

         break;
      }
   }

   if(subkey[strlen(subkey) - 1] == ']')
      subkey[strlen(subkey) - 1] = '\0';

   if(i >= len)
      tail[0] = '\0';
   else
      strcpy(tail, subkey + i + 1);
   
   len = strlen(tail);

   if((sv = svc_struct(structstr)) != (SVC *)NULL)
   {
      for(i=0; i<sv->count; ++i)
      {
	 if(strcmp(sv->key[i], subkey) == 0)
	 {
	    if(!len)
	    {
	       strcpy(val, sv->val[i]);
	       svc_free(sv);
	       return(val);
	    }

	    else if(svc_val(sv->val[i], tail, subval))
	    {
	       strcpy(val, subval);
	       svc_free(sv);
	       return(val);
	    }

	    else
	    {
	       svc_free(sv);
	       return((char *) NULL);
	    }
	 }
      }

      svc_free(sv);            /* RBH added */
      return((char *) NULL);
   }
   else
      return((char *) NULL);
}
