/* imprint.c  (image pixel printout program) */
static char version[] = "@(#)imprint.c %I% %E%";

main(argc,argv)
int argc;
char *argv[];
{
#define BUFSIZE 1000
#define MAXINFILES 32

   char infiles[MAXINFILES][64];
   int n_infiles;
   float limit_minval;
   int n_minval;
   float limit_maxval;
   int n_maxval;
   int limit_blanks; /* 0=no 1=also  2=only */
   int want_axes;
   int want_value;
   int want_header;

   float buffer[BUFSIZE];
   float *pix;
   int n_pix_in_buffer;

   int fid;
   int axis_type[10];
   int len_dn_col;
   int len_coord_col;
   int min_len_col;

   char bunit[12];
   int naxes;
   int naxis[10];
   char ctype[10][12];
   float crpix[10];
   float crval[10];
   float cdelt[10];
   int c_pix[10]; /* center pixel */
   int map_block[50];
   int i[10];
   int k;
   int n_ref; /* tmp val used when printing hdr info for ref pixel */
   char center_type[100];

   int qx, qy;
   double x0, y0;
   double lon, lat;
   double crota_x, crota_y;

   int j;
   int stat;

   par_f(infiles,"in,-");
   par_a(&n_infiles,"in,1,32");
   par_r(&limit_minval,"minval");
   par_a(&n_minval,"minval,0,1");
   par_r(&limit_maxval,"maxval");
   par_a(&n_maxval,"maxval,0,1");
   par_i(&limit_blanks,"blanks,0,0,2,0=no 1=also  2=only");
   par_i(&want_header,"header,1,0,1,want header info printout: 0=no 1=yes");
   par_i(&want_value,"value,1,0,1,pixel values: 0=no 1=yes");
   par_i(&want_axes,
     "axes,1,0,4,axes: 0=no 1=real 2=rel-CRPIX 3=rel-center 4=abs");
   if (par_g(argc,argv) <0) return(-1);
   par_vers(version+4);


   min_len_col = 6;
   len_coord_col = 8;

   for (k=0; k<n_infiles; k++) {
      fid = im_open(infiles[k],"r-");
      if (fid < 0) {
         msg1(1,-2,"Cannot read image file %s",infiles[k]);
         continue;
      }


      /* get_header_info */
      stat = im_rkey_i(fid,&naxes,"NAXIS");
      for (j=1; j<=naxes; j++) {
         im_rkey_i(fid,naxis+j,im_key("NAXIS",j));
         stat = im_rkey_c(fid,ctype[j],im_key("CTYPE",j));
         if (stat<0) switch(j) {
            case 1:  strcpy (ctype[j],"SAMP"); break;
            case 2:  strcpy (ctype[j],"LINE"); break;
            default: sprintf(ctype[j],"AXIS%d",j);
         }
         if (naxis[j] <= 1) axis_type[j] = 0;
         else {
            axis_type[j] = strlen(ctype[j]);
            if (axis_type[j] < min_len_col) axis_type[j] = min_len_col;
         }

         stat = im_rkey_r(fid,crval+j,im_key("CRVAL",j));
         if (stat<0) crval[j] = 1.0;
         stat = im_rkey_r(fid,cdelt+j,im_key("CDELT",j));
         if (stat<0) cdelt[j] = 1.0;
         stat = im_rkey_r(fid,crpix+j,im_key("CRPIX",j));
         if (stat<0) crpix[j] = 1.0;
         switch (want_axes) {
            case 3:
             c_pix[j] = (naxis[j]+1) / 2; break;
            case 2:
             if (crpix[j] > 0) c_pix[j] = crpix[j] + 0.5;
             else              c_pix[j] = crpix[j] - 0.5;
             break;
         }
      }

      stat = im_rkey_c(fid,bunit,"BUNIT");
      if (stat<0) strcpy(bunit,"VALUE");
      len_dn_col = strlen(bunit);
      if (len_dn_col < 10) len_dn_col = 10;

      qx = qy = 0;
      for (j=1; j<=naxes; j++) {
         if      (strcmp (ctype[j],"RA")    ==0) qx = j;
         else if (strncmp(ctype[j],"RA--",4)==0) qx = j;
         else if (strncmp(ctype[j],"ELON",4)==0) qx = j;
         else if (strncmp(ctype[j],"GLON",4)==0) qx = j;
         else if (strcmp (ctype[j],"DEC")   ==0) qy = j;
         else if (strncmp(ctype[j],"DEC-",4)==0) qy = j;
         else if (strncmp(ctype[j],"ELAT",4)==0) qy = j;
         else if (strncmp(ctype[j],"GLAT",4)==0) qy = j;
      }

      if (qy==0) qx = 0;

      crota_x = crota_y = 0.0;
      im_rkey_r(fid,&crota_x,im_key("CROTA",qx));
      im_rkey_r(fid,&crota_y,im_key("CROTA",qy));

      if (qx) {
         stat = map_set(map_block, ctype[qx], ctype[qy], cdelt[qx], cdelt[qy],
                 crval[qx], crval[qy], crpix[qx], crpix[qy], crota_x, crota_y);
         if (stat) {
            msg1(1,-2,"Bad geometric header info in %s",infiles[k]);
            qx = 0;
         }
      }

      if (want_axes==1) if (qx) {axis_type[qx] = -1; axis_type[qy] = -2; }

      /* print_header_info */
      if (want_header) {
         printf("\\C IMAGE = %s\n",infiles[k]);
         if (n_minval) printf("\\R FLUXMIN = %G\n",limit_minval);
         if (n_maxval) printf("\\R FLUXMAX = %G\n",limit_maxval);
         switch (limit_blanks) {
            case 1: printf("\\C BLANKS = YES\n"); break;
            case 2: printf("\\C BLANKS = ONLY\n"); break;
         }

         /* print header info for NAXISn =1 */
         for (j=1; j<=naxes; j++) {
            if (axis_type[j] == 0) printf("\\R %s = %G\n",ctype[j],
                                    (1-crpix[j])*cdelt[j]+crval[j]  );
         }

         /* print center if relative coordinates selected */
         if (want_axes >= 2) {
            if      (want_axes == 2) strcpy(center_type,"reference");
            else if (want_axes == 3) strcpy(center_type,"center");
            else if (want_axes == 4) strcpy(center_type,"first");
            if (want_axes == 4) {x0 = 1.0; y0 = 1.0; n_ref = 1;}
            else {x0 = c_pix[qx]; y0 = c_pix[qy]; n_ref = 0;}
            printf("\\C PIX%d = '%s pixel'\n",n_ref,center_type);
            stat = mapi(map_block,' ',x0,y0,&lon,&lat);
            if (stat ==0) {
               printf("\\R LON%d = %f\n",n_ref,lon);
               printf("\\R LAT%d = %+f\n",n_ref,lat);
            }
         }

         /* print column headers */
         printf("|%*s",len_dn_col,bunit);
         if (want_axes==1) if (qx) {
            printf("|%*s",len_coord_col,ctype[qx]);
            printf("|%*s",len_coord_col,ctype[qy]);
         }
         if (want_axes) for (j=1; j<=naxes; j++) { /* if NAXISn !=1 */
            if (axis_type[j] > 0) printf("|%*s",axis_type[j],ctype[j]);
         }
         printf("|\n");
      }


      /* print_pixel_lines */
      i[1] = 0;
      for (j = 2; j<=naxes; j ++) i[j] = 1;
      n_pix_in_buffer = 1;
      pix = buffer + n_pix_in_buffer; /* force read */

      while (n_pix_in_buffer > 0) {
         pix ++;
         for (j=1; j<=naxes; j++) { /* increment axis coordinates */
            if (i[j] < naxis[j]) { i[j]++; break; }
            else i[j] = 1;
         }
         /* read pixels if needed */
         if (pix >= buffer + n_pix_in_buffer) {
            n_pix_in_buffer = im_rpix_r(fid, buffer, BUFSIZE);
            if (n_pix_in_buffer <= 0) continue;
            pix = buffer;
         }
      
         /* check if qualified */
         if ( *pix != *pix) /* (blank pixel) */{
            if (limit_blanks == 0) continue;
         }
         else {
            if (limit_blanks == 2) continue;
            if (n_minval) if (*pix < limit_minval) continue;
            if (n_maxval) if (*pix > limit_maxval) continue;
         }
      
         /* print pixel value */
         if (want_value) {
            if (*pix == *pix) printf(" %#*.4G",len_dn_col,*pix);
            else              printf(" %*s",len_dn_col," *BLANK*");
         }

         /* print coordinates */
         switch (want_axes) {
            case 1: /* real */

            if (qx) { /* print lon&lat */
               x0 = i[qx];
               y0 = i[qy];
               stat = mapi(map_block,' ',x0,y0,&lon,&lat);
               if (stat == 0) {
                  printf(" %*.4f",len_coord_col,lon);
                  printf(" %+*.4f",len_coord_col,lat);
               }
               else {
                  printf(" %s",len_coord_col," ???");
                  printf(" %s",len_coord_col," ???");
               }
            }

             for (j=1; j<=naxes; j++) {
                if (axis_type[j] > 0) printf(" % #*.4G",axis_type[j],
                    (i[j]-crpix[j])*cdelt[j]+crval[j]  );
             }
             break;
            case 2: /* rel-center */
            case 3: /* rel-CRPIX */
             for (j=1; j<=naxes; j++) {
                if (axis_type[j] >0) printf(" %+*d",axis_type[j],i[j]-c_pix[j]);
             }
             break;
            case 4: /* absolute */
             for (j=1; j<=naxes; j++) {
                if (axis_type[j] > 0) printf(" %*d",axis_type[j],i[j]);
             }
             break;
         }

         printf("\n");

      }
      im_close(fid);
   } /* end for(k)  */
   msg(711,0,"DONE");
}

/* -----------------------------------
   format: table, FITS, simple, s "take" file(pi,da,etc.)
PER PIXEL:
   DN/flux: yes/no
   geo coords: none, rel to 1, center, rpix, RWC(native,RA,RLON,GLON,ELON)
   non-geom coords: none, abs, rwc
   distance: yes/no
HEADER INFO:
   file name
   limits
   mode=RWC:
      for NAXISn=1: CTYPE, effective CRVAL
   mode!=RWC:
      for all NAXIS: CTYPE, CRVAL, CRPIX, CDELT, CROTA
      BUNITS, BSCALE, BZERO
   col hdr info (none,in col, in hdr):
      for NAXISn>1: CTYPE
      BUNITS
  

/* NI:
distance and for "relative pixels"  ?? rel to [CRPIX] vs. [center]
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 FORMAT: 
  - FITS
  - IPAC table
  - plain
  - 's'
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 increments (per axis)
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 print more geo info when printing pixel type coords (?)
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 combine mult DN's on one line (all HCONS etc. (?)
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 proper print coords
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
   limit_radius 0=all  1=within circle 
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
 limit n_pix (bomb if too many    vs.  only do so many)
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
DIST TO CENT:
distance
*0 = none
 1 = real - degrees
 2 = pixels
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
HANDLING OF NAXIS=1's:
len1
 0 = none
*1 = print as header info
 2 = print in every line
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
COLUMN HEADERS WANTED:
columns
 0 = none
 1 = name
*2 = name & units
 3 = full
>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>

----------------------------------- */
